/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *        Copyright (c) 1990, ..., 1996 Bellcore            *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *    Deborah F. Swayne            Dianne Cook              *
 *   dfs@research.att.com       dicook@iastate.edu          *
 *      (973) 360-8423    www.public.iastate.edu/~dicook/   *
 *                                                          *
 *                    Andreas Buja                          *
 *                andreas@research.att.com                  *
 *              www.research.att.com/~andreas/              *
 *                                                          *
 ************************************************************/

#include <math.h>
#include <stdio.h>
#include <string.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"
#include "xgobiexterns.h"

/* To plot the case profile, linked to identification */
static Widget cprof_plot_shell, cprof_form, cprof_panel, cprof_plot_wksp;
static Widget cprof_cmd[5];
static Drawable cprof_plot_window, cprof_pixmap;
#define LBMARGIN 50  /* left margin */
#define RTMARGIN 20  /* right and top margins -- and bottom? */
static Dimension cprof_height, cprof_width;
static Dimension left_margin = LBMARGIN;

static int ncolors_used = 1;
static unsigned long colors_used[NCOLORS+1];

/*
 * By making these first three into arrays, I can plot points
 * and lines for nearest_point plus the sticky guys -- or parallel
 * coordinate plots.
*/
static fcoords *cprof_tform;
static lcoords *cprof_plane;
static icoords *cprof_screen;

static XPoint *points;
static XSegment *segs;
static XRectangle *open_rectangles;
static XRectangle *filled_rectangles;
static XArc *open_circles;
static XArc *filled_circles;

/*
 * If we're doing parallel coordinates, stickysegs contains segments
 * for all the points.
*/
/* nprofiles either points to xg->nsticky_ids or xg->nrows_in_plot. */
static int *nprofiles;
/* profile_ids either points to xg->sticky_ids or xg->rows_in_plot */
static int *profile_ids;
static XSegment *connsegs, *stickysegs;
static int nconnsegs;

static int *cprof_selectedvars;
static int cprof_nselectedvars;
static lcoords cprof_is;
static int cprof_midx;
static int cprof_midy;
static long cprof_shift_wrld_x;
static long cprof_shift_wrld_y;
static lims cprof_xlim;
static lims cprof_ylim;
/*static lims cprof_xnicelim;*/
static lims cprof_ynicelim;
static icoords cprof_screen_axes[3];
static int cprof_xdeci;
static int cprof_ydeci;
/*static float cprof_xtickdelta;*/
static float cprof_ytickdelta;
static tickinfo cprof_ticks;
static XSegment cprof_xtick_segs[100];
static XSegment cprof_ytick_segs[100];

/*
typedef struct {
  int nticks[NCOLS];   [0] for x; [1] for y;
  float xticks[NTICKS];
  float yticks[NTICKS];
  lcoords plane[NTICKS];
  icoords screen[NTICKS];
} tickinfo;
*/

static Boolean parallel_coords = False;
static Boolean cprof_showlines = True;
static Boolean cprof_showpoints = True;
static Boolean cprof_doublebuffer = False;
static Boolean cprof_fix_ymax = False;
static Boolean plot_imputed_values = True;

/* For the case profile */
#define STICKYSEGS(case,varoffset) \
  (stickysegs[case * (cprof_nselectedvars-1) + varoffset])

#define NDISPLAYBTNS 3
static Widget display_menu_cmd, display_menu, display_menu_btn[NDISPLAYBTNS];
#define SHOWPOINTS      0
#define SHOWLINES       1
#define DOUBLEBUFFER    2

#define NLINKBTNS 1
static Widget link_menu_cmd, link_menu, link_menu_btn[NLINKBTNS];
#define LINKPARCOORDS   0

#define NYAXISBTNS 2
static Widget yaxstyle_menu_cmd, yaxstyle_menu, yaxstyle_menu_btn[NYAXISBTNS];
#define SCALE_TOGETHER      0
#define SCALE_SEPARATELY 1
static int yaxis_style = SCALE_SEPARATELY;

void init_parcoords(xg)
  xgobidata *xg;
{
  xg->is_cprof_plotting = False;
  xg->link_cprof_plotting = True;
  cprof_selectedvars = (int *) XtMalloc((Cardinal)
    xg->ncols * sizeof(int));
}

/*
 * Code for ticks and axes on case profile plotting window.
*/

void
make_cprof_screen_axes(void)
/*
 * Since there's no shifting or scaling of this plot, just
 * force the axes to be stable.
*/
{
/*     .(axes[0].x, axes[0].y
 *     |
 *     |
 *     |
 *     |
 *     |
 *     |
 *     .----------------. (axes[2].x, axes[2].y)
 * (axes[1].x, axes[1].y)
*/

  cprof_screen_axes[0].x =
    cprof_screen_axes[1].x = left_margin ;
  cprof_screen_axes[2].x = cprof_width - RTMARGIN ;

/*
 * Boy, is this a kludge.  In order to be able to fix the
 * y axis, I have to be able to make sure that the number of
 * y ticks doesn't keep shifting around for no apparent reason.
 * This shifting around is due, it seems at the moment, to
 * rounding in generate_cprof_yticks(), in the generation of
 * cprof_ticks.plane[j].y.  I tried making things double, and
 * it made no difference.  So instead of trying to halt the
 * variation there, instead I'm making the axis a tiny bit
 * taller.
*/
  /*cprof_screen_axes[0].y = RTMARGIN - 5 ;*/
/* I'm not sure this makes a difference at the moment */

  cprof_screen_axes[0].y = RTMARGIN ;
  cprof_screen_axes[1].y =
    cprof_screen_axes[2].y = cprof_height - LBMARGIN ;
}

/*
 * Let's force a tick at every variable
 * Rewritten; dfs June 96
*/
void
generate_cprof_xticks(xgobidata *xg)
{
/*
 * nticks: number of ticks used by each column
 * This routine will take the ticks up to their planar
 * values -- that is, the locations of the x ticks in
 * plane coordinates.
*/
  int j;
  float tmpf, precis;

  cprof_ticks.nticks[0] = cprof_nselectedvars;
  cprof_xdeci = 0;

  for (j=0; j<cprof_nselectedvars; j++)
  {
    cprof_ticks.xticks[j] = j+1;

    if (j == NTICKS-1)
    {
      (void) fprintf(stderr, "warning: (generate_ticks) too many x ticks\n");
      cprof_ticks.nticks[0] = NTICKS;
      return;
    }
  }

/*
 * Drawn from scale_ticks()
*/
  precis = PRECISION1;

  for (j=0; j<cprof_ticks.nticks[0]; j++)
  {
    tmpf = -1.0 + 2.0*(cprof_ticks.xticks[j] - cprof_xlim.min)
        /(cprof_xlim.max - cprof_xlim.min);
    cprof_ticks.plane[j].x = (long) (precis * tmpf);
  }

}

void
generate_cprof_yticks(void)
{
/*
 * nticks: number of ticks used by each column
 * cprof_ytickdelta: increment between subsequent ticks
 * This routine will take the ticks up to their planar
 * values -- that is, the locations of the y ticks in
 * plane coordinates.
*/
  int j;
  float ftmp, precis, fdiff;
  float nicearg;

/*
 * Drawn from SetNiceRange()
*/
  cprof_ticks.nticks[1] = 4;

  nicearg = (cprof_ylim.max - cprof_ylim.min) /
            (float) (cprof_ticks.nticks[1] - 1) ;
  cprof_ytickdelta = NiceValue( nicearg );
  /* sometimes trouble occurs here, for no good reason */

  cprof_ynicelim.min = (float)
    floor((double) (cprof_ylim.min / cprof_ytickdelta)) *
    cprof_ytickdelta;
  cprof_ynicelim.max = (float)
    ceil((double) (cprof_ylim.max / cprof_ytickdelta)) *
    cprof_ytickdelta;

  /* add .01 for rounding */
  cprof_ticks.nticks[1] = 1 +
    (.01 + (cprof_ynicelim.max - cprof_ynicelim.min) / cprof_ytickdelta);

  cprof_ydeci = set_deci(cprof_ytickdelta);

/*
 * Drawn from generate_ticks()
*/
  cprof_ticks.yticks[0] = cprof_ynicelim.min;
  j = 1;
  while (cprof_ticks.yticks[j-1] + cprof_ytickdelta <=
     cprof_ylim.max + cprof_ytickdelta/2 )
  {
    cprof_ticks.yticks[j] = cprof_ticks.yticks[j-1] + cprof_ytickdelta ;
    if (j++ == NTICKS-1)
    {
      (void) fprintf(stderr, "warning: (generate_ticks) too many y ticks\n");
      return;
    }
  }

  cprof_ticks.nticks[1] = j;

/*
 * Drawn from scale_ticks()
*/
  precis = (float) PRECISION1;
  fdiff = cprof_ylim.max - cprof_ylim.min;

  for (j=0; j<cprof_ticks.nticks[1]; j++)
  {
    ftmp = -1.0 + 2.0 * (cprof_ticks.yticks[j] - cprof_ylim.min) / fdiff;
    cprof_ticks.plane[j].y = - (long) (precis * ftmp);
  }

}

void
convert_cprof_ticks(void)
{
  int j;
  long nx, ny;

  for (j=0; j<cprof_ticks.nticks[0]; j++)
  {
    nx = (cprof_ticks.plane[j].x + cprof_shift_wrld_x) * cprof_is.x ;
    cprof_ticks.screen[j].x = (int) (nx >> EXP1) ;
    cprof_ticks.screen[j].x += cprof_midx ;
  }

  for (j=0; j<cprof_ticks.nticks[1]; j++)
  {
    ny = (cprof_ticks.plane[j].y - cprof_shift_wrld_y) * cprof_is.y ;

    cprof_ticks.screen[j].y = (int) (ny >> EXP1) ;
    cprof_ticks.screen[j].y += cprof_midy ;
  }

}

void
build_cprof_tick_segs(void)
{
  int j;

/* build x tick segments */
  for (j=0; j<cprof_ticks.nticks[0]; j++)
  {
    cprof_xtick_segs[j].x1 = cprof_ticks.screen[j].x;
    cprof_xtick_segs[j].x2 = cprof_ticks.screen[j].x;
    cprof_xtick_segs[j].y1 = cprof_screen_axes[1].y;
    cprof_xtick_segs[j].y2 = cprof_screen_axes[1].y + 5;
  }

/* build y tick segments */
  for (j=0; j<cprof_ticks.nticks[1]; j++)
  {
    cprof_ytick_segs[j].x1 = cprof_screen_axes[1].x;
    cprof_ytick_segs[j].x2 = cprof_screen_axes[1].x - 5;
    cprof_ytick_segs[j].y1 = cprof_ticks.screen[j].y;
    cprof_ytick_segs[j].y2 = cprof_ticks.screen[j].y;
  }
}

void
find_cprof_tick_label(int deci, int tickn, float *ticknos, char *str)
{
  int length;
  float ftmp;
  int src, dest;

/*
 * deci:  number of decimal places in each tickdelta; one per column
 * ticknos: xticks or yticks, vector of ticklabel values
 * tickn: indec into ticknos array
*/
  if ((deci > 4 ||
       fabs((double) ticknos[tickn]) >= 10000.) &&
       ticknos[tickn] != 0.0)
  {
    number_length(ticknos[tickn], &length);
    ftmp = (float) ticknos[tickn]/pow((double)10, (double)(length-1));
    if (ftmp == (float)(int)ftmp)
      (void) sprintf (str, "%.0e", ticknos[tickn]);
    else
      (void) sprintf (str, "%.2e", ticknos[tickn]);
  }
  else
  {
    switch(deci)
    {
      case 0:
        (void) sprintf (str, "%3.0f", ticknos[tickn]);
        break;
      case 1:
        (void) sprintf (str, "%3.1f", ticknos[tickn]);
        break;
      case 2:
        (void) sprintf (str, "%3.2f", ticknos[tickn]);
        break;
      case 3:
        (void) sprintf (str, "%3.3f", ticknos[tickn]);
        break;
      case 4:
        (void) sprintf (str, "%3.4f", ticknos[tickn]);
        break;
    }
    /*
     * To get better placement, strip blanks.
    */
    for (src=0, dest=0; src<(INT(strlen(str))+1); src++)
      if (str[src] != ' ')
        str[dest++] = str[src];
  }
}

static int
find_cprof_ytick_offset(void)
{
  int j, offset = 0;

  for (j=0; j<cprof_ticks.nticks[1]; j++)
  {
    if (cprof_screen_axes[1].y < cprof_ticks.screen[j].y )
      offset++;
    else
      break;
  }
  return(offset);
}

static int
find_cprof_ytick_toomany(void)
{
  int j, toomany = 0;

  for (j=(cprof_ticks.nticks[1]-1); j>=0; j--)
  {
    if (cprof_screen_axes[0].y > cprof_ticks.screen[j].y )
      toomany++;
    else
      break;
  }
  return(toomany);
}

void
add_cprof_axes(xgobidata *xg)
{
  int j, offset, toomany;
  unsigned int width;
  XPoint vpnts[2];
  char str[64];

  if (!mono)
    XSetForeground(display, copy_GC, plotcolors.fg);
/*
 * Draw x axis
*/
  for (j=0; j<2; j++) {
    vpnts[j].x = cprof_screen_axes[j+1].x;
    vpnts[j].y = cprof_screen_axes[j+1].y;
  }
  if (cprof_doublebuffer)
    XDrawLines(display, cprof_pixmap, copy_GC, vpnts, 2, CoordModeOrigin);
  else
    XDrawLines(display, cprof_plot_window, copy_GC, vpnts, 2, CoordModeOrigin);

/*
 * Draw y axis
*/
  if (yaxis_style == SCALE_TOGETHER) {
    for (j=0; j<2; j++) {
      vpnts[j].x = cprof_screen_axes[j].x;
      vpnts[j].y = cprof_screen_axes[j].y;
    }
    if (cprof_doublebuffer)
      XDrawLines(display,
        cprof_pixmap, copy_GC, vpnts, 2, CoordModeOrigin);
    else
      XDrawLines(display,
        cprof_plot_window, copy_GC, vpnts, 2, CoordModeOrigin);
  }

/*
 * Draw x ticks
*/
  if (cprof_doublebuffer)
    XDrawSegments(display, cprof_pixmap, copy_GC,
    cprof_xtick_segs, cprof_ticks.nticks[0]);
  else
    XDrawSegments(display, cprof_plot_window, copy_GC,
      cprof_xtick_segs, cprof_ticks.nticks[0]);

  for (j=0; j<cprof_ticks.nticks[0]; j++)
  {
    sprintf(str, "%d", cprof_selectedvars[j]+1);

    /*
     * Arbitrary:  if there are more than 20 ticks, then
     * only draw every other tick label.  If there are
     * more than 30; draw every fourth.
    */
    if (cprof_ticks.nticks[0] > 30 && j%4 != 0)
      ;
    else if (cprof_ticks.nticks[0] <= 30 &&
             cprof_ticks.nticks[0] > 20 && j%2 != 0)
      ;
    else {
      width = XTextWidth(appdata.plotFont, str, strlen(str));
      if (cprof_doublebuffer)
        XDrawImageString(display, cprof_pixmap, copy_GC,
          (int) (cprof_ticks.screen[j].x - width/2),
          cprof_screen_axes[1].y + FONTHEIGHT(appdata.plotFont)+7,
          str, strlen(str));
      else
        XDrawImageString(display, cprof_plot_window, copy_GC,
          (int) (cprof_ticks.screen[j].x - width/2),
          cprof_screen_axes[1].y + FONTHEIGHT(appdata.plotFont)+7,
          str, strlen(str));
    }
  }

/*
 * Draw y ticks -- if the yaxis scaling style is appropriate.
*/
  if (yaxis_style == SCALE_TOGETHER) {
    offset = find_cprof_ytick_offset();
    toomany = find_cprof_ytick_toomany();

    if (cprof_doublebuffer)
      XDrawSegments(display, cprof_pixmap, copy_GC,
        cprof_ytick_segs + offset,
        cprof_ticks.nticks[1] - offset - toomany);
    else
      XDrawSegments(display, cprof_plot_window, copy_GC,
        cprof_ytick_segs + offset,
        cprof_ticks.nticks[1] - offset - toomany);
    /*
     * Simply skip the first tick if it's to the below the X axis
    */
    for (j=offset; j<(cprof_ticks.nticks[1] - toomany); j++)
    {
      find_cprof_tick_label(cprof_ydeci, j, cprof_ticks.yticks, str);
      width = XTextWidth(appdata.plotFont, str, strlen(str));
      if (cprof_doublebuffer)
        XDrawImageString(display, cprof_pixmap, copy_GC,
          (int) (cprof_screen_axes[1].x-width-7),
          cprof_ticks.screen[j].y + 5,
          str, strlen(str));
      else
        XDrawImageString(display, cprof_plot_window, copy_GC,
          (int) (cprof_screen_axes[1].x-width-7),
          cprof_ticks.screen[j].y + 5,
          str, strlen(str));
    }
  }
}

/*
 * End of ticks and axes section.
*/

static void
init_cprof_plane_x(void)
{
  int i;
  float min, max, rdiff, dtmp;
  float precis = PRECISION1;

/*
 * Create cprof_tform[].x
*/
  for (i=0; i<cprof_nselectedvars; i++)
    cprof_tform[i].x = (float) (i+1) ;

/*
 * Scale cprof_tform[].x into cprof_plane[].x
*/
  min = max = cprof_tform[0].x ;
  for (i=1; i<cprof_nselectedvars; i++)
  {
    if (cprof_tform[i].x < min)
      min = cprof_tform[i].x ;
    else if (cprof_tform[i].x > max)
      max = cprof_tform[i].x ;
  }
  adjust_limits(&min, &max);
  rdiff = max - min;
  for (i=0; i<cprof_nselectedvars; i++)  /* the minimum number of these */
  {
    dtmp = -1.0 + 2.0*(cprof_tform[i].x - min)/rdiff;
    cprof_plane[i].x = (long) (precis * dtmp);
  }
/*
 * For use in constructing ticks and axes.
*/
  cprof_xlim.min = min;
  cprof_xlim.max = max;
}

void
cprof_tform_to_plane(xgobidata *xg)
{
/*
 * Create cprof_tform[].y from tform_data[] and scale it
 * into cprof_plane[].y, the planar values.
*/
  int i, j, k, m;
  float ftmp;
  float precis = PRECISION1;
  float min, max;
  float rdiff;
  static int id = 0;

  /*
   * Since this routine is performed whenever the case profile
   * is about to be changed, this is probably a safe place
   * to perform this intialization.
  */
  if (parallel_coords) {
    nprofiles = (int *) &xg->nrows_in_plot;
    profile_ids = (int *) xg->rows_in_plot;
  } else {
    nprofiles = (int *) &xg->nsticky_ids;
    profile_ids = (int *) xg->sticky_ids;
  }

  if (xg->nearest_point != -1)
    id = xg->nearest_point;

/*
 * Create cprof_tform[].y.  The first cprof_nselectedvars positions
 * are occupied by nearest_point; the remainder by the values
 * corresponding to the sticky ids.
*/
  for (j=0; j<cprof_nselectedvars; j++) {
    i = cprof_selectedvars[j];
    cprof_tform[j].y = xg->tform_data[id][i] ;
  }

  for (i=0; i<*nprofiles; i++) {
    k = (i+1) * cprof_nselectedvars ;
    for (m=0; m<cprof_nselectedvars; m++)
    {
      j = cprof_selectedvars[m];
      cprof_tform[k + m].y = xg->tform_data[ profile_ids[i] ][j] ;
    }
  }

/*
 * Scale cprof_tform[].y into cprof_plane[].y
*/

  min = max = cprof_tform[0].y ;

  if (cprof_fix_ymax) {
    for (i=0; i<cprof_nselectedvars; i++) {
      k = cprof_selectedvars[i];
      if (xg->lim_tform[k].min < min)
        min = xg->lim_tform[k].min ;
      if (xg->lim_tform[k].max > max)
        max = xg->lim_tform[k].max ;
    }
  }
  else {
    for (i=1; i<(*nprofiles+1)*cprof_nselectedvars; i++) {
      if (cprof_tform[i].y < min)
        min = cprof_tform[i].y ;
      if (cprof_tform[i].y > max)
        max = cprof_tform[i].y ;
    }
  }
  adjust_limits(&min, &max);
  rdiff = max - min;

/*
 * Using xg->lim_tform[] has a problem:  If we have imputed but
 * not rescaled, then xg->lim_tform[] has not been updated.  Hmm.
*/

/*
 * Here tform is taken straight to the plane ... using min or
 * lim_tform[].min for scaling everything as widely as possible or
 * scaling to a common scale.
*/
  for (j=0; j<cprof_nselectedvars; j++)
  {
    i = cprof_selectedvars[j];

    if (yaxis_style == SCALE_SEPARATELY) {
      min = xg->lim_tform[ i ].min;
      rdiff = xg->lim_tform[ i ].max - min;
    }

    ftmp = -1.0 + 2.0*(cprof_tform[j].y - min)/rdiff;
    cprof_plane[j].y = - (long) (precis * ftmp) ;
    cprof_plane[j].y -= xg->jitter_data[id][i];
  }

  for (i=0; i<*nprofiles; i++) {
    k = (i+1) * cprof_nselectedvars ;
    for (m=0; m<cprof_nselectedvars; m++) {
      j = cprof_selectedvars[m];

      if (yaxis_style == SCALE_SEPARATELY) {
        min = xg->lim_tform[ j ].min;
        rdiff = xg->lim_tform[ j ].max - min;
      }

      ftmp = -1.0 + 2.0*(cprof_tform[k+m].y - min)/rdiff;
      cprof_plane[k+m].y = - (long) (precis * ftmp) ;
      cprof_plane[k+m].y -= xg->jitter_data[ profile_ids[i] ][j];

    }
  }

/*
 * For use in constructing ticks and axes.
 * Maybe putting the axis back in for the SCALE_SEPARATELY style
 * involves setting these values based on the the first
 * variable instead of the overall value.
*/
  cprof_ylim.min = min;
  cprof_ylim.max = max;
}

void
cprof_plane_to_screen(xgobidata *xg)
{
/*
 * Take the values in
 *  xg->world_data[nearest_point, sticky][] for the y variable
 *  a scaled up version of 1 to the number of columns for x
 * and scale them into cprof_screen for plotting.
*/
  int j, k;
  long nx, ny;
  

  /* Don't know where this 40 came from. */
  float cprof_scale_x = (FLOAT(cprof_width) - FLOAT(left_margin) - 40.) /
    FLOAT(cprof_width) ;
  /* Also subtract the right&top margin -- dfs 6/13/97 */
  float cprof_scale_y = (FLOAT(cprof_height) -
    FLOAT(LBMARGIN) - FLOAT(RTMARGIN) - 40.0) / FLOAT(cprof_height) ;

/* Why do I have a shift_wrld at all?  There's no scaling here */
  cprof_shift_wrld_x = FLOAT(left_margin) * PRECISION1 /
    (FLOAT(cprof_width) * FLOAT(cprof_scale_x)) ;
  cprof_shift_wrld_y = FLOAT(RTMARGIN) * PRECISION1 /
    (FLOAT(cprof_height) * FLOAT(cprof_scale_y)) ;

  cprof_midx = FLOAT(cprof_width)/2. ;
  cprof_midy = FLOAT(cprof_height)/2. ;

  cprof_is.x = (long) (FLOAT(cprof_width) * FLOAT(cprof_scale_x) / 2.);
  cprof_is.y = (long) (FLOAT(cprof_height) * FLOAT(cprof_scale_y) / 2.);

  for (j=0; j<(*nprofiles+1)*cprof_nselectedvars; j++)
  {
    /* Only have nselectedvars cprof_plane[].x values */
    k = j % cprof_nselectedvars;
    nx = (cprof_plane[k].x + cprof_shift_wrld_x) * cprof_is.x ;
    /* But we're building all the cprof_screen[].x values */
    cprof_screen[j].x = (int) (nx >> EXP1);
    cprof_screen[j].x += cprof_midx ;

    ny = (cprof_plane[j].y - cprof_shift_wrld_y) * cprof_is.y ;
    cprof_screen[j].y = (int) (ny >> EXP1);
    cprof_screen[j].y += cprof_midy ;
  }
}

void
build_cprof_connect_segs(xgobidata *xg)
{
  int i, j, k, var1, var2, id;

  nconnsegs = 0;
  if (xg->nearest_point != -1)
  {
    id = xg->nearest_point;
    for (i=0; i<cprof_nselectedvars-1; i++)
    {
      var1 = cprof_selectedvars[i];
      var2 = cprof_selectedvars[i+1];
      if (!plot_imputed_values &&
          (xg->is_missing[id][var1] || xg->is_missing[id][var2]))
        ;
      else
      {
        connsegs[nconnsegs].x1 = (short) cprof_screen[i].x ;
        connsegs[nconnsegs].y1 = (short) cprof_screen[i].y ;
        connsegs[nconnsegs].x2 = (short) cprof_screen[i+1].x ;
        connsegs[nconnsegs].y2 = (short) cprof_screen[i+1].y ;
        nconnsegs++;
      }
    }
  }

/*
 * The first cprof_nselectedvars positions of cprof_screen
 * are occupied by nearest_point; the remainder by the values
 * corresponding to the sticky ids.
*/

/*
 * This is pretty screwy:  I'm going to start the segments
 * for each new id at j*(nselectedvars-1):  that's because I need
 * to be able to find these boundaries when I determine what
 * color to use and draw each one separately.  Maybe these
 * should be figured and plotted one at a time ...
*/

  /* build the segments for the sticky guys */
  /* ... moved to cprof_plot_once */
}

static void
find_point_colors_used(xgobidata *xg)
{
  Boolean new_color;
  int i, k, m;

  if (mono) {
    ncolors_used = 1;
    colors_used[0] = plotcolors.fg;
  } else {

    /*
     * Loop once through xg->color_now[], collecting the colors currently
     * in use by sticky_ids into the colors_used[] vector.
    */
    ncolors_used = 0;
    for (i=0; i<*nprofiles; i++) {
      m = profile_ids[i];
      new_color = True;
      for (k=0; k<ncolors_used; k++) {
        if (colors_used[k] == xg->color_now[m]) {
          new_color = False;
          break;
        }
      }
      if (new_color) {
        colors_used[ncolors_used] = xg->color_now[m];
        (ncolors_used)++;
      }
    }

    if (ncolors_used == 0) {
      ncolors_used = 1;
      colors_used[0] = plotcolors.fg;
    }

    /*
     * Make sure that the current brushing color is
     * last in the list, so that it is drawn on top of
     * the pile of points.
    */
    for (k=0; k<(ncolors_used-1); k++) {
      if (colors_used[k] == xg->color_id) {
        colors_used[k] = colors_used[ncolors_used-1];
        colors_used[ncolors_used-1] = xg->color_id;
        break;
      }
    }
  }
}


void
cprof_plot_once(xg)
  xgobidata *xg;
{
  int i, j, sticky_id;
  unsigned int width;
  static int id = 0;
  int np, ns, nr_open, nr_filled, nc_open, nc_filled;
  XGCValues *gcv, gcv_inst;

/*
 * In order to make the plot update itself during brushing,
 * there are two lines at the end of brush_once().  This
 * is not pretty, since it is another violation of modularity.
 * Can this be handled in the event loop?
*/

  if (!mono && *nprofiles > 1)
    find_point_colors_used(xg);

  XGetGCValues(display, copy_GC, GCCapStyle|GCJoinStyle, &gcv_inst);
  gcv = &gcv_inst;

  if (cprof_doublebuffer)
    XFillRectangle(display, cprof_pixmap, clear_GC,
      0, 0, cprof_width, cprof_height );
  else
    XFillRectangle(display, cprof_plot_window, clear_GC,
      0, 0, cprof_width, cprof_height );

  if (xg->nearest_point != -1 || *nprofiles != 0)
  {
    id = xg->nearest_point;
    /* This needs to be done once for for the sticky and non-sticky guys */
    build_cprof_connect_segs(xg);

    /* If there's a nearest_point: ie, if the cursor is in the window .. */
    if (id != -1)
    {
      if (cprof_showpoints)
      {
        np = ns = nr_open = nr_filled = nc_open = nc_filled = 0;

        /* Build the glyphs for the nearest point */
        for (i=0; i<cprof_nselectedvars; i++)
          if (!plot_imputed_values &&
              xg->is_missing[id][ cprof_selectedvars[i] ])
          {
            /*fprintf(stderr, "not including (%d,%d)\n",*/
            /*id, cprof_selectedvars[i]);*/
          }
          else
          {
            /*fprintf(stderr, "including (%d,%d)\n",
              id, cprof_selectedvars[i]);*/
            build_glyph(xg, id, cprof_screen, i,
              points, &np,
              segs, &ns,
              open_rectangles, &nr_open,
              filled_rectangles, &nr_filled,
              open_circles, &nc_open,
              filled_circles, &nc_filled);
          }

        /*
         * Draw the glyphs for nearest point.
        */
        if (!mono)
          XSetForeground(display, copy_GC, xg->color_now[id]);

        if (np)
          if (cprof_doublebuffer)
            XDrawPoints(display, cprof_pixmap, copy_GC,
            points, np, CoordModeOrigin);
          else
            XDrawPoints(display, cprof_plot_window, copy_GC,
              points, np, CoordModeOrigin);
        else if (ns)
          if (cprof_doublebuffer)
            XDrawSegments(display, cprof_pixmap, copy_GC,
              segs, ns);
          else
            XDrawSegments(display, cprof_plot_window, copy_GC,
              segs, ns);
        else if (nr_open)
          if (cprof_doublebuffer)
            XDrawRectangles(display, cprof_pixmap, copy_GC,
              open_rectangles, nr_open);
          else
            XDrawRectangles(display, cprof_plot_window, copy_GC,
              open_rectangles, nr_open);
        else if (nr_filled)
        {
          if (cprof_doublebuffer) {
            XDrawRectangles(display, cprof_pixmap, copy_GC,
              filled_rectangles, nr_filled);
            XFillRectangles(display, cprof_pixmap, copy_GC,
              filled_rectangles, nr_filled);
          } else {
            XDrawRectangles(display, cprof_plot_window, copy_GC,
              filled_rectangles, nr_filled);
            XFillRectangles(display, cprof_plot_window, copy_GC,
              filled_rectangles, nr_filled);
          }
        }
        else if (nc_open)
          if (cprof_doublebuffer)
            XDrawArcs(display, cprof_pixmap, copy_GC,
              open_circles, nc_open);
          else
            XDrawArcs(display, cprof_plot_window, copy_GC,
              open_circles, nc_open);
        else if (nc_filled)
        {
          if (cprof_doublebuffer) {
            XDrawArcs(display, cprof_pixmap, copy_GC,
              filled_circles, nc_filled);
            XFillArcs(display, cprof_pixmap, copy_GC,
              filled_circles, nc_filled);
          } else {
            XDrawArcs(display, cprof_plot_window, copy_GC,
              filled_circles, nc_filled);
            XFillArcs(display, cprof_plot_window, copy_GC,
              filled_circles, nc_filled);
          }
        }
      }
      if (cprof_showlines)
      {
        /*
         * Draw the connected lines for nearest_point; use plotcolors.fg.
        */

        /* Thin lines for sticky guys; thick for current guy */
        width = 5;

        if (!mono)
          XSetForeground(display, copy_GC, xg->color_now[id]);
        XSetLineAttributes(display, copy_GC, width, LineDoubleDash,
          gcv->cap_style, gcv->join_style);

        if (cprof_doublebuffer)
          XDrawSegments(display, cprof_pixmap, copy_GC,
          connsegs, nconnsegs);
        else
          XDrawSegments(display, cprof_plot_window, copy_GC,
            connsegs, nconnsegs);

        XSetLineAttributes(display, copy_GC, 0, LineSolid,
          gcv->cap_style, gcv->join_style);
      }
    }

/*** Sticky guys, or parallel coordinates plot ***/
/*
 * First draw the points:  loop over colors, gathering vectors
 * of glyphs, and do all the drawing for each glyph type for
 * each color.
*/

    if (cprof_showpoints)
    {
      int k;
      long current_color;

      for (k=0; k<ncolors_used; k++) {
        np = ns = nr_open = nr_filled = nc_open = nc_filled = 0;
        current_color = colors_used[k];

        XSetForeground(display, copy_GC, current_color);
      
        for (j=0; j<*nprofiles; j++) {
          sticky_id = *(profile_ids+j);
          if (xg->color_now[sticky_id] == current_color) {

            if (!xg->erased[sticky_id]) {
              for (i=0; i<cprof_nselectedvars; i++) {
                if (!plot_imputed_values && xg->missing_values_present &&
                     xg->is_missing[sticky_id][ cprof_selectedvars[i] ])
                  ;
                else
                  build_glyph(xg,
                    sticky_id, &cprof_screen[cprof_nselectedvars*(j+1)], i,
                    points, &np,
                    segs, &ns,
                    open_rectangles, &nr_open,
                    filled_rectangles, &nr_filled,
                    open_circles, &nc_open,
                    filled_circles, &nc_filled);
              }
            }
          }
        }

        if (np)
          if (cprof_doublebuffer)
            XDrawPoints(display, cprof_pixmap, copy_GC,
              points, np, CoordModeOrigin);
          else
            XDrawPoints(display, cprof_plot_window, copy_GC,
              points, np, CoordModeOrigin);
        if (ns)
          if (cprof_doublebuffer)
            XDrawSegments(display, cprof_pixmap, copy_GC,
              segs, ns);
          else
            XDrawSegments(display, cprof_plot_window, copy_GC,
              segs, ns);
        if (nr_open)
          if (cprof_doublebuffer)
            XDrawRectangles(display, cprof_pixmap, copy_GC,
              open_rectangles, nr_open);
          else
            XDrawRectangles(display, cprof_plot_window, copy_GC,
              open_rectangles, nr_open);
        if (nr_filled) {
          if (cprof_doublebuffer) {
            XDrawRectangles(display, cprof_pixmap, copy_GC,
              filled_rectangles, nr_filled);
            XFillRectangles(display, cprof_pixmap, copy_GC,
              filled_rectangles, nr_filled);
          } else {
            XDrawRectangles(display, cprof_plot_window, copy_GC,
              filled_rectangles, nr_filled);
            XFillRectangles(display, cprof_plot_window, copy_GC,
              filled_rectangles, nr_filled);
          }
        }
        if (nc_open) {
          /* work-around for X bug; see plot_once.c */
          int c = 0;
          while (nc_open > MAXARCS) {
            if (cprof_doublebuffer)
              XDrawArcs(display, cprof_pixmap, copy_GC,
                &open_circles[c*MAXARCS], MAXARCS);
            else
              XDrawArcs(display, cprof_plot_window, copy_GC,
                &open_circles[c*MAXARCS], MAXARCS);
            nc_open -= MAXARCS;
            c++;
          }
          if (cprof_doublebuffer)
            XDrawArcs(display, cprof_pixmap, copy_GC,
              &open_circles[c*MAXARCS], nc_open);
          else
            XDrawArcs(display, cprof_plot_window, copy_GC,
              &open_circles[c*MAXARCS], nc_open);
        }
        if (nc_filled) {
          /* work-around for X bug; see plot_once.c */
          int c = 0;
          while (nc_filled > MAXARCS) {
            if (cprof_doublebuffer) {
              XDrawArcs(display, cprof_pixmap, copy_GC,
                &filled_circles[c*MAXARCS], MAXARCS);
              XFillArcs(display, cprof_pixmap, copy_GC,
                &filled_circles[c*MAXARCS], MAXARCS);
            } else {
              XDrawArcs(display, cprof_plot_window, copy_GC,
                &filled_circles[c*MAXARCS], MAXARCS);
              XFillArcs(display, cprof_plot_window, copy_GC,
                &filled_circles[c*MAXARCS], MAXARCS);
            }
            nc_filled -= MAXARCS;
            c++;
          }
          if (cprof_doublebuffer) {
            XDrawArcs(display, cprof_pixmap, copy_GC,
              &filled_circles[c*MAXARCS], nc_filled);
            XFillArcs(display, cprof_pixmap, copy_GC,
              &filled_circles[c*MAXARCS], nc_filled);
          } else {
            XDrawArcs(display, cprof_plot_window, copy_GC,
              &filled_circles[c*MAXARCS], nc_filled);
            XFillArcs(display, cprof_plot_window, copy_GC,
              &filled_circles[c*MAXARCS], nc_filled);
          }
        }
      }
    }

/*
 * Then draw the line segments.  Loop over colors again, gathering
 * all the segments to be drawn in a particular color.
*/
    if (cprof_showlines)
    {
      int k, nv, var1, var2, jsegs;
      long current_color;

      /*
       * Draw the connected lines for the sticky ids; use thin lines.
      */
      width = 1;
      XSetLineAttributes(display, copy_GC, width, LineSolid,
        gcv->cap_style, gcv->join_style);

      for (k=0; k<ncolors_used; k++) {
        current_color = colors_used[k];
        XSetForeground(display, copy_GC, current_color);

        nv = cprof_nselectedvars;
        jsegs = 0;
        for (j=0; j<*nprofiles; j++) {
          id = *(profile_ids+j);
          if (xg->color_now[id] != current_color || xg->erased[id]) {
            nv += (cprof_nselectedvars-1);
          } else {
            for (i=0; i<cprof_nselectedvars-1; i++) {
              var1 = cprof_selectedvars[i];
              var2 = cprof_selectedvars[i+1];
              if (!plot_imputed_values && xg->missing_values_present &&
                  (xg->is_missing[id][var1] || xg->is_missing[id][var2]))
                ;
              else {
                stickysegs[ jsegs ].x1 = (short) cprof_screen[i].x ;
                stickysegs[ jsegs ].y1 = (short) cprof_screen[nv].y ;
                stickysegs[ jsegs ].x2 = (short) cprof_screen[i+1].x ;
                stickysegs[ jsegs ].y2 = (short) cprof_screen[nv+1].y ;
                jsegs++;
              }
              nv++;
            }
          }
          nv++;  /* ok */
        }

        if (cprof_doublebuffer)
          XDrawSegments(display, cprof_pixmap, copy_GC,
            stickysegs, jsegs);
        else
          XDrawSegments(display, cprof_plot_window, copy_GC,
            stickysegs, jsegs);
      }

      /* back to the usual line */
      XSetLineAttributes(display, copy_GC, 0, LineSolid,
        gcv->cap_style, gcv->join_style);
    }
  }
/*** end of section needing rewrite ***/

  if (xg->nearest_point != -1 || *nprofiles != 0) {
    build_cprof_tick_segs();
    add_cprof_axes(xg);
  }

  if (cprof_doublebuffer)
    XCopyArea(display, cprof_pixmap, cprof_plot_window, copy_GC,
      0, 0, cprof_width, cprof_height, 0, 0 );
}

void
update_cprof_selectedvars(xg)
  xgobidata *xg;
{
  int j, k = 0;

  for (j=0; j<xg->ncols_used; j++)
    if (xg->selectedvars[j])
      cprof_selectedvars[k++] = j ;

  cprof_nselectedvars = k;
}

void
update_cprof_plot(xg)
  xgobidata *xg;
{
/*
 * This is called when a variable transformation occurs,
 * when nselectedvars changes, when the passive window
 * receives new data, and so forth.
*/
  int j;
  char str[128];
  Dimension width;

  cprof_tform_to_plane(xg);
  cprof_plane_to_screen(xg);
  /*
   * Calculate the ticks and axes for the
   * y axis; scale ticks and axes.
  */
  generate_cprof_yticks();
  convert_cprof_ticks();

  /*
   * Check the length of the y axis labels.  If the length
   * of any axis plus 5 for the tick length plus 2 for breathing
   * room is greater than the margin, then reinitialize the axes.
  */
  width = 0;
  for (j=0; j<cprof_ticks.nticks[1]; j++)
  {
    find_cprof_tick_label(cprof_ydeci, j, cprof_ticks.yticks, str);
    width = MAX(width,
      ((Dimension)XTextWidth(appdata.plotFont, str, strlen(str))));
  }
  width = width+5+2 ;
  if (width > left_margin)
  {
    left_margin = width;
    make_cprof_screen_axes();
    cprof_plane_to_screen(xg);
    convert_cprof_ticks();
  }

  cprof_plot_once(xg);
}

void
realloc_tform(xg)
  xgobidata *xg;
{
  int npts;

  if (parallel_coords)
    npts = xg->nrows_in_plot+1;
  else
    npts = xg->nsticky_ids+1;


  cprof_tform = (fcoords *) XtRealloc( (XtPointer) cprof_tform,
    (Cardinal) (npts * xg->ncols_used * sizeof(fcoords)));
  cprof_plane = (lcoords *) XtRealloc( (XtPointer) cprof_plane,
    (Cardinal) (npts * xg->ncols_used * sizeof(lcoords)));
  cprof_screen = (icoords *) XtRealloc( (XtPointer) cprof_screen,
    (Cardinal) (npts * xg->ncols_used * sizeof(icoords)));

  points = (XPoint *) XtRealloc( (XtPointer) points,
    (Cardinal) (npts * xg->ncols_used) * sizeof(XPoint));
  segs = (XSegment *) XtRealloc( (XtPointer) segs,
    (Cardinal) (2 * npts * xg->ncols_used) * sizeof(XSegment));
  open_rectangles = (XRectangle *) XtRealloc( (XtPointer) open_rectangles,
    (Cardinal) (npts * xg->ncols_used) * sizeof(XRectangle));
  filled_rectangles = (XRectangle *) XtRealloc( (XtPointer) filled_rectangles,
    (Cardinal) (npts * xg->ncols_used) * sizeof(XRectangle));
  open_circles = (XArc *) XtRealloc( (XtPointer) open_circles,
    (Cardinal) (npts * xg->ncols_used) * sizeof(XArc));
  filled_circles = (XArc *) XtRealloc( (XtPointer) filled_circles,
    (Cardinal) (npts * xg->ncols_used) * sizeof(XArc));

  connsegs = (XSegment *) XtRealloc( (XtPointer) connsegs,
    (Cardinal) (xg->ncols_used-1) * sizeof(XSegment));
  stickysegs = (XSegment *) XtRealloc( (XtPointer) stickysegs,
    (Cardinal) (npts * (xg->ncols_used-1)) * sizeof(XSegment));
}

/* ARGSUSED */
XtCallbackProc
cprof_resize_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  XtVaGetValues(cprof_plot_wksp,
    XtNwidth, &cprof_width,
    XtNheight, &cprof_height,
    NULL);

/*  Allocate, and free it even if it isn't going to be used */
  XFreePixmap(display, cprof_pixmap);
  cprof_pixmap = XCreatePixmap(display, cprof_plot_window,
    cprof_width, cprof_height, depth);

  if (cprof_doublebuffer)
    XFillRectangle(display, cprof_pixmap, clear_GC,
      0, 0, cprof_width, cprof_height );
  else
    XFillRectangle(display, cprof_plot_window, clear_GC,
      0, 0, cprof_width, cprof_height );

  cprof_tform_to_plane(xg);
  cprof_plane_to_screen(xg);

  make_cprof_screen_axes();
  generate_cprof_yticks();
  convert_cprof_ticks();

/*
  cprof_plot_once(xg);
*/
}

/* ARGSUSED */
XtEventHandler
cprof_expose_cback(w, xg, evnt, cont)
  Widget w;
  xgobidata *xg;
  XEvent *evnt;
  Boolean *cont;
{
  if (evnt->xexpose.count == 0)  /* Compress expose events */
  {
    cprof_tform_to_plane(xg);
    cprof_plane_to_screen(xg);
    cprof_plot_once(xg);
  }
}

void
setFixYScale()
{
/* 
 * The 'Fix Y scale' button only makes sense if SCALE_TOGETHER
 * is True and 'Show all profiles', ie parallel_coordinates,
 * is False.
*/
  if (parallel_coords == False && yaxis_style == SCALE_TOGETHER)
    XtSetSensitive(cprof_cmd[1], True);
  else {
    XtSetSensitive(cprof_cmd[1], False);

    /* If all profiles are plotted, the scale <is> fixed. */
    if (parallel_coords)
      XtVaSetValues(cprof_cmd[1], XtNstate, True, NULL);
  }
}

/* ARGSUSED */
XtCallbackProc
parallel_coords_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  parallel_coords = !parallel_coords;

/* For Andreas, trying to leave this sensitive */
  /*setFixYScale();*/

  realloc_tform(xg);
  update_cprof_plot(xg);
}

/* ARGSUSED */
XtCallbackProc
cprof_fix_ymax_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  cprof_fix_ymax = !cprof_fix_ymax;
  update_cprof_plot(xg);
}

/* ARGSUSED */
static XtCallbackProc
plot_imputed_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  plot_imputed_values = !plot_imputed_values;
  update_cprof_plot(xg);
}

/* ARGSUSED */
static XtCallbackProc
cprof_close_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  XtPopdown(cprof_plot_shell);

  xg->is_cprof_plotting = False;

  XtDestroyWidget( cprof_plot_shell );
  XFreePixmap( display, cprof_pixmap );
  XtFree( (XtPointer) cprof_tform );
  XtFree( (XtPointer) cprof_plane );
  XtFree( (XtPointer) cprof_screen );
  XtFree( (XtPointer) segs );
  XtFree( (XtPointer) connsegs );
  XtFree( (XtPointer) stickysegs );
}

void
set_parcoord_link_menu_marks(xgobidata *xg)
{
  if (xg->link_cprof_plotting)
    XtVaSetValues(link_menu_btn[0],
      XtNleftBitmap, (Pixmap) menu_mark,
      NULL);
  else
    XtVaSetValues(link_menu_btn[0],
      XtNleftBitmap, (Pixmap) None,
      NULL);
}

/* ARGSUSED */
XtCallbackProc
parcoord_link_menu_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  int btn;

  for (btn=0; btn<1; btn++)
    if (link_menu_btn[btn] == w)
      break;

  switch (btn) {
    case 0 :
      xg->link_cprof_plotting = !xg->link_cprof_plotting;
      break;
  }
  set_parcoord_link_menu_marks(xg);
}

void
make_parcoord_link_menu(xgobidata *xg, Widget parent)
{
  int k;

  static char *link_menu_name[] = {
    "Link par coords plot",
  };

  link_menu_cmd = XtVaCreateManagedWidget("LinkParCoord",
    menuButtonWidgetClass, parent,
    XtNlabel, (String) "Link",
    XtNmenuName, (String) "Menu",
    NULL);
  if (mono) set_mono(link_menu_cmd);
  add_menupb_help(&xg->nhelpids.menupb,
    link_menu_cmd, "IdentifyLink");

  link_menu = XtVaCreatePopupShell("Menu",
    simpleMenuWidgetClass, link_menu_cmd,
    XtNinput, True,
    NULL);
  if (mono) set_mono(link_menu);

  for (k=0; k<1; k++)
  {
    link_menu_btn[k] = XtVaCreateWidget("Command",
      smeBSBObjectClass, link_menu,
      XtNleftMargin, (Dimension) 24,
      XtNleftBitmap, menu_mark,
      XtNlabel, (String) link_menu_name[k],
      NULL);
    if (mono) set_mono(link_menu_btn[k]);

    XtAddCallback(link_menu_btn[k], XtNcallback,
      (XtCallbackProc) parcoord_link_menu_cback, (XtPointer) xg);
  }

  XtManageChildren(link_menu_btn, 1);
}

static void
set_parcoord_display_menu_marks(xgobidata *xg)
{
  int k;
  for (k=0; k<NDISPLAYBTNS; k++)
    XtVaSetValues(display_menu_btn[k],
      XtNleftBitmap, (Pixmap) None,
      NULL);

  if (cprof_showpoints)
    XtVaSetValues(display_menu_btn[SHOWPOINTS],
      XtNleftBitmap, (Pixmap) menu_mark,
      NULL);
  if (cprof_showlines)
    XtVaSetValues(display_menu_btn[SHOWLINES],
      XtNleftBitmap, (Pixmap) menu_mark,
      NULL);
  if (cprof_doublebuffer)
    XtVaSetValues(display_menu_btn[DOUBLEBUFFER],
      XtNleftBitmap, (Pixmap) menu_mark,
      NULL);
}

/* ARGSUSED */
static XtCallbackProc
display_menu_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  int btn;

  for (btn=0; btn<NDISPLAYBTNS; btn++)
    if (display_menu_btn[btn] == w)
      break;

  switch (btn) {
    case SHOWPOINTS :
      cprof_showpoints = !cprof_showpoints;
      break;
    case SHOWLINES :
      cprof_showlines = !cprof_showlines;
      break;
    case DOUBLEBUFFER :
      cprof_doublebuffer = !cprof_doublebuffer;
      break;
  }
  cprof_plot_once(xg);
  set_parcoord_display_menu_marks(xg);
}


static void
make_display_menu(xgobidata *xg, Widget parent)
{
  int k;

  static char *display_menu_name[] = {
    "Plot the points",
    "Plot the lines",
    "Use doublebuffering",
  };

  display_menu_cmd = XtVaCreateManagedWidget("Command",
    menuButtonWidgetClass, parent,
    XtNlabel, (String) "Display",
    XtNmenuName, (String) "Menu",
    NULL);
  if (mono) set_mono(display_menu_cmd);
  add_menupb_help(&xg->nhelpids.menupb,
    display_menu_cmd, "DisplayOptions");

  display_menu = XtVaCreatePopupShell("Menu",
    simpleMenuWidgetClass, display_menu_cmd,
    XtNinput, True,
    NULL);
  if (mono) set_mono(display_menu);

  for (k=0; k<NDISPLAYBTNS; k++) {
    display_menu_btn[k] = XtVaCreateWidget("Command",
      smeBSBObjectClass, display_menu,
      XtNleftMargin, (Dimension) 24,
      XtNleftBitmap, menu_mark,
      XtNlabel, (String) display_menu_name[k],
      NULL);
    if (mono) set_mono(display_menu_btn[k]);

    XtAddCallback(display_menu_btn[k], XtNcallback,
      (XtCallbackProc) display_menu_cback, (XtPointer) xg);
  }

  XtManageChildren(display_menu_btn, NDISPLAYBTNS);
}

static void
set_parcoord_yaxstyle_menu_marks(xgobidata *xg)
{
  int k;
  for (k=0; k<NYAXISBTNS; k++)
    XtVaSetValues(yaxstyle_menu_btn[k],
      XtNleftBitmap, (Pixmap) None,
      NULL);

  if (yaxis_style == SCALE_TOGETHER)
    XtVaSetValues(yaxstyle_menu_btn[SCALE_TOGETHER],
      XtNleftBitmap, (Pixmap) menu_mark,
      NULL);
  else if (yaxis_style == SCALE_SEPARATELY)
    XtVaSetValues(yaxstyle_menu_btn[SCALE_SEPARATELY],
      XtNleftBitmap, (Pixmap) menu_mark,
      NULL);
}

/* ARGSUSED */
static XtCallbackProc
parcoord_yaxstyle_menu_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  int btn;

  for (btn=0; btn<NYAXISBTNS; btn++)
    if (yaxstyle_menu_btn[btn] == w)
      break;

  yaxis_style = btn;
  set_parcoord_yaxstyle_menu_marks(xg);

/* For Andreas, trying to leave this sensitive */
  /*setFixYScale();*/

  update_cprof_plot(xg);
}

void
make_parcoord_yaxstyle_menu(xgobidata *xg, Widget parent)
{
  int i, k;
  int nvgroups = numvargroups(xg);
  char lab[64];

  static char *yaxstyle_menu_name[] = {
    "Scale variables together",
    "Scale variables separately",
  };
  static char *yaxstyle_menu_name_vgr[] = {
    "Scale variable groups together",
    "Scale variable groups separately",
  };


  yaxstyle_menu_cmd = XtVaCreateManagedWidget("PCoordYAxStyle",
    menuButtonWidgetClass, parent,
    XtNlabel, (String) "Y scale style",
    XtNmenuName, (String) "Menu",
    NULL);
  if (mono) set_mono(yaxstyle_menu_cmd);
  add_menupb_help(&xg->nhelpids.menupb,
    yaxstyle_menu_cmd, "PCoordYAxStyle");

  yaxstyle_menu = XtVaCreatePopupShell("Menu",
    simpleMenuWidgetClass, yaxstyle_menu_cmd,
    XtNinput, True,
    NULL);
  if (mono) set_mono(yaxstyle_menu);

  for (k=0; k<NYAXISBTNS; k++) {
    /* If vgroups are present, use a slightly different name as a clue */
    if (nvgroups == xg->ncols_used)
      sprintf(lab, yaxstyle_menu_name[k]);
    else
      sprintf(lab, yaxstyle_menu_name_vgr[k]);

    yaxstyle_menu_btn[k] = XtVaCreateWidget("Command",
      smeBSBObjectClass, yaxstyle_menu,
      XtNleftMargin, (Dimension) 24,
      XtNleftBitmap, menu_mark,
      /*XtNlabel, (String) yaxstyle_name[k],*/
      XtNlabel, (String) lab,
      NULL);
    if (mono) set_mono(yaxstyle_menu_btn[k]);

    XtAddCallback(yaxstyle_menu_btn[k], XtNcallback,
      (XtCallbackProc) parcoord_yaxstyle_menu_cback, (XtPointer) xg);
  }

  XtManageChildren(yaxstyle_menu_btn, NYAXISBTNS);

  set_parcoord_yaxstyle_menu_marks(xg);
}


/* ARGSUSED */
XtCallbackProc
cprof_plot_cback(w, xg, callback_data)
  Widget w;
  xgobidata *xg;
  XtPointer callback_data;
{
  static Boolean initd = False;
  String main_title;
  static char cp_title[256];
  Widget dform;

  if (!xg->is_cprof_plotting)
  {
    int npts;
    Widget cpaned, cprof_close, panelform;

    /*
     * Weird behavior here:  if I free this and reallocate it,
     * the title does not reappear.
    */
    if (!initd) {
      main_title = XtMalloc((Cardinal) 132 * sizeof(char));
      XtVaGetValues(xg->shell, XtNtitle, &main_title, NULL);
      sprintf(cp_title, "%s, parallel coordinates", main_title);
      XtFree(main_title);
      initd = True;
    }

    if (parallel_coords)
      npts = xg->nrows_in_plot + 1;
    else
      npts = xg->nsticky_ids + 1;

    /*
     * Allocate enough space to plot the points and lines
     * for all the sticky ids for the current id.
    */
    cprof_tform = (fcoords *) XtMalloc( (Cardinal)
      npts * xg->ncols_used * sizeof(fcoords));
    cprof_plane = (lcoords *) XtMalloc( (Cardinal)
      npts * xg->ncols_used * sizeof(lcoords));
    cprof_screen = (icoords *) XtMalloc( (Cardinal)
      npts * xg->ncols_used * sizeof(icoords));

    points = (XPoint *) XtMalloc( (Cardinal)
      npts * xg->ncols_used * sizeof(XPoint));
    segs = (XSegment *) XtMalloc( (Cardinal)
      2 * npts * xg->ncols_used * sizeof(XSegment));
    open_rectangles = (XRectangle *) XtMalloc( (Cardinal)
      npts * xg->ncols_used * sizeof(XRectangle));
    filled_rectangles = (XRectangle *) XtMalloc( (Cardinal)
      npts * xg->ncols_used * sizeof(XRectangle));
    open_circles = (XArc *) XtMalloc( (Cardinal)
      npts * xg->ncols_used * sizeof(XArc));
    filled_circles = (XArc *) XtMalloc( (Cardinal)
      npts * xg->ncols_used * sizeof(XArc));

    connsegs = (XSegment *) XtMalloc( (Cardinal)
      (xg->ncols_used-1) * sizeof(XSegment));
    stickysegs = (XSegment *) XtMalloc( (Cardinal)
      npts * (xg->ncols_used-1) * sizeof(XSegment));

    cprof_width = 400;
    cprof_height = 200;

    cprof_plot_shell = XtVaCreatePopupShell("CaseProfile",
      topLevelShellWidgetClass, xg->shell,
      XtNtitle, (String) cp_title,
      XtNiconName, "XGobi: ParCoords",
      NULL);
    if (mono) set_mono(cprof_plot_shell);

    cprof_form = XtVaCreateManagedWidget("Form1",
      formWidgetClass, cprof_plot_shell,
      NULL);
    if (mono) set_mono(cprof_form);

    panelform = XtVaCreateManagedWidget("Form1",
      formWidgetClass, cprof_form,
      XtNvertDistance, 5,
      XtNhorizDistance, 5,
      XtNtop, (XtEdgeType) XtChainTop,
      XtNleft, (XtEdgeType) XtChainLeft,
      XtNright, (XtEdgeType) XtChainLeft,
      XtNbottom, (XtEdgeType) XtChainTop,
      NULL);
    if (mono) set_mono(cprof_form);
    /*
     * Create a paned widget so the 'Click here ...'
     * can be all across the bottom.
    */
    cpaned = XtVaCreateManagedWidget("CProfPaned",
      panedWidgetClass, panelform,
      XtNorientation, (XtOrientation) XtorientVertical,
      XtNright, (XtEdgeType) XtChainLeft,
      XtNbottom, (XtEdgeType) XtChainTop,
      XtNinternalBorderWidth, 2,
      NULL);

    cprof_panel = XtVaCreateManagedWidget("CProfPanel",
      boxWidgetClass, cpaned,
      NULL);
    if (mono) set_mono(cprof_panel);

    cprof_cmd[0] = (Widget) CreateToggle(xg, "Show all profiles",
      True, (Widget) NULL, (Widget) NULL, (Widget) NULL,
      parallel_coords, ANY_OF_MANY,
      cprof_panel, "CProfPlot");
    XtAddCallback(cprof_cmd[0], XtNcallback,
      (XtCallbackProc) parallel_coords_cback, (XtPointer) xg);

    make_display_menu(xg, cprof_panel);
    set_parcoord_display_menu_marks(xg);
    make_parcoord_link_menu(xg, cprof_panel);
    make_parcoord_yaxstyle_menu(xg, cprof_panel);

    cprof_cmd[1] = (Widget) CreateToggle(xg, "Fix Y scale",
      True, (Widget) NULL, (Widget) NULL, (Widget) NULL,
      cprof_fix_ymax, ANY_OF_MANY,
      cprof_panel, "CProfPlot");
    XtAddCallback(cprof_cmd[1], XtNcallback,
      (XtCallbackProc) cprof_fix_ymax_cback, (XtPointer) xg);

    /*
     * Let this button insensitive and unhighlighted for the
     * missing values xgobi
    */
    if (xg->is_missing_values_xgobi) {
      cprof_cmd[2] = (Widget) CreateToggle(xg, "Display missings",
        False, (Widget) NULL, (Widget) NULL, (Widget) NULL,
        False, ANY_OF_MANY,
        cprof_panel, "CProfPlot");
    } else {
      cprof_cmd[2] = (Widget) CreateToggle(xg, "Display missings",
        /* the button is insensitive if there are no missings */
        xg->missing_values_present,
        (Widget) NULL, (Widget) NULL, (Widget) NULL,
        plot_imputed_values, ANY_OF_MANY,
        cprof_panel, "CProfPlot");
      XtAddCallback(cprof_cmd[2], XtNcallback,
        (XtCallbackProc) plot_imputed_cback, (XtPointer) xg);
    }

    XtManageChildren(cprof_cmd, 3);

    /* Add a form so that the dismiss button can be farther
       away from the 'display missings' button
    */
    dform = XtVaCreateManagedWidget("Form",
      formWidgetClass, cpaned,
      XtNshowGrip, (Boolean) False,
      XtNskipAdjust, (Boolean) True,
      NULL);
    if (mono) set_mono(dform);

    cprof_close = XtVaCreateManagedWidget("Close",
      commandWidgetClass, dform,
      XtNlabel, (String) "Click here to dismiss",
      XtNvertDistance, 10,
      XtNleft, (XtEdgeType) XtChainLeft,
      XtNright, (XtEdgeType) XtChainRight,
      NULL);
    if (mono) set_mono(cprof_close);
    XtAddCallback(cprof_close, XtNcallback,
      (XtCallbackProc) cprof_close_cback, (XtPointer) xg);

    cprof_plot_wksp = XtVaCreateManagedWidget("CaseProfile",
      labelWidgetClass, cprof_form,
      XtNlabel, (String) "",
      XtNforeground, (Pixel) plotcolors.fg,
      XtNbackground, (Pixel) plotcolors.bg,
      XtNwidth, (Dimension) cprof_width,
      /*XtNfromHoriz, (Widget) cpaned,*/
      XtNfromHoriz, (Widget) panelform,
      XtNheight, (Dimension) cprof_height,
      XtNleft, (XtEdgeType) XtChainLeft,
      XtNright, (XtEdgeType) XtChainRight,
      XtNtop, (XtEdgeType) XtChainTop,
      XtNbottom, (XtEdgeType) XtChainBottom,
      NULL);
    if (mono) set_mono(cprof_plot_wksp);

    XtAddEventHandler(cprof_plot_wksp, ExposureMask,
      FALSE, (XtEventHandler) cprof_expose_cback, (XtPointer) xg);
    XtAddEventHandler(cprof_plot_wksp, StructureNotifyMask,
      FALSE, (XtEventHandler) cprof_resize_cback, (XtPointer) xg);

    XtPopup(cprof_plot_shell, XtGrabNone);
    XRaiseWindow(display, XtWindow(cprof_plot_shell));

    cprof_plot_window = XtWindow( cprof_plot_wksp );
    cprof_pixmap = XCreatePixmap(display, cprof_plot_window,
      cprof_width, cprof_height, depth);

    update_cprof_selectedvars(xg);

    /*
     * The cprof_plane[].x values don't change, so just
     * set them up here.  The same goes for the x ticks,
     * and the axes only need to be initialized once.
    */
    init_cprof_plane_x();

    make_cprof_screen_axes();
    generate_cprof_xticks(xg);

    xg->is_cprof_plotting = True;
  }
  else
  {
    XtPopup(cprof_plot_shell, XtGrabNone);
    XRaiseWindow(display, XtWindow(cprof_plot_shell));
  }
}

void
passive_update_cprof_plot(xg)
  xgobidata *xg;
{
/*
 * This is called when the passive window receives new data.
*/
  if (xg->is_cprof_plotting && xg->link_cprof_plotting)
  {
    realloc_tform(xg);
    update_cprof_plot(xg);
  }
}

void
reset_nvars_cprof_plot(xg)
  xgobidata *xg;
{
/*
 * If case profile plotting when an additional variable is
 * added or changed, especially the group variable, then do this:
*/
  if (xg->is_cprof_plotting)
  {
    /*
     * This isn't always necessary, but I don't know
     * how to tell when it is, so I'll do it in any case.
    */
    realloc_tform(xg);

    init_cprof_plane_x();

    generate_cprof_xticks(xg);
    update_cprof_plot(xg);
  }
}

/*
 * Used by print_plotwin
*/
void
get_cprof_win_dims(maxx, maxy, xg)
  float *maxx, *maxy;
  xgobidata *xg;
{
  if (xg->is_cprof_plotting)
  {
    *maxx = (float) cprof_width ;
    *maxy = (float) cprof_height ;
  }
  else
    *maxx = *maxy = 0 ;
}

void
check_cprof_fac_and_offsets(minx, maxx, maxy, fac, xoff, yoff,
pointsize)
  float *minx, *maxx, *maxy, *fac, *xoff, *yoff ;
  int pointsize;
{
  /*
   * Shift x axis.  The constant 4.5 comes from the header file.
  */
  while (72.*((*maxy - cprof_screen_axes[1].y) * *fac + *yoff) -
             4.0*pointsize < 72. * *yoff)
  {
    *maxy = 1.02 * *maxy ;
    set_fac_and_offsets(*minx, *maxx, *maxy, fac, xoff, yoff);
  }

  /*
   * Shift x axis.  The constant 4.5 comes from the header file.
  */
  while (72.*((cprof_screen_axes[1].x - *minx) * *fac + *xoff) -
         4.0*pointsize < 72. * *xoff)
  {
    *minx = *minx - .02 * (*maxx - *minx) ;
    /**maxx = 1.02 * *maxx ;*/
    set_fac_and_offsets(*minx, *maxx, *maxy, fac, xoff, yoff);
  }
}

void
print_cprof_win(xg, psfile, minx, maxy, fac, xoff, yoff, fg, rgb_table, ncolors)
  xgobidata *xg;
  FILE *psfile;
  float minx, maxy, fac, xoff, yoff;
  XColor *fg;
  XColor *rgb_table;
  unsigned int ncolors;
{
  int i, j, k, indx, id;
  int offset, toomany;
  char str[64];
  int pgsize;
  XColor *rgb;

  rgb = (XColor *) XtMalloc((Cardinal) *nprofiles * sizeof(XColor));
  for (j=0; j<*nprofiles; j++) {
    id = *(profile_ids+j);  
    for (k=0; k<ncolors; k++) {
      if (xg->color_now[id] == color_nums[k]) {
        rgb[j].red = rgb_table[k].red;
        rgb[j].green = rgb_table[k].green;
        rgb[j].blue = rgb_table[k].blue;
        break;
      }
    }
  }

/*
 * id is the row number of the point in question;
 * indx indexes the points of the profile curve.
*/
  /* first the points */
  if (cprof_showpoints) {
    for (j=0; j<*nprofiles; j++) {
      id = *(profile_ids+j);

      pgsize = find_pgsize(xg->glyph_now[id].type, xg->glyph_now[id].size);
      for (i=0; i<cprof_nselectedvars; i++) {
        /*
         * cprof_screen[] for the sticky guys starts at
         * cprof_nselectedvars; this accounts for the (j+1) here
        */
        indx = (j+1)*cprof_nselectedvars + i ;

        if (!xg->erased[id]) {
          if (plot_imputed_values ||
              !xg->is_missing[id][ cprof_selectedvars[i] ])
          {
            (void) fprintf(psfile, "%f %f %f %d %d %f %f pg\n",
              (float) rgb[j].red / (float) 65535,
              (float) rgb[j].green / (float) 65535,
              (float) rgb[j].blue / (float) 65535,
              xg->glyph_now[id].type,
              pgsize,
              (float) (cprof_screen[indx].x - minx) * fac + xoff,
              (float) (maxy - cprof_screen[indx].y) * fac + yoff );
           }
         }
       }
    }
  }

  /*
   * Then the lines.  I have to use the same colors_used
   * vector that's been used to draw to the window.
  */
  if (cprof_showlines) {
    int nv, i, var1, var2;
    long current_color;
    for (k=0; k<ncolors_used; k++) {
      current_color = colors_used[k];
      nv = cprof_nselectedvars;
      for (j=0; j<*nprofiles; j++) {
        id = *(profile_ids+j);
        if (xg->color_now[id] != current_color || xg->erased[id]) 
          nv += (cprof_nselectedvars-1);
        else {  /* current_color && !erased */
          for (i=0; i<cprof_nselectedvars-1; i++) {
            var1 = cprof_selectedvars[i];
            var2 = cprof_selectedvars[i+1];
            if (!plot_imputed_values && xg->missing_values_present &&
                (xg->is_missing[id][var1] || xg->is_missing[id][var2]))
              ;
            else {
              (void) fprintf(psfile, "%f %f %f 0 %f %f %f %f ln\n",
                (float) rgb[j].red / (float) 65535,
                (float) rgb[j].green / (float) 65535,
                (float) rgb[j].blue / (float) 65535,
                (float) (cprof_screen[i].x - minx) * fac + xoff,
                (float) (maxy - cprof_screen[nv].y) * fac + yoff,
                (float) (cprof_screen[i+1].x - minx) * fac + xoff,
                (float) (maxy - cprof_screen[nv+1].y) * fac + yoff );
            }
            nv++;
          }
        }
        nv++;
      }
    }
  }

/*
 * At the moment, once the cursor leaves the plot window,
 * no profile is drawn.  So it is meaningless to draw
 * the point and line for nearest_point.  Leave them out.
*/

  /*
   * Draw y axis
  */
  (void) fprintf(psfile, "%% yax: (label) red green blue x1 y1 x2 y2\n");
  (void) fprintf(psfile, "%%  draw y axis (and null label)\n");
  (void) fprintf(psfile, "(%s) %f %f %f %f %f %f %f yax\n",
    "",
    (float) fg->red / (float) 65535,
    (float) fg->green / (float) 65535,
    (float) fg->blue / (float) 65535,
    (float) (cprof_screen_axes[1].x - minx) * fac + xoff,
    (float) (maxy - cprof_screen_axes[1].y) * fac + yoff,
    (float) (cprof_screen_axes[0].x - minx) * fac + xoff,
    (float) (maxy - cprof_screen_axes[0].y) * fac + yoff );

  (void) fprintf(psfile, "%% ytx: (label) red green blue x y\n");
  (void) fprintf(psfile, "%%  draw y axis tick and label\n");
  offset = find_cprof_ytick_offset();
  toomany = find_cprof_ytick_toomany();
  for (i=offset; i<cprof_ticks.nticks[1]-toomany; i++)
  {
    find_cprof_tick_label(cprof_ydeci, i, cprof_ticks.yticks, str);
    (void) fprintf(psfile, "(%s) %f %f %f %f %f ytx\n",
      str,
      (float) fg->red / (float) 65535,
      (float) fg->green / (float) 65535,
      (float) fg->blue / (float) 65535,
      (float) (cprof_screen_axes[1].x - minx) * fac + xoff,
      (float) (maxy - cprof_ticks.screen[i].y) * fac + yoff);
  }

  /*
   * Draw x axis
  */
  (void) fprintf(psfile, "%% xax: (label) red green blue x1 y1 x2 y2\n");
  (void) fprintf(psfile, "%%  draw x axis (and null label)\n");
  (void) fprintf(psfile, "(%s) %f %f %f %f %f %f %f xax\n",
    "Variable Number",
    (float) fg->red / (float) 65535,
    (float) fg->green / (float) 65535,
    (float) fg->blue / (float) 65535,
    (float) (cprof_screen_axes[1].x - minx) * fac + xoff,
    (float) (maxy - cprof_screen_axes[1].y) * fac + yoff,
    (float) (cprof_screen_axes[2].x - minx) * fac + xoff,
    (float) (maxy - cprof_screen_axes[2].y) * fac + yoff );

  (void) fprintf(psfile, "%% xtx: (label) red green blue x y\n");
  (void) fprintf(psfile, "%%  draw x axis tick and label\n");
  for (i=0; i<cprof_ticks.nticks[0]; i++)
  {
    find_cprof_tick_label(cprof_xdeci, cprof_selectedvars[i],
      cprof_ticks.xticks, str);
    /*
     * we're printing 'i' here instead of the variable inde;
     * this should fix it.  It assumes we're labelling all
     * ticks -- but we can use the same arbitrary scheme used
     * elsewhere in this file.
     */
    if (cprof_ticks.nticks[0] > 30 && i%4 != 0)
      sprintf(str, "");
    else if (cprof_ticks.nticks[0] <= 30 &&
             cprof_ticks.nticks[0] > 20 && i%2 != 0)
      sprintf(str, "");

    /* find_cprof_tick_label(cprof_xdeci, i, cprof_ticks.xticks, str); */
    (void) fprintf(psfile, "(%s) %f %f %f %f %f xtx\n",
      str,
      (float) fg->red / (float) 65535,
      (float) fg->green / (float) 65535, 
      (float) fg->blue / (float) 65535,
      (float) (cprof_ticks.screen[i].x - minx) * fac + xoff,
      (float) (maxy - cprof_screen_axes[1].y) * fac + yoff);
  }

  XtFree((XtPointer) rgb);
}

