/*************************************************************************\
**  file	 * adhoc.c
**               * 
**  description  * SOLID SQL API usage sample:
**               * 
**               * Very simple ad-hoc SQL statement execution utility
**               * 
**               * (C) Copyright Solid Information Technology Ltd 1996
\*************************************************************************/

/* Environment specific headers */
#ifdef SS_WINDOWS
#include <windows.h>
#include <stdio.h>
#endif

/* Common system headers */
#include <c.h>
#include <assert.h>


/* SOLID SQL API headers */
#include <cli0defs.h>
#include <cli0core.h>
#include <cli0ext1.h>

/* Constants */
#define MAXCOLS              30
#define NAME_MAXLEN         250
#define STATEMENT_MAXLEN   2500

/* Utility macros - crude error handling */
#ifdef DEBUG
	#define DBG(s)	printf("%s.\n", (s));fflush(stdout)
#else
	#define DBG(s)
#endif

#define FUNC(s) \
        char* err_fun = (s); \
	char* err_oper = "init";\
	DBG(s)

#define CALL(s) \
        err_oper = (s);\
	DBG(s)

#define CHECK(rc) \
        if ((rc) != SQL_SUCCESS && (rc) != SQL_SUCCESS_WITH_INFO) \
            error_exit((rc), henv, hdbc, hstmt, \
                       err_oper, err_fun, __FILE__, __LINE__)


/* Function prototypes */
static void usage();
static void welcome();
static void goodbye(UCHAR* network_name);
static void show_sample_statements();
static void ask_connection_info(
        UCHAR*  network_name, UCHAR* username, UCHAR* password
);
static void initialize_connection(
        HENV* phenv, HDBC* phdbc, HSTMT* phstmt,
        UCHAR*  network_name, UCHAR* username, UCHAR* password
);
static void terminate_connection(HENV henv, HDBC hdbc, HSTMT hstmt);
static void process_SQL_statements(HENV henv, HDBC hdbc, HSTMT hstmt);
static void retrieve_results(HENV henv, HDBC hdbc, HSTMT hstmt);
static void print_error(
        HENV henv, HDBC hdbc, HSTMT hstmt
);
static void error_exit(
        RETCODE rc, HENV henv, HDBC hdbc, HSTMT hstmt,
        UCHAR* err_oper, UCHAR* err_fun, UCHAR* err_file, SWORD err_line
);
static SWORD display_size(SWORD coltype, SWORD collen, UCHAR *colname);
static UCHAR* read_line(UCHAR* string, SWORD maxlen);



/*##**********************************************************************\
 *
 *              main
 *
 * ad hoc SQL query execution program main function.
 *
 * Parameters : none
 *
 * Return value - give :
 *       0      = Success return
 *      -1      = Error occurred
 */
int SS_CDECL main(int argc, char* argv[])
{
        UCHAR   network_name[NAME_MAXLEN];
        UCHAR   username[NAME_MAXLEN];
        UCHAR   password[NAME_MAXLEN];
        HENV    henv;
        HDBC    hdbc;
        HSTMT   hstmt;

        FUNC("main");

        if (argc > 1) {
            usage();
            return(-1);
        }

        welcome();

        ask_connection_info(network_name, username, password);

        initialize_connection(&henv, &hdbc, &hstmt,
                              network_name, username, password);

        process_SQL_statements(henv, hdbc, hstmt);

        terminate_connection(henv, hdbc, hstmt);

        goodbye(network_name);

        return(0);
}


/*##**********************************************************************\
 *
 *              usage
 *
 * Print usage instructions on standard output.
 */
static void usage()
{
        FUNC("usage");

        printf("\nUsage:  adhoc \n\n");
        welcome();
}


/*##**********************************************************************\
 *
 *              welcome
 *
 * Print welcome texts on standard output. 
 */
static void welcome()
{
        FUNC("welcome");

        printf("This program allows you to connect to a SOLID Server and\n");
        printf("to dynamically execute SQL statements.\n\n");
}


/*##**********************************************************************\
 *
 *              show_sample_statements
 *
 * Output examples of possible SQL statements.
 */
static void show_sample_statements()
{
        FUNC("show_sample_statements");

        printf("Samples of recognized SQL statements:\n");
        printf("\tCREATE TABLE TESTTABLE (ID INTEGER, NAME VARCHAR(50))\n");
        printf("\tSELECT TABLE_NAME FROM TABLES WHERE TABLE_SCHEMA NOT LIKE \'_SYS%%\'\n");
        printf("\tINSERT INTO TESTTABLE (ID, NAME) VALUES (123, 'SPIFF')\n");
        printf("\tUPDATE TESTTABLE SET NAME = 'CALVIN' WHERE ID = 123\n");
        printf("\tSELECT * FROM TESTTABLE\n");
        printf("\tDELETE FROM TESTTABLE\n");
        printf("\tDROP TABLE TESTTABLE\n");
        printf("\tSELECT * FROM SYS_USERS\n");
        printf("\tADMIN COMMAND 'backup'\n");
}


/*##**********************************************************************\
 *
 *              goodbye
 *
 * Output normal termination message.
 */
static void goodbye(UCHAR* network_name)
{
        FUNC("goodbye");

        printf("Disconnected from '%s'.\n", network_name);
}


/*##**********************************************************************\
 *
 *              ask_connection_info
 *
 * Prompt the user for SOLID Server network name, username and password.
 *
 * Parameters - output:
 *      network name, username, password: character strings 
 */
static void ask_connection_info(
        UCHAR*  network_name, UCHAR* username, UCHAR* password
) {
        FUNC("ask_connection_info");

        printf("Try connecting with the default network name ");
#ifdef SS_WINDOWS
        printf("'shmem solid'.\n");
#else
        printf("'tcp [host] 1313'.\n");
#endif

        printf("\nConnect to SOLID Server with network name:\n");
        read_line(network_name, NAME_MAXLEN);

        printf("\nUsername:\n");
        read_line(username, NAME_MAXLEN);

        printf("\nPassword:\n");
        read_line(password, NAME_MAXLEN);
}


/*##**********************************************************************\
 *
 *              initialize_connection
 *
 * Allocate and initialize connection elements and establish connection.
 *
 * Parameters - input:
 *      network name, username, password: character strings 
 *
 * Parameters - output:
 *      phenv, phdbc, phstmt:   pointers to environment, connection and
 *                              statement handles
 */
static void initialize_connection(
        HENV* phenv, HDBC* phdbc, HSTMT* phstmt,
        UCHAR*  network_name, UCHAR* username, UCHAR* password
) {
        RETCODE rc;
        HENV henv = NULL;   /* local variables - for error handling macros */
        HDBC hdbc = NULL;
        HSTMT hstmt = NULL;
        FUNC("initialize_connection");

	/* Allocate environment and connection handles. */

        CALL("SQLAllocenv");
	rc = SQLAllocEnv(phenv);
        henv = *phenv;
        CHECK(rc);

        CALL("SQLAllocConnect");
	rc = SQLAllocConnect(henv, phdbc);
        hdbc = *phdbc;
        CHECK(rc);


	/* Connect to the data source. */

        printf("Connecting to '%s'...\n", network_name);
        CALL("SQLConnect");
	rc = SQLConnect(hdbc, network_name, SQL_NTS,
                        username, SQL_NTS, password, SQL_NTS);
        CHECK(rc);
        printf("Connection established.\n");


	/* Allocate a statement handle. */

        CALL("SQLAllocStmt");
	SQLAllocStmt(*phdbc, phstmt);
        hstmt = *phstmt;
        CHECK(rc);
}


/*##**********************************************************************\
 *
 *              process_SQL_statements
 *
 * Prompt user for SQL statement, execute it and retrieve results.
 *
 * Parameters - input:
 *      henv, hdbc, hstmt:   environment, connection and statement handles
 */
static void process_SQL_statements(
        HENV henv, HDBC hdbc, HSTMT hstmt
) {
        UCHAR   statement[STATEMENT_MAXLEN];
        RETCODE rc;
        FUNC("process_SQL_statements");

        printf("\nNext you can enter SQL statements to be executed.\n");
        printf("When line ends with '\\', statement continues on the following line.\n");
        printf("Enter 'hint' to see some sample SQL statements.\n");
        printf("Enter 'quit' to exit from this program.\n\n");

	/* Ask for a SQL statement */
        printf("Enter SQL statement:\n");
        read_line(statement, STATEMENT_MAXLEN);

        while (strncmp(statement, "quit", 4)) {

            if (!strlen(statement))
                /* Empty string -> do nothing */
                ;

            else if (!strncmp(statement, "hint", 4)) {

                show_sample_statements();

            } else {

            	/* Execute the SQL statement. */
                CALL("SQLExecDirect");
                rc = SQLExecDirect(hstmt, statement, SQL_NTS);

                /* Retrieve the results & Commit */
                if (rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO) {
                    retrieve_results(henv, hdbc, hstmt);
                } else {
                    print_error(henv, hdbc, hstmt);
                    /* Close any possibly open cursors */
                    CALL("SQLFreeStmt");
                    rc = SQLFreeStmt(hstmt, SQL_CLOSE );
                    CHECK(rc);

                    CALL("SQLTransact");
            	    rc = SQLTransact(henv, hdbc, SQL_ROLLBACK);
                    CHECK(rc);
                }
            }

            /* Ask for the next SQL statement */
            printf("\nEnter SQL statement:\n");
            read_line(statement, STATEMENT_MAXLEN);
        }
}


/*##**********************************************************************\
 *
 *              retrieve_results
 *
 * Prompt user for SQL statement, execute it and retrieve results.
 *
 * Parameters - input:
 *      henv, hdbc, hstmt:   environment, connection and statement handles
 */
static void retrieve_results(
        HENV henv, HDBC hdbc, HSTMT hstmt
) {
	int     i;
        RETCODE rc;
	UCHAR   errmsg[256];
	UCHAR   colname[32];
	SWORD   coltype;
	SWORD   colnamelen;
	SWORD   nullable;
	UDWORD  collen[MAXCOLS];
	SWORD   scale;
	SDWORD  outlen[MAXCOLS];
	UCHAR * data[MAXCOLS];
	SWORD   nresultcols;
	SDWORD  rowcount;

        FUNC("retrieve_results");

	/* See what kind of statement it was.  If there are */
	/* no result columns, the statement is not a SELECT */
	/* statement.  If the number of affected rows is    */
	/* greater than 0, the statement was probably an    */
	/* UPDATE, INSERT, or DELETE statement, so print    */
	/* the number of affected rows.  If the number of   */
	/* affected rows is 0, the statement is probably a  */
	/* DDL statement, so print that the operation was   */
	/* successful and commit it.                        */
	
        CALL("SQLNumResultCols");
	rc = SQLNumResultCols(hstmt, &nresultcols);
        CHECK(rc);

	if (nresultcols == 0) {

            CALL("SQLRowCount");
	    rc = SQLRowCount(hstmt, &rowcount);
            CHECK(rc);

	    if (rowcount > 0 ) {
		    printf("%ld rows affected.\n", rowcount);
	    } else {
        		printf("Operation successful.\n");
	    }

            CALL("SQLTransact");
	    rc = SQLTransact(henv, hdbc, SQL_COMMIT);
            CHECK(rc);
	
	/* Otherwise, display the column names of the result */
	/* set and use the display_size() function to        */
	/* compute the length needed by each data type.      */
	/* Next, bind the columns and specify all data will  */
	/* be converted to char.  Finally, fetch and print   */
	/* each row, printing truncation messages as         */
	/* necessary. */
	} else {

	    for (i = 0; i < nresultcols; i++) {

                CALL("SQLDescribeCol");
			rc = SQLDescribeCol(hstmt, i + 1, colname,
				(SWORD)sizeof(colname), &colnamelen,
				&coltype, &collen[i], &scale, 
				&nullable
                );
                CHECK(rc);

			collen[i] = display_size(coltype, collen[i], colname);
			printf("%*.*s", collen[i], collen[i], colname);
			data[i] = (UCHAR *) malloc(collen[i] + 1);

                CALL("SQLBindCol");
			rc = SQLBindCol(hstmt, i + 1, SQL_C_CHAR, data[i],
                                collen[i], &outlen[i]);
                CHECK(rc);
 	    }
            printf("\n");

	    while (TRUE) {

                CALL("SQLFetch");
			rc = SQLFetch(hstmt);

			if (rc == SQL_SUCCESS || rc == SQL_SUCCESS_WITH_INFO) {
 				errmsg[0] = '\0';
				for (i = 0; i < nresultcols; i++) {
					if (outlen[i] == SQL_NULL_DATA) {
						strcpy((char *)data[i], "NULL");
     			    	} else if (outlen[i] >= collen[i]) {
						sprintf((char *)&errmsg[strlen((const char *)errmsg)],
					    	"%d chars truncated, col %d\n",
        				    		outlen[i] - collen[i] + 1, i);
     	    			}
					printf("%*.*s", collen[i], collen[i], data[i]);
   				}
				printf("\n%s", errmsg);
  			} else {
                    if (rc == SQL_ERROR)
                        print_error(henv, hdbc, hstmt);
				break;
 			}
		}

	    /* Free the data buffers. */
	    for (i = 0; i < nresultcols; i++) {
	        free(data[i]);
	    }

            /* Close open cursors */
            CALL("SQLFreeStmt - close");
            rc = SQLFreeStmt(hstmt, SQL_CLOSE );
            CHECK(rc);

            /* Unbind variables */
            CALL("SQLFreeStmt - unbind");
            rc = SQLFreeStmt(hstmt, SQL_UNBIND );
            CHECK(rc);

            CALL("SQLTransact");
            rc = SQLTransact(henv, hdbc, SQL_COMMIT);
            CHECK(rc);
	}
}


/*##**********************************************************************\
 *
 *              terminate_connection
 *
 * Close connection and free resources.
 *
 * Parameters - input:
 *      henv, hdbc, hstmt:   environment, connection and statement handles
 */
static void terminate_connection(
        HENV henv, HDBC hdbc, HSTMT hstmt
) {
        FUNC("terminate_connection");

	/* Free the statement handle.       */
	/* Disconnect from the database.    */
	/* Free the connection handle.      */
	/* Free the environment handle.     */

	SQLFreeStmt(hstmt, SQL_DROP );
	SQLDisconnect(hdbc);
	SQLFreeConnect(hdbc);
	SQLFreeEnv(henv);
}


/*##**********************************************************************\
 *
 *              error_exit
 *
 * Output error message, disconnect from database and exit.
 */
static void error_exit(
        RETCODE rc, HENV henv, HDBC hdbc, HSTMT hstmt,
        UCHAR* err_oper, UCHAR* err_fun, UCHAR* err_file, SWORD err_line
) {
        printf("Error (%d) encountered when calling '%s' in ", rc, err_oper);
        printf("'%s' [file: %s, line: %d] \n", err_fun, err_file, err_line);

        print_error(henv, hdbc, hstmt);
        terminate_connection(henv, hdbc, hstmt);
        printf("Exiting...\n");
        exit(-1);
}


/*##**********************************************************************\
 *
 *              print_error
 *
 * Output error message.
 */
static void print_error(HENV henv, HDBC hdbc, HSTMT hstmt) {
        UCHAR   SQLstate[NAME_MAXLEN];
        SDWORD  errorcode;
        UCHAR   errormsg[SQL_MAX_MESSAGE_LENGTH];
        SWORD   msglen;
        RETCODE rc;

        DBG("print_error.\n");

        /* Print more error info, if available */
        rc = SQLError(henv, hdbc, hstmt, SQLstate, &errorcode, errormsg,
                     SQL_MAX_MESSAGE_LENGTH - 1, &msglen);
        if (rc != SQL_NO_DATA_FOUND) {
            printf("%s\n", errormsg);
        }
}


/*##**********************************************************************\
 *
 *              display_size
 *
 * Returns the display width of column type in bytes.
 */
static SWORD display_size(
        SWORD coltype, SWORD collen, UCHAR *colname
) {
	switch (coltype) {

	    case SQL_CHAR:
	    case SQL_VARCHAR:
			return(SS_MAX(15, strlen((const char *)colname)));

	    case SQL_SMALLINT:
	    	return(SS_MAX(6, strlen((const char *)colname)));

	    case SQL_INTEGER:
			return(SS_MAX(11, strlen((const char *)colname)));

	    case SQL_DECIMAL:
	    case SQL_NUMERIC:
	    case SQL_REAL:
	    case SQL_FLOAT:
	    case SQL_DOUBLE:
			return(SS_MAX(15, strlen((const char *)colname)));
  
	    /* Note that this function only supports the */
	    /* core data types. */
	    default:
			return(15);
 	}
}
	

/*##**********************************************************************\
 *
 *              read_line
 *
 * Read one line from standard input into the parameter buffer.
 * Allows continuing to the following by preceding the end-of-line with '\'.
 * Trim also possible trailing semicolon ';'.
 */
static UCHAR* read_line(
        UCHAR* string, SWORD maxlen
) {

	static UCHAR retbuf[STATEMENT_MAXLEN + 1];
	static UCHAR buf[STATEMENT_MAXLEN + 1];
	UCHAR* sp = NULL; 
	UCHAR* tp = NULL;
	bool loopend = FALSE;

	*string = '\0';
	retbuf[0] = '\0';
	retbuf[STATEMENT_MAXLEN] = '\0';
	buf[0] = '\0';
	buf[STATEMENT_MAXLEN] = '\0';

	while(!loopend) {

            sp = buf;

	 	/* read a line */
		fgets(sp, STATEMENT_MAXLEN - 1, stdin);

		/* trim leading non-alpha characters */
		while (*sp && *sp != '\n' && isspace(*sp)) 
			sp++;

            if (*sp == '\n') *sp = '\0';

            if (!*sp) {
                /* empty string */
                break;
            }

		/* trim trailing non-alpha characters */
		tp = sp + strlen(sp) - 1;
		while (tp > sp && *tp != '\\' && (isspace(*tp) || *tp == ';') )
			tp--;

		/* check if input is continued on the following line */
            if (*tp == '\\') {
			*tp = '\0';
		} else {
			*++tp = '\0';
			loopend = TRUE;
		}
		strcat(retbuf, sp);
	}

	/* check and enforce maxlen */
	if (strlen(retbuf) > maxlen)
		retbuf[maxlen] = '\0';

	strcpy(string, retbuf);
	return(string);
}

