#!/usr/local/bin/perl5 
#
# ABSOLUTELY NO WARRANTY WITH THIS PACKAGE. USE IT AT YOUR OWN RISK.
#
# Parse DIABLO logfile and generate stats about incoming & outgoing feeds.
# URL: ftp://ftp.bricbrac.de/pub/news/diablo-utils
#
# diablo-stats.pl v3.65  980101  Iain Lea  iain@bricbrac.de
#
# Usage: diablo-stats.pl [options]
#     -h       help
#     -d date  date to gather statistics in YYMMDD (default: 970703)
#     -D days  number of days of statistics to display (default: all)
#     -B date  begin time to start gathering statistics in YYMMDD
#     -E date  end time to stop gathering statistics in YYMMDD
#     -e       report articles/sec and kbytes/sec using elapsed wall time
#     -g       generate various graphs of traffic statistics
#     -l file  diablo log file (default: /var/log/news/news.notice)
#     -o file  output file (default: stdout)
#     -s host  server hostname to use in report (default: newsfeed.foo.net)
#     -q       report queue by hostname instead of newsfeed name
#     -S       read logfile in pipe mode from STDIN
#     -t       output format as text (default: HTML)
#     -v       verbose
#     -V       debug
#     -w dir   create dir/index.html page to list daily statistics
#     -W dir   create dir/index.html page to list daily statistics and exit
#
# Acknowledgements:
# -----------------
#   Jeff Garzik             jeff.garzik@spinne.com
#   Terry Kennedy           terry@spcvxa.spc.edu
#   Steve Rawlinson         steve@clara.net
#   Pierre Belanger         belanger@risq.qc.ca
#   Miquel van Smoorenburg  miquels@cistron.nl
#   David Bonner            dbonner@bu.edu
#   Georg v.Zezschwitz      gvz@hamburg.pop.de
#   Christophe Wolfhugel    wolf@pasteur.fr
#
# TODO:
# -----
# - add table of last connect messages for all outgoing feeds (ala algo)
#   Last connection to ecrc resulted in: StreamOK. (streaming)
#
# - add InArts/Sec & OutArts/Sec (ala algo)
# - add .lastday file to contain 2359 values from prevoius day
# - add ftell(logfile) + store 1st line of logfile (save in .file) so that we can 
#   jump to the current date in file and not have to parse all the old days
# - add 'alias: real-hostname alias-to-use' to conf file
# - add header & footer boilerplate placeholders that are read in from file
# - FIX %3.2f formatting in ascii mode
# - add a cmd line switch to delete files that are more than -D N days in index.html
#
# ChangeLog:
# ----------
# v3.65
#   - added 'DIABLO uptime=66:38 ...' to system info at top of page
#   - added link to Freenix stats page
#   - changed index.html to not show incoming 'Errs' field
#   - fixed cosmetic problem with SPAM Top25 table headers
# v3.63
#   - changed $MetaRefresh variable to 4200 (secs)
#   - fixed $Month variable in GetMetaExpires() function
#   - fixed key length in MakeVolKey() function to handle large volumes
# v3.62
#   - changed table headers to 'bgcolor=lightblue' background
#   - changed all '<strong></strong>' tags to '<b></b>' tags to save space
# v3.60
#   - added <meta http-equiv="expires"...> tag to expire pages every 60 mins
#   - added <meta http-equiv="refresh"...> tag to refresh pages every 60 mins
#   - added -e cmdline option to use wall-time when calculating arts/sec etc.
#   - added system uptime, df | egrep news, Num. Diablo & Num. Dnewslink procs
#   - added URL: pointer to -h usage output
#   - added new logo icon for BricBrac Consulting
#   - changed IconGif to be $ImgDir/logo.gif (was ecrclogo.gif)
#   - changed border=0 on logo icon
#   - fixed possible divide by 0 error in GetAvgArtSize()
#   - fixed alt= tags to use comment
# v3.53
#   - error message fixups for diablo-1.14
# v3.51
#   - cosmetic nitpickings
# v3.49
#   - added Kb/art field to incoming & outgoing feeds tables
#   - fixed 1 off error in all GIF images (ouch!)
# v3.48
#   - added <table> around the GIF images 
# v3.47
#   - fixed graphs to show last hour and not last but 1
#   - fixed top1000 code to case insensitive compares
# v3.46
#   - added check for syslog error msg concerning incorrect Path headers
#   - fixed spam table to use &nbsp; for better formatting of empty cells
# v3.44
#   - fixed Arts/sec & kbps by using feed->{InSecs}->{00} and test if for 
#      each hour >3600 and if so reset to 3600
#   - fixed embedded values in incoming table if no diablo lines were parsed
# v3.42
#   - fixed Art/sec & kbps to use wall seconds instead of feed seconds
# v3.41
#   - fixed possible divide by zero error for $InSecs & $OutSecs
# v3.4
#   - added Art/sec & Kbps for incoming & outgoing feed & deleted hh:mm & Cons
#   - added GIF graphs of incoming feeds for articles, volume & time
#   - added checks for more syslog error msgs from diablo & dnewslink
#   - added feed position in Freenix Top1000 list for in & out feeds.
#   - changed curious table to list only time part of 1st & last dates
#   - fixed &GetVolume() to handle large numbers correctly
#   - fixed graphs to not frop to 0 after current hour (used undef values)
#   - fixed tables to use &nbsp; for better formatting of empty cells
# v3.2
#   - added by-hour table showing total arts & vol per hour
#   - added http links to feedname stats page for reverse feed checks:
#     link:  news.foo.com  http://news.foo.com/stats/
#     link:  feed.bar.com  http://feef.bar.com/diablo/
#     etc. etc. (default file is /news/diablo-stats.conf)
#   - added SPAM totals for incoming feeds to top level index.html file
#   - added logo to page headers (default is logo.gif) + link to www site
#   - added $BgColor to <body> tags for defining background color 
#   - added rank field to incoming & outgoing tables
#   - added check for 'lost backchannel to master server' in logfile
#   - added 1stDate-LastDate to curious table instead of just 1stDate
#   - changed fields in incoming & outgoing tables for readability
#   - changed index.html to place Total+Volume as first fields
#   - changed the 'Total' field text to be centered
#   - fixed another BigFloat problem when no initialized value is used
#   - fixed BigFloat panic line 31 error by initializing $InBytes to 0
#   - fixed sorted order of >4GB feeds by using &MakeVolKey
# v3.0 970801
#   - added SPAM table 
#   - added more checks for '400/500/502' type error messages in logfile

use Math::BigInt;
use Math::BigFloat;
require 'getopts.pl';
require "timelocal.pl";

BEGIN {
	eval "use Chart::Lines;";
	$::GraphMode = ($@ eq "");
}

$Version = 'v3.65';
$LogFile = "/var/log/news/news.notice";
$CmdUptime = "uptime";
$CmdDf = "df -k | egrep -i 'avail|news'";
$Uname = `uname -a`;
if ($Uname =~ /IRIX|SunOS.*5.5/) {
	$CmdPs = "ps -defa";
} else {
	$CmdPs = "ps -ax";
}
$CmdDiablo = "$CmdPs | egrep -v grep | egrep diablo | wc -l";
$CmdDnewslink = "$CmdPs | egrep -v grep | egrep dnewslink | wc -l";
$DiabloUptime = "";
#
$CfgFile = "/news/diablo-stats.conf";
$CtlFile = "/news/dnntpspool.ctl";
$QueueCmd = "/news/dbin/doutq";
$Top1000Dir = "/news/top1000";
$ImgDir = "";
$ImgLogo = "$ImgDir/logo.gif";
$FtpUrl = "ftp://ftp.bricbrac.de/pub/news/diablo-utils/";
$WwwUrl = "http://www.bricbrac.de/";
$ImgText = "www.bricbrac.de";
$ImgInfo = "border=0 height=\"50\" width=\"50\"";
$BgColor = "bgcolor=\"#FFFFFF\"";
$OutFile = "";
$HtmlDir = "";
$Verbose = 0;
$Debug = 0;
$MaxSpamList = 25;
$MaxLinesGIF = 7;
$WidthGIF = 670;
# $HeightGIF = 400;
$HeightGIF = 600;
#
$MetaRefresh = 4200;
$MetaExpires = &GetMetaExpires ($MetaRefresh / 60);
# print "EXPIRE=[$MetaExpires]\n"; exit 0;
#
$DiabloUrl = "http://www.backplane.com/diablo/";
$Top1000Url = "http://www.freenix.fr/top1000/";
$WallTime = 0;
$QueueHost = 0;
$TextFormat = 0;
$TotalSpamList = 0;
$TopSpamList = 0;
$ScriptName = 'diablo-stats';
chop ($DiabloHost = `hostname`);
chop ($GenYear = `date +%y`);
chop ($GenMonth = `date +%m`);
chop ($GenDay = `date +%d`);
chop ($CurrHour = `date +%H`);
chop ($CurrMin = `date +%M`);
chop ($CurrSec = `date +%S`);
$GenDate = "$GenYear$GenMonth$GenDay";
$GenTime = "$CurrHour$CurrMin";
$ElapsedSecs = $CurrHour * 3600 + $CurrMin * 60 + $CurrSec;
$MaxDays = 0;
$CreateIndexAndExit = 0;
$BegDate = $EndDate = $GenDate;
$BegTime = $EndTime = "";
$TDL = "<td align=\"left\">";
$TDR = "<td align=\"right\">";
$TDC = "<td align=\"center\">";
@DayName = ("Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday");
@DayNameShort = ("Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat");
%MonthNumByName = (
	'Jan', 1, 'Feb', 2, 'Mar', 3, 'Apr', 4,  'May', 5,  'Jun', 6,
	'Jul', 7, 'Aug', 8, 'Sep', 9, 'Oct', 10, 'Nov', 11, 'Dec', 12);
%MonthNameByNum = (
	'01', 'Jan', '02', 'Feb', '03', 'Mar', '04', 'Apr', '05', 'May', '06', 'Jun',
	'07', 'Jul', '08', 'Aug', '09', 'Sep', '10', 'Oct', '11', 'Nov', '12', 'Dec');
@XLabelHours = (
	'00', '01', '02', '03', '04', '05', '06', '07',
	'08', '09', '10', '11', '12', '13', '14', '15',
	'16', '17', '18', '19', '20', '21', '22', '23' );
$BgColor="bgcolor=\"#FFFFFF\"";
# Colorize outgoing queue. Defaults are 20/25/50/65/80/95%
$QueColor1 = 'lightgreen';
$QueColor2 = 'green';
$QueColor3 = 'lightblue';
$QueColor4 = 'lightyellow';
$QueColor5 = 'orange';
$QueColor6 = 'red';
$QuePercent1 = 20;
$QuePercent2 = 35;
$QuePercent3 = 50;
$QuePercent4 = 65;
$QuePercent5 = 80;
$QuePercent6 = 95;
$TableColor = "bgcolor=\"lightblue\"";

##############################################################################
# 
#

if ($GraphMode == 0 && $TextFormat) {
	print <<EOT

Warning: Chart.pm not installed on your system. Graphs will not be generated.

You can download the Perl5 Chart.pm module from any good CPAN site.  Example:

ftp://ftp.ecrc.de/pub/perl/cpan/modules/by-module/Chart

EOT
;
}

&ParseCmdLine ($0);
&ParseTop1000;

if ($CreateIndexAndExit) {
	&CreateHtmlIndex;
} else {
	&ReadCfgFile;

	&ParseLogFile;

	&ParseOutQueue;

	&CalcHourlyStats;

	&GenerateOutputFile;

	&CreateHtmlIndex;

	if ($GraphMode) {
		$Title = "Top $MaxLinesGIF incoming feeds by articles per hour ($GenDate $GenTime)";
		&MakeIncomingByArticleGIF ("$HtmlDir/$BegDate-in-art.gif", $Title, $WidthGIF, $HeightGIF);
		$Title = "Top $MaxLinesGIF incoming feeds by volume (KB) per hour ($GenDate $GenTime)";
		&MakeIncomingByVolumeGIF ("$HtmlDir/$BegDate-in-vol.gif", $Title, $WidthGIF, $HeightGIF);
		$Title = "Incoming feeds by articles per hour ($GenDate $GenTime)";
		&MakeIncomingByTimeGIF ("$HtmlDir/$BegDate-in-time.gif", $Title, $WidthGIF, $HeightGIF);

#		$Title = "Incoming and outgoing feeds by articles per day ($GenDate $GenTime)";
#		&MakeIndexGIF ("$HtmlDir/$BegDate-index-art.gif", $Title, $WidthGIF, $HeightGIF);
#
#		$Title = "Incoming and outgoing feeds by volume per day ($GenDate $GenTime)";
#		&MakeIndexGIF ("$HtmlDir/$BegDate-index-vol.gif", $Title, $WidthGIF, $HeightGIF);
	}
}

exit 0;

##############################################################################
# 
#

sub ParseCmdLine
{
	my ($ProgName) = @_;
	
	&Getopts('B:E:d:D:eghl:o:s:qStvVw:W:');

	if ($opt_h) { 
		print <<EOT
$ScriptName $Version  $FtpUrl

Create statistics for DIABLO news relay server in Ascii / HTML format.
Copyright 1997 Iain Lea (iain\@bricbrac.de) NOTE: Use at your own risk.

Usage: $ProgName [options]
       -h       help
       -d date  date to gather statistics in YYMMDD (default: $GenDate)
       -D days  number of days of statistics to display (default: all)
       -B date  begin time to start gathering statistics in YYMMDD
       -E date  end time to stop gathering statistics in YYMMDD
       -e       report articles/sec and kbytes/sec using elapsed wall time
       -g       generate various graphs of traffic statistics
       -l file  diablo log file (default: $LogFile)
       -o file  output file (default: stdout)
       -s host  server hostname to use in report (default: $DiabloHost)
       -q       report queue by hostname instead of newsfeed name
       -S       read logfile in pipe mode from STDIN
       -t       output format as text (default: HTML)
       -v       verbose
       -V       debug
       -w dir   create dir/index.html page to list daily statistics
       -W dir   create dir/index.html page to list daily statistics and exit

URL:  $FtpUrl
EOT
;
		exit 1;
	}
	$BegDate = $opt_B if (defined($opt_B));
	$EndDate = $opt_E if (defined($opt_E));
	$BegDate = $EndDate = $GenDate = $opt_d if (defined($opt_d));
	$MaxDays = $opt_D if (defined($opt_D));
	$WallTime++ if (defined($opt_e));
	$GraphMode++ if (defined($opt_g));
	$LogFile = $opt_l if (defined($opt_l));
	$OutFile = $opt_o if (defined($opt_o));
	$DiabloHost = $opt_s if (defined($opt_s));
	$QueueHost++ if (defined($opt_q));
	$TextFormat++ if (defined($opt_t));
	$Verbose++ if (defined($opt_v));
	$Debug++ if (defined($opt_V));
	$HtmlDir = $opt_w if (defined($opt_w));

	if (defined($opt_S)) {
		$LogFile = "-";
	}

	if (defined($opt_W)) {
		$HtmlDir = $opt_W;
		$CreateIndexAndExit++;
	} else {
		die "Error: $LogFile - $!\n" if ($LogFile eq '');
	}

	$GraphMode = 0 if $TextFormat;
}


sub ParseLogFile
{
	my ($Error, $Line, $FullDate, $LineDate);

	printf STDERR "Parsing %s for $BegDate - $EndDate ...\n",
		 ($LogFile eq "-" ? "STDIN" : $LogFile) if $Verbose;

	open (FILE, $LogFile) || die "Error: $LogFile - $!\n";

	while ($Line = <FILE>) 
	{
		next unless (($Line =~ /diablo/o) || ($Line =~ /newslink/o));

		if ($Line =~ /^(... .. ..:..:..)/o) {
			
			$FullDate = $1;
			$LineDate = &GetYyMmDd ($FullDate);
			if ($LineDate >= $BegDate && $LineDate <= $EndDate) {
				chop $Line;

				# ... diablo[230]: news.foo.org secs=15 ihave=0 chk=19 rec=3 rej=0 predup=0 posdup=0 pcoll=101 spam=0 err=0 added=3 bytes=2137 (1/sec)
				if ($Line =~ /(.*)\s+diablo\[(\d+)\]:\s+(.*)\s+secs=(\d+)\s+ihave=(\d+)\s+chk=(\d+)\s+rec=(\d+)\s+rej=(\d+)\s+.*\s+err=(\d+)\s+added=(\d+)/o) {
#					$DiabloHost = $1;
#					$DiabloPid = $2;
					$InHost = $3;
					$InSecs = $4;
					$InIhave = $5;
					$InChk = $6;
					$InRec = $7;
					$InRej = $8;
					$InErr = $9;
					$InAdded = $10;

					# Diablo >= 1.10  ... predup=25 posdup=0 pcoll=10 spam=0 ... bytes=18908
					if ($Line =~ /predup=(\d+)\s+posdup=(\d+)\s+pcoll=(\d+)\s+spam=(\d+).*bytes=(\d+)/o) {
						$InPredup = $1;
						$InPosdup = $2;
						$InPcoll = $3;
						$InSpam = $4;
						$InBytes = $5;
					}

					$InHost =~ s/\s//g;
					$BegTime = $LineDate if !$BegTime;
					$EndTime = $LineDate;

					$Feeds{$InHost}->{InIhave} += $InIhave;
					$Feeds{$InHost}->{InChk} += $InChk;
					$Feeds{$InHost}->{InRec} += $InRec;
					$Feeds{$InHost}->{InRej} += $InRej;
					$Feeds{$InHost}->{InPredup} += $InPredup;
					$Feeds{$InHost}->{InPosdup} += $InPosdup;
					$Feeds{$InHost}->{InPcoll} += $InPcoll;
					$Feeds{$InHost}->{InSpam} += $InSpam;
					$Feeds{$InHost}->{InErr} += $InErr;
					$Feeds{$InHost}->{InAdded} += $InAdded;
					$Feeds{$InHost}->{InBytes} += $InBytes;

					&AddToInTimeList (
						$FullDate, $InHost, $InSecs, $InAdded, $InBytes, 
						$InChk, $InIhave, $InSpam, $InRej, $InErr); # if $InAdded;

					printf STDERR "IN  [%s] %4d %4d %4d %4d %4d %4d %4d %4d %4d\n", 
						$InHost, $InSecs, $InIhave, $InChk, $InRec, $InRej, 
						$InSpam, $InErr, $InAdded, $InBytes if $Debug;

				# ... newslink[14596]: news.foo.de:news.foo.de.S00013 final secs=74 acc=0 dup=48 rej=1086 tot=1134 ...
				# ... newslink[22406]: news.foo.de:/news/dqueue/news.foo.de final secs=129 acc=4 dup=299 rej=0 tot=303 ...
				} elsif ($Line =~ /newslink\[(\d+)\]:\s+(.*):(.*)\s+final\s+secs=(\d+)\s+acc=(\d+)\s+dup=(\d+)\s+rej=(\d+)\s+tot=(\d+)/o) {

#					$NewslinkPid = $1;
					$OutHost = $2;
					$OutBatch = $3;
					$OutSecs = $4;
					$OutAcc = $5;
					$OutDup = $6;
					$OutRej = $7;

					$OutHost =~ s/\s//g;
					$BegTime = $LineDate if !$BegTime;
					$EndTime = $LineDate;

					if ($Line =~ /bytes=(\d+)(.*)/o) {
						$OutBytes = $1;
					}

					$Feeds{$OutHost}->{OutCons} += 1;
					$Feeds{$OutHost}->{OutAcc} += $OutAcc;
					$Feeds{$OutHost}->{OutDup} += $OutDup;
					$Feeds{$OutHost}->{OutRej} += $OutRej;
					$Feeds{$OutHost}->{OutBytes} += $OutBytes;

					&AddToOutTimeList ($FullDate, $OutHost, $OutSecs, $OutAcc, 
						$OutBytes, $OutDup, $OutRej); # if $OutAcc;

					printf STDERR "OUT [%s] %4d %4d %4d %4d %4d %4d\n", 
						$OutHost, $OutSecs, $OutAcc, $OutDup, $OutRej, $OutBytes if $Debug;

				# ... newslink[7390]: news.crg.net:/news/dqueue/news.crg.net.S03454 connect: 500 Syntax error or bad command (nostreaming) ...
				} elsif ($Line =~ /newslink\[(\d+)\]:\s+(.*):(.*)\s+connect:\s+(.*)/o) {

#					$NewslinkPid = $1;
					$OutHost = $2;
					$OutBatch = $3;
					$OutMsg = $4;

					if ($OutMsg =~ /((400|500|502)\s+.*)/ ||
						$OutMsg =~ /(Connection timed out.*)/ ||
						$OutMsg =~ /(Connection refused.*)/) {
						$OutHost =~ s/\s//g;
						$Error = "$OutHost: $1";
						&AddToErrorList ($Error, $FullDate, $LineDate);
					}

				# ... diablo[423]: SpamFilter/by-post-rate copy #33: <5rplh4$12p@ron.ipa.com>  206.1.57.29
				# ... diablo[423]: SpamFilter/by-dup-body copy #1: <5rpvin$8ro$1@demdwu11.telemedia.de> essn-m103-83.pool.mediaways.net
				# ... diablo[423]: SpamFilter/dnewsfeeds copy #-1: <5r4dpg$le8$1@news.utu.fi> apus.astro.utu.fi
				} elsif ($Line =~ /diablo.*SpamFilter\/(.*)/o) {

					&AddToSpamList ($1, $FullDate, $LineDate);

				# ... diablo[423]: Connection 8 from 194.59.191.4 (no permission)
				} elsif ($Line =~ /diablo.*Connection.*from\s+(.*)\s+\(no permission\)/o) {

					$Error = "$1 (no permission)";
					&AddToErrorList ($Error, $FullDate, $LineDate);

				# ... diablo[423]: Connect Limit exceeded for 194.108.168.3
				# ... diablo[423]: Connect Limit exceeded (from -M/diablo.config) for 205.252.116.205 (8)
				} elsif ($Line =~ /diablo.*Connect Limit exceeded .* for\s+(.*)/o) {

					$Error = "Connect Limit exceeded for $1";
					&AddToErrorList ($Error, $FullDate, $LineDate);

				# ... diablo[423]: diablo.hosts entry for localhost missing label
				} elsif ($Line =~ /diablo.*diablo.hosts entry for (.*) missing label/o) {

					$Error = "diablo.hosts entry for $1 missing label";
					&AddToErrorList ($Error, $FullDate, $LineDate);

				# ... diablo[423]: dhistory file corrupted on lookup
				# ... diablo[423]: dhistory file realigned by 4 @20439040
				} elsif ($Line =~ /diablo.*(dhistory file.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[423]: lost backchannel to master server
				} elsif ($Line =~ /diablo.*(lost backchannel to master server)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[423]: NumForks [exceeded|ok, reaccepting]
				} elsif ($Line =~ /diablo.*(NumForks.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[423]: Maximum file descriptors exceeded
				} elsif ($Line =~ /diablo.*(Maximum file descriptors exceeded.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[423]: fork failed:
				} elsif ($Line =~ /diablo.*(fork failed:.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[423]: pipe() failed:
				} elsif ($Line =~ /diablo.*(pipe\(\) failed:.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[423]: failure writing to feed
				} elsif ($Line =~ /diablo.*(failure writing to feed.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[423]: .*diablo.hosts file not found
				} elsif ($Line =~ /diablo.*(\/.*diablo.hosts file not found.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

#				# ... diablo[423]: message-id mismatch, command:
#				} elsif ($Line =~ /diablo.*(message-id mismatch, command:.*)/o) {
#
#					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[423]: fdopen() of socket failed
				} elsif ($Line =~ /diablo.*(fdopen\(\) of socket failed.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... newslink[7390]: article batch corrupted:
				} elsif ($Line =~ /newslink.*(article batch corrupted:.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[134]: news.nettuno.it Path element fails to match aliases: news.nettuno.it
				} elsif ($Line =~ /diablo.*\]: (.*Path element fails to match aliases:.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[134]: Diablo misconfiguration, label news.maxwell.syr.edu not found in dnewsfeeds
				} elsif ($Line =~ /diablo.*\]:  Diablo misconfiguration, (.*)/o) {

					&AddToErrorList ($1, $FullDate, $LineDate);

				# ... diablo[134]: DIABLO uptime=3:44 arts=83.000K tested=0 bytes=895.835M fed=4.074M
				} elsif ($Line =~ /(DIABLO uptime=.*)/o) {

					$DiabloUptime = $1;

				}
			}
		}
	}
	close (FILE);

	$Num = 0;
	foreach $Host (sort keys %Feeds) {
		# InAdded
		$Key = sprintf "%012lu.%03d", $Feeds{$Host}->{InAdded}, $Num;
		$OrderByInAdded{$Key} = $Host;
		# InBytes
		$Key = &MakeVolKey ($Feeds{$Host}->{InBytes}, $Num);
		$OrderByInBytes{$Key} = $Host;

		# OutAcc
		$Key = sprintf "%012lu.%03d", $Feeds{$Host}->{OutAcc}, $Num;
		$OrderByOutAcc{$Key} = $Host;
		# OutBytes
		$Key = &MakeVolKey ($Feeds{$Host}->{OutBytes}, $Num);
		$OrderByOutBytes{$Key} = $Host;

		$Num++;
	}

	$Num = 0;
	foreach $Msg (sort keys %ErrorList) {
		$Key = sprintf "%08lu.%03d%s", $ErrorList{$Msg}->{Tot}, $Num, $Msg;
		$OrderByErrorTotal{$Key} = $Msg;

		$Num++;
	}

	$Num = 0;
	foreach $Host (sort keys %SpamList) {
		# SpamTotal
		$Key = sprintf "%08lu.%03d", $SpamList{$Host}->{Total}, $Num;
		$OrderBySpamTotal{$Key} = $Host;

		$Num++;
	}
	$TotalSpamList = $Num;
	$TopSpamList = $MaxSpamList;
	if ($TopSpamList > $TotalSpamList) {
		$TopSpamList = $TotalSpamList
	}

	&DumpTimeList if $Debug;
}


sub ParseOutQueue
{
	my ($Key, $Host, $Num);

	print STDERR "Parsing output of $QueueCmd ...\n" if $Verbose;

	open (CMD, "$QueueCmd |") || die "Error: $QueueCmd - $!\n";

	while (<CMD>) 
	{
#		print if $Debug;

		chop;

		# news.foo.de  2699-2700  (  1/200 files   0% full)	02699 
		if (/^(.*)\s+(\d+)-(\d+)\s+\((.*)\/(.*)\s+files\s+(\d+)%\s+full\)/o) {

			$Host = $1;
			$BatchBeg = $2;
			$BatchEnd = $3;
			$BatchNum = $4;
			$BatchMax = $5;
			$BatchFull = $6;
			$Host =~ s/\s//g;

			$Feeds{$Host}->{BatchBeg} += $BatchBeg;
			$Feeds{$Host}->{BatchEnd} += $BatchEnd;
			$Feeds{$Host}->{BatchNum} += $BatchNum;
			$Feeds{$Host}->{BatchMax} += $BatchMax;
			$Feeds{$Host}->{BatchFull} += $BatchFull;

			printf STDERR "BATCH  %-30s %4d %4d %4d %4d %4d\n", 
				$Host, $Feeds{$Host}->{BatchBeg}, $Feeds{$Host}->{BatchEnd},
				$Feeds{$Host}->{BatchNum}, $Feeds{$Host}->{BatchMax},
				$Feeds{$Host}->{BatchFull} if $Debug;
		}
	}
	close CMD;

	$Num = 0;
	foreach $Host (sort keys %Feeds) {
		$Key = sprintf "%10d.%3d", $Feeds{$Host}->{BatchFull}, $Num;
		$OrderByBatchFull{$Key} = $Host;
		$Num++;
	}
}


sub GenerateOutputFile
{
#	print STDERR "OUT=[$OutFile]\n";

	if ($OutFile) {
		open (FILE, "> $OutFile") || warn "Warning: $OutFile - $!\n";
	} else {
		open (FILE, "> STDOUT") || warn "Warning: STDOUT - $!\n";
	}

	&PrintOutHeader;
 
	print FILE "<p>\n<a name=\"01\">\n<hr>\n<p>\n" if !$TextFormat;
	&PrintIncomingFeeds ("1. Summary of Incoming Feeds by Article", 0);

	&PrintSeparator ("02") if !$TextFormat;
	&PrintIncomingFeeds ("2. Summary of Incoming Feeds by Volume", 1);

	&PrintSeparator ("03") if !$TextFormat;
	&PrintIncomingTimes ("3. Summary of Incoming Feeds by Time");

	&PrintSeparator ("04") if !$TextFormat;
	&PrintOutgoingFeeds ("4. Summary of Outgoing Feeds by Article", 0);

	&PrintSeparator ("05") if !$TextFormat;
	&PrintOutgoingFeeds ("5. Summary of Outgoing Feeds by Volume", 1);

	&PrintSeparator ("06") if !$TextFormat;
	&PrintQueueStats ("6. Summary of Outgoing Queue");

	&PrintSeparator ("07") if !$TextFormat;
	&PrintCuriousActivity ("7. Summary of Curious Activity");

	&PrintSeparator ("08") if !$TextFormat;
	&PrintSpamActivity ("8. Summary of SPAM Activity (Top $TopSpamList of $TotalSpamList)");

	&PrintSeparator ("end") if !$TextFormat;

	&PrintOutFooter;

	close FILE;
}


sub CreateHtmlIndex
{
	my ($IndexFile) = "$HtmlDir/index.html";
	my ($Title1) = "DIABLO statistics for '$DiabloHost'";
	my ($Title2) = "<a href=$DiabloUrl>DIABLO</a>  statistics for '$DiabloHost'";
	my ($InVolume, $OutVolume);
	my (@DailyStats) = "";
	my ($NumDays) = 0;

	print STDERR "INDEX=[$IndexFile]\n" if $Debug;
	print STDERR "TITLE=[$Title]\n" if $Debug;

	open (DIR, "cd $HtmlDir; find . -name \"[0-9]*.html\" -print | sort -r |") || 
		die "Error: $HtmlDir - $!\n";

	open (FILE, ">$IndexFile") || die "Error: $IndexFile - $!\n";

	print FILE <<EOT
<html>
<head>
<title>$Title1</title>
<meta http-equiv="Refresh" content="$MetaRefresh">
<meta http-equiv="Expires" content="$MetaExpires">
</head>
<body $BgColor>
<table border=0>
<tr>
<td><a href="$WwwUrl"> <img $ImgInfo alt="$ImgText" src="$ImgLogo"></a></td>
<td><h3>$Title2</h3></td>
</tr>
</table>
EOT
;
		&PrintOutCopyright;
	print FILE <<EOT
<hr>
<p>
<table border=2>
<tr>
<th $TableColor rowspan=2 colspan=2>Daily Statistics<th $TableColor colspan=4>Incoming Feeds<th $TableColor colspan=4>Outgoing Feeds
<tr>
<th $TableColor>Accepted<th $TableColor>Volume<th $TableColor>Spam<th $TableColor>Rejs<th $TableColor>Accepted<th $TableColor>Volume<th $TableColor>Dups<th $TableColor>Rejs
EOT
;

	while (<DIR>) {
		chop;
		s/^\.\///;
		if (/^(......)\.html/o && ($MaxDays == 0 || ($NumDays < $MaxDays))) {
			$NumDays++;
			$Date = $1;
			$FancyDate = &GetFancyNameDate ($Date, 1, 0);
			@DailyStats = &GetDailyStats ($Date);
			$InVolume = &GetVolume ($DailyStats[3]);
			$OutVolume = &GetVolume ($DailyStats[7]);

			$FancyDate =~ s/\ /&nbsp;/g;

			print FILE <<EOT
<tr>
<td align="right"><b>$NumDays</b>
<td>&nbsp;<a href="$Date.html"><tt>$FancyDate</tt></a>
<td align="right"><b>$DailyStats[2]</b>
<td align="right"><b>$InVolume</b>
<td align="right">$DailyStats[8]
<td align="right">$DailyStats[1]
<td align="right"><b>$DailyStats[6]</b>
<td align="right"><b>$OutVolume</b>
<td align="right">$DailyStats[4]
<td align="right">$DailyStats[5]
EOT
;
		}
	}
	print FILE <<EOT
</table>
<p>
<hr>
EOT
;
	&PrintOutFooter;

	close (FILE);
	close (DIR);
}

	
sub PrintSeparator
{
	my ($Num) = @_;

	if ($Num =~ /beg/i) {
		print FILE "<p>\n<hr>\n<p>\n";
	} elsif ($Num =~ /end/i) {
		print FILE "<a href=\"#00\">Goto top of page</a>\n";
		print FILE "<p>\n<hr>\n";
	} else {
		print FILE "<a name=\"$Num\">\n";
#		print FILE "<a href=\"#00\">Goto top of page</a>, Goto top of table\n";
		print FILE "<a href=\"#00\">Goto top of page</a>\n";
		print FILE "<p>\n<hr>\n<p>\n";
	}
}


sub PrintIncomingFeeds
{
	my ($Title, $ListByVolume) = @_;
	my ($Num, $InSecs, $InIhave, $InChk, $InErr, $InSpam, $InRej);
	my ($InRec, $InAdded, $InBytes, $InPercent, $InSubPercent, $Key, $Comment);
	my ($InPercentAcc, $InSubPercentAcc, $InPercentVol, $InSubPercentVol, $FileGIF);
	my ($SubTotal1, $SubTotal2, $SubTotal3, $SubTotal4, $SubTotal5, $SubTotal6);
	my ($Field1, $Field2, $Field3, $Field4, $Field5, $Field6);
	my ($Total1, $Total2, $Total4, $Total5, $Total6);
	my ($Format, $SubFormat, $AscFormat);

	&PrintTableHeader ($Title);

	if ($ListByVolume) {
		my ($Total3) = new Math::BigFloat 0;
		%OrderedList = %OrderByInBytes;
		$Field1 = "Volume"; 
		$Field2 = "%Vol";
		$Field3 = "Kbps";
		$Field4 = "Accepted"; 
		$Field5 = "%Acc";
		$Field6 = "Kb/art";
		$AscFormat = "%8s %3.2f %3.2f %8d %3.2f %3.2f";
		$SubFormat = "%s%s %s%3.2f %s%3.2f %s%d %s%3.2f %s%3.2f";
		$Format = "%s%s%s%s %s%s%3.2f%s %s%s%3.2f%s %s%s%d%s %s%s%3.2f%s %s%s%3.2f%s";
		$FileGIF = "$BegDate-in-vol.gif";
	} else {
		my ($Total3) = 0;
		%OrderedList = %OrderByInAdded;
		$Field1 = "Accepted"; 
		$Field2 = "%Acc";
		$Field3 = "Art/sec";
		$Field4 = "Volume"; 
		$Field5 = "%Vol";
		$Field6 = "Kb/art";
		$AscFormat = "%8d %3.2f %3.2f %8s %3.2f %3.2f";
		$SubFormat = "%s%d %s%3.2f %s%3.2f %s%s %s%3.2f %s%3.2f";
		$Format = "%s%s%d%s %s%s%3.2f%s %s%s%3.2f%s %s%s%s%s %s%s%3.2f%s %s%s%3.2f%s";
		$FileGIF = "$BegDate-in-art.gif";
	}

	if ($TextFormat) {
		printf FILE "\n%-30s %8s %8s %8s %8s %8s %8s %8s %8s %8s %8s %8s %8s\n\n",
			"Incoming Feed", $Field1, $Field2, $Field3, $Field4, $Field5, $Field6, "Check", "Ihave", "Spam", "Rejs", "Errs";
	} else {
		print FILE "<th $TableColor colspan=3>Incoming Feed (+ <a href=\"http://www.freenix.fr/top1000/\">Freenix #</a>)<th $TableColor>$Field1<th $TableColor>$Field2<th $TableColor>$Field3<th $TableColor>$Field4<th $TableColor>$Field5<th $TableColor>$Field6<th $TableColor>Check<th $TableColor>Ihave<th $TableColor>Spam<th $TableColor>Rejs<th $TableColor>Errs\n";
	}

	$InAdded = $InBytes = $InErr = $InSpam = $InRej = 0;
	$InSecs = 1;

	foreach $Key (reverse sort keys %OrderedList) {
		$Host = $OrderedList{$Key};

		if ($Feeds{$Host}->{InAdded}) {

			$InSecs += $Feeds{$Host}->{InSecs};
			$InIhave += $Feeds{$Host}->{InIhave};
			$InChk += $Feeds{$Host}->{InChk};
			$InSpam += $Feeds{$Host}->{InSpam};
			$InErr += $Feeds{$Host}->{InErr};
			$InRej += $Feeds{$Host}->{InRej};
			$InAdded += $Feeds{$Host}->{InAdded};
			$InBytes += $Feeds{$Host}->{InBytes};
		}
	}

	foreach $Key (reverse sort keys %OrderedList) {
		$Host = $OrderedList{$Key};

		if ($Feeds{$Host}->{InSecs} && $Feeds{$Host}->{InAdded}) {

			$InSubPercentAcc = &GetPercent ($InAdded, $Feeds{$Host}->{InAdded});
			$InPercentAcc += $InSubPercentAcc;

			$InSubPercentVol = &GetPercent ($InBytes, $Feeds{$Host}->{InBytes});
			$InPercentVol += $InSubPercentVol;

			if ($ListByVolume) {
				$SubTotal1 = &GetVolume ($Feeds{$Host}->{InBytes});
				$SubTotal2 = $InSubPercentVol;
				if ($WallTime) {
					$SubTotal3 = &GetKbps ($Feeds{$Host}->{InBytes}, $ElapsedSecs);
				} else {
					$SubTotal3 = &GetKbps ($Feeds{$Host}->{InBytes}, $Feeds{$Host}->{InSecs});
					$Total3 += $SubTotal3;
				}
				$SubTotal4 = $Feeds{$Host}->{InAdded};
				$SubTotal5 = $InSubPercentAcc;
			} else {
				$SubTotal1 = $Feeds{$Host}->{InAdded};
				$SubTotal2 = $InSubPercentAcc;
				if ($WallTime) {
					$SubTotal3 = $Feeds{$Host}->{InAdded} / $ElapsedSecs;
				} else {
					$SubTotal3 = $Feeds{$Host}->{InAdded} / $Feeds{$Host}->{InSecs};
					$Total3 += $SubTotal3;
				}
				$SubTotal4 = &GetVolume ($Feeds{$Host}->{InBytes});
				$SubTotal5 = $InSubPercentVol;
			}
			$SubTotal6 = &GetAvgArtSize ($Feeds{$Host}->{InBytes}, $Feeds{$Host}->{InAdded});

			if ($TextFormat) {
				printf FILE "%-30s $AscFormat %8d %8d %8d %8d %8d\n",
					&GetHostLink ($Host), 
					$SubTotal1, 
					$SubTotal2, 
					$SubTotal3, 
					$SubTotal4,
					$SubTotal5,
					$SubTotal6,
					$Feeds{$Host}->{InChk},
					$Feeds{$Host}->{InIhave}, 
					$Feeds{$Host}->{InSpam}, 
					$Feeds{$Host}->{InRej}, 
					$Feeds{$Host}->{InErr};
			} else {
				printf FILE "<tr>\n%s%s%d%s %s%s %s&nbsp;%s $SubFormat %s%d %s%d %s%d %s%d %s%d\n",
					$TDR, "<b>", ++$Num, "</b>", 
					$TDL, &GetHostLink ($Host), 
					$TDR, &GetTop1000 ($Host),
					$TDR, $SubTotal1,
					$TDR, $SubTotal2,
					$TDR, $SubTotal3,
					$TDR, $SubTotal4,
					$TDR, $SubTotal5,
					$TDR, $SubTotal6,
					$TDR, $Feeds{$Host}->{InChk}, 
					$TDR, $Feeds{$Host}->{InIhave}, 
					$TDR, $Feeds{$Host}->{InSpam}, 
					$TDR, $Feeds{$Host}->{InRej},
					$TDR, $Feeds{$Host}->{InErr};
			}
		}
	}

	if ($ListByVolume) {
		$Total1 = &GetVolume ($InBytes);
		$Total2 = $InPercentVol;
		if ($WallTime) {
			$Total3 = &GetKbps ($InBytes, $ElapsedSecs);
		}
		$Total4 = $InAdded;
		$Total5 = $InPercentAcc;
	} else {
		$Total1 = $InAdded;
		$Total2 = $InPercentAcc;
		if ($WallTime) {
			$Total3 = $InAdded / $ElapsedSecs;
		}
		$Total4 = &GetVolume ($InBytes);
		$Total5 = $InPercentVol;
	}
	$Total6 = &GetAvgArtSize ($InBytes, $InAdded);

	if ($TextFormat) {
		printf FILE "\n%s $AscFormat %8d %8d %8d %8d %8d\n\n",
			"                              ",
			$Total1, $Total2, $Total3, $Total4, $Total5, $Total6,
			$InChk, $InIhave, $InSpam, $InRej, $InErr; 
	} else {
		printf FILE "<tr>\n%s%s $Format %s%s%d%s %s%s%d%s %s%s%d%s %s%s%d%s %s%s%d%s\n",
			"<td align=\"center\" colspan=3>", "<b>Total</b>",
			$TDR, "<b>", $Total1, "</b>",
			$TDR, "<b>", $Total2, "</b>",
			$TDR, "<b>", $Total3, "</b>",
			$TDR, "<b>", $Total4, "</b>",
			$TDR, "<b>", $Total5, "</b>",
			$TDR, "<b>", $Total6, "</b>",
			$TDR, "<b>", $InChk, "</b>", 
			$TDR, "<b>", $InIhave, "</b>", 
			$TDR, "<b>", $InSpam, "</b>", 
			$TDR, "<b>", $InRej, "</b>", 
			$TDR, "<b>", $InErr, "</b>";
	}

	$Comment = "INCOMING ERR=$InErr REJ=$InRej SPAM=$InSpam ADD=$InAdded VOL=$InBytes";

	&PrintTableFooter ($Comment);

	&PrintUrlGIF ($FileGIF);
}


sub PrintIncomingTimes
{
	my ($Title) = @_;
	my ($Num, $Hour, $InVolume, $InSubVolume, $FileGIF);
	my ($InPercentAcc, $InSubPercentAcc, $InPercentVol, $InSubPercentVol);

	$FileGIF = "$BegDate-in-time.gif";

	&PrintTableHeader ($Title);

	if ($TextFormat) {
		printf FILE "\n%5s %8s %6s %8s %6s %8s %8s %8s %8s %8s\n\n",
			"Hour", "Accepted", "%Acc", "Volume", "%Vol", "Check", "Ihave", "Spam", "Rejs", "Errs";
	} else {
		print FILE "<th $TableColor>Hour<th $TableColor>Accepted<th $TableColor>%Acc<th $TableColor>Volume<th $TableColor>%Vol<th $TableColor>Check<th $TableColor>Ihave<th $TableColor>Spam<th $TableColor>Rejs<th $TableColor>Errs\n";
	}

	for ($Num = 0; $Num < 24; $Num++) {
		$Hour = sprintf ("%02d", $Num);

		$InSubPercentAcc = &GetPercent ($HourlyStats{Total}->{InAdded}, $HourlyStats{$Hour}->{InAdded});
		$InPercentAcc += $InSubPercentAcc;

		$InSubPercentVol = &GetPercent ($HourlyStats{Total}->{InBytes}, $HourlyStats{$Hour}->{InBytes});
		$InPercentVol += $InSubPercentVol;

		$InSubVolume = &GetVolume ($HourlyStats{$Hour}->{InBytes});

		if ($TextFormat) {
			printf FILE "%5s %8d %3.2f %8s %3.2f %8d %8d %8d %8d %8d\n",
				$Hour, 
				$HourlyStats{$Hour}->{InAdded}, 
				$InSubPercentAcc,
				$InSubVolume,
				$InSubPercentVol,
				$HourlyStats{$Hour}->{InChk},
				$HourlyStats{$Hour}->{InIhave},
				$HourlyStats{$Hour}->{InSpam},
				$HourlyStats{$Hour}->{InRej}, 
				$HourlyStats{$Hour}->{InErr};
		} else {
			printf FILE "<tr>\n%s%s%s%s %s%d %s%3.2f %s%s %s%3.2f %s%d %s%d %s%d %s%d %s%d\n",
				$TDR, "<b>", $Hour, "</b>",
				$TDR, $HourlyStats{$Hour}->{InAdded},
				$TDR, $InSubPercentAcc,
				$TDR, $InSubVolume,
				$TDR, $InSubPercentVol,
				$TDR, $HourlyStats{$Hour}->{InChk},
				$TDR, $HourlyStats{$Hour}->{InIhave},
				$TDR, $HourlyStats{$Hour}->{InSpam},
				$TDR, $HourlyStats{$Hour}->{InRej},
				$TDR, $HourlyStats{$Hour}->{InErr};
		}
	}

	$InVolume = &GetVolume ($HourlyStats{Total}->{InBytes});

	if ($TextFormat) {
		printf FILE "\n%5s %8d %3.2f %8s %3.2f %8d %8d %8d %8d %8d\n\n",
			"Total",
			$HourlyStats{Total}->{InAdded},
			$InPercentAcc,
			$InVolume,
			$InPercentVol,
			$HourlyStats{Total}->{InChk},
			$HourlyStats{Total}->{InIhave},
			$HourlyStats{Total}->{InSpam},
			$HourlyStats{Total}->{InRej},
			$HourlyStats{Total}->{InErr};
	} else {
		printf FILE "<tr>\n%s%s%s%s %s%s%d%s %s%s%3.2f%s %s%s%s%s %s%s%3.2f%s %s%s%d%s %s%s%d%s %s%s%d%s %s%s%d%s %s%s%d%s\n",
			$TDC, "<b>", "Total", "</b>",
			$TDR, "<b>", $HourlyStats{Total}->{InAdded}, "</b>",
			$TDR, "<b>", $InPercentAcc, "</b>",
			$TDR, "<b>", $InVolume, "</b>",
			$TDR, "<b>", $InPercentVol, "</b>",
			$TDR, "<b>", $HourlyStats{Total}->{InChk}, "</b>",
			$TDR, "<b>", $HourlyStats{Total}->{InIhave}, "</b>",
			$TDR, "<b>", $HourlyStats{Total}->{InSpam}, "</b>",
			$TDR, "<b>", $HourlyStats{Total}->{InRej}, "</b>",
			$TDR, "<b>", $HourlyStats{Total}->{InErr}, "</b>";
	}

	&PrintTableFooter ("");

	&PrintUrlGIF ($FileGIF);
}


sub PrintOutgoingFeeds
{
	my ($Title, $ListByVolume) = @_;
	my ($Key, $Comment, $OutCons, $OutSecs, $OutDup, $OutRej, $OutAcc, $OutBytes);
	my ($Num, $OutPercentAcc, $OutSubPercentAcc, $OutPercentVol, $OutSubPercentVol); 
	my ($SubTotal1, $SubTotal2, $SubTotal3, $SubTotal4, $SubTotal5, $SubTotal6);
	my ($Field1, $Field2, $Field3, $Field4, $Field5, $Field6);
	my ($Total1, $Total2, $Total4, $Total5, $total6);
	my ($Format, $SubFormat, $FileGIF);

	&PrintTableHeader ($Title);

	if ($ListByVolume) {
		my ($Total3) = new Math::BigFloat 0;
		%OrderedList = %OrderByOutBytes;
		$Field1 = "Volume"; 
		$Field2 = "%Vol";
		$Field3 = "Kbps";
		$Field4 = "Accepted"; 
		$Field5 = "%Total";
		$Field6 = "Kb/art";
		$AscFormat = "%8s %3.2f %3.2f %8d %3.2f %3.2f";
		$SubFormat = "%s%s %s%3.2f %s%3.2f %s%d %s%3.2f %s%3.2f";
		$Format = "%s%s%s%s %s%s%3.2f%s %s%s%3.2f%s %s%s%d%s %s%s%3.2f%s %s%s%3.2f%s";
		$FileGIF = "$BegDate-out-vol.gif";
	} else {
		my ($Total3) = 0;
		%OrderedList = %OrderByOutAcc;
		$Field1 = "Accepted"; 
		$Field2 = "%Acc";
		$Field3 = "Art/sec";
		$Field4 = "Volume"; 
		$Field5 = "%Vol";
		$Field6 = "Kb/art";
		$AscFormat = "%8d %3.2f %3.2f %8s %3.2f %3.2f";
		$SubFormat = "%s%d %s%3.2f %s%3.2f %s%s %s%3.2f %s%3.2f";
		$Format = "%s%s%d%s %s%s%3.2f%s %s%s%3.2f%s %s%s%s%s %s%s%3.2f%s %s%s%3.2f%s";
		$FileGIF = "$BegDate-out-art.gif";
	}

	if ($TextFormat) {
		printf FILE "\n%-30s %8s %8s %8s %8s %8s %8s %8s %8s\n\n",
			"Outgoing Feed", $Field1, $Field2, $Field3, $Field4, $Field5, $Field6, "Dups", "Rejs";
	} else {
		print FILE "<th $TableColor colspan=3>Outgoing Feed (+ <a href=\"http://www.freenix.fr/top1000/\">Freenix #</a>)<th $TableColor>$Field1<th $TableColor>$Field2<th $TableColor>$Field3<th $TableColor>$Field4<th $TableColor>$Field5<th $TableColor>$Field6<th $TableColor>Dups<th $TableColor>Rejs\n";
	}

	$OutAcc = $OutBytes = 0;
	$OutSecs = 1;

	foreach $Key (reverse sort keys %OrderedList) {
		$Host = $OrderedList{$Key};

		if ($Feeds{$Host}->{OutAcc}) {

			$OutSecs += $Feeds{$Host}->{OutSecs};
			$OutDup += $Feeds{$Host}->{OutDup};
			$OutRej += $Feeds{$Host}->{OutRej};
			$OutAcc += $Feeds{$Host}->{OutAcc};
			$OutBytes += $Feeds{$Host}->{OutBytes};
		}
	}

	foreach $Key (reverse sort keys %OrderedList) {
		$Host = $OrderedList{$Key};

		if ($Feeds{$Host}->{OutSecs} && $Feeds{$Host}->{OutAcc}) {

			$OutSubPercentAcc = &GetPercent ($OutAcc, $Feeds{$Host}->{OutAcc});
			$OutPercentAcc += $OutSubPercentAcc;

			$OutSubPercentVol = &GetPercent ($OutBytes, $Feeds{$Host}->{OutBytes});
			$OutPercentVol += $OutSubPercentVol;

			if ($ListByVolume) {
				$SubTotal1 = &GetVolume ($Feeds{$Host}->{OutBytes});
				$SubTotal2 = $OutSubPercentVol;
				if ($WallTime) {
					$SubTotal3 = &GetKbps ($Feeds{$Host}->{OutBytes}, $ElapsedSecs);
				} else {
					$SubTotal3 = &GetKbps ($Feeds{$Host}->{OutBytes}, $Feeds{$Host}->{OutSecs});
					$Total3 += $SubTotal3;
				}
				$SubTotal4 = $Feeds{$Host}->{OutAcc};
				$SubTotal5 = $OutSubPercentAcc;
			} else {
				$SubTotal1 = $Feeds{$Host}->{OutAcc};
				$SubTotal2 = $OutSubPercentAcc;
				if ($WallTime) {
					$SubTotal3 = $Feeds{$Host}->{OutAcc} / $ElapsedSecs;
				} else {
					$SubTotal3 = $Feeds{$Host}->{OutAcc} / $Feeds{$Host}->{OutSecs};
					$Total3 += $SubTotal3;
				}
				$SubTotal4 = &GetVolume ($Feeds{$Host}->{OutBytes});
				$SubTotal5 = $OutSubPercentVol;
			}
			$SubTotal6 = &GetAvgArtSize ($Feeds{$Host}->{OutBytes}, $Feeds{$Host}->{OutAcc});

			if ($TextFormat) {
				printf FILE "%-30s $AscFormat %8d %8d\n",
					&GetHostLink ($Host),  
					$SubTotal1, 
					$SubTotal2, 
					$SubTotal3, 
					$SubTotal4,
					$SubTotal5,
					$SubTotal6,
					$Feeds{$Host}->{OutDup}, 
					$Feeds{$Host}->{OutRej};
			} else {
				printf FILE "<tr>\n%s%s%d%s %s%s %s&nbsp;%s $SubFormat %s%d %s%d\n",
					$TDR, "<b>", ++$Num, "</b>",
					$TDL, &GetHostLink ($Host),
					$TDR, &GetTop1000 ($Host),
					$TDR, $SubTotal1,
					$TDR, $SubTotal2,
					$TDR, $SubTotal3,
					$TDR, $SubTotal4,
					$TDR, $SubTotal5,
					$TDR, $SubTotal6,
					$TDR, $Feeds{$Host}->{OutDup}, 
					$TDR, $Feeds{$Host}->{OutRej};
			}
		}
	}

	if ($ListByVolume) {
		$Total1 = &GetVolume ($OutBytes);
		$Total2 = $OutPercentVol;
		if ($WallTime) {
			$Total3 = &GetKbps ($OutBytes, $ElapsedSecs);
		}
		$Total4 = $OutAcc;
		$Total5 = $OutPercentAcc;
	} else {
		$Total1 = $OutAcc;
		$Total2 = $OutPercentAcc;
		if ($WallTime) {
			$Total3 = $OutAcc / $ElapsedSecs;
		}
		$Total4 = &GetVolume ($OutBytes);
		$Total5 = $OutPercentVol;
	}
	$Total6 = &GetAvgArtSize ($OutBytes, $OutAcc);

	if ($TextFormat) {
		printf FILE "\n%s $AscFormat %8d %8d\n\n",
			"                              ", $Total1, $Total2, 
			$Total3, $Total4, $Total5, $Total6, $OutDup, $OutRej;
	} else {
		printf FILE "<tr>\n%s%s $Format %s%s%d%s %s%s%d%s\n",
			"<td align=\"center\" colspan=3>", "<b>Total</b>",
			$TDR, "<b>", $Total1, "</b>",
			$TDR, "<b>", $Total2, "</b>",
			$TDR, "<b>", $Total3, "</b>",
			$TDR, "<b>", $Total4, "</b>",
			$TDR, "<b>", $Total5, "</b>",
			$TDR, "<b>", $Total6, "</b>",
			$TDR, "<b>", $OutDup, "</b>",
			$TDR, "<b>", $OutRej, "</b>";
	}

	$Comment = "OUTGOING DUP=$OutDup REJ=$OutRej ACC=$OutAcc VOL=$OutBytes";

	&PrintTableFooter ($Comment);
}


sub PrintQueueStats
{
	my ($Title) = @_;
	my ($OutCons, $OutSecs, $OutDup, $OutRej, $OutAcc);
	my ($Key, $Color, $TDL, $TDR);

	if (! $TextFormat) {
		print FILE <<EOT
<table border=2 width=100%>
<td align="center" bgcolor=$QueColor1><b>&gt;= $QuePercent1% Full</b></td>
<td align="center" bgcolor=$QueColor2><b>&gt;= $QuePercent2% Full</b></td>
<td align="center" bgcolor=$QueColor3><b>&gt;= $QuePercent3% Full</b></td>
<td align="center" bgcolor=$QueColor4><b>&gt;= $QuePercent4% Full</b></td>
<td align="center" bgcolor=$QueColor5><b>&gt;= $QuePercent5% Full</b></td>
<td align="center" bgcolor=$QueColor6><b>&gt;= $QuePercent6% Full</b></td>
</table>
<p>
EOT
;
	}

	&PrintTableHeader ($Title);

	if ($TextFormat) {
		printf FILE "\n%-30s %19s %9s %9s %9s\n\n",
			"Outgoing Feed", "Batch Seq", 
			"Batch Num", "Batch Max", "%Full";
	} else {
		print FILE "<th $TableColor>Outgoing Feed<th $TableColor>Batch Seq<th $TableColor>Batch Num<th $TableColor>Batch Max<th $TableColor>%Full\n";
	}

	foreach $Key (reverse sort keys %OrderByBatchFull) {
		$Host = $OrderByBatchFull{$Key};

		if ($Feeds{$Host}->{BatchMax}) {

			if ($TextFormat) {
				printf FILE "%-30s %9d-%9d %9d %9d %9d\n",
				        &GetQueueHost ($Host),
					$Feeds{$Host}->{BatchBeg}, 
					$Feeds{$Host}->{BatchEnd},
					$Feeds{$Host}->{BatchNum},
					$Feeds{$Host}->{BatchMax},
					$Feeds{$Host}->{BatchFull};
			} else {
				if ($Feeds{$Host}->{BatchFull} >= $QuePercent6) {
					$Color = "bgcolor=$QueColor6";
				} elsif ($Feeds{$Host}->{BatchFull} >= $QuePercent5) {
					$Color = "bgcolor=$QueColor5";
				} elsif ($Feeds{$Host}->{BatchFull} >= $QuePercent4) {
					$Color = "bgcolor=$QueColor4";
				} elsif ($Feeds{$Host}->{BatchFull} >= $QuePercent3) {
					$Color = "bgcolor=$QueColor3";
				} elsif ($Feeds{$Host}->{BatchFull} >= $QuePercent2) {
					$Color = "bgcolor=$QueColor2";
				} elsif ($Feeds{$Host}->{BatchFull} >= $QuePercent1) {
					$Color = "bgcolor=$QueColor1";
				} else {
					$Color = "";
				}
				$TDL = "<td $Color>";
				$TDR = "<td $Color align=\"right\">";

				printf FILE "<tr>\n%s%s %s%d-%d %s%d %s%d %s%d\n",
				        $TDL, &GetQueueHost ($Host),
					$TDR, $Feeds{$Host}->{BatchBeg}, $Feeds{$Host}->{BatchEnd},
					$TDR, $Feeds{$Host}->{BatchNum},
					$TDR, $Feeds{$Host}->{BatchMax},
					$TDR, $Feeds{$Host}->{BatchFull};
			}
		}
	}

	&PrintTableFooter ("");
}


sub PrintCuriousActivity
{
	my ($Title) = @_;
	my ($Key, $BegTime, $EndTime);

	&PrintTableHeader ($Title);

	if ($TextFormat) {
		printf FILE "\n%s %s %-8s %50s\n\n",
			"1st Time", "Last Time", "# Msgs", "Message";
	} else {
		print FILE "<th $TableColor>1st Time<th $TableColor>Last Time<th $TableColor># Msgs<th $TableColor>Message\n";
	}

	foreach $Key (reverse sort keys %OrderByErrorTotal) {
		$Msg = $OrderByErrorTotal{$Key};
		
		if ($ErrorList{$Msg}->{Tot}) {

			$BegTime = $EndTime = "";
			$BegTime = $1 if ($ErrorList{$Msg}->{BegDate} =~ /(..:..:..)/);
			$EndTime = $1 if ($ErrorList{$Msg}->{EndDate} =~ /(..:..:..)/);

			if ($TextFormat) {
				printf FILE "%s %s %8d %-50s\n",
					$BegTime, $EndTime, $ErrorList{$Msg}->{Tot}, $Msg;
			} else {
				printf FILE "<tr>\n%s%s %s&nbsp;%s %s%d %s%s\n",
					$TDL, $BegTime,
					$TDL, $EndTime,
					$TDR, $ErrorList{$Msg}->{Tot},
					$TDL, $Msg;
			}
		}
	}

	&PrintTableFooter ("");
}


sub PrintSpamActivity
{
	my ($Title) = @_;
	my ($Key, $Time, $Num, $Host, $ByDups, $ByRate, $Total);

	&PrintTableHeader ($Title);

	if ($TextFormat) {
		printf FILE "\n%-50s %8d %8d %8d\n\n",
			"Host", "By Post Rate", "By Dup Body", "# Articles";
	} else {
		print FILE "<th $TableColor>Host<th $TableColor>By Post Rate<th $TableColor>By Dup Body<th $TableColor># Articles\n";
	}

	$Num = 0;
	foreach $Key (reverse sort keys %OrderBySpamTotal) {
		if ($Num < $MaxSpamList) {

			$Host = $OrderBySpamTotal{$Key};
	
			$Total = $SpamList{$Host}->{Total};
			$ByRate = $SpamList{$Host}->{ByRate};
			$ByDups = $SpamList{$Host}->{ByDups};
#			print "SPAM=[$Key] $Host\n";

			if ($Total) {
				if ($TextFormat) {
					printf FILE "%-50s %8d %8d %8d\n",
						$Host, $ByRate, $ByDups, $Total; 
				} else {	
					printf FILE "<tr>\n%s&nbsp;%s %s%d %s%d %s%d\n",
						$TDL, $Host,
						$TDR, $ByRate,
						$TDR, $ByDups,
						$TDR, $Total;
				}
			}
			$Num++;
		} else {
			goto SpamDone;
		}
	}
SpamDone:

	&PrintTableFooter ("");
}


sub PrintTableHeader
{
	my ($Title) = @_;

	if ($TextFormat) {
		print FILE "$Title\n";
	} else {
		print FILE <<EOT
<table border=2 width=100%>
<caption><h3>$Title</h3></caption>
EOT
;
	}
}


sub PrintTableFooter
{
	my ($Comment) = @_;

	if (! $TextFormat) {
		print FILE "</table>\n";
		print FILE "<!-- $Comment -->\n" if $Comment ne "";
	}
}


sub PrintOutHeader
{
	my ($TimePeriod) = $BegDate;
	my ($Title, $Title2);

	if ($BegDate ne $EndDate) {
		$TimePeriod = "$BegDate - $EndDate";
	} else {
		$TimePeriod = &GetFancyNameDate ($BegDate, 1, 0);
	}

	$Title = "DIABLO statistics for '$DiabloHost' for $TimePeriod";
	$Title2 = "<a href=$DiabloUrl>DIABLO</a>  statistics for '$DiabloHost' on $TimePeriod";

	if ($TextFormat) {
		print FILE "$Title\n\n\n";
	} else {
		print FILE <<EOT
<html>
<head>
<title>$Title</title>
<meta http-equiv="Refresh" content="$MetaRefresh">
<meta http-equiv="Expires" content="$MetaExpires">
</head>
<body $BgColor>
<a name="00">
<table>
<tr>
<td><a href="$WwwUrl"> <img $ImgInfo alt="$ImgText" src="$ImgLogo"></a></td>
<td><h3>$Title2</h3></td>
</tr>
</table>
EOT
;
		&PrintOutCopyright;
		print FILE <<EOT
<hr>
<p>
<ol>
<li><a href="#01">Summary of Incoming Feeds by Article</a>
<li><a href="#02">Summary of Incoming Feeds by Volume</a>
<li><a href="#03">Summary of Incoming Feeds by Time</a>
<li><a href="#04">Summary of Outgoing Feeds by Article</a>
<li><a href="#05">Summary of Outgoing Feeds by Volume</a>
<li><a href="#06">Summary of Outgoing Queue</a>
<li><a href="#07">Summary of Curious Activity</a>
<li><a href="#08">Summary of SPAM Activity (Top $TopSpamList of $TotalSpamList)</a>
</ol>
EOT
;
		chop (($OutputUptime = `$CmdUptime`));
		chop (($OutputDf = `$CmdDf`));
		chop (($OutputDiablo = `$CmdDiablo`));
		chop (($OutputDnewslink = `$CmdDnewslink`));
		$OutputDiablo =~ s/ //g;
		$OutputDnewslink =~ s/ //g;

		print FILE <<EOT
<hr>
<p>
<pre>
Diablo running processes:  $OutputDiablo    Dnewslink running processes:  $OutputDnewslink

$DiabloUptime

$OutputUptime

$OutputDf
</pre>
EOT
;

	}
}


sub PrintOutFooter
{
	&PrintOutCopyright;

	if (! $TextFormat) {
		print FILE "</body>\n</html>\n";
	}
}


sub PrintOutCopyright
{
	if ($TextFormat) {
		print FILE "\nGenerated on $GenDate $GenTime by $ScriptName $Version. Copyright (C) 1997 Iain Lea (iain\@bricbrac.de)\n";
	} else {
		print FILE <<EOT
Generated on $GenDate $GenTime by 
<a href="$FtpUrl">$ScriptName $Version</a>. 
Copyright &copy; 1997 
<a href="mailto:iain\@bricbrac.de">Iain Lea</a>. 
EOT
;
	}
}


sub GetYyMmDd
{
	my ($Date) = @_;

	# Apr 10 11:36:54
	if ($Date =~ /^(...) (..)/) {
		return sprintf "%02d%02d%02d", $GenYear, $MonthNumByName{$1}, $2;
	} else {
		return "";
	}
}


sub GetHhMmSs
{
	my ($Secs) = @_;
#	my ($Time) = "00:00:00";
	my ($Time) = "00:00";
	my ($Hh, $Mm, $Ss);

	if ($Secs) {
		$Hh = int($Secs / 3600);
		$Mm = int(($Secs - ($Hh * 3600)) / 60);
		$Ss = $Secs % 60;

#		$Time = sprintf "%02d:%02d:%02d", $Hh, $Mm, $Ss;
		$Time = sprintf "%02d:%02d", $Hh, $Mm;
	}

#	print "Secs=[$Secs] Time=[$Time]\n";

	return $Time;
}


sub GetFancyDate
{
	my ($Date) = @_;

	if (/^(..)(..)(..)/) {
		$Date = sprintf "%s %s %s", $3, $MonthNameByNum{$2}, $1;
	}

	return $Date;
}


# $Date = &GetFancyNameDate ("960109", [0|1], [0|1]);
# par2: 0 = Long / 1 = Short form of dayname ie. Monday / Mon
# par3: 0 = not split / 1 = split returned line ie. Mon\n14 Oct 95

sub GetFancyNameDate 
{
	local ($CurDate, $ShortName, $SplitLine) = @_;
	local ($Break, $DayName);

	if ($SplitLine) { 
		$Break = '<br>'; 
	} else { 
		$Break = ' '; 
	}
	$CurDate =~ /^(..)(..)(..)/;
	$DayName = &GetDayOfWeek ($1, $2, $3, $ShortName);

	return "$DayName$Break$3 $MonthNameByNum{$2} $1";
}


sub GetDayOfWeek 
{
	# Parameters: YY, MM, DD, [0|1]
	local($Yy, $Mm, $Dd, $ShortName) = @_;
	local($DayNum);
    
	$DayNum = (localtime(timelocal(0,0,0,$Dd,$Mm-1,$Yy,0,0)))[6];

	if ($ShortName) { 
		return $DayNameShort[$DayNum]; 
	} else { 
		return $DayName[$DayNum]; 
	}
}


sub GetDailyStats
{
	my ($Date) = @_;
	my ($InErr, $InRej, $InSpam, $InTot, $InVol, $OutDup, $OutRej, $OutAcc, $OutVol);

	$InErr = $InRej = $InSpam = $InTot = $InVol = $OutDup = $OutRej = $OutAcc = $OutVol = 0;

	if (open (STATS, "$Date.html")) {
		while (<STATS>) {
			if (/INCOMING ERR=([\ 0-9]*) REJ=([\ 0-9]*) SPAM=([\ 0-9]*) ADD=([\ 0-9]*) VOL=([\ 0-9]*)/) {
				$InErr = $1;
				$InRej = $2;
				$InSpam = $3;
				$InTot = $4;
				$InVol = $5;
			} elsif (/INCOMING ERR=([\ 0-9]*) REJ=([\ 0-9]*) ADD=([\ 0-9]*) VOL=([\ 0-9]*)/) {
				$InErr = $1;
				$InRej = $2;
				$InTot = $3;
				$InVol = $4;
			} elsif (/INCOMING ERR=([\ 0-9]*) REJ=([\ 0-9]*) ADD=([\ 0-9]*)/) {	# < v1.10
				$InErr = $1;
				$InRej = $2;
				$InTot = $3;
			}
			if (/OUTGOING DUP=([\ 0-9]*) REJ=([\ 0-9]*) ACC=([\ 0-9]*) VOL=([\ 0-9]*)/) {
				$OutDup = $1;
				$OutRej = $2;
				$OutAcc = $3;
				$OutVol = $4;
			} elsif (/OUTGOING DUP=([\ 0-9]*) REJ=([\ 0-9]*) ACC=([\ 0-9]*)/) {	# < v1.10
				$OutDup = $1;
				$OutRej = $2;
				$OutAcc = $3;
			}
		}
		close STATS;
	} else {
		warn "Error: $Date.html - $!\n";
	}
	
	return ($InErr, $InRej, $InTot, $InVol, $OutDup, $OutRej, $OutAcc, $OutVol, $InSpam);
}


sub GetPercent
{
	my ($Total, $SubTotal) = @_;

	return ($SubTotal && $Total) ? ($SubTotal / $Total) * 100 : 0; 	
}


sub AddToInTimeList
{
	my ($Date, $Host, $InSecs, $InAdded, $InBytes, $InChk, $InIhave, $InSpam, $InRej, $InErr) = @_;

# print "IN:  $Host, $InSecs, $InAdded, $InBytes, $InChk, $InIhave, $InSpam, $InRej, $InErr\n";

	# ... 23:59:59
	if ($Date =~ / (..):..:../) {
		$Hour = $1;
		$Feeds{$Host}->{$Hour}->{InAdded} += $InAdded;
		$Feeds{$Host}->{$Hour}->{InBytes} += $InBytes;
		$Feeds{$Host}->{$Hour}->{InChk} += $InChk;
		$Feeds{$Host}->{$Hour}->{InIhave} += $InIhave;
		$Feeds{$Host}->{$Hour}->{InSpam} += $InSpam;
		$Feeds{$Host}->{$Hour}->{InRej} += $InRej;
		$Feeds{$Host}->{$Hour}->{InErr} += $InErr;
		if ($InAdded || $InBytes) {
			$Feeds{$InHost}->{$Hour}->{InSecs} += $InSecs;
		} 

# print "IN: $Hour  $Host  $Feeds{$Host}->{$Hour}->{InSecs}  $Feeds{$Host}->{$Hour}->{InAdded}  $Feeds{$Host}->{$Hour}->{InBytes}\n";
	}
}


sub AddToOutTimeList
{
	my ($Date, $Host, $OutSecs, $OutAcc, $OutBytes, $OutDup, $OutRej) = @_;

	# ... 23:59:59
	if ($Date =~ / (..):..:../) {
		$Hour = $1;
		$Feeds{$Host}->{$Hour}->{OutAcc} += $OutAcc;
		$Feeds{$Host}->{$Hour}->{OutBytes} += $OutBytes;
		$Feeds{$Host}->{$Hour}->{OutDup} += $OutDup;
		$Feeds{$Host}->{$Hour}->{OutRej} += $OutRej;
		$Feeds{$Host}->{$Hour}->{OutSecs} += $OutSecs;
	}
}


sub AddToErrorList
{
	my ($Error, $FullDate, $LineDate) = @_;

	$ErrorList{$Error}->{Tot} += 1;
	if ($ErrorList{$Error}->{BegDate} eq "") {
		$ErrorList{$Error}->{BegDate} = $FullDate;
	} else {
		$ErrorList{$Error}->{EndDate} = $FullDate;
	}

	print "$LineDate  $Error\n" if $Debug;
}


sub AddToSpamList
{
	my ($Spam, $FullDate, $LineDate) = @_;
	my ($Num, $Host, $MsgId, $ByDups, $ByRate, $ByFeed);

	$ByDups = $ByRate = $ByFeed = 0;

	# ... by-post-rate copy #33: <5rplh4$12p@ron.ipa.com> 206.1.57.29
	# ... by-dup-body copy #1: <5rpvin$8ro$1@telemedia.de> essn.mediaways.net
	# ... dnewsfeeds copy #-1: <5r4dpg$le8$1@news.utu.fi> apus.astro.utu.fi
	if ($Spam =~ /by-post-rate copy #([\-0-9]*):\s+([A-z0-9\.\-_\<>\$@]*)\s+([A-z0-9\.\-_]*)/) {
		$Num = $1;
		$MsgId = $2;
		$Host = $3;
		$ByRate = 1;
	} elsif ($Spam =~ /by-dup-body copy #([\-0-9]*):\s+([A-z0-9\.\-_\<>\$@]*)\s+([A-z0-9\.\-_]*)/) {
		$Num = $1;
		$MsgId = $2;
		$Host = $3;
		$ByDups = 1;
	} elsif ($Spam =~ /dnewsfeeds copy #([\-0-9]*):\s+([A-z0-9\.\-_\<>\$@]*)\s+([A-z0-9\.\-_]*)/) {
		$Num = $1;
		$MsgId = $2;
		$Host = $3;
		$ByFeed = 1;
	}
	if ($ByRate || $ByDups || $ByFeed) {
		$SpamList{$Host}->{Total} += 1;
		$SpamList{$Host}->{ByRate} += $ByRate;
		$SpamList{$Host}->{ByDups} += $ByDups;
		$SpamList{$Host}->{ByFeed} += $ByFeed;

		print "SPAM:  Type=$ByDups ($SpamList{$Host}->{ByDups})| $ByRate ($SpamList{$Host}->{ByRate})| $ByFeed ($SpamList{$Host}->{ByFeed})  MsgID=$MsgId  Site=$Host\n" if $Debug;
	}
}


sub GetVolume 
{
	# This is disgusting, but fcmp does lexical compares of floats expressed in
	# scientific notation, which is completely useless... If I knew more Perl I
	# could probably compute <X>Int from <X> instead of having duplicate defines.

#print "GetVolume(@_)\n";

	$_[0] += 0;		# In case $_[0] is empty

	my ($Bytes) = new Math::BigFloat @_;
	my ($BytesInt) = new Math::BigInt @_;
	my ($Vol) = new Math::BigFloat "0.0";
	my ($Gb) = new Math::BigFloat "1073741824.0";
	my ($GbInt) = new Math::BigInt "1073741824";
	my ($Mb) = new Math::BigFloat "1048576.0";
	my ($MbInt) = new Math::BigInt "1048576";
	my ($Kb) = new Math::BigFloat "1024.0";
	my ($KbInt) = new Math::BigInt "1024";

#print "Bytes=[$Bytes]  BytesInt=[$BytesInt]";

	if ($BytesInt > $GbInt) {
		$Sign = "GB";
		$Vol = $Bytes / $Gb;
	} elsif ($BytesInt > $MbInt) {
CalcMb:
		$Sign = "MB";
		$Vol = $Bytes / $Mb;
	} else {
CalcKb:
		$Sign = "KB";
		$Vol = $Bytes / $Kb;
	}

#printf "Num=$Bytes  Format(%g)=%3.2f%s\n", $Vol, $Vol, $Sign;

	$Vol = sprintf "%3.2f%s", $Vol, $Sign;

	return $Vol;
}


sub GetKbps
{
	$_[0] += 0; # In case @_[0] bytes is empty

	my ($KBytes) = new Math::BigFloat ($_[0] / 1024);
	my ($Secs) = new Math::BigFloat $_[1];
	my ($Kbps) = new Math::BigFloat (($KBytes * 8) / $Secs);

#print "KBytes=[$KBytes]  Secs=[$Secs]  Kbps=[$Kbps]\n";

	return $Kbps;
}


sub GetAvgArtSize
{
	$_[0] += 0; # In case @_[0] bytes is empty

	my ($KBytes) = new Math::BigFloat ($_[0] / 1024);
	my ($NumArts) = $_[1];
	my ($Size);

	if ($NumArts) {
		$Size = sprintf "%3.2f", $KBytes / $NumArts;
	} else {
		$Size = sprintf "%3.2f", 0;
	}

# print "NEW:  KBytes=[$KBytes]  NumArts=[$NumArts]  KbSize=[$Size]\n";

	return $Size;
}


sub GetQueueHost
{
	my ($QHost) = @_;
	my ($QLine, $First, $Second, $Rest);

	$QHost =~ s/[ \t]+//;

	if ($QueueHost) {
		open (QFILE, $CtlFile) || die "Error: $CtlFile - $!\n";

		do {
			$QLine = <QFILE>;
			chop ($QLine);
			($First, $Second, $Rest) = split(/[ \t]+/, $QLine, 3);

		} until ($First eq $QHost || eof);

		close (QFILE);

		if ($Second eq '') {
			$Rest = $QHost;
		}
		$Rest = $Second.'  ('.$QHost.')';

		return $Rest;
	} else {
		return $QHost;
	}
}


sub MakeVolKey                                 
{
	my ($Bytes, $Num) = @_;
                
	until (length($Bytes) > 10) {                                        
		$Bytes = " " . $Bytes;
	}
	until (length($Num) > 2) {
		$Num = " " . $Num;                                   
	}
                
	return "$Bytes.$Num";                                                
}


sub GetHostLink
{
	my ($Host) = @_;
	my ($Link, $HostLink, $Pos1000);

	$HostLink = $Host;
	if ($HostLink) {
		$Link = $HostLinkList{$Host};

		if ($Link) {
			$HostLink = sprintf ("<a href=\"%s\">%s</a>", $Link, $Host);
		}
		print "HOST:  Host=[$Host]  Link=[$Link]  HostLink=[$HostLink]\n" if $Debug;
	}

	return $HostLink;
}


sub ReadCfgFile
{
	if (open (FILE, $CfgFile)) {
		while (<FILE>) {
			if (/^link:\s+([A-z0-9_\.\/\-\~:]*)\s+([A-z0-9_\.\/\-\~:]*)\n/) {
				$Host = $1;
				$Link = $2;
				$HostLinkList{$Host} = $Link;
				print "LINK:  [$Host]==[$HostLinkList{$Host}]\n" if $Debug;
			}
			if (/^top1000:\s+([A-z0-9_\.\/\-\~:]*)\s+([A-z0-9_\.\/\-\~:]*)\n/) {
				$Host = $1;
				$TopHost = $2;
				$Top1000Host{$Host} = $TopHost;
				print "TOP1000:  [$Host]==[$Top1000Host{$Host}]\n" if $Debug;
			}
		}
		close FILE;
	}
}


sub CalcHourlyStats
{
	my ($Num, $Hour);

	print "Calculating hourly statistics ...\n";

	for ($Num = 0; $Num < 24; $Num++) {
		$Hour = sprintf ("%02d", $Num);

		foreach $Host (keys %Feeds) {
			# Incoming feeds
			$HourlyStats{$Hour}->{InAdded} += $Feeds{$Host}->{$Hour}->{InAdded};
			$HourlyStats{$Hour}->{InBytes} +=  $Feeds{$Host}->{$Hour}->{InBytes};
			$HourlyStats{$Hour}->{InChk} += $Feeds{$Host}->{$Hour}->{InChk};
			$HourlyStats{$Hour}->{InIhave} += $Feeds{$Host}->{$Hour}->{InIhave};
			$HourlyStats{$Hour}->{InSpam} += $Feeds{$Host}->{$Hour}->{InSpam};
			$HourlyStats{$Hour}->{InRej} += $Feeds{$Host}->{$Hour}->{InRej};
			$HourlyStats{$Hour}->{InErr} += $Feeds{$Host}->{$Hour}->{InErr};

			# reset if multiple connections have more than 1 hour walltime
			if ($Feeds{$Host}->{$Hour}->{InSecs} > 3600) {
# print "IN  SECS: [$Hour]  [$Host]  Secs=[$Feeds{$Host}->{$Hour}->{InSecs}]\n";
				$Feeds{$Host}->{$Hour}->{InSecs} = 3600;
			}
			$Feeds{$Host}->{InSecs} += $Feeds{$Host}->{$Hour}->{InSecs};

			# Outgoing feeds
			$HourlyStats{$Hour}->{OutAdded} += $Feeds{$Host}->{$Hour}->{OutAcc};
			$HourlyStats{$Hour}->{OutBytes} += $Feeds{$Host}->{$Hour}->{OutBytes};
			$HourlyStats{$Hour}->{OutDup} += $Feeds{$Host}->{$Hour}->{OutDup};
			$HourlyStats{$Hour}->{OutRej} += $Feeds{$Host}->{$Hour}->{OutRej};

			# reset if multiple connections have more than 1 hour walltime
			if ($Feeds{$Host}->{$Hour}->{OutSecs} > 3600) {
# print "OUT SECS: [$Hour]  [$Host]  Secs=[$Feeds{$Host}->{$Hour}->{OutSecs}]\n";
				$Feeds{$Host}->{$Hour}->{OutSecs} = 3600;
			}
			$Feeds{$Host}->{OutSecs} += $Feeds{$Host}->{$Hour}->{OutSecs};

# print "HOST: [$Hour]  [$Host]  [$HourlyStats{$Hour}->{InAdded}]  [$Feeds{$Host}->{$Hour}->{InAdded}]\n";
		}
		$HourlyStats{Total}->{InAdded} += $HourlyStats{$Hour}->{InAdded};
		$HourlyStats{Total}->{InBytes} += $HourlyStats{$Hour}->{InBytes};
		$HourlyStats{Total}->{InChk} += $HourlyStats{$Hour}->{InChk};
		$HourlyStats{Total}->{InIhave} += $HourlyStats{$Hour}->{InIhave};
		$HourlyStats{Total}->{InSpam} += $HourlyStats{$Hour}->{InSpam};
		$HourlyStats{Total}->{InRej} += $HourlyStats{$Hour}->{InRej};
		$HourlyStats{Total}->{InErr} += $HourlyStats{$Hour}->{InErr};

#		printf "HOUR: [$Hour]  Acc=%d Vol=%ld Chk=%d Ihave=%d Spam=%d Rejs=%d Errs=%d\n",
#			$HourlyStats{$Hour}->{InAdded}, $HourlyStats{$Hour}->{InBytes},
#			$HourlyStats{$Hour}->{InChk}, $HourlyStats{$Hour}->{InIhave},
#			$HourlyStats{$Hour}->{InSpam}, $HourlyStats{$Hour}->{InRej},
#			$HourlyStats{$Hour}->{InErr};
	}
#	printf "TOTAL:  Acc=%d Vol=%ld Chk=%d Ihave=%d Spam=%d Rejs=%d Errs=%d\n",
#		$HourlyStats{Total}->{InAdded}, $HourlyStats{Total}->{InBytes},
#		$HourlyStats{Total}->{InChk}, $HourlyStats{Total}->{InIhave},
#		$HourlyStats{Total}->{InSpam}, $HourlyStats{Total}->{InRej},
#		$HourlyStats{Total}->{InErr};

}


sub MakeIncomingByTimeGIF
{
	my ($File, $Title, $WidthGIF, $HeightGIF) = @_;
	my ($Num, $Hour, $Graph);
	my (@LegendLabels) = ('Accepted', 'Spam', 'Rejected', 'Errors');
	my (@Data1) = (
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef);
	my (@Data2) = (
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef);
	my (@Data3) = (
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef);
	my (@Data4) = (
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef);
	my ($CurrHourForGIF) = &GetCurrHourForGIF ($CurrHour, $CurrMin);

	$Graph = Chart::Lines->new ($WidthGIF, $HeightGIF);

	$Graph->set ('x_label' => $Title);
	$Graph->set ('transparent' => 'true');
	$Graph->set ('grid_lines' => 'true');
	$Graph->set ('legend_placement' => 'bottom');
	$Graph->add_dataset (@XLabelHours);

	$Data1[0] = $Data2[0] = $Data3[0] = $Data4[0] = 0;	# 1 off error

	for ($Num = 1; $Num <= $CurrHourForGIF; $Num++) {
		$Hour = sprintf ("%02d", $Num - 1);

		if ($HourlyStats{$Hour}->{InAdded}) {
			$Data1[$Num] = $HourlyStats{$Hour}->{InAdded};
		} else {
			$Data1[$Num] = 0;
		}
		if ($HourlyStats{$Hour}->{InSpam}) {
			$Data2[$Num] = $HourlyStats{$Hour}->{InSpam};
		} else {
			$Data2[$Num] = 0;
		}
		if ($HourlyStats{$Hour}->{InRej}) {
			$Data3[$Num] = $HourlyStats{$Hour}->{InRej};
		} else {
			$Data3[$Num] = 0;
		}
		if ($HourlyStats{$Hour}->{InErr}) {
			$Data4[$Num] = $HourlyStats{$Hour}->{InErr};
		} else {
			$Data4[$Num] = 0;
		}
	}

	$Graph->add_dataset (@Data1);
	$Graph->add_dataset (@Data2);
	$Graph->add_dataset (@Data3);
	$Graph->add_dataset (@Data4);
	$Graph->set ('legend_labels' => \@LegendLabels);

	print "TIME GIF=$File\n" if $Verbose;

	$Graph->gif ($File);
}


sub MakeIncomingByArticleGIF
{
	my ($File, $Title, $WidthGIF, $HeightGIF) = @_;
	my (@Data, $Num, $Hour, $Graph, $Key, $Host, $NumLinesGIF) = 0;
	my ($CurrHourForGIF) = &GetCurrHourForGIF ($CurrHour, $CurrMin);
	my (@Total) = (
		0, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef);

	$Graph = Chart::Lines->new ($WidthGIF, $HeightGIF);
	$Graph->set ('x_label' => $Title);
	$Graph->set ('transparent' => 'true');
	$Graph->set ('grid_lines' => 'true');
	$Graph->set ('legend_placement' => 'bottom');
	$Graph->add_dataset (@XLabelHours);

	foreach $Key (reverse sort keys %OrderByInAdded) {
		$Host = $OrderByInAdded{$Key};

		if ($NumLinesGIF < $MaxLinesGIF) {
			$LegendLabels[$NumLinesGIF] = $Host;
			@Data = (
				0, undef, undef, undef, undef, undef, undef, undef,
				undef, undef, undef, undef, undef, undef, undef, undef,
				undef, undef, undef, undef, undef, undef, undef, undef);
			for ($Num = 1; $Num <= $CurrHourForGIF; $Num++) {
				$Hour = sprintf ("%02d", $Num - 1);
	
				if ($Feeds{$Host}->{$Hour}->{InAdded}) {
					$Data[$Num] = $Feeds{$Host}->{$Hour}->{InAdded};
				} else {
					$Data[$Num] = 0;
				}
#				$Total[$Num] += $Data[$Num];
# printf STDERR "TEST [$Host] [$Hour] [$Feeds{$Host}->{$Hour}->{InAdded}]\n";
			}
# printf STDERR "HOST: [%s]  [%s]\n", $Host, join (" ", @Data);
			if ($Feeds{$Host}->{InAdded}) {
				$Graph->add_dataset (@Data);
				$NumLinesGIF++;
			}
		} else {
			goto IncomingByArticleDone;
		}
#		$Total[$Num] += $Feeds{$Host}->{$Hour}->{InAdded};
	}
IncomingByArticleDone:

	for ($Num = 1; $Num <= $CurrHourForGIF; $Num++) {
		$Hour = sprintf ("%02d", $Num - 1);
		$Total[$Num] = $HourlyStats{$Hour}->{InAdded};
	}

	$LegendLabels[$NumLinesGIF] = "TOTAL (all feeds)";
	$Graph->set ('legend_labels' => \@LegendLabels);
	$Graph->add_dataset (@Total);

	print "ARTS GIF=$File\n" if $Verbose;

	$Graph->gif ($File);
}


sub MakeIncomingByVolumeGIF
{
	my ($File, $Title, $WidthGIF, $HeightGIF) = @_;
	my (@Data, $Num, $Hour, $Graph, $Key, $Host, $NumLinesGIF) = 0;
	my (@Total) = (
		0, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef,
		undef, undef, undef, undef, undef, undef, undef, undef);
	my ($CurrHourForGIF) = &GetCurrHourForGIF ($CurrHour, $CurrMin);

	$Graph = Chart::Lines->new ($WidthGIF, $HeightGIF);
	$Graph->set ('x_label' => $Title);
	$Graph->set ('transparent' => 'true');
	$Graph->set ('grid_lines' => 'true');
	$Graph->set ('legend_placement' => 'bottom');
	$Graph->add_dataset (@XLabelHours);

	foreach $Key (reverse sort keys %OrderByInBytes) {
		$Host = $OrderByInBytes{$Key};

		if ($NumLinesGIF < $MaxLinesGIF) {
			$LegendLabels[$NumLinesGIF] = $Host;
			@Data = (
				0, undef, undef, undef, undef, undef, undef, undef,
				undef, undef, undef, undef, undef, undef, undef, undef,
				undef, undef, undef, undef, undef, undef, undef, undef);
			for ($Num = 1; $Num <= $CurrHourForGIF; $Num++) {
				$Hour = sprintf ("%02d", $Num - 1);
	
				if ($Feeds{$Host}->{$Hour}->{InBytes}) {
					$Data[$Num] = $Feeds{$Host}->{$Hour}->{InBytes} / 1024;  # KB
				} else {
					$Data[$Num] = 0;
				}
				$Total[$Num] += $Data[$Num];

# printf STDERR "TEST [$Host] [$Hour] [$Feeds{$Host}->{$Hour}->{InBytes}]\n";
			}
# printf STDERR "HOST: [%s]  [%s]\n", $Host, join (" ", @Data);
			if ($Feeds{$Host}->{InAdded}) {
				$Graph->add_dataset (@Data);
				$NumLinesGIF++;
			}
		} else {
			goto IncomingByVolumeDone;
		}
	}
IncomingByVolumeDone:

	$LegendLabels[$NumLinesGIF] = "TOTAL (all feeds)";
	$Graph->set ('legend_labels' => \@LegendLabels);
	$Graph->add_dataset (@Total);

	print "VOL GIF=$File\n" if $Verbose;

	$Graph->gif ($File);
}


sub PrintUrlGIF
{
	my ($FileGIF) = @_;

	if ($GraphMode) {
		print FILE <<EOT
<p>
<table border=2 align=center>
<tr>
<td><img src="$FileGIF" width=$WidthGIF height=$HeightGIF>
</table>
<br>
EOT
;
	}
}


sub ParseTop1000
{
	my ($File) = "$Top1000Dir/current";
	my ($Index);

	print "FILE: $File\n" if $Debug;

	if (open (FILE, $File)) {
		print "Parsing Freenix Top1000 file $File ...\n" if $Verbose;
		while (<FILE>) {
			if (/^\s+(\d+)\s+[0-9\.]*\s+(.*)/o) {
				print "$1  $2\n" if $Debug;
 				$Index = lc $2;
 				$Top1000List{$Index} = $1;
			}
		}
		close (FILE);
	}
}


sub GetTop1000
{
	my ($Host) = @_;
	my ($TopHost, $Pos1000) = "";

	$Host = lc $Host;
	$TopHost = $Top1000Host{$Host} ? $Top1000Host{$Host} : $Host;

	if ($Top1000List{$TopHost}) {
		$Pos1000 = "#$Top1000List{$TopHost}";
		print "POS: $Host  $TopHost  $Pos1000\n" if $Debug;
	}

	return $Pos1000;
}


sub GetCurrHourForGIF
{
	my ($Hour, $Min) = @_;
	my ($CurrHour) = $Hour;

	if ($Min > 28) {	# already ~30 minutes into current hour
		$CurrHour++;
		$CurrHour = 0 if ($Hour > 23);
	}

	return $CurrHour;
}


sub GetMetaExpires
{
  my ($Time) = time + $_[0] * 60 + 5;
  my ($Wday) = ('Sun','Mon','Tue','Wed','Thu','Fri','Sat')[(gmtime($Time))[6]];
  my ($Month) = ('Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep', 
		 'Oct','Nov','Dec')[(gmtime($Time))[4]];
  my ($Mday, $Year, $Hour, $Min, $Sec) = (gmtime($Time))[3,5,2,1,0];

  if ($Mday < 10) {$Mday = "0$Mday"};

  if ($Hour < 10) {$Hour = "0$Hour"};

  if ($Min < 10) {$Min = "0$Min";}

  if ($Sec < 10) {$Sec = "0$Sec";}

  return "$Wday, $Mday $Month " . ($Year + 1900) . " $Hour:$Min:$Sec GMT";
}
