//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  This is the M-Network Bankswitching method (E7).  There are two 2K
  slices, 8 ROM banks, and 2K of RAM.  The RAM is broken into two
  1K pieces.  One 1K piece goes into slice 0 if selected and the other
  one is broken into four 256-byte parts.

  @author  Bradford W. Mott
  @version $Id: CartE7.cxx,v 1.2 1997/05/17 19:00:03 bwmott Exp $
*/

#include "CartE7.hxx"
#include "System.hxx"

//============================================================================
// Constructor
//============================================================================
CartridgeE7::CartridgeE7(System& system, uByte* image)
    : Cartridge(system)
{
  // Copy the image into my image buffer
  for(uWord t = 0; t < 16384; ++t)
    myImage[t] = image[t];

  // Map all of my addresses in the system
  for(uWord addr = 0; addr < 8192; ++addr)
  {
    if(addr & 0x1000)
    {
      if(((addr & 0x0FFF) >= 0x0FE0) && ((addr & 0x0FFF) <= 0x0FFB))
        mySystem.mapPeek(addr, *this);
      else if((addr & 0x0FFF) <= 0x07FF)
        mySystem.mapPeek(addr, *this, &myImage[addr & 0x07ff], 
          &myImageOffset[0]);
      else if(((addr & 0x0FFF) >= 0x900) && ((addr & 0x0FFF) <= 0x9FF))
        mySystem.mapPeek(addr, *this, &myRAM[addr & 0x00ff], 
          &myRAMOffset);
      else if((addr & 0x0FFF) >= 0x0A00)
        mySystem.mapPeek(addr, *this, &myImage[addr & 0x07ff], 
          &myImageOffset[1]);
      else
        mySystem.mapPeek(addr, *this);

      mySystem.mapPoke(addr, *this);
    }
  }

  // Reset myself
  reset();
}

//============================================================================
// Destructor
//============================================================================
CartridgeE7::~CartridgeE7()
{
}

//============================================================================
// Reset to power on state
//============================================================================
void CartridgeE7::reset()
{
  myImageOffset[0] = 0 * 2048;
  myImageOffset[1] = 7 * 2048;
}

//============================================================================
// Answer the byte at the given address
//============================================================================
uByte CartridgeE7::peek(uWord addr)
{
  if((addr >= 0x1fe0) && (addr <= 0x1fe6))
    myImageOffset[0] = (addr & 07) * 2048;
  else if(addr == 0x1fe7)
    myImageOffset[0] = 8 * 2048;
  else if((addr >= 0x1fe8) && (addr <= 0x1feb))
    myRAMOffset = (addr & 03) * 256;

  return 0;
}

//============================================================================
// Store value in the given address
//============================================================================
void CartridgeE7::poke(uWord addr, uByte value)
{
  if((addr >= 0x1000) && (addr <= 0x13ff))
    myImage[8 * 2048 + (addr & 0x03ff)] = value;
  else if((addr >= 0x1800) && (addr <= 0x18ff))
    myRAM[myRAMOffset + (addr & 0x00ff)] = value;
  else if((addr >= 0x1fe0) && (addr <= 0x1fe6))
    myImageOffset[0] = (addr & 07) * 2048;
  else if(addr == 0x1fe7)
    myImageOffset[0] = 8 * 2048;
  else if((addr >= 0x1fe8) && (addr <= 0x1feb))
    myRAMOffset = (addr & 03) * 256;
}
 
