//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/** 
  Code to emulate the 6507 instructions.

  @author  Bradford W. Mott
  @version $Id: M6507.m4,v 1.1 1997/05/17 19:04:12 bwmott Exp $
*/

define(M6507_IMMEDIATE, `{
  operandAddress = PC++;
}')

define(M6507_ABSOLUTE, `{
  operandAddress = iDPeek(PC);
  PC += 2;

#ifdef SCHACK
  PokePossible = true;
#endif
}')

define(M6507_ABSOLUTEX, `{
  uWord addr = iDPeek(PC);
  PC += 2;

  // See if we need to add one cycle for indexing across a page boundary
  if((ourInstructionCycleTable[IR] == 4) && notSamePage(addr, addr + X))
    ++myCycles;

  operandAddress = ((addr + X) & 0xffff);

#ifdef SCHACK
  PokePossible = true;
#endif
}')

define(M6507_ABSOLUTEY, `{
  uWord addr = iDPeek(PC);
  PC += 2;

  // See if we need to add one cycle for indexing across a page boundary
  if((ourInstructionCycleTable[IR] == 4) && notSamePage(addr, addr + Y))
    ++myCycles;

  operandAddress = ((addr + Y) & 0xffff);

#ifdef SCHACK
  PokePossible = true;
#endif
}')

define(M6507_ZERO, `{
  operandAddress = iPeek(PC++);
}')

define(M6507_ZEROX, `{
  operandAddress = ((uByte)iPeek(PC++) + X);
}')

define(M6507_ZEROY, `{
  operandAddress = ((uByte)iPeek(PC++) + Y);
}')

define(M6507_INDIRECT, `{
  uWord addr = iDPeek(PC);
  PC += 2;

  // Simulate the error in the indirect addressing mode!
  uWord high = notSamePage(addr, addr + 1) ? (addr & 0xff00) : (addr + 1);

  operandAddress = (iPeek(addr) | (iPeek(high) << 8));
}')

define(M6507_INDIRECTX, `{
  uByte zp = iPeek(PC++) + X;

#ifdef M6507_FAST_ACCESS
  operandAddress = iDPeek(zp);
#else
  operandAddress = iPeek(zp++);
  operandAddress |= iPeek(zp) << 8;
#endif

#ifdef SCHACK
  PokePossible = true;
#endif
}')

define(M6507_INDIRECTY, `{
  uByte zp = iPeek(PC++);

#ifdef M6507_FAST_ACCESS
  uWord addr = iDPeek(zp);
#else
  uWord addr = iPeek(zp++);
  addr |= iPeek(zp) << 8;
#endif

  // See if we need to add one cycle for indexing across a page boundary
  if((ourInstructionCycleTable[IR] == 5) && notSamePage(addr, addr + Y))
    ++myCycles;

  operandAddress = ((addr + Y) & 0xffff);

#ifdef SCHACK
  PokePossible = true;
#endif
}')

define(M6507_ADC, `{
  uByte value = peek(operandAddress);
  uByte oldA = A;

  if(!D)
  {
    Word sum = (Word)((Byte)A) + (Word)((Byte)value) + (C ? 1 : 0);
    V = ((sum > 127) || (sum < -128));

    sum = (Word)A + (Word)value + (C ? 1 : 0);
    A = sum;
    C = (sum > 0xff);
    Z = !A;
    N = A & 0x80;
  }
  else
  {
    Word sum = ourBCDTable[0][A] + ourBCDTable[0][value] + (C ? 1 : 0);

    C = (sum > 99);
    A = ourBCDTable[1][sum & 0xff];
    Z = !A;
    N = A & 0x80;
    V = ((oldA ^ A) & 0x80) && ((A ^ value) & 0x80);
  }
}')

define(M6507_AND, `{
  A &= peek(operandAddress);

  Z = !A;
  N = A & 0x80;
}')

define(M6507_ASL, `{
  uByte value = peek(operandAddress);

  // Set carry flag according to the left-most bit in value
  C = value & 0x80;

  value <<= 1;
  poke(operandAddress, value);

  Z = !value;
  N = value & 0x80;
}')

define(M6507_ASLA, `{
  // Set carry flag according to the left-most bit in A
  C = A & 0x80;

  A <<= 1;

  Z = !A;
  N = A & 0x80;
}')

define(M6507_BCC, `{
  if(!C)
  {
    Byte offset = peek(PC++);
    uWord addr = (PC + offset) & 0xffff;
    myCycles += (notSamePage(PC, addr) ? 2 : 1);
    PC = addr;
  }
  else
  {
    PC++;
  }
}')

define(M6507_BCS, `{
  if(C)
  {
    Byte offset = peek(PC++);
    uWord addr = (PC + offset) & 0xffff;
    myCycles += (notSamePage(PC, addr) ? 2 : 1);
    PC = addr;
  }
  else
  {
    PC++;
  }
}')

define(M6507_BEQ, `{
  if(Z)
  {
    Byte offset = peek(PC++);
    uWord addr = (PC + offset) & 0xffff;
    myCycles += (notSamePage(PC, addr) ? 2 : 1);
    PC = addr;
  }
  else
  {
    PC++;
  }
}')


define(M6507_BIT, `{
  uByte value = peek(operandAddress);

  Z = !(A & value);
  N = value & 0x80;
  V = value & 0x40;
}')

define(M6507_BMI, `{
  if(N)
  {
    Byte offset = peek(PC++);
    uWord addr = (PC + offset) & 0xffff;
    myCycles += (notSamePage(PC, addr) ? 2 : 1);
    PC = addr;
  }
  else
  {
    PC++;
  }
}')

define(M6507_BNE, `{
  if(!Z)
  {
    Byte offset = peek(PC++);
    uWord addr = (PC + offset) & 0xffff;
    myCycles += (notSamePage(PC, addr) ? 2 : 1);
    PC = addr;
  }
  else
  {
    PC++;
  }
}')

define(M6507_BPL, `{
  if(!N)
  {
    Byte offset = peek(PC++);
    uWord addr = (PC + offset) & 0xffff;
    myCycles += (notSamePage(PC, addr) ? 2 : 1);
    PC = addr;
  }
  else
  {
    PC++;
  }
}')

define(M6507_BRK, `{
  PC++;
  poke(0x0100 + SP--, PC >> 8);
  poke(0x0100 + SP--, PC & 0x00ff);
  poke(0x0100 + SP--, PS());

  B = true;
  I = true;

  PC = peek(0xfffe) | (peek(0xffff) << 8);
}')

define(M6507_BVC, `{
  if(!V)
  {
    Byte offset = peek(PC++);
    uWord addr = (PC + offset) & 0xffff;
    myCycles += (notSamePage(PC, addr) ? 2 : 1);
    PC = addr;
  }
  else
  {
    PC++;
  }
}')

define(M6507_BVS, `{
  if(V)
  {
    Byte offset = peek(PC++);
    uWord addr = (PC + offset) & 0xffff;
    myCycles += (notSamePage(PC, addr) ? 2 : 1);
    PC = addr;
  }
  else
  {
    PC++;
  }
}')


define(M6507_CLC, `{
  C = false;
}')

define(M6507_CLD, `{
  D = false;
}')

define(M6507_CLI, `{
  I = false;
}')

define(M6507_CLV, `{
  V = false;
}')

define(M6507_CMP, `{
  uWord value = (uWord)A - peek(operandAddress);

  Z = !value;
  N = value & 0x0080;
  C = !(value & 0x0100);
}')

define(M6507_CPX, `{
  uWord value = (uWord)X - peek(operandAddress);

  Z = !value;
  N = value & 0x0080;
  C = !(value & 0x0100);
}')

define(M6507_CPY, `{
  uWord value = (uWord)Y - peek(operandAddress);

  Z = !value;
  N = value & 0x0080;
  C = !(value & 0x0100);
}')

define(M6507_DEC, `{
  uByte value = peek(operandAddress) - 1;
  poke(operandAddress, value);

  Z = !value;
  N = value & 0x80;
}')


define(M6507_DEX, `{
  X--;

  Z = !X;
  N = X & 0x80;
}')


define(M6507_DEY, `{
  Y--;

  Z = !Y;
  N = Y & 0x80;
}')

define(M6507_EOR, `{
  A ^= peek(operandAddress);

  Z = !A;
  N = A & 0x80;
}')


define(M6507_INC, `{
  uByte value = peek(operandAddress) + 1;
  poke(operandAddress, value);

  Z = !value;
  N = value & 0x80;
}')

define(M6507_INX, `{
  X++;

  Z = !X;
  N = X & 0x80;
}')

define(M6507_INY, `{
  Y++;

  Z = !Y;
  N = Y & 0x80;
}')

define(M6507_JMP, `{
  PC = operandAddress;
}')

define(M6507_JSR, `{
  // It seems that the 650x does not push the address of the next instruction
  // on the stack it actually pushes the address of the next instruction
  // minus one.  This is compensated for in the RTS instruction
  poke(0x0100 + SP--, (PC - 1) >> 8);
  poke(0x0100 + SP--, (PC - 1) & 0xff);

  PC = operandAddress;
}')

define(M6507_LDA, `{
  A = peek(operandAddress);

  Z = !A;
  N = A & 0x80;
}')

define(M6507_LDX, `{
  X = peek(operandAddress);

  Z = !X;
  N = X & 0x80;
}')

define(M6507_LDY, `{
  Y = peek(operandAddress);

  Z = !Y;
  N = Y & 0x80;
}')

define(M6507_LSR, `{
  uByte value = peek(operandAddress);

  // Set carry flag according to the right-most bit in value
  C = value & 0x01;

  value = (value >> 1) & 0x7f;
  poke(operandAddress, value);

  Z = !value;
  N = value & 0x80;
}')

define(M6507_LSRA, `{
  // Set carry flag according to the right-most bit
  C = A & 0x01;

  A = (A >> 1) & 0x7f;

  Z = !A;
  N = A & 0x80;
}')

define(M6507_NOP, `{
}')

define(M6507_ORA, `{
  A |= peek(operandAddress);

  Z = !A;
  N = A & 0x80;
}')

define(M6507_PHA, `{
  poke(0x0100 + SP--, A);
}')


define(M6507_PHP, `{
  poke(0x0100 + SP--, PS());
}')

define(M6507_PLA, `{
  A = peek(++SP + 0x0100);

  Z = !A;
  N = A & 0x80;
}')

define(M6507_PLP, `{
  PS(peek(++SP + 0x0100));
}')

define(M6507_ROL, `{
  uByte value = peek(operandAddress);

  bool oldC = C;

  // Set carry flag according to the left-most bit in value
  C = value & 0x80;

  value = (value << 1) | (oldC ? 1 : 0);
  poke(operandAddress, value);

  Z = !value;
  N = value & 0x80;
}')

define(M6507_ROLA, `{
  bool oldC = C;

  // Set carry flag according to the left-most bit
  C = A & 0x80;

  A = (A << 1) | (oldC ? 1 : 0);

  Z = !A;
  N = A & 0x80;
}')

define(M6507_ROR, `{
  uByte value = peek(operandAddress);

  bool oldC = C;

  // Set carry flag according to the right-most bit
  C = value & 0x01;

  value = ((value >> 1) & 0x7f) | (oldC ? 0x80 : 0x00);
  poke(operandAddress, value);

  Z = !value;
  N = value & 0x80;
}')

define(M6507_RORA, `{
  bool oldC = C;

  // Set carry flag according to the right-most bit
  C = A & 0x01;

  A = ((A >> 1) & 0x7f) | (oldC ? 0x80 : 0x00);

  Z = !A;
  N = A & 0x80;
}')


define(M6507_RTI, `{
  PS(peek(++SP + 0x0100));
  PC = peek(++SP + 0x0100);
  PC |= peek(++SP + 0x0100) << 8;
}')

define(M6507_RTS, `{
  PC = peek(++SP + 0x0100);
  PC |= peek(++SP + 0x0100) << 8;
  ++PC;
}')

define(M6507_SBC, `{
  uByte oldA = A;

  if(!D)
  {
    uByte value = ~(peek(operandAddress));
    Word difference = (Word)((Byte)A) + (Word)((Byte)value) + (C ? 1 : 0);
    V = ((difference > 127) || (difference < -128));

    difference = ((Word)A) + ((Word)value) + (C ? 1 : 0);
    A = difference;
    C = (difference > 0xff);
    Z = !A;
    N = A & 0x80;
  }
  else
  {
    uByte value = peek(operandAddress);
    Word difference = ourBCDTable[0][A] - ourBCDTable[0][value] - (C ? 0 : 1);

    if(difference < 0)
      difference += 100;

    A = ourBCDTable[1][difference];
    Z = !A;
    N = A & 0x80;

    C = (oldA >= (value + (C ? 0 : 1)));
    V = ((oldA ^ A) & 0x80) && ((A ^ value) & 0x80);
  }
}')

define(M6507_SEC, `{
  C = true;
}')

define(M6507_SED, `{
  D = true;
}')

define(M6507_SEI, `{
  I = true;
}')

define(M6507_STA, `{
  poke(operandAddress, A);
}')

define(M6507_STX, `{
  poke(operandAddress, X);
}')

define(M6507_STY, `{
  poke(operandAddress, Y);
}')

define(M6507_TAX, `{
  X = A;

  Z = !X;
  N = X & 0x80;
}')

define(M6507_TAY, `{
  Y = A;

  Z = !Y;
  N = Y & 0x80;
}')

define(M6507_TSX, `{
  X = SP;

  Z = !X;
  N = X & 0x80;
}')

define(M6507_TXA, `{
  A = X;

  Z = !A;
  N = A & 0x80;
}')

define(M6507_TXS, `{
  SP = X;
}')

define(M6507_TYA, `{
  A = Y;

  Z = !A;
  N = A & 0x80;
}')


case 0x69:
M6507_IMMEDIATE
M6507_ADC
break;

case 0x65:
M6507_ZERO
M6507_ADC
break;

case 0x75:
M6507_ZEROX
M6507_ADC
break;

case 0x6D:
M6507_ABSOLUTE
M6507_ADC
break;

case 0x7D:
M6507_ABSOLUTEX
M6507_ADC
break;

case 0x79:
M6507_ABSOLUTEY
M6507_ADC
break;

case 0x61:
M6507_INDIRECTX
M6507_ADC
break;

case 0x71:
M6507_INDIRECTY
M6507_ADC
break;



case 0x29:
M6507_IMMEDIATE
M6507_AND
break;

case 0x25:
M6507_ZERO
M6507_AND
break;

case 0x35:
M6507_ZEROX
M6507_AND
break;

case 0x2D:
M6507_ABSOLUTE
M6507_AND
break;

case 0x3D:
M6507_ABSOLUTEX
M6507_AND
break;

case 0x39:
M6507_ABSOLUTEY
M6507_AND
break;

case 0x21:
M6507_INDIRECTX
M6507_AND
break;

case 0x31:
M6507_INDIRECTY
M6507_AND
break;


case 0x0a:
M6507_ASLA
break;

case 0x06:
M6507_ZERO
M6507_ASL
break;

case 0x16:
M6507_ZEROX
M6507_ASL
break;

case 0x0e:
M6507_ABSOLUTE
M6507_ASL
break;

case 0x1e:
M6507_ABSOLUTEX
M6507_ASL
break;


case 0x90:
M6507_BCC
break;


case 0xb0:
M6507_BCS
break;


case 0xf0:
M6507_BEQ
break;


case 0x24:
M6507_ZERO
M6507_BIT
break;

case 0x2C:
M6507_ABSOLUTE
M6507_BIT
break;


case 0x30:
M6507_BMI
break;


case 0xD0:
M6507_BNE
break;


case 0x10:
M6507_BPL
break;


case 0x00:
M6507_BRK
break;


case 0x50:
M6507_BVC
break;


case 0x70:
M6507_BVS
break;


case 0x18:
M6507_CLC
break;


case 0xd8:
M6507_CLD
break;


case 0x58:
M6507_CLI
break;


case 0xb8:
M6507_CLV
break;


case 0xc9:
M6507_IMMEDIATE
M6507_CMP
break;

case 0xc5:
M6507_ZERO
M6507_CMP
break;

case 0xd5:
M6507_ZEROX
M6507_CMP
break;

case 0xcd:
M6507_ABSOLUTE
M6507_CMP
break;

case 0xdd:
M6507_ABSOLUTEX
M6507_CMP
break;

case 0xd9:
M6507_ABSOLUTEY
M6507_CMP
break;

case 0xc1:
M6507_INDIRECTX
M6507_CMP
break;

case 0xd1:
M6507_INDIRECTY
M6507_CMP
break;


case 0xe0:
M6507_IMMEDIATE
M6507_CPX
break;

case 0xe4:
M6507_ZERO
M6507_CPX
break;

case 0xec:
M6507_ABSOLUTE
M6507_CPX
break;


case 0xc0:
M6507_IMMEDIATE
M6507_CPY
break;

case 0xc4:
M6507_ZERO
M6507_CPY
break;

case 0xcc:
M6507_ABSOLUTE
M6507_CPY
break;


case 0xc6:
M6507_ZERO
M6507_DEC
break;

case 0xd6:
M6507_ZEROX
M6507_DEC
break;

case 0xce:
M6507_ABSOLUTE
M6507_DEC
break;

case 0xde:
M6507_ABSOLUTEX
M6507_DEC
break;


case 0xca:
M6507_DEX
break;


case 0x88:
M6507_DEY
break;


case 0x49:
M6507_IMMEDIATE
M6507_EOR
break;

case 0x45:
M6507_ZERO
M6507_EOR
break;

case 0x55:
M6507_ZEROX
M6507_EOR
break;

case 0x4d:
M6507_ABSOLUTE
M6507_EOR
break;

case 0x5d:
M6507_ABSOLUTEX
M6507_EOR
break;

case 0x59:
M6507_ABSOLUTEY
M6507_EOR
break;

case 0x41:
M6507_INDIRECTX
M6507_EOR
break;

case 0x51:
M6507_INDIRECTY
M6507_EOR
break;


case 0xe6:
M6507_ZERO
M6507_INC
break;

case 0xf6:
M6507_ZEROX
M6507_INC
break;

case 0xee:
M6507_ABSOLUTE
M6507_INC
break;

case 0xfe:
M6507_ABSOLUTEX
M6507_INC
break;


case 0xe8:
M6507_INX
break;


case 0xc8:
M6507_INY
break;


case 0x4c:
M6507_ABSOLUTE
M6507_JMP
break;

case 0x6c:
M6507_INDIRECT
M6507_JMP
break;


case 0x20:
M6507_ABSOLUTE
M6507_JSR
break;


case 0xa9:
M6507_IMMEDIATE
M6507_LDA
break;

case 0xa5:
M6507_ZERO
M6507_LDA
break;

case 0xb5:
M6507_ZEROX
M6507_LDA
break;

case 0xad:
M6507_ABSOLUTE
M6507_LDA
break;

case 0xbd:
M6507_ABSOLUTEX
M6507_LDA
break;

case 0xb9:
M6507_ABSOLUTEY
M6507_LDA
break;

case 0xa1:
M6507_INDIRECTX
M6507_LDA
break;

case 0xb1:
M6507_INDIRECTY
M6507_LDA
break;


case 0xa2:
M6507_IMMEDIATE
M6507_LDX
break;

case 0xa6:
M6507_ZERO
M6507_LDX
break;

case 0xb6:
M6507_ZEROY
M6507_LDX
break;

case 0xae:
M6507_ABSOLUTE
M6507_LDX
break;

case 0xbe:
M6507_ABSOLUTEY
M6507_LDX
break;


case 0xa0:
M6507_IMMEDIATE
M6507_LDY
break;

case 0xa4:
M6507_ZERO
M6507_LDY
break;

case 0xb4:
M6507_ZEROX
M6507_LDY
break;

case 0xac:
M6507_ABSOLUTE
M6507_LDY
break;

case 0xbc:
M6507_ABSOLUTEX
M6507_LDY
break;


case 0x4a:
M6507_LSRA
break;


case 0x46:
M6507_ZERO
M6507_LSR
break;

case 0x56:
M6507_ZEROX
M6507_LSR
break;

case 0x4e:
M6507_ABSOLUTE
M6507_LSR
break;

case 0x5e:
M6507_ABSOLUTEX
M6507_LSR
break;


case 0xea:
M6507_NOP
break;


case 0x09:
M6507_IMMEDIATE
M6507_ORA
break;

case 0x05:
M6507_ZERO
M6507_ORA
break;

case 0x15:
M6507_ZEROX
M6507_ORA
break;

case 0x0D:
M6507_ABSOLUTE
M6507_ORA
break;

case 0x1D:
M6507_ABSOLUTEX
M6507_ORA
break;

case 0x19:
M6507_ABSOLUTEY
M6507_ORA
break;

case 0x01:
M6507_INDIRECTX
M6507_ORA
break;

case 0x11:
M6507_INDIRECTY
M6507_ORA
break;


case 0x48:
M6507_PHA
break;


case 0x08:
M6507_PHP
break;


case 0x68:
M6507_PLA
break;


case 0x28:
M6507_PLP
break;


case 0x2a:
M6507_ROLA
break;


case 0x26:
M6507_ZERO
M6507_ROL
break;

case 0x36:
M6507_ZEROX
M6507_ROL
break;

case 0x2e:
M6507_ABSOLUTE
M6507_ROL
break;

case 0x3e:
M6507_ABSOLUTEX
M6507_ROL
break;


case 0x6a:
M6507_RORA
break;

case 0x66:
M6507_ZERO
M6507_ROR
break;

case 0x76:
M6507_ZEROX
M6507_ROR
break;

case 0x6e:
M6507_ABSOLUTE
M6507_ROR
break;

case 0x7e:
M6507_ABSOLUTEX
M6507_ROR
break;


case 0x40:
M6507_RTI
break;


case 0x60:
M6507_RTS
break;


case 0xe9:
M6507_IMMEDIATE
M6507_SBC
break;

case 0xe5:
M6507_ZERO
M6507_SBC
break;

case 0xf5:
M6507_ZEROX
M6507_SBC
break;

case 0xed:
M6507_ABSOLUTE
M6507_SBC
break;

case 0xfd:
M6507_ABSOLUTEX
M6507_SBC
break;

case 0xf9:
M6507_ABSOLUTEY
M6507_SBC
break;

case 0xe1:
M6507_INDIRECTX
M6507_SBC
break;

case 0xf1:
M6507_INDIRECTY
M6507_SBC
break;


case 0x38:
M6507_SEC
break;


case 0xf8:
M6507_SED
break;


case 0x78:
M6507_SEI
break;


case 0x85:
M6507_ZERO
M6507_STA
break;

case 0x95:
M6507_ZEROX
M6507_STA
break;

case 0x8d:
M6507_ABSOLUTE
M6507_STA
break;

case 0x9d:
M6507_ABSOLUTEX
M6507_STA
break;

case 0x99:
M6507_ABSOLUTEY
M6507_STA
break;

case 0x81:
M6507_INDIRECTX
M6507_STA
break;

case 0x91:
M6507_INDIRECTY
M6507_STA
break;


case 0x86:
M6507_ZERO
M6507_STX
break;

case 0x96:
M6507_ZEROY
M6507_STX
break;

case 0x8e:
M6507_ABSOLUTE
M6507_STX
break;


case 0x84:
M6507_ZERO
M6507_STY
break;

case 0x94:
M6507_ZEROX
M6507_STY
break;

case 0x8c:
M6507_ABSOLUTE
M6507_STY
break;


case 0xaa:
M6507_TAX
break;


case 0xa8:
M6507_TAY
break;


case 0xba:
M6507_TSX
break;


case 0x8a:
M6507_TXA
break;


case 0x9a:
M6507_TXS
break;


case 0x98:
M6507_TYA
break;


