//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  This class emulates a 6507 microprocessor.  The memory accesses
  and cycle counts it generates are valid at the sub-instruction
  level and "false" reads are generated (such as the ones produced
  by the Indirect,X addressing when it crosses a page boundary).
  This provides better compatibility for hardware that has side
  effects and for games which are very time sensitive.

  NOTE: The description given above is what this class "should"
  be, however, currently it's just like M6507Low except it keeps
  information about the last operand fetched for Supercharger support.

  @author  Bradford W. Mott
  @version $Id: M6507Hi.cxx,v 1.1 1997/05/17 19:04:12 bwmott Exp $
*/

#include <assert.h>
#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>
#include "M6507Hi.hxx"

bool PokePossible = false;
uWord LastOperandAddress = 0;
uWord LastOperandOffset = 0;
uLong LastOperandCycle = 0;

//============================================================================
// Answers true iff the two addresses are not on the same page
//============================================================================
static inline bool notSamePage(uWord addr1, uWord addr2)
{
  return (addr1 ^ addr2) & 0xff00;
}

//============================================================================
// Default constructor
//============================================================================
M6507High::M6507High(System& system)
    : M6507(system)
{
}

//============================================================================
// Destructor
//============================================================================
M6507High::~M6507High()
{
}

//============================================================================
// Answer the byte at the given address (only lower 8-bits used)
//============================================================================
inline uWord M6507High::peek(uWord addr)
{
  return mySystem.peek(addr);
}

//============================================================================
// Answer the uByte at the given address that should be part of
// an instruction we're trying to execute
//============================================================================
inline uWord M6507High::iPeek(uWord addr)
{
  return mySystem.peek(addr);
}

//============================================================================
// Answer the uWord at the given address
//============================================================================
inline uWord M6507High::iDPeek(uWord addr)
{
  uWord value = mySystem.peek(addr++);
  return value | (((uWord)mySystem.peek(addr)) << 8);
}

//============================================================================
// Write given byte to the given address 
//============================================================================
inline void M6507High::poke(uWord addr, uByte value)
{
  mySystem.poke(addr,value);
}

//============================================================================
// Execute instructions until at least the given number of
// cycles have passed or execution is halted by someone
//============================================================================
void M6507High::execute(uLong numberOfCycles)
{
  uLong stop = myCycles + numberOfCycles;
  myHaltExecution = false;

  // Loop until execution is halted or we're finished
  while(!myHaltExecution && (myCycles < stop))
  {
    uWord operandAddress = 0;
    uByte oldX = X;
    uByte oldY = Y;

#ifdef DEBUG
    debugStream << "PC=" << hex << setw(4) << PC << " ";
#endif

    // Fetch instruction at the program counter
    IR = iPeek(PC++);

    // Update cycles
    myCycles += ourInstructionCycleTable[IR];

#ifdef DEBUG
    debugStream << "IR=" << hex << setw(2) << (int)IR << " ";
    debugStream << "<" << ourAddressingModeTable[IR] << " ";
#endif

    // Call the code to execute the instruction
    switch(IR)
    {
      // Turn on the Supercharger hack in 6507 instruction emulation
      #define SCHACK

      // The code to simulate the 6507 instructions is 
      // generated by an M4 macro file
      #include "M6507.ins"

      default:
        invalidInstruction();
    } 

    // Needed for Supercharger support
    if(operandAddress)
    {
      if(PokePossible)
      {
        if(ourAddressingModeTable[IR] == AbsoluteX)
        {
          LastOperandAddress = (operandAddress - oldX);
          LastOperandOffset = oldX;
        }
        else if(ourAddressingModeTable[IR] == IndirectY || 
            ourAddressingModeTable[IR] == AbsoluteY)
        {
          LastOperandAddress = (operandAddress - oldY);
          LastOperandOffset = oldY;
        }
        else
        {
          LastOperandAddress = operandAddress;
          LastOperandOffset = 0;
        }

        LastOperandCycle = myCycles;
      }
      else
      {
        LastOperandAddress = 0;
        LastOperandOffset = 0;
      }
    }

    PokePossible = false;

#ifdef DEBUG
    debugStream << hex << setw(4) << operandAddress << " ";
    debugStream << setw(4) << ourInstructionMnemonicTable[IR];
    debugStream << "> ";
    debugStream << "A=" << ::hex << setw(2) << (int)A << " ";
    debugStream << "X=" << ::hex << setw(2) << (int)X << " ";
    debugStream << "Y=" << ::hex << setw(2) << (int)Y << " ";
    debugStream << "PS=" << ::hex << setw(2) << (int)PS() << " ";
    debugStream << "SP=" << ::hex << setw(2) << (int)SP << " ";
    debugStream << "Cyc=" << ::dec << cycles();
    debugStream << endl;
#endif
  }
}

