 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/


class Unit;

#ifndef __UNIT_H__
#define __UNIT_H__

#include "EST.h"
#include "EST_TVector.h"

#if defined(INCLUDE_LISP)
#   include "siod.h"
#   include "festival.h"
#endif

#include "UnitDatabase.h"
#include "SubtypedObject.h"

/** Abstraction for a chunk of information. We string together units
  * to make an utterance.
  * @see UnitDatabase
  * @see UnitIndex
  * @author Richard Caley <rjc@cstr,ed,ac,uk>
  * @version $Id: Unit.h,v 1.1 1998/08/12 09:35:52 awb Exp $
  */

class Unit : public SubtypedObject<Unit> {

private:

  /// Name. Used in Indexes and catalogues to refer to this unit.
  UnitName p_name;
  /// Properties of this unit.
  UnitProperties p_properties;
  /// Database this unit came from, if any.
  UnitDatabase *p_database;
  /// Unit which this one is a modification of.
  Unit *p_underlying;

protected:

  /// Set the name.
  void set_name(EST_String name)  {p_name = name;};
  /// Record where this unit came from.
  void set_database(UnitDatabase *database)  {p_database = database;};
  /// Record where to look for information not set in this unit.
  void set_underlying(Unit *unit)  {p_underlying = unit;};
  /// record the properties.
  void set_unit_properties(UnitProperties properties) {p_properties = properties;};

  /**@name Subtyping
    * Required by the FestivalObject interface, but not useful for units
    * since they are always in databases, so we define default ones here
    * which do nothing.
    */
  //@{
  /// Read in from the file.
  EST_read_status fill_from(FILE *) {return misc_read_error; };
  /// Record the filename.
  void set_filename(EST_String) { };
  //@}

public:

  /// Default constructor
  Unit(void);
  /// Destructor.
  virtual ~Unit(void);

  /// Get name.
  EST_String name(void) const {return p_name;};
  /// Get the database this unit came from.
  UnitDatabase *database(void) const {return p_database;};
  /// Get the unit this is a varient of.
  Unit *underlying(void) const {return p_underlying; };
  /// Get the properties.
  UnitProperties &properties(void) {return p_properties;};

  /// Record the source database.
  virtual void associate_database(UnitDatabase *db);

#if 0
  /**@name Properties */
  //@{
  /// Set a single property.
  virtual int set_property(EST_String property, ValueType value);
  //@}
#endif

  /// Units are memory-managed as part of their database.
  virtual SchemeObject *gc_master_pointer(void) { return p_database?(SchemeObject *)p_database:(SchemeObject *)this;};

  /**@name Segmentation
    * Units are divided into segments so they can be lined up
    * with elements ofthe Segment stream.
    * @see Segmentation_notes
    */
  //@{
  /// Length of this unit in arbitrary units.
  virtual int length(void);
  /// How it divides up.
  virtual EST_TVector<int> *segment_ends(void);
  //@}

  /// Return a new unit based on this one which can be modified.
  virtual Unit *derived_unit(void);

#if defined(INCLUDE_LISP)

  /**@name Scheme interface
    * These are the functions which are imported into Scheme. 
    */
  //@{

  /// Print out the unit.
  static void lisp_print(LISP, FILE *);
  /// Print to string.
  static void lisp_print_string(LISP, char *);
  /// Initialisation function.
  static void lisp_declare(void);
#endif

  friend class SubtypedObject<Unit>;
  friend class UnitDatabase;
};

#endif
