 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                       Copyright (c) 1996,1997                         */
 /*                        All Rights Reserved.                           */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/
 /*                                                                       */
 /*                  Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Wed Mar 19 1997                               */
 /* --------------------------------------------------------------------- */
 /* A Unit type for units which have content which consists of frames     */
 /* each representing a pitch period containing LPCs, residual,           */
 /* original signal or whatever.                                          */
 /*                                                                       */
 /*************************************************************************/

#include "FramesUnit.h"
#include "FramesJoin.h"
#include "EST_ContourType.h"
#include "sigpr/EST_lpc.h"

FramesUnit::FramesUnit(void)
{
  p_signal = p_residual = NULL;
  p_analysis = NULL;
  p_lpc = NULL;
  p_rfc = NULL;
  p_pitchmarks = NULL;
  p_segment_ends = NULL;
  p_n_mods = 0;
  for(int i=0; i< MAX_UNITS_IN_CHUNK; i++)
      p_modifications[i] = NULL;
}

FramesUnit::~FramesUnit(void)
{
  if (p_signal)
    delete p_signal;
  if (p_residual)
    delete p_residual;
  if (p_lpc && p_lpc != p_analysis)
    delete p_lpc;
  if (p_rfc && p_rfc != p_analysis)
    delete p_rfc;
  if (p_analysis)
    delete p_analysis;
  if (p_pitchmarks)
    delete p_pitchmarks;
  if (p_segment_ends)
    delete p_segment_ends;
  for(int i=0; i< MAX_UNITS_IN_CHUNK; i++)
      if (p_modifications[i])
	  delete p_modifications[i];
}

Unit *FramesUnit::create(void) 
{
  FramesUnit *unit = new FramesUnit;

  return unit;
}

void FramesUnit::property_names(EST_TList<EST_String> &list) const
{
  Unit::property_names(list);
  list.append("frames_subtype");
}

ValueType FramesUnit::property(EST_String name) const
{
  if (name == "frames_subtype")
    return c_string_as_value("yes");

  return Unit::property(name);
}


int FramesUnit::length(void)
{
  EST_Track *pms = pitchmarks();

  return pms->num_frames();
}

int FramesUnit::add_modifications(EST_Track *mods)
{
  if (p_n_mods >= MAX_UNITS_IN_CHUNK)
    err("Too many modification tracks", name());

  // cout << "add mod " << name() << " " << p_n_mods << "\n";

  p_modifications[p_n_mods++] = mods;

  return p_n_mods;
}


EST_TVector<int> *FramesUnit::segment_ends(void)
{
    if (underlying())
	return underlying()->segment_ends();

    if (!p_segment_ends && database())
    {
	EST_Track *pms = pitchmarks();


	EST_TVector<float> *segments = database()->get_segments(*this);

	if (!segments)
	    return NULL;

	p_segment_ends = new EST_TVector<int>(segments->length());
	cout << "d\n";
	for(int i=0; i< segments->length(); i++)
	{
//	    cout << "t offset:" << pms->t_offset() << endl;
//	    cout << "orig:" << i << " " << (*segments)(i) << endl;

//	    for (int j = 0; j < pms->num_frames(); ++j)
//		cout << pms->t(j) << " ";
//	    cout << endl;
	    if (pms->has_channel(channel_length))
		(*p_segment_ends)[i] = nearest_boundary(*pms, (*segments)(i), sample_rate(), pms->t_offset());
	    else
		(*p_segment_ends)[i] = pms->index((*segments)(i) 
						  - pms->t_offset());
//	    cout << "index:" << i << " " << (*p_segment_ends)[i] << endl;
	}

	if ((*p_segment_ends)[0] == (*p_segment_ends)[1] && (*p_segment_ends)[0] > 0)
	    (*p_segment_ends)[0] -=1;

	if ((*p_segment_ends)[segments->length()-1] 
	    == (*p_segment_ends)[segments->length()-2] 
	    && (*p_segment_ends)[segments->length()-1] < pms->num_frames())
	    (*p_segment_ends)[segments->length()-1] +=1;
      
	for(int i2=1; i2< segments->length()-1; i2++)
	{
	    if ((*p_segment_ends)[i2] == (*p_segment_ends)[i2+1])
	    {
		(*p_segment_ends)[i2] -= 1;
	    }
	}

    }
    return p_segment_ends;
}

EST_Wave *FramesUnit::signal(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->signal();
  if (!p_signal && database())
    p_signal = database()->get_wave(UnitDatabase::content_signal, *this);

  return p_signal;
}

EST_Wave *FramesUnit::residual(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->residual();

  if (!p_residual && database())
    p_residual = database()->get_wave(UnitDatabase::content_residual, *this);

  return p_residual;
}

EST_Track *FramesUnit::analysis(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->analysis();

  if (!p_analysis && database())
    p_analysis = database()->get_coefficients(UnitDatabase::content_analysis, 
					      ct_other, 
					      *this);

  return p_analysis;
}

EST_Track *FramesUnit::lpc(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->lpc();

  if (p_lpc)
    return p_lpc;

  if (database()->coefficient_information(UnitDatabase::content_analysis)->coefficient_format == ct_lpc)
    {
      return p_lpc = analysis();
    }

  cout << "can't convert to LPC\n";

  return NULL;
}

EST_Track *FramesUnit::rfc(void)
{
  if (underlying())
    return ((FramesUnit *)underlying())->rfc();

  if (p_rfc)
    return p_rfc;

  EST_ContourType type = database()->coefficient_information(UnitDatabase::content_analysis)->coefficient_format;

  if ( type == ct_reflection)
      return p_rfc = analysis();
  else if ( type == ct_lpc)
    {
      p_rfc = new EST_Track(*analysis());
      lpc_to_reflection(*p_rfc);
      return p_rfc;
    }

  cout << "can't convert to RFC\n";

  return NULL;
}

EST_Track *FramesUnit::pitchmarks(void)
{
    if (underlying())
	return ((FramesUnit *)underlying())->pitchmarks();

    if (!p_pitchmarks && database())
	p_pitchmarks = 
	    database()->get_coefficients(UnitDatabase::content_analysis, 
					 ct_pitchmarks, *this);

    return p_pitchmarks;
}

/**@name Frames_modifications
  * Modifications describe how the information in a Unit should be rearanged
  * to conform to the required prosody.
  * 
  * For a FramesUnit, a modification is a track with the following
  * channels:
  * \begin{description}
  * \item[{\tt channel_frame}]
  *	Which of the  original frames to use at each point.
  * \item[{\tt channel_length}]
  *	New length for the frame.
  * \item[{\tt channel_power}]
  *	New power for the frame.
  * \end{description}
  * 
  * The unit should then be synthesised as if the frames had
  * been rearanged into the order given and with the given lengths and 
  * powers.
  * @see FramesUnit
  */
//@{ 

EST_Track *FramesUnit::modifications(int i) const
{
  //  cout << "get mod " << i << "\n";
  return p_modifications[i];
}

int FramesUnit::n_modifications(void) const
{
  return p_n_mods;
}
//@}

int FramesUnit::sample_rate(void)
{
  EST_Wave *w = residual();
  return w?w->sample_rate():0;
}


/**@name Chunking
  * The full generality of the {\tt Utterance} structure can be hard
  * to interpret. When writing festival modules which have to deal
  * with utterances which have joins and modifications and so on and
  * relate these to Segments the code of the module can become quite complex.
  * 
  * At least for development it is often preferable to have a simpler
  * structure to represent the utterance to be synthesised and {\tt Chunk}s
  * provide this. 
  *
  * The Utterance structure can be mapped into a list of chunks which can be
  * scanned with a simple one itteration per Segment loop.
  *
  * A chunk represents the part of an utterance which corresponds to
  * one element of the Segment stream. Each chunk has a number of `bits'
  * which are parts of units. For instance in diphone synthesis there
  * are two bits per chunk, each of whichis half of a unit.
  *
  * @see Chunks
  * @see FramesUnit 
  */
//@{ 


FramesUnit::Chunk *FramesUnit::chunk_utterance(EST_Utterance *utterance,
					 const EST_String unit_stream_name,
					 const EST_String join_stream_name)
{
  if(!utterance->has_relation(unit_stream_name))
    return NULL;
  if(!utterance->has_relation(join_stream_name))
    return NULL;

  EST_Relation * unit_stream = utterance->relation(unit_stream_name);
  EST_Relation * join_stream = utterance->relation(join_stream_name);

  return FramesUnit::chunk_utterance(utterance, unit_stream, join_stream);
}

static 
void get_joins(EST_Relation *unit_stream, EST_Relation *join_stream, 
		 EST_Item *un, FramesUnit *unit,
		 FramesJoin *&start_join, FramesJoin *&end_join)
{
  (void)unit_stream;
  EST_Item *join_item1, *join_item2;

  un->get_linked(join_stream, join_item1, join_item2);

  if (join_item1)
    {
	FramesJoin *j = (FramesJoin *)(un->f("Join").ptr());

      if (j && (j->after() == unit || j->after() == unit->underlying()))
	start_join = j;
      else if (j && (j->before() == unit || j->before() == unit->underlying()))
	end_join = j;
      else
	cerr << "join not to here " << join_item1->name() << "\n";
    }
  if (join_item2)
    {
//	FramesJoin *j = (FramesJoin *)join_item2->contents();
	FramesJoin *j=(FramesJoin *)(join_item2->f("Join").ptr());

      if (j && (j->after() == unit || j->after() == unit->underlying()))
	start_join = j;
      else if (j && (j->before() == unit || j->before() == unit->underlying()))
	end_join = j;
      else
	cerr << "join not to here " << join_item2->name() << "\n";
    }
}

static
void get_unit_info(EST_Item *un,
		   EST_Relation *unit_stream,
		   EST_Relation *join_stream,
		   FramesUnit *&unit,
		   FramesJoin *&start_join, 
		   FramesJoin *&end_join,
		   EST_TVector<int> *&segment_ends
		   )
{

//  unit  = (FramesUnit *)(un->contents());
  unit  = (FramesUnit *)(un->f("Unit").ptr());
  start_join=NULL;
  end_join=NULL;
  segment_ends = unit->segment_ends();
  if (join_stream)
    get_joins(unit_stream, join_stream, 
	      un, unit,
	      start_join, end_join);
}

FramesUnit::Chunk * FramesUnit::chunk_utterance(EST_Utterance *utt,
			    EST_Relation *unit_stream,
			    EST_Relation *join_stream
			    )
{
  (void)utt;

  static EST_TBuffer<FramesUnit::Chunk> chunks(100);

  EST_Item *un = unit_stream->head();
  FramesUnit *unit;
  FramesJoin *start_join, *end_join;
  EST_TVector<int> *segment_ends;

  get_unit_info(un, 
		unit_stream, join_stream, 
		unit, start_join, end_join, segment_ends);

  int bit_start = (start_join
		   ? start_join->start_index()
		   : 0
		   );
  int current_unit_end = (end_join
			  ? end_join->end_index()
			  : unit->analysis()->num_frames()
			  );

  int current_seg = 1;
  int current_seg_end = (*segment_ends)(current_seg);

    
  int nchunks=0;
  Chunk *chunk = chunks;


  // Loop, one itteration per segment, hence one per chunk
  while(un)
    {
      chunk->n=0;

      // Loop once per bit
      while (1)
	{
	  if (current_seg_end <= current_unit_end)
	    {
	      // Segment boundary inside unit
	      chunk->bit[chunk->n].unit = unit;
	      chunk->bit[chunk->n].start_frame = bit_start;
	      chunk->bit[chunk->n].end_frame = current_seg_end;
	      chunk->n++;

	      bit_start = current_seg_end;
	      current_seg++;
	      current_seg_end = (current_seg < segment_ends->length()
				 ? (*segment_ends)(current_seg)
				 : current_seg_end + 100
				 );
	      // end of chunk
	      break;
	    }
	  else 
	    {
	      // Unit boundary within segment
	      un=next(un);
	      // end of units
	      if (!un)
		break;

	      chunk->bit[chunk->n].unit = unit;
	      chunk->bit[chunk->n].start_frame = bit_start;
	      chunk->bit[chunk->n].end_frame = current_unit_end;
	      chunk->n++;

	      get_unit_info(un, 
			    unit_stream, join_stream, 
			    unit, start_join, end_join, segment_ends);
	      
	      bit_start = (start_join
			   ? start_join->start_index()
			   : 0
			   );
	      current_unit_end = (end_join
				  ? end_join->end_index()
				  : unit->analysis()->num_frames()
				  );
	      current_seg = 1;
	      current_seg_end = (*segment_ends)(current_seg);
	    }
	}

      nchunks++;
      chunks.ensure(nchunks, 100);
      chunk = &(chunks[nchunks]);
    }

  chunk->n = 0;

  return chunks;
}


void FramesUnit::dump_chunks(ostream &stream, const Chunk *chunks)
{
  stream << "{ CHUNKS\n";
  while (chunks->n > 0)
    {
      stream << "      [ ";
      for(int i=0; i<chunks->n;i++)
	{
	  if (i>0)
	    stream << "\t| ";
	  stream << chunks->bit[i].unit->name() 
		 << "\t" << chunks->bit[i].start_frame 
		 << " " << chunks->bit[i].end_frame;
	}
      stream << "\t]\n";
      chunks++;
    }
  stream << "}\n";
}

//@}

#if defined(INSTANTIATE_TEMPLATES)
#include "../base_class/EST_TBuffer.cc"
template class EST_TBuffer<FramesUnit::Chunk>;
#endif
