/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1998                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Alan W Black and Paul Taylor                    */
/*             Date   :  February 1998                                   */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*  An implementation of Metrical Tree Phonology                         */
/*                                                                       */
/*=======================================================================*/

#include <math.h>
#include "festival.h"
#include "EST_error.h"
#include "us_features.h"

void add_feature_function(EST_Relation &r, const EST_String &fname,
			  EST_feature_function f)
{
    for (EST_Item *p = r.head(); p; p = next(p))
	p->fset(fname, f);
}

EST_Val duration_time(EST_Item *s)
{
    return s->fF("end", 1) - s->fF("start", 1);
}

EST_Val ff_start_time(EST_Item *s)
{
    
    EST_Relation *r = s->relation();
    if (r->f.S("timing_style") != "segment")
    {
	EST_error("Attempted to use start() feature function in non segment relation\n");
	festival_error();
    }

    return  (prev(s) == 0) ? 0.0 : prev(s)->fF("end");
}

EST_Val leaf_end_time(EST_Item *s)
{
    if (!s->relation()->f.present("time_path"))
    {
	EST_error("Attempted to use end() feature function in relation with no time_path feature defined: %\n", s->relation());
	festival_error();
    }

    EST_String rel_name = s->relation()->f.S("time_path");
    EST_Item *t, *a;
    
    if ((t = s->as_relation(rel_name)) == 0)
    {
	EST_error("No relation %s for item\n", (const char *)rel_name);
	festival_error();
    }

    a = last_leaf_in_tree(t);
//    cout << "ll: 1" << *a << endl;
    a  = a->as_relation(a->relation()->f.S("time_relation"));
//    cout << "ll: 2" << *a << endl;
    return a->fF("end", 1);
}

EST_Val vowel_start_time(EST_Item *s)
{
    EST_Relation *r = s->relation();

    if (!r->f.present("time_path"))
    {
	EST_error("Attempted to use vowel_time() feature function in relation with no time_relation feature defined\n");
	festival_error();
    }

    EST_String rel_name = r->f.S("time_path");

    EST_Item *n;

    n = syl_nucleus(s->as_relation(rel_name));
//    cout <<"nucleus " << *n << ":::" << endl;
    n = n->as_relation("LexicalPhone");
//    cout <<"nucleus2 " << *n << ":::" << endl;

    return n->fF("start", 1);
}

EST_Item *named_daughter(EST_Item *syl, const EST_String &fname, 
			 const EST_String &fval)
{
    if ((daughter1(syl) != 0) && (daughter1(syl)->f(fname) == fval))
	return daughter1(syl);
    if ((daughter2(syl) != 0) && (daughter2(syl)->f(fname) == fval))
	return daughter2(syl);
    return 0;
}

EST_Item *syl_nucleus(EST_Item *syl_struct)
{
    EST_Item *t;
    if (syl_struct == 0)
	return 0;

//    cout << "x " << *syl_struct << endl;

//    cout << "y " << *named_daughter(syl_struct, "sylval", "Rhyme") << endl;

    if ((t = named_daughter(syl_struct, "sylval", "Rhyme")) != 0)
    {
//	cout << "rhyme: " << *t << endl;
	t = named_daughter(t, "sylval", "Nucleus");
//	cout << "nucleus: " << *t << endl;
	return daughter1(t);
    }

    return 0;
}


bool verify_utterance_relations(EST_Utterance &u, const EST_String &names, 
				int err)
{
    EST_StrList s;
    BracketStringtoStrList(names, s);
    int e = 0;
    
    for (EST_Litem *p = s.head(); p; p = next(p))
    {
	if (!u.has_relation(s(p)))
	{
	    cerr << "Error: relation " << s(p)<< " not found in utterance\n";
	    e = 1;
	}
    }

    if (e && err)
	festival_error();
    else if (e)
	return 0;
    return 1;
}

bool verify_relation_features(EST_Relation &r, const EST_String &names, int err)
{
    EST_StrList s;
    BracketStringtoStrList(names, s);
    int e = 0;

    for (EST_Item *t = r.head(); t; t = next(t))    
	for (EST_Litem *p = s.head(); p; p = next(p))
	    if (!t->f_present(s(p)))
	    {
		cerr << "Error: feature " <<s(p)<< " not found on item " << *t
		    << " relation\n";
		e = 1;
	}

    if (e && err)
	festival_error();
    else if (e)
	return 0;
    return 1;
}

bool verify_relation_features(EST_Utterance &u, const EST_String rname, const EST_String &names, int err)
{
    return verify_relation_features(*u.relation(rname), names, err);

}

EST_Item *nth(EST_Relation &r, int n)
{
    int i = 1;
    for (EST_Item *s = r.head(); s; s = next(s), ++i)
	if (n == i)
	    return s;

    cerr << "Couldn't find item " << n << " in relation of length " << 
	r.length() << endl;
    festival_error();
    return 0;
}

EST_Item *nth_leaf(EST_Item *r, int n)
{
    int i = 1;
    EST_Item *p;

    for (p = first_leaf_in_tree(r);
	 p != next_leaf(last_leaf_in_tree(r)); p = next_leaf(p), ++i)
	if (n == i)
	    return p;

    cerr << "Couldn't find item " << n << " in relation \n";
    festival_error();
    return 0;
}



