 /*************************************************************************/
 /*                                                                       */
 /*                Centre for Speech Technology Research                  */
 /*                     University of Edinburgh, UK                       */
 /*                       Copyright (c) 1996,1997                         */
 /*                        All Rights Reserved.                           */
 /*                                                                       */
 /*  Permission to use, copy, modify, distribute this software and its    */
 /*  documentation for research, educational and individual use only, is  */
 /*  hereby granted without fee, subject to the following conditions:     */
 /*   1. The code must retain the above copyright notice, this list of    */
 /*      conditions and the following disclaimer.                         */
 /*   2. Any modifications must be clearly marked as such.                */
 /*   3. Original authors' names are not deleted.                         */
 /*  This software may not be used for commercial purposes without        */
 /*  specific prior written permission from the authors.                  */
 /*                                                                       */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
 /*  THIS SOFTWARE.                                                       */
 /*                                                                       */
 /*************************************************************************/
 /*                                                                       */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)             */
 /*                   Date: Fri Aug 8 1997                                */
 /* --------------------------------------------------------------------- */
 /* LPC residual synthesis.                                               */
 /*                                                                       */
 /*************************************************************************/

#include "festival.h"
#include "UnitDatabase.h"
#include "FramesUnit.h"
#include "frames_lpc_synthesis.h"
#include "module_support.h"
#include "lpc_utils.h"
#include "sigpr/EST_PpMod.h"
#include "sigpr/EST_lpc.h"

static ModuleDescription description =
{
  "frames_lpc_synthesis", 1.0,
  "CSTR",
  "Richard Caley <rjc@cstr.ed.ac.uk>",
  {
    "Residual excited LPC synthesis from Units derrived from the",
    "FramesUnit class.",
    NULL
  },
  {
    { "ModifiedUnit",	"Units to be synthesised." },
    {NULL,NULL}
  },
  {
    { "Join",		"Information on how to join units."},
    {NULL,NULL}
  },
  {
    { "Wave",		"Synthesised waveform." },
    {NULL,NULL}
  },
  {
    { "SmoothDistance",		mpt_natnum,	"0",	"Distance over which joins between frames are smoothed" },
    { "Preemphasis",		mpt_float,	"0.0",	"Preemphasis used in analysis" },
    { "SaveResidual",		mpt_string,	"",	"If set to a string, the residual used is saved under this name."},
    { "SaveFrames",		mpt_string,	"",	"If set to a string, the frames used are saved under this name."},
    { "SplitSynthesis",		mpt_bool,	"nil",	"Use alternative two-pass synthesis method." },
    { "PpModVoiced",		mpt_string,	"",	"Function to modify voiced pulses." },
    { "PpModUnvoiced",		mpt_string,	"",	"Function to modify unvoiced pulses." },
    { "VerboseSynthesis",	mpt_bool,	"nil",	"Print trace of synthesis." },
    {NULL,NULL,NULL,NULL}
  }
};

static EST_PpMod::Func *pp_mod_func(EST_String name)
{
  EST_PpMod::Func *f = EST_PpMod::mod_function(name);

  if (f==NULL)
    return NULL;


  printf(" -> %s %x\n", (const char *)name, (int)f);

  return f;
}

static void copy_frames(EST_Track &frames, int &at, float &time, int sample_rate, EST_Track &lpcs, EST_Track *mods, int start, int end)
{
  if (mods)
    {
      start =0;
      end = mods->num_frames();
    }

  for(int i=start; i<end;i++)
    {
	int f = mods?((int)(mods->a(i,channel_frame))):i;
	int l;

	for (int c=0; c<lpcs.num_channels(); c++)
	    frames.a(at,c) = lpcs.a(f,c);
	// changed by pault to remove dependence
	cout << "hello\n";
	if (mods->has_channel(channel_length) 
	    || lpcs.has_channel(channel_length))
	{

	    if (mods)
		l = (int)(mods->a(i, channel_length));
	    else
		l = (int)(lpcs.a(f, channel_length));
	    frames.a(at,"frame_len") = l;
	    time += (float)l/sample_rate;
	}
	frames.t(at) = lpcs.t(f);
	if (frames.has_channel(channel_time))
	    frames.a(at,channel_time) = time;
	at++;
    }
  
}


LISP frames_lpc_synthesis(LISP args)
{
  EST_Utterance *utt;

  EST_String unit_stream_name("ModifiedUnit");	
  EST_String join_stream_name("Join");
  EST_String wave_stream_name("Wave");

  EST_Stream *unit_stream=NULL, *join_stream=NULL, *wave_stream=NULL;

  unpack_module_args(args, 
		     utt, 
		     unit_stream_name, unit_stream, sat_existing,
		     join_stream_name, join_stream, sat_as_is,
		     wave_stream_name, wave_stream, sat_replace);

  bool save_residual =  bool_parameter_get("SaveResidual");
  bool save_frames =  bool_parameter_get("SaveFrames");
  bool split_synthesis =  bool_parameter_get("SplitSynthesis");
  int smooth_distance = int_parameter_get("SmoothDistance", 0);
  float preemphasis = float_parameter_get("Preemphasis", 0.0);
  bool verbose =  bool_parameter_get("VerboseSynthesis");


	
  EST_PpMod::Func *pp_mod_func_v = pp_mod_func(string_parameter_get("PpModVoiced"));
  EST_PpMod::Func *pp_mod_func_u = pp_mod_func(string_parameter_get("PpModUnvoiced"));

  const FramesUnit::Chunk *chunks = FramesUnit::chunk_utterance(utt, unit_stream, join_stream);

  if (verbose)
    FramesUnit::dump_chunks(cout, chunks);

  const FramesUnit::Chunk *ch;

  // Preliminary pass to extract information.

  int num_samples=0, num_frames=0, order= -1, sample_rate= -1, num_channels = -1;

  Unit *last_unit=NULL;
  int mod_i=0;
  for(ch=chunks; ch->n != 0; ch++)
    for(int i=0; i < ch->n ; i++)
      {
	if (ch->bit[i].unit == last_unit)
	  mod_i++;
	else
	  mod_i=0;
	last_unit = ch->bit[i].unit;
	  
	EST_Track *lpcs = ch->bit[i].unit->lpc();
	if (!lpcs)
	  return NIL;

	EST_Track *mods = ch->bit[i].unit->modifications(mod_i);

	if (mods)
	  {
	  num_samples  += sum_lengths(*mods, sample_rate);
	  num_frames   += mods->num_frames();
	  }
	else
	  {
	  num_samples  += sum_lengths(*lpcs, sample_rate,
				      ch->bit[i].start_frame, 
				      ch->bit[i].end_frame);
	  num_frames   += ch->bit[i].end_frame - ch->bit[i].start_frame;
	  }

	if (order < 0)
	  order = get_order(*lpcs);
	else if (order != get_order(*lpcs))
	  cwarn << "lpcs of different order " << order << " " << get_order(*lpcs) << "\n";

	if (num_channels < 0)
	  num_channels = lpcs->num_channels();
	else if (lpcs->num_channels() != num_channels)
	  cwarn << "lpcs with different number of channels " << num_channels << " " << lpcs->num_channels() << "\n";

	if (sample_rate <0)
	  sample_rate = ch->bit[i].unit->residual()->sample_rate();

	ch->bit[i].unit->residual();

	if (verbose)
	  cout << "n_s " << num_samples 
	       << " res_len " << ch->bit[i].unit->residual()->num_samples() 
	       << "\n";
      }

  if (verbose)
    cout << "Sum of Lengths " << num_samples << "\n";

  short *sig = new short[num_samples+order];
  short *sig_at = sig;
  int num_samples_left = num_samples;

  if (split_synthesis)
    memset(sig_at, 0, num_samples*sizeof(short));

  short *res=NULL;
  EST_Track *frames=NULL;

  if (save_residual)
    res = new short[num_samples];

  short *res_at = res;
  int frame_at=0;
  float time=0;
  

  while (sig_at < sig+order)
    *(sig_at++) = 0;

  last_unit=NULL;
  mod_i=0;
  for(ch=chunks; ch->n != 0; ch++)
    {
#if 0
      // Really ugly debugging code.
      int check_length = 0;
      int t_mod_i = mod_i;
      Unit * t_last_unit = last_unit;
      for(int ix=0; ix < ch->n ; ix++)
	{
	  if (ch->bit[ix].unit == t_last_unit)
	    t_mod_i++;
	  else
	    t_mod_i=0; 
	  t_last_unit = ch->bit[ix].unit;
	  EST_Track *mods = ch->bit[ix].unit->modifications(t_mod_i);
	  check_length += sum_lengths(*mods);
	}
      cout << "sdur " << check_length/(float)sample_rate << "\n";
#endif
	
      for(int i=0; i < ch->n ; i++)
	{
	  if (verbose)
	    cout << " " << ch->bit[i].unit->name();
	  if (ch->bit[i].unit == last_unit)
	    mod_i++;
	  else
	    mod_i=0;
	  last_unit = ch->bit[i].unit;
	  EST_Track *lpcs = ch->bit[i].unit->lpc();
	  EST_Track *mods = ch->bit[i].unit->modifications(mod_i);
	  EST_Wave *res2 = ch->bit[i].unit->residual();
  
	  if (!res2)
	    return NIL;

	  if (sample_rate < 0)
	    sample_rate = res2->sample_rate();
	  else if (sample_rate != res2->sample_rate())
	    cwarn << "residuals of different sample rate " << sample_rate << " " << res2->sample_rate() << "\n";

	  int frame_num_samples = 
	    mods?sum_lengths(*mods, sample_rate):
	         sum_lengths(*lpcs, sample_rate,
			     ch->bit[i].start_frame, 
			     ch->bit[i].end_frame);

	  // cout << "synth bit " <<  ch->bit[i].unit->name() << " " 
	  // << ch->bit[i].start_frame << " " 
	  // << ch->bit[i].end_frame ;

	  res2->save(ch->bit[i].unit->name() + ".res", "nist");

	  // if (mods) cout << " with mods " << mods->num_frames();
	  // cout << "\n";

	  // Hacky copy until this either uses Wave's or we
	  // get a pointer to the shorts
	  short *data = walloc(short,res2->num_samples());
	  for (int ii=0; ii < res2->num_samples(); ii++)
	      data[i] = res2->a_no_check(i);
	  if (split_synthesis)
	    lpc_resynth_chunk_split
	      (sig_at,
	       *lpcs,
	       data,
	       res2->num_samples(),
	       sample_rate,
	       num_samples_left, 
	       mods?0:ch->bit[i].start_frame, 
	       mods?(mods->num_frames()):
		    (ch->bit[i].end_frame-ch->bit[i].start_frame),
	       order,	       
	       mods,
	       pp_mod_func_v,
	       pp_mod_func_u,
	       res_at
	       );
	  else
	    lpc_resynth_chunk
	      (sig_at,
	       *lpcs,
	       data,
	       res2->num_samples(),
	       sample_rate,
	       num_samples_left, 
	       mods?0:ch->bit[i].start_frame, 
	       mods?(mods->num_frames()):
		    (ch->bit[i].end_frame-ch->bit[i].start_frame),
	       order,	       
	       mods,
	       pp_mod_func_v,
	       pp_mod_func_u,
	       res_at
	       );
	  wfree(data);

	  if (smooth_distance >0 && sig_at-sig > smooth_distance)
	    {
	      float start = sig_at[-smooth_distance];
	      float step  = (start - sig_at[0])/(float)smooth_distance;
	      for(int ii=1; ii<=smooth_distance; ii++)
		sig_at[-ii] = irint(sig_at[0] + ii*step);
	    }

	  sig_at += frame_num_samples;
	  num_samples_left -= frame_num_samples;

	  if (save_residual)
	    res_at += frame_num_samples;

	  if (save_frames)
	    {
	      if (!frames)
		{
		  frames = new EST_Track;
		  frames->copy_setup(*lpcs);
		  frames->resize(num_frames, num_channels);
		  frame_at = 0;
		}
	      
	      copy_frames(*frames, frame_at, time, sample_rate, *lpcs, mods, ch->bit[i].start_frame, ch->bit[i].end_frame);
	    }
	      

	  // cout << "Frame " << (float)frame_num_samples/16000.0 << " " << (float)(sig_at - (sig+order))/16000.0 << "\n";
	}
    }

  if (verbose)
    cout << "\n";


  if (save_residual)
    {
      EST_Wave residual;
      EST_String filename(string_parameter_get("SaveResidual"));

      residual.resize(num_samples,1);
      for (int i=0; i<residual.length(); i++)
	  residual.a_no_check(i) = res[i];
      residual.set_sample_rate(sample_rate);
      delete[] res;
      residual.save(filename, "nist");
    }

  if (save_frames && frames)
    {
      frames->save(string_parameter_get("SaveFrames"), "esps");
    }

  EST_Wave *result = new EST_Wave;

  result->resize(num_samples,1);
  for (int i=0; i<result->length(); i++)
      result->a_no_check(i) = sig[order+i];
  result->set_sample_rate(sample_rate);

  if (preemphasis > 0.0)
    {
/*      EST_Wave *dresult = new EST_Wave;

      lpc_post_deemphasis(*result, *dresult, preemphasis);
      delete result;
      result = dresult;
*/
	post_emphasis(*result, preemphasis);

    }

  EST_Stream_Item item;

  item.init(wave_stream_name);
  item.set_name("[waveform]");

  item.f.set("Wave", (void *)result, gc_wave);

  wave_stream->append(item);

  delete[] sig;
  return NIL;
}

void frames_lpc_synthesis_init(void)
{
  proclaim_module("frames_lpc_synthesis", &description);
  init_module_subr("frames_lpc_synthesis", frames_lpc_synthesis, &description);
}
