/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.i18n;

import java.util.List;
import java.util.LinkedList;
import java.util.Locale;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.Configurable;

/**
 * Used to map bundle information to string representation (e.g. URI),
 * to find the relevant bundle.
 *
 * @author <a href="mailto:neeme@apache.org">Neeme Praks</a>
 * @version CVS $Revision: 1.5 $ $Date: 2002/01/02 19:04:56 $ $Author: neeme $
 */
public class DefaultBundleMatcher implements BundleMatcher, Configurable {

    private List bundleInfos = new LinkedList();
    private String type;

    /**
     * Get the string form of the bundle, based on bundle info.
     *
     * @return      the string form
     */
    public String getType(BundleInfo bundleInfo) {
        if (bundleInfos != null) {
            for (int i = 0; i < bundleInfos.size(); i++) {
                if (match((BundleInfo) bundleInfos.get(i), bundleInfo))
                    return this.type;
            }
            return null;
        }
        return this.type;
    }

    /**
     * Configure the component.
     *
     * @param configuration the configuration
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        this.type = configuration.getAttribute("type");
        Configuration[] bundleInfos = configuration.getChildren("bundle-info");
        for (int i = 0; i < bundleInfos.length; i++) {

            Configurable bundleInfo = null;
            String className = bundleInfos[i].getAttribute("class", null);
            if (className != null) {
                try {
                    bundleInfo = (Configurable) Class.forName(className).newInstance();
                }
                catch (ClassCastException e) {
                    throw new ConfigurationException("bundleinfo implementation does not implement Configurable interface! " +
                                                     "All bundleinfo implementations passed to matchers have to be Configurable!", e);
                }
                catch (Exception e) {
                    throw new ConfigurationException("unable to load bundleinfo implementation", e);
                }
            }
            if (bundleInfo == null) bundleInfo = new ConfigurableBundleInfo();
            bundleInfo.configure(bundleInfos[i]);
            this.bundleInfos.add(bundleInfo);
        }
    }

    protected boolean match(BundleInfo src, BundleInfo compareTo) {
        return
            match(src.getName(), compareTo.getName()) &&
            match(src.getLocale(), compareTo.getLocale()) &&
            match(src.getExtensionParameter(), compareTo.getExtensionParameter());
    }

    protected boolean match(String str, String reference) {
        if (str == null) return true;
        if (reference == null) return false;
        return str.equals(reference);
    }

    protected boolean match(Locale locale, Locale reference) {
        if (locale == null) return true;
        if (reference == null) return false;
        return
            match(locale.getLanguage(), reference.getLanguage()) &&
            match(locale.getCountry(), reference.getCountry()) &&
            match(locale.getVariant(), reference.getVariant());
    }

}
