/*
 * Copyright  The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.util.test;

import junit.framework.TestCase;
import junit.framework.Test;
import junit.framework.TestSuite;
import org.apache.avalon.excalibur.util.StringUtil;

/**
 * This is used to test StringUtil for correctness.
 *
 * @author <a href="mailto:peter@apache.org">Peter Donald</a>
 */
public final class StringUtilTestCase
    extends TestCase
{
    private final static String S1 = "TestMe";
    private final static String S2 = "TestMeTestMe";
    private final static String S3 = "TestMeTestMeTestMe";
    private final static String S4 = "MeeeMer";
    private final static String S5 = "MeeeeMer";

    private final static String P11 = "eT";
    private final static String P21 = "eT";
    private final static String P31 = "eT";
    private final static String P41 = "ee";
    private final static String P51 = "ee";

    private final static String P12 = "Te";
    private final static String P22 = "Te";
    private final static String P32 = "Te";
    private final static String P42 = "ff";
    private final static String P52 = "ff";

    private final static String SR1 = "TestMe";
    private final static String SR2 = "TestMTeestMe";
    private final static String SR3 = "TestMTeestMTeestMe";
    private final static String SR4 = "MffeMer";
    private final static String SR5 = "MffffMer";

    private final static String ST1 = "TestMe";
    private final static String ST2 = "TestMeT";
    private final static String ST3 = "TestMeT";
    private final static String ST4 = "MeeeMer";
    private final static String ST5 = "MeeeeMe";

    private final static String STN1 = "TestMe";
    private final static String STN2 = "Test...";
    private final static String STN3 = "Test...";
    private final static String STN4 = "MeeeMer";
    private final static String STN5 = "Meee...";

    private final static String[] SS1 = new String[] { "T", "stM" };
    private final static String[] SS2 = new String[] { "T", "stM", "T", "stM" };
    private final static String[] SS3 = new String[] { "T", "stM", "T", "stM", "T", "stM" };
    private final static String[] SS4 = new String[] { "M", "M", "r" };
    private final static String[] SS5 = new String[] { "M", "M", "r" };

    private final static String SP  = " ";

    private final static String WU1 = S1 + SP + S2 + SP + S3 + SP + S4 + SP + S5;
    private final static String WW1 =
        "TestMe\n" +
        "TestMeTe\n" +
        "stMe\n" +
        "TestMeTe\n" +
        "stMeTest\n" +
        "Me\n" +
        "MeeeMer\n" +
        "MeeeeMer";
    private final static String WW2 =
        "TestMe\n" +
        "TestMeT\n" +
        "estMe\n" +
        "TestMeT\n" +
        "estMeTe\n" +
        "stMe\n" +
        "MeeeMer\n" +
        "MeeeeMe\n" +
        "r";

    public StringUtilTestCase( final String name )
    {
        super( name );
    }

    public void testReplaceSubString()
        throws Exception
    {
        final String result1 = StringUtil.replaceSubString( S1, P11, P12 );
        final String result2 = StringUtil.replaceSubString( S2, P21, P22 );
        final String result3 = StringUtil.replaceSubString( S3, P31, P32 );
        final String result4 = StringUtil.replaceSubString( S4, P41, P42 );
        final String result5 = StringUtil.replaceSubString( S5, P51, P52 );

        assertEquals( "ReplaceSubString SR1", SR1, result1 );
        assertEquals( "ReplaceSubString SR2", SR2, result2 );
        assertEquals( "ReplaceSubString SR3", SR3, result3 );
        assertEquals( "ReplaceSubString SR4", SR4, result4 );
        assertEquals( "ReplaceSubString SR5", SR5, result5 );
    }

    public void testWordWrap()
        throws Exception
    {
        assertEquals( "WordWrap S1", WW1, StringUtil.wordWrap( WU1, 8, true ) );
        assertEquals( "WordWrap S1", WW2, StringUtil.wordWrap( WU1, 7, true ) );
    }

    public void testTruncate()
        throws Exception
    {
        assertEquals( "Truncate S1", ST1, StringUtil.truncate( S1, 7 ) );
        assertEquals( "Truncate S2", ST2, StringUtil.truncate( S2, 7 ) );
        assertEquals( "Truncate S3", ST3, StringUtil.truncate( S3, 7 ) );
        assertEquals( "Truncate S4", ST4, StringUtil.truncate( S4, 7 ) );
        assertEquals( "Truncate S5", ST5, StringUtil.truncate( S5, 7 ) );
    }

    public void testTruncateNicely()
        throws Exception
    {
        assertEquals( "Truncate Nicely S1", STN1, StringUtil.truncateNicely( S1, 7 ) );
        assertEquals( "Truncate Nicely S2", STN2, StringUtil.truncateNicely( S2, 7 ) );
        assertEquals( "Truncate Nicely S3", STN3, StringUtil.truncateNicely( S3, 7 ) );
        assertEquals( "Truncate Nicely S4", STN4, StringUtil.truncateNicely( S4, 7 ) );
        assertEquals( "Truncate Nicely S5", STN5, StringUtil.truncateNicely( S5, 7 ) );

        assertEquals( "Truncate Nicely 1", ".", StringUtil.truncateNicely( S5, 1 ) );
        assertEquals( "Truncate Nicely 2", "..", StringUtil.truncateNicely( S5, 2 ) );
        assertEquals( "Truncate Nicely 3", "...", StringUtil.truncateNicely( S5, 3 ) );
    }

    public void testSplitString()
        throws Exception
    {
        assertEqualArrays( SS1, StringUtil.split( S1, "e" ) );
        assertEqualArrays( SS2, StringUtil.split( S2, "e" ) );
        assertEqualArrays( SS3, StringUtil.split( S3, "e" ) );
        assertEqualArrays( SS4, StringUtil.split( S4, "e" ) );
        assertEqualArrays( SS5, StringUtil.split( S5, "e" ) );
    }

    public void testConcatStrings()
        throws Exception
    {
        assertEquals( "TestMeTruly", StringUtil.concat("Test", "Me", "Truly"));
        assertEquals( "<prefix:name/>", StringUtil.concat(new String[] {
                                                            "<",
                                                            "prefix",
                                                            ":",
                                                            "name",
                                                            "/>"
                                                          }));
    }

    private void assertEqualArrays( final String[] s1, final String[] s2 )
    {
        assertEquals( "Array Length Equality", s1.length, s2.length );
        assertEquals( "Array Type Equality",
                      s1.getClass().getComponentType(),
                      s2.getClass().getComponentType() );

        for( int i = 0; i < s1.length; i++ )
        {
            assertEquals( "Array Element " + i, s1[ i ], s2[ i ] );
        }
    }
}
