//
// File:        RMIStubSource.java
// Package:     gov.llnl.babel.backend.rmi
// Revision:    @(#) $Id: RMIStubSource.java 4462 2005-03-23 19:29:24Z leek2 $
// Description: generate IOR stub source to a pretty writer stream
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.rmi;

import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.BabelConfiguration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Class <code>RMIStubSource</code> generates an IOR implementation source file
 * to a language writer output stream.  The constructor takes a language
 * writer stream and method <code>generateCode</code> generates intermediate
 * object representation for the specified symbol to the output stream.  The
 * language writer output stream is not closed by this object.
 */
public class RMIStubSource {
    private static int s_longestBuiltin;
    private static final String s_deleteBuiltin = 
                                  IOR.getBuiltinName(IOR.DELETE);
    private static final String s_castBuiltin = IOR.getBuiltinName(IOR.CAST);
    private static final String s_execBuiltin = IOR.getBuiltinName(IOR.EXEC);
    private static final String s_superBuiltin = "_super";

    private static final String s_self          = Utilities.s_self;

    /**
     * Store the SymbolID for sidl.BaseClass, if the extendable being
     * printed is not abstract (i.e., is a concrete class).
     */
    SymbolID d_baseClass = null;

    /**
     * Store the SymbolID for sidl.ClassInfo, if the extendable being
     * printed is not abstract (i.e., is a concrete class).
     */
    SymbolID d_classInfo = null;

    /**
     * Store the SymbolID for sidl.ClassInfoI, if the extendable being
     * printed is not abstract (i.e., is a concrete class).
     */
    SymbolID d_classInfoI = null;

    static {
        s_longestBuiltin = 0;
        for(int j = 0; j < IOR.CLASS_BUILTIN_METHODS; ++j) {
            String mname = IOR.getBuiltinName(j);
            if (mname.length() > s_longestBuiltin) {
                s_longestBuiltin = mname.length();
            }
        }
    }

    private LanguageWriterForC d_writer;

    /**
     * This is a convenience utility function that writes the symbol
     * source information into the provided language writer output
     * stream.  The output stream is not closed on exit.  A code
     * generation exception is thrown if an error is detected.
     */
    public static void generateCode(Symbol symbol, LanguageWriterForC writer)
        throws CodeGenerationException 
    {
        RMIStubSource source = new RMIStubSource(writer);
        source.generateCode(symbol);
    }

    /**
     * Create a <code>RMIStubSource</code> object that will write symbol 
     * information to the provided output writer stream.
     */
    public RMIStubSource(LanguageWriterForC writer) {
        d_writer = writer;
    }

    /**
     * Write IOR source information for the provided symbol to the language
     * writer output stream provided in the constructor.  This method does
     * not close the language writer output stream and may be called for more
     * than one symbol (although the generated source may not be valid input
     * for the C compiler).  A code generation exception is generated if an
     * error is detected.  No code is generated for enumerated and package
     * symbols.
     */
    public void generateCode(Symbol symbol) throws CodeGenerationException {
        if (symbol != null) {
            switch (symbol.getSymbolType()) {
            case Symbol.PACKAGE:
            case Symbol.INTERFACE:
            case Symbol.ENUM:
                break;
            case Symbol.CLASS:
                generateSource((Class) symbol);
                break;
            }
        }
    }

    /**
     * Lookup the SymbolIDs for sidl.BaseClass, sidl.ClassInfo and
     * sidl.ClassInfoI.
     */
    private void lookupSymbolIDs() {
        SymbolTable table = SymbolTable.getInstance();
        d_baseClass = table.
            lookupSymbol(BabelConfiguration.getBaseClass()).getSymbolID();
        d_classInfo = table.
            lookupSymbol(BabelConfiguration.getClassInfo()).getSymbolID();
        d_classInfoI = table.
            lookupSymbol(BabelConfiguration.getClassInfoI()).getSymbolID();
    }

    /**
     * Generate the RStub source for a sidl class .  The source
     * file begins with a banner and include files followed by a declaration
     * of static methods and (for a class) external methods expected in the
     * skeleton file.  For classes, the source file then defines a number
     * of functions (cast, delete, initialize EPVs, new, init, and fini).
     */
    private void generateSource(Class cls) throws CodeGenerationException {
        /*
         * Generate the file banner and include files.
         */
        SymbolID id = cls.getSymbolID();
        SymbolID baseClass = null;
        SymbolID classInfo = null;
        SymbolID classInfoI = null;
        String source = IOR.getSourceFile(id);

        d_writer.printlnUnformatted("#include <stdlib.h>");
        d_writer.printlnUnformatted("#include <string.h>");
        if (!cls.isAbstract()) {
            lookupSymbolIDs();
            d_writer.generateInclude(C.getHeaderFile(d_baseClass), true);
            d_writer.generateInclude(C.getHeaderFile(d_classInfo), true);
        }
        d_writer.printlnUnformatted("#include \"sidl_rmi_ProtocolFactory.h\"");
        d_writer.printlnUnformatted("#include \"sidl_rmi_InstanceHandle.h\"");
        d_writer.printlnUnformatted("#include \"sidl_rmi_Invocation.h\"");
        d_writer.printlnUnformatted("#include \"sidl_rmi_Response.h\"");
        d_writer.println();

        d_writer.printlnUnformatted("#ifndef " + C.NULL);
        d_writer.printlnUnformatted("#define " + C.NULL + " 0");
        d_writer.printlnUnformatted("#endif");
        d_writer.println();

        String my_mutex=IOR.getSymbolName(id) + "__mutex";
        String lock_name=IOR.getLockStaticGlobalsMacroName();
        String unlock_name=IOR.getUnlockStaticGlobalsMacroName();
        String have_lock= IOR.getHaveLockStaticGlobalsMacroName();
        d_writer.printlnUnformatted("#include \"sidl_thread.h\"");
        d_writer.printlnUnformatted("#ifdef HAVE_PTHREAD");
        d_writer.printlnUnformatted("static struct sidl_recursive_mutex_t " 
          + my_mutex + "= SIDL_RECURSIVE_MUTEX_INITIALIZER;");
        d_writer.printlnUnformatted("#define " + lock_name 
          + " sidl_recursive_mutex_lock( &" + my_mutex + " )" );
        d_writer.printlnUnformatted("#define " + unlock_name 
          + " sidl_recursive_mutex_unlock( &" + my_mutex + " )" );
        d_writer.printlnUnformatted("/* #define " + have_lock 
          + " (sidl_recursive_mutex_trylock( &" + my_mutex 
          + " )==EDEADLOCK) */");
        d_writer.printlnUnformatted("#else");
        d_writer.printlnUnformatted("#define " + lock_name);
        d_writer.printlnUnformatted("#define " + unlock_name);
        d_writer.printlnUnformatted("/* #define " + have_lock + " (1) */");
        d_writer.printlnUnformatted("#endif");
        d_writer.println();

        /*
         * Generate internal static variables and external references to be
         * supplied by the skeleton file.
         */
        generateStaticVariables(cls);

        /*
         * Generate remote method support for interfaces and classes.
         */
/*
 * TBD...Need to determine if this appropriate/necessary here.
 *
        boolean doStatic   = cls.hasStaticMethod(true);
        boolean doIceptors = IOR.supportInterceptors(cls);
 */

        generateRemoteCastFunction(cls);
/*
 * TBD...Need to determine if this appropriate/necessary here.
 *
        if (doIceptors) {
          if (doStatic) {
            generateInterceptorFunction(cls, true);
          }
          generateInterceptorFunction(cls, false);
          generateAllInterceptors(cls);
        }
 */
        generateRemoteDeleteFunction(cls);
        generateRemoteExecFunction(cls);
        generateRemoteMethodBodies(cls);
        generateRemoteInitEPV(cls);
/*
 * TBD...Need to determine if this appropriate/necessary here.
 *
        generateStaticFunction(cls, false);
        if (doIceptors) {
          generateStaticFunction(cls, true);
        }
 */
        generateRemoteConstructor(cls);
    }

    /**
     * Generate a single line comment.  This is called out as a separate
     * method to make the code formatting below a little prettier.
     */
    private void comment(String s) {
        d_writer.writeCommentLine(s);
    }

    /**
     * Generate the static variables used to store the EPVs and also the
     * initialization flags for the EPVs.  For interfaces, we only need
     * to generate a remote EPV structure.  Classes require EPVs for their
     * static methods (if present), standard methods, remote methods, and
     * old, new, and remote versions for all parent classes and interfaces.
     */
    private void generateStaticVariables(Class cls) {
        comment("Static variables to hold version of IOR");
        d_writer.println("static const int32_t s_IOR_MAJOR_VERSION = " 
          + IOR.MAJOR_VERSION + ";");
        d_writer.println("static const int32_t s_IOR_MINOR_VERSION = " 
          + IOR.MINOR_VERSION + ";");
        d_writer.println();

        if (!cls.isAbstract()) {
            /* TODO: ClassInfo and RMI?
               comment("Static variable to hold shared ClassInfo interface.");
               d_writer.println("static " + C.getObjectName(d_classInfo) +
               " s_classInfo = " + C.NULL + ";");
               d_writer.println("static int s_classInfo_init = 1;");
               d_writer.println();
            */
        }

        comment("Static variables for managing EPV initialization.");

        /*
         * Output the initialization flags for the EPV structures.
         */
        boolean has_static = cls.hasStaticMethod(true);
        boolean doIceptors = IOR.supportInterceptors(cls);
    
        d_writer.println("static int s_remote_initialized = 0;");
        /* TODO: Support static methods in RMI ? NO!
           if (has_static) {
           d_writer.println("static int s_static_initialized = 0;");
           }*/
        d_writer.println();

        /*
         * Output the EPV, remote EPV, and static EPV for this object.
         */
        IOR.generateStaticEPVVariables(d_writer, cls, false, true, 
                                       IOR.SET_PUBLIC);
/*
 * TBD...Need to determine if this appropriate/necessary here.
 *
        if (doIceptors) {
          IOR.generateStaticEPVVariables(d_writer, cls, has_static,
                                         IOR.SET_INTERCEPTORS);
        }
*/

        /*
         * If the object is a class, then collect all the parents in a set
         * and output EPV structures for the parents.
         */
        Set   parents        = Utilities.getAllParents(cls);
        Set   new_interfaces = Utilities.getUniqueInterfaceIDs(cls);

        if (!parents.isEmpty()) {
            List sorted = Utilities.sort(parents);
            for (Iterator i = sorted.iterator(); i.hasNext(); ) {
                SymbolID p_id    = (SymbolID) i.next();
                String   p_name  = IOR.getSymbolName(p_id);
                String   p_lower = p_name.toLowerCase();
                String   p_epv   = "static " + IOR.getEPVName(p_id);
                boolean  is_old  = !new_interfaces.contains(p_id);
                String   p_epvStr = (is_old ? p_epv + "  " : p_epv + " ");

                d_writer.print(p_epvStr);
                d_writer.println(IOR.getStaticEPVVariable(p_id, IOR.EPV_REMOTE,
                                                         IOR.SET_PUBLIC) + ";");
                d_writer.println();
            }
      	}

/*
 * TBD...Need to determine if this appropriate/necessary here.
 *
        if (has_static && doIceptors) {
            comment("Static variables for interceptor controls.");
            String asCtrls = "static " 
                             + IOR.getControlsStruct(cls.getSymbolID());
            int asWidth = asCtrls.length() + 1;
            d_writer.printAligned(asCtrls, asWidth);
            d_writer.println(IOR.S_CONTROLS + ";");
            d_writer.println();
        }
*/
    }

    /**
     * Generate external references for skeleton routines that define the
     * functions in the EPVs.  A class will have a method EPV.  If there are
     * static functions, then there must also be a static EPV.
     */
    private void generateExternalReferences(Class cls) {
        comment("Declare EPV routines defined in the skeleton file.");

        SymbolID id = cls.getSymbolID();
        d_writer.openCxxExtern();
        d_writer.println("extern void " + IOR.getSetEPVName(id) + "(");
        d_writer.tab();
        d_writer.println(IOR.getEPVName(id) + "* epv);");
        d_writer.backTab();

        if (cls.hasStaticMethod(true)) {
            d_writer.println("extern void " + IOR.getSetSEPVName(id) + "(");
            d_writer.tab();
            d_writer.println(IOR.getSEPVName(id) + "* sepv);");
            d_writer.backTab();
        }
        d_writer.println("extern void " +IOR.getCallLoadName(id) + "(void);");
        d_writer.closeCxxExtern();

        d_writer.println();
    }

    /**
     * Generate the cast function for a class.  This will return null if
     * the cast is invalid and a pointer to the object otherwise.  The logic
     * generates tests for the current class and then recursively queries the
     * parent classes.
     */
    private void generateCastFunction(Class cls) {
        comment("CAST: dynamic type casting support.");

        /*
         * Define the method signature and begin the method implementation.
         */
        SymbolID id = cls.getSymbolID();

        d_writer.println("static void* ior_" + IOR.getSymbolName(id) + '_' 
          + s_castBuiltin + '(');
        d_writer.tab();
        d_writer.println(IOR.getObjectName(id) + "* self,");
        d_writer.println("const char* name)");
        d_writer.backTab();
        d_writer.println("{");
        d_writer.tab();

        /*
         * Output parent pointers to simplify access to parent classes.
         * Recursively generate the casting checks for this class and
         * all parent classes.
         */
        d_writer.println("void* cast = " + C.NULL + ";");
        d_writer.println();

        generateParentSelf(cls, 0, 0);
        d_writer.println();

        generateCastLogic(cls, 0);
        d_writer.println();

        /*
         * Return the cast pointer and close the function body.
         */
        d_writer.println("return cast;");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    /**
     * Recursively generate the cast logic for a class and its interfaces.
     * This routine first generates the check against the class type and
     * then against all newly defined interfaces in the class.  Finally,
     * it recurses on the class parent (if it exists).
     */
    private void generateCastLogic(Class cls, int level) {
        String self = "s" + String.valueOf(level);

        /*
         * Check the cast against the class type.
         */
        if (level > 0) {
            d_writer.print("} else ");
        }
        d_writer.println("if (!strcmp(name, \"" + cls.getFullName() + "\")) {");
        d_writer.tab();
        d_writer.println("cast = (void*) " + self + ";");
        d_writer.backTab();

        /*
         * Check the cast against all newly defined interfaces in this class.
         */
        List interfaces = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
        for (Iterator i = interfaces.iterator(); i.hasNext(); ) {
            SymbolID id    = (SymbolID) i.next();
            String   fn    = id.getFullName();
            String   lower = IOR.getSymbolName(id).toLowerCase();;
            d_writer.println("} else if (!strcmp(name, \"" + fn + "\")) {");
            d_writer.tab();
            d_writer.println("cast = (void*) &" + self + "->d_" + lower + ";");
            d_writer.backTab();
        }

        /*
         * If there is a parent class, then recursively call the cast function
         * on the parent class.  Otherwise, close the if test block.
         */
        Class parent = cls.getParentClass();
        if (parent != null) {
            generateCastLogic(parent, level+1);
        } else {
            d_writer.println("}");
        }
    }

    /**
     * Returns the controls variable base for the specified version.
     */
    private String getBaseControls(boolean doStatic, String self) {
      return (doStatic ? IOR.S_CONTROLS : self + "->" + IOR.D_CONTROLS);
    }

    /**
     * Returns the name of the built-in method, prepending "ior_" and the
     * name of the symbol.
     */
    private String getRemoteMethodName(SymbolID id, String name) {
      return "remote_" + IOR.getSymbolName(id) + '_' + name;
    }

    /**
     * Generate call to the base method.
     */
    private void generateBaseCall(String name, String suffix, String var,
                                  List args, boolean isStatic, boolean doThrows,
                                  Type returnType)
      throws CodeGenerationException
    {
       boolean isOrig = suffix.equals("");
       if (  (isOrig) && (returnType != null)
          && (returnType.getType() != Type.VOID) ) 
       {
         d_writer.print(C.FUNCTION_RESULT + " = ");
       }
       //d_writer.print("(*" + var + "->");
       d_writer.print("(" + var + "->");
       d_writer.println(IOR.getVectorEntry(name + suffix) + ")(");
       d_writer.tab();
       d_writer.tab();
       C.generateArguments(d_writer, s_self, args, isStatic, doThrows,
         ((suffix.equals(IOR.GENERIC_POST_SUFFIX)) ? returnType : null), false,
         false, false);
       d_writer.println(");");
       d_writer.backTab();
       d_writer.backTab();
    }

    /**
     * Return the specified built-in error check function name.
     */
    private String getBuiltinInterceptorName(SymbolID id, boolean doStatic) {
      return getRemoteMethodName(id, IOR.getBuiltinName(IOR.INTERCEPTORS, 
                              doStatic));
    }
  
    /**
     * Generate the specified function to set interceptors activation.
     */
    private void generateInterceptorFunction(Class cls, boolean doStatic) {
      String desc = doStatic ? "static " : "";
      comment("INTERCEPTORS: set " + desc + "interceptor activation.");
  
      SymbolID id   = cls.getSymbolID();
      d_writer.println("static void " + getBuiltinInterceptorName(id, doStatic)
        + '(');
      d_writer.tab();
      if (!doStatic) {
        d_writer.println(IOR.getObjectName(id) + "* " + s_self + ",");
      }
      d_writer.println("int on)");
      d_writer.backTab();
      d_writer.println("{");
      d_writer.tab();
  
      if (IOR.supportInterceptors(cls)) {
        String base = getBaseControls(doStatic, s_self) + ".";
        String use  = base + IOR.D_INTERCEPTORS;
        d_writer.println(use + " = on;");
        if (!doStatic) {
          d_writer.println();
          d_writer.println("/* Ensure the EPVs are set properly. */");
          String publicepv = s_self + "->" + IOR.getEPVVar(IOR.PUBLIC_EPV);
          d_writer.println("if (" + use + ") {");
          d_writer.tab();
          d_writer.println(publicepv + "  = &"
            + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_INTERCEPTORS)
            + ";");
          d_writer.backTab();
          d_writer.println("} else {");
          d_writer.tab();
          d_writer.println(publicepv + "  = &"
            + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
          d_writer.backTab();
          d_writer.println("}");
          d_writer.println("");
        }
      } else {
        comment("Nothing to do since interceptor support not needed.");
      }
  
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }
  
    /**
     * Returns the name of the interceptor method for the specified method.
     */
    private String getInterceptorMethodName(Extendable ext, Method meth) {
      return "iceptor_" + IOR.getSymbolName(ext.getSymbolID()) + '_'
             + meth.getLongMethodName();
    }
    /**
     * Generate all interceptor functions.  These are the functions that
     * invoke the internal pre- and post-method calls before invoking the
     * actual call.  It is assumed these methods are only generated when
     * interceptor generation is enabled.
     */
    private void generateAllInterceptors(Class cls) 
       throws CodeGenerationException
    {
      ArrayList invariants = cls.getAllInvariants();
  
      List     normal = (List) cls.getMethods(true);
      Iterator i      = normal.iterator();
      Method   meth   = null;
      while (i.hasNext()) {
        meth = (Method)i.next();
        generateInterceptor(cls, meth);
      }
    }
  
    /**
     * Generate the interceptor method associated with the specified method.
     */
    private void generateInterceptor(Class cls, Method meth)
      throws CodeGenerationException
    {
      SymbolID  id        = cls.getSymbolID();
      String    name      = meth.getLongMethodName();
      String    methname  = getInterceptorMethodName(cls, meth);
      Type      type      = meth.getReturnType();
      String    retStr    = getReturnString(type);
      String    var;
      boolean   hasReturn = false;
      if ( (type != null) && (type.getType() != Type.VOID) ) {
         hasReturn = true;
      }
      comment("Sandwich the execution of the method between the pre- and post- "
        + "calls.");
      d_writer.println("static " + retStr + " " + methname + "(");
      d_writer.tab();
      List args = meth.getArgumentList();
      boolean hasThrows = !meth.getThrows().isEmpty();
      boolean isStatic  = meth.isStatic();
      C.generateArguments(d_writer, IOR.getObjectName(id) + "* " + s_self, args,
                          isStatic, hasThrows, null, true, true, false);
      d_writer.println(")");
      d_writer.backTab();
  
      d_writer.println("{");
      d_writer.tab();
  
      boolean addBlank = false;
      if (hasReturn) {
         d_writer.println(retStr + " " + C.FUNCTION_RESULT + ";");
         addBlank = true;
      }
      if (isStatic) {
         var = "sepv";
         d_writer.println(IOR.getSEPVName(id) + "* " + var + " = "
           + IOR.getLocalStaticsName(id) + "("
           + IOR.getStaticTypeOption(id, IOR.SET_PUBLIC) + ");");
         addBlank = true;
      } else {
         var = "epv";
         d_writer.println(IOR.getEPVName(id) + "* " + var + " = &"
           + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
         addBlank = true;
      }
      if (addBlank) {
         d_writer.println();
      }
      generateBaseCall(name, IOR.GENERIC_PRE_SUFFIX, var, args, isStatic, false,
                       null);
      generateBaseCall(name, "", var, args, isStatic, hasThrows, type);
      generateBaseCall(name, IOR.GENERIC_POST_SUFFIX, var, args, isStatic, 
                       false, type);
  
      if (hasReturn) {
        d_writer.println();
        d_writer.println("return " + C.FUNCTION_RESULT + ";");
      }
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }

    /**
     * Generate a return string for the specified SIDL type.  Most
     * of the SIDL return strings are listed in the static structures defined
     * at the start of this class.  Symbol types and array types require
     * special processing.
     */
    private static String getReturnString(Type type)
      throws CodeGenerationException
    {
      return IOR.getReturnString(type, true, false);
    }

    /**
     * Call the destructor for the object and then deallocate the associated
     * storage using <code>free</code>.
     */
    private void generateDeleteFunction(Class cls) {
        comment("DELETE: call destructor and free object memory.");

        SymbolID id   = cls.getSymbolID();
        String   name = IOR.getSymbolName(id);

        d_writer.println("static void ior_" + name + '_' + s_deleteBuiltin 
          + '(');
        d_writer.tab();
        d_writer.println(IOR.getObjectName(id) + "* self)");
        d_writer.backTab();

        d_writer.println("{");
        d_writer.tab();

        d_writer.println(IOR.getFiniName(id) + "(self);");
        d_writer.println("memset((void*)self, 0, sizeof(" 
          + IOR.getObjectName(id) + "));");
        d_writer.println("free((void*) self);");

        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    /**
     * Call the destructor for the object and then deallocate the associated
     * storage using <code>free</code>.
     */
    private void generateSuperFunction(Class cls) {
        comment("SUPER: return's parent's non-overrided EPV");
  
        SymbolID id   = cls.getSymbolID();
        String   name = IOR.getSymbolName(id);
        Class  parent = cls.getParentClass();
        if (parent != null ) { 
            SymbolID pid = parent.getSymbolID();

            d_writer.println("static " + IOR.getEPVName(pid) + "* " 
              + IOR.getSymbolName(id) + '_' + s_superBuiltin + "(void) {");
            d_writer.tab();
            d_writer.println( "return " 
              + IOR.getStaticEPVVariable(pid, IOR.EPV_OLD, IOR.SET_PUBLIC)
              + ';');
            d_writer.backTab();
            d_writer.println("}");
            d_writer.println();
        }
    }

    /**
     * Generate the function that initializes the method entry point vector
     * for this class.  This class performs three functions.  First, it saves
     * the EPVs from the parent classes and interfaces (assuming there is a
     * parent).  Second, it fills the EPV for this class using information from
     * the parent (if there is a parent, and NULL otherwise).  It then calls
     * the skeleton method initialization function for the EPV.  Third, this
     * function resets interface EPV pointers and parent EPV pointers to use
     * the correct function pointers from the class EPV structure.
     */
    private void generateInitEPV(Class cls) throws CodeGenerationException {
        comment("EPV: create method entry point vector (EPV) structure.");
        final String epvVar = "epv";

        /*
         * Declare the method signature and open the implementation body.
         */
        SymbolID id         = cls.getSymbolID();
        String   name       = IOR.getSymbolName(id);
        String   object     = IOR.getObjectName(id);
        boolean  doIceptors = IOR.supportInterceptors(cls);

        d_writer.println("static void " + name + "__init_epv(");
        d_writer.tab();
        comment( "assert( " + IOR.getHaveLockStaticGlobalsMacroName() + " );" );
        d_writer.println(object + "* self)");
        d_writer.backTab();

        d_writer.println("{");
        d_writer.tab();

        /*
         * Output pointers to the parent classes to simplify access to
         * their data structures.
         */
        if (cls.getParentClass() != null) {
            generateParentSelf(cls, 0, 0);
            d_writer.println();
        }

        /*
         * Output entry point vectors aliases for each parent class and
         * interface as well as a special one for the current object.
         */
        List parents = Utilities.sort(Utilities.getAllParents(cls));

        /*
         * Save all parent interface and class EPVs in static pointers.
         */
        Class parent = cls.getParentClass();
        if (parent != null) {
            saveEPVs(parent, 1);
        }

        /*
         * Generate a list of the nonstatic methods in the class and get
         * the width of every method name.
         */
        List methods = (List) cls.getNonstaticMethods(true);
        int  mwidth  = Math.max(Utilities.getWidth(methods), s_longestBuiltin) 
                       + IOR.getVectorEntry("").length();

        /*
         * Output builtin methods.
         */
        for (int j = 0; j < IOR.CLASS_BUILTIN_METHODS; j++) {
          if (  IOR.isBuiltinBasic(j)
             || ( (j == IOR.INTERCEPTORS) && doIceptors ) ) {
            String mname = IOR.getBuiltinName(j);
            d_writer.print(epvVar + "->");
            d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
            if (  (j == IOR.CONSTRUCTOR) || (j == IOR.DESTRUCTOR)
               || (j == IOR.CHECK_ERROR) )
            {
                d_writer.println(" = " + C.NULL + ";");
            } else {
                d_writer.println(" = " + getRemoteMethodName(id, mname) + ';');
            }
          }
        }

        /*
         * Iterate through all of the nonstatic methods.  Assign them to
         * NULL if the parent class does not have the method and to the parent
         * function pointer if the parent has the method.
         */
        for (Iterator i = methods.iterator(); i.hasNext(); ) {
            Method method = (Method) i.next();
            String mname  = method.getLongMethodName();
            String ename  = IOR.getVectorEntry(mname);
            boolean parentHas = (parent != null) 
                                && (parent.hasMethodByLongName(mname, true));
            if (doIceptors) {
                d_writer.print(epvVar + "->");
                d_writer.printAligned(ename + IOR.GENERIC_PRE_SUFFIX, mwidth);
                d_writer.println(" = " + C.NULL + ";");
                generateNonstaticEPV(method, object, epvVar, ename, parentHas, 
                                     mwidth);
                d_writer.print(epvVar + "->");
                d_writer.printAligned(ename + IOR.GENERIC_POST_SUFFIX, mwidth);
                d_writer.println(" = " + C.NULL + ";");
            } else {
                generateNonstaticEPV(method, object, epvVar, ename, parentHas, 
                                     mwidth);
            }
        }
        d_writer.println();

        /*
         * Call the user initialization function to set up the EPV structure.
         */
        // d_writer.println(IOR.getSetEPVName(id) + "(" + epvVar + ");");
        // d_writer.println();

        /*
         * Now, if necessary, clone the static that has been set up and
         * overlay with local methods for assertions and/or interceptors.
         */
        d_writer.println();
        if (doIceptors) {
          String xEPV = "iepv";
          d_writer.println("memcpy((void*)" + xEPV + ", " + epvVar + ", sizeof("
            + IOR.getEPVName(id) + "));");
          generateEPVMethodAssignments(cls, IOR.SET_INTERCEPTORS, xEPV, false);
          d_writer.println();
        }

        /*
         * Iterate through all parent EPVs and set each of the function pointers
         * to use the corresponding function in the parent EPV structure.
         */
        copyEPVs(parents);
        d_writer.println("s_method_initialized = 1;");
        // d_writer.println("ior_" + name + "__ensure_load_called();");

        /*
         * Close the function definition.
         */
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    /**
     * Generate the non-static method epv entry for the method.
     */
    private void generateNonstaticEPV(Method m, String object, String var,
                                      String ename, boolean parentHas, 
                                      int width)
      throws CodeGenerationException
    {
      d_writer.print(var + "->");
      d_writer.printAligned(ename, width);
      if (parentHas) {
        d_writer.print(" = ");
        d_writer.print(IOR.getCast(m, object + "*"));
        d_writer.println(" s1->" + IOR.getEPVVar(IOR.PUBLIC_EPV) + "->" + ename
          + ";");
      } else {
        d_writer.println(" = " + C.NULL + ";");
      }
    }

    /**
     * Generate the function that initializes the static entry point
     * vector (assuming the class contains static methods).  If the
     * parent class contains static methods, then copy the pointers
     * from the parent class, since static methods are "inherited" from
     * the parent (in a funny kinda way).  Call the skeleton-supplied
     * function to initialize the remaining static methods.
     */
    private void generateInitSEPV(Class cls) throws CodeGenerationException {
        if (cls.hasStaticMethod(true)) {
            comment("SEPV: create the static entry point vector (SEPV).");

            /*
             * Define the method signature and begin the method body.
             */
            SymbolID id    = cls.getSymbolID();
            String   name  = IOR.getSymbolName(id);
            String   lower = name.toLowerCase();
            int      width = name.length();

            d_writer.println("static void " + name + "__init_sepv(void)");
            d_writer.println("{");
            d_writer.tab();
            comment( "assert( " + IOR.getHaveLockStaticGlobalsMacroName() 
              + " );" );

            /*
             * Get information about the parent (if it exists and has static
             * methods).
             */
            Class    parent        = cls.getParentClass();
            SymbolID pid           = null;
            boolean  includeParent = (parent != null) 
                                     && parent.hasStaticMethod(true);
            boolean  doIceptors    = IOR.supportInterceptors(cls);

            if (includeParent) {
                pid = parent.getSymbolID();
                int w = IOR.getSymbolName(pid).length();
                if (w > width) {
                    width = w;
                }
            }
            width += "struct __sepv*".length();

            /*
             * Create pointers to the local static EPV structure and one
             * to the parent SEPV structure (if the parent exists and has
             * static members).
             */
            d_writer.printAligned(IOR.getSEPVName(id) + "*", width);
            d_writer.println(" s = &" 
              + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_PUBLIC) 
              + ";");
            if (doIceptors) {
              d_writer.printAligned(IOR.getSEPVName(id) + "*", width);
              d_writer.println("is = &"
                + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, 
                                           IOR.SET_INTERCEPTORS) + ";");
            }
            if (includeParent) {
                d_writer.printAligned(IOR.getSEPVName(pid) + "*", width);
                d_writer.println(" p = " + IOR.getStaticsName(pid) + "();");
            }
            d_writer.println();

            /*
             * Iterate through all of the static methods.  Assign them to NULL
             * if the parent class does not have the method and to the parent
             * function pointer if the parent has the method.
             */
            String var = "s";
            List methods = (List) cls.getStaticMethods(true);
            int  mwidth  = Utilities.getWidth(methods) 
                           + IOR.getVectorEntry("").length()
                           + IOR.GENERIC_SUFFIX_MAXLEN + var.length() + 2;

            for (Iterator i = methods.iterator(); i.hasNext(); ) {
                Method method = (Method) i.next();
                String mname  = method.getLongMethodName();
                String ename  = IOR.getVectorEntry(mname);
                boolean parentHas = (parent != null)
                                    && (parent.hasMethodByLongName(mname,true));
                if (doIceptors) {
                    d_writer.print(var + "->");
                    d_writer.printAligned(ename + IOR.GENERIC_PRE_SUFFIX, 
                                          mwidth);
                    d_writer.print(" = " + C.NULL + ";");
                    generateStaticEPV(var, ename, parentHas, mwidth);
                    d_writer.print(var + "->");
                    d_writer.printAligned(ename + IOR.GENERIC_POST_SUFFIX, 
                                          mwidth);
                    d_writer.print(" = " + C.NULL + ";");
                } else {
                    generateStaticEPV(var, ename, parentHas, mwidth);
                }
            }
            d_writer.println();

            /*
             * Initialize the remainder of the static functions by calling
             * the skeleton initialization routine.
             */
            d_writer.println(IOR.getSetSEPVName(id) + "(s);");
            d_writer.println();
            d_writer.println("s_static_initialized = 1;");
            d_writer.println("ior_" + name + "__ensure_load_called();");

            /*
             * Close the initialization guard and end the function body.
             */
            d_writer.backTab();
            d_writer.println("}");
            d_writer.println();
        }
    }

    /**
     * Generate the static method epv entry for the method.
     */
    private void generateStaticEPV(String var, String ename, boolean parentHas,
                                   int mwidth)
      throws CodeGenerationException
    {
      d_writer.print(var + "->");
      d_writer.printAligned(ename, mwidth);
      if (parentHas) {
        d_writer.println(" = p->" + ename + ";");
      } else {
        d_writer.println(" = " + C.NULL + ";");
      }
    }

    /**
     * Generate the appropriate assertion or interceptor methods assignments.
     */
    void generateEPVMethodAssignments(Extendable ext, int setType, 
                                      String ptrVar, boolean doStatics)
    {
      List     methods = (doStatics ? (List)ext.getStaticMethods(true)
                                    : (List)ext.getNonstaticMethods(true));
      Iterator i       = methods.iterator();
      Method   m       = null;
      String   mName;
      int      mwidth  = Math.max(Utilities.getWidth(methods), s_longestBuiltin)
                         + IOR.getVectorEntry("").length()
                         + ptrVar.length() + 2;
      while (i.hasNext()) {
        m     = (Method)i.next();
        mName = m.getLongMethodName();
        d_writer.printAligned(ptrVar + "->"
          + IOR.getVectorEntry(mName), mwidth);
        d_writer.println(" = " + getInterceptorMethodName(ext, m) + ";");
      }
    }

    /**
     * Generate a function that will return a pointer to the static entry
     * point vector.  This function is called by clients to get access to
     * static methods in the class.  If this class has no static methods,
     * then do not generate an implementation.
     */
    private void generateStaticFunction(Class cls, boolean doLocal) {
      boolean  doIceptors = IOR.supportInterceptors(cls);

      if (  cls.hasStaticMethod(true) && ((!doLocal) || (doIceptors)) ) {
        String desc = doLocal ? "specified " : "";
        comment("STATIC: return pointer to " + desc + "static EPV structure.");
  
        SymbolID id         = cls.getSymbolID();
        String   name       = IOR.getSymbolName(id);
        String   lower      = name.toLowerCase();
  
        d_writer.println(IOR.getSEPVName(id) + "*");
        if (doLocal) {
          d_writer.println(IOR.getLocalStaticsName(id) + "(int type)");
        } else {
          d_writer.println(IOR.getStaticsName(id) + "(void)");
        }
        d_writer.println("{");
        d_writer.tab();
  
        d_writer.println( IOR.getLockStaticGlobalsMacroName() + ";" );
  
        if (doIceptors) {
          d_writer.println(IOR.getSEPVName(id) + "* sepv;");
        }
        d_writer.println("if (!s_static_initialized) {");
        d_writer.tab();
        d_writer.println(name + "__init_sepv();");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println( IOR.getUnlockStaticGlobalsMacroName() + ";" );
  
        boolean cont = false;
        if (doIceptors) {
          if (cont) {
            d_writer.print(" else ");
          }
          if (doLocal) {
            d_writer.println("if (type == "
              + IOR.getStaticTypeOption(id, IOR.SET_INTERCEPTORS) + ") {");
          } else {
            d_writer.println("if (" + getBaseControls(true, "") + "."
              + IOR.D_INTERCEPTORS + ") {");
          }
          d_writer.tab();
          d_writer.println("sepv = &"
            + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_INTERCEPTORS)
            + ";");
          d_writer.backTab();
          d_writer.print("}");
        }
        if (doIceptors) {
          d_writer.println(" else {");
          d_writer.tab();
          d_writer.println("sepv = &"
            + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_PUBLIC) 
            + ";");
          d_writer.backTab();
          d_writer.println("}");
          d_writer.println("return sepv;");
        } else {
          d_writer.println("return &"
            + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_PUBLIC) 
            + ";");
        }
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
      }
    }

    /**
     */
    private void generateEnsureLoad(Class cls ) { 
        SymbolID id    = cls.getSymbolID();
        String   name  = IOR.getSymbolName(id);
        String   lower = name.toLowerCase();
    
        d_writer.println("static void ior_" + name 
          + "__ensure_load_called(void) {");
        d_writer.tab();
        comment( "assert( " + IOR.getHaveLockStaticGlobalsMacroName() + " );" );
        d_writer.println("if (! s_load_called ) {");
        d_writer.tab();
        d_writer.println(name + "__call_load();");
        d_writer.println("s_load_called=1;");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.backTab();
        d_writer.println("}");
    }

    /**
     * Generate a function that will return a pointer to the static entry
     * point vector.  This function is called by clients to get access to
     * static methods in the class.  If this class has no static methods,
     * then do not generate an implementation.
     */
    private void generateStaticFunction(Class cls) {
        if (cls.hasStaticMethod(true)) {
            comment("STATIC: return static EPV structure for static methods.");

            SymbolID id    = cls.getSymbolID();
            String   name  = IOR.getSymbolName(id);
            String   lower = name.toLowerCase();

            d_writer.println(IOR.getSEPVName(id) + "*");
            d_writer.println(IOR.getStaticsName(id) + "(void)");
            d_writer.println("{");
            d_writer.tab();

            d_writer.println( IOR.getLockStaticGlobalsMacroName() + ";" );
            d_writer.println("if (!s_static_initialized) {");
            d_writer.tab();
            d_writer.println(name + "__init_sepv();");
            d_writer.backTab();
            d_writer.println("}");
            d_writer.println( IOR.getUnlockStaticGlobalsMacroName() + ";" );

            d_writer.println("return &"  
              + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_PUBLIC) 
              + ";");

            d_writer.backTab();
            d_writer.println("}");
            d_writer.println();
        }
    }

    private void generateInitClassInfo(Class cls) {
        if (!cls.isAbstract()) {
            if (d_classInfoI.equals(cls.getSymbolID())){
                d_writer.println("static void initMetadata(" 
                  + IOR.getObjectName(cls.getSymbolID()) + "*);");
                d_writer.println();
            }
            comment("initClassInfo: create a ClassInfo interface if necessary"
              + ".");
            d_writer.println("static void");
            d_writer.println("initClassInfo(" + C.getObjectName(d_classInfo) 
              + " *info)");
            d_writer.println("{");
            d_writer.tab();
            if (! d_classInfoI.equals(cls.getSymbolID())){
                d_writer.println( IOR.getLockStaticGlobalsMacroName() + ";" );
            }else { 
                d_writer.writeCommentLine("ClassInfo for ClassInfoI is a "
                  + "special case");
                d_writer.writeCommentLine(IOR.getUnlockStaticGlobalsMacroName()
                  + ";" );
            }
            d_writer.println("if (s_classInfo_init) {");
            d_writer.tab();
            d_writer.println(C.getObjectName(d_classInfoI) + " impl;");
            d_writer.println("s_classInfo_init = 0;");
            d_writer.println("impl = " 
              + C.getFullMethodName(d_classInfoI, "_create") + "();");
            d_writer.println("s_classInfo = " 
              + C.getFullMethodName(d_classInfo, "_cast") + "(impl);");
            d_writer.println("if (impl) {");
            d_writer.tab();
            d_writer.println(C.getFullMethodName(d_classInfoI, "setName") 
              + "(impl, \"" + cls.getFullName() + "\");");
            d_writer.println(C.getFullMethodName(d_classInfoI, "setIORVersion") 
              + "(impl, s_IOR_MAJOR_VERSION, s_IOR_MINOR_VERSION);");
            if (d_classInfoI.equals(cls.getSymbolID())) {
                d_writer.println("initMetadata(impl);");
            }
            d_writer.backTab();
            d_writer.println("}");
            d_writer.backTab();
            d_writer.println("}");
            d_writer.println("if (s_classInfo) {");
            d_writer.tab();
            d_writer.println("if (*info) {");
            d_writer.tab();
            d_writer.println(C.getFullMethodName(d_classInfo, "deleteRef") 
              + "(*info);");
            d_writer.backTab();
            d_writer.println("}");
            d_writer.println("*info = s_classInfo;");
            d_writer.println(C.getFullMethodName(d_classInfo, "addRef") 
              + "(*info);");
            d_writer.backTab();
            d_writer.println("}");
            d_writer.backTab();
            if (! d_classInfoI.equals(cls.getSymbolID())){
                d_writer.println( IOR.getUnlockStaticGlobalsMacroName() + ";" );
            } else { 
                d_writer.writeCommentLine("ClassInfo for ClassInfoI is a "
                  + "special case");
                d_writer.writeCommentLine(IOR.getUnlockStaticGlobalsMacroName()
                  + ";" );
            }
            d_writer.println("}");
            d_writer.println();
        }
    }

    private void generateInitMetadata(Class cls) {
        if (!cls.isAbstract()) {
            SymbolID id     = cls.getSymbolID();
            String   object = IOR.getObjectName(id);
            comment("initMetadata: store IOR version & class in sidl.BaseClass"
              + "'s data");
            d_writer.println("static void");
            d_writer.println("initMetadata(" + object + "* self)");
            d_writer.println("{");
            d_writer.tab();
            d_writer.println("if (self) {");
            d_writer.tab();
            d_writer.println(C.getDataName(d_baseClass) + " *data = " 
              + C.getDataGetName(d_baseClass) + "(" 
              + C.getFullMethodName(d_baseClass, "_cast") + "(self));");
            d_writer.println("if (data) {");
            d_writer.tab();
            d_writer.println("data->d_IOR_major_version = s_IOR_MAJOR_VERSION"
              + ";");
            d_writer.println("data->d_IOR_minor_version = s_IOR_MINOR_VERSION"
              + ";");
            d_writer.println("initClassInfo(&(data->d_classinfo));");
            d_writer.backTab();
            d_writer.println("}");
            d_writer.backTab();
            d_writer.println("}");
            d_writer.backTab();
            d_writer.println("}");
            d_writer.println();
        }
    }

    /**
     * Allocate the memory for a new instance of the class and then call
     * the constructor (initializer) on the object.  This method is not
     * output for abstract classes.
     */
    private void generateNewFunction(Class cls) {
        if (!cls.isAbstract()) {
            comment("NEW: allocate object and initialize it.");

            SymbolID id     = cls.getSymbolID();
            String   object = IOR.getObjectName(id);

            d_writer.println(object + "*");
            d_writer.println(IOR.getNewName(id) + "(void)");
            d_writer.println("{");
            d_writer.tab();

            d_writer.println(object + "* self =");
            d_writer.tab();
            d_writer.println("(" + object + "*) malloc(");
            d_writer.tab();
            d_writer.println("sizeof(" + object + "));");
            d_writer.backTab();
            d_writer.backTab();

            d_writer.println(IOR.getInitName(id) + "(self);");
            d_writer.println("initMetadata(self);");
            d_writer.println("return self;");

            d_writer.backTab();
            d_writer.println("}");
            d_writer.println();
        }
    }

    /**
     * Generate the initialization function that acts as the constructor
     * for a class object.  The logic is as follows.  First, if there is
     * a parent for this class, call its constructor.  Then, make sure that
     * the EPVs have been initialized.  If there are parents, make sure that
     * all parent EPVs are updated to point to the corrected EPVs.  Then
     * initialize the new interfaces in this object.  Finally, call the
     * user-defined constructor.
     */
    private void generateInitFunction(Class cls) {
        comment("INIT: initialize a new instance of the class object.");

        /*
         * Declare the method signature and open the implementation body.
         */
        SymbolID id = cls.getSymbolID();
        d_writer.println("void " + IOR.getInitName(id) + "(");
        d_writer.tab();
        d_writer.println(IOR.getObjectName(id) + "* self)");
        d_writer.backTab();

        d_writer.println("{");
        d_writer.tab();

        /*
         * Output parent pointers to simplify access to parent classes.
         */
        generateParentSelf(cls, 0, 0);
        d_writer.println();

        /*
         * If there is a parent class, then call its constructor.
         */
        Class parent = cls.getParentClass();
        if (parent != null) {
            String p = IOR.getSymbolName(parent.getSymbolID());
            d_writer.println(IOR.getInitName(parent.getSymbolID()) + "(s1);");
            d_writer.println();
        }

        /*
         * Ensure that the EPV has been initialized.
         */
        d_writer.println( IOR.getLockStaticGlobalsMacroName() + ";" );
        d_writer.println("if (!s_method_initialized) {");
        d_writer.tab();
        d_writer.println(IOR.getSymbolName(id) + "__init_epv(s0);");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println( IOR.getUnlockStaticGlobalsMacroName() + ";" );
        d_writer.println();

        /*
         * Modify the EPVs in parent classes and their interfaces.
         */
        fixEPVs(cls, 0, true);

        /*
         * Iterate through the interfaces defined in this class and allocate
         * the symbol identifier and set the self pointer.
         */
        List interfaces = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
        for (Iterator i = interfaces.iterator(); i.hasNext(); ) {
            SymbolID iid = (SymbolID) i.next();
            String n = IOR.getSymbolName(iid).toLowerCase();
            d_writer.println("s0->d_" + n + ".d_object = self;");
            d_writer.println();
        }

        /*
         * Finally, set the data pointer for this class and allocate the
         * symbol identifier.
         */
        d_writer.println("s0->d_data = " + C.NULL + ";");
        d_writer.println();
        if (IOR.supportInterceptors(cls)) {
          d_writer.println(getBuiltinInterceptorName(id, false) + "(s0, "
            + IOR.DEFAULT_OPTION_INTERCEPTORS + ");");
        }
        d_writer.println();

        /*
         * Call the user constructor now that the object has been constructed.
         */
        d_writer.println(methodCall(cls, "self",
                                    IOR.getBuiltinName(IOR.CONSTRUCTOR),
                                    ""));

        /*
         * Close the function definition.
         */
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    /**
     * Generate the finalization function that acts as the destructor
     * for a class object.  The logic is as follows.  First, call the
     * user-defined destructor for the object.  Deallocate all identifiers
     * for this object.  If there are parents, reset the EPVs to their
     * previous values and call the parent destructor.
     */
    private void generateFiniFunction(Class cls) {
        comment("FINI: deallocate a class instance (destructor).");

        /*
         * Declare the method signature and open the implementation block.
         */
        SymbolID id = cls.getSymbolID();
        d_writer.println("void " + IOR.getFiniName(id) + "(");
        d_writer.tab();
        d_writer.println(IOR.getObjectName(id) + "* self)");
        d_writer.backTab();

        d_writer.println("{");
        d_writer.tab();

        /*
         * Output parent pointers to simplify access to parent classes.
         */
        generateParentSelf(cls, 0, 0);
        d_writer.println();

        /*
         * Call the user-defined destructor for this class.
         */
        d_writer.println(methodCall(cls, "s0", 
                                    IOR.getBuiltinName(IOR.DESTRUCTOR), ""));

        /*
         * If there is a parent class, then reset all parent pointers and
         * call the parent destructor.
         */
        Class parent = cls.getParentClass();
        if (parent != null) {
            d_writer.println();
            fixEPVs(parent, 1, false);
            String p = IOR.getSymbolName(parent.getSymbolID());
            d_writer.println(IOR.getFiniName(parent.getSymbolID()) + "(s1);");
        }

        /*
         * Close the function definition.
         */
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    private void generateVersionFunction(Class cls) {
        comment("VERSION: Return the version of the IOR used to generate this "
          + "IOR.");

        /*
         * Declare the method signature and open the implementation block.
         */
        SymbolID id = cls.getSymbolID();
        d_writer.println("void");
        d_writer.println(IOR.getVersionName(id) + "(int32_t *major, int32_t *"
          + "minor)");
        d_writer.println("{");
        d_writer.tab();
        d_writer.println("*major = s_IOR_MAJOR_VERSION;");
        d_writer.println("*minor = s_IOR_MINOR_VERSION;");
        d_writer.backTab();
        d_writer.println("}");
    }

    /**
     * TODO: Generate the remote cast method body (not yet implemented).
     */
    private void generateRemoteCastFunction(Class cls) {
        comment("REMOTE CAST: dynamic type casting for remote objects.");

        SymbolID id   = cls.getSymbolID();
        String   name = IOR.getSymbolName(id);

        d_writer.println("static void* remote_" + name + '_' + s_castBuiltin 
          + '(');
        d_writer.tab();
        if (cls.isInterface()) {
            d_writer.println("void* self,");
        } else {
            d_writer.println(IOR.getObjectName(id) + "* self,");
        }
        d_writer.println("const char* name)");
        d_writer.backTab();

        d_writer.println("{");
        d_writer.tab();

        d_writer.println("return " + C.NULL + ";");

        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    /**
     * TODO: Generate the remote delete method body (not yet implemented).
     */
    private void generateRemoteDeleteFunction(Class cls) {
        comment("REMOTE DELETE: call the remote destructor for the object.");

        SymbolID id   = cls.getSymbolID();
        String   name = IOR.getSymbolName(id);

        d_writer.println("static void remote_" + name + '_' + s_deleteBuiltin 
          + '(');
        d_writer.tab();
        if (cls.isInterface()) {
            d_writer.println("void* self)");
        } else {
            d_writer.println(IOR.getObjectName(id) + "* self)");
        }
        d_writer.backTab();
        d_writer.println("{");
        d_writer.tab();
        d_writer.println("free((void*) self);");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    /**
     * TODO: Generate the remote exec method body (not yet implemented).
     */
    private void generateRemoteExecFunction(Class cls) 
        throws CodeGenerationException 
    {
        comment("REMOTE EXEC: call the exec function for the object.");

        SymbolID id   = cls.getSymbolID();
        String   name = IOR.getSymbolName(id);
        String   object = IOR.getObjectName(id);
        Method m_exec = C.getExecMethod();

        d_writer.println("static void remote_" + name + '_' + s_execBuiltin 
          + '(');
        d_writer.tab();
        boolean has_throws = !m_exec.getThrows().isEmpty();
        List args = m_exec.getArgumentList();
        
        d_writer.print(object + "* self");
        if ((args.size() > 0) || has_throws) {
          d_writer.println(",");
        }
        
        for (Iterator a = args.iterator(); a.hasNext(); ) {
          Argument arg = (Argument) a.next();
          d_writer.print(IOR.getArgumentWithFormal(arg, false, false, true));
          if (a.hasNext() || has_throws) {
            d_writer.println(",");
          }
        }
        d_writer.println(")");
        //if (cls.isInterface()) {
        //    d_writer.println("void* self)");
        //} else {
        //    d_writer.println(IOR.getObjectName(id) + "* self)");
        //}
        d_writer.backTab();
        d_writer.println("{");
        d_writer.tab();
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    /**
     * TODO: Generate remote method bodies (not yet implemented).
     * TODO: Fix exceptions so they actually get thrown
     */
    private void generateRemoteMethodBodies(Class cls)
        throws CodeGenerationException 
    {
        SymbolID id     = cls.getSymbolID();
        String   name   = IOR.getSymbolName(id);
        String   object = IOR.getObjectName(id);

        /*
         * Generate method bodies for all non-static methods.
         */
        Iterator m = cls.getNonstaticMethods(true).iterator();
        while (m.hasNext()) {
            Method method = (Method) m.next();
            String method_name = method.getLongMethodName();
            boolean hasThrows = !method.getThrows().isEmpty();
            Type returnType = method.getReturnType();
            comment("REMOTE METHOD STUB:" + method_name);

            /*
             * Write the method signature.
             */
            d_writer.println("static " + getReturnString(returnType));
            d_writer.println("remote_" + name + "_" + method_name + "(");
            d_writer.tab();

            /*
             * Write the method arguments.
             */
            C.generateArgumentList(d_writer, "self /* TLD */", cls.isInterface(),
                                   id, method, true, true, true, hasThrows, 
                                   false, false, false); 
            d_writer.println(")");
            d_writer.backTab();

            /*
             * Write the method body.
             */
            d_writer.println("{");
            d_writer.tab();

                
            if (method_name.compareTo("addRef") == 0) { 
                comment("FIXME  need to think through all of these special "
                  + "cases");                
            } else if (  (method_name.compareTo("isSame") == 0) 
                      || (method_name.compareTo("queryInt") == 0) 
                      || (method_name.compareTo("isType") == 0) 
                      || (method_name.compareTo("getClassInfo") == 0))  {
              if ( !hasThrows) { 
                d_writer.println("sidl_BaseInterface _ex = NULL;");
                d_writer.println("sidl_BaseInterface *_ex2 =&_ex;");
              } else { 
                d_writer.println("sidl_BaseInterface *_ex2 =_ex;");
              }
              comment("FIXME  need to think through all of these special "
                      + "cases");                
              d_writer.println("return 0;");
              
              d_writer.println("sidl_rmi_InstanceHandle conn = "
                               + "(sidl_rmi_InstanceHandle)self->d_data;");
              d_writer.println("self->d_data = " + C.NULL + ";");
              d_writer.println("sidl_rmi_InstanceHandle_close(conn, _ex2);");
              d_writer.println("sidl_rmi_InstanceHandle_deleteRef(conn);");
            } else {
              if ( !hasThrows) { 
                d_writer.println("sidl_BaseInterface _ex = NULL;");
                d_writer.println("sidl_BaseInterface *_ex2 =&_ex;");
              } else { 
                d_writer.println("sidl_BaseInterface *_ex2 =_ex;");
              }
              comment("initialize a new invocation");

              d_writer.println("sidl_rmi_InstanceHandle conn = "
                               + "(sidl_rmi_InstanceHandle)self->d_data;");
              d_writer.println("sidl_rmi_Invocation inv = "
                               + "sidl_rmi_InstanceHandle_createInvocation( conn, \"" 
                               + method_name + "\", _ex2 );");
              d_writer.println("sidl_rmi_Response rsvp = " + C.NULL + ";");
              /*      Pretty sure this is extraneous
                      if (hasThrows) { 
                      d_writer.println("sidl_BaseInterface *_ex2 = _ex;");
                      } else {
                      d_writer.println("sidl_BaseInterface *_ex2 = " + C.NULL 
                      + ";");
                      }*/
              
              //TODO: Fix returning objects
              if ( returnType.getType() != Type.VOID ) {
                if(returnType.getDetailedType() >= Type.CLASS) {
                  comment("TODO: Fix returning objects and arrays");
                  d_writer.println( getReturnString(returnType) 
                                    + " _retval = "+ C.NULL+";");
                } else { 
                  d_writer.println( getReturnString(returnType)  
                                    + " _retval;");
                }
              }
              d_writer.println();
              
              comment("pack in and inout arguments");
              List args = method.getArgumentList();
              for (Iterator a = args.iterator(); a.hasNext(); ) {
                Argument arg = (Argument) a.next();
                if (arg.getType().isArray() || arg.getType().isSymbol() 
                    || ( arg.getType().getType() == Type.OPAQUE) ) { 
                  //FIXME: implement the complicated stuff later
                  continue;
                }
                if (  arg.getMode() == Argument.IN 
                      || arg.getMode() == Argument.INOUT ) { 
                  d_writer.println(RMI.packArg("sidl_rmi_Invocation", 
                                               "inv", arg));
                }
              }
              d_writer.println();
                
                comment("send actual RMI request");
                
                d_writer.println("rsvp = sidl_rmi_Invocation_invokeMethod(inv,"
                                 + "_ex2);");
                
                d_writer.println();
                
                if (hasThrows) {
                  comment("check if exception thrown.");
                  comment("FIXME");
                  d_writer.println();
                }

                comment("extract return value");
                if (returnType.isArray() || returnType.isSymbol() 
                   || ( returnType.getType() == Type.OPAQUE)  ) { 
                    //FIXME: implement the complicated stuff later
                    ;
                } else if ( returnType.getType() != Type.VOID ) { 
                    d_writer.println(RMI.unpackType("sidl_rmi_Response", 
                                                    "rsvp", returnType , 
                                                    "_retval", "&_retval") );
                }
                d_writer.println();

                comment("unpack out and inout arguments");
                for (Iterator a = args.iterator(); a.hasNext(); ) {
                    Argument arg = (Argument) a.next();
                    if (arg.getType().isArray() || arg.getType().isSymbol() 
                       || ( arg.getType().getType() == Type.OPAQUE)  ) { 
                        //FIXME: implement the complicated stuff later
                        continue;
                        }
                    if (  arg.getMode() == Argument.OUT 
                       || arg.getMode() == Argument.INOUT ) { 
                        d_writer.println(RMI.unpackArg("sidl_rmi_Response", 
                                                       "rsvp", arg));
                    }
                }
                d_writer.println();

                comment("cleanup and return");
                d_writer.println("sidl_rmi_Response_done(rsvp, _ex2);");
                d_writer.println("sidl_rmi_Invocation_deleteRef(inv);");
                d_writer.println("sidl_rmi_Response_deleteRef(rsvp);");
                if ( returnType.getType() != Type.VOID ) { 
                    d_writer.println( "return _retval;");
                }
            }
            
            d_writer.backTab();
            d_writer.println("}");
            d_writer.println();
        }
    }

    /**
     * Generate the function that initializes the method entry point vector
     * for a remote instance.  This method performs two basic functions.
     * First, it initializes the remote EPV with pointers to the remote stubs
     * in the IOR file.  Second, it sets interface and parent EPV function
     * pointers using the pointer values generated in the first step.
     */
    private void generateRemoteInitEPV(Class cls)
        throws CodeGenerationException 
    {
        comment("REMOTE EPV: create remote entry point vectors (EPVs).");

        /*
         * Decleare the method signature and open the implementation body.
         */
        SymbolID id    = cls.getSymbolID();
        String   name  = IOR.getSymbolName(id);
        String   lower = name.toLowerCase();

        d_writer.println("static void " + name + "__init_remote_epv(void)");
        d_writer.println("{");
        d_writer.tab();
        comment( "assert( " + IOR.getHaveLockStaticGlobalsMacroName() + " );" );

        /*
         * Output entry point vectors aliases for each parent class and
         * interface as well as a special one for the current object.
         */
        List parents = null;
        if (cls.isInterface()) {
            d_writer.print(IOR.getEPVName(id) + "*");
            d_writer.println(" epv = &" 
              + IOR.getStaticEPVVariable(id, IOR.EPV_REMOTE, IOR.SET_PUBLIC) 
              + ";");
            d_writer.println();
        } else {
            parents = Utilities.sort(Utilities.getAllParents((Class) cls));
            aliasEPVs((Class) cls, parents, true);
        }

        /*
         * Generate a list of the nonstatic methods in the class and get
         * the width of every method name.
         */
        List methods = (List) cls.getNonstaticMethods(true);
        int  mwidth  = Math.max(Utilities.getWidth(methods), s_longestBuiltin)
                       + IOR.getVectorEntry("").length()
                       + IOR.GENERIC_POST_SUFFIX.length();

        /*
         * Output standard methods.
         */
        for (int j = 0; j < IOR.CLASS_BUILTIN_METHODS; j++) {
          if (  IOR.isBuiltinBasic(j)
             || ( (j == IOR.INTERCEPTORS) && IOR.supportInterceptors(cls) ) ) {
            String mname = IOR.getBuiltinName(j);
            d_writer.print("epv->");
            d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
            if (  (j == IOR.CONSTRUCTOR) || (j == IOR.DESTRUCTOR)
               || (j == IOR.CHECK_ERROR) )
            {
                d_writer.println(" = " + C.NULL + ";");
            } else {
                d_writer.println(" = " + getRemoteMethodName(id, mname) + ';');
            }
          }
        }

        /*
         * Iterate through all of the nonstatic methods.  Assign them to
         * the remote stub function.
         */
        for (Iterator i = methods.iterator(); i.hasNext(); ) {
            Method method = (Method) i.next();
            String mname  = method.getLongMethodName();
            String ename  = IOR.getVectorEntry(mname);
            if (IOR.supportInterceptors(cls)) {
                d_writer.print("epv->");
                d_writer.printAligned(ename + IOR.GENERIC_PRE_SUFFIX, mwidth);
                d_writer.println(" = " + C.NULL + ";");
                d_writer.print("epv->");
                d_writer.printAligned(ename, mwidth);
                d_writer.println(" = remote_" + name + "_" + mname + ";");
                d_writer.print("epv->");
                d_writer.printAligned(ename + IOR.GENERIC_POST_SUFFIX, mwidth);
                d_writer.println(" = " + C.NULL + ";");
            } else {
                d_writer.print("epv->");
                d_writer.printAligned(ename, mwidth);
                d_writer.println(" = remote_" + name + "_" + mname + ";");
            } 
        }

        /*
         * Iterate through all of the nonstatic methods.  Assign them to
         * NULL if the parent class does not have the method and to the parent
         * function pointer if the parent has the method.
         */
        /*for (Iterator i = methods.iterator(); i.hasNext(); ) {
          Method method = (Method) i.next();
          String mname  = method.getLongMethodName();
          d_writer.print("epv->");
          d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
          if ((parent != null) && (parent.hasMethodByLongName(mname, true))) {
          d_writer.print(" = ");
          d_writer.print(IOR.getCast(method, object + "*"));
          d_writer.println(" s1->" + IOR.getEPVVar(IOR.PUBLIC_EPV) + "->" 
            + IOR.getVectorEntry(mname) + ";");
          } else {
          d_writer.println(" = NULL;");
          }
          }
          d_writer.println();
        */
        /*
         * If this is a class, then iterate through all parent EPVs and set
         * each of the function pointers to use the corresponding function in
         * the class EPV structure.
         */
        if (parents != null) {
            d_writer.println();
            copyEPVs(parents);
        }
        d_writer.println("s_remote_initialized = 1;");

        /*
         * Close the function definition.
         */
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }

    /**
     * Generate remote method support (not yet implemented).
     */
    private void generateRemoteConstructor(Class cls) {
        comment("REMOTE: generate remote instance given URL string.");

        SymbolID id     = cls.getSymbolID();
        String idName = id.getFullName();
        String   name   = IOR.getSymbolName(id);
        String   lower  = name.toLowerCase();
        String   object = IOR.getObjectName(id);

        /*
         * Declare the function prototype.
         */
        d_writer.println("static " + object + "*");
        d_writer.println(IOR.getRemoteName(id) + "(const char *url, sidl_BaseInterface *_ex)");
        d_writer.println("{");
        d_writer.tab();

        /* 
         * 	Data
         */

        d_writer.println("sidl_rmi_InstanceHandle instance = "
          + "sidl_rmi_ProtocolFactory_createInstance(url, \"" + idName + "\", _ex );");

        d_writer.println("if ( instance == NULL) { return NULL; }");
    
        /*
         * Allocate the object data.
         */
        d_writer.println(object + "* self =");
        d_writer.tab();
        d_writer.println("(" + object + "*) malloc(");
        d_writer.tab();
        d_writer.println("sizeof(" + object + "));");
        d_writer.backTab();
        d_writer.backTab();
        d_writer.println();

        /*
         * Output parent pointers to simplify access to parent classes.
         */
        if (!cls.isInterface()) {
            generateParentSelf((Class) cls, 0, 0);
            d_writer.println();
        }

        /*
         * Ensure that the EPV has been initialized.
         */
        d_writer.println( IOR.getLockStaticGlobalsMacroName() + ";" );
        d_writer.println("if (!s_remote_initialized) {");
        d_writer.tab();
        d_writer.println(name + "__init_remote_epv();");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println( IOR.getUnlockStaticGlobalsMacroName() + ";" );
        d_writer.println();

        /*
         * Recursively modify the parent classes and set up interface pointers.
         */
        if (cls.isInterface()) {
            d_writer.println("self->" + IOR.getEPVVar(IOR.PUBLIC_EPV) 
              + "    = &" 
              + IOR.getStaticEPVVariable(id, IOR.EPV_REMOTE, IOR.SET_PUBLIC) 
              + ";");
            d_writer.println("self->d_object = (void*) instance;");
        } else {
            remoteEPVs((Class) cls, 0);
            d_writer.println("self->d_data = (void*) instance;"); 
        }
        d_writer.println();
        d_writer.println("return self;");
        d_writer.backTab();
        d_writer.println("}"); 
    }

    /**
     * Recursively output self pointers to the sidl objects for this class
     * and its parents.  The self pointers are of the form sN, where N is
     * an integer represented by the level argument.  If the width is zero,
     * then the width of all parents is generated automatically.
     */
    private void generateParentSelf(Class cls, int level, int width) {
        if (cls != null) {

            /*
             * Calculate the width of this class and all parents for pretty
             * output.  Ooh, very pretty.
             */
            if (width == 0) {
                Class parent = cls;
                while (parent != null) {
                    int w = IOR.getObjectName(parent.getSymbolID()).length();
                    if (w > width) {
                        width = w;
                    }
                    parent = parent.getParentClass();
                }
            }

            /*
             * Now use the width information to print out symbols.
             */
            SymbolID id = cls.getSymbolID();
            d_writer.printAligned(IOR.getObjectName(id) + "*", width+1);
            d_writer.print(" s" + String.valueOf(level) + " = ");
            if (level == 0) {
                d_writer.println("self;");
            } else {
                d_writer.print("&s" + String.valueOf(level-1) + "->d_");
                d_writer.println(IOR.getSymbolName(id).toLowerCase() + ";");
            }
            generateParentSelf(cls.getParentClass(), level+1, width);
        }
    }

    /*
     * Recursively modify the EPVs in parent classes and set up interface
     * pointers.  Nothing is done if the class argument is null.  The flag
     * is_new determines whether the EPVs are being set to a newly defined
     * EPV or to a previously saved EPV.
     */
    private void fixEPVs(Class cls, int level, boolean is_new) {
        if (cls != null) {
            String epvVar = IOR.getEPVVar(IOR.PUBLIC_EPV);
            fixEPVs(cls.getParentClass(), level+1, is_new);

            /*
             * Update the EPVs for all of the new interfaces in this
             * particular class.
             */
            String self  = "s" + String.valueOf(level);
            String news  = is_new ? "new" : "old";
            List   ifce  = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
            int    width = Utilities.getWidth(ifce) + "d_.".length()
                           + epvVar.length();

            for (Iterator i = ifce.iterator(); i.hasNext(); ) {
                SymbolID id = (SymbolID) i.next();
                String n = IOR.getSymbolName(id).toLowerCase();
                d_writer.print(self + "->");
                d_writer.printAligned("d_" + n + "." + epvVar, width);
                d_writer.print(" = ");
                if (is_new) {
                    d_writer.print("&");
                }
                d_writer.println("s_" + news + "__" + n + ";");
            }

            /*
             * Modify the class entry point vector.
             */
            String lower = IOR.getSymbolName(cls.getSymbolID()).toLowerCase();
            d_writer.print(self + "->");
            d_writer.printAligned(epvVar, width);
            d_writer.print(" = ");
            if (is_new) {
                d_writer.print("&");
            }
            d_writer.println("s_" + news + "__" + lower + ";");
            d_writer.println();
        }
    }

    /*
     * Recursively save the class and interface EPVs in this class and all
     * parent classes.  Nothing is done if the class argument is null.
     */
    private void saveEPVs(Class cls, int level) {
        if (cls != null) {
            String epvVar = IOR.getEPVVar(IOR.SET_PUBLIC);
            saveEPVs(cls.getParentClass(), level+1);

            /*
             * Save the EPVs for all of the new interfaces in this class.
             */
            String self  = "s" + String.valueOf(level);
            List   ifce  = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
            int    width = Utilities.getWidth(ifce);

            for (Iterator i = ifce.iterator(); i.hasNext(); ) {
                SymbolID id = (SymbolID) i.next();
                String n = IOR.getSymbolName(id).toLowerCase();
                d_writer.print(IOR.getStaticEPVVariable(id, IOR.EPV_OLD,
                  IOR.SET_PUBLIC));
                d_writer.printSpaces(width - n.length());
                d_writer.println(" = " + self + "->d_" + n + "." + epvVar 
                  + ";");
            }

            /*
             * Save the class entry point vector.
             */
            String lower = IOR.getSymbolName(cls.getSymbolID()).toLowerCase();
            d_writer.print(IOR.getStaticEPVVariable(cls.getSymbolID(),
              IOR.EPV_OLD, IOR.SET_PUBLIC));
            d_writer.printSpaces(width - lower.length());
            d_writer.println(" = " + self + "->" + epvVar + ";");
            d_writer.println();
        }
    }

    /*
     * Recursively modify the remote EPVs in parent classes and set up
     * the interfaces.
     */
    private void remoteEPVs(Class cls, int level) {
        if (cls != null) {
            String epvVar = IOR.getEPVVar(IOR.SET_PUBLIC);
            remoteEPVs(cls.getParentClass(), level+1);

            /*
             * Update the EPVs for all of the new interfaces in this
             * particular class.
             */
            String self = "s" + String.valueOf(level);
            List   ifce = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
            for (Iterator i = ifce.iterator(); i.hasNext(); ) {
                SymbolID id = (SymbolID) i.next();
                String n = IOR.getSymbolName(id).toLowerCase();
                d_writer.println(self + "->d_" + n + "." + epvVar + "    = &" 
                  + IOR.getStaticEPVVariable(id, IOR.EPV_REMOTE, IOR.SET_PUBLIC)
                  + ";");
                d_writer.println(self + "->d_" + n + ".d_object = (void*) "
                  + "instance;");
                d_writer.println();
            }

            /*
             * Modify the class entry point vector.
             */
            String lower = IOR.getSymbolName(cls.getSymbolID()).toLowerCase();
            d_writer.println(self + "->d_data = (void*) instance;");
            d_writer.println(self + "->" + epvVar + "  = &"
              + IOR.getStaticEPVVariable(cls.getSymbolID(), IOR.EPV_REMOTE, 
                                         IOR.SET_PUBLIC) + ";");
            d_writer.println();
        }
    }

    /**
     * Generate the entry point vector alias for each parent class and each
     * interface as well as a special one for the current object.
     */
    private void aliasEPVs(Class cls, Collection parents, boolean remote) {
        /*
         * Get the width of the symbols for pretty printing.
         */
        SymbolID id      = cls.getSymbolID();
        String   name    = IOR.getSymbolName(id);
        String   epv     = IOR.getEPVName(id);
        int      epvType = remote ? IOR.EPV_REMOTE : IOR.EPV_NEW;

        int width = epv.length() + 1;
        int w     = Utilities.getWidth(parents) + "struct __epv*".length();
        if (w > width) {
            width = w;
        }

        /*
         * Output the EPV pointer for this class and its class and interface
         * parents.
         */
        d_writer.printAligned(epv + "*", width);
        d_writer.println(" epv = &" 
          + IOR.getStaticEPVVariable(id, epvType, IOR.SET_PUBLIC) + ";");
        if (IOR.supportInterceptors(cls)) {
          d_writer.printAligned(epv + "*", width);
          d_writer.println(" iepv = &"
            + IOR.getStaticEPVVariable(id, epvType, IOR.SET_INTERCEPTORS) 
            + ";");
        }


        int e = 0;
        for (Iterator i = parents.iterator(); i.hasNext(); ) {
            SymbolID sid   = (SymbolID) i.next();
            String   lower = IOR.getSymbolName(sid).toLowerCase();
            d_writer.printAligned(IOR.getEPVName(sid) + "*", width);
            d_writer.printAligned(" e" + String.valueOf(e++), 4);
            d_writer.println(" = &" 
              + IOR.getStaticEPVVariable(sid, epvType, IOR.SET_PUBLIC) + ";");
        }
        d_writer.println();
    }

    /*
     * Copy EPV function pointers from the most derived EPV data structure
     * into all parent class and interface EPVs.
     */
    private void copyEPVs(Collection parents) throws CodeGenerationException {
        int e = 0;
        for (Iterator i = parents.iterator(); i.hasNext(); ) {
            /*
             * Extract information about the parent extendable object.
             * Generate a list of the nonstatic methods and calculate
             * the width of every method name.
             */
            SymbolID   id   = (SymbolID) i.next();
            Extendable ext  = (Extendable) Utilities.lookupSymbol(id);
            String     name = IOR.getSymbolName(id);

            List methods = (List) ext.getNonstaticMethods(true);
            int  mwidth  = Math.max(Utilities.getWidth(methods), 
                                    s_longestBuiltin) 
                           + IOR.getVectorEntry("").length();

            /*
             * Calculate the "self" pointer for the cast function.
             */
            String self = null;
            if (ext.isInterface()) {
                self = "void*";
            } else {
                self = IOR.getObjectName(id) + "*";
            }

            /*
             * Generate the assignments for the cast and delete functions.
             */
            String estring = "e" + String.valueOf(e) + "->";
            String vecEntry = IOR.getVectorEntry(s_castBuiltin);

            d_writer.print(estring);
            d_writer.printAligned(vecEntry, mwidth);
            d_writer.print(" = (void* (*)(" + self);
            d_writer.print(",const char*)) epv->");
            d_writer.print(vecEntry);
            d_writer.println(";");

            vecEntry = IOR.getVectorEntry(s_deleteBuiltin);
            d_writer.print(estring);
            d_writer.printAligned(vecEntry, mwidth);
            d_writer.println(" = (void (*)(" + self + ")) epv->" + vecEntry 
              + ";");

            /**
             * Generate the assignment for exec function
             */
            vecEntry = IOR.getVectorEntry(s_execBuiltin);
            Method m_exec = C.getExecMethod();
            String ename  = m_exec.getLongMethodName();
            d_writer.print(estring);
            d_writer.printAligned(IOR.getVectorEntry(ename), mwidth);
            d_writer.print(" = " + IOR.getCast(m_exec, self));
            d_writer.println(" epv->" + IOR.getVectorEntry(ename) + ";");


            /*
             * Iterate over all methods in the EPV and set the method pointer.
             */
            for (Iterator j = methods.iterator(); j.hasNext(); ) {
                Method method = (Method) j.next();
                String mname  = method.getLongMethodName();
                d_writer.print(estring);
                d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
                d_writer.print(" = " + IOR.getCast(method, self));
                d_writer.println(" epv->" + IOR.getVectorEntry(mname) + ";");
            }

            d_writer.println();
            e++;
        }
    }

    private static String methodCall(Extendable ext, String var, String method,
                                     String args)
    {
        String result = "(*(" + var + "->" + IOR.getEPVVar(IOR.PUBLIC_EPV) 
                        + "->" + IOR.getVectorEntry(method) + "))(" + var;
        if (ext.isInterface()) {
            result += "->d_object";
        }
        result += args;
        result += ");";
        return result;
    }

    private void generateReplaceValue(Symbol symbol) {
        d_writer.println("if (result) {");
        d_writer.tab();
        if (symbol instanceof Extendable) {
            d_writer.println("if (value) {");
            d_writer.tab();
            d_writer.println(methodCall((Extendable)symbol, "value", "addRef", 
                                        ""));
            d_writer.backTab();
            d_writer.println("}");
            d_writer.println("if (*result) {");
            d_writer.tab();
            d_writer.println(methodCall((Extendable)symbol, "(*result)", 
                                        "deleteRef", ""));
            d_writer.backTab();
            d_writer.println("}");
        }
        d_writer.println("*result = value;");
        d_writer.backTab();
        d_writer.println("}");
    }

    public static void generateExternalSignature(LanguageWriter lw, Symbol sym,
                                                 String terminator)
    {
        final SymbolID id = sym.getSymbolID();
        lw.beginBlockComment(false);
        lw.println("This function returns a pointer to a static structure of");
        lw.println("pointers to function entry points.  Its purpose is to "
          + "provide");
        lw.println("one-stop shopping for loading DLLs.");
        lw.endBlockComment(false);
        lw.println("const " + IOR.getExternalName(id) + "*");
        lw.println(IOR.getExternalFunc(id) + "(void)" + terminator);
    }

    private void generateExternalFunc(Symbol sym) {
        SymbolID id = sym.getSymbolID();
/*
        if (IOR.supportInterceptors(cls) || IOR.supportAssertions(cls) ) {
            d_writer.println("const " + IOR.getExternalName(id));
        } else {
*/
            d_writer.println("static const " + IOR.getExternalName(id));
/*
        }
*/

        d_writer.println("s_externalEntryPoints = {");
        d_writer.tab();
        if (sym instanceof Class) {
            Class cls = (Class)sym;
            if (!cls.isAbstract()){
                d_writer.println(IOR.getNewName(id) + ",");
            }
            d_writer.println(IOR.getRemoteName(id) + ",");
            if (cls.hasStaticMethod(true)) {
                d_writer.println(IOR.getStaticsName(id) + ",");
            }
            if (cls.getParentClass() != null ) { 
                d_writer.println(IOR.getSymbolName(id) + "__super");
            }
        }
        d_writer.backTab();
        d_writer.println("};");
        d_writer.println();
        generateExternalSignature(d_writer, sym, "");
        d_writer.println("{");
        d_writer.tab();
        d_writer.println("return &s_externalEntryPoints;");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }
}
