/*
    LIB - a librarian for compatible OBJ/LIB files
    Copyright (C) 1995,1996  Steffen Kaiser

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: LIB.C $
   $Locker: ska $	$Name: v3_2 $	$State: Rel $

	Library oriented functions.

*/

#ifndef _MICROC_
#include <string.h>
#include <malloc.h>
#include <io.h>
#else
#include <file.h>
#endif
#include <getopt.h>
#include "types.h"
#include "lib.h"
#include "tmpList.h"
#include "mlib.h"
#include "list.h"
#include "yerror.h"

#ifndef lint
static char const rcsid[] = 
	"$Id: LIB.C 3.3 1999/02/17 05:18:08 ska Rel ska $";
#endif

FLAG skipModule = NUL;	/* don't copy the current module */
FLAG modifyTHEADR = !NUL;	/* modify the module names */
FLAG addModule = NUL;	/* this is a newly added module */
FLAG dblMod = !NUL;		/* deny to add a module with the same name twice */
FLAG alignDir = !NUL;	/* align library directory at 512 boundary */

struct LIB_INFO libHeaderOld, libHeaderNew;
byte readCRC;		/* compute the CRC of the modules */
char *modName = NULL;	/* name of the current OBJ */
word modPage;			/* page number, on which the current OBJ starts */
dword modLength;		/* #of bytes of current OBJ module copied so far */
struct LIB_TRAILER module;	/* temporary buffer for the modules */
#define modType module.type
#define modLen (module.length - 1)	/* spare the Check Sum byte */

#ifdef _MICROC_			/* expand till now undefined macros */
#undef __CONFIG_H
#include "config.h"
#endif

FILEP extFile = NULL;	/* external file where to copy current module to */
char *extFName = NULL;	/* name of extFile */

int dumpTHEADR(struct tmpList *h, FILEP f)
/* write the THEADR module

                T-MODULE Record Format -- T-MODULE Header

                        Ŀ
                           RecordT-MODULE NAMECheck
                        80HLength              Sum 
                        Ĵ
                         1   2       NAME       1  
                         */
{	struct LIB_TRAILER buf;
	int i, j;

	buf.type = NR_LIB_THEADR;
	buf.length = h->symLen + 2;	/* name + name length + check sum */
	if(ww(aS(buf), sizeof(buf), f)	/* module start */
	 || ww(&h->symLen, 1 + h->symLen, f))		/* name length & name */
		return 1;
	for(i = j = h->symLen; j--; i += h->symbol[j]);
	for(j = sizeof(buf); j--; i += ((byte*)aS(buf))[j]);
	i = -i;
	return ww(&i, 1, f);			/* check sum */
}

void closeExtFile(void)
/* end the copying of a module into an external file */
{	if(extFile) {
#ifndef _MICROC_
		if(ferror(extFile))
			error(E_writeFile, extFName);
#endif
		Close(extFile);
		extFile = NULL;
	}
	if(extFName) {
		U_free(extFName);
		extFName = NULL;
	}
	chkHeap()
}

void openExtFile(char *modName, struct tmpList *h, char *p)
/* start the copying of this module into the external file modName */
{	char *name, *ext, *path, *drive;

	closeExtFile();
	if(!fsplit(modName, NULL, NULL, &name, NULL, 0)
	 || !fsplit(p, &drive, &path, NULL, &ext, 0)
	 || (extFName = fmerge(NULL, drive, path, name, ext)) == NULL)
		fatal(E_noMem);

#ifdef _MICROC_
	if((extFile = open(extFName, F_READ | F_BINARY)) == NULL)
#else
	if((extFile = fopen(extFName, "wb")) == NULL)
#endif
		error(E_openFile, extFName);
	dumpTHEADR(h, extFile);

	U_free(drive);
	U_free(path);
	U_free(name);
	U_free(ext);
	chkHeap()
}

void Lread(void *buf, unsigned len, struct LIB_INFO *lib)
/* read len bytes from the library into the buffer buf and
   emit an error message, if necessary */
{	if(rr(buf, len, libFp))
		fatal(E_readFile, libName);

	longaddu(modLength, len);

/* compute CRC */
	while(len--)
		readCRC += *((byte*)buf)++;
}

void LreadMod(struct LIB_TRAILER *buf, struct LIB_INFO *lib)
/* read the next module header from the library.
*/
{	byte c;

/* skip padding characters */
	do Lread(&c, sizeof(c), lib);
	while(c == NUL);
/* CRC calculation begins with the type byte */
	readCRC = buf->type = c;
	Lread(&buf->length, 2, lib);
}

void Lseek(int delta, struct LIB_INFO *lib)
/* seek delta bytes from the current position of the library and
   emit an error message, if necessary */
{	
#ifndef _MICROC_
	if(fseek(libFp, delta, 1 /* SEEK_CUR */))
#else
	if(lseek(libFp, 0, delta, 1 /* SEEK_CUR */))
#endif
		fatal(E_accessFile, libName);
}

#ifndef _MICROC_
void Lset(dword offset, struct LIB_INFO *lib)
/* set the file pointer the offset *offset */
{
	if(fseek(libFp, offset, 0 /* SEEK_SET */))
		fatal(E_accessFile, libName);
}
#else
void Lset(dword *offset, struct LIB_INFO *lib)
/* set the file pointer the offset *offset */
{
	if(lseek(libFp, offset->hi, offset->lo, 0 /* SEEK_SET */))
		fatal(E_accessFile, libName);
}
#endif

int Ltell(unsigned *diff, unsigned *rem, unsigned unit, struct LIB_INFO *lib)
/* calculate: tell(libFp) == (Ltell() * 65536 + (*diff)) * unit + (*rem);
	if diff or rem is NULL, *diff or *rem is not stored.
*/
{	dword pos;

#ifdef _MICROC_
	dword tmp;

	if(ltell(libFp, &pos.hi, &pos.lo))
		error(E_accessFile, libName);
	longsetu(tmp, unit);
	longdiv(pos, tmp);	/* Longreg = pos % unit; pos /= unit; */
	if(rem) *rem = Longreg.lo;
	if(diff) *diff = pos.lo;
	return pos.hi;
#else
	if((pos = ftell(libFp)) == (dword)~0l)
		error(E_accessFile, libName);
	if(rem) *rem = (unsigned)(pos % unit);
	pos /= unit;
	if(diff) *diff = (unsigned)pos;
	return (unsigned long)pos >> (sizeof(unsigned) * 8);
#endif
}

void Lwrite(void *buf, int len, struct LIB_INFO *lib)
/* write len bytes from buf into library, then
   emit a fatal error message, if anything goes wrong
*/
{	if(!skipModule && ww(buf, len, libFp))
		fatal(E_writeFile, libName);
	if(extFile && ww(buf, len, extFile))
		fatal(E_writeFile, extFName);
}

void LwriteMult(byte b, int len, struct LIB_INFO *lib)
/* write the byte b len times into lib */
{	while(len--)
		Lwrite(&b, 1, lib);
}

void LwriteNxtPage(struct LIB_INFO *lib)
/* advance to the next page while padding out that region and
   emit a fatal error message, if anything goes wrong
*/
{	word topad;

	Ltell(NULL, &topad, pageSize, lib);
	if(topad)
		LwriteMult(NUL, pageSize - topad, lib);
}

int trailerSize(struct LIB_INFO *lib)
/* evaluate the difference to the next 512-aligned page */
{	word topad;

	Ltell(NULL, &topad, DIR_PAGE_SIZE, lib);
	return DIR_PAGE_SIZE - topad;
}


void Lrw(void *buf, unsigned len)
{	Lread(buf, len, libOld);
	Lwrite(buf, len, libNew);
}

int Lcreate(char *name, struct LIB_INFO *lib)
/* creates library, initialize the library variables and 
   return 0 if successful */
{	struct LIB_HEADER lhead;

	DB_ENTER("Lcreate");
	DB_PRINT("arg", ("name = %s", name));

	informative(M_createLib, name);
#ifdef _MICROC_		/* does not support '+' */
	if((libFp = creat(libName = name, 0)) != -1) {
#else
	if((libFp = fopen(libName = name, "w+b")) != NULL) {
#endif
		if(pageSize < MINPAGESIZE) {
			pageSize = MINPAGESIZE;
			warning(W_pageSize, MINPAGESIZE);
		}
		lhead.type = NR_LIB_HEADER;
		lhead.length = pageSize - 3;
		Lwrite(aS(lhead), sizeof(lhead), lib);
		LwriteNxtPage(lib);
		chkHeap()
		DB_RETURN(0);		/* OK */
	}
	DB_RETURN(1);
}

void Lcloseup(struct LIB_INFO *lib)
{
	Close(libFp);
	U_free(libName);
}

void Lclose(struct LIB_INFO *lib)
{	struct LIB_HEADER lhead;
#ifdef _MICROC_	
	dword zero;

	longclr(zero);
	Lset(zero, lib);
#else
	DB_ENTER("Lclose");
	DB_PRINT("arg", ("lib = %s", lib->_name));

	Lset(0, lib);
#endif
	lhead.type = NR_LIB_HEADER;
	lhead.length = pageSize - 3;
	longcpy(lhead.dirOff, dirOffset);
	lhead.dirPag = dirPages;
	Lwrite(aS(lhead), sizeof(lhead), lib);
	LwriteNxtPage(lib);
	Lcloseup(lib);
	chkHeap()
	DB_EXIT
}

void Linit(struct LIB_INFO *lib)
/* Initialize the structure members from the library */
{	struct LIB_HEADER lhead;

	DB_ENTER("Linit");
	Lread(aS(lhead), sizeof(lhead), lib);
	if(lhead.type != NR_LIB_HEADER)
		fatal(E_libHeader);
	longcpy(dirOffset, lhead.dirOff);
	pageSize = lhead.length + 3;
	dirPages = lhead.dirPag;
	Lseek(pageSize - sizeof(lhead), lib);
	chkHeap()
	DB_EXIT
}

int Lopen(char *name, struct LIB_INFO *lib)
{
	DB_ENTER("Lopen");
	DB_PRINT("arg", ("name = %s", name));

	informative(M_openLib, name);
#ifdef _MICROC_
	if((libFp = open(libName = name, F_READ | F_BINARY)) != NULL) {
#else
	if((libFp = fopen(libName = name, "rb")) != NULL) {
#endif
		Linit(lib);
		DB_RETURN(0);
	}
	pageSize = MINPAGESIZE;
	libFp = NULL;	/* indicate: no old lib */
	DB_RETURN(1);
}


void chkLibExceed(struct LIB_INFO *lib)
{	dword pos;

#ifdef _MICROC_
	if(ltell(libFp, &pos.hi, &pos.lo))
		error(E_accessFile, libName);
	if(longcmp(pos, dirOffset) > 0)
	 	error(E_libEnd);
#else
	if((pos = ftell(libFp)) == (dword)~0l)
		error(E_accessFile, libName);
	if(pos > dirOffset)
	 	error(E_libEnd);
#endif
}
void chkLibDir(struct LIB_INFO *lib)
{	dword pos;

#ifdef _MICROC_
	if(ltell(libFp, &pos.hi, &pos.lo))
		error(E_accessFile, libName);
	if(longcmp(pos, dirOffset) != 0)
	 	error(E_libEnd);
#else
	if((pos = ftell(libFp)) == (dword)~0l)
		error(E_accessFile, libName);
	if(pos != dirOffset)
	 	error(E_libEnd);
#endif
}

void chkCRC(void)
/* check CRC */
{	byte crc;

	Lrw(&crc, sizeof(crc));
	if(readCRC)
		chkBadCrc()
}
void writeHeader(void)
/* write module header & initiate module copying */
{	Lwrite(aS(module), sizeof(struct LIB_TRAILER), libNew);	}


#define rwIndex(a) 	(readLen += rwIdx(&a))
int rwIdx(byte *a)
{	byte dummy;

	Lrw(a, 1);
	if(*a & 0x80) {
		Lrw(&dummy, 1);
		return 2;		/* two bytes read */
	}
	return 1;
}
struct tmpList *getName(void)
{	struct tmpList *h;
	word len;

	len = 0;			/* Make HiByte zero */
	Lrw(&len, 1);
	h = getmem(sizeof(struct tmpList) + len);
	h->pageNr = modPage;
	h->symLen = len;
	if(len) Lrw(h->symbol, len);
	h->symbol[len] = h->flags = NUL;	/* for a better C-like handling */
	chkHeap()
	return h;
}


void LcopyDEFAULT(void)
{	word len;

	writeHeader();
	for(len = modLen; len > COPYBUF; len -= COPYBUF)
		Lrw(ybuf, COPYBUF);
	Lrw(ybuf, len);
	chkCRC();
}
void LcopyMODEND(void)
/* MODEND == finish the external module */
{	LcopyDEFAULT();
	closeExtFile();
	SwriteSize(modLength);
}
void LcopyTHEADR(void)
/* THEADR == name of OBJ */
{	struct tmpList *h;
	word i;
	char *p, *q;
	struct cmdList *cmd;
	FLAG8 modi;

	DB_ENTER("LcopyTHEADR");

/* check, if the latter external file retrieved no MODEND */
	if(extFile) {
		warning(W_noModEnd, extFName);
		closeExtFile();				/* close the external file */
	}
/* each module starts at a page break */
	LwriteNxtPage(libNew);
	if(Ltell(&modPage, &i, pageSizeNew, libNew) || i) {
		/* restart the librarian with increased page size */
		//DB_LONGJMP(restart, 1);
		ovrrun = !NUL;
#ifdef NICHT_COMPILIEREN
		fatal(E_pageOverflow, pageSizeNew);	/* no page break */
#endif
	}

	skipModule = !NUL;		/* don't copy the name right now, so subtraction
								can be determined */
/* initialize the OBJ module size counting,
	already read: sizeof(struct LIB_TRAILER) */
	longsetu(modLength, sizeof(struct LIB_TRAILER));

/* read in the name */
	h = getName();
	h->flags |= TL_THEADR;
	if(modLen - 1 != h->symLen || modLen < 2)
		error(E_corruptMod, NR_LIB_THEADR, modName);
	chkCRC();

/* modify module name */
	chkHeap()
	U_free(modName);
	if(addModule && modifyTHEADR) {	/* modify the module's name? */
		modName = dupstr(libNameOld);
		fnameupr(modName);
		i = h->symLen;			/* original length of the module name */
		if((h = U_realloc(h, sizeof(struct tmpList) + (h->symLen = strlen(modName)))) == NULL)
			fatal(E_noMem);
		memcpy(h->symbol, modName, h->symLen + 1);
	/* modLength contains the old module name length til now */
		longaddi(modLength, h->symLen - i);
	}
	else modName = dupmem(h->symLen, h->symbol);

	modi = 0;
	if((cmd = inList(CMD_ALL, modName, 2)) != NULL)	/* do we know an action for this module? */
		p = cmd->fl_name;

/* check out, if to extract this module */
	if(cmd && (cmd->fl_flags & CMD_EXT)) {		/* extract this module */
		if(xtractFile) {		/* special extraction mode */
			U_free(modName);
			sprintf(modName = getmem(14), "%u.OBJ", ++xtractCnt);
			message(xtractFile, M_xtrctOld);
			fwrite(h->symbol, h->symLen, 1, xtractFile);
			message(xtractFile, M_xtrctNew, modName);
		}
		openExtFile(modName, h, p);
		cmd->fl_flags |= CMD_EXTED;
		if(!wildcarded(p))		/* extract only the first module */
			cmd->fl_flags &= ~CMD_EXT;
		modi = 1;
	}

/* check out, if to subtract this module */
	if(cmd && (cmd->fl_flags & CMD_SUB)) {		/* subtract this module */
	/* leave skipModule == 1 to indicate skipping */
		cmd->fl_flags |= CMD_SUBED;
		modi |= 2;
		if(wildcarded(p)) {		/* internal globbing replacement */
			if(replInternal && (cmd->fl_flags & CMD_ADD)) {
				if(access(q = replName(p, modName), 4)) {
					warning(W_notFound, q);
					modi &= ~2;					/* do not subtract module */
					cmd->fl_flags &= ~CMD_ADD;	/* do not newly add module */
				}
				else
					appList(CMD_ADD | CMD_THROUGH, q, 2);	/* file names are to upper-case */
				U_free(q);
			}
		}
		else	/* subtract only the first module */
			cmd->fl_flags &= ~CMD_SUB;
	}

	if((modi & 2) == 0) {
	/* leave this module in the library */
		skipModule = NUL;
		dumpTHEADR(h, libOut);	/* this was disable prior to allow subtracting */

		if(dblMod && cmd && (cmd->fl_flags & CMD_ADD)) {
		/* Then this module is about to be added twice */
			warning(W_modAlreadyIn, modName);
			returnNumber = msgErrorNumber(E_nrDblMod);
			cmd->fl_flags |= CMD_ADDED | CMD_ERROR;		/* do not add the same module name twice */
			if(!wildcarded(p))
				cmd->fl_flags &= ~CMD_ADD;
		}
	}

	Fupdate();			/* save the flag modifications */

/* dupe name into temporary symbol's list*/
	Swrite(h);
/* prepare name for later emitting to screen */
	for(i = h->symLen; i--;)
		 if(h->symbol[i] < ' ')	h->symbol[i] = ' ';	/* pad */
	U_free(modName);
	modName = dupstr(h->symbol);
	U_free(h);
	if(modi && dispDoing)
		informative(modi == 2? M_modSubtract:
			modi == 1? M_modExtract: M_modSubExt, modName);
	chkHeap()
	DB_EXIT
}
#if 0			/* not necessary for the library directory */
void LcopyLNAME(void)
/* LNAME List of Names */
{	struct tmpList *h;	/* name buffer */
	word len;

	writeHeader();
	len = modLen;
	while(len--) {	/* skip name's length byte */
	/* read name length byte */
		h = getName();
		if(!h->symLen || h->symLen > len)
			error(E_corruptMod, NR_LIB_LNAME, modName);
		len -= hlen;
	/* dupe name into temporary list */
		Swrite(h);
		U_free(h);
	}
	chkCRC();
}
#endif
void LcopyPUBDEF(void)
/* public definition

          Ŀ
             RecordGroupSegmentFRAME PublicPublicType Check
          B6HLengthINDEX INDEX NUMBER NAME OffsetINDEX Sum 
          Ĵ
           1   2   INDEX INDEX  0-2   NAME   2   INDEX  1  
          
                                          repeated
*/
{	byte idx1, idx2;
	int dummy;
	struct tmpList *h;
	word readLen, len;

	writeHeader();
	len = modLen;
	readLen = 0;
/* reading INDEXs */
	rwIndex(idx1);
	rwIndex(idx2);
	if(!idx1 && !idx2) {	/* Frame Number comes */
		Lrw(&dummy, 2);
		readLen += 2;
	}
	do {	/* read the "repeated"; first readLen comes from leading items */
		if(len < readLen)
			error(E_corruptMod, NR_LIB_PUBDEF, modName);
		if(!(len -= readLen))
			break;
		readLen = (h = getName())->symLen + 2 + 1;
					/* & len of pub off & name's length byte */
					/* no overflow: because (word)readLen, but (byte)symLen */
		Swrite(h);
		U_free(h);
		Lrw(&dummy, 2);		/* read public offset */
		rwIndex(idx1);
	} while(1);
	chkCRC();
}
void LcopyLIBHEAD(void)
/* This is a library to copy; This module must be the first! */
{	word pos;

	if(Ltell(&pos, NULL, 1, libOld) || pos != sizeof(module))
		fatal(E_LIBHEAD, libNameOld);
	Rewind(libIn, libNameOld);		/* Linit() needs file offset 0 */
	informative(M_copyLib, libNameOld);
	Linit(aS(libHeaderOld));
	addModule = NUL;
}

void writeLibEnd(void)
{	struct LIB_TRAILER buf;

	LwriteNxtPage(libNew);
	buf.type = NR_LIB_TRAILER;
	buf.length = alignDir? trailerSize(libNew) - sizeof(struct LIB_TRAILER)
		: 0;
	Lwrite(aS(buf), sizeof(struct LIB_TRAILER), libNew);
	LwriteMult(0, buf.length, libNew);	/* pad the page */
}

void Lcopy(void)
/* copy a library of modules */
{
	DB_ENTER("Lcopy");
	addModule = !NUL;		/* presume newly added module */

	if(libIn) {
		do {
			if(!addModule)	/* try to read a module in dir space? */
				chkLibExceed(libOld);

			LreadMod(aS(module), libOld);

			if(!addModule && modType == NR_LIB_TRAILER)	{
				/* This record has no ordinary length field */
				Lseek(module.length, libOld);	/* should now point to dirOffset */
				chkLibDir(libOld);
				break;		/* Library fully copied */
			}

			if(!module.length)
				chkEmptyMod()
			else switch(modType) {
				case NR_LIB_HEADER: LcopyLIBHEAD(); break;
				case NR_LIB_THEADR:
					if(addModule && dispDoing)
						informative(M_modAdd, libNameOld);
					LcopyTHEADR();
					break;
				case NR_LIB_MODEND:
					LcopyMODEND();
					if(addModule)	/* in OBJ files this is the very last */
						DB_EXIT
					break;
				case NR_LIB_PUBDEF: LcopyPUBDEF(); break;
				/* case NR_LIB_LNAME: LcopyLNAME(); break; */
				default: LcopyDEFAULT(); break;
			}
		} while(1);
	}

	DB_EXIT
}


void Ldir(struct LIB_INFO *lib)
{	FLAG noFirst;

	DB_ENTER("Ldir");
	DB_PRINT("arg", ("name = %s", lib->_name));

 	noFirst = NUL;

	informative(M_createDir);
#ifdef _MICROC_
	if(ltell(libFp, &dirOffset.hi, &dirOffset.lo))
#else
	if((dirOffset = ftell(libFp)) == (dword)~0l)
#endif
		fatal(E_accessFile, libName);

/* write the dir as often, until they fit into dirPages pages */
	dirPages = minPages - 1;
#ifndef NDEBUG
	informative("Min dir pages: %u", dirPages);
#endif
	while((dirPages = nextPrime(dirPages + 1)) != 0) {
		if(noFirst)
			informative(M_recreateDir, dirPages);
		else noFirst = !NUL;
		if(createDir(lib)) DB_EXIT
	}
	error(E_noPages);
}

int LaddMod(struct cmdList *modp)
/* add the module mod to libNew */
{	char *mod;

	DB_ENTER("LaddMod");

	mod = modp->fl_name;

	if((modp->fl_flags & (CMD_ERROR | CMD_TWICE | CMD_ADDED | CMD_ADD))
	  != CMD_ADD				/* this entry contains no module to add */
	 || replInternal && wildcarded(mod)		/* was globbed internally */
	) DB_RETURN(0);

	modp->fl_flags |= CMD_ADDED;
	modp->fl_flags &= ~CMD_ADD;

	DB_PRINT("arg", ("name = %s", mod));

	chkHeap()
	if(libIn) {			/* replace old module/library */
		Close(libIn);
		libIn = NULL;
		U_free(libNameOld);
		libNameOld = NULL;
	}
	chkHeap()
#ifdef _MICROC_
	if((libIn = open(mod, F_READ | F_BINARY)) == NULL) {
#else
	if((libIn = fopen(mod, "rb")) == NULL) {
#endif
		warning(E_openFile, mod);
		returnNumber = msgErrorNumber(E_nrAddMod);
	}
	else {
		libNameOld = dupstr(mod);
		Lcopy();				/* copy the currently opened OBJ/LIB */
	}
	DB_RETURN(0);
}
