/***************************************************************************/
/* XCOPY                                                                   */
/*                                                                         */
/* A replacement for the MS DOS(tm) XCOPY command.                         */
/*-------------------------------------------------------------------------*/
/* compatible compilers:                                                   */
/* Borland C++ (tm) v3.0 or higher                                         */
/*-------------------------------------------------------------------------*/
/* (C)opyright 2001 by Rene Ableidinger (rene.ableidinger@gmx.at)          */
/*                                                                         */
/* This program is free software; you can redistribute it and/or modify it */
/* under the terms of the GNU General Public License version 2 as          */
/* published by the Free Software Foundation.                              */
/*                                                                         */
/* This program is distributed in the hope that it will be useful, but     */
/* WITHOUT ANY WARRANTY; without even the implied warranty of              */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU        */
/* General Public License for more details.                                */
/*                                                                         */
/* You should have received a copy of the GNU General Public License along */
/* with this program; if not, write to the Free Software Foundation, Inc., */
/* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                */
/***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <dos.h>
#include <sys/stat.h>
#include "shared.inc"

/*-------------------------------------------------------------------------*/
/* GLOBAL CONSTANTS                                                        */
/*-------------------------------------------------------------------------*/
#define MAXSWITCH 255

/*-------------------------------------------------------------------------*/
/* GLOBAL VARIABLES                                                        */
/*-------------------------------------------------------------------------*/
char switch_archive = 0,
     switch_confirm = 2,  /* ask for confirmation to overwrite file if */
                          /* it exists:                                */
                          /* 0 - don't ask and overwrite file          */
                          /* 1 - don't ask and skip file               */
                          /* 2 - ask                                   */
     switch_dir = 0,
     switch_emptydir = 0,
     switch_file = 0,
     switch_hidden = 0,
     switch_archive_reset = 0,
     switch_prompt = 0,
     switch_quiet = 0,
     switch_readonly = 0,
     switch_subdir = 0,
     switch_verify = 0,
     switch_wait = 0,
     bak_verify = 0,
     dest_drive;
int file_counter = 0,
    file_found = 0;

/*-------------------------------------------------------------------------*/
/* PROTOTYPES                                                              */
/*-------------------------------------------------------------------------*/
void help(void);
void exit_fn(void);
int cyclic_path(const char *src_pathname,
                const char *dest_pathname);
int make_dir(const char *path);
void build_filename(char *dest_filename,
                    const char *src_filename,
                    const char *filemask);
void build_name(char *dest,
                const char *src,
                const char *mask);
void xcopy_files(const char *src_pathname,
                 const char *src_filename,
                 const char *dest_pathname,
                 const char *dest_filename);
void xcopy_file(const char *src_filename,
                const char *dest_filename);

/*-------------------------------------------------------------------------*/
/* MAIN-PROGRAM                                                            */
/*-------------------------------------------------------------------------*/
int main(int argc, char **argv) {
  char fileargc,
       *fileargv[255],
       switchargc,
       *switchargv[255],
       env_prompt[MAXSWITCH],
       tmp_switch[MAXSWITCH] = "",
       src_pathname[MAXPATH] = "",
       src_filename[MAXFILE + MAXEXT] = "",
       dest_pathname[MAXPATH] = "",
       dest_filename[MAXFILE + MAXEXT] = "",
       *ptr,
       i,
       ch;


  classify_args(argc, argv, &fileargc, fileargv, &switchargc, switchargv);

  if (fileargc < 1 || switchargv[0] == "?") {
    help();
    exit(4);
  }

  /* activate termination function */
  /* (writes no. of copied files at exit) */
  atexit(exit_fn);

  /* read COPYCMD environment to set confirmation switch */
  strmcpy(env_prompt, getenv("COPYCMD"), sizeof(env_prompt));
  if (env_prompt[0] != '\0') {
    strupr(env_prompt);
    if (strcmp(env_prompt, "/Y") == 0)
      /* overwrite existing file(s) */
      switch_confirm = 0;
    else if (strcmp(env_prompt, "/N") == 0)
      /* skip existing file(s) */
      switch_confirm = 1;
    else
      /* ask for confirmation */
      switch_confirm = 2;
  }

  /* get switches */
  for (i = 0; i < switchargc; i++) {
    strmcpy(tmp_switch, switchargv[i], sizeof(tmp_switch));
    strupr(tmp_switch);
    if (strcmp(tmp_switch, "A") == 0)
      switch_archive = -1;
    else if (strcmp(tmp_switch, "D") == 0)
      switch_dir = -1;
    else if (strcmp(tmp_switch, "E") == 0)
      switch_emptydir = -1;
    else if (strcmp(tmp_switch, "F") == 0)
      switch_file = -1;
    else if (strcmp(tmp_switch, "H") == 0)
      switch_hidden = -1;
    else if (strcmp(tmp_switch, "M") == 0)
      switch_archive_reset = -1;
    else if (strcmp(tmp_switch, "N") == 0)
      switch_confirm = 1;
    else if (strcmp(tmp_switch, "P") == 0)
      switch_prompt = -1;
    else if (strcmp(tmp_switch, "Q") == 0)
      switch_quiet = -1;
    else if (strcmp(tmp_switch, "R") == 0)
      switch_readonly = -1;
    else if (strcmp(tmp_switch, "S") == 0)
      switch_subdir = -1;
    else if (strcmp(tmp_switch, "V") == 0) {
      switch_verify = -1;
      bak_verify = getverify();
      setverify(1);
    }
    else if (strcmp(tmp_switch, "W") == 0)
      switch_wait = -1;
    else if (strcmp(tmp_switch, "Y") == 0)
      switch_confirm = 0;
    else if (strcmp(tmp_switch, "-Y") == 0)
      switch_confirm = 2;
    else {
      printf("Invalid switch - %s\n", switchargv[i]);
      exit(4);
    }
  }
  if (switch_file && switch_dir) {
    printf("Invalid combination of switches\n");
    exit(4);
  }

  /* get source pathname (with trailing backspace) and filename */
  _fullpath(src_pathname, fileargv[0], MAXPATH);
  if (src_pathname[0] == '\0') {
    printf("Invalid source drive specification\n");
    exit(4);
  }
  /* check source path */
  if (! dir_exists(src_pathname)) {
    /* source path contains a filename/-mask -> separate it */
    ptr = strrchr(src_pathname, (char)* DIR_SEPARATOR);
    strmcpy(src_filename, ptr + 1, sizeof(src_filename));
    *(ptr + 1) = '\0';
    /* check source path */
    if (! dir_exists(src_pathname)) {
      printf("Source path not found - %s\n", src_pathname);
      exit(4);
    }
  }
  else {
    /* source path is a directory -> filename = "*.*" */
    strmcpy(src_filename, "*.*", sizeof(src_filename));
  }
  cat_separator(src_pathname);

  /* get destination pathname (with trailing backspace) and filename/-mask */
  if (fileargc < 2) {
    /* no destination path specified -> use current */
    getcwd(dest_pathname, MAXPATH);
    strmcpy(dest_filename, "*.*", sizeof(dest_filename));
  }
  else {
    /* destination path specified */
    _fullpath(dest_pathname, fileargv[1], MAXPATH);
    if (dest_pathname[0] == '\0') {
      printf("Invalid destination drive specification\n");
      exit(4);
    }
    /* check destination path */
    if (! dir_exists(dest_pathname)) {
      ptr = strrchr(dest_pathname, (char)* DIR_SEPARATOR);
      strmcpy(dest_filename, ptr + 1, sizeof(dest_filename));
      *(ptr + 1) = '\0';
      if (! dir_exists(dest_pathname)) {
        printf("Destination path not found - %s\n", dest_pathname);
        exit(4);
      }

      if ((ptr = strchr(dest_filename, '*')) == NULL &&
          (ptr = strchr(dest_filename, '?')) == NULL) {
        /* last destination entry doesn't include wildcards -> look */
        /* for switches /F or /D */
        if (! switch_dir &&
            ! switch_file) {
          /* no switch set -> ask if destination should be a file or */
          /* a directory */
          printf("Does %s specify a file name\n", dest_filename);
          ch = confirm("or directory name on the target", "File", "Directory", NULL, NULL);
          switch (ch) {
            case 'D':
              /* directory */
              switch_dir = -1;
              break;
            case 'F':
              /* file */
              switch_file = -1;
              break;
          }
        }
        if (switch_dir) {
          /* destination is a directory */
          strmcat(dest_pathname, dest_filename, sizeof(dest_pathname));
          strmcpy(dest_filename, "*.*", sizeof(dest_filename));
        }
      }
    }
    else {
      /* source path is a directory -> filemask = "*.*" */
      strmcpy(dest_filename, "*.*", sizeof(dest_filename));
    }
  }
  cat_separator(dest_pathname);

  /* check for cyclic path */
  if ((switch_emptydir || switch_subdir) &&
      cyclic_path(src_pathname, dest_pathname)) {
    printf("Cannot perform a cyclic copy\n");
    exit(4);
  }

  /* get destination drive (1 = A, 2 = B, 3 = C, ...) */
  dest_drive = toupper(dest_pathname[0]) - 64;

  if (switch_wait) {
    printf("Press any key to continue...\n");
    getch();
    fflush(stdin);
  }

  xcopy_files(src_pathname, src_filename, dest_pathname, dest_filename);
  if (! file_found) {
    printf("File not found - %s\n", src_filename);
    exit(1);
  }

  return 0;
}

/*-------------------------------------------------------------------------*/
/* SUB-PROGRAMS                                                            */
/*-------------------------------------------------------------------------*/
void help(void) {
  printf("XCOPY v1.0 - (C)opyright 2001 by Rene Ableidinger\n"
         "Copies files and directory trees.\n"
         "\n"
         "XCOPY [drive1:][path1]filename [drive2:][path2][filename2] [/switches]\n"
         "\n"
         "  [drive1:][path1]filename     Specifies directory and name of file(s) to copy.\n"
         "  [drive2:][path2][filename2]  Specifies directory and/or name of new file(s).\n"
         "  /A        Copies only files with the archive attribute set and\n"
         "            doesn't change the attribute.\n"
         "  /D        The destination specifies a directory.\n"
         "            Cannot be used with the /F switch.\n"
         "  /E        Copies any subdirectories, even if empty.\n"
         "  /F        The destination specifies a file.\n"
         "            Cannot be used with the /D switch.\n"
         "  /H        Copies hidden and system files as well as unprotected files.\n"
         "  /M        Copies only files with the archive attribute set and\n"
         "            turns off the archive attribute.\n"
         "  /N        Suppresses prompting to confirm you want to overwrite an\n"
         "            existing destination file and skips these files.\n"
         "  /P        Prompts for confirmation before creating each destination file.\n"
         "  /Q        Quiet mode, don't show copied filenames.\n"
         "  /R        Overwrite read-only files as well as unprotected files.\n"
         "  /S        Copies directories and subdirectories except empty ones.\n"
         "  /V        Verifies each new file.\n"
         "  /W        Waits for a keypress before beginning.\n"
         "  /Y        Suppresses prompting to confirm you want to overwrite an\n"
         "            existing destination file and overwrites these files.\n"
         "  /-Y       Causes prompting to confirm you want to overwrite an\n"
         "            existing destination file.\n"
         "\n"
         "The switch /Y or /N may be preset in the COPYCMD environment variable.\n"
         "This may be overridden with /-Y on the command line.\n");
}

void exit_fn(void) {
/*-------------------------------------------------------------------------*/
/* Gets called by the "exit"-command and is used to write a                */
/* status-message.                                                         */
/*-------------------------------------------------------------------------*/
  if (switch_verify) {
    /* restore value of verify flag */
    setverify(bak_verify);
  }

  printf("%d file(s) copied\n", file_counter);
}

int cyclic_path(const char *src_pathname,
                const char *dest_pathname) {
/*-------------------------------------------------------------------------*/
/* Checks, if the destination path is a subdirectory of the source path.   */
/*-------------------------------------------------------------------------*/
  char tmp_dest_pathname[MAXPATH];
  int length;


  /* format pathnames for comparison */
  length = strlen(src_pathname);
  strncpy(tmp_dest_pathname, dest_pathname, length);
  tmp_dest_pathname[length] = '\0';

  return stricmp(src_pathname, tmp_dest_pathname) == 0;
}

int make_dir(const char *path) {
/*-------------------------------------------------------------------------*/
/* Creates a directory or a whole directory path. The pathname may contain */
/* a trailing directory separator.                                         */
/*-------------------------------------------------------------------------*/
  char tmp_path1[MAXPATH],
       tmp_path2[MAXPATH],
       i,
       length,
       mkdir_error;


  if (path[0] == '\0') {
    return -1;
  }

  strmcpy(tmp_path1, path, sizeof(tmp_path1));
  cat_separator(tmp_path1);
  length = strlen(tmp_path1);
  strncpy(tmp_path2, tmp_path1, 3);
  i = 3;
  while (i < length) {
    if (tmp_path1[i] == '\\') {
      tmp_path2[i] = '\0';
      if (! dir_exists(tmp_path2)) {
        mkdir_error = mkdir(tmp_path2);
        if (mkdir_error) {
          return mkdir_error;
        }
      }
    }
    tmp_path2[i] = tmp_path1[i];

    i = i + 1;
  }

  return 0;
}

void build_filename(char *dest_filename,
                    const char *src_filename,
                    const char *filemask) {
/*-------------------------------------------------------------------------*/
/* Uses the source filename and the filemask (which may contain wildcards  */
/* '?' and '*' in any possible combination) to create a new filename. The  */
/* source filename may contain a pathname.                                 */
/*-------------------------------------------------------------------------*/
  char drive[MAXDRIVE],
       dir[MAXDIR],
       filename_file[MAXFILE],
       filename_ext[MAXEXT],
       filemask_file[MAXFILE],
       filemask_ext[MAXEXT],
       tmp_filename[MAXFILE],
       tmp_ext[MAXEXT];


  _splitpath(src_filename, drive, dir, filename_file, filename_ext);
  _splitpath(filemask, drive, dir, filemask_file, filemask_ext);
  build_name(tmp_filename, filename_file, filemask_file);
  build_name(tmp_ext, filename_ext, filemask_ext);
  strmcpy(dest_filename, drive, MAXPATH);
  strmcat(dest_filename, dir, MAXPATH);
  strmcat(dest_filename, tmp_filename, MAXPATH);
  strmcat(dest_filename, tmp_ext, MAXPATH);
}

void build_name(char *dest,
                const char *src,
                const char *mask) {
  int i,
      mask_i,
      src_length,
      mask_length;


  src_length = strlen(src);
  mask_length = strlen(mask);
  i = 0;
  mask_i = 0;
  while ((i < src_length ||
          (mask[mask_i] != '\0' &&
           mask[mask_i] != '?' &&
           mask[mask_i] != '*')) &&
         mask_i < mask_length) {
    switch (mask[mask_i]) {
      case '*':
        dest[i] = src[i];
        break;
      case '?':
        dest[i] = src[i];
        mask_i = mask_i + 1;
        break;
      default:
        dest[i] = mask[mask_i];
        mask_i = mask_i + 1;
        break;
    }

    i = i + 1;
  }
  dest[i] = '\0';
}

void xcopy_files(const char *src_pathname,
                 const char *src_filename,
                 const char *dest_pathname,
                 const char *dest_filename) {
/*-------------------------------------------------------------------------*/
/* Searchs through the source directory (and its subdirectories) and calls */
/* function "xcopy_file" for every found file.                             */
/*-------------------------------------------------------------------------*/
  char filemask[MAXPATH],
       new_src_pathname[MAXPATH],
       new_dest_pathname[MAXPATH],
       src_path_filename[MAXPATH],
       dest_path_filename[MAXPATH],
       tmp_filename[MAXFILE + MAXEXT];
  struct ffblk fileblock;
  int fileattrib,
      done;


  if (switch_emptydir ||
      switch_subdir) {
    /* copy files in subdirectories too */
    strmcpy(filemask, src_pathname, sizeof(filemask));
    strmcat(filemask, "*.*", sizeof(filemask));
    done = findfirst(filemask, &fileblock, FA_DIREC);
    while (! done) {
      if (fileblock.ff_attrib == FA_DIREC &&
          strcmp(fileblock.ff_name, ".") != 0 &&
          strcmp(fileblock.ff_name, "..") != 0) {
        /* build source pathname */
        strmcpy(new_src_pathname, src_pathname, sizeof(new_src_pathname));
        strmcat(new_src_pathname, fileblock.ff_name, sizeof(new_src_pathname));
        strmcat(new_src_pathname, DIR_SEPARATOR, sizeof(new_src_pathname));

        /* build destination pathname */
        strmcpy(new_dest_pathname, dest_pathname, sizeof(new_dest_pathname));
        strmcat(new_dest_pathname, fileblock.ff_name, sizeof(new_dest_pathname));
        strmcat(new_dest_pathname, DIR_SEPARATOR, sizeof(new_dest_pathname));

        xcopy_files(new_src_pathname, src_filename,
                    new_dest_pathname, dest_filename);
      }

      done = findnext(&fileblock);
    }
  }

  fileattrib = FA_RDONLY + FA_ARCH;
  if (switch_hidden) {
    /* replace hidden and system files too */
    fileattrib = fileattrib + FA_HIDDEN + FA_SYSTEM;
  }

  /* find first source file */
  strmcpy(filemask, src_pathname, sizeof(filemask));
  strmcat(filemask, src_filename, sizeof(filemask));
  done = findfirst(filemask, &fileblock, fileattrib);

  if (! done) {
    file_found = -1;
  }

  /* check if destination directory must be created */
  if ((! done || switch_emptydir) &&
      ! dir_exists(dest_pathname)) {
    if (make_dir(dest_pathname) != 0) {
      printf("Unable to create directory %s\n", dest_pathname);
      exit(4);
    }
  }

  /* copy files */
  while (! done) {
    /* check, if copied files should have archive attribute set */
    if ((switch_archive == 0 && switch_archive_reset == 0) ||
        fileblock.ff_attrib & FA_ARCH) {
      /* build source filename including path */
      strmcpy(src_path_filename, src_pathname, sizeof(src_path_filename));
      strmcat(src_path_filename, fileblock.ff_name, sizeof(src_path_filename));

      /* build destination filename including path */
      strmcpy(dest_path_filename, dest_pathname, sizeof(dest_path_filename));
      build_filename(tmp_filename, fileblock.ff_name, dest_filename);
      strmcat(dest_path_filename, tmp_filename, sizeof(dest_path_filename));

      xcopy_file(src_path_filename, dest_path_filename);
    }

    done = findnext(&fileblock);
  }
}

void xcopy_file(const char *src_filename,
                const char *dest_filename) {
/*-------------------------------------------------------------------------*/
/* Checks all dependencies of the source and destination file and calls    */
/* function "copy_file".                                                   */
/*-------------------------------------------------------------------------*/
  int dest_file_exists,
      fileattrib;
  struct stat src_statbuf,
              dest_statbuf;
  struct dfree disktable;
  unsigned long free_diskspace;
  char ch,
       msg_prompt[255];


  if (switch_prompt) {
    /* ask for confirmation to create file */
    ch = confirm(dest_filename, "Yes", "No", NULL, NULL);
    if (ch == 'N') {
      /* no -> skip file */
      return;
    }
  }

  /* check if source and destination file are equal */
  if (stricmp(src_filename, dest_filename) == 0) {
    printf("File cannot be copied onto itself - %s\n", src_filename);
    exit(4);
  }

  /* check source file for read permission */
  /* (only usefull under an OS with the ability to deny read access) */
  if (access(src_filename, R_OK) != 0) {
    printf("Read access denied - %s\n", src_filename);
    exit(5);
  }

  /* get info of source and destination file */
  stat(src_filename, &src_statbuf);
  dest_file_exists = ! stat(dest_filename, &dest_statbuf);

  /* get amount of free disk space in destination drive */
  getdfree(dest_drive, &disktable);
  free_diskspace = (unsigned long) disktable.df_avail *
                   disktable.df_sclus * disktable.df_bsec;

  if (dest_file_exists) {
    switch (switch_confirm) {
      case 1:
        /* skip file */
        return;
      case 2:
        /* ask for confirmation to overwrite file */
        strmcpy(msg_prompt, "Overwrite ", sizeof(msg_prompt));
        strmcat(msg_prompt, dest_filename, sizeof(msg_prompt));
        ch = confirm(msg_prompt, "Yes", "No", "Overwrite all", "Skip all");
        switch (ch) {
          case 'N':
            /* no -> skip file */
            return;
          case 'O':
            /* overwrite all -> set confirm switch */
            switch_confirm = 0;
            break;
          case 'S':
            /* skip all -> set confirm switch */
            switch_confirm = 1;
            return;
        }
        break;
    }

    /* check free space on destination disk */
    if (src_statbuf.st_size > free_diskspace - dest_statbuf.st_size) {
      printf("Insufficient disk space in destination path - %s\n", dest_filename);
      exit(39);
    }

    /* get file attribute from destination file */
    fileattrib = _chmod(dest_filename, 0);

    /* check destination file for write permission */
    if (fileattrib & (64 ^ FA_RDONLY) != 0) {
      if (! switch_readonly) {
        printf("Write access denied - %s\n", dest_filename);
        exit(5);
      }

      /* remove readonly attribute from destination file */
      fileattrib = fileattrib ^ FA_RDONLY;
      _chmod(dest_filename, 1, fileattrib);
    }
  }
  else {
    /* check free space on destination disk */
    if (src_statbuf.st_size > free_diskspace) {
      printf("Insufficient disk space in destination path - %s\n", dest_filename);
      exit(39);
    }
  }

  if (! switch_quiet) {
    printf("Copying %s\n", dest_filename);
  }
  copy_file(src_filename, dest_filename);

  if (switch_archive_reset) {
    /* remove archive attribute from source file */
    fileattrib = _chmod(src_filename, 0);
    _chmod(src_filename, 1, fileattrib ^ FA_ARCH);
  }

  file_counter = file_counter + 1;
}