/* GSnes9x -- callbacks.c: All the callbacks (except the prefs ones).
 * Copyright (C) 1999 Canek Pelez Valds <canek@abulafia.fciencias.unam.mx>
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gsnes9x.h"

/* It's not my fault: tell to ESD. */
#ifndef NO_ESD_SUPPORT
extern int gnome_sound_connection;
#endif

void
play_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	ROMInfo  *rom_info;
	GList    *list = g_list_first (gsnes_data->roms_list);
	gchar   **process = NULL;
	gchar    *rom_name;
	pid_t     pid;
	gint      status;
	gint      cont = 0;
#ifndef NO_ESD_SUPPORT
	gint      connection = -1;
#endif
	
	F_NAME();
	
	if (gsnes_data->current_rom == -1)
		return;
	
	gtk_clist_get_text (GTK_CLIST (gsnes_data->clist),
			    gsnes_data->current_rom, 0, &rom_name);

	while (strcmp (rom_name,
		       ((ROMInfo*)(list->data))->string_prefs[NAME]))
		list = g_list_next (list);

	rom_info = (ROMInfo*)(list->data);

	process = get_command_line (rom_info, gsnes_data->prefs);

	while (process[cont])
		printf ("%s ", process[cont++]);
	printf ("\n");
	cont = 0;
	
#ifndef NO_ESD_SUPPORT
	if (gsnes_data->prefs->bool_prefs[WITH_ESD-ROM_BOOL_PREFS]) {
		connection = esd_standby (gnome_sound_connection);
		if (connection == -1)
			g_warning ("A error has ocurred when tried "
				   "to put Esound in standby,");
	}
#endif
	
	switch((pid = fork())) {
	case -1:
#ifndef NO_ESD_SUPPORT
		if (gsnes_data->prefs->bool_prefs[WITH_ESD-ROM_BOOL_PREFS]) {
			connection = esd_resume (gnome_sound_connection);
			if (connection == -1)
				g_warning ("A error has ocurred when tried "
					   "to resume Esound.");
		}
#endif
		g_error (_("I can't fork."));
		break;
	case 0:
		execvp (process[0], process);
		_exit (255);
	default:
		gtk_widget_hide (gsnes_data->app);
		while (gtk_events_pending())
			gtk_main_iteration();
		waitpid (pid, &status, 0);
		gtk_widget_show (gsnes_data->app);
#ifndef NO_ESD_SUPPORT
		if (gsnes_data->prefs->bool_prefs[WITH_ESD-ROM_BOOL_PREFS]) {
			connection = esd_resume (gnome_sound_connection);
			if (connection == -1)
				g_warning ("A error has ocurred when tried "
					   "to resume Esound.");
		}
#endif
		
		if (!WIFEXITED (status))
			g_warning (_("Snes9X terminated abnormally."));
		else if (WEXITSTATUS (status) == 255)
			g_warning (_("I can't exec %s."), process[0]);
		else if ((status) && (status != 768))
			g_warning (_("%s returned error code %d."),
				   process[0], WEXITSTATUS(status));
	}

	g_strfreev (process);
}

void
add_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	AddROMInfo  *addrom_info = g_new (AddROMInfo, 1);
	GtkWidget   *add_dialog;
	GtkWidget   *table;
	GtkWidget   *label;
	GtkWidget   *text;
	GtkWidget   *frame;
	GlobalPrefs *prefs = g_new (GlobalPrefs, 1);
	gchar       *path;


	F_NAME();

	prefs = read_conf_file ();	
	path = prefs->string_prefs[ROMS_DIR-ROM_STR_PREFS];
	g_free (prefs);
	if (!path)
		path = get_default_roms_dir();
	
	addrom_info->data       = gsnes_data;
	gsnes_data->dialog_open = TRUE;
	
	add_dialog = gnome_dialog_new (_("Add new ROM"),
				       GNOME_STOCK_BUTTON_OK,
				       GNOME_STOCK_BUTTON_CANCEL,
				       NULL);

	gtk_window_set_modal (GTK_WINDOW (add_dialog), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (add_dialog),
				      GTK_WINDOW (gsnes_data->app));
	
	addrom_info->dialog = add_dialog;
	
	gnome_dialog_button_connect (GNOME_DIALOG (add_dialog),
				     0, GTK_SIGNAL_FUNC (add_ok_cb),
				     (gpointer) addrom_info);

	gnome_dialog_button_connect (GNOME_DIALOG (add_dialog),
				     1, GTK_SIGNAL_FUNC (add_cancel_cb),
				     (gpointer) addrom_info);

	gtk_signal_connect (GTK_OBJECT (add_dialog), "destroy",
			    GTK_SIGNAL_FUNC (no_dialog_cb),
			    (gpointer) gsnes_data);
	gtk_signal_connect (GTK_OBJECT (add_dialog), "destroy",
			    GTK_SIGNAL_FUNC (add_free_cb),
			    (gpointer) addrom_info);
	
	frame = gtk_frame_new (_("New ROM"));
	
	table = gtk_table_new (1, 2, FALSE);
	
	label  = gtk_label_new (_("File:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);
	
	text   = gnome_file_entry_new(NULL, _("Select ROM file"));
	gnome_file_entry_set_default_path (GNOME_FILE_ENTRY(text), path);
	gtk_table_attach (GTK_TABLE (table), text, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	addrom_info->entry[ROM_FILE] = gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(text));
		
	gtk_widget_show (table);
	gtk_container_add (GTK_CONTAINER (frame), GTK_WIDGET (table));
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (add_dialog)->vbox),
			    frame, TRUE, TRUE, 0);
	
	frame = gtk_frame_new (_("Optional Information"));

	table = gtk_table_new (3, 2, FALSE);
	
	label  = gtk_label_new (_("Name:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);
	
	text   = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), text, 1, 2, 0, 1,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	addrom_info->entry[NAME] = text;

	label  = gtk_label_new (_("Company:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);
	
	text   = gtk_entry_new ();
	gtk_table_attach (GTK_TABLE (table), text, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	addrom_info->entry[COMPANY] = text;

	label  = gtk_label_new (_("Image File:"));
	gtk_misc_set_alignment (GTK_MISC (label), 1.0, 0.5);
	gtk_table_attach (GTK_TABLE (table), label, 0, 1, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (label);
	
	text   = gnome_file_entry_new(NULL, _("Select Image file"));
	gnome_file_entry_set_default_path (GNOME_FILE_ENTRY(text), path);
	gtk_table_attach (GTK_TABLE (table), text, 1, 2, 2, 3,
			  GTK_FILL, GTK_FILL, 5, 2);
	gtk_widget_show (text);
	addrom_info->entry[IMAGE_FILE] =  gnome_file_entry_gtk_entry(
			GNOME_FILE_ENTRY(text));
	
	gtk_widget_show (table);
	gtk_container_add (GTK_CONTAINER (frame), GTK_WIDGET (table));
	gtk_widget_show (frame);
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (add_dialog)->vbox),
			    frame, TRUE, TRUE, 0);
		
	gtk_widget_show (add_dialog);
}

void
remove_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	GtkWidget *confirm_dialog;

	F_NAME();
	
	if (gsnes_data->current_rom == -1)
		return;

	gsnes_data->dialog_open = TRUE;
	
	confirm_dialog = gnome_message_box_new(_("Are you sure you want\n"
						 "to remove the ROM?"),
					       GNOME_MESSAGE_BOX_WARNING,
					       GNOME_STOCK_BUTTON_YES,
					       GNOME_STOCK_BUTTON_NO,
					       NULL);

	gtk_window_set_modal (GTK_WINDOW (confirm_dialog), TRUE);
	gtk_window_set_transient_for (GTK_WINDOW (confirm_dialog),
				      GTK_WINDOW (gsnes_data->app));

	gnome_dialog_button_connect (GNOME_DIALOG (confirm_dialog), 0,
				     GTK_SIGNAL_FUNC (remove_ok_cb),
				     (gpointer) gsnes_data);
	gnome_dialog_button_connect (GNOME_DIALOG (confirm_dialog), 0,
				     GTK_SIGNAL_FUNC (destroy_data_cb),
				     (gpointer) confirm_dialog);

	gnome_dialog_button_connect (GNOME_DIALOG (confirm_dialog), 1,
				     GTK_SIGNAL_FUNC (destroy_data_cb),
				     (gpointer) confirm_dialog);

	gtk_signal_connect (GTK_OBJECT (confirm_dialog), "destroy",
			    GTK_SIGNAL_FUNC (no_dialog_cb),
			    (gpointer) gsnes_data);
	
	gtk_widget_show (confirm_dialog);
}

gint
quit_by_close_cb (GtkWidget *app)
{
	GSnes9xData *gsnes_data =
		(GSnes9xData*) gtk_object_get_data (GTK_OBJECT (app),
						    "gsnes_data");

	F_NAME();

	if (gsnes_data->dialog_open)
		return TRUE;
	else
		gtk_main_quit ();

	return FALSE;
}

void
quit_by_menu_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	F_NAME();

	if (gsnes_data->dialog_open)
		return;

	gtk_widget_destroy (gsnes_data->app);
	gtk_main_quit ();
}

void
readme_cb (GtkWidget *widget, gint part)
{
	GtkWidget *readme = NULL;
	GtkWidget *text;
	GtkWidget *hbox;
	GtkWidget *vscrollbar;
	
#include "doc.h" /* Nasty, nasty. */

	F_NAME();
	
	hbox       = gtk_hbox_new (FALSE, 0);
	text       = gtk_text_new (NULL, NULL);	
	vscrollbar = gtk_vscrollbar_new (GTK_TEXT(text)->vadj);
	
	gtk_box_pack_start (GTK_BOX (hbox), text, FALSE, FALSE, 0);
        gtk_box_pack_start (GTK_BOX (hbox), vscrollbar, FALSE, FALSE, 0);

	switch (part) {
	case 0:
		readme = gnome_dialog_new (_("Snes9X Original README.TXT"),
					   GNOME_STOCK_BUTTON_OK, NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readme1, -1);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmeinfo, -1);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmegraph, -1);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmesound, -1);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmecheat, -1);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmespeed, -1);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmeformat, -1);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmejoystick, -1);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readme2, -1);
		break;
	case 1:
		readme = gnome_dialog_new (_("GSnes9x Info Options Help"),
					   GNOME_STOCK_BUTTON_OK, NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmeinfo, -1);
		break;
	case 2:
		readme = gnome_dialog_new (_("GSnes9x Graphics Options Help"),
					   GNOME_STOCK_BUTTON_OK, NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmegraph, -1);
		break;
	case 3:
		readme = gnome_dialog_new (_("GSnes9x Sound Options Help"),
					   GNOME_STOCK_BUTTON_OK, NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmesound, -1);
		break;
	case 4:
		readme = gnome_dialog_new (_("GSnes9x Cheat Codes Options "
					     "Help"), GNOME_STOCK_BUTTON_OK,
					   NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmecheat, -1);
		break;
	case 5:
		readme = gnome_dialog_new (_("GSnes9x Speed Up/Down Options "
					     "Help"), GNOME_STOCK_BUTTON_OK,
					   NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmespeed, -1);
		break;
	case 6:
		readme = gnome_dialog_new (_("GSnes9x ROM Image Format Options"
					   "Help"), GNOME_STOCK_BUTTON_OK,
					   NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmeformat, -1);
		break;
	case 7:
		readme = gnome_dialog_new (_("GSnes9x Joystick Options Help"),
					   GNOME_STOCK_BUTTON_OK, NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmejoystick, -1);
		break;
	case 8:
		readme = gnome_dialog_new (_("GSnes9x System Options Help"),
					   GNOME_STOCK_BUTTON_OK, NULL);
		gtk_text_insert (GTK_TEXT (text), NULL,
				 NULL, NULL, readmesystem, -1);
	}
	
	gtk_text_set_editable (GTK_TEXT (text), FALSE);
	
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (readme)->vbox),
			    hbox, FALSE, FALSE, 10);

	gtk_widget_set_usize (text, 500, 300);

	gnome_dialog_button_connect (GNOME_DIALOG (readme),
				     0, GTK_SIGNAL_FUNC (destroy_data_cb),
				     (gpointer) readme);

	gtk_window_set_modal (GTK_WINDOW (readme), TRUE);
	
	gtk_widget_show (vscrollbar);
	gtk_widget_show (text);
	gtk_widget_show (hbox);
	gtk_widget_show (readme);
}

void
about_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	const gchar *authors[] = { "Canek Pelez Valds", NULL };
	GtkWidget   *about;


	if (gsnes_data->about_open)
		return;

	gsnes_data->about_open = TRUE;

	about = gnome_about_new (PACKAGE, VERSION,
				 "Copyright Canek Pelez Valds 1999",
				 authors,
				 _("A launcher and ROMs and Cheat Code "
				   "Database for the BEST Super Nintendo "
				   "Entertainment System Emulator available"),
				 NULL);

	F_NAME();

	gtk_signal_connect (GTK_OBJECT (about), "destroy",
			    GTK_SIGNAL_FUNC (about_destroyed_cb),
			    (gpointer)gsnes_data);
	
	gtk_widget_show (about);   
}

void
add_ok_cb (GtkWidget *widget, AddROMInfo *addrom_info)
{
	ROMInfo    *rom_info = rom_info_new ();
	GList      *list     = g_list_first (addrom_info->data->roms_list);
	gboolean    repeat   = FALSE;
	gint        i, r = 0, l  = g_list_length (list), len;
	gchar      *name, *company, *rom_file, *img_file, *status;

	F_NAME();
	
	if (!check_rom_info (addrom_info->entry))
       	{
		no_rom_file (addrom_info->dialog);
		return;
	}

	name     = g_strdup
		(gtk_entry_get_text (GTK_ENTRY (addrom_info->entry[NAME])));
	company  = g_strdup
		(gtk_entry_get_text (GTK_ENTRY (addrom_info->entry[COMPANY])));
	rom_file = g_strdup
		(gtk_entry_get_text (GTK_ENTRY (addrom_info->entry[ROM_FILE])));
	img_file = g_strdup
		(gtk_entry_get_text (GTK_ENTRY (addrom_info->entry[IMAGE_FILE])));
		
	len = strlen (name);

	for (i = 0; i < len; i++)
		if (name[i] == '/') {
			g_free (name);
			g_free (company);
			g_free (rom_file);
			g_free (img_file);
			no_correct_name (addrom_info->dialog);
			return;
		}
	
	for (i = 0; i < l; i++) {
		if (!strcmp (((ROMInfo*)(list->data))->string_prefs[NAME],
			     name))
			repeat = TRUE;
		list = g_list_next (list);
	}

	if (repeat) {
		rom_repeated (name, addrom_info->dialog);
		g_free (rom_info);
		g_free (name);
		g_free (company);
		g_free (rom_file);
		g_free (img_file);
		return;
	}
		    
	rom_get_defaults (rom_info, addrom_info->data->prefs);
	
	rom_info->string_prefs[NAME]       = name;
	rom_info->string_prefs[COMPANY]    = company;
	rom_info->string_prefs[ROM_FILE]   = rom_file;
	rom_info->string_prefs[IMAGE_FILE] = img_file;
	
	gtk_clist_append (GTK_CLIST (addrom_info->data->clist),
			  &(rom_info->string_prefs[NAME]));

	addrom_info->data->roms_list =
		g_list_prepend (addrom_info->data->roms_list,
				(gpointer) rom_info);

	save_rom (rom_info);

	list = g_list_first (addrom_info->data->roms_list);

	l = g_list_length (list);

	for (i = 0; i < l; i++) {
		gtk_clist_get_text (GTK_CLIST (addrom_info->data->clist),
				    i, 0, &name);
		if (!strcmp (name, rom_info->string_prefs[NAME]))
			r = i;
	}

	gtk_clist_select_row (GTK_CLIST (addrom_info->data->clist), r, 0);

	status = g_strdup_printf (_("Total ROMs: %d"), g_list_length
				  (g_list_first
				   (addrom_info->data->roms_list)));
	gtk_label_set_text (GTK_LABEL (addrom_info->data->status), status);
	g_free (status);
	
	gtk_widget_destroy (addrom_info->dialog);
}

void
add_cancel_cb (GtkWidget *widget, AddROMInfo *addrom_info)
{
	F_NAME();
	
	gtk_widget_destroy (addrom_info->dialog);
}

void
add_free_cb (GtkWidget *widget, AddROMInfo *addrom_info)
{
	F_NAME();
	
	g_free (addrom_info);
}

void
remove_ok_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	GList   *list = g_list_first (gsnes_data->roms_list);
	FILE    *file;
	ROMInfo *rom_info;
	gchar   *rom_name;
	gchar   *rom_info_file;
	gchar   *roms_file = gnome_util_home_file (ROMS_FILE);
	gchar   *status;
	gint     i, l;

	F_NAME();
	
	if (gsnes_data->current_rom == -1)
		return;

	gtk_clist_get_text (GTK_CLIST (gsnes_data->clist),
			    gsnes_data->current_rom, 0, &rom_name);

	while (strcmp (rom_name,
		       ((ROMInfo*)(list->data))->string_prefs[NAME]))
		list = g_list_next (list);

	rom_info = (ROMInfo*)(list->data);

	rom_info_file = g_strconcat (CONF_DIR, "/", rom_name, NULL);
	remove (gnome_util_home_file (rom_info_file));
	g_free (rom_info_file);
	rom_info_file = g_strconcat (CONF_DIR, "/", rom_name, ".gg", NULL);
	remove (gnome_util_home_file (rom_info_file));
	g_free (rom_info_file);
	rom_info_file = g_strconcat (CONF_DIR, "/", rom_name, ".ar", NULL);
	remove (gnome_util_home_file (rom_info_file));
	g_free (rom_info_file);
	rom_info_file = g_strconcat (CONF_DIR, "/", rom_name, ".gf", NULL);
	remove (gnome_util_home_file (rom_info_file));
	g_free (rom_info_file);
	
	gtk_clist_remove (GTK_CLIST (gsnes_data->clist),
			  gsnes_data->current_rom);

	list = g_list_first (list);

	list = g_list_remove (list, (gpointer) rom_info);

	gsnes_data->roms_list = list;
	
	gsnes_data->current_rom = -1;
	
	gtk_widget_destroy (gsnes_data->pixmap);

	list = g_list_first (gsnes_data->roms_list);

	if (list) {
	
		file = fopen (roms_file, "w");

		l = g_list_length (list);

		for (i = 0; i < l; i++) {
			fprintf (file, "%s\n",
				 ((ROMInfo*)(list->data))->string_prefs[NAME]);
			list = g_list_next (list);
		}

		fclose (file);
	} else
		remove (roms_file);
	
	gsnes_data->pixmap = get_image_pixmap (DEFAULT_IMAGE,
					       gsnes_data->width,
					       gsnes_data->height);
	
	gtk_scrolled_window_add_with_viewport (
		GTK_SCROLLED_WINDOW (gsnes_data->image_w),
		gsnes_data->pixmap);

	gtk_window_set_title (GTK_WINDOW (gsnes_data->app), "GSnes9x");

	gtk_widget_show (gsnes_data->pixmap);

	status = g_strdup_printf (_("Total ROMs: %d"), g_list_length
				  (g_list_first (gsnes_data->roms_list)));
	gtk_label_set_text (GTK_LABEL (gsnes_data->status), status);

	g_free (rom_info);
	g_free (status);
}

void
about_destroyed_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	F_NAME();

	gsnes_data->about_open = FALSE;
}

void
press_key_cb (GtkWidget *widget, GdkEvent *event, GSnes9xData* gsnes_data)
{
	F_NAME();
	
	if (gsnes_data->current_rom == -1 || !event)
		return;
	
	switch (event->key.keyval) {
	case GDK_KP_Enter:
		play_cb (NULL, gsnes_data);
		break;
	case GDK_Return:
		play_cb (NULL, gsnes_data);
		break;
	case GDK_Delete:
		remove_cb (NULL, gsnes_data);
		break;
	}
}

void
select_rom_cb (GtkWidget *clist, gint row, gint col,
	       GdkEventButton *event, GSnes9xData* gsnes_data)
{
	ROMInfo *rom_info;
	GList   *list = g_list_first (gsnes_data->roms_list);
	gchar   *name;
	gchar   *rom_title;

	F_NAME();
	
	gsnes_data->current_rom = row;
	
	gtk_clist_get_text (GTK_CLIST (clist), row, col, &name);
	
	while (strcmp (name,
		       ((ROMInfo*)(list->data))->string_prefs[NAME]))
		list = g_list_next (list);

	rom_info = (ROMInfo*)(list->data);
	
	gtk_widget_destroy (gsnes_data->pixmap);

	gsnes_data->pixmap =
		get_image_pixmap (rom_info->string_prefs[IMAGE_FILE],
				  gsnes_data->width,
				  gsnes_data->height);

	gtk_scrolled_window_add_with_viewport (
		GTK_SCROLLED_WINDOW (gsnes_data->image_w),
		gsnes_data->pixmap);

	rom_title = g_strconcat ("GSnes9x: ", rom_info->string_prefs[NAME],
				 NULL);
	
	gtk_window_set_title (GTK_WINDOW (gsnes_data->app),
			      rom_title);

	g_free (rom_title);
	
	gtk_widget_show (gsnes_data->pixmap);

	if (event && event->type == GDK_2BUTTON_PRESS)
		play_cb (NULL, gsnes_data);
}

gint
right_select_rom_cb (GtkWidget *clist, GdkEventButton *event,
		     GSnes9xData* gsnes_data)
{
	gdouble absh = 0;
	gdouble rowh = 0;
	gint    row  = -1;
	gint    roms = g_list_length (g_list_first (gsnes_data->roms_list));
	
	GtkAdjustment *adj;
	
	F_NAME();

	if (!roms)
		return FALSE;
	
	if (!event || !gsnes_data)
		return FALSE;

	if (event->button != 3)
		return FALSE;

	adj  = gtk_clist_get_vadjustment (GTK_CLIST (gsnes_data->clist));
	absh = adj->value + event->y;
	rowh = (gint)(adj->upper/roms);
	row  = (gint)(absh/rowh);

	gtk_clist_select_row (GTK_CLIST (gsnes_data->clist), row, 0);

	gtk_menu_popup (GTK_MENU (gsnes_data->popmenu), NULL, NULL,
			NULL, NULL, event->button, event->time);

	return TRUE;
}

gint
right_click_pixmap_cb (GtkWidget *clist, GdkEventButton *event,
		       GSnes9xData* gsnes_data)
{
	F_NAME();
	
	if (gsnes_data->current_rom == -1)
		return FALSE;

	if (event->button < 1 || event->button > 3)
		return FALSE;

	gtk_menu_popup (GTK_MENU (gsnes_data->popmenu), NULL, NULL,
			NULL, NULL, event->button, event->time);

	return TRUE;
}

void
destroy_data_cb (GtkWidget *widget, GtkWidget *data)
{
	F_NAME();
	
	gtk_widget_destroy (data);
}

void
toggle_to_active_cb (GtkWidget *widget, GtkWidget *data)
{
	F_NAME();
	
	gtk_widget_set_sensitive (data, GTK_TOGGLE_BUTTON (widget)->active);
}

void
no_dialog_cb (GtkWidget *widget, GSnes9xData *gsnes_data)
{
	F_NAME();

	gsnes_data->dialog_open = FALSE;
}

void
resize_pixmap_cb (GtkWidget *widget, GdkRectangle *area,
		  GSnes9xData *gsnes_data)
{
	ROMInfo *rom_info = NULL;
	GList   *list = g_list_first (gsnes_data->roms_list);
	gchar   *rom_name;

	F_NAME();
	
	gsnes_data->width  = area->width;
	gsnes_data->height = area->height;

	if (gsnes_data->width > 20)
		gsnes_data->width -= 20;
	if (gsnes_data->height > 20)
		gsnes_data->height -= 20;
	
	if (gsnes_data->current_rom != -1) {
		gtk_clist_get_text (GTK_CLIST (gsnes_data->clist),
				    gsnes_data->current_rom, 0, &rom_name);
	
		while (strcmp (rom_name,
			       ((ROMInfo*)(list->data))->
			       string_prefs[NAME]))
			list = g_list_next (list);
		
		rom_info = (ROMInfo*)(list->data);
	}
	
	gtk_widget_destroy (gsnes_data->pixmap);

	if (rom_info)
		gsnes_data->pixmap = get_image_pixmap
			(rom_info->string_prefs[IMAGE_FILE],
			 gsnes_data->width, gsnes_data->height);
	else
		gsnes_data->pixmap = get_image_pixmap
			(DEFAULT_IMAGE, gsnes_data->width,
			 gsnes_data->height);
	
	gtk_scrolled_window_add_with_viewport (
		GTK_SCROLLED_WINDOW (gsnes_data->image_w),
		gsnes_data->pixmap);
	
	gtk_widget_show (gsnes_data->pixmap);
}

void
popup_menu_detacher (GtkWidget *attach_widget, GtkMenu *menu)
{
	F_NAME();
}

/* callbacks.c ends here. */
