/*********************************************************************
 *
 *      Copyright (C) 1999-2002 Nathan Fiedler
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * PROJECT:     JSwat
 * MODULE:      View
 * FILE:        SourceView.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      04/08/99        Initial version
 *      nf      01/28/01        Use JavaParser for more accurate
 *                              breakpoint setting.
 *      nf      04/21/01        Fixed bug #97, view will scroll after
 *                              view becomes visible
 *      nf      05/04/01        Fixed bug #88, highlight current line
 *      nf      05/28/01        Fix bug 97, again, using Timer.
 *      nf      06/16/01        Changed to use new breakpoints code
 *      he      07/02/01        Added configurable tab size
 *      nf      07/22/01        Fixed bug 131, handle parse errors
 *      nf      08/18/01        Fixed bug 146, show parser errors in Log
 *      nf      08/26/01        Fixed bug 196
 *      nf      08/26/01        Fixed bug 190: making the impossible possible
 *      nf      10/25/01        Fixed bug 249
 *      nf      11/10/01        Fixed bug 293
 *      nf      11/14/01        Fixed bug 313
 *      nf      11/17/01        Fixed bug 318: merged ColorSourcView code
 *      nf      11/25/01        Fixed bug 320
 *      nf      11/28/01        Fixed bug 327
 *      nf      11/29/01        Fixed bug 332
 *      nf      12/04/01        Made colorization 10% faster
 *      tr      12/08/01        Fixed bugs 323, 325
 *      nf      12/15/01        Fixed bug 368
 *      nf      01/06/02        Fixed bug 380 with an ugly hack
 *      nf      04/03/02        Fixed bug 419
 *
 * DESCRIPTION:
 *      This file contains the SourceView class definition.
 *
 * $Id: SourceView.java,v 1.62 2002/04/08 04:41:48 nfiedler Exp $
 *
 ********************************************************************/

/* Ideas for improvement:

  - SourceView needs to be redesigned from scratch.
    - BasicView
      - Operations:
        + findString
        + foundString
        + removeHighlight
        + scrollToLine
        + showHighlight
      - Requires:
        + Highlight painter
        + Text area
        + Textual content
        + view title (in lieu of source file name)

    - UIAdapter needs class name when building a SourceView.
    - JavaSourceView can figure out the classname, if needed.

    - SourceView
      - Operations:
        - Showing breakpoints
        + contextChanged
        - matches
        + refresh(InputStream)
      - Requires:
        - Session
        - ContextManager
        - BreakpointManager

    - JavaSourceView:
      - Operations:
        - parseClassDefs
      - Requires:

    - SourceViewPopup: aware of breakpoints and sets them anywhere
    - JavaSourceViewPopup: aware of Java class definitions,
      sets breakpoints only where it makes sense to do so.

    - Need a view that is not associated with a File.
      - View of source generated by application server of JSP page.
        - Not associated with a file, must be via an input stream.
*/

package com.bluemarsh.jswat.view;

import com.bluemarsh.adt.SkipList;
import com.bluemarsh.config.ConfigureListener;
import com.bluemarsh.config.JConfigure;
import com.bluemarsh.jswat.ContextManager;
import com.bluemarsh.jswat.JSwat;
import com.bluemarsh.jswat.Log;
import com.bluemarsh.jswat.Session;
import com.bluemarsh.jswat.SessionListener;
import com.bluemarsh.jswat.breakpoint.BreakpointManager;
import com.bluemarsh.jswat.event.ContextChangeEvent;
import com.bluemarsh.jswat.event.ContextListener;
import com.bluemarsh.jswat.parser.java.lexer.LexerException;
import com.bluemarsh.jswat.parser.java.parser.ParserException;
import com.sun.jdi.Location;
import java.awt.Font;
import java.io.*;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import javax.swing.*;
import javax.swing.text.*;

/**
 * Defines the SourceView class which will be responsible for displaying
 * the source file on the screen. This class displays a row header down
 * the left side of the source view, indicating the line numbers.
 * A popup menu is attached to the view for managing breakpoints.
 *
 * @author  Nathan Fiedler
 */
public class SourceView extends BasicView implements ContextListener, ConfigureListener, SessionListener {
    /** Scrollable component for text component. */
    protected JScrollPane viewScroller;
    /** Popup menu for managing breakpoints. Listens to text component. */
    protected SourceViewPopup popupMenu;
    /** Session to which we belong. Set in <code>init()</code>. */
    protected Session owningSession;
    /** Width of tabs in characters. Defaults to 8. */
    protected int tabSize;
    /** Classlines of the parsed source file. */
    protected List classLines;
    /** Draw layer responsible for colorizing Java code. */
    protected JavaDrawLayer javaDrawLayer;
    /** Gutter draw layer for showing breakpoint locations and states. */
    protected BreakpointDrawLayer breakpointDrawLayer;
    /** Original view content, with any tab characters.
     * XXX - waste of space, ought to have text area deal with the tabs */
    protected char[] contentWithTabs;

    /**
     * Creates a SourceView object.
     *
     * @param  name  name of the file to display.
     */
    public SourceView(String name) {
        super(name);

        // Create the text component.
        textComponent = new SourceViewTextArea(viewContent);
        textComponent.addDrawLayer(lineHighlighter,
                                   DrawLayer.PRIORITY_STEPPING);
        javaDrawLayer = new JavaDrawLayer();
        textComponent.addDrawLayer(javaDrawLayer,
                                   DrawLayer.PRIORITY_COLORIZER);
        breakpointDrawLayer = new BreakpointDrawLayer(textComponent);
        textComponent.addGutterLayer(breakpointDrawLayer,
                                     GutterDrawLayer.PRIORITY_BREAKPOINT);

        // Set the text component preferences.
        setPreferences();

        // Hook into the configuration system.
        JConfigure config = JSwat.instanceOf().getJConfigure();
        config.addListener(this);

        // Set up the scroller for the text component.
        viewScroller = new JScrollPane(textComponent);

        // Create the breakpoint popup menu gadget for the text component.
        popupMenu = new SourceViewPopup("Breakpoints");
        textComponent.add(popupMenu);
        textComponent.addMouseListener(popupMenu);
    } // SourceView

    /**
     * Called when the Session is about to begin an active debugging
     * session. That is, JSwat is about to debug a debuggee VM.
     *
     * @param  session  Session being activated.
     */
    public void activate(Session session) {
        // Register for debugger events.
        ContextManager cmgr = (ContextManager)
            session.getManager(ContextManager.class);
        cmgr.addContextListener(this);
    } // activate

    /**
     * Called when the Session is about to close down.
     * Closes this view, detaching it from all other objects
     * and preparing it for finalization.
     *
     * @param  session  Session being closed.
     */
    public void close(Session session) {
        BreakpointManager bpman = (BreakpointManager)
            session.getManager(BreakpointManager.class);
        bpman.removeBreakListener(breakpointDrawLayer);
        JConfigure config = JSwat.instanceOf().getJConfigure();
        config.removeListener(this);
        owningSession = null;
        viewScroller = null;
        textComponent = null;
        popupMenu = null;
    } // close

    /**
     * Called then the prefernces change.
     */
    public void configurationChanged() {
        // Load the preferences.
        setPreferences();

        // Ugly hack to deal with tab width changes.
        viewContent = SourceContent.replaceTabs(contentWithTabs, tabSize);

        // Reload the text contents.
        if (viewContent != null) {
            setTextContent();
        }
    } // configurationChanged

    /**
     * Invoked when the current context has changed. The context
     * change event identifies which aspect of the context has
     * changed.
     *
     * @param  cce  context change event
     */
    public void contextChanged(ContextChangeEvent cce) {
        ContextManager cmgr = (ContextManager) cce.getSource();
        Location loc = cmgr.getCurrentLocation();
        // See if this event belongs to our source file.
        if ((loc != null) && matches(loc)) {
            int line = loc.lineNumber();
            // Show where the pc is located.
            // Minus one for some reason.
            showHighlight(line - 1);
            scrollToLine(line);
        } else {
            // Clear the current stepping line.
            removeHighlight();
        }
    } // contextChanged

    /**
     * Called when the Session is about to end an active debugging
     * session. That is, JSwat is about to terminate the connection
     * with the debuggee VM.
     *
     * @param  session  Session being deactivated.
     */
    public void deactivate(Session session) {
        ContextManager cmgr = (ContextManager)
            session.getManager(ContextManager.class);
        cmgr.removeContextListener(this);
        // Clear the current stepping line.
        removeHighlight();
        if (logCategory.isEnabled()) {
            logCategory.report("setting current line to 0 in " + viewTitle);
        }
    } // deactivate

    /**
     * Returns a reference to the UI component.
     *
     * @return  UI component object
     */
    public JComponent getUI() {
        return viewScroller;
    } // getUI

    /**
     * Called after the Session has added this listener to the
     * Session listener list.
     *
     * @param  session  Session adding this listener.
     */
    public void init(Session session) {
        owningSession = session;
        BreakpointManager bpman = (BreakpointManager)
            session.getManager(BreakpointManager.class);
        bpman.addBreakListener(breakpointDrawLayer);
    } // init

    /**
     * Check if the Location is in the same source file as the
     * file we are displaying.
     *
     * @param  location  Location to check.
     * @return  True if breakpoint source name same as our filename.
     */
    protected boolean matches(Location location) {
        if (classLines == null) {
            return false;
        } else {
            Iterator iter = classLines.iterator();
            String clazz = location.declaringType().name();
            while (iter.hasNext()) {
                ClassDefinition cd = (ClassDefinition) iter.next();
                if (cd.getClassName().equals(clazz)) {
                    return true;
                }
            }
        }
        return false;
    } // matches

    /**
     * Reads the class definition information from the source file
     * using the JavaParser class. Give this information to the
     * text component and row header popup menus.
     */
    protected void parseClassDefs() {
        // Read in the class definitions from the source file.
        CharArrayReader car = new CharArrayReader(viewContent);
        JavaParser parser = new JavaParser(car);
        try {
            classLines = parser.parse();
            // Parser closes the reader in all cases.
        } catch (ParserException pe) {
            owningSession.getStatusLog().writeStackTrace(pe);
        } catch (LexerException le) {
            owningSession.getStatusLog().writeStackTrace(le);
        } catch (IOException ioe) {
            owningSession.getStatusLog().writeStackTrace(ioe);
        }
        if (classLines != null) {
            // Make the list read-only to prevent errors.
            classLines = Collections.unmodifiableList(classLines);
            // Report that the class definitions were loaded successfully.
            if (logCategory.isEnabled()) {
                logCategory.report("class definitions loaded for " +
                                   viewTitle);
            }
        }
        popupMenu.setClassDefs(classLines);
        breakpointDrawLayer.setClassDefs(classLines);
    } // parseClassDefs

    /**
     * Read the input stream text into the text component. The view must
     * be added to the Session as a session listener before calling
     * this method.
     *
     * @param  input  input stream providing source.
     * @param  line   line to make visible.
     * @return  true if successful, false if error.
     */
    public boolean refresh(InputStream input, int line) {
        try {
            // Average Java source is under 32K characters.
            CharArrayWriter caw = new CharArrayWriter(32768);
            char[] buffer = new char[8192];
            InputStreamReader isr = new InputStreamReader(input);
            int bytesRead = isr.read(buffer, 0, buffer.length);
            while (bytesRead > 0) {
                caw.write(buffer, 0, bytesRead);
                bytesRead = isr.read(buffer, 0, buffer.length);
            }
            contentWithTabs = caw.toCharArray();
            // Convert the tabs to spaces because that causes problems
            // for searching and the text area width estimation.
            viewContent = SourceContent.replaceTabs(contentWithTabs, tabSize);
            isr.close();
            caw.close();
        } catch (SecurityException se) {
            se.printStackTrace();
            return false;
        } catch (IOException ioe) {
            ioe.printStackTrace();
            return false;
        }
        setTextContent();

        if (logCategory.isEnabled()) {
            logCategory.report("source view refreshed for " + viewTitle);
        }

        // Set all the breakpoints and the current line marker.
        // Parse first so the breakpoints can use the information.
        parseClassDefs();
        BreakpointManager bpman = (BreakpointManager)
            owningSession.getManager(BreakpointManager.class);
        breakpointDrawLayer.setBreakpoints(bpman);
        ContextManager cmgr = (ContextManager)
            owningSession.getManager(ContextManager.class);
        Location loc = cmgr.getCurrentLocation();
        if ((loc != null) && matches(loc)) {
            // Show where the pc is located.
            // Minus one for some reason.
            showHighlight(loc.lineNumber() - 1);
        }
        if (logCategory.isEnabled()) {
            logCategory.report(
                "breakpoints and other line attributes set for " +
                viewTitle);
        }
        scrollToLine(line);
        return true;
    } // refresh

    /**
     * Called to update this view's preferences, either when this object
     * is constructed or when the preferences change.
     */
    protected void setPreferences() {
        JConfigure config = JSwat.instanceOf().getJConfigure();

        // Notify the colorizer of possible changes.
        JavaDrawLayer.updateColors(config);

        // Set the font size of the text component.
        // Do this before setting the tab size.
        int fontSize = 12;
        try {
            fontSize = Integer.parseInt(config.getProperty("view.fontSize"));
        } catch(NumberFormatException nfe) {
            // we don't expect this to happen
            nfe.printStackTrace();
        }
        Font font = new Font("Monospaced", Font.PLAIN, fontSize);
        textComponent.setFont(font);

        // This is part of Hugh Emberson's tab size support.
        // Set the tab width of the text area.
        tabSize = 8;
        try {
            tabSize = Integer.parseInt(config.getProperty("view.tabWidth"));
        } catch(NumberFormatException nfe) {
            // we don't expect this to happen
            nfe.printStackTrace();
        }
    } // setPreferences

    /**
     * Set the content of the text component using the text defined
     * by <code>viewContent</code>. This implementation applies styles
     * to the Java source to colorize it syntactically.
     */
    protected void setTextContent() {
        JConfigure config = JSwat.instanceOf().getJConfigure();
        Boolean color = Boolean.valueOf(
            config.getProperty("view.colorSource"));

        SourceContent doc = new SourceContent(viewContent);

        if (color.booleanValue()) {
            // Colorize the source.
            CharArrayReader car = new CharArrayReader(viewContent);
            JavaScanner scanner = new JavaScanner(car, doc);
            SkipList tokenInfo = null;
            tokenInfo = scanner.scan();
            // Scanner closes the reader.
            // Give the JavaDrawLayer the token information.
            javaDrawLayer.setTokens(tokenInfo);
        } else {
            // Indicate that we have no tokens and the layer
            // should be inactive.
            javaDrawLayer.setTokens(null);
        }

        // Do this when we're all done, so it forces a repaint.
        textComponent.setContent(doc);
    } // setTextContent
} // SourceView
