/*
 * Jeffrey Friedl
 * Omron Corporation			ʳ
 * Nagaokakyoshi, Japan			617Ĺ
 *
 * jfriedl@nff.ncl.omron.co.jp
 *
 * This work is placed under the terms of the GNU General Purpose License
 * (the "GNU Copyleft").
 */

#include "config.h"
#include "assert.h"
#include <stdio.h>
#include <setjmp.h>
#include <ctype.h>
#include "output.h"
#include "jreadline.h"
#include "strsave.h"
#include "input.h"
#include "xmalloc.h"
#include "euc.h"

#include "system.h"
#if defined(_HAVE_STRINGS_H_) /* might be defined in system.h */
# include <strings.h>
#else
# include <string.h>
# define index strchr
# define rindex strrchr
#endif

#if !defined(__GNUC__)
#  if !defined(__volatile__)
#    define __volatile__ /*nothing; for use with volatile functions */
#  endif
#  if !defined(__inline__)
#    define __inline__ /*nothing; for use with volatile functions */
#  endif
#endif

#define array_elements(array)   (sizeof(array)/sizeof(array[0]))

/* idea and rough implementation for HANDLE_SIGNALS from GNU's readline */
#if !defined(NO_HANDLE_SIGNALS) && !defined(HANDLE_SIGNALS)
# define HANDLE_SIGNALS
#endif

#ifndef MOVE_MEMORY
   /* this must be a "safe" memory copy */
   #define MOVE_MEMORY(FROM, TO, LENGTH) \
       (void)bcopy((const char*)(FROM), (char*)(TO), (int)(LENGTH))
#endif

#define bindfunc static /* maybe someday we'll allow outside binding */

/*
 * The input buffer is where all input and editing are done.
 * It is accessed by
 *	dot -- buffer pointer. This points just beyond the "cursor"
 *		character.  Inserts are done at this pointer.
 *		Unless one has used the editing commands to move
 *		backward, dot will usually be the same as end_of_line.
 *
 * 	end_of_line -- end of the used input buffer.  Points just beyond the
 *		last character on the current input line.
 */
static unsigned char start_of_line[MAX_INPUT_LINE_LENGTH + 1];
static unsigned char *dot = start_of_line; /* dot = "cursor" pointer */
static unsigned char *end_of_line = start_of_line; /* points beyond last char*/


/*
 * Some macros to check the status of the start_of_line with respect to size.
 *
 *    ok_to_insert()
 *		is false if a character can't be added to the buffer.
 *    ok_to_move_forwrad
 *		is false if one is already at the end of the line.
 *    ok_to_move_backward
 *		is false if one is already at the beginning of the line.
 */
#define line_length()           (end_of_line - start_of_line)
#define bytes_into_line()       (dot - start_of_line)
#define bytes_from_end()        (end_of_line - dot)
#define eol()	                (dot == end_of_line)
#define bol()	                (dot == start_of_line)
#define ok_to_insert()		(line_length() < MAX_INPUT_LINE_LENGTH)
#define ok_to_insert2()		(line_length() < (MAX_INPUT_LINE_LENGTH-1))
#ifdef JIS0212_SUPPORT
#define ok_to_insert3()		(line_length()+1 < (MAX_INPUT_LINE_LENGTH-2))
#endif
#define ok_to_move_forward()	(!eol())
#define ok_to_move_backward()	(!bol())


/*
 * Returns the number of bytes in the character starting at PTR.
 */
static __inline__ int char_size(unsigned char *ptr)
{
   int size;
   kibishii_assert(ptr >= start_of_line && ptr < end_of_line);
   size = EUC_CHAR_LENGTH(*ptr);
   if (size == 0) {
       /*
        * This should never happen with normal input, but major problems if
	* it does, so better take care of it here.
	*/
       kibishii_assert(size);
       return 1;
   }
   return size;
}

/*
 * Returns the number of bytes in the character before the one starting at PTR.
 */
static __inline__ int prev_char_size(unsigned char *mark)
{
   unsigned char *ptr;
   int lastsize = 1;
   kibishii_assert(mark > start_of_line && mark <= end_of_line);

   for (ptr = start_of_line; ptr < mark; ptr += (lastsize=char_size(ptr)))
       ;
   kibishii_assert(mark == ptr);
   return lastsize;
}


/*
 * The history mechanism is a simple linked list of command texts.
 */
struct history_struct
{
    struct history_struct
	*h_prev, 	/* command previous to this one */
	*h_next;	/* command that happend after this one */

    const unsigned char *h_text;	/* text of the command */
};

/* head->prev is the last command, head->prev->prev is the one before that */
static struct history_struct history_head = {&history_head, &history_head, 0 };

/*
 * Current_history_pointer used by PREV, NEXT, etc for movement, and is
 * reset at each ENTER to be the head.
 */
static struct history_struct *current_history_ptr;

const unsigned char *jreadline_last_prompt = (void *)"> ";

const unsigned char *
jreadline_mod_prompt(const unsigned char *new)
{
    const unsigned char *old = jreadline_last_prompt;
    jreadline_last_prompt = new;
    return old;
}


/*
 * Given the text (of the most recently executed command), add it to the
 * head of the history list.
 */
void add_history(const unsigned char *text)
{
    struct history_struct *new;
    
    /* allocate and fill the new history struct */
    new = (struct history_struct *)xmalloc(sizeof(struct history_struct));
    new->h_text = strsave((void*)text);
    
    /* link it in */
    new->h_prev = history_head.h_prev;
    new->h_prev->h_next = new;
    history_head.h_prev = new;
    history_head.h_prev->h_next = &history_head;
}

static unsigned jreadline_active; /* true if getting an input line */
static unsigned waiting_for_input; /* true if waiting for a character */
static unsigned jreadline_display_width = 80;


#define max(x,y) ((x)>(y)?(x):(y))
#define min(x,y) ((x)<(y)?(x):(y))


/*
 * Print a copy of the current input buffer to the output.
 * If newline is true, one is appended.
 *
 * This is called each time the line is changed, so may be thought
 * of as updating the printed version of the internal buffer.
 */
static void update_display(int newline)
{
    static int printed_last_time = 0;
    int text_len = end_of_line - start_of_line;
    int prompt_len = strlen((void*)jreadline_last_prompt);
    int total_len = prompt_len + text_len;

    int starting_index, ending_index;
    int len_printed_before_dot;
    int len_printed_after_dot;
    int printed_this_time;
    int dot_index = prompt_len + (dot - start_of_line);
    int print_only_until_dot = 0;

    if (total_len < (jreadline_display_width - 2))
    {
	starting_index = 0;
	ending_index = total_len;
    }
    else
    {
	unsigned available_width = jreadline_display_width - 1;

	ending_index = dot_index + (available_width / 2);
	if (ending_index > total_len)
	    ending_index = total_len;

	starting_index = ending_index - (available_width - 2);
	if (starting_index < 0)
	    starting_index = 0;
        if (starting_index > 0)
	{
	    /* first, save room for an initial "" */
	    starting_index += 2;
	    available_width -= 2;

	    /*
	     * Too much to print, must pick some part of prompt+line to
	     * display. We propose to start printing at STARTING_INDEX, but
	     * must make sure that's not in the middle of a multi-byte
	     * character. If it is, we'll bump it up to the start of the
	     * next character.
	     */
	    if (starting_index < prompt_len)
	    {
		/*
		 * Proposed starting place is in the prompt... check from the
		 * beginning of the prompt to the proposed location...
		 */
		int i = 0;
		while (i < starting_index)
		{
		    if (jreadline_last_prompt[i] & 0x80)
		    {
                        #ifdef JIS0212_SUPPORT
		            if (jreadline_last_prompt[i] == /*CodeSet3*/ 143)
				i += 3;
			    else
                        #endif
			    i += 2;
		    }
		    else
			i++;
		}
		starting_index = i;
	    }
	    else
	    {
		/*
		 * Proposed starting place is in the line somewhere...
		 * check down the line to the proposed starting index.
		 */
		int i = 0;

		while (prompt_len + i < starting_index)
		{
		    unsigned char c = start_of_line[i];
		    if (c & 0x80)
		    {
			#ifdef JIS0212_SUPPORT
			    if (c == /*CodeSet3*/ 143)
				 i += 3;
			     else
			#endif
			    i += 2;
		    }
		    else
			i ++;
		}
		starting_index = prompt_len + i;
	    }
	}
	ending_index = starting_index + available_width;
	if (ending_index > total_len)
	    ending_index = total_len;
	else
	    ending_index -= 2; /* save room for a final "" */
    }

    soft_assert(dot_index >= starting_index);
    soft_assert(dot_index <= ending_index);

    output_pager_transparent(1);

  again:
    outchar('\r');
    len_printed_after_dot = 0;
    if (starting_index == 0)
	len_printed_before_dot = 0;
    else
    {
	len_printed_before_dot = 2;
	output("");
    }

    /* output any prompt */
    if (starting_index < prompt_len)
    {
	const unsigned char
	    *this_start = jreadline_last_prompt + starting_index,
	    *this_end = jreadline_last_prompt + min(prompt_len, ending_index);

	while (this_start < this_end)
	{
	    unsigned char c = *this_start;
	    if (!(c & 0x80))
	    {
		outchar(c);
		this_start++;
		len_printed_before_dot++;
	    }
	    else
	    {
		if (this_start + 2 > this_end)
		    break;
		outchar(c);
		this_start++;
                #ifdef JIS0212_SUPPORT
		    if (c == 143)
		        outchar(*(this_start++));
                #endif
		outchar(*(this_start++));
		len_printed_before_dot += 2;
	    }
	}
	
	if (this_start > this_end)
	     len_printed_before_dot += (this_start - this_end);
    }

    if (ending_index > prompt_len)
    {
	const unsigned char
	    *this_start,
	    *this_end = start_of_line + (ending_index - prompt_len);

	if (starting_index <= prompt_len)
	    this_start = start_of_line;
	else
	    this_start = start_of_line + (starting_index - prompt_len);

	while (this_start < this_end)
	{
	    unsigned char c = *this_start;
	    if (!(c & 0x80))
	    {
		outchar(c);
		if (this_start < dot)
		    len_printed_before_dot++;
		else
		    len_printed_after_dot++;
		this_start++;
	    }
	    else
	    {
		if (this_start + 2 > this_end)
		    break;
		outchar(c);
		if (this_start < dot)
		    len_printed_before_dot += 2;
		else
		    len_printed_after_dot += 2;
		this_start++;

                #ifdef JIS0212_SUPPORT
		    if (c == 143)
		       outchar(*(this_start++));
                #endif
		outchar(*(this_start++));
	    }
	}
    }

    if (print_only_until_dot)
	goto done;

    if (ending_index != total_len)
    {
	output("");
	len_printed_after_dot += 2;
    }

    printed_this_time = len_printed_before_dot + len_printed_after_dot;
    if (printed_this_time < printed_last_time)
    {
	int needed = min(printed_last_time, jreadline_display_width)
	    - printed_this_time;
	len_printed_after_dot += needed;
	while (needed--)
	    outchar(' ');
    }
    printed_last_time = printed_this_time;

    if (len_printed_after_dot && !newline)
    {
	if (len_printed_after_dot > len_printed_before_dot)
	{
	    ending_index = dot_index;
	    print_only_until_dot = 1;
	    goto again;
	} else {
	    while (len_printed_after_dot--)
		outchar('\b');
	}
    }

  done:
    if (newline)
	outchar('\n');
    else
	flush_output();

    output_pager_transparent(0);
}

unsigned set_jreadline_width(unsigned new)
{
    unsigned old = jreadline_display_width;
    jreadline_display_width = new;
    if (waiting_for_input)
	update_display(0);
    return old;
}


/*
 * Return values for the various functions that get invoked directly
 * by a users typing a character.
 *
 * These functions, all in the form
 *	int bind_FUNCTION(unsigned char c)
 *
 * Are called with the character that was entered to invoke the command.
 * For most such commands, the character is ignored.
 */
#define CONTINUE	1	/* continue accepting input */
#define END_OF_INPUT 	2	/* EOF ...*/
#define RETURN_TO_USER	3	/* ENTER */

int (*jreadline_access)(unsigned char *,unsigned char **,unsigned char **) = 0;

#ifndef NO_AUTO_ROMAJI_CONVERSION
int jreadline_auto_romaji = 0;

static romaji_converter_t romaji_converter;

romaji_converter_t set_romaji_converter(romaji_converter_t new)
{
    romaji_converter_t old = romaji_converter;
    romaji_converter = new;
    return old;
}

#define romaji_conversion(force)                                             \
macro_start {                                                                \
    if (romaji_converter)                                                    \
	(*romaji_converter)(start_of_line,                                   \
			    &start_of_line[sizeof(start_of_line)],           \
			    &dot,                                            \
			    (const unsigned char **)&end_of_line,            \
			    (force));                                        \
} macro_end

bindfunc int bind_convert_romaji(unsigned char c)
{
    romaji_conversion(1);
    return CONTINUE;
}
#endif

/* Ring the bell (as some sign of user error) */
bindfunc int bind_ring_bell(unsigned char c)
{
    outchar('\007');
    flush_output();
    return CONTINUE;
}

/* select the next command in the history list, if there is one */
bindfunc int bind_next_history_line(unsigned char c)
{
    if (current_history_ptr->h_next == &history_head)
	(void)bind_ring_bell(c);	/* oops, not one there */
    else
    {
	int i = 0;
	current_history_ptr = current_history_ptr->h_next;
	
	/*
	 * Copy the text to the input buffer, setting dot
	 * and end_of_line to their proper values.
	 */
	dot = start_of_line;
	while (c = current_history_ptr->h_text[i++], c != 0)
	    *dot++ = c;
	    ;
	end_of_line = dot;
    }
    return CONTINUE;
}

/* select the previous command in the history list, if there is one */
bindfunc int bind_previous_history_line(unsigned char c)
{
    if (current_history_ptr->h_prev == &history_head)
	(void)bind_ring_bell(c);	/* not one */
    else
    {
	int i = 0;
	
	current_history_ptr = current_history_ptr->h_prev;
	
	/*
	 * Copy the text to the input buffer, setting dot
	 * and end_of_line to their proper values.
	 */
	dot = start_of_line;
	while (c = current_history_ptr->h_text[i++], c != 0)
	    *dot++ = c;
	end_of_line = dot;
    }
    
    return CONTINUE;
}

/* Redraw the current input line */
bindfunc int bind_redraw(unsigned char c)
{
    update_display(1);
    return CONTINUE;
}

/* the ENTER command */
bindfunc int bind_EOL(unsigned char c)
{
    return RETURN_TO_USER;
}

/* the EOF "command"  */
static int at_EOF(void)
{
    if (end_of_line - start_of_line == 0)
	return END_OF_INPUT;
    else
	return RETURN_TO_USER;	/* if there's input, use it */
}

/* destructive backspace. */
bindfunc int bind_delete_char_backward(unsigned char c)
{
    if (!ok_to_move_backward())
	(void)bind_ring_bell(c);	/* can't move back */
    else
    {
	unsigned size = prev_char_size(dot);
	/*
	 * Delete char to the left of the cursor.
	 * If there's text to the right, it must be shifted
	 * over.
	 */
	if (ok_to_move_forward())
	    MOVE_MEMORY(/*from*/dot, /*to*/dot-size, end_of_line - dot);
	dot -= size;
	end_of_line -= size;
    }
    return CONTINUE;
}

/* gobble the current character -- opposit of the above */
bindfunc int bind_delete_char_forward(unsigned char c)
{
    if (!ok_to_move_forward())
	return bind_delete_char_backward(c);
    else
    {
	unsigned size = char_size(dot);
	/* delete the char under the cursor */
	MOVE_MEMORY(/*from*/dot + size, /*to*/dot, end_of_line - dot);
	end_of_line -= size;
    }
    return CONTINUE;
}

/*
 * Move forward one character, if possible.
 * If not, return an at_EOF
 */
bindfunc int bind_delete_char_forward_or_EOF(unsigned char c)
{
    if (ok_to_move_forward())
	return bind_delete_char_forward(c);
    else if (ok_to_move_backward())
	return bind_delete_char_backward(c);
    else
	return at_EOF();
}

/* move forward one character, if possible */
bindfunc int bind_forward_character(unsigned char c)
{
    if (!ok_to_move_forward())
	(void)bind_ring_bell(c);
    else
	dot += char_size(dot);
    return CONTINUE;
}

/* move back one character, if possible */
bindfunc int bind_backward_character(unsigned char c)
{
    if (!ok_to_move_backward())
	(void)bind_ring_bell(c);
    else
	dot -= prev_char_size(dot);
    return CONTINUE;
}

/* Insert the given character into the current input buffer, if possible */
bindfunc int bind_self_insert(unsigned char c)
{
    if (!ok_to_insert())
	(void)bind_ring_bell(c);
    else
    {
	/* push any characters that are to the right of the cursor */
	int length = end_of_line - dot;
	if (length > 0)
	    MOVE_MEMORY(/*from*/dot, /*to*/dot + 1, length);
	
	/* add the character */
	*dot++ = c;
	
	/* bump up the end pointer, too */
	end_of_line++;

        #ifndef NO_AUTO_ROMAJI_CONVERSION
	    if (jreadline_auto_romaji && isascii(c) && !isspace(c))
		romaji_conversion(0);
        #endif
    }
    return CONTINUE;
}

/* Insert the given characters into the current input buffer, if possible */
bindfunc int bind_self_insert2(unsigned char c1, unsigned char c2)
{
    if (!ok_to_insert2())
	(void)bind_ring_bell(c1);
    else
    {
	/* push any characters that are to the right of the cursor */
	int length = end_of_line - dot;
	if (length > 0)
	    MOVE_MEMORY(/*from*/dot, /*to*/dot + 2, length);
	
	/* add the characters */
	dot[0] = c1;
	dot[1] = c2;
	dot += 2;
	
	/* bump up the end pointer, too */
	end_of_line += 2;
    }
    return CONTINUE;
}

#ifdef JIS0212_SUPPORT
/* Insert the given characters into the current input buffer, if possible */
bindfunc int bind_self_insert3(unsigned char c1, unsigned char c2, unsigned char c3)
{
    if (!ok_to_insert3())
	(void)bind_ring_bell(c1);
    else
    {
	/* push any characters that are to the right of the cursor */
	int length = end_of_line - dot;
	if (length > 0)
	    MOVE_MEMORY(/*from*/dot, /*to*/dot + 3, length);
	
	/* add the characters */
	dot[0] = c1;
	dot[1] = c2;
	dot[2] = c3;
	dot += 3;
	
	/* bump up the end pointer, too */
	end_of_line += 3;
    }
    return CONTINUE;
}
#endif

/* Tabs become spaces... */
bindfunc int bind_tab(unsigned char c)
{
    return bind_self_insert(' ');
}

/* move to the end of the line */
bindfunc int bind_end_of_line(unsigned char c)
{
    dot = end_of_line;
    return CONTINUE;
}

/* delete to the end of the line */
bindfunc int bind_kill_to_end(unsigned char c)
{
    end_of_line = dot;
    return CONTINUE;
}

/* move to the start of the line */
bindfunc int bind_start_of_line(unsigned char c)
{
    dot = start_of_line;
    return CONTINUE;
}

/* delete to the start of the line */
bindfunc int bind_kill_to_start(unsigned char c)
{
    int len = end_of_line - dot;
    
    if (len == 0)
	end_of_line = dot = start_of_line;
    else
    {
	MOVE_MEMORY(/*from*/dot, /*to*/start_of_line, len);
	dot = start_of_line;
	end_of_line = dot + len;
    }
    return CONTINUE;
}


static enum jis_mode {
    ascii,
    roman = ascii,	 /* we'll treat them the same */
    jis78,
    jis83 = jis78,	 /* we'll treat them the same */
    jis90 = jis78,	 /* we'll treat them the same */
#ifdef JIS0212_SUPPORT
    jis_0212,
#endif
    hw_kata
} jis_mode;

static enum hi_bit_mode {
    EUC,
    SJIS
} hi_bit_mode = EUC;

unsigned jreadline_highbit_input(unsigned selection)
{
    unsigned old = (hi_bit_mode == EUC) ? JREADLINE_EUC :  JREADLINE_SJIS;
    if (selection == JREADLINE_EUC)
	hi_bit_mode = EUC;
    else if (selection == JREADLINE_SJIS)
	hi_bit_mode = SJIS;
    return old;
}

static int bind_have_escape(unsigned char x)
{
    struct
    {
	const unsigned char *string;
	enum jis_mode mode;
	int marker;
    }
    escape[] =
    {
	{ (const unsigned char *)"$@",		jis78 },
	{ (const unsigned char *)"$B",		jis83 },
	{ (const unsigned char *)"$&@\33$B",	jis90 },
#ifdef JIS0212_SUPPORT
	{ (const unsigned char *)"$(D",	jis_0212 },
#endif
	{ (const unsigned char *)"(J",	        roman },
	{ (const unsigned char *)"(H",	        roman },
	{ (const unsigned char *)"(B", 	ascii },
	{ (const unsigned char *)"(I",	        hw_kata },
    };
    #define escapes array_elements(escape)
    int i, j = 0, count;

    for (i = 0; i < escapes; i++)
	escape[i].marker = 1;

    do
    {
	unsigned char c = next_raw_input_byte();

	for (count = i = 0; i < escapes; i++)
	    if (escape[i].marker)
	    {
		if (escape[i].string[j] != c)
		    escape[i].marker = 0;
		else if (escape[i].string[j+1] == '\0')
		{
		    jis_mode = escape[i].mode;
		    return CONTINUE;
		} else
		    count++;
	    }
	j++;
    } while(count);

    return bind_ring_bell(0);
}


static int (*bind_action[])(unsigned char c) =
{
#ifndef NO_AUTO_ROMAJI_CONVERSION
  /*   0 : '^@'*/   bind_convert_romaji,
#else
  /*   0 : '^@'*/   bind_ring_bell,
#endif
  /*   1 : '^A'*/   bind_start_of_line,
  /*   2 : '^B'*/   bind_backward_character,
  /*   3 : '^C'*/   bind_ring_bell,
  /*   4 : '^D'*/   bind_delete_char_forward_or_EOF,
  /*   5 : '^E'*/   bind_end_of_line,
  /*   6 : '^F'*/   bind_forward_character,
  /*   7 : '^G'*/   bind_delete_char_forward,
  /*   8 : '^H'*/   bind_delete_char_backward,
  /*   9 : '^I'*/   bind_tab,
  /*  10 : '^J'*/   bind_EOL,
  /*  11 : '^K'*/   bind_kill_to_end,
  /*  12 : '^L'*/   bind_redraw,
  /*  13 : '^M'*/   bind_EOL,
  /*  14 : '^N'*/   bind_next_history_line,
  /*  15 : '^O'*/   bind_ring_bell,
  /*  16 : '^P'*/   bind_previous_history_line,
  /*  17 : '^Q'*/   bind_ring_bell,
  /*  18 : '^R'*/   bind_redraw,
  /*  19 : '^S'*/   bind_ring_bell,
  /*  20 : '^T'*/   bind_ring_bell,
  /*  21 : '^U'*/   bind_kill_to_start,
  /*  22 : '^V'*/   bind_ring_bell,
  /*  23 : '^W'*/   bind_ring_bell,
  /*  24 : '^X'*/   bind_ring_bell,
  /*  25 : '^Y'*/   bind_ring_bell,
  /*  26 : '^Z'*/   bind_ring_bell,
  /*  27 : ''*/   bind_have_escape,
  /*  28 : ''*/   bind_ring_bell,
  /*  29 : ''*/   bind_ring_bell,
  /*  30 :     */   bind_ring_bell,
  /*  31 :     */   bind_ring_bell,
};

#ifdef HANDLE_SIGNALS
#include <signal.h>

#ifndef SIG_TYPE
# ifdef __GNUC__
#  define SIG_TYPE __typeof__(SIG_DFL)
# else
   typedef void (*SIG_TYPE)(); /* take a guess with it */
# endif
#endif
static jmp_buf top_level;

SIG_TYPE sig_int;   /* interrupt */
SIG_TYPE sig_quit;  /* quit */
SIG_TYPE sig_ill;   /* illegal instruction */
SIG_TYPE sig_fpe;   /* floating */
SIG_TYPE sig_bus;   /* bus error */
SIG_TYPE sig_segv;  /* segmentation error */
#ifdef SIGTSTP
 SIG_TYPE sig_tstp;  /* tty stop */
#endif

static void release_signals(void)
{
    (void)signal(SIGINT,  sig_int);
    (void)signal(SIGQUIT, sig_quit);
    (void)signal(SIGILL,  sig_ill);
    (void)signal(SIGFPE,  sig_fpe);
    (void)signal(SIGBUS,  sig_bus);
    (void)signal(SIGSEGV, sig_segv);
#ifdef SIGTSTP
    (void)signal(SIGTSTP, sig_tstp);
#endif
}
#ifdef luna88k
static int signal_handler(int sig, int code, struct sigcontext *context);
#else
static int signal_handler(int sig, int code);
#endif

static void grab_signals(void)
{
    sig_int  = signal(SIGINT,  (void*)signal_handler);
    sig_quit = signal(SIGQUIT, (void*)signal_handler);
    sig_ill  = signal(SIGILL,  (void*)signal_handler);
    sig_fpe  = signal(SIGFPE,  (void*)signal_handler);
    sig_bus  = signal(SIGBUS,  (void*)signal_handler);
    sig_segv = signal(SIGSEGV, (void*)signal_handler);
#ifdef SIGTSTP
    sig_tstp = signal(SIGTSTP, (void*)signal_handler);
#endif
}


#ifdef luna88k
static int signal_handler(int sig, int code, struct sigcontext *context)
#else
static int signal_handler(int sig, int code)
#endif
{
    reset_tty_state();
    release_signals();

    kill(getpid(), sig);
    #if !defined(__svr4__) && !defined(__DGUX__)
    sigsetmask (0);
    #endif

    /* if we make it back... */
#ifdef SIGTSTP
    if (sig == SIGTSTP)
    {
	update_display(0);
	set_tty_state_to_cbreak();
	grab_signals();
	return 0;
    }
#endif
    longjmp(top_level, 1);
    return 0; /* notreached */
}
#endif /* HANDLE_SIGNALS */


/*
 * Get one line of input.
 */
unsigned char *readline(const unsigned char *prompt)
{
    int action = CONTINUE; /* "may be clobbered by longjmp" OK here */

    /*
     * reset the input buffer pointers to point to a clear
     * input area.
     */
    end_of_line = dot = start_of_line;

    /* want history searches to start from here */
    current_history_ptr = &history_head;

    jreadline_last_prompt = prompt ? prompt : (void*)"";
    /* print the initial prompt if was given one */

    jreadline_active = 1;

    /* GRAB THE TTY */
    set_tty_state_to_cbreak();

    #ifdef HANDLE_SIGNALS
    if (setjmp(top_level)) {
	jreadline_active = 0;
	return strsave((const unsigned char *)"");
    }
    grab_signals();
    #endif

    jis_mode = ascii;
    while (action == CONTINUE)
    {
	unsigned char c1, c2;

	/* no need to redisplay if more input ready */
	if (!input_pending())
	{
	    if (jreadline_access)
		(*jreadline_access)(start_of_line, &dot, &end_of_line);
	    update_display(0);
	}

	c1 = next_raw_input_byte();

	#define retry_c1_with(X)                                             \
	macro_start {                                                        \
	    c1 = (X);                                                        \
	    goto use_c1;                                                     \
	} macro_end

      use_c1:

	if (c1 & 0x80)
	{
 	    /* SJIS or EUC */
	    if (hi_bit_mode == EUC)
	    {
		if (c2 = next_raw_input_byte(), (c2 & 0x80) == 0)
		    retry_c1_with(c2);

#ifdef JIS0212_SUPPORT
		if (c1 == 0x143 /* Code Set 3 */)
		{
		    unsigned char c3;
		    /* have a JIX 0212 */
		    if (c3 = next_raw_input_byte(), (c3 & 0x80) == 0)
			retry_c1_with(c3);

		    action = bind_self_insert3(c1,c2,c3);
		} else
#endif
		if (c1 == 142 /* Code Set 2 */) {
		    /* should convert to full-width here */
		    action = bind_self_insert2(c1,c2);
		} else {
		    action = bind_self_insert2(c1,c2);
		}
	    } else { /* Shift-JIS */
		if (c1 >= 161 && c1 <= 223) {
		    /* should convert to full-width here */
		    action = bind_self_insert2(142, c1);
		} else {
		    if (c2 = next_raw_input_byte(), c2 < 64)
			retry_c1_with(c2);

		    /* convert c1:c2 to EUC */
		    action = bind_self_insert2(
		      (((c1 - (c1<160 ? 112:176))<<1) - (c2<159)) | 0x80,
		      (c2 - (c2<159 ? (c2>127?32:31) : 126)) | 0x80);
		}
	    }
	}
	else if (c1 == 127)
	    action = bind_delete_char_backward(c1);
	else if (c1 < array_elements(bind_action))
	{
	    jis_mode = ascii; /* probably should do this */
	    action = (*bind_action[(int)(unsigned char)c1])(c1);
	}
	else if (jis_mode == ascii)
	    action = bind_self_insert(c1);
	else if (jis_mode == hw_kata) {
	    bind_self_insert2(142, c1 | 0x80);
	} else {
	    c2 = next_raw_input_byte();
#ifdef JIS0212_SUPPORT
	    if (jis_mode == jis_0212)
		bind_self_insert3(143, c1|0x80, c2|0x80);
	    else
#endif
		bind_self_insert2(c1|0x80, c2|0x80);
	}
	*end_of_line = 0;	/* cap off the text -- now a string */
    } while (action == CONTINUE);

    #ifndef NO_AUTO_ROMAJI_CONVERSION
        /*
	 * To fake-force any final conversion that might need to be
	 * done ('n', 'h', etc.), tack on an 'x' (a rather safe character)
	 * and offer a conversion. Then take back off.
	 */
	if (jreadline_auto_romaji)
	{
	    *end_of_line++ = 'x';
	    *(dot = end_of_line) = '\0';
	    romaji_conversion(0);
	    *(dot = --end_of_line) = '\0';
	}
    #endif

    update_display(1); /* show final line */

    reset_tty_state();
    #ifdef HANDLE_SIGNALS
    release_signals();
    #endif

    jreadline_active = 0;
    return action == END_OF_INPUT ? NULL : strsave(start_of_line);
}
