/* CompoundGraphFilter.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.conjecture.filter;

import java.util.ResourceBundle;
import javax.swing.event.ListDataEvent;

import org.grinvin.graphs.GraphBundleView;
import org.grinvin.list.DefaultFilterListModel;
import org.grinvin.list.FilterList;
import org.grinvin.list.FilterListModel;
import org.grinvin.list.FilterListModelListener;

import org.jdom.Element;

/**
 *
 */
public class CompoundGraphFilter extends AbstractGraphFilter implements FilterListModelListener {
    
    public enum Operator {
        AND,
        OR;
    }
    
    private FilterListModel filters;
    
    private static final ResourceBundle RBUNDLE = ResourceBundle.getBundle("org.grinvin.conjecture.filter.resources");
    
    private Operator operator;
    
    /** Creates a new instance of CompoundGraphFilter */
    CompoundGraphFilter() {
        //to be used by GraphFilterManager
        this(new DefaultFilterListModel());
    }

    public CompoundGraphFilter(FilterListModel filters) {
        this(filters, Operator.AND);
    }

    public CompoundGraphFilter(FilterListModel filters, Operator operator) {
        this(filters, operator, false);
    }

    public CompoundGraphFilter(FilterListModel filters, Operator operator, boolean negate) {
        super(negate);
        this.operator = operator;
        setFilterList(filters);
    }
    
    public FilterList getFilters() {
        return filters;
    }
    
    public void setFilterList(FilterListModel newFilters) {
        if (filters != null) {
            filters.removeFilterListModelListener(this);
        }
        filters = newFilters;
        filters.addFilterListModelListener(this);
        fireFilterChanged();
    }

    public Element toElement() {
        Element element = new Element("CompoundFilter").setAttribute("negate", Boolean.toString(isNegated())).setAttribute("operator", operator.toString());
        for(GraphFilter filter : filters)
            element.addContent(filter.toElement());
        return element;
    }

    public void fromElement(Element element) {
        if(element.getName().equalsIgnoreCase("CompoundFilter")){
            for(Object obj : element.getChildren())
                filters.add(GraphFilterManager.getInstance().fromElement((Element)obj));
            negate(Boolean.parseBoolean(element.getAttributeValue("negate")));
            try {
                operator = Operator.valueOf(element.getAttributeValue("operator"));
            } catch (IllegalArgumentException ex) {
                operator = Operator.AND;
            }
        }
    }

    public boolean accepts_impl(GraphBundleView gb) {
        boolean accept = (operator == Operator.AND);
        if(operator == Operator.OR)
            for(GraphFilter filter : filters)
                accept = accept || filter.accepts(gb);
        else
            for(GraphFilter filter : filters)
                accept = accept && filter.accepts(gb);
        return accept;
    }

    public String getDescription_impl() {
        String operatorName = "filter.name.and"; //and is default
        if(operator == Operator.OR)
            operatorName = "filter.name.or";
        StringBuffer buf = new StringBuffer();
        if(filters.size()>0){
            buf.append("(");
            buf.append(filters.get(0).getDescription());
            for(int i=1; i<filters.size();i++)
                buf.append(" ").append(RBUNDLE.getString(operatorName)).append(" ").append(filters.get(i).getDescription());
            buf.append(")");
        }
        return buf.toString();
    }

    public void intervalAdded(ListDataEvent e) {
        fireFilterChanged();
    }

    public void intervalRemoved(ListDataEvent e) {
        fireFilterChanged();
    }

    public void contentsChanged(ListDataEvent e) {
        fireFilterChanged();
    }
    
}
