/* ====================================================================
 * Copyright (c) 1995-2002 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *	notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *	notice, this list of conditions and the following disclaimer in
 *	the documentation and/or other materials provided with the
 *	distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *	software must display the following acknowledgment:
 *	"This product includes software developed by the Apache Group
 *	for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *	endorse or promote products derived from this software without
 *	prior written permission. For written permission, please contact
 *	apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *	nor may "Apache" appear in their names without prior written
 *	permission of the Apache Group.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by the Apache Group
 *	for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */

/*
 * fpexec.c -- "Wrapper" support program for fpEXEC behaviour for Apache
 *
 ***********************************************************************
 *
 * NOTE! : DO NOT edit this code!!!  Unless you know what you are doing,
 *	 editing this code might open up your system in unexpected 
 *	 ways to would-be crackers.  Every precaution has been taken 
 *	 to make this code as safe as possible; alter it at your own
 *	 risk.
 *
 ***********************************************************************
 *
 */

#include "ap_config.h"
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <stdarg.h>

#include "fpexec.h"
#include "path.h"

/*
 ***********************************************************************
 * There is no initgroups() in QNX, so I believe this is safe :-)
 * Use cc -ofpexec -3 -O -mf -DQNX fpexec.c to compile.
 *
 * May 17, 1997.
 * Igor N. Kovalenko -- infoh@mail.wplus.net
 ***********************************************************************
 */

#if defined(NEED_INITGROUPS)
int
initgroups(const char *name, gid_t basegid) {
/* QNX and MPE do not appear to support supplementary groups. */
	return 0;
}
#endif

#if defined(PATH_MAX)
#define AP_MAXPATH PATH_MAX
#elif defined(MAXPATHLEN)
#define AP_MAXPATH MAXPATHLEN
#else
#define AP_MAXPATH 8192
#endif

#define AP_ENVBUF 256

extern char **environ;
static FILE *log = NULL;

char *safe_env_lst[] = {
	"AUTH_TYPE",
	"CONTENT_LENGTH",
	"CONTENT_TYPE",
	"DATE_GMT",
	"DATE_LOCAL",
	"DOCUMENT_NAME",
	"DOCUMENT_PATH_INFO",
	"DOCUMENT_ROOT",
	"DOCUMENT_URI",
	"FILEPATH_INFO",
	"GATEWAY_INTERFACE",
	"LAST_MODIFIED",
	"PATH_INFO",
	"PATH_TRANSLATED",
	"QUERY_STRING",
	"QUERY_STRING_UNESCAPED",
	"REMOTE_ADDR",
	"REMOTE_HOST",
	"REMOTE_IDENT",
	"REMOTE_PORT",
	"REMOTE_USER",
	"REDIRECT_QUERY_STRING",
	"REDIRECT_STATUS",
	"REDIRECT_URL",
	"REQUEST_METHOD",
	"REQUEST_URI",
	"SCRIPT_FILENAME",
	"SCRIPT_NAME",
	"SCRIPT_URI",
	"SCRIPT_URL",
	"SERVER_ADMIN",
	"SERVER_NAME",
	"SERVER_PORT",
	"SERVER_PROTOCOL",
	"SERVER_SOFTWARE",
	"UNIQUE_ID",
	"USER_NAME",
	"TZ",
	NULL
};


static void
err_output(int severe, const char *fmt, va_list ap) {
#ifdef FP_LOG_EXEC
	time_t timevar;
	struct tm *lt;

	if (!log) {
		if ((log = fopen(FP_LOG_EXEC, "a")) == NULL) {
			fprintf(stderr, "failed to open fpEXEC log file\n");
			perror("fopen");
			exit(1);
		}
	}

	time(&timevar);
	lt = localtime(&timevar);

	fprintf(log, "[%d-%.2d-%.2d %.2d:%.2d:%.2d]: ",
	    lt->tm_year + 1900, lt->tm_mon + 1, lt->tm_mday,
	    lt->tm_hour, lt->tm_min, lt->tm_sec);

	vfprintf(log, fmt, ap);

	fflush(log);
#endif /* FP_LOG_EXEC */

	if (severe) {
		printf ("Content-Type: text/html\n\n");
		printf ("<HTML>\n<ul>\n<li>status=1\n<li>osstatus=0\n");
		printf ("<li>msg=FrontPage security violation.\n");
		printf ("<li>osmsg=(not displayed for security reasons)\n");
		printf ("</ul>\n</HTML>\n");
	}
	return;
}

static void
log_msg(const char *fmt,...) {
	va_list ap;

	va_start(ap, fmt);
	err_output(0, fmt, ap);
	va_end(ap);
	return;
}

static void
log_err(const char *fmt,...) {
	va_list ap;

	va_start(ap, fmt);
	err_output(1, fmt, ap);
	va_end(ap);
	return;
}

static void
clean_env(void) {
	char pathbuf[512];
	char **cleanenv;
	char **ep;
	int cidx = 0;
	int idx;


	if ((cleanenv = (char **) calloc(AP_ENVBUF, sizeof(char *))) == NULL) {
		log_err("failed to malloc memory for environment\n");
		exit(120);
	}

	snprintf(pathbuf, sizeof(pathbuf), "PATH=%s", FP_SAFE_PATH);
	cleanenv[cidx] = strdup(pathbuf);
	cidx++;

	for (ep = environ; *ep && cidx < AP_ENVBUF-1; ep++) {
		if (*ep == NULL) {
			log_err("HTTP header mismatch, stop\n");
			exit(120);
		}

		if (!strncmp(*ep, "HTTP_", 5) || !strncmp(*ep,"HTTPS",5)
		    || !strncmp(*ep,"SSL_",4)) {
			cleanenv[cidx] = *ep;
			cidx++;
		} else {
			for (idx = 0; safe_env_lst[idx]; idx++) {
				if (!strncmp(*ep, safe_env_lst[idx],
				    strlen(safe_env_lst[idx]))) {
					cleanenv[cidx] = *ep;
					cidx++;
				break;
				}
			}
		}
	}

	cleanenv[cidx] = NULL;

	environ = cleanenv;
}

int
main(int argc, char *argv[]) {
	int userdir;			/* userdir flag			*/
	uid_t uid;			/* user information		*/
	gid_t gid;			/* target group placeholder	*/
	uid_t fpuid;			/* FrontPage owner UID		*/
	gid_t fpgid;			/* FrontPage owner GID		*/
	const char *target_uname;	/* target user name		*/
	const char *target_gname;	/* target group name		*/
	char *target_homedir;		/* target home directory	*/
	char *actual_uname;		/* actual user name		*/
	char *actual_gname;		/* actual group name		*/
	const char *cmd;		/* command to be executed	*/
	const char *cwd;		/* current working directory	*/
	char dwd[AP_MAXPATH];		/* docroot working directory	*/
	char fpd[AP_MAXPATH];		/* workarea for FrontPage	*/
	struct passwd *pw;		/* password entry holder	*/
	struct group *gr;		/* group entry holder		*/
	struct stat dir_info;		/* directory info holder	*/
	struct stat prg_info;		/* program info holder		*/
	struct stat fp_info;		/* FrontPage info holder	*/

	userdir = 0;

	/*
	 * If there are a proper number of arguments, set
	 * all of them to variables.  Otherwise, error out.
	 */
	target_uname = getenv ("FPUID");
	target_gname = getenv ("FPGID");
	cmd = getenv ("FPEXE");
	cwd = getenv ("FPDIR");

	/*
	 * Check existence/validity of the UID of the user
	 * running this program.  Error out if invalid.
	 */
	uid = getuid();
	if ((pw = getpwuid(uid)) == NULL) {
		log_err("invalid uid: (%ld)\n", uid);
		exit(102);
	}


	/*
	 * Check the size of cwd and cmd
	 * This should save us against buffer
	 * overflows. A local user on the system
	 * could set the local variables and overflow
	 * the fpd buffer. We have to do additional
	 * checks later.
	 */
	if (cmd != NULL && (strlen(cmd) + 1 > AP_MAXPATH)) {
		log_err("Prevented buffer overflow hack from uid (%ld)\n", uid);
		exit(102);
	}
	if (cwd != NULL && (strlen(cwd) + 1 > AP_MAXPATH)) {
		log_err("Prevented buffer overflow hack from uid (%ld)\n", uid);
		exit(102);
	}

	/*
	 * Check to see if the user running this program
	 * is the user allowed to do so as defined in
	 * fpexec.h.  If not the allowed user, error out.
	 */
#ifdef _OSD_POSIX
	/* User name comparisons are case
	 *  insensitive on BS2000/OSD
	 */
	if (strcasecmp(FP_HTTPD_USER, pw->pw_name)) {
		log_err("user mismatch (%s instead of %s)\n", pw->pw_name, FP_HTTPD_USER);
		exit(103);
	}
#else  /*_OSD_POSIX*/
	if (strcmp(FP_HTTPD_USER, pw->pw_name)) {
		log_err("user mismatch (%s instead of %s)\n", pw->pw_name, FP_HTTPD_USER);
		exit(103);
	}
#endif /*_OSD_POSIX*/

	if (target_uname == NULL) {
		log_err("invalid target user name: (NULL)\n");
		exit(105);
	}

	if (!strncmp("~", target_uname, 1)) {
		target_uname++;
		userdir = 1;
	}

	/*
	 * Check this is one of the allowed commands.
	 */
	if ((strcmp (cmd, SHTML ) != 0) &&
	    (strcmp (cmd, FPCOUNT) != 0) &&
	    (strcmp (cmd, AUTHOR ) != 0) &&
	    (strcmp (cmd, ADMIN ) != 0) &&
	    (strcmp (cmd, ADMINCGI ) != 0)) {
		log_err("invalid command (%s)\n", cmd);
		exit(104);
	}

	/*
	 * Get the FrontPage owner UID and GID.
	 */
	if ((pw = getpwnam(FP_USER)) == NULL) {
		log_err("invalid FrontPage Server Extensions directory owner user name: (%s)\n",
		    FP_USER);
		exit(105);
	}
	fpuid = pw->pw_uid;

	if (strspn(FP_GROUP, "1234567890") != strlen(FP_GROUP)) {
		if ((gr = getgrnam(FP_GROUP)) == NULL) {
			log_err("invalid FrontPage Server Extensions directory owner group name: (%s)\n",
			    FP_GROUP);
			exit(106);
		}
		gid = gr->gr_gid;
	}
	else
		gid = atoi(FP_GROUP);

	fpgid = gid;

	/*
	 * Error out if the target username is invalid.
	 */
	if ((pw = getpwnam(target_uname)) == NULL) {
		log_err("invalid target user name: (%s)\n", target_uname);
		exit(105);
	}

	/*
	 * Error out if the target group name is invalid.
	 */
	if (strspn(target_gname, "1234567890") != strlen(target_gname)) {
		if ((gr = getgrnam(target_gname)) == NULL) {
			log_err("invalid target group name: (%s)\n", target_gname);
			exit(106);
		}
		gid = gr->gr_gid;
		actual_gname = strdup(gr->gr_name);
	}
	else {
		gid = atoi(target_gname);
		actual_gname = strdup(target_gname);
	}

#ifdef _OSD_POSIX
	/*
	 * Initialize BS2000 user environment
	 */
	{
	pid_t pid;
	int status;

		switch (pid = ufork(target_uname)) {
		case -1:	/* Error */
			log_err("failed to setup bs2000 environment for user %s: %s\n",
			    target_uname, strerror(errno));
			exit(150);
		case 0:	/* Child */
			break;
		default:	/* Father */
			while (pid != waitpid(pid, &status, 0))
			;
			/* XXX FIXME: should we deal
			 * with STOP signals as well?
			 */
			if (WIFSIGNALED(status))
				kill (getpid(), WTERMSIG(status));
			exit(WEXITSTATUS(status));
		}
	}
#endif /*_OSD_POSIX*/

	/*
	 * Save these for later since initgroups will hose the struct
	 */
	uid = pw->pw_uid;
	actual_uname = strdup(pw->pw_name);
	target_homedir = strdup(pw->pw_dir);

	/*
	 * Log the transaction here to be sure we have an open log 
	 * before we setuid().
	 */
	log_msg("uid: (%s/%s) gid: (%s/%s) cmd: %s\n",
	    target_uname, actual_uname,
	    target_gname, actual_gname,
	    cmd);

	/*
	 * Error out if attempt is made to execute as root or as
	 * a UID less than FP_UID_MIN.  Tsk tsk.
	 */
	if ((uid == 0) || (uid < FP_UID_MIN)) {
		log_err("cannot run as forbidden uid (%d/%s)\n", uid, cmd);
		exit(107);
	}

	/*
	 * Error out if attempt is made to execute as root group
	 * or as a GID less than FP_GID_MIN.  Tsk tsk.
	 */
	if ((gid == 0) || (gid < FP_GID_MIN)) {
		log_err("cannot run as forbidden gid (%d/%s)\n", gid, cmd);
		exit(108);
	}

	/*
	 * Change UID/GID here so that the following tests work over NFS.
	 *
	 * Initialize the group access list for the target user,
	 * and setgid() to the target group. If unsuccessful, error out.
	 */
	if (((setgid(gid)) != 0) || (initgroups(actual_uname, gid) != 0)) {
		log_err("failed to setgid (%ld: %s)\n", gid, cmd);
		exit(109);
	}

	/*
	 * setuid() to the target user.  Error out on fail.
	 */
	if ((setuid(uid)) != 0) {
		log_err("failed to setuid (%ld: %s)\n", uid, cmd);
		exit(110);
	}

	/*
	 * Get the current working directory.
	 */
	if (chdir (cwd) != 0) {
		log_err("cannot set current working directory\n");
		exit(111);
	}

	if (userdir) {
		if (((chdir(target_homedir)) != 0) ||
		    ((chdir(FP_USERDIR_SUFFIX)) != 0) ||
		    ((getcwd(dwd, AP_MAXPATH)) == NULL) ||
		    ((chdir(cwd)) != 0)) {
			log_err("cannot get docroot information (%s)\n", target_homedir);
			exit(112);
		}
	} else {
		if (((chdir(FP_DOC_ROOT)) != 0) ||
		    ((getcwd(dwd, AP_MAXPATH)) == NULL) ||
		    ((chdir(cwd)) != 0)) {
			log_err("cannot get docroot information (%s)\n", FP_DOC_ROOT);
			exit(113);
		}
	}

	/*
	 * Stat the cwd and verify it is a directory, or error out.
	 */
	if (((lstat(cwd, &dir_info)) != 0) || !(S_ISDIR(dir_info.st_mode))) {
		log_err("cannot stat directory: (%s)\n", cwd);
		exit(115);
	}

	/*
	 * Error out if cwd is writable by others.
	 */
	if ((dir_info.st_mode & S_IWOTH) || (dir_info.st_mode & S_IWGRP)) {
		log_err("directory is writable by others: (%s)\n", cwd);
		exit(116);
	}

	/*
	 * Error out if the target name/group is different from
	 * the name/group of the cwd.
	 */
	if ((uid != dir_info.st_uid) ||
	    (gid != dir_info.st_gid)) {
		log_err("target uid/gid (%ld/%ld) mismatch "
		    "with directory (%ld/%ld) %s\n",
		    uid, gid, dir_info.st_uid,
		    dir_info.st_gid, cwd);
		exit(120);
	}

	/*
	 * Be careful if we copy strings
	 */
	snprintf(fpd, sizeof(fpd), "%s%c%s", cwd, '/', FP_HTACCESS);

	/*
	 * Stat the .htaccess file in the cwd to be sure it exists and is
	 * not writable by others.
	 */
	if (((lstat(fpd, &fp_info)) != 0) || (S_ISLNK(fp_info.st_mode))) {
		log_err("cannot stat: %s\n", fpd);
		exit(117);
	}

	if ((fp_info.st_mode & S_IWOTH) || (fp_info.st_mode & S_IWGRP)) {
		log_err("file is writable by others: %s\n", fpd);
		exit(118);
	}

	if ((uid != fp_info.st_uid) ||
	    (gid != fp_info.st_gid)) {
		log_err("target uid/gid (%ld/%ld) mismatch "
		    "with file (%ld/%ld): %s\n",
		    uid, gid, fp_info.st_uid,
		    fp_info.st_gid, fpd);
		exit(120);
	}

	/*
	 * Be careful if we copy strings
	 */
	snprintf(fpd, sizeof(fpd), "%s%s", cwd, _VTI_BIN);

	/*
	 * Stat the _vti_bin directory in the cwd to be sure it exists and is
	 * not writable by others.
	 */

	if (((lstat(fpd, &fp_info)) != 0) || !(S_ISDIR(fp_info.st_mode))) {
		log_err("cannot stat: %s\n", fpd);
		exit(117);
	}

	if ((fp_info.st_mode & S_IWOTH) || (fp_info.st_mode & S_IWGRP)) {
		log_err("directory is writable by others: %s\n", fpd);
		exit(118);
	}

	if ((uid != fp_info.st_uid) ||
	    (gid != fp_info.st_gid)) {
		log_err("target uid/gid (%ld/%ld) mismatch "
		    "with directory (%ld/%ld): %s\n",
		    uid, gid, fp_info.st_uid,
		    fp_info.st_gid, fpd);
		exit(120);
	}

	/*
	 * Be careful if we copy strings
	 */
	snprintf(fpd, sizeof(fpd), "%s%s%c%s", cwd, _VTI_BIN, '/', FP_HTACCESS);

	/*
	 * Stat the .htaccess file in the _vti_bin directory to be sure it
	 * exists and is not writable by others.
	 */
  
	if (((lstat(fpd, &fp_info)) != 0) ||
	    (S_ISLNK(fp_info.st_mode))) {
		log_err("cannot stat: %s\n", fpd);
		exit(117);
	}

	if ((fp_info.st_mode & S_IWOTH) || (fp_info.st_mode & S_IWGRP)) {
		log_err("file is writable by others: %s\n", fpd);
		exit(118);
	}

	if ((uid != fp_info.st_uid) || (gid != fp_info.st_gid)) {
		log_err("target uid/gid (%ld/%ld) mismatch "
		    "with file (%ld/%ld): %s\n",
		    uid, gid, fp_info.st_uid,
		    fp_info.st_gid, fpd);
		exit(120);
	}

	/*
	 * If the command is AUTHOR or ADMIN, stat the _vti_bin/_vti_aut or
	 * _vti_bin/_vti_adm directory to be sure it exists and is not
	 * writable by others.
	 */

	if ((strcmp(cmd, ADMIN) == 0) || (strcmp(cmd, AUTHOR) == 0)) {
		/*
		 * Be careful if we copy strings
		 */
		if ((strcmp(cmd, ADMIN) == 0) || (strcmp(cmd, ADMINCGI) == 0)) {
			snprintf(fpd, sizeof(fpd), "%s%s", cwd, _VTI_ADM);
		} else {
			snprintf(fpd, sizeof(fpd), "%s%s", cwd, _VTI_AUT);
		}

		if (((lstat(fpd, &fp_info)) != 0) || !(S_ISDIR(fp_info.st_mode))) {
			log_err("cannot stat: %s\n", fpd);
			exit(117);
		}

		if ((fp_info.st_mode & S_IWOTH) || (fp_info.st_mode & S_IWGRP)) {
			log_err("directory is writable by others: %s\n", fpd);
			exit(118);
		}

		if ((uid != fp_info.st_uid) ||
			(gid != fp_info.st_gid)) {
			log_err("target uid/gid (%ld/%ld) mismatch "
			    "with directory (%ld/%ld): %s\n",
			    uid, gid, fp_info.st_uid,
			    fp_info.st_gid, fpd);
			exit(120);
	}

	/*
	 * Be careful if we copy strings
	 */
	if ((strcmp(cmd, ADMIN) == 0) || (strcmp(cmd, ADMINCGI) == 0)) {
		snprintf(fpd, sizeof(fpd), "%s%s%c%s", cwd, _VTI_ADM, '/', FP_HTACCESS);
	} else {
		snprintf(fpd, sizeof(fpd), "%s%s%c%s", cwd, _VTI_AUT, '/', FP_HTACCESS);
	}

	if (((lstat(fpd, &fp_info)) != 0) || (S_ISLNK(fp_info.st_mode))) {
		log_err("cannot stat: %s\n", fpd);
		exit(117);
	}

	if ((fp_info.st_mode & S_IWOTH) || (fp_info.st_mode & S_IWGRP)) {
		log_err("file is writable by others: %s\n", fpd);
		exit(118);
	}

	if ((uid != fp_info.st_uid) ||
		(gid != fp_info.st_gid)) {
		log_err("target uid/gid (%ld/%ld) mismatch "
			"with file (%ld/%ld): %s\n",
			uid, gid,
			fp_info.st_uid, fp_info.st_gid, fpd);
		exit(120);
	}
	}

	/*
	 * Be careful if we copy strings
         */
	snprintf(fpd, sizeof(fpd), "%s", FPDIR);

	if (((lstat(fpd, &fp_info)) != 0) || !(S_ISDIR(fp_info.st_mode))) {
		log_err("cannot stat: %s\n", fpd);
		exit(117);
	}

	if ((fp_info.st_mode & S_IWOTH) || (fp_info.st_mode & S_IWGRP)) {
		log_err("directory is writable by others: %s\n", fpd);
		exit(118);
	}

	if ((fpuid != fp_info.st_uid) ||
	    (fpgid != fp_info.st_gid)) {
		log_err("owner uid/gid (%ld/%ld) mismatch "
		    "with directory (%ld/%ld): %s\n",
		    fpuid, fpgid, fp_info.st_uid,
		    fp_info.st_gid, fpd);
		exit(120);
	}

	/*
	 * Be careful if we copy strings
         */
	snprintf(fpd, sizeof(fpd), "%s%s", FPDIR, cmd);

	/*
	 * Error out if we cannot stat the program.
	 */
	if (((lstat(fpd, &prg_info)) != 0) || (S_ISLNK(prg_info.st_mode))) {
		log_err("cannot stat program: %s\n", fpd);
		exit(117);
	}

	/*
	 * Error out if the program is writable by others.
	 */
	if ((prg_info.st_mode & S_IWOTH) || (prg_info.st_mode & S_IWGRP)) {
		log_err("file is writable by others: %s\n", fpd);
		exit(118);
	}

	/*
	 * Error out if the file is setuid or setgid.
	 */
	if ((prg_info.st_mode & S_ISUID) || (prg_info.st_mode & S_ISGID)) {
		log_err("file is either setuid or setgid: %s\n", fpd);
		exit(119);
	}

	/*
	 * Error out if the owner name/group is different from
	 * the name/group of the program.
	 */
	if ((fpuid != prg_info.st_uid) ||
	    (fpgid != prg_info.st_gid)) {
		log_err("owner uid/gid (%ld/%ld) mismatch "
		    "with program (%ld/%ld): %s\n",
		    fpuid, fpgid, prg_info.st_uid,
		    prg_info.st_gid, fpd);
		exit(120);
	}
	/*
	 * Error out if the program is not executable for others.
	 * Otherwise, she won't find any error in the logs except for
	 * "[error] Premature end of script headers: ..."
	 */
	if (!(prg_info.st_mode & S_IXOTH)) {
		log_err("file has no execute permission: %s\n", fpd);
		exit(121);
	}

	clean_env();

	/* 
	 * Be sure to close the log file so the CGI can't
	 * mess with it.  If the exec fails, it will be reopened 
	 * automatically when log_err is called.  Note that the log
	 * might not actually be open if FP_LOG_EXEC isn't defined.
	 * However, the "log" cell isn't ifdef'd so let's be defensive
	 * and assume someone might have done something with it
	 * outside an ifdef'd FP_LOG_EXEC block.
	 */
	if (log != NULL) {
		fclose(log);
		log = NULL;
	}


	argv[0] = fpd;

	/*
	 * Set a safe umask.  FP_UMASK is used as a minimum mask.  If the
	 * Apache is running with other bits masked, they remain in effect.
	 */
#ifdef FP_UMASK
	umask(FP_UMASK | umask(FP_UMASK));
#endif /*FP_UMASK*/

	/*
	 * fpadmcgi.exe seems not to send the content type "text/html" for every
	 * case. The official Frontpage patch suffers as we do. This ugly workaround
	 * does save us. (Note that there is only one "\n", else we have the ctype
	 * printed above the html site.
	 */
	if ((strcmp (cmd, ADMINCGI )) == 0) {
		printf("Content-Type: text/html\n");
		fflush(stdout);
	}

	/*
	 * Execute the command, replacing our image with its own.
	 */
	execv(fpd, &argv[0]);

	/* (I can't help myself...sorry.)
	 *
	 * Uh oh.  Still here.  Where's the kaboom?  There was supposed to be an
	 * EARTH-shattering kaboom!
	 *
	 * Oh well, log the failure and error out.
	 */
	log_err("(%d)%s: exec failed (%s)\n", errno, strerror(errno), fpd);
	exit(255);
}
