/* ====================================================================
 * Copyright (c) 1995-2002 The Apache Group.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *	notice, this list of conditions and the following disclaimer. 
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *	notice, this list of conditions and the following disclaimer in
 *	the documentation and/or other materials provided with the
 *	distribution.
 *
 * 3. All advertising materials mentioning features or use of this
 *	software must display the following acknowledgment:
 *	"This product includes software developed by the Apache Group
 *	for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * 4. The names "Apache Server" and "Apache Group" must not be used to
 *	endorse or promote products derived from this software without
 *	prior written permission. For written permission, please contact
 *	apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *	nor may "Apache" appear in their names without prior written
 *	permission of the Apache Group.
 *
 * 6. Redistributions of any form whatsoever must retain the following
 *	acknowledgment:
 *	"This product includes software developed by the Apache Group
 *	for use in the Apache HTTP server project (http://www.apache.org/)."
 *
 * THIS SOFTWARE IS PROVIDED BY THE APACHE GROUP ``AS IS'' AND ANY
 * EXPRESSED OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE APACHE GROUP OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Group and was originally based
 * on public domain software written at the National Center for
 * Supercomputing Applications, University of Illinois, Urbana-Champaign.
 * For more information on the Apache Group and the Apache HTTP server
 * project, please see <http://www.apache.org/>.
 *
 */

/*
 * fpstatic.c -- "Wrapper" support program for static frontpage files.
 */

#include "ap_config.h"
#include <sys/param.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>

#include <stdarg.h>

#include "fpexec.h"
#include "path.h"

#if defined(PATH_MAX)
#define AP_MAXPATH PATH_MAX
#elif defined(MAXPATHLEN)
#define AP_MAXPATH MAXPATHLEN
#else
#define AP_MAXPATH 8192
#endif
#define AP_ENVBUF 256

int	translate (const char *);

typedef enum { GIF, HTM } content_type;
extern char **environ;
static FILE *log = NULL;

static void
err_output(int severe, const char *fmt, va_list ap) {
#ifdef FP_LOG_EXEC
	time_t timevar;
	struct tm *lt;

	if (!log) {
		if ((log = fopen(FP_LOG_EXEC, "a")) == NULL) {
			fprintf(stderr, "failed to open fpEXEC log file\n");
			perror("fopen");
			exit(1);
		}
	}

	time(&timevar);
	lt = localtime(&timevar);

	fprintf(log, "[%d-%.2d-%.2d %.2d:%.2d:%.2d]: ",
	    lt->tm_year + 1900, lt->tm_mon + 1, lt->tm_mday,
	    lt->tm_hour, lt->tm_min, lt->tm_sec);

	vfprintf(log, fmt, ap);

	fflush(log);
#endif /* FP_LOG_EXEC */

        if (severe) {
                printf ("Content-Type: text/html\n\n");
                printf ("<HTML>\n<ul>\n<li>status=1\n<li>osstatus=0\n");
                printf ("<li>msg=FrontPage security violation.\n");
                printf ("<li>osmsg=(not displayed for security reasons)\n");
                printf ("</ul>\n</HTML>\n");
        }
	return;
}

static void
log_msg(const char *fmt,...) {
	va_list ap;

	va_start(ap, fmt);
	err_output(0, fmt, ap);
	va_end(ap);
	return;
}

static void
log_err(const char *fmt,...) {
	va_list ap;

	va_start(ap, fmt);
	err_output(1, fmt, ap);
	va_end(ap);
	return;
}

int
main(int argc, char *argv[]) {
	int userdir = 0;		/* userdir flag			*/
	int ret = 0;			/* return value of transl.	*/
	uid_t uid;			/* user information		*/
	gid_t gid;			/* target group placeholder	*/
	uid_t fpuid;			/* FrontPage owner UID		*/
	gid_t fpgid;			/* FrontPage owner GID		*/
	const char *target_uname;	/* target user name		*/
	const char *target_gname;	/* target group name		*/
	char *target_homedir;		/* target home directory	*/
	char *actual_uname;		/* actual user name		*/
	char *actual_gname;		/* actual group name		*/
	const char *cwd;		/* current working directory	*/
	const char *uri;		/* file location		*/
	char dwd[AP_MAXPATH];		/* docroot working directory	*/
	struct passwd *pw;		/* password entry holder	*/
	struct group *gr;		/* group entry holder		*/
	struct stat dir_info;		/* directory info holder	*/

	/*
	 * If there are a proper number of arguments, set
	 * all of them to variables.  Otherwise, error out.
	 */
	target_uname = getenv ("FPUID");
	target_gname = getenv ("FPGID");
	cwd = getenv ("FPDIR");
	uri = getenv ("REQUEST_URI");

	/*
	 * Check existence/validity of the UID of the user
	 * running this program.  Error out if invalid.
	 */
	uid = getuid();
	if ((pw = getpwuid(uid)) == NULL) {
		log_err("invalid uid: (%ld)\n", uid);
		exit(102);
	}

	/*
	 * Check to see if the user running this program
	 * is really the apache user. If not the allowed user,
	 * error out.
	 */
#ifdef _OSD_POSIX
	/* User name comparisons are case insensitive on BS2000/OSD */
	if (strcasecmp(FP_HTTPD_USER, pw->pw_name)) {
		log_err("user mismatch (%s instead of %s)\n",
		    pw->pw_name, FP_HTTPD_USER);
		exit(103);
	}
#else  /*_OSD_POSIX*/
	if (strcmp(FP_HTTPD_USER, pw->pw_name)) {
		log_err("user mismatch (%s instead of %s)\n",
		    pw->pw_name, FP_HTTPD_USER);
		exit(103);
	}
#endif /*_OSD_POSIX*/

	if (target_uname == NULL) {
		log_err("invalid target user name: (NULL)\n");
		exit(105);
	}

	if (!strncmp("~", target_uname, 1)) {
		target_uname++;
		userdir = 1;
	}

	/*
	 * Get the FrontPage owner UID and GID.
	 */
	if ((pw = getpwnam(FP_USER)) == NULL) {
		log_err("invalid FrontPage Server Extensions directory owner user name: (%s)\n",
		    FP_USER);
		exit(105);
	}
	fpuid = pw->pw_uid;

	if (strspn(FP_GROUP, "1234567890") != strlen(FP_GROUP)) {
		if ((gr = getgrnam(FP_GROUP)) == NULL) {
			log_err("invalid FrontPage Server Extensions directory owner group name: (%s)\n",
			    FP_GROUP);
			exit(106);
		}
		gid = gr->gr_gid;
	} else
		gid = atoi(FP_GROUP);

	fpgid = gid;

	/*
	 * Error out if the target username is invalid.
	 */
	if ((pw = getpwnam(target_uname)) == NULL) {
		log_err("invalid target user name: (%s)\n", target_uname);
		exit(105);
	}

	/*
	 * Error out if the target group name is invalid.
	 */
	if (strspn(target_gname, "1234567890") != strlen(target_gname)) {
		if ((gr = getgrnam(target_gname)) == NULL) {
			log_err("invalid target group name: (%s)\n", target_gname);
			exit(106);
		}
		gid = gr->gr_gid;
		actual_gname = strdup(gr->gr_name);
	} else {
		gid = atoi(target_gname);
		actual_gname = strdup(target_gname);
	}

#ifdef _OSD_POSIX
	/*
	 * Initialize BS2000 user environment
	 */
	{
		pid_t pid;
		int status;

		switch (pid = ufork(target_uname)) {
		case -1:	/* Error */
			log_err("failed to setup bs2000 environment for user %s: %s\n",
				target_uname, strerror(errno));
			exit(150);
		case 0:	/* Child */
			break;
		default:	/* Father */
			while (pid != waitpid(pid, &status, 0))
			;
			/* @@@ FIXME: should we deal with STOP signals as well? */
			if (WIFSIGNALED(status))
			kill (getpid(), WTERMSIG(status));
			exit(WEXITSTATUS(status));
		}
	}
#endif /*_OSD_POSIX*/

	/*
	 * Save these for later since initgroups will hose the struct
	 */
	uid = pw->pw_uid;
	actual_uname = strdup(pw->pw_name);
	target_homedir = strdup(pw->pw_dir);

	/*
	 * Log the transaction here to be sure we have an open log 
	 * before we setuid().
	 */
	log_msg("uid: (%s/%s) gid: (%s/%s) static file: %s\n",
	    target_uname, actual_uname,
	    target_gname, actual_gname,
	    uri);

	/*
	 * Error out if attempt is made to fetch as root
	 */
	if ((uid == 0) || (gid == 0)) {
		log_err("cannot fetch as root: %s\n", uri);
		exit(107);
	}

	/*
	 * Get the current working directory.
	 */
	if (chdir (cwd) != 0) {
		log_err("cannot set current working directory\n");
		exit(111);
	}

	if (userdir) {
		if (((chdir(target_homedir)) != 0) ||
		    ((chdir(FP_USERDIR_SUFFIX)) != 0) ||
		    ((getcwd(dwd, AP_MAXPATH)) == NULL) ||
		    ((chdir(cwd)) != 0)) {
			   log_err("cannot get docroot information (%s)\n", target_homedir);
			   exit(112);
			}
		} else {
		if (((chdir(FP_DOC_ROOT)) != 0) ||
			   ((getcwd(dwd, AP_MAXPATH)) == NULL) ||
			   ((chdir(cwd)) != 0)) {
			   log_err("cannot get docroot information (%s)\n", FP_DOC_ROOT);
			   exit(113);
		}
	}

	/*
	 * Stat the cwd and verify it is a directory, or error out.
	 */
	if (((lstat(cwd, &dir_info)) != 0) || !(S_ISDIR(dir_info.st_mode))) {
		log_err("cannot stat directory: (%s)\n", cwd);
		exit(115);
	}

	/*
	 * Error out if cwd is writable by others.
	 */
	if ((dir_info.st_mode & S_IWOTH) || (dir_info.st_mode & S_IWGRP)) {
		log_err("directory is writable by others: (%s)\n", cwd);
		exit(116);
	}

	/*
	 * Error out if the target name/group is different from
	 * the name/group of the cwd.
	 */
	if ((uid != dir_info.st_uid) ||
	    (gid != dir_info.st_gid)) {
		log_err("target uid/gid (%ld/%ld) mismatch "
		    "with directory (%ld/%ld) %s\n",
		    uid, gid, dir_info.st_uid,
		    dir_info.st_gid, cwd);
		exit(120);
	}

	/* 
	 * Be sure to close the log file so the CGI can't
	 * mess with it.  If the exec fails, it will be reopened 
	 * automatically when log_err is called.  Note that the log
	 * might not actually be open if FP_LOG_EXEC isn't defined.
	 * However, the "log" cell isn't ifdef'd so let's be defensive
	 * and assume someone might have done something with it
	 * outside an ifdef'd FP_LOG_EXEC block.
	 */
	if (log != NULL) {
		fclose(log);
		log = NULL;
	}

	/*
	 * Set a safe umask.  FP_UMASK is used as a minimum mask.  If the
	 * Apache is running with other bits masked, they remain in effect.
	 */
#ifdef FP_UMASK
	umask(FP_UMASK | umask(FP_UMASK));
#endif /*FP_UMASK*/

	 ret = translate(uri);
	 if (ret != 0) {
		log_err("Error in translate(), static binary does not exist. %s\n", uri);
		exit(118);
	 }
	 exit (0);
}

int translate (const char * uri) {
	content_type type;
	char *pos;
	FILE *fp;
	char *fnbuf[256];
	struct stat statbuf;
	int c;
	int Lcid;
	
	Lcid = 0;

	/*
	 * Initial check for the content type
	 */
	if ((pos = strrchr(uri,'.')) != NULL) {
		if (strcmp(pos, ".css\0") == 0)
			type = HTM;
		else if (strcmp(pos, ".gif\0") == 0)
			type = GIF;
		else if (strcmp(pos, ".js\0") == 0)
			type = HTM;
		else if (strcmp(pos, ".htm\0") == 0)
			type = HTM;
		else
			return (-1);

		/*
		 * Check for the help pages. If we match, we calculate
		 * the Lcid from the string.
		 */
		if ((pos = strstr(uri, _HELP)) != NULL) {
			pos = pos + 6;
			Lcid = (int)strtol(pos, &pos, 10);
			pos = strrchr((char *)uri, '/');
			snprintf((char *)fnbuf, sizeof(fnbuf), "%s%s%c%04d%s",
			    FPBASE, _HELP, '/', Lcid, pos);
		/*
		 * Check for the images. If we have a .gif ending,
		 * we can be sure that it's a gif in the help
		 * directory.
		 */
		} else if ((pos = strstr(uri, ".gif")) != NULL) {
			pos = strrchr((char *)uri, '/');
			snprintf((char *)fnbuf, sizeof(fnbuf), "%s%s%s%s%s",
			    FPBASE, _EXES, _VTI_ADM, _IMAGES, pos);
		/*
		 * If we still have .htm or .css, we end up
		 * with the admin dir. If we match, we calculate
		 * the Lcid from the string.
		 */
		} else if (((pos = strstr(uri, ".htm")) != NULL)
		|| ((pos = strstr(uri, ".css")) != NULL)) {
			pos = strstr(uri, _VTI_ADM);
			pos = pos + 19;
			Lcid = (int)strtol(pos, &pos, 10);
			snprintf((char *)fnbuf, sizeof(fnbuf), "%s%s%c%04d%s",
			    FPBASE, _ADMIN, '/', Lcid, pos);
		}
		if ((stat((char *)fnbuf,&statbuf)<0) || (!S_ISREG(statbuf.st_mode)))
			return (-1);

		if ((fp = fopen((char *)fnbuf, "r")) != NULL) {
			if (type == HTM) {
				printf("Content-Type: text/html\n\n");
			} else if (type == GIF) {
				printf("Content-Type: image/gif\n\n");
			} else {
				return (-1);
			}
			while((c = fgetc(fp)) != EOF) {
				fputc(c, stdout);
			}
		}
		return (0);
	}
	return (-1);
}
