// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/register_allocator.cpp,v 1.2 2001/08/13 10:00:42 xhshi Exp $
//

#include "defines.h"
#include <iostream.h>
#include "jit_intf.h"
#include "register_allocator.h"
#include "cg_prepass.h"
#include "jcfg.h"
#include "x86.h"

////////////////////////////////////////////////////////////////////
///   Register_Allocator_None
////////////////////////////////////////////////////////////////////

X86_Reg_No Register_Allocator_None::reg_var_allocated_to(int index,
        const unsigned char *where)
{
    return n_reg;
}

unsigned Register_Allocator_None::var_in_reg(X86_Reg_No reg_no, const unsigned char *where)
{
    return (unsigned) -1;
}

unsigned Register_Allocator_None::callee_saved_regs(int ebp_based)
{
	if (ebp_based)
		return (callee_saved_esi_mask | callee_saved_edi_mask | callee_saved_ebx_mask);
    else
        return 0;
}

int Register_Allocator_None::should_restore_callee(Jit_Method_Info *mi, X86_Reg_No reg)
{
    return (!mi->is_esp_based && reg != ebp_reg);
}

void Register_Allocator_None::should_restore_ebp(Jit_Method_Info *method_info, int out_args,
                                                 int &restore, int &off)
{
    restore = 0;
}

////////////////////////////////////////////////////////////////////
///   Register_Allocator_Simple
////////////////////////////////////////////////////////////////////

void Register_Allocator_Simple::get_register_mapping_and_liveness(unsigned k,
                                                                  Call_Site_Info *csi,
                                                                  Jit_Method_Info *mi,
                                                                  X86_Reg_No &reg, int &live)
{
    live = 1;
    
    if (k == (unsigned) mi->esi_local) {
        reg = esi_reg;
    } else if (k == (unsigned) mi->edi_local) {
        reg = edi_reg;
    } else if (k == (unsigned) mi->ebx_local) {
        reg = ebx_reg;
    } else if (k == (unsigned) mi->ebp_local) {
        reg = ebp_reg;
    } else {
        reg = n_reg;
    }
}

int Register_Allocator_Simple::should_restore_callee(Jit_Method_Info *mi, X86_Reg_No reg)
{
    switch (reg)
    {
    case ebx_reg:
        return (mi->ebx_local != -1 || !mi->is_esp_based);
        break;
    case ebp_reg:
        return (mi->ebp_local != -1 || !mi->is_esp_based);
        break;
    case esi_reg:
        return (mi->esi_local != -1 || !mi->is_esp_based);
        break;
    case edi_reg:
        return (mi->edi_local != -1 || !mi->is_esp_based);
        break;
    default:
        return 0;
        break;
    }
}

void Register_Allocator_Simple::should_restore_ebp(Jit_Method_Info *method_info, int out_args,
                                                   int &restore, int &off)
{
    restore = (method_info->ebp_local != -1);
    if (restore)
    {
        off = 
            out_args + 
            (method_info->edi_local != -1) + 
            (method_info->esi_local != -1);
        
    }
}

X86_Reg_No Register_Allocator_Simple::reg_var_allocated_to(int index, const unsigned char *where)
{
	if (ebx_local == index) return ebx_reg;
	if (ebp_local == index) return ebp_reg;
	if (esi_local == index) return esi_reg;
	if (edi_local == index) return edi_reg;
	// default
	return n_reg;
}

unsigned Register_Allocator_Simple::var_in_reg(X86_Reg_No reg_no, const unsigned char *where)
{
    switch (reg_no)
    {
    case ebx_reg:
        return ebx_local;
        break;
    case ebp_reg:
        return ebp_local;
        break;
    case esi_reg:
        return esi_local;
        break;
    case edi_reg:
        return edi_local;
        break;
    default:
        return (unsigned)-1;
        break;
    }
}

unsigned Register_Allocator_Simple::callee_saved_regs(int ebp_based)
{
    unsigned result = 0;

	if (esi_local != -1 || ebp_based)
		result |= callee_saved_esi_mask;
	if (edi_local != -1 || ebp_based)
		result |= callee_saved_edi_mask;
	if (ebx_local != -1 || ebp_based)
		result |= callee_saved_ebx_mask;
	if (ebp_local != -1)
		result |= callee_saved_ebp_mask;

    return result;
}

void Register_Allocator_Simple::update_method_info(Jit_Method_Info *mi)
{
	mi->esi_local = esi_local;
	mi->edi_local = edi_local;
	mi->ebx_local = ebx_local;
	mi->ebp_local = ebp_local;
#ifdef _DEBUGxx
	cout << mi->esi_local << " allocated to esi\n";
	cout << mi->edi_local << " allocated to edi\n";
	cout << mi->ebx_local << " allocated to ebx\n";
	cout << mi->ebp_local << " allocated to ebp\n";
#endif // _DEBUG
}

unsigned Register_Allocator_Simple::register_allocate(class CG_Prepass *prepass,
                                                      Method_Handle methodHandle,
                                                      Class_Handle classHandle,
                                                      unsigned max_locals,
                                                      const unsigned char *bc,
                                                      unsigned code_length)
{
    if (method_get_num_handlers(methodHandle) != 0)
        return 3;

    unsigned *ref_count = prepass->ref_count;
    unsigned avail_regs = (1 << esi_reg) | (1 << edi_reg) | (1<< ebx_reg) | (1 << ebp_reg);
    //	unsigned avail_regs = (1<< ebx_reg) | (1 << ebp_reg) | (1 << esi_reg);
    //	unsigned avail_regs = (1<< ebx_reg) ;
    //	unsigned avail_regs = 0 ;
    unsigned i;
    int n_assigned = 0;
    
   	DWORD method_flags = method_get_flags(methodHandle);
    if (method_flags & ACC_SYNCHRONIZED) 
        return 0;
    
    for ( i = 0; i < n_reg; i++) {
        // if not available, then continue
		if (!(avail_regs & (1<<i))) continue;
		// 
		// find one that has the biggest ref count (> 0)
		//
		int var_idx = -1;
		unsigned max_ref_count = 1;
		for (unsigned j = 0; j < max_locals; j++)
			if (ref_count[j] > max_ref_count) {
				max_ref_count = ref_count[j];
				var_idx = j;
			}
		if (var_idx == -1) return n_assigned;
		n_assigned++;
        X86_Reg_No reg = (X86_Reg_No) i;
        switch (reg)
        {
        case esi_reg:
            esi_local = var_idx;
            break;
        case edi_reg:
            edi_local = var_idx;
            break;
        case ebp_reg:
            ebp_local = var_idx;
            break;
        case ebx_reg:
            ebx_local = var_idx;
            break;
        default:
            break;
        }
		//reg_assigned_to_var[i] = var_idx;
		//
		// clear so that we won't pick it again
		//
		ref_count[var_idx] = 0;
	}
	return n_assigned;
}


////////////////////////////////////////////////////////////////////
///   Register_Allocator_Priority
////////////////////////////////////////////////////////////////////

unsigned Register_Allocator_Priority::var_in_reg(X86_Reg_No reg_no, const unsigned char *where)
{
    if (fg == NULL)
        return (unsigned)-1;
    else
        return fg->var_in_register(reg_no,where);
}

int Register_Allocator_Priority::var_is_live_on_entry(unsigned var)
{
    if (fg == NULL)
        return 0;
    else
        return fg->var_is_live_on_entry(var);
}

void Register_Allocator_Priority::set_call_site_info(Jit_Method_Info *new_mi, unsigned size)
{
    if (fg != NULL)
        fg->set_call_site_info(new_mi,size);
}

void Register_Allocator_Priority::set_register_saved_info(Jit_Method_Info *new_mi)
{
    if (fg == NULL)
        new_mi->registers_saved = 0;
    else
        new_mi->registers_saved = fg->callee_save_registers_used();
}

unsigned char Register_Allocator_Priority::registers_available_in_codegen_bb(const unsigned char *bc, int &hint)
{
    if (fg == NULL)
        return 0;
    else
        return fg->registers_available_in_codegen_bb(bc, hint);
}

void Register_Allocator_Priority::get_register_mapping_and_liveness(unsigned k,
                                                                    Call_Site_Info *csi,
                                                                    Jit_Method_Info *mi,
                                                                    X86_Reg_No &reg, int &live)
{
    if (k == (unsigned) csi->register_mapping[2])
    {
        reg = esi_reg;
        live = 1;
    }
    else if (k == (unsigned) csi->register_mapping[3])
    {
        reg = edi_reg;
        live = 1;
    }
    else if (k == (unsigned) csi->register_mapping[0])
    {
        reg = ebx_reg;
        live = 1;
    }
    else if (k == (unsigned) csi->register_mapping[1])
    {
        reg = ebp_reg;
        live = 1;
    }
    else if (mi->vars_register_allocated[k/8] & (1 << (k%8)))
    {
        reg = n_reg;
        live = 0;
    }
    else
    {
        reg = n_reg;
        live = 1;
    }
}

unsigned Register_Allocator_Priority::callee_saved_regs(int ebp_based)
{
    if (ebp_based)
    {
        return (callee_saved_esi_mask |
            callee_saved_edi_mask |
            callee_saved_ebx_mask);
    }
    else
        return fg->callee_save_registers_used();
}


unsigned Register_Allocator_Priority::register_allocate(class CG_Prepass *prepass,
                                                        Method_Handle methodHandle,
                                                        Class_Handle classHandle,
                                                        unsigned max_locals,
                                                        const unsigned char *bc,
                                                        unsigned code_length)
{
    unsigned result = 0;

    fg = new(*mm) Jcfg(bc,code_length,max_locals,*mm,methodHandle,classHandle);
    fg->register_allocate();
    if (method_get_num_handlers(methodHandle) != 0)
        return 3;
    unsigned char tmp = fg->callee_save_registers_used();
	if (tmp & (1 << esi_reg))
		result ++;
	if (tmp & (1 << edi_reg))
		result ++;
	if (tmp & (1 << ebx_reg))
		result ++;
	if (tmp & (1 << ebp_reg))
		result ++;

    return result;
}


int Register_Allocator_Priority::should_restore_callee(Jit_Method_Info *mi, X86_Reg_No reg)
{
    switch (reg)
    {
    case ebx_reg:
        return ((mi->registers_saved & (1 << ebx_reg)) || !mi->is_esp_based);
        break;
    case ebp_reg:
        return ((mi->registers_saved & (1 << ebp_reg)));
        break;
    case esi_reg:
        return ((mi->registers_saved & (1 << esi_reg)) || !mi->is_esp_based);
        break;
    case edi_reg:
        return ((mi->registers_saved & (1 << edi_reg)) || !mi->is_esp_based);
        break;
    default:
        return 0;
        break;
    }
}

void Register_Allocator_Priority::should_restore_ebp(Jit_Method_Info *method_info, int out_args,
                                                     int &restore, int &off)
{
    restore = (method_info->registers_saved & (1<<ebp_reg));
    if (restore)
    {
        off = 
            out_args + 
            ((method_info->registers_saved & (1 << edi_reg)) != 0) +
            ((method_info->registers_saved & (1 << esi_reg)) != 0);
    }
}

X86_Reg_No Register_Allocator_Priority::reg_var_allocated_to(int index,
        const unsigned char *where)
{
    if (fg == NULL)
        return n_reg;

    unsigned char result = fg->register_assigned_to_var(index);
    if (result & (1 << ebx_reg))
        return ebx_reg;
    if (result & (1 << ebp_reg))
        return ebp_reg;
    if (result & (1 << esi_reg))
        return esi_reg;
    if (result & (1 << edi_reg))
        return edi_reg;
    return n_reg;
}
