// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/jni_array.cpp,v 1.3 2001/11/20 20:09:22 rlhudson Exp $
//


#include "platform.h"
#include "Class.h"
#include "environment.h"
#include "orp_utils.h"
#include "orp_threads.h"
#include "root_set_enum.h"
#include "ini.h"


#include "jni.h"
#include "jni_direct.h"



jsize JNICALL GetArrayLength(JNIEnv *env,
                             jarray array)
{
    Object_Handle h = (Object_Handle)array;

    bool prev_gc_state = orp_disable_gc();  //-----------------v

    // All arrays have the length field at the same offset.
    JavaArrayOfChar *java_array = (JavaArrayOfChar *)h->java_reference;
    uint32 length= java_array->length;

    if(prev_gc_state) {
        orp_enable_gc();    //---------------------------------^
    }

    return length;
} //GetArrayLength



jarray JNICALL NewObjectArray(JNIEnv *env,
                              jsize length,
                              jclass elementClass,
                              jobject initialElement)
{
    Object_Handle elem_class_handle = (Object_Handle)elementClass;
    Class *clss = (Class *)elem_class_handle->java_reference;
    Object_Handle elem_handle = (Object_Handle)initialElement;
    Loader_Exception ld_exc;

    orp_disable_gc();       //---------------------------------v

    Class *arr_clss =
        resolve_class_array_of_class(ORP_Global_State::loader_env, clss, &ld_exc);

    if(!arr_clss) {
        orp_enable_gc();    //---------------------------------^

        return 0;
    }


    Object_Handle new_handle = orp_create_local_object_handle();
    JavaArrayOfObject *arr = orp_anewarray_resolved_array_type(arr_clss, length);
    Java_java_lang_Object *elem;
    if(elem_handle) {
        elem = elem_handle->java_reference;
    } else {
        elem = 0;
    }
    for(int i = 0; i < length; i++) {
        arr->body[i] = elem;
    }
    new_handle->java_reference = (Java_java_lang_Object *)arr;

    orp_enable_gc();        //---------------------------------^

    return new_handle;
} //NewObjectArray



jobject JNICALL GetObjectArrayElement(JNIEnv *env, jobjectArray array, jsize index)
{
    if(!array) {
        // This should not happen.
        assert(0);
        return 0;
    }

    jsize length = GetArrayLength(env, array);
    if(index < 0 || index >= length) {
        char msg[20];
        sprintf(msg, "%d", index);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return 0;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfObject *java_array = (JavaArrayOfObject *)h->java_reference;
    Java_java_lang_Object *elem = java_array->body[index];
    Object_Handle new_handle;
    if(elem) {
        new_handle = orp_create_local_object_handle();
        new_handle->java_reference = elem;
    } else {
        new_handle = 0;
    }

    orp_enable_gc();        //---------------------------------^

    return new_handle;
} //GetObjectArrayElement



void JNICALL SetObjectArrayElement(JNIEnv *env,
                                   jobjectArray array,
                                   jsize index,
                                   jobject value)
{
    if(!array) {
        // This should not happen.
        assert(0);
        return;
    }

    jsize length = GetArrayLength(env, array);
    if(index < 0 || index >= length) {
        char msg[20];
        sprintf(msg, "%d", index);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    if(value) {
        jclass array_jclass = GetObjectClass(env, array);
        Class *array_class = (Class *)((Object_Handle)array_jclass)->java_reference;
        jclass actual_element_jclass = GetObjectClass(env, value);
        Class *actual_element_class = (Class *)((Object_Handle)actual_element_jclass)->java_reference;

        DeleteLocalRef(env, array_jclass);
        DeleteLocalRef(env, actual_element_jclass);

        if(!orp_instanceof_class(actual_element_class, array_class->array_element_class)) {

            ThrowNew_Quick(env, "java/lang/ArrayStoreException", actual_element_class->name->bytes);
            return;
        }
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfObject *java_array = (JavaArrayOfObject *)h->java_reference;
    Java_java_lang_Object *elem;
    if(value) {
        Object_Handle vh = (Object_Handle)value;
        elem = vh->java_reference;
    } else {
        elem = 0;
    }

//    java_array->body[index] = elem;
    gc_heap_slot_write_ref ((Java_java_lang_Object *)java_array,
        (Java_java_lang_Object **)&(java_array->body[index]),
        (Java_java_lang_Object *)elem);

    orp_enable_gc();        //---------------------------------^
} //SetObjectArrayElement




/////////////////////////////////////////////////////////////////////////////
// begin New<Type>Array functions



jbooleanArray JNICALL NewBooleanArray(JNIEnv *env, jsize length)
{
    Object_Handle h = orp_create_local_object_handle();
    Class *clss = ORP_Global_State::loader_env->ArrayOfBoolean_Class;
    
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof (J_Boolean);
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif 
    orp_disable_gc();       //---------------------------------v
#ifdef GC_REWORK
    JavaArrayOfBoolean *array = (JavaArrayOfBoolean *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfBoolean *array = (JavaArrayOfBoolean *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
//    array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)array,
        &(array->length),
        length);

    h->java_reference = (Java_java_lang_Object *)array;

    orp_enable_gc();        //---------------------------------^

    return h;
} //NewBooleanArray



jbyteArray JNICALL NewByteArray(JNIEnv *env, jsize length)
{
    Object_Handle h = orp_create_local_object_handle();
    Class *clss = ORP_Global_State::loader_env->ArrayOfByte_Class;
 
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof (J_Byte);
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif 
    orp_disable_gc();       //---------------------------------v
#ifdef GC_REWORK
    JavaArrayOfByte *array = (JavaArrayOfByte *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfByte *array = (JavaArrayOfByte *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
    //    array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)array,
        &(array->length),
        length);
    h->java_reference = (Java_java_lang_Object *)array;

    orp_enable_gc();        //---------------------------------^

    return h;
} //NewByteArray



jcharArray JNICALL NewCharArray(JNIEnv *env, jsize length)
{
    Object_Handle h = orp_create_local_object_handle();
    Class *clss = ORP_Global_State::loader_env->ArrayOfChar_Class;
 
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof(J_Char); // sizeof J_Char == 2;
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif 
    orp_disable_gc();       //---------------------------------v
#ifdef GC_REWORK
    JavaArrayOfChar *array = (JavaArrayOfChar *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfChar *array = (JavaArrayOfChar *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
//    array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)array,
        &(array->length),
        length);

    h->java_reference = (Java_java_lang_Object *)array;

    orp_enable_gc();        //---------------------------------^

    return h;
} //NewCharArray



jshortArray JNICALL NewShortArray(JNIEnv *env, jsize length)
{
    Object_Handle h = orp_create_local_object_handle();
    Class *clss = ORP_Global_State::loader_env->ArrayOfShort_Class;
 
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof (J_Short); // 2;
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif 
    orp_disable_gc();       //---------------------------------v

#ifdef GC_REWORK
    JavaArrayOfShort *array = (JavaArrayOfShort *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfShort *array = (JavaArrayOfShort *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
//    array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)array,
        &(array->length),
        length);

    h->java_reference = (Java_java_lang_Object *)array;

    orp_enable_gc();        //---------------------------------^

    return h;
} //NewShortArray



jintArray JNICALL NewIntArray(JNIEnv *env, jsize length)
{
    Object_Handle h = orp_create_local_object_handle();
    Class *clss = ORP_Global_State::loader_env->ArrayOfInt_Class;
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof(J_Int);
 
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif 
    orp_disable_gc();       //---------------------------------v
#ifdef GC_REWORK
    JavaArrayOfInt *array = (JavaArrayOfInt *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfInt *array = (JavaArrayOfInt *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
//    array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)array,
        &(array->length),
        length);

    h->java_reference = (Java_java_lang_Object *)array;

    orp_enable_gc();        //---------------------------------^

    return h;
} //NewIntArray



jlongArray JNICALL NewLongArray(JNIEnv *env, jsize length)
{
    Object_Handle h = orp_create_local_object_handle();
    Class *clss = ORP_Global_State::loader_env->ArrayOfLong_Class;
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof(J_Long); // 9
 
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif 
    orp_disable_gc();       //---------------------------------v
#ifdef GC_REWORK
    JavaArrayOfLong *array = (JavaArrayOfLong *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfLong *array = (JavaArrayOfLong *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
//    array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)array,
        &(array->length),
        length);

    h->java_reference = (Java_java_lang_Object *)array;

    orp_enable_gc();        //---------------------------------^

    return h;
} //NewLongArray



jfloatArray JNICALL NewFloatArray(JNIEnv *env, jsize length)
{
    Object_Handle h = orp_create_local_object_handle();
    Class *clss = ORP_Global_State::loader_env->ArrayOfFloat_Class;
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof(J_Float);
 
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif 
    orp_disable_gc();       //---------------------------------v
#ifdef GC_REWORK
    JavaArrayOfFloat *array = (JavaArrayOfFloat *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfFloat *array = (JavaArrayOfFloat *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
//    array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)array,
        &(array->length),
        length);

    h->java_reference = (Java_java_lang_Object *)array;

    orp_enable_gc();        //---------------------------------^

    return h;
} //NewFloatArray



jdoubleArray JNICALL NewDoubleArray(JNIEnv *env, jsize length)
{
    Object_Handle h = orp_create_local_object_handle();
    Class *clss = ORP_Global_State::loader_env->ArrayOfDouble_Class;
#if 1
    unsigned sz = orp_array_size(clss, length);
#else
    unsigned sz = sizeof(JavaArray) + (length) * sizeof (J_Double);
 
    sz = sz + OBJECT_HEADER_SIZE;
    // align the object upwards
    sz = (((sz + (GC_OBJECT_ALIGNMENT - 1)) 
          / GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT);
#endif 
    orp_disable_gc();       //---------------------------------v
#ifdef GC_REWORK
    JavaArrayOfDouble *array = (JavaArrayOfDouble *)gc_malloc(sz, (VTable *)clss->vtable);
#else
    JavaArrayOfDouble *array = (JavaArrayOfDouble *)gc_malloc(sz, (Partial_Reveal_VTable *)clss->vtable);
#endif
//    array->length = length;
    gc_heap_slot_write_int32 ((Java_java_lang_Object *)array,
        &(array->length),
        length);

    h->java_reference = (Java_java_lang_Object *)array;

    orp_enable_gc();        //---------------------------------^

    return h;
} //NewDoubleArray


// end New<Type>Array functions
/////////////////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////////////
// begin Get<Type>ArrayElements functions


jboolean *JNICALL GetBooleanArrayElements(JNIEnv *env,
                                          jbooleanArray array,
                                          jboolean *isCopy)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v
    JavaArrayOfBoolean *java_array = (JavaArrayOfBoolean *)h->java_reference;
    assert(java_array->vt->clss == ORP_Global_State::loader_env->ArrayOfBoolean_Class);
    bool is_pinned = gc_is_object_pinned((Java_java_lang_Object *)java_array);
    int length= java_array->length;
    orp_enable_gc();        //---------------------------------^

    if(is_pinned) {
        // No copy needed.
        if(isCopy) {
            *isCopy = JNI_FALSE;
        }
        JavaArrayOfBoolean *java_array = (JavaArrayOfBoolean *)h->java_reference;
        return (jboolean *)&(java_array->body[0]);
    } else {
        jboolean *primitive_array = (jboolean *)orp_malloc_gc_safe(sizeof(jboolean) * length);

        orp_disable_gc();       //---------------------------------v

        JavaArrayOfBoolean *java_array = (JavaArrayOfBoolean *)h->java_reference;
        memcpy(primitive_array, &(java_array->body[0]), sizeof(jboolean) * length);

        orp_enable_gc();        //---------------------------------^

        if(isCopy) {
            *isCopy = JNI_TRUE;
        }
        return primitive_array;
    }
} //GetBooleanArrayElements



jbyte *JNICALL GetByteArrayElements(JNIEnv *env,
                                    jbyteArray array,
                                    jboolean *isCopy)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v
    JavaArrayOfByte *java_array = (JavaArrayOfByte *)h->java_reference;
    bool is_pinned = gc_is_object_pinned((Java_java_lang_Object *)java_array);
    int length= java_array->length;
    orp_enable_gc();        //---------------------------------^

    if(is_pinned) {
        // No copy needed.
        if(isCopy) {
            *isCopy = JNI_FALSE;
        }
        JavaArrayOfByte *java_array = (JavaArrayOfByte *)h->java_reference;
        return (jbyte *)&(java_array->body[0]);
    } else {
        jbyte *primitive_array = (jbyte *)orp_malloc_gc_safe(sizeof(jbyte) * length);

        orp_disable_gc();       //---------------------------------v

        JavaArrayOfByte *java_array = (JavaArrayOfByte *)h->java_reference;
        memcpy(primitive_array, &(java_array->body[0]), sizeof(jbyte) * length);

        orp_enable_gc();        //---------------------------------^

        if(isCopy) {
            *isCopy = JNI_TRUE;
        }
        return primitive_array;
    }
} //GetByteArrayElements



jchar *JNICALL GetCharArrayElements(JNIEnv *env,
                                    jcharArray array,
                                    jboolean *isCopy)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v
    JavaArrayOfChar *java_array = (JavaArrayOfChar *)h->java_reference;
    bool is_pinned = gc_is_object_pinned((Java_java_lang_Object *)java_array);
    int length= java_array->length;
    orp_enable_gc();        //---------------------------------^

    if(is_pinned) {
        // No copy needed.
        if(isCopy) {
            *isCopy = JNI_FALSE;
        }
        JavaArrayOfChar *java_array = (JavaArrayOfChar *)h->java_reference;
        return (jchar *)&(java_array->body[0]);
    } else {
        jchar *primitive_array = (jchar *)orp_malloc_gc_safe(sizeof(jchar) * length);

        orp_disable_gc();       //---------------------------------v

        JavaArrayOfChar *java_array = (JavaArrayOfChar *)h->java_reference;
        memcpy(primitive_array, &(java_array->body[0]), sizeof(jchar) * length);

        orp_enable_gc();        //---------------------------------^

        if(isCopy) {
            *isCopy = JNI_TRUE;
        }
        return primitive_array;
    }
} //GetCharArrayElements



jshort *JNICALL GetShortArrayElements(JNIEnv *env,
                                      jshortArray array,
                                      jboolean *isCopy)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v
    JavaArrayOfShort *java_array = (JavaArrayOfShort *)h->java_reference;
    bool is_pinned = gc_is_object_pinned((Java_java_lang_Object *)java_array);
    int length= java_array->length;
    orp_enable_gc();        //---------------------------------^

    if(is_pinned) {
        // No copy needed.
        JavaArrayOfShort *java_array = (JavaArrayOfShort *)h->java_reference;
        if(isCopy) {
            *isCopy = JNI_FALSE;
        }
        return (jshort *)&(java_array->body[0]);
    } else {
        jshort *primitive_array = (jshort *)orp_malloc_gc_safe(sizeof(jshort) * length);

        orp_disable_gc();       //---------------------------------v

        JavaArrayOfShort *java_array = (JavaArrayOfShort *)h->java_reference;
        memcpy(primitive_array, &(java_array->body[0]), sizeof(jshort) * length);

        orp_enable_gc();        //---------------------------------^

        if(isCopy) {
            *isCopy = JNI_TRUE;
        }
        return primitive_array;
    }
} //GetShortArrayElements



jint *JNICALL GetIntArrayElements(JNIEnv *env,
                                  jintArray array,
                                  jboolean *isCopy)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v
    JavaArrayOfInt *java_array = (JavaArrayOfInt *)h->java_reference;
    bool is_pinned = gc_is_object_pinned((Java_java_lang_Object *)java_array);
    int length= java_array->length;
    orp_enable_gc();        //---------------------------------^

    if(is_pinned) {
        // No copy needed.
        if(isCopy) {
            *isCopy = JNI_FALSE;
        }
        JavaArrayOfInt *java_array = (JavaArrayOfInt *)h->java_reference;
        return (jint *)&(java_array->body[0]);
    } else {
        jsize length = GetArrayLength(env, array);

        jint *primitive_array = (jint *)orp_malloc_gc_safe(sizeof(jint) * length);

        orp_disable_gc();       //---------------------------------v

        JavaArrayOfInt *java_array = (JavaArrayOfInt *)h->java_reference;
        memcpy(primitive_array, &(java_array->body[0]), sizeof(jint) * length);

        orp_enable_gc();        //---------------------------------^

        if(isCopy) {
            *isCopy = JNI_TRUE;
        }
        return primitive_array;
    }
} //GetIntArrayElements



jlong *JNICALL GetLongArrayElements(JNIEnv *env,
                                    jlongArray array,
                                    jboolean *isCopy)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v
    JavaArrayOfLong *java_array = (JavaArrayOfLong *)h->java_reference;
    bool is_pinned = gc_is_object_pinned((Java_java_lang_Object *)java_array);
    int length= java_array->length;
    orp_enable_gc();        //---------------------------------^

    if(is_pinned) {
        // No copy needed.
        if(isCopy) {
            *isCopy = JNI_FALSE;
        }
        JavaArrayOfLong *java_array = (JavaArrayOfLong *)h->java_reference;
        return (jlong *)&(java_array->body[0]);
    } else {
        jlong *primitive_array = (jlong *)orp_malloc_gc_safe(sizeof(jlong) * length);

        orp_disable_gc();       //---------------------------------v

        JavaArrayOfLong *java_array = (JavaArrayOfLong *)h->java_reference;
        memcpy(primitive_array, &(java_array->body[0]), sizeof(jlong) * length);

        orp_enable_gc();        //---------------------------------^

        if(isCopy) {
            *isCopy = JNI_TRUE;
        }
        return primitive_array;
    }
} //GetLongArrayElements



jfloat *JNICALL GetFloatArrayElements(JNIEnv *env,
                                      jfloatArray array,
                                      jboolean *isCopy)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v
    JavaArrayOfFloat *java_array = (JavaArrayOfFloat *)h->java_reference;
    bool is_pinned = gc_is_object_pinned((Java_java_lang_Object *)java_array);
    int length= java_array->length;
    orp_enable_gc();        //---------------------------------^

    if(is_pinned) {
        // No copy needed.
        if(isCopy) {
            *isCopy = JNI_FALSE;
        }
        JavaArrayOfFloat *java_array = (JavaArrayOfFloat *)h->java_reference;
        return (jfloat *)&(java_array->body[0]);
    } else {
        jsize length = GetArrayLength(env, array);

        jfloat *primitive_array = (jfloat *)orp_malloc_gc_safe(sizeof(jfloat) * length);

        orp_disable_gc();       //---------------------------------v

        JavaArrayOfFloat *java_array = (JavaArrayOfFloat *)h->java_reference;
        memcpy(primitive_array, &(java_array->body[0]), sizeof(jfloat) * length);

        orp_enable_gc();        //---------------------------------^

        if(isCopy) {
            *isCopy = JNI_TRUE;
        }
        return primitive_array;
    }
} //GetFloatArrayElements



jdouble *JNICALL GetDoubleArrayElements(JNIEnv *env,
                                        jdoubleArray array,
                                        jboolean *isCopy)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v
    JavaArrayOfDouble *java_array = (JavaArrayOfDouble *)h->java_reference;
    bool is_pinned = gc_is_object_pinned((Java_java_lang_Object *)java_array);
    int length= java_array->length;
    orp_enable_gc();        //---------------------------------^

    if(is_pinned) {
        // No copy needed.
        if(isCopy) {
            *isCopy = JNI_FALSE;
        }
        JavaArrayOfDouble *java_array = (JavaArrayOfDouble *)h->java_reference;
        return (jdouble *)&(java_array->body[0]);
    } else {
        jdouble *primitive_array = (jdouble *)orp_malloc_gc_safe(sizeof(jdouble) * length);

        orp_disable_gc();       //---------------------------------v

        JavaArrayOfDouble *java_array = (JavaArrayOfDouble *)h->java_reference;
        memcpy(primitive_array, &(java_array->body[0]), sizeof(jdouble) * length);

        orp_enable_gc();        //---------------------------------^

        if(isCopy) {
            *isCopy = JNI_TRUE;
        }
        return primitive_array;
    }
} //GetDoubleArrayElements



// end Get<Type>ArrayElements functions
/////////////////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////////////
// begin Release<Type>ArrayElements functions


void JNICALL ReleaseBooleanArrayElements(JNIEnv *env,
                                         jbooleanArray array,
                                         jboolean *elems,
                                         jint mode)
{
    Object_Handle h = (Object_Handle)array;
    orp_disable_gc();       //---------------------------------v

    JavaArrayOfBoolean *java_array = (JavaArrayOfBoolean *)h->java_reference;
    jboolean *elems_in_array = (jboolean *)&(java_array->body[0]);
    bool no_copy = (elems_in_array == elems);

    orp_enable_gc();        //---------------------------------^

    if(no_copy) {
        // Nothing to do.
        return;
    }

    switch(mode) {
    case 0:
    case JNI_COMMIT:
        {
            orp_disable_gc();       //---------------------------------v

            JavaArrayOfBoolean *java_array = (JavaArrayOfBoolean *)h->java_reference;
            int length = java_array->length;
            jboolean *elems_in_array = (jboolean *)&(java_array->body[0]);
            memcpy(elems_in_array, elems, sizeof(jboolean) * length);
            gc_heap_wrote_object((Java_java_lang_Object *)java_array);

            orp_enable_gc();        //---------------------------------^

            if(mode == 0) {
                orp_free_gc_safe(elems);
            }
        }
        break;
    case JNI_ABORT:
        orp_free_gc_safe(elems);
        break;
    default:
        assert(0);
        break;
    }
} //ReleaseBooleanArrayElements



void JNICALL ReleaseByteArrayElements(JNIEnv *env,
                                      jbyteArray array,
                                      jbyte *elems,
                                      jint mode)
{
    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfByte *java_array = (JavaArrayOfByte *)h->java_reference;
    jbyte *elems_in_array = (jbyte *)&(java_array->body[0]);
    bool no_copy = (elems_in_array == elems);

    orp_enable_gc();        //---------------------------------^

    if(no_copy) {
        // Nothing to do.
        return;
    }

    switch(mode) {
    case 0:
    case JNI_COMMIT:
        {
            jsize length = GetArrayLength(env, array);
            //Object_Handle h = (Object_Handle)array;
            orp_disable_gc();       //---------------------------------v

            JavaArrayOfByte *java_array = (JavaArrayOfByte *)h->java_reference;
            memcpy(&(java_array->body[0]), elems, sizeof(jbyte) * length);
            gc_heap_wrote_object ((Java_java_lang_Object *)java_array);
            orp_enable_gc();        //---------------------------------^

            if(mode == 0) {
                orp_free_gc_safe(elems);
            }
        }
        break;
    case JNI_ABORT:
        orp_free_gc_safe(elems);
        break;
    default:
        assert(0);
        break;
    }
} //ReleaseByteArrayElements



void JNICALL ReleaseCharArrayElements(JNIEnv *env,
                                      jcharArray array,
                                      jchar *elems,
                                      jint mode)
{
    Object_Handle h = (Object_Handle)array;
    orp_disable_gc();       //---------------------------------v

    JavaArrayOfChar *java_array = (JavaArrayOfChar *)h->java_reference;
    jchar *elems_in_array = (jchar *)&(java_array->body[0]);
    bool no_copy = (elems_in_array == elems);

    orp_enable_gc();        //---------------------------------^

    if(no_copy) {
        // Nothing to do.
        return;
    }

    switch(mode) {
    case 0:
    case JNI_COMMIT:
        {
            jsize length = GetArrayLength(env, array);
            Object_Handle h = (Object_Handle)array;
            orp_disable_gc();       //---------------------------------v

            JavaArrayOfChar *java_array = (JavaArrayOfChar *)h->java_reference;
            memcpy(&(java_array->body[0]), elems, sizeof(jchar) * length);
            gc_heap_wrote_object ((Java_java_lang_Object *)java_array);
            orp_enable_gc();        //---------------------------------^

            if(mode == 0) {
                orp_free_gc_safe(elems);
            }
        }
        break;
    case JNI_ABORT:
        orp_free_gc_safe(elems);
        break;
    default:
        assert(0);
        break;
    }
} //ReleaseCharArrayElements



void JNICALL ReleaseShortArrayElements(JNIEnv *env,
                                       jshortArray array,
                                       jshort *elems,
                                       jint mode)
{
    Object_Handle h = (Object_Handle)array;
    orp_disable_gc();       //---------------------------------v

    JavaArrayOfShort *java_array = (JavaArrayOfShort *)h->java_reference;
    jshort *elems_in_array = (jshort *)&(java_array->body[0]);
    bool no_copy = (elems_in_array == elems);

    orp_enable_gc();        //---------------------------------^

    if(no_copy) {
        // Nothing to do.
        return;
    }

    switch(mode) {
    case 0:
    case JNI_COMMIT:
        {
            jsize length = GetArrayLength(env, array);
            Object_Handle h = (Object_Handle)array;
            orp_disable_gc();       //---------------------------------v

            JavaArrayOfShort *java_array = (JavaArrayOfShort *)h->java_reference;
            memcpy(&(java_array->body[0]), elems, sizeof(jshort) * length);
            gc_heap_wrote_object ((Java_java_lang_Object *)java_array);
            orp_enable_gc();        //---------------------------------^

            if(mode == 0) {
                orp_free_gc_safe(elems);
            }
        }
        break;
    case JNI_ABORT:
        orp_free_gc_safe(elems);
        break;
    default:
        assert(0);
        break;
    }
} //ReleaseShortArrayElements



void JNICALL ReleaseIntArrayElements(JNIEnv *env,
                                     jintArray array,
                                     jint *elems,
                                     jint mode)
{
    Object_Handle h = (Object_Handle)array;
    orp_disable_gc();       //---------------------------------v

    JavaArrayOfInt *java_array = (JavaArrayOfInt *)h->java_reference;
    jint *elems_in_array = (jint *)&(java_array->body[0]);
    bool no_copy = (elems_in_array == elems);

    orp_enable_gc();        //---------------------------------^

    if(no_copy) {
        // Nothing to do.
        return;
    }

    switch(mode) {
    case 0:
    case JNI_COMMIT:
        {
            jsize length = GetArrayLength(env, array);
            Object_Handle h = (Object_Handle)array;
            orp_disable_gc();       //---------------------------------v

            JavaArrayOfInt *java_array = (JavaArrayOfInt *)h->java_reference;
            memcpy(&(java_array->body[0]), elems, sizeof(jint) * length);
            gc_heap_wrote_object ((Java_java_lang_Object *)java_array);
            orp_enable_gc();        //---------------------------------^

            if(mode == 0) {
                orp_free_gc_safe(elems);
            }
        }
        break;
    case JNI_ABORT:
        orp_free_gc_safe(elems);
        break;
    default:
        assert(0);
        break;
    }
} //ReleaseIntArrayElements



void JNICALL ReleaseLongArrayElements(JNIEnv *env,
                                      jlongArray array,
                                      jlong *elems,
                                      jint mode)
{
    Object_Handle h = (Object_Handle)array;
    orp_disable_gc();       //---------------------------------v

    JavaArrayOfLong *java_array = (JavaArrayOfLong *)h->java_reference;
    jlong *elems_in_array = (jlong *)&(java_array->body[0]);
    bool no_copy = (elems_in_array == elems);

    orp_enable_gc();        //---------------------------------^

    if(no_copy) {
        // Nothing to do.
        return;
    }

    switch(mode) {
    case 0:
    case JNI_COMMIT:
        {
            jsize length = GetArrayLength(env, array);
            Object_Handle h = (Object_Handle)array;
            orp_disable_gc();       //---------------------------------v

            JavaArrayOfLong *java_array = (JavaArrayOfLong *)h->java_reference;
            memcpy(&(java_array->body[0]), elems, sizeof(jlong) * length);
            gc_heap_wrote_object ((Java_java_lang_Object *)java_array);
            orp_enable_gc();        //---------------------------------^

            if(mode == 0) {
                orp_free_gc_safe(elems);
            }
        }
        break;
    case JNI_ABORT:
        orp_free_gc_safe(elems);
        break;
    default:
        assert(0);
        break;
    }
} //ReleaseLongArrayElements



void JNICALL ReleaseFloatArrayElements(JNIEnv *env,
                                       jfloatArray array,
                                       jfloat *elems,
                                       jint mode)
{
    Object_Handle h = (Object_Handle)array;
    orp_disable_gc();       //---------------------------------v

    JavaArrayOfFloat *java_array = (JavaArrayOfFloat *)h->java_reference;
    jfloat *elems_in_array = (jfloat *)&(java_array->body[0]);
    bool no_copy = (elems_in_array == elems);

    orp_enable_gc();        //---------------------------------^

    if(no_copy) {
        // Nothing to do.
        return;
    }

    switch(mode) {
    case 0:
    case JNI_COMMIT:
        {
            jsize length = GetArrayLength(env, array);
            Object_Handle h = (Object_Handle)array;
            orp_disable_gc();       //---------------------------------v

            JavaArrayOfFloat *java_array = (JavaArrayOfFloat *)h->java_reference;
            memcpy(&(java_array->body[0]), elems, sizeof(jfloat) * length);

            gc_heap_wrote_object ((Java_java_lang_Object *)java_array);
            orp_enable_gc();        //---------------------------------^

            if(mode == 0) {
                orp_free_gc_safe(elems);
            }
        }
        break;
    case JNI_ABORT:
        orp_free_gc_safe(elems);
        break;
    default:
        assert(0);
        break;
    }
} //ReleaseFloatArrayElements



void JNICALL ReleaseDoubleArrayElements(JNIEnv *env,
                                        jdoubleArray array,
                                        jdouble *elems,
                                        jint mode)
{
    Object_Handle h = (Object_Handle)array;
    orp_disable_gc();       //---------------------------------v

    JavaArrayOfDouble *java_array = (JavaArrayOfDouble *)h->java_reference;
    jdouble *elems_in_array = (jdouble *)&(java_array->body[0]);
    bool no_copy = (elems_in_array == elems);

    orp_enable_gc();        //---------------------------------^

    if(no_copy) {
        // Nothing to do.
        return;
    }

    switch(mode) {
    case 0:
    case JNI_COMMIT:
        {
            jsize length = GetArrayLength(env, array);
            Object_Handle h = (Object_Handle)array;
            orp_disable_gc();       //---------------------------------v

            JavaArrayOfDouble *java_array = (JavaArrayOfDouble *)h->java_reference;
            memcpy(&(java_array->body[0]), elems, sizeof(jdouble) * length);

            orp_enable_gc();        //---------------------------------^

            if(mode == 0) {
                orp_free_gc_safe(elems);
            }
        }
        break;
    case JNI_ABORT:
        orp_free_gc_safe(elems);
        break;
    default:
        assert(0);
        break;
    }
} //ReleaseDoubleArrayElements



// end Release<Type>ArrayElements functions
/////////////////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////////////
// begin Get<Type>ArrayRegion functions


void JNICALL GetBooleanArrayRegion (JNIEnv *env,
                                    jobjectArray array,
                                    jsize start,
                                    jsize len,
                                    jboolean *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfBoolean *java_array = (JavaArrayOfBoolean *)h->java_reference;
    memcpy(buf, &(java_array->body[start]), sizeof(jboolean) * len);

    orp_enable_gc();        //---------------------------------^
} //GetBooleanArrayRegion



void JNICALL GetByteArrayRegion    (JNIEnv *env,
                                    jobjectArray array,
                                    jsize start,
                                    jsize len,
                                    jbyte *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    // All arrays have the length field at the same offset.
    JavaArrayOfByte *java_array = (JavaArrayOfByte *)h->java_reference;
    memcpy(buf, &(java_array->body[start]), sizeof(jbyte) * len);

    orp_enable_gc();        //---------------------------------^
} //GetByteArrayRegion



void JNICALL GetCharArrayRegion(JNIEnv *env,
                                jobjectArray array,
                                jsize start,
                                jsize len,
                                jchar *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfChar *java_array = (JavaArrayOfChar *)h->java_reference;
    memcpy(buf, &(java_array->body[start]), sizeof(jchar) * len);

    orp_enable_gc();        //---------------------------------^
} //GetCharArrayRegion



void JNICALL GetShortArrayRegion(JNIEnv *env,
                                 jobjectArray array, 
                                 jsize start,
                                 jsize len,
                                 jshort *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfShort *java_array = (JavaArrayOfShort *)h->java_reference;
    memcpy(buf, &(java_array->body[start]), sizeof(jshort) * len);

    orp_enable_gc();        //---------------------------------^
} //GetShortArrayRegion



void JNICALL GetIntArrayRegion(JNIEnv *env,
                               jobjectArray array,
                               jsize start,
                               jsize len,
                               jint *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfInt *java_array = (JavaArrayOfInt *)h->java_reference;
    memcpy(buf, &(java_array->body[start]), sizeof(jint) * len);

    orp_enable_gc();        //---------------------------------^
} //GetIntArrayRegion



void JNICALL GetLongArrayRegion(JNIEnv *env,
                                jobjectArray array,
                                jsize start,
                                jsize len,
                                jlong *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfLong *java_array = (JavaArrayOfLong *)h->java_reference;
    memcpy(buf, &(java_array->body[start]), sizeof(jlong) * len);

    orp_enable_gc();        //---------------------------------^
}  //GetLongArrayRegion



void JNICALL GetFloatArrayRegion(JNIEnv *env,
                                 jobjectArray array,
                                 jsize start,
                                 jsize len,
                                 jfloat *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfFloat *java_array = (JavaArrayOfFloat *)h->java_reference;
    memcpy(buf, &(java_array->body[start]), sizeof(jfloat) * len);

    orp_enable_gc();        //---------------------------------^
} //GetFloatArrayRegion



void JNICALL GetDoubleArrayRegion(JNIEnv *env,
                                  jobjectArray array,
                                  jsize start,
                                  jsize len,
                                  jdouble *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfDouble *java_array = (JavaArrayOfDouble *)h->java_reference;
    memcpy(buf, &(java_array->body[start]), sizeof(jdouble) * len);

    orp_enable_gc();        //---------------------------------^
} //GetDoubleArrayRegion


// end Get<Type>ArrayRegion functions
/////////////////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////////////////////////////
// begin Set<Type>ArrayRegion functions


void JNICALL SetBooleanArrayRegion(JNIEnv *env,
                                   jobjectArray array,
                                   jsize start,
                                   jsize len,
                                   jboolean *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfBoolean *java_array = (JavaArrayOfBoolean *)h->java_reference;
    memcpy(&(java_array->body[start]), buf, sizeof(jboolean) * len);
    
    gc_heap_wrote_object ((Java_java_lang_Object *)java_array);

    orp_enable_gc();        //---------------------------------^
} //SetBooleanArrayRegion



void JNICALL SetByteArrayRegion(JNIEnv *env,
                                jobjectArray array,
                                jsize start,
                                jsize len,
                                jbyte *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfByte *java_array = (JavaArrayOfByte *)h->java_reference;
    memcpy(&(java_array->body[start]), buf, sizeof(jbyte) * len);

    gc_heap_wrote_object ((Java_java_lang_Object *)java_array);

    orp_enable_gc();        //---------------------------------^
} //SetByteArrayRegion



void JNICALL SetCharArrayRegion(JNIEnv *env,
                                jobjectArray array,
                                jsize start,
                                jsize len,
                                jchar *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfChar *java_array = (JavaArrayOfChar *)h->java_reference;
    memcpy(&(java_array->body[start]), buf, sizeof(jchar) * len);

    gc_heap_wrote_object ((Java_java_lang_Object *)java_array);

    orp_enable_gc();        //---------------------------------^
} //SetCharArrayRegion



void JNICALL SetShortArrayRegion(JNIEnv *env,
                                 jobjectArray array,
                                 jsize start,
                                 jsize len,
                                 jshort *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfShort *java_array = (JavaArrayOfShort *)h->java_reference;
    memcpy(&(java_array->body[start]), buf, sizeof(jshort) * len);

    gc_heap_wrote_object ((Java_java_lang_Object *)java_array);

    orp_enable_gc();        //---------------------------------^
} //SetShortArrayRegion



void JNICALL SetIntArrayRegion(JNIEnv *env,
                               jobjectArray array,
                               jsize start,
                               jsize len,
                               jint *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfInt *java_array = (JavaArrayOfInt *)h->java_reference;
    memcpy(&(java_array->body[start]), buf, sizeof(jint) * len);

    gc_heap_wrote_object ((Java_java_lang_Object *)java_array);

    orp_enable_gc();        //---------------------------------^
} //SetIntArrayRegion



void JNICALL SetLongArrayRegion(JNIEnv *env,
                                jobjectArray array,
                                jsize start,
                                jsize len,
                                jlong *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfLong *java_array = (JavaArrayOfLong *)h->java_reference;
    memcpy(&(java_array->body[start]), buf, sizeof(jlong) * len);

    gc_heap_wrote_object ((Java_java_lang_Object *)java_array);

    orp_enable_gc();        //---------------------------------^
} //SetLongArrayRegion



void JNICALL SetFloatArrayRegion(JNIEnv *env,
                                 jobjectArray array,
                                 jsize start,
                                 jsize len,
                                 jfloat *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfFloat *java_array = (JavaArrayOfFloat *)h->java_reference;
    memcpy(&(java_array->body[start]), buf, sizeof(jfloat) * len);

    gc_heap_wrote_object ((Java_java_lang_Object *)java_array);

    orp_enable_gc();        //---------------------------------^
} //SetFloatArrayRegion



void JNICALL SetDoubleArrayRegion(JNIEnv *env,
                                  jobjectArray array,
                                  jsize start,
                                  jsize len,
                                  jdouble *buf)
{
    assert(len >= 0);
    jsize length = GetArrayLength(env, array);
    jsize end = start + len;
    if(start < 0 || start >= length || end < 0 || end > length) {
        char msg[30];
        sprintf(msg, "%d..%d", start, end);
        ThrowNew_Quick(env, "java/lang/ArrayIndexOutOfBoundsException", msg);
        return;
    }

    Object_Handle h = (Object_Handle)array;

    orp_disable_gc();       //---------------------------------v

    JavaArrayOfDouble *java_array = (JavaArrayOfDouble *)h->java_reference;
    memcpy(&(java_array->body[start]), buf, sizeof(jdouble) * len);

    gc_heap_wrote_object ((Java_java_lang_Object *)java_array);

    orp_enable_gc();        //---------------------------------^
} //SetDoubleArrayRegion



// end Set<Type>ArrayRegion functions
/////////////////////////////////////////////////////////////////////////////


