// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/base/mem_alloc.cpp,v 1.1.1.1 2001/07/23 07:25:38 xli18 Exp $
//



#include "platform.h"
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>


#include "orp_utils.h"
#include "nogc.h"


static void check_for_out_of_memory(void *p)
{
    if(!p) {
        printf("Out of memory\n");
        orp_exit(1);
    }
} //check_for_out_of_memory



////////////////////////////////////////////////////////////
// begin allocating memory for code


#ifdef ORP_NT
static unsigned page_size = 0;
#endif
static unsigned initial_code_pool_size = 8192;


//
// We have to separate memory pools to avoid an extra lock.
// Class loading and JIT compilation are serialized,
// so memory allocation doesn't need to if we guarantee
// a separate pool for each of those operations which
// are serialized by other means.
//
static Byte *memory_pool_for_class_loading_start = 0;
static Byte *memory_pool_for_class_loading_end;
static Byte *memory_pool_for_jit_start = 0;
static Byte *memory_pool_for_jit_end;


static void allocate_code_pool(Byte **p_pool_start, Byte **p_pool_end, unsigned size)
{
#ifdef ORP_NT
    if(size % page_size) {
        unsigned num_pages = size / page_size;
        size = (num_pages + 1) * page_size;
    }
    Byte *pool_start = (Byte *)VirtualAlloc(0, size, MEM_COMMIT, PAGE_EXECUTE_READWRITE);
    assert(!(15 & (int)pool_start));
#else
    uint64 pool_start_i = (uint64)malloc(size);
    pool_start_i = (pool_start_i + 15) & ~15;
    Byte *pool_start = (Byte *)pool_start_i;
#endif

    *p_pool_start = pool_start;
    check_for_out_of_memory(*p_pool_start);
    *p_pool_end = *p_pool_start + size;
} //allocate_code_pool



static void *malloc_fixed_code_from_pool(Byte **p_pool_start, Byte **p_pool_end, unsigned size)
{
    // align size to 16
    size = (size + 15) & ~15;

    Byte *pool_start = *p_pool_start;
    Byte *pool_end = *p_pool_end;
    unsigned mem_in_pool = pool_end - pool_start;
    if(mem_in_pool < size) {
        unsigned new_pool_size = size > initial_code_pool_size ? size : initial_code_pool_size;
        allocate_code_pool(p_pool_start, p_pool_end, new_pool_size);
        pool_start = *p_pool_start;
        pool_end = *p_pool_end;
    }
    void *p = pool_start;
    *p_pool_start = pool_start + size;
    return p;
} //malloc_fixed_code_from_pool




void *gc_malloc_fixed_code_for_class_loading(unsigned size)
{
    assert(memory_pool_for_class_loading_start);
    void *p = malloc_fixed_code_from_pool(&memory_pool_for_class_loading_start,
                                          &memory_pool_for_class_loading_end,
                                          size);
    return p;
} //gc_malloc_fixed_code_for_class_loading



void *gc_malloc_fixed_code_for_jit(unsigned size)
{
    assert(memory_pool_for_jit_start);
    void *p = malloc_fixed_code_from_pool(&memory_pool_for_jit_start,
                                          &memory_pool_for_jit_end,
                                          size);
    return p;
} //gc_malloc_fixed_code_for_jit



void gc_free_fixed_code_for_class_loading(void *a)
{
} //gc_free_fixed_code_for_class_loading



void gc_free_fixed_code_for_jit(void *a)
{
} //gc_free_fixed_code_for_jit



// end allocating memory for code
////////////////////////////////////////////////////////////



////////////////////////////////////////////////////////////
// begin allocating memory for data


void *gc_malloc_fixed_data_C(unsigned size)
{
    return malloc(size);
} //gc_malloc_fixed_data_C



void gc_free_fixed_data_C(void *a)
{
    free(a);
} //gc_free_fixed_data_C



// end allocating memory for data
////////////////////////////////////////////////////////////



void orp_init_mem_alloc()
{
#ifdef ORP_NT
    SYSTEM_INFO si;
    GetSystemInfo(&si);
    page_size = si.dwPageSize;
    initial_code_pool_size = 4 * page_size;
#endif

    allocate_code_pool(&memory_pool_for_class_loading_start,
                       &memory_pool_for_class_loading_end,
                       initial_code_pool_size);
    allocate_code_pool(&memory_pool_for_jit_start,
                       &memory_pool_for_jit_end,
                       initial_code_pool_size);
} //orp_init_mem_alloc



