// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/gc_v2/include/gc_interface.h,v 1.5 2002/01/09 14:50:13 weldon Exp $
//

#ifndef _gc_interface_H_
#define _gc_interface_H_

#include "block_store.h"

//
// The Interface Between The Garbage Collector and The ORP.
// Version 0.1 November 6, 1997.
//


/********************************************************************
*
* This include file defines the proposed interface between the Garbage
* Collector and any Java Virtual Machine.
*
* ORPs intending to support this GC interface need to implement 
* Class ORPControl and follow the discipline imposed by Class
* GarbageCollector for memory allocation.
*
* This interface attempts, as far as possible, to delegate to the
* ORP activities that require knowledge of internal ORP structure.
* It also attempts to claim for the Garbage Collector any tasks
* that ownership of which can enable the GC to optimize heap
* management.
*
* An attempt is made to minimize the potential for this interface
* to become obsolete with advances in garbage collection algorithms
* and implementations.
*
* NOTE: the private portions of the classes below are not part of the
* interface. They are simply there for the convenience of the initial
* implementation. Please ignore.
*
********************************************************************/

/********************************************************************
ISSUES:

  Candidates for finalization only become apparent during a barrier
  collection. Any consequent finalization activity is delegated to
  the ORP, to be executed after the barrier collection has completed.
  We might consider extending the interface to interleave barrier
  collection with running finalizers as an optimization.

  The interface attempts to make performance critical routines
  in-band. For example, instead of having the JIT make a function
  call to record a write barrier, it will download a write barrier
  from the GC that is customized for the JIT.

  Simple interpreters may make an out-of-band call to the GC for
  registering writes. This will be ignored when there is no
  generational or incremental/concurrent style collection underway.
  For sufficient generality the out-of-band default write barrier
  needs to be replication style. This means that *all* writes need
  to be reported, not just pointer updates.

  What about interior pointers?
  Will we support them?
  If so, how will the GC interface accomodate this?
  Need your comments.

  This interface will either support JNI or RNI. JNI is GC-friendly,
  but comparatively inefficient. RNI is relatively efficient, but
  it is possible to write native libraries using it that are GC-
  hostile (i.e. it is possible to subvert the GC and compromise
  the ORP).

********************************************************************/

/*
                                 |
                   ------------  | ------------------
       GC  ->      |OREControl | | |GarbageCollector |   <- ORP
                   |Instance   | | |Instance         |
       Freeze Thrds|           | | |                 |Allocate
       Enumerate   |           | | |                 |Report write
       Thaw Threads|           | | |                 |Get WriteBarrier
       GetStats    |           | | |                 |Get Hash
                   |           | | |                 |Dump/Debug
                   |           | | |                 |Policy Hints
                   |           | | |                 |
                   |           | | |                 |
                   |           | | |                 |
                   |           | | |                 |
                   ------------- | -------------------
                                 |
*/

/*******************************************************************
*
* Bootstrap: Here is a brief description of how the GC interface is
* set up during ORP initialization: 
*
* (1) Beforp the ORP allocates any garbage collected object, 
* it creates an object of class ORPControl (described below).
*
* (2) It then passes this object as an argument to the creation
* of an object of class GarbageCollector (also described below).
*
* That ORPControl instance will be used by this Garbage
* Collector for controlling and communicating with the ORP during
* its lifetime. 
*
* (3) As part of the creation of the GarbageCollector instance,
* the following types of initialization activities are performed:
*
*   (a) The requisite heap spaces are allocated
*   (b) Any/all GC Daemon threads are created
*   (c) Meta objects for method area components such as ClassClasses
*       and VTables are created.
*   (d) Other sundry GC bookkeeping.
*
* (4) When the ORP gets back a handle to the GarbageCollector
* instance, GC initialization has completed and the ORP now 
* retains that instance for its lifetime for memory management.
*
* (5) Beforp any Java byte codes are executed, the ORP needs to
* download a write barrier sequence and a patch from the Garbage
* collector. A null write barrier implies that no generational,
* asynchronous or concurrent collection is being attempted. Hence
* one is not necessary.
*
* (6) The ORP will use the appropriate GcMalloc calls to allocate
* storage.
*
* (7) Zero or more GC Threads may be concurrently executing GC
* activity that does not require mutators to freeze, such as
* incrementally marking or replicating. 
*
* (8) Some condition (typically a low memory watermark) will result
* in the Garbage Collector using ORPControl to enumerate all live
* references. The ORP will return the live references, and keep all
* mutator threads frozen until a subsequent Thaw call from the GC.
*
* (9) When the method area is collected, the mutator threads will
* include both Java as well as some types of ORP Daemon threads.
*
* (10) The Hash information for an object is part of the GC header.
* Hence the ORP needs to call the GC to obtain this information.
*
* (11) Similarly the Lock information is part of the GC header.
* We may consider uploading a lock sequence to the ORP to maintain
* modularity. I'm not sure about this. 
*
* (12) For ORPs using RNI-style interfaces, the ORP will demarcate
* certain time periods when no barrier collection may occur ("turn
* off GC" in Microsoft RNI parlance.) However other incremental/
* concurrent non-barrier GC activity may continue, and the ORP will
* continue making write barrier calls.
*
* (13) While more efficient, RNI-style interfaces can subvert the
* collector by turning off barrier collection too long.
* A sophisticated and defensive GC implementation may revert to
* conservative mark/sweep to continue to perform, albeit in a
* degraded manner.
*
* (14) Since write barriers need to be explicitely inserted by
* the programmer in RNI-style interfaces, it is trivial for the
* coder to subvert the GC. We might consider providing a hint
* to the GC about the presence of RNI-style native code. The GC
* might then choose to avoid any generational or incremental/
* concurrent styles of collection.
*
* (15) Objects that are candidates for finalization are identified
* in the middle of a barrier collection. The GC will retain those,
* and other F-reachable, objects during that cycle of GC, and 
* provide the list of candidates for finalization back to the ORP,
* who can run the finalizers in any order at any time. Until then
* the ORP needs to augment the root set with these candidates.
*
*******************************************************************/

// The following header file defines the "secret" padding that
// the GC prepends to ORP structures that are managed by thes
// GC.
#include "gc_header.h"
#include "remembered_set.h"

class Nursery;

/******************************************************************
*
* The following two classes define the interface between the ORP
* and the GC.
*
* ORPControl - is used by the GC to communicate with the ORE.
*
* GarbageCollector - is used by the ORP to communicate with the GC.
*
*******************************************************************/

// An instance of the ORPControl class is created by the ORP
// and passed to the Garbage Collector during the latter's 
// initialization, and is subsequently used by the latter to 
// send commands to the ORP.

class ORP_Control {

public:


    // An EnumerateRootSet call to the ORP causes it to 
    // freeze all mutator threads (Java threads) at the earliest
    // possible time, then collaborate with the JIT and the
    // Interpreter in collecting and compacting all
    // live references, and returning a nonredundant list to the GC.
    // ORP Daemon threads will continue to run, but all mutator threads
    // will remain frozen until a subsequent 'ResumeMutators' call 
    // from the GC.

    Remembered_Set *enumerate_root_set();

	ORP_Control();

    void
    enumerate_live_references();

    void update_live_references(void **);

    // After doing a barrier collection, the GC uses the following
    // ResumeMutators call to notify the ORP that it is ok to thaw
    // the frozen mutators.

    VOID resume_mutators();

    // A finalizable unfinalized object that has become unreachable
    // is passed back to the ORP so that the finalizer may be invoked.
    // These calls are made when the mutator threads are frozen.
    // The ORP will collect all these finalizable objects and run
    // the finalizers in some order, perhaps concurrently, after the
    // subsequent call from the GC to thaw all mutators. 
    // Note that finalizable unfinalized objects that have become
    // unreachable are not reclaimed by the GC 

    VOID finalize(IN Java_java_lang_Object *p_object);

    // ORPStatistics is obtained from the ORP by the GC. It will contain
    // runtime information collected by the ORP that might be useful
    // to an Adaptive Garbage Collector. The exact mechanism and format
    // are TBD.

    typedef struct _ORP_Statistics_ {
        // TBD
    } ORP_Statistics;


    PVOID *p_get_policy_and_statistics(IN char *PolicyFormat);


protected:
    Remembered_Set *xxxp_root_set;
private:
};


// A single instance of GarbageCollector is created by the ORP
// upon startup, prior to creating any instances. This
// object is subsequently used for requesting memory allocations
// from garbage collected space, and for interacting with the
// Garbage Collector. One of the arguments passed to the GarbageCollector
// upon creation is an instance of Class VirtualMachine, which is
// used by the Garbage Collector to communicate with the ORP.

class Gc_Interface {

public:

    //
    // This method is called just before ORP exit if the user app
    // specified runFinalizersOnExit on class System.
    //
    virtual void run_all_finalizers() = 0;

    // Constructor for creating a garbage collected heap that the GC
    // will determine the dimensions of. At this time, a heap of size
    // "MinHeap" will be obtained from the Operating System and managed.
    // This heap size is selected in order to maximize OS VM performance
    // and minimze paging. If at any time the GC determines that forward
    // progress or efficiency are negatively impacted by the small heap,
    // the GC will obtain more memory from the OS, up to the max heap size:

    Gc_Interface(IN ORP_Control *p_orp_control) {
		_p_orp_control = p_orp_control;
	}

    // Constructor that also specifies the maximum size of GC heap to use.
    // If this is not specified, the maximum size will be either statically
    // selected, or it might be heuristically determined at initialization
    // time based on the size of the page file:

    Gc_Interface(IN ORP_Control *p_orp_control, 
                     IN ULONG max_heap);

    // Constructor that specifies the minimum and the maximum sizes of the
    // GC heap. If MinHeap isn't specified (by using one of the previous
    // interfaces) it is either statically selected, or it may be 
    // heuristically determined during initialization, based on the
    // available physical memory:

    Gc_Interface(IN ORP_Control *p_orp_control,
                     IN ULONG min_heap, 
                     IN ULONG max_heap);

    // If the ORP retires this GC for some reason, it will be a good
    // citizen and execute the requisite cleanup:

    virtual ~Gc_Interface() {
    }


    //
	// A callback from the ORP, enumerating a single live reference.
	// Several of these calls are made from the ORP into the JIT
	// in response to a call to ORP_Control to enumerate all live refs.
	//
    virtual void gc_add_root_set_entry(Java_java_lang_Object **ref) = 0;

    //
    // A callback from the ORP delivering a weak reference.
    //
    virtual void gc_add_weak_reference(Java_java_lang_Object **ref) = 0;
	// The thread is telling us that the nursery is exhausted.
	// Add this to the used list and give the thread a new one.
	// If all nurseries are exhausted, trigger a collection.
	//
    // Force a barrier collection:
	//
    virtual void reclaim_heap(unsigned int size, bool forced) = 0;
    
    //
    // Thaw is called after a Freeze. It may be triggered by an exit
    // from an RNI native method, or it may be in response to the native
    // method turning GC back on. Freeze and Thaw do not actually
    // "turn GC off". The specific behavior is dependent on the GC
    // scheme. In certain incremental and concurrent collectors,
    // "turning GC off" would curtail some activities such as barrier
    // collects, moving, etc. But routine incremental/concurrent stuff
    // may proceed.
	//
    virtual void thaw() = 0;
	//
    // The mutator can give the GC a hint that an object has just become
    // unreachable, with the hope that the GC can do some opportunistic
    // optimization.
	//
    virtual void free(IN Java_java_lang_Object *p_object) = 0;

    // The following version returns an object that will never be moved.
    // The client ORP should not use this call for large (but movable)
    // objects. Instead, if the client ORP really wants to flag large
    // objects, it should use the GcMallocLarge call, so that the GC
    // can know which optimizations to perform.
 
    
    virtual Java_java_lang_Object *gc_pinned_malloc(long size, 
		                           VTable *p_vtable_pointer,
								   bool return_null_on_fail,
                                   bool double_align
                                   ) = 0;
 
	//
	// The following special version is used during bootstrapping
	// when objects need to be created, but the corresponding
	// classes are not yet available. (Later when the classes
	// get loaded, the ORP goes back and patches up the VTables
	// for these objects.)
	//
    virtual Java_java_lang_Object *gc_pinned_malloc_noclass(long size) = 0;

    // Though the Garbage Collector will identify and handle large 
    // movable objects appropriately, the ORP can give it a hint by
    // using the following allocation call:
 
    virtual void *gc_large_malloc(long size, 
		                          VTable *p_vtable_pointer) = 0;
 
    // The following calls allocate objects from the method area.
    // The Garbage Collector will create and manage the respective
    // meta objects. In the initial implementation the GC will not
    // move or collect these objects. Even in subsequent versions
    // of the GC, the Code will not move till it is ready.

    PVOID gc_vtable_malloc(IN long size);

    PVOID gc_class_class_malloc(IN long size);

    PVOID gc_method_bloc_malloc(IN long size);

    PVOID gc_field_block_malloc(IN long size);

    PVOID gc_code_malloc(IN long size);

    // The following provides a hint from the ORP that the object being
    // allocated is a key object. There is no correctness issue here,
    // since this is an advisory, and key object handling is an optimization.

    PVOID gc_malloc_key_object(IN long size, IN void *p_vtable_pointer);

    // The following routines are used by the ORP in order to
    // allow the method area to be reclaimed:

    PVOID gc_malloc_class_class(IN long size);

    PVOID gc_malloc_method(IN long size);

    PVOID gc_malloc_code(IN long size);

    // Since the object header is managed by the GC, the client
    // ORP should make the following call into the ORP to 
    // obtain the hash value.

    inline ULONG get_object_hash(IN Java_java_lang_Object *p_object);

    // Copying collectors either have the mutator view the TO
    // space or the FROM space exclusively. However the ORP Daemon
    // threads may need to identify if an object has been forwarded.
    // If the object isn't forwarded, the value is NULL. Otherwise
    // it is the forward address.

    inline PVOID object_forward_pointer(IN Java_java_lang_Object *p_object);

    // PinObject instructs the GC to stop moving the (possibly) movable
    // object specified, until a corresponding UnPin call is made.
    // Beforp making this call the ORP needs to first all threads.
    // The GC may move the object before pinning it. The return
    // value is the new address of the Object. The second argument
    // to the function is the live references after the threads were
    // frozen. After pinning the object the Garbage Collector will make
    // a "Thaw" call to the ORP to restart the mutators.


    VOID pin_object(IN Java_java_lang_Object *p_object, 
                   IN Remembered_Set *p_live_references);

    // UnPinObject instructs the GC that it may subsequently move the
    // previously pinned object if it chooses. UnPinning an object that
    // was not previously pinned is a noop.

    VOID unpin_object(IN Java_java_lang_Object *p_object);

    // Using the following call, the GC polls the VM for statistics and
    // policy hints that can be used to optimize GC policy.
    // The input and returned arguments are not specified in the interface.
    // Compliant Virtual Machines may choose to ignore these calls (i.e.
    // return NULL). A complete list of policy and statistics formats and
    // objectives is TBD, and will accompany the GC interface spec.

    typedef struct _Write_Barrier_ {
        // Look at Michal's interface for specifics
    } Write_Barrier;

    typedef struct _Register_State_ {
        // Look at Michal's interface for specifics
    } Register_State;

    Write_Barrier *p_get_write_barrier(IN Register_State);

    // The following is an out-of-band write barrier call that 
    // the ORP can use, perhaps on behalf of naive interpreters
    // or native code. It is a replicating write barrier. This means
    // that all writes are reported - not just pointer updates.
    // pObject is the object that was modified, and Offset is the
    // byte offset from the reference to the object.

    inline VOID register_write(IN Java_java_lang_Object pObject, ULONG Offset);

    // The following routine instructs the Garbage Collector to do a
    // consistency check of the heap. Successively higher levels do
    // checks of increasingly higher detail, but incur higher costs.
    // The ORP should first freeze all mutators before making this
    // call. Otherwise the GC returns FALSE without checking. The GC
    // also returns FALSE if consistency problems were detected (along
    // with a diagnostic dump to stderr). If the check succeeds the
    // routine returns TRUE.

    typedef enum _Consistency_Check_Level_ {
        Level1  = 0,
        Level2,
        Level3,
        Level4,
        Level5
    } Consistency_Check_Level;

    bool execute_consistency_check(IN Consistency_Check_Level cc_level);

    // The following routines are used by the ORP and its components
    // to help debug Garbage Collector related problems:

	//
	// Set the level of verboseness of the GC.
	//
	virtual void set_verbose_level(unsigned int level) = 0;

    // The following routine, Reachable, determines if the
    // object in question is reachable from the specified
    // root set:

    void check_reachable(Remembered_Set Root, Java_java_lang_Object *pObject);

    // The following routine dumps all known live objects 
    // between the two specified memory ranges:

    void display_objects_in_range(void *start, void *end);

    // GCPolicy is used by the ORP to provide hints to the GC, which
    // may be used by an Adaptive Garbage Collector. The exact mechanism
    // and format are TBD.
    //
    typedef struct _GCPolicy_ {
        // TBD
    } GC_Policy;

    VOID gc_policy_hint(IN GC_Policy *p_gc_policy);

    // Locking: TBD

	//
	// ORP notification to GC to clean up.
	//
	virtual void wrapup_gc() = 0;

#ifdef OBJECT_SPLITTING
	virtual int get_large_nursery_size() = 0;
#endif // OBJECT_SPLITTING

protected:

    // A handle to ORPControl for communicating with the client ORE

    ORP_Control *_p_orp_control;

};

#endif //  _gc_interface_H_

