/* This file is part of q-tools, a collection of performance tools
   Copyright (c) 2003 Hewlett-Packard Development Company, L.P.
   Contributed by David Mosberger-Tang <davidm@hpl.hp.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330,
   Boston, MA  02111-1307  USA  */

#include <signal.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/time.h>

#include "q-syscollect.h"

#define HASH_SHIFT	16
#define HASH_TABLE_LEN	(1 << HASH_SHIFT)

#define BLOCK_SHIFT	17
#define BLOCK_LEN	(1 << BLOCK_SHIFT)

struct block
  {
    uintptr_t addr;		/* base-address of this histogram-block */
    struct block *next;		/* collision-chain */
    unsigned long counter[BLOCK_LEN];
  };

/* It should be OK to use a fixed-size hash-table here because we move
   recently accessed elements to the front of the collision-queue on
   each access.  */
struct time_hist
  {
    struct block *htable[HASH_TABLE_LEN];
  };

#if __WORDSIZE == 64
static inline unsigned long
hash (uintptr_t addr)
{
  uintptr_t t = 65535 * addr;

  return (t ^ (t >> 32)) & (HASH_TABLE_LEN - 1);
}
#else
static inline unsigned long
hash (uintptr_t addr)
{
  uintptr_t t = 32768 * addr;

  return (t ^ (t >> 16)) & (HASH_TABLE_LEN - 1);
}
#endif

static inline void
increment_bin (struct block *b, uintptr_t addr, unsigned long count)
{
  unsigned long *c, old_count;

  c = &b->counter[((addr - b->addr) >> INSN_SHIFT) & (BLOCK_LEN - 1)];
  old_count = *c;
  *c = old_count + count;
  if (old_count + count < count)
    /* saturate on overflow */
    *c = ~0UL;
}

static inline void
new_block (struct time_hist *th, uintptr_t addr, unsigned long index,
	   unsigned long count)
{
  struct block *b;

  b = calloc (1, sizeof (struct block));
  if (!b)
    {
      fprintf (stderr, "time-hist.c.new_block: out of memory\n");
      return;
    }
  b->addr = addr & ~((BLOCK_LEN << INSN_SHIFT) - 1);
  increment_bin (b, addr, count);

  b->next = th->htable[index];
  th->htable[index] = b;
}

struct time_hist *
time_hist_create (void)
{
  struct time_hist *th = malloc (sizeof (*th));

  if (!th)
    return NULL;

  memset (th, 0, sizeof (*th));
  return th;
}

void
time_hist_destroy (struct time_hist *th)
{
  free (th);
}

void
time_hist_add (struct time_hist *th, uintptr_t addr, unsigned long count)
{
  struct block *b, *prev;
  unsigned long index = hash (addr >> (BLOCK_SHIFT + INSN_SHIFT));

  b = th->htable[index];
  if (likely (b))
    {
      if (likely (addr - b->addr < (BLOCK_LEN << INSN_SHIFT)))
	{
	  /* fast path... */
	  increment_bin (b, addr, count);
	  return;
	}
      else
	{
	  /* walk the collision-chain:  */
	  for (prev = b, b = b->next; b; prev = b, b = b->next)
	    if (addr - b->addr < (BLOCK_LEN << INSN_SHIFT))
	      {
		increment_bin (b, addr, count);

		/* move to front of collision chain: */
		prev->next = b->next;
		b->next = th->htable[index];
		th->htable[index] = b;
		return;
	      }
	}
    }
  new_block (th, addr, index, count);
}

int
time_hist_extract (struct time_hist *th,
		   int (*callback) (void *arg, uintptr_t addr, size_t len,
				    unsigned long *counts),
		   void *arg)
{
  unsigned long i;
  struct block *b;
  int ret = 0;

  for (i = 0; i < HASH_TABLE_LEN; ++i)
    if (th->htable[i])
      {
	for (b = th->htable[i]; b; b = b->next)
	  {
	    ret = (*callback) (arg, b->addr, BLOCK_LEN, b->counter);
	    if (ret < 0)
	      break;
	  }
      }
  return ret;
}
