/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain;

import java.io.*;
import java.sql.*;
import java.util.*;
import java.text.*;

/**
 * EXPLAIN a query. Should only be used by classes in the
 * com.redhat.rhdb.explain package. The query should have been passed
 * through Explain.query(). Meant to be used if the connection is to a
 * PostgreSQL 7.0, 7.1 or 7.2 database.
 *
 * @author Liam Stewart
 * @author Maintained by <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 */
public class ExplainParserV7 implements ExplainParser
{
	private Explain explain;
	private Statement stmt;
	private	double total_cost = 0;
	private	double total_runtime = 0;

	public ExplainParserV7(Explain exp)
	{
		this.explain = exp;
	}

	public void explain(Connection con, boolean analyze) throws ExplainException
	{
		String dump;
		StringReader strreader;
		StringWriter strwriter, planwriter;
		PrintWriter printer;
		BufferedReader bufread;
		StringTokenizer strtok;
		String str, token;
		SQLWarning dump_warn, plan_warn;
		ExplainTreeNode root;
		ExplainTreeNode toproot = null;
		boolean explain_analyze;

		if (con == null || explain.getQuery() == null)
			throw new ExplainException("Null connection or query");

		try {
			if (con.isClosed())
				throw new ExplainException("Connection is closed");

			DatabaseMetaData dbmd = con.getMetaData();

			// Attempt EXPLAIN ANALYZE?
			String q;
			if (analyze)
			{
				if (dbmd.getDatabaseProductVersion().startsWith("7.2")) {
					q = "EXPLAIN ANALYZE VERBOSE " + explain.getQuery();
					explain_analyze = true;
				} else {
					throw new ExplainException("ANALYZE option of EXPLAIN unsupported on this database version");
				}
			}
			else
			{
				q = "EXPLAIN VERBOSE " + explain.getQuery();
				explain_analyze = false;
			}

			stmt = con.createStatement();
			con.clearWarnings();

			// Turn off auto-commit mode. explain analyze actually
			// runs the query so if it as insert, update, or delete,
			// modifications will be made. we don't want that.. A
			// rollback is performed after the explain is done.
			// 
			// If the connection is already out of auto-commit mode,
			// rollback before doing the explain. After the explain,
			// the auto-commit mode will be restored to whatever it
			// was before the explain.
			boolean autoCommitState = con.getAutoCommit();

			if (explain_analyze)
			{
				if (autoCommitState)
					con.setAutoCommit(false);
				else
					con.rollback();
			}

			// Set planner options, if any specified
			if (!explain.getNewPlannerOptions().trim().equals(""))
				stmt.executeUpdate(explain.getNewPlannerOptions());

			// Now execute (explain) the query itself
			stmt.executeUpdate(q);

			// And save the plan
			dump_warn = con.getWarnings();

			// If necessary, rollback
			if (explain_analyze)
			{
				con.rollback();
				con.setAutoCommit(autoCommitState);
			}

			// Reset planner options to defaults if we changed any
			if (!explain.getNewPlannerOptions().trim().equals(""))
				stmt.executeUpdate(explain.getResetCommands());

			// Parse the first notice. Ignore all 'bad' notices.
			while (true)
			{
				// Parse the first notice (the dump)
				// The form of the dump is:
				//   NOTICE:  QUERY DUMP:
				//   <blank line>
				//   <dump - all on one line>
				if (dump_warn == null)
					throw new ExplainException("Unexpected EXPLAIN output");
				bufread = new BufferedReader(new StringReader(dump_warn.getMessage()));

				// first line
				str = bufread.readLine();
				strtok = new StringTokenizer(str);

				if (strtok.countTokens() == 3 &&
					strtok.nextToken().equals("NOTICE:") &&
					strtok.nextToken().equals("QUERY") &&
					strtok.nextToken().equals("DUMP:"))
					break;
				else
				{
					System.err.println("Ignoring: " + str);
					dump_warn = dump_warn.getNextWarning();
				}
			}

			// If we get here, know that first line of last dump_warn was "NOTICE: QUERY DUMP:"
			// second line
			str = bufread.readLine();
			strtok = new StringTokenizer(str);
			if (strtok.countTokens() != 0)
				throw new ExplainException("Bad number of tokens on second dump line: " + strtok.countTokens());

			// third line
			dump = bufread.readLine();

			plan_warn = dump_warn.getNextWarning();
			if (plan_warn == null)
				throw new ExplainException("Got nothing when QUERY PLAN expected");

			// Parse the second notice (the plan)
			// Form:
			//   NOTICE:  QUERY PLAN:
			//   <blank line>
			//   <root>
			//     ->  <child>
			//       ->  <child of child> ...
			//     ->  <child>
			bufread = new BufferedReader(new StringReader(plan_warn.getMessage()));

			// first line
			str = bufread.readLine();
			strtok = new StringTokenizer(str);
			if (strtok.countTokens() != 3) {
				throw new ExplainException("Bad number of tokens on third dump line: " + strtok.countTokens());
			}
			token = strtok.nextToken();
			if (!token.equals("NOTICE:")) {
				throw new ExplainException("Bad first plan token: " + token);
			}
			token = strtok.nextToken();
			if (!token.equals("QUERY")) {
				throw new ExplainException("Bad second plan token: " + token);
			}
			token = strtok.nextToken();
			if (!token.equals("PLAN:")) {
				throw new ExplainException("Bad third plan token: " + token);
			}

			// second line
			str = bufread.readLine();
			strtok = new StringTokenizer(str);
			if (strtok.countTokens() != 0) {
				throw new ExplainException("Bad number of tokens on second plan line: " + strtok.countTokens());
			}

			// now for the tree...
			planwriter = new StringWriter();
			printer = new PrintWriter(planwriter);

			HashMap ret = buildExplainTree(bufread, printer, toproot, explain_analyze);
			
			calcNodeCost(ret, (ExplainTreeNode) ret.get("root"));
			
			explain.setExplainTree(new ExplainTree((ExplainTreeNode) ret.get("root")));
			explain.setOptions();
			explain.setPlan(planwriter.toString());
			explain.setDump(dump);
			
			// done
		} catch (SQLException e) {
			if ((e.getSQLState() != null) && (e.getSQLState().equals("08S01")))
				throw new ExplainException("Communication link error: " + e.getMessage());
			else
				throw new ExplainException("SQL error: " + e.getMessage());
		} catch (IOException e) {
			throw new ExplainException("I/O error: " + e.getMessage());
		} catch (ExplainException e) {
			throw new ExplainException("Processing error: " + e.getMessage());
		} catch (Exception e) {
			e.printStackTrace();
			throw new ExplainException("Other error: " + e);
		}
	}
	
	public void cancel() {
		try {
			stmt.cancel();
		} catch (SQLException e) {
			System.out.println("exception when trying to cancel() = " + e);
		}
	}

	/*
	 * Recursively calculate the node cost by subtracting the sum of the child costs
	 * from the total cost for the node.  Do the same for time, if available.
	 */
	private void calcNodeCost(HashMap hm, ExplainTreeNode pnode) {
		NumberFormat formatter = NumberFormat.getNumberInstance();
		int numChildren = pnode.getChildCount();
		
		if (pnode.getDetail(ExplainTreeNode.TOTAL_COST) != null)
		{
			try {
				String text = pnode.getDetail(ExplainTreeNode.TOTAL_COST);
				double node_cost = formatter.parse(text).doubleValue();
				for (int i = 0; i < numChildren; i++) {
					text = pnode.getChild(i).getDetail(ExplainTreeNode.TOTAL_COST);
					if (text != null)
						node_cost -= formatter.parse(text).doubleValue();
				}
				pnode.setDetail(ExplainTreeNode.NODE_COST, formatter.format(node_cost));
			} catch (ParseException pe) {
				pe.printStackTrace();	// And do not generate the NODE_COST
			}
		}

		if (pnode.getDetail(ExplainTreeNode.TOTAL_TIME) != null)
		{
			try {
				String text = pnode.getDetail(ExplainTreeNode.TOTAL_TIME);
				double node_time = formatter.parse(text).doubleValue();
				for (int i = 0; i < numChildren; i++) {
					text = pnode.getChild(i).getDetail(ExplainTreeNode.TOTAL_TIME);
					if (text != null)
						node_time -= formatter.parse(text).doubleValue();
				}
				pnode.setDetail(ExplainTreeNode.NODE_TIME, formatter.format(node_time));
			} catch (ParseException pe) {
				pe.printStackTrace();	// And do not generate the NODE_TIME
			}
		}
			
		for (int i = 0; i < numChildren; i++) {
			calcNodeCost(hm, pnode.getChild(i));
		}
	}

	/*
	 * Build an ExplainTree. Return the ExplainTree in a HashMap,
	 * which can be accessed with the ROOT key. The last line read is
	 * also stored in the HashMap. It can be accessed with the LASTLINE
	 * key.
	 */
	private HashMap buildExplainTree(BufferedReader bufread, PrintWriter planwriter, ExplainTreeNode toproot, boolean explain_analyze) throws IOException
	{
		HashMap rv = new HashMap();
		
		String str = bufread.readLine();
		planwriter.println(str);
		ExplainTreeNode root = createNode(str, explain_analyze);
		rv.put("root", root);

		if (toproot == null)
			toproot = root;

		int level = 0;
		ExplainTreeNode parent = root, child = root;
		ArrayList levels = new ArrayList(10);
		levels.add(level, new Integer(whitespaceBeforeArrow(str)));

		str = null;

		while (true)
		{
			if (str == null)
			{
				str = bufread.readLine();
				if (str == null)
					break;
				planwriter.println(str);
			}
			// else str was read last time and needs to be processed again
			// (usually since it was passed back from another call to
			// buildExplainTree
			
			if (str.trim().length() == 0) {
				str = null;
				continue;
			}

			if (str.trim().startsWith("SubPlan") || str.trim().startsWith("InitPlan")) {
				ExplainTreeNode old = parent;
				parent = child;
				child = createNode(str, explain_analyze);
				parent.addChild(child);
				child.setParent(parent);
				
				HashMap ret = buildExplainTree(bufread, planwriter, toproot, explain_analyze);
				ExplainTreeNode n = (ExplainTreeNode) ret.get("root");
				str = (String) ret.get("lastline");
				
				child.addChild(n);
				n.setParent(child);
				parent = old;
			} else if (str.trim().startsWith("Total runtime:")) {
				String s = str.trim();
				s = s.substring(s.indexOf(':') + 2);
				StringTokenizer detailtok = new StringTokenizer(s, ".");
				String intpart = detailtok.nextToken();
				try {
					int ivalue = Integer.parseInt(intpart);
					NumberFormat formatter = NumberFormat.getNumberInstance();
					toproot.setDetail(ExplainTreeNode.TOTAL_RUNTIME, formatter.format(ivalue) + "." + detailtok.nextToken());
				} catch (NumberFormatException pe) {
					toproot.setDetail(ExplainTreeNode.TOTAL_RUNTIME, intpart + "." + detailtok.nextToken());
				}
				str = null;
			} else {
				int n = whitespaceBeforeArrow(str);
				int p = ((Integer) levels.get(level)).intValue();

				if (n > p) {
					levels.add(++level, new Integer(n));
					parent = child;
				} else if (n == p) {
					// same level
				} else if (n < p) {
					// back up to appropriate level
					while (n != p && level != 0) {
						level--;
						p = ((Integer) levels.get(level)).intValue();
						parent = parent.getParent();
					}
					if (level == 0)
						break;
				}
				
				child = createNode(str, explain_analyze);
				parent.addChild(child);
				child.setParent(parent);
				
				str = null;
			}
		}

		rv.put("lastline", str);

		return rv;
	}
	
	/*
	 * Assuming (worthless) initial indentation and no "->", a plan
	 * tree node has the following form:
	 *
	 *   [algorithm] (key1=val1 key2=val2 ...)
	 *
	 * [algorithm] can contain (and usually does contain) more than
	 * one word. Examples: "Seq Scan on hosts", "Merge Join", "Sort"
	 *
	 * -> More error checking/handling is needed.
	 */
	private ExplainTreeNode createNode(String str, boolean explain_analyze) {
		StringTokenizer strtok;
		StringBuffer name = new StringBuffer();
		String type, token, first = null;

		ExplainTreeNode node = new ExplainTreeNode();

		str = stripCruft(str);

		// special "nodes"
		if (str.startsWith("SubPlan"))
		{
			node.setName("SubPlan");
			node.setType(ExplainTreeNode.NODE_SUBPLAN);
			return node;
		}
		else if (str.startsWith("InitPlan"))
		{
			node.setName("InitPlan");
			node.setType(ExplainTreeNode.NODE_INITPLAN);
			return node;
		}

		int detail_start = str.indexOf('(');
		int detail_end = str.indexOf(')', detail_start);
		int analyze_start = -1;
		int analyze_end = -1;

		if (explain_analyze)
		{
			analyze_start = str.indexOf('(', detail_end);
			analyze_end = str.indexOf(')', analyze_start);
		}

		String algo = str.substring(0, detail_start);

		strtok = new StringTokenizer(algo);

		first = strtok.nextToken();
		if (first.equals("Result"))
		{
			node.setType(ExplainTreeNode.NODE_RESULT);
			node.setDetail(ExplainTreeNode.TYPE, first);
			
			name.append(first);
		}
		else if (first.equals("Append"))
		{
			node.setType(ExplainTreeNode.NODE_APPEND);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Hash"))
		{
			if (strtok.hasMoreTokens())
			{
				String next = strtok.nextToken();

				if (next.equals("Join"))
				{
					node.setType(ExplainTreeNode.NODE_JOIN);
					node.setDetail(ExplainTreeNode.TYPE, next);
					node.setDetail(ExplainTreeNode.SUBTYPE, first + " " + next);
					name.append(first + " " + next);
				}
				else
				{
					node.setType(ExplainTreeNode.NODE_UNKNOWN);
					node.setDetail(ExplainTreeNode.TYPE, "unknown: " + first);
					name.append("???");
				}
			}
			else
			{
				node.setType(ExplainTreeNode.NODE_HASH);
				node.setDetail(ExplainTreeNode.TYPE, first);
				name.append(first);
			}
		}
		else if (first.equals("Nested") ||
				 first.equals("Merge"))
		{
			String next = strtok.nextToken();

			node.setType(ExplainTreeNode.NODE_JOIN);
			node.setDetail(ExplainTreeNode.TYPE, "Join");
			node.setDetail(ExplainTreeNode.SUBTYPE, first + " " + next);

			name.append(first + " " + next);
		}
		else if (first.equals("Seq")   ||
				 first.equals("Index") ||
				 first.equals("Tid")   ||
				 first.equals("Subquery"))
		{
			node.setType(ExplainTreeNode.NODE_SCAN);
			node.setDetail(ExplainTreeNode.TYPE, "Scan");
			
			name.append(first + " " + strtok.nextToken());

			if (first.equals("Seq"))
			{
				node.setDetail(ExplainTreeNode.SUBTYPE, "Sequential Scan");
			}
			else if (first.equals("Tid"))
			{
				node.setDetail(ExplainTreeNode.SUBTYPE, "Tid Scan");
			}
			else if (first.equals("Index"))
			{
				node.setDetail(ExplainTreeNode.DIRECTION, "forward");
				node.setDetail(ExplainTreeNode.SUBTYPE, "Index Scan");
			}
			else if (first.equals("Subquery"))
			{
				StringBuffer text = new StringBuffer();

				while (strtok.hasMoreTokens())
				{
					text.append(strtok.nextToken() + " ");
				}

				node.setDetail(ExplainTreeNode.SUBQUERY_ON, text.toString());
				node.setDetail(ExplainTreeNode.SUBTYPE, "Subquery Scan");
			}
			
			while (strtok.hasMoreTokens()) {
				token = strtok.nextToken();
				
				if (token.equals("Backward"))
				{
					node.setDetail(ExplainTreeNode.DIRECTION, "backward");
				}
				else if (token.equals("using"))
				{
					StringBuffer indexes = new StringBuffer();
					String next = null;
					
					while (strtok.hasMoreTokens())
					{
						next = strtok.nextToken();
						if (next.indexOf(',') == -1)
							break;
						indexes.append(next + " ");
					}
					indexes.append(next);
					
					node.setDetail(ExplainTreeNode.INDEXES, indexes.toString());
				}
				else if (token.equals("on"))
				{
					node.setDetail(ExplainTreeNode.TABLE, strtok.nextToken());
					if (strtok.hasMoreTokens())
					{
						node.setDetail(ExplainTreeNode.TABLE_ALIAS, strtok.nextToken());
					}
				}
			}
		}
		else if (first.equals("Materialize"))
		{
			node.setType(ExplainTreeNode.NODE_MATERIALIZE);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Sort"))
		{
			node.setType(ExplainTreeNode.NODE_SORT);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Group"))
		{
			node.setType(ExplainTreeNode.NODE_GROUP);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Aggregate"))
		{
			node.setType(ExplainTreeNode.NODE_AGGREGATE);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("Unique"))
		{
			node.setType(ExplainTreeNode.NODE_UNIQUE);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else if (first.equals("SetOp"))
		{
			node.setType(ExplainTreeNode.NODE_SETOP);
			node.setDetail(ExplainTreeNode.TYPE, first);

			String next = strtok.nextToken();
			node.setDetail(ExplainTreeNode.SUBTYPE, next);
			name.append(next);

			if (strtok.countTokens() == 3 &&
				strtok.nextToken().equals("All"))
			{
				node.setDetail(ExplainTreeNode.SETOP_ALL, "yes");
			}
			else
			{
				node.setDetail(ExplainTreeNode.SETOP_ALL, "no");
			}
		}
		else if (first.equals("Limit"))
		{
			node.setType(ExplainTreeNode.NODE_LIMIT);
			node.setDetail(ExplainTreeNode.TYPE, first);

			name.append(first);
		}
		else
		{
			node.setType(ExplainTreeNode.NODE_UNKNOWN);
			node.setDetail(ExplainTreeNode.TYPE, "unknown: " + first);

			name.append("???");
		}
		
		node.setName(name.toString());

		// details
		String details = str.substring(detail_start + 1, detail_end);
		strtok = new StringTokenizer(details);
		NumberFormat formatter = NumberFormat.getNumberInstance();
		NumberFormat pformatter = NumberFormat.getPercentInstance();

		while (strtok.hasMoreTokens()) {
			String str2 = strtok.nextToken();
			StringTokenizer detailtok = new StringTokenizer(str2, "=");
			String key = detailtok.nextToken();
			String value = detailtok.nextToken();
			
			if (key.equals("cost")) {
				detailtok = new StringTokenizer(value, ".");
				try {
					double cost = Double.parseDouble(detailtok.nextToken()) + (Double.parseDouble(detailtok.nextToken()) / 100);
					node.setDetail(ExplainTreeNode.STARTUP_COST, formatter.format(cost));
				} catch (NumberFormatException pe) {
					node.setDetail(ExplainTreeNode.STARTUP_COST, value.substring(0, value.indexOf(".") + 2));
				}
				try {
					double cost = Double.parseDouble(detailtok.nextToken()) + (Double.parseDouble(detailtok.nextToken()) / 100);
					if (total_cost == 0) {
						// If it is the root, save total cost
						total_cost = cost;
					}
					if (total_cost > 0)
						// Can the total cost be zero? 
						node.setDetail(ExplainTreeNode.TOTAL_COST, formatter.format(cost) +
										"  (" + pformatter.format(cost / total_cost) + ")");
					else
						node.setDetail(ExplainTreeNode.TOTAL_COST, formatter.format(cost));
				} catch (NumberFormatException pe) {
					if (total_cost == 0)
						total_cost = -1;  // If could not get the total cost stop trying
					node.setDetail(ExplainTreeNode.TOTAL_COST, value.substring(value.indexOf(".") + 4));
				}
			} else if (key.equals("rows") || key.equals("width")) {
				try {
					double ivalue = Double.parseDouble(value);
					node.setDetail(key, formatter.format(ivalue));
				} catch (NumberFormatException pe) {
					node.setDetail(key, value);
				}
			} else {
				node.setDetail(key, value);
			}
		}

		// analyze
		if (explain_analyze && analyze_start >= 0)
		{
			String analyze = str.substring(analyze_start + 1, analyze_end);
			strtok = new StringTokenizer(analyze);

			while (strtok.hasMoreTokens())
			{
				String str2 = strtok.nextToken();
				if (str2.equals("actual"))
					str2 = strtok.nextToken();
				StringTokenizer detailtok = new StringTokenizer(str2, "=");
				String key = detailtok.nextToken();
				String value = detailtok.nextToken();

				if (key.equals("time")) {
					detailtok = new StringTokenizer(value, ".");
					try {
						double time = Double.parseDouble(detailtok.nextToken()) + (Double.parseDouble(detailtok.nextToken()) / 100);
						node.setDetail(ExplainTreeNode.STARTUP_TIME, formatter.format(time));
					} catch (NumberFormatException pe) {
						node.setDetail(ExplainTreeNode.STARTUP_TIME, value.substring(0, value.indexOf(".") + 2));
					}
					try {
						double time = Double.parseDouble(detailtok.nextToken()) + (Double.parseDouble(detailtok.nextToken()) / 100);
						if (total_runtime == 0) {
							// If it is the root, save total runtime
							total_runtime = time;
						}
						if (total_runtime > 0)
							// Can the total runtime be zero? 
							node.setDetail(ExplainTreeNode.TOTAL_TIME, formatter.format(time) +
										"  (" + pformatter.format(time / total_runtime) + ")");
						else
							node.setDetail(ExplainTreeNode.TOTAL_TIME, formatter.format(time));
					} catch (NumberFormatException pe) {
						if (total_runtime == 0)
							total_runtime = -1;  // If could not get the total runtime stop trying
						node.setDetail(ExplainTreeNode.TOTAL_TIME, value.substring(value.indexOf(".") + 4));
					}
				} else if (key.equals("rows")) {
					try {
				    	double ivalue = Double.parseDouble(value);
						node.setDetail(ExplainTreeNode.ACTUAL_ROWS, formatter.format(ivalue));
					} catch (NumberFormatException pe) {
						node.setDetail(ExplainTreeNode.ACTUAL_ROWS, value);
					}
				} else if (key.equals("loops")) {
					try {
				    	double ivalue = Double.parseDouble(value);
						node.setDetail(ExplainTreeNode.ACTUAL_LOOPS, formatter.format(ivalue));
					} catch (NumberFormatException pe) {
						node.setDetail(ExplainTreeNode.ACTUAL_LOOPS, value);
					}
				} else {
					node.setDetail(key, value);
				}
			}
		}

		return node;
	}

	/*
	 * count the amount of whitespace before the "->"
	 */
	private static int whitespaceBeforeArrow(String str) {
		int n = str.indexOf("->");

		return n;
	}

	/*
	 * Strip all leading and trailing cruft from a string.
	 * This includes all leading and trailing whitespace and
	 * the arrow ("->"). All whitespace after the arrow and before the
	 * next non-whitespace character is also stripped.
	 */
	private static String stripCruft(String str) {
		str = str.trim();
		if (str.startsWith("->")) {
			str = str.substring(2);
			str = str.trim();
		}

		return str;
	}

}
