/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain.view;

import java.awt.*;
import java.awt.font.*;
import java.awt.geom.*;

import com.redhat.rhdb.treedisplay.DefaultTreeLayoutNode;

/**
 * Class used when laying out the plan tree. ExplainTreeLayoutNodes
 * add an icon property - each object can have an icon that is set
 * either when creation occurs or through the {@link #setIcon}
 * method. The height of the node is the height of the icon + 2*ICON_SPACE
 + TEXT_HEIGHT. The width of the node is the maximum of the width of the
 * text and the width of the icon.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public class ExplainTreeLayoutNode extends DefaultTreeLayoutNode {

	/** Spacing above/below icon. */
	public static final int ICON_SPACE = 15;

	/** Height of text plus above/below spacing. */
	public static final int TEXT_HEIGHT = 20;

	private Image icon;
	
	/**
	 * Creates a new <code>ExplainTreeLayoutNode</code> instance.
	 */
	public ExplainTreeLayoutNode()
	{
		super();
	}

	/**
	 * Creates a new <code>ExplainTreeLayoutNode</code> instance that
	 * will have o as its associated data.
	 *
	 * @param o an <code>Object</code> value
	 */
	public ExplainTreeLayoutNode(Object o)
	{
		super(o);
	}

	/**
	 * Creates a new <code>ExplainTreeLayoutNode</code> instance that
	 * will have o as its associated data and i as its icon.
	 *
	 * @param o an <code>Object</code> value
	 * @param i an <code>Image</code> value
	 */
	public ExplainTreeLayoutNode(Object o, Image i)
	{
		super(o);

		icon = i;
	}

	/**
	 * Get the icon of the node.
	 *
	 * @return an <code>Image</code> value
	 */
	public Image getIcon()
	{
		return icon;
	}

	/**
	 * Set the icon of the node.
	 *
	 * @param i an <code>Image</code> value
	 */
	public void setIcon(Image i)
	{
		icon = i;
	}

	/**
	 * Get the width of the node.
	 *
	 * @return an <code>int</code> value
	 */
	public int getWidth()
	{
		String text;
		if (getData() == null)
			text = "";
		else if (getData().toString() == null)
			text = "<null>";
		else
			text = getData().toString();
		
		TextLayout tl = new TextLayout(text,
									   new Font("Helvetica", Font.PLAIN, 12),
									   new FontRenderContext(null, true, false));
		Rectangle2D bounds = tl.getBounds();

		return Math.max((int)bounds.getWidth(), icon.getWidth(null));
	}

	/**
	 * Get the height of the node.
	 *
	 * @return an <code>int</code> value
	 */
	public int getHeight()
	{
		return icon.getHeight(null) + (2*ICON_SPACE) + TEXT_HEIGHT;
	}
}// ExplainTreeLayoutNode
