/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain.view;

import java.io.File;
import java.sql.*;
import java.util.*;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import javax.swing.*;
import javax.swing.tree.*;
import javax.swing.event.*;
import javax.swing.table.*;

import com.redhat.rhdb.misc.RHDBUtils;
import com.redhat.rhdb.explain.*;
import com.redhat.rhdb.treedisplay.*;
import com.redhat.rhdb.vise.*;
import com.redhat.rhdb.vise.resources.ActionResources;

/**
 * ExplainView.java
 *
 * The ExplainView class is a custom Swing component. An instance of
 * ExplainView can be used for viewing an {@link
 * com.redhat.rhdb.explain.Explain Explain} object.
 * <p>
 * The component consists of a split-pane which contains a customized
 * {@link com.redhat.rhdb.treedisplay.TreeDisplay TreeDisplay} in the
 * left container and a nice details panel in the
 * right container. Some methods are provided for manipulating the
 * TreeDisplay component.
 * <p>
 * ExplainView also consists of three dialog windows:
 * one that shows an overview of the current EXPLAIN plan, one that
 * shows the EXPLAIN output, and one that shows the EXPLAIN VERBOSE
 * output. These are not visible by default, but may be made visible by
 * the invoking the {@link #showOverview}, {@link #showExplainWindow},
 * and {@link #showExplainVerboseWindow} methods respectively.
 * <p>
 * An ExplainView component will notify any interested listeners if
 * certain events occur ({@link ExplainViewEvent}). Objects wishing to
 * be notified should implement the {@link ExplainViewListener}
 * interface and register themselves with the ExplainView object using
 * the {@link #addExplainViewListener} method. They may un-register
 * themselves using the {@link #removeExplainViewListener}.
 *
 * @author Liam Stewart
 * @author Maintained by <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 * @see com.redhat.rhdb.explain.Explain
 * @see com.redhat.rhdb.treedisplay.TreeDisplay
 *
 */
public class ExplainView extends JPanel implements Observer, TreeSelectionListener, TreeDisplayListener {
	private ResourceBundle res;
	private JTextArea options_ta, plan_ta, dump_ta;
 	private ExplainTreeDisplay explain_tree;
	private ExplainTreeDisplayOverviewWindow overview;
	private	JScrollPane scroller;
	private JTable details_table;
	private JLabel details_label;
	private Explain exp;
	private EventListenerList listeners;
	private JPopupMenu popup_options, popup_exp, popup_expvb;
	private JFileChooser chooser_opt, chooser_exp, chooser_expvb;
	private JDialog optionsWindow, explainWindow, explainVerboseWindow;
	
	private ExplainTreeModel emptytree;
	
	private final int MINWIDTH = 122;	// Minimum size for first column of details_table
										// empirically determined

	public static final int TOP = ExplainTreeDisplay.TOP;
	public static final int BOTTOM = ExplainTreeDisplay.BOTTOM;
	public static final int LEFT = ExplainTreeDisplay.LEFT;
	public static final int RIGHT = ExplainTreeDisplay.RIGHT;
	
	/**
	 * Creates a new <code>ExplainView</code> instance.
	 *
	 */
	public ExplainView()
	{
		this(new Explain());
	}

	/**
	 * Creates a new <code>ExplainView</code> instance.
	 *
	 * @param e an <code>Explain</code> value
	 */
	public ExplainView(Explain e)
	{
		emptytree = new ExplainTreeModel(new ExplainTree());
		listeners = new EventListenerList();

		res = ResourceBundle.getBundle("com.redhat.rhdb.vise.resources.ActionResources");			
		initComponents();
		setExplain(e);
	}

	//
	// overview window operations
	//

	/**
	 * Show the overview window.
	 *
	 * @see #hideOverview
	 * @see #setOverviewVisible
	 * @see #isOverviewVisible
	 */
	public void showOverview()
	{
		setOverviewVisible(true);
	}

	/**
	 * Hide the overview window.
	 *
	 * @see #showOverview
	 * @see #setOverviewVisible
	 * @see #isOverviewVisible
	 */
	public void hideOverview()
	{
		setOverviewVisible(false);
	}

	/**
	 * Checks if the overview window is currently visible.
	 *
	 * @return a <code>boolean</code> value: true if the window is
	 * open, false if the window is closed.
	 *
	 * @see #showOverview
	 * @see #hideOverview
	 * @see #setOverviewVisible
	 */
	public boolean isOverviewVisible()
	{
		return overview.isVisible();
	}

	/**
	 * Sets the overview window visible or hidden.
	 *
	 * @param b a <code>boolean</code> value. Use true to show the
	 * window, false to hide the window.
	 *
	 * @see #showOverview
	 * @see #hideOverview
	 * @see #isOverviewVisible
	 */
	public void setOverviewVisible(boolean b)
	{
		overview.setVisible(b);
	}

	//
	// options window operations
	//

	/**
	 * Show the planner options window.
	 *
	 * @see #hideOptionsWindow
	 * @see #setOptionsWindowVisible
	 * @see #isOptionsWindowVisible
	 */
	public void showOptionsWindow()
	{
		setOptionsWindowVisible(true);
	}

	/**
	 * Hide the planner options window.
	 *
	 * @see #showOptionsWindow
	 * @see #setOptionsWindowVisible
	 * @see #isOptionsWindowVisible
	 */
	public void hideOptionsWindow()
	{
		setOptionsWindowVisible(false);
	}

	/**
	 * Checks if the planner options window is currently visible.
	 *
	 * @return a <code>boolean</code> value: true if the window is
	 * open, false if the window is closed.
	 *
	 * @see #showOptionsWindow
	 * @see #hideOptionsWindow
	 * @see #setOptionsWindowVisible
	 */
	public boolean isOptionsWindowVisible()
	{
		return optionsWindow.isVisible();
	}

	/**
	 * Sets the planner options window visible or hidden.
	 *
	 * @param b a <code>boolean</code> value. Use true to show the
	 * window, false to hide the window.
	 *
	 * @see #showOptionsWindow
	 * @see #hideOptionsWindow
	 * @see #isOptionsWindowVisible
	 */
	public void setOptionsWindowVisible(boolean b)
	{
		optionsWindow.setVisible(b);
	}

	//
	// explain window operations
	//

	/**
	 * Show the EXPLAIN output window.
	 *
	 * @see #hideExplainWindow
	 * @see #setExplainWindowVisible
	 * @see #isExplainWindowVisible
	 */
	public void showExplainWindow()
	{
		setExplainWindowVisible(true);
	}

	/**
	 * Hide the EXPLAIN output window.
	 *
	 * @see #showExplainWindow
	 * @see #setExplainWindowVisible
	 * @see #isExplainWindowVisible
	 */
	public void hideExplainWindow()
	{
		setExplainWindowVisible(false);
	}

	/**
	 * Checks if the EXPLAIN output window is currently visible.
	 *
	 * @return a <code>boolean</code> value: true if the window is
	 * open, false if the window is closed.
	 *
	 * @see #showExplainWindow
	 * @see #hideExplainWindow
	 * @see #setExplainWindowVisible
	 */
	public boolean isExplainWindowVisible()
	{
		return explainWindow.isVisible();
	}

	/**
	 * Sets the EXPLAIN output window visible or hidden.
	 *
	 * @param b a <code>boolean</code> value. Use true to show the
	 * window, false to hide the window.
	 *
	 * @see #showExplainWindow
	 * @see #hideExplainWindow
	 * @see #isExplainWindowVisible
	 */
	public void setExplainWindowVisible(boolean b)
	{
		explainWindow.setVisible(b);
	}

	//
	// explain verbose window operations
	//

	/**
	 * Show the EXPLAIN VERBOSE output window.
	 *
	 * @see #hideExplainVerboseWindow
	 * @see #setExplainVerboseWindowVisible
	 * @see #isExplainVerboseWindowVisible
	 */
	public void showExplainVerboseWindow()
	{
		setExplainVerboseWindowVisible(true);
	}

	/**
	 * Hide the EXPLAIN VERBOSE output window.
	 *
	 * @see #showExplainVerboseWindow
	 * @see #setExplainVerboseWindowVisible
	 * @see #isExplainVerboseWindowVisible
	 */
	public void hideExplainVerboseWindow()
	{
		setExplainVerboseWindowVisible(false);
	}

	/**
	 * Checks if the EXPLAIN VERBOSE output window is currently visible.
	 *
	 * @return a <code>boolean</code> value: true if the window is
	 * open, false if the window is closed.
	 *
	 * @see #showExplainVerboseWindow
	 * @see #hideExplainVerboseWindow
	 * @see #setExplainVerboseWindowVisible
	 */
	public boolean isExplainVerboseWindowVisible()
	{
		return explainVerboseWindow.isVisible();
	}

	/**
	 * Sets the EXPLAIN VERBOSE output window visible or hidden.
	 *
	 * @param b a <code>boolean</code> value. Use true to show the
	 * window, false to hide the window.
	 *
	 * @see #showExplainVerboseWindow
	 * @see #hideExplainVerboseWindow
	 * @see #isExplainVerboseWindowVisible
	 */
	public void setExplainVerboseWindowVisible(boolean b)
	{
		explainVerboseWindow.setVisible(b);
	}

	//
	// other methods
	//

	/**
	 * Sets the Explain object to view.
	 *
	 * @param e an <code>Explain</code> value
	 */
	public void setExplain(Explain e)
	{
		this.exp = e;
		update();
	}

	/**
	 * Gets the Explain object that is being viewed.
	 *
	 * @return an <code>Explain</code> value
	 */
	public Explain getExplain()
	{
		return exp;
	}

	/**
	 * Checks if the TreeDisplay component is doing anti-aliasing.
	 *
	 * @return a <code>boolean</code> value
	 * @see com.redhat.rhdb.treedisplay.TreeDisplay#isAntiAlias
	 *
	 */
	public boolean isAntiAlias()
	{
		return explain_tree.isAntiAlias();
	}

	/**
	 * Turns anti-aliasing of the TreeDisplay component either on or
	 * off.
	 *
	 * @param v a <code>boolean</code> value
	 * @see com.redhat.rhdb.treedisplay.TreeDisplay#setAntiAlias
	 *
	 */
	public void setAntiAlias(boolean v)
	{
		explain_tree.setAntiAlias(v);
	}

	public boolean isUseLargeIcons()
	{
		return explain_tree.isUseLargeIcons();
	}

	public void setUseLargeIcons(boolean v)
	{
		explain_tree.setUseLargeIcons(v);
	}

	public void setOrientation(int o) 
	{
		try {
			explain_tree.setOrientation(o);
		} catch (TreeDisplayException ex) { 
			System.out.println(ex.getMessage());
		}
	}

	public int getOrientation()
	{
		return explain_tree.getOrientation();
	}

	/**
	 * Gets the current zoom level of the TreeDisplay component.
	 *
	 * @return a <code>double</code> value
	 * @see com.redhat.rhdb.treedisplay.TreeDisplay#getZoomLevel
	 *
	 */
	public double getZoomLevel()
	{
		return explain_tree.getZoomLevel();
	}

	/**
	 * Sets the current zoom level of the TreeDisplay component.
	 *
	 * @param d a <code>double</code> value. d must be greater than 0.
	 * @see com.redhat.rhdb.treedisplay.TreeDisplay#setZoomLevel
	 *
	 */
	public void setZoomLevel(double d)
	{
		explain_tree.setZoomLevel(d);
	}
	
	/**
	 * Gets the current zoom mode of the TreeDisplay component.
	 *
	 * @return an <code>int</code> value
	 * @see com.redhat.rhdb.treedisplay.TreeDisplay#getZoomMode
	 *
	 */
	public int getZoomMode()
	{
		return explain_tree.getZoomMode();
	}

	/**
	 * Sets the current zoom mode of the TreeDisplay component.
	 *
	 * @param i an <code>int</code> value
	 * @see com.redhat.rhdb.treedisplay.TreeDisplay#setZoomMode
	 *
	 */
	public void setZoomMode(int i)
	{
		explain_tree.setZoomMode(i);
	}

	/**
	 * Redraw the tree. Also redraw the tree in the overview window.
	 */
	public void repaint()
	{
		super.repaint();
		if (overview != null)
			overview.repaint();
		if (explain_tree != null)
			explain_tree.repaint();
	}

	//
	// Methods implementing Observer
	//
	
	/**
	 * The Observable o has been updated.
	 *
	 * @param o an <code>Observable</code> value
	 * @param arg an <code>Object</code> value
	 */
	public void update(Observable o, Object arg)
	{
		try {
			Explain e = (Explain) o;
			if (this.exp != e) {
				return;
			}
			update();
		} catch (ClassCastException ex) {
			// ignore
		}
	}

	//
	// Methods implementing TreeSelectionListener
	//

	/**
	 * Selection has changed.
	 *
	 * @param e a <code>TreeSelectionEvent</code> value
	 */
	public void valueChanged(TreeSelectionEvent e)
	{
		ExplainTreeNode node = (ExplainTreeNode) explain_tree.getLastSelectedPathComponent();
		
		if (node == null)
			updateDetailsTable();
		else
			updateDetailsTable((ExplainTreeNode) node);
	}

	//
	// Methods implementing TreeDisplayListener
	//

	/**
	 * The TreeDisplay has been zoomed.
	 *
	 * @param e a <code>TreeDisplayEvent</code> value
	 */
	public void treeDisplayZoomed(TreeDisplayEvent e)
	{
		notifyExplainViewListeners(new ExplainViewEvent(this, ExplainViewEvent.ZOOM_PERFORMED));
	}
	
	//
	// private methods
	//

	private void initComponents()
	{
		Component visual = createVisualPanel();

		GridBagLayout layout = new GridBagLayout();
		GridBagConstraints c = new GridBagConstraints();
		this.setLayout(layout);

        c.gridx = 0;
		c.gridy = 0;
		c.fill = GridBagConstraints.BOTH;
		c.weightx = 1.0;
		c.weighty = 1.0;
		layout.setConstraints(visual, c);
		this.add(visual);

		// file choosers
		chooser_opt = new JFileChooser();
		chooser_opt.setDialogTitle("Save Planner Option Commands");

		chooser_exp = new JFileChooser();
		chooser_exp.setDialogTitle("Save EXPLAIN Output");

		chooser_expvb = new JFileChooser();
		chooser_expvb.setDialogTitle("Save EXPLAIN VERBOSE Output");

		// windows
		explainWindow = createExplainWindow();
		optionsWindow = createOptionsWindow();
		explainVerboseWindow = createExplainVerboseWindow();
		overview = createOverviewWindow();
	}

	private void overviewWindowClosed()
	{
		notifyExplainViewListeners(new ExplainViewEvent(this, ExplainViewEvent.OVERVIEW_WINDOW_CLOSED));
		hideOverview();
	}

	private void optionsWindowClosed()
	{
		notifyExplainViewListeners(new ExplainViewEvent(this, ExplainViewEvent.OPTIONS_WINDOW_CLOSED));
		hideOptionsWindow();
	}

	private void explainWindowClosed()
	{
		notifyExplainViewListeners(new ExplainViewEvent(this, ExplainViewEvent.EXPLAIN_WINDOW_CLOSED));
		hideExplainWindow();
	}

	private void explainVerboseWindowClosed()
	{
		notifyExplainViewListeners(new ExplainViewEvent(this, ExplainViewEvent.EXPLAIN_VERBOSE_WINDOW_CLOSED));
		hideExplainVerboseWindow();
	}

	private void update()
	{
		updateVisual();
		updateOptions();
		updatePlan();
		updateDump();
	}

	private void updateVisual()
	{
		ExplainTreeNode node;
		ExplainTreeModel m;
		
		if (exp != null && exp.getExplainTree() != null && !exp.getExplainTree().isEmpty()) {
			m = new ExplainTreeModel(exp.getExplainTree());
			
			explain_tree.setModel(m);
			node = exp.getExplainTree().getRoot();
			
			// select the root node
			Object root = m.getRoot();
			// Note that setSelectionPath causes valueChanged to be
			// called and that already calls updateDetailsTable()
			// so we make sure we don't call it twice to minimize flickering
			if (root != null) {
				explain_tree.setSelectionPath(new TreePath(root));
			} else
				updateDetailsTable(node);
		} else {
			explain_tree.setModel(emptytree);
			node = new ExplainTreeNode();

			updateDetailsTable();
		}
	}

	private void updateDetailsTable()
	{
		// Check if we already cleared the pane and don't do it again
		// We know it is empty because otherwise we would have AUTO_RESIZE_OFF
		if (details_table.getAutoResizeMode() == JTable.AUTO_RESIZE_LAST_COLUMN)
			return;

		// No node to fill in the label
		details_label.setText("");
		
		// A model with no details at all
		details_table.setModel(new ExplainTreeNodeDetailModel());
		
		// Make sure we fill all the available space with our headers
		details_table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
	}
	
	private void updateDetailsTable(ExplainTreeNode node)
	{
		// The node type goes in the label above the table 
		details_label.setText(node.getName());

		// Hide our resizing efforts and make scrollbar stabilize faster
		details_table.setVisible(false);
		
		// Turn off resizing so that the horizontal scrollbar shows up
		details_table.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		
		// A model with the details for this node
		details_table.setModel(new ExplainTreeNodeDetailModel(node));
		
		// Add a header renderer so that we can use sizeWidthToFit()
		details_table.getColumnModel().getColumn(1).setHeaderRenderer(new DetailsTableCellRenderer());

		// Make sure first column has an adequate size
		details_table.getColumnModel().getColumn(0).setPreferredWidth(MINWIDTH);

		// Redo the layout so scrollbar is accounted for
		scroller.validate();
		// Wait for it to be done
		try {Thread.sleep(10);} catch (InterruptedException ie) {}
		
		// We need to find what is the real available space and split it between the columns
		Dimension dim = scroller.getViewport().getExtentSize();
		int available = dim.width - MINWIDTH;

		// The second column has enough trailing spaces to cover the filter field width
		// but we don't let it be less than the container available space
		details_table.getColumnModel().getColumn(1).sizeWidthToFit();
		// The "available space" is smaller if the vertical scrollbar is present, not a fixed value
		if (details_table.getColumnModel().getColumn(1).getWidth() < available)
		{
			details_table.getColumnModel().getColumn(1).setMinWidth(15);
			details_table.getColumnModel().getColumn(1).setMaxWidth(Integer.MAX_VALUE);
			details_table.getColumnModel().getColumn(1).setPreferredWidth(available);
		}

		// We are done, show the table again
		details_table.setVisible(true);
	}		

	private void updateOptions()
	{
		options_ta.setText("");
		
		if (exp != null) {
			if (RHDBUtils.isJava14())
				options_ta.setText(exp.getPlannerOptions().replaceAll("; ", ";\n"));
			else
				options_ta.setText(" " + exp.getPlannerOptions().replace(';', '\n'));
		}
	}
	
	private void updatePlan()
	{
		plan_ta.setText("");
		
		if (exp != null) {
			plan_ta.setText(exp.getOutputExplain());
		}
	}
		
	private void updateDump()
	{
		dump_ta.setText("");
		
		if (exp != null) {
			dump_ta.setText(exp.getOutputExplainVerbose());
		}
	}

	private Component makeTreePane()
	{
		JScrollPane pane;

		explain_tree = new ExplainTreeDisplay(emptytree);
		explain_tree.addTreeSelectionListener(this);
		explain_tree.addTreeDisplayListener(this);

  		pane = new JScrollPane(explain_tree);

		return pane;
	}

	private Component makeDetailsPane()
	{
		JPanel pane;

		pane = new JPanel();
		pane.setLayout(new BorderLayout(0, 5));
		pane.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));

		details_label = new JLabel();
		details_label.setForeground(Color.black);
		details_label.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createLineBorder(Color.black),
																   BorderFactory.createEmptyBorder(5, 3, 5, 0)));
		details_label.setBackground(Color.lightGray);
		details_label.setOpaque(true);
		details_label.setPreferredSize(new Dimension(100, 25));

		details_table = new JTable();
		details_table.setShowGrid(false);
		details_table.setCellSelectionEnabled(false);
		if (RHDBUtils.isJava14())
			details_table.setFocusable(false);
		details_table.getTableHeader().setReorderingAllowed(false);

		scroller = new JScrollPane(details_table);
		scroller.getViewport().setBackground(details_table.getBackground());

		pane.add(details_label, BorderLayout.NORTH);
		pane.add(scroller, BorderLayout.CENTER);
		
		return pane;
	}
	
	private Component createVisualPanel()
	{
		JSplitPane splitPane = new JSplitPane();

		Component treePane = makeTreePane();
		Component detailsPane = makeDetailsPane();

		splitPane.setLeftComponent(treePane);
		splitPane.setRightComponent(detailsPane);

		// hmmm
		splitPane.setDividerLocation(500);
		splitPane.setResizeWeight(1);
		
		splitPane.addPropertyChangeListener(new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if (evt.getPropertyName().equals(JSplitPane.DIVIDER_LOCATION_PROPERTY))
				{
					ExplainTreeNode node =
						(ExplainTreeNode) explain_tree.getLastSelectedPathComponent();

					((JSplitPane)evt.getSource()).validate();
		
					if (node == null)
						updateDetailsTable();
					else
						updateDetailsTable((ExplainTreeNode) node);
				}
			}
		});

		return splitPane;
	}

	private JDialog createOptionsWindow()
	{
		JMenuItem menuitem;
		JDialog d;

		options_ta = new JTextArea();
		
		options_ta.setLineWrap(false);
		options_ta.setEditable(false);
		options_ta.addMouseListener(new PopupListener());

		// FIXME This should be a button on the bottom of the window
		popup_options = new JPopupMenu();
		menuitem = new JMenuItem(res.getString(ActionResources.SAVE_NAME));
		menuitem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e)
			{
				saveOptions();
			}
		});
		popup_options.add(menuitem);
		
		JScrollPane scrollPane = new JScrollPane(options_ta);
		
		d = new JDialog();
		d.setSize(new Dimension(600, 200));
		d.setTitle(ExplainResources.getString(ExplainResources.OPTIONS_TITLE));
		d.setLocationRelativeTo(this);
		d.setModal(false);
		d.getContentPane().add(scrollPane);

		d.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				optionsWindowClosed();
			}
		});
		
		return d;
	}

	private JDialog createExplainWindow()
	{
		JMenuItem menuitem;
		JDialog d;

		plan_ta = new JTextArea();
		
		plan_ta.setLineWrap(false);
		plan_ta.setEditable(false);
		plan_ta.addMouseListener(new PopupListener());

		popup_exp = new JPopupMenu();
		// FIXME This should be a button on the bottom of the window
		menuitem = new JMenuItem(res.getString(ActionResources.SAVE_NAME));
		menuitem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e)
			{
				saveExplain();
			}
		});
		popup_exp.add(menuitem);
		
		JScrollPane scrollPane = new JScrollPane(plan_ta);
		
		d = new JDialog();
		d.setSize(new Dimension(600, 400));
		d.setTitle(ExplainResources.getString(ExplainResources.PLAN_TITLE));
		d.setLocationRelativeTo(this);
		d.setModal(false);
		d.getContentPane().add(scrollPane);

		d.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				explainWindowClosed();
			}
		});
		
		return d;
	}

	private JDialog createExplainVerboseWindow()
	{
		JMenuItem menuitem;
		JDialog d;

		dump_ta = new JTextArea();
		
		dump_ta.setLineWrap(true);
		dump_ta.setWrapStyleWord(true);
		dump_ta.setEditable(false);
		dump_ta.addMouseListener(new PopupListener());

		popup_expvb = new JPopupMenu();
		// FIXME This should be a button on the bottom of the window
		menuitem = new JMenuItem(res.getString(ActionResources.SAVE_NAME));
		menuitem.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e)
			{
				saveExplainVerbose();
			}
		});
		popup_expvb.add(menuitem);
		
		JScrollPane scrollPane = new JScrollPane(dump_ta);

		d = new JDialog();
		d.setSize(new Dimension(600, 400));
		d.setTitle(ExplainResources.getString(ExplainResources.DUMP_TITLE));
		d.setModal(false);
		d.setLocationRelativeTo(this);
		d.getContentPane().add(scrollPane);

		d.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				explainVerboseWindowClosed();
			}
		});
		
		return d;
	}

	private ExplainTreeDisplayOverviewWindow createOverviewWindow()
	{
		ExplainTreeDisplayOverviewWindow d;

		d = new ExplainTreeDisplayOverviewWindow(this);
		d.setTreeDisplayClass(ExplainTreeDisplay.class);
		d.setSisterTreeDisplay(explain_tree);
		d.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				overviewWindowClosed();
			}
		});

		return d;
	}

	private void saveOptions()
	{
		int rv = chooser_opt.showSaveDialog(optionsWindow);

		if (rv == JFileChooser.APPROVE_OPTION)
			saveText(chooser_opt.getSelectedFile(), options_ta.getText());
	}

	private void saveExplain()
	{
		int rv = chooser_exp.showSaveDialog(explainWindow);

		if (rv == JFileChooser.APPROVE_OPTION)
			saveText(chooser_exp.getSelectedFile(), plan_ta.getText());
	}

	private void saveExplainVerbose()
	{
		int rv = chooser_expvb.showSaveDialog(explainVerboseWindow);

		if (rv == JFileChooser.APPROVE_OPTION)
			saveText(chooser_expvb.getSelectedFile(), dump_ta.getText());
	}

	private void saveText(File f, String text)
	{
		if (f.exists())
		{
			int rv = JOptionPane.showConfirmDialog(this,
												   "File " + f.getName() + " exists. Overwrite?",
												   "File " + f.getName() + " Exists",
												   JOptionPane.YES_NO_OPTION);

			if (rv == JOptionPane.NO_OPTION)
				return;
		}

		try {
			RHDBUtils.save(f, text);
		} catch (Exception ex) {
			JOptionPane.showMessageDialog(this,
										  new String[] { "Unable to save file " + f.getName(),
														 "Reason: " + ex.getMessage() },
										  "Error Saving " + f.getName(),
										  JOptionPane.ERROR_MESSAGE);
		}
	}

	//
	// stuff for listeners
	//

	/**
	 * Add a given ExplainViewListener.
	 *
	 * @param tdl an <code>ExplainViewListener</code> value
	 */
	public void addExplainViewListener(ExplainViewListener tdl)
	{
		listeners.add(ExplainViewListener.class, tdl);
	}

	/**
	 * Remove a given ExplainViewListener.
	 *
	 * @param tdl an <code>ExplainViewListener</code> value
	 */
	public void removeExplainViewListener(ExplainViewListener tdl)
	{
		listeners.remove(ExplainViewListener.class, tdl);
	}

	/**
	 * Notify all ExplainViewListeners that an ExplainViewEvent has
	 * occurred.
	 *
	 * @param tde an <code>ExplainViewEvent</code> value
	 */
	protected void notifyExplainViewListeners(ExplainViewEvent tde)
	{
		Object[] list = listeners.getListenerList();

		for (int i = 1; i < list.length; i += 2)
		{
			ExplainViewListener l = (ExplainViewListener) list[i];
			switch (tde.getID())
			{
				case ExplainViewEvent.ZOOM_PERFORMED:
					l.explainViewZoomed(tde);
					break;
				case ExplainViewEvent.OVERVIEW_WINDOW_CLOSED:
					l.explainViewOverviewWindowClosed(tde);
					break;
				case ExplainViewEvent.EXPLAIN_WINDOW_CLOSED:
					l.explainViewExplainWindowClosed(tde);
					break;
				case ExplainViewEvent.EXPLAIN_VERBOSE_WINDOW_CLOSED:
					l.explainViewExplainVerboseWindowClosed(tde);
					break;
				case ExplainViewEvent.OPTIONS_WINDOW_CLOSED:
					l.explainViewOptionsWindowClosed(tde);
					break;
				default:
					// ignore
			}
		}
	}

	//
	// Inner classes
	//

	/*
	 * Adapter: ExplainTree -> TreeModel
	 */
	private class ExplainTreeModel implements TreeModel {
		private ExplainTree tree;

		public ExplainTreeModel(ExplainTree t) {
			this.tree = t;
		}

		public Object getChild(Object parent, int index) {
			ExplainTreeNode p = (ExplainTreeNode) parent;
			return p.getChild(index);
		}

		public int getChildCount(Object parent) {
			if (parent == null)
				return 0;

			ExplainTreeNode p = (ExplainTreeNode) parent;
			return p.getChildCount();
		}

		public int getIndexOfChild(Object parent, Object child) {
			ExplainTreeNode p = (ExplainTreeNode) parent;
			ExplainTreeNode c = (ExplainTreeNode) child;

			return p.getIndexOfChild(c);
		}

		public boolean isLeaf(Object node) {
			ExplainTreeNode n = (ExplainTreeNode) node;
			return n.isLeaf();
		}

		public Object getRoot() {
			return tree.getRoot();
		}

		public void valueForPathChanged(TreePath path, Object newValue) { }

		public void addTreeModelListener(TreeModelListener l) { }

		public void removeTreeModelListener(TreeModelListener l) { }

	}

	/*
	 * TableModel for use with ExplainTreeNodes.
	 */
	private class ExplainTreeNodeDetailModel extends AbstractTableModel {
		private String[] colnames = ExplainResources.getStringArray(ExplainResources.DETAIL_COLUMN_NAMES);
		private String[] vcolname = { " ", " "}; 
		private String[][] values;
		private int size;
		private final String BLANK = "blank";

		public ExplainTreeNodeDetailModel()
		{
			values = new String[0][0];
			size = 0;
			vcolname[0] = colnames[0];
			vcolname[1] = colnames[1];
		}
		
		public ExplainTreeNodeDetailModel(ExplainTreeNode n) {
			Set s = n.getDetailTypes();
			Iterator it;
			boolean containstypeinfo = false;
			boolean containsqualexpr = false;
			boolean containscost = false;
			boolean containsactual = false;
			boolean containssubquery = false;
                        
			String str;
			vcolname[0] = colnames[0];
			vcolname[1] = colnames[1];
			ArrayList v = new ArrayList(s.size() + 5);

			if (s.contains(ExplainTreeNode.TYPE))
			{
				containstypeinfo = true;
				v.add(ExplainTreeNode.TYPE);
			}
			if (s.contains(ExplainTreeNode.SUBTYPE))
			{
				containstypeinfo = true;
				v.add(ExplainTreeNode.SUBTYPE);
			}
                        
			if (containstypeinfo)
				v.add(BLANK);

			if (s.contains(ExplainTreeNode.FILTER))
			{
				containsqualexpr = true;
				v.add(ExplainTreeNode.FILTER);
			}
			if (s.contains(ExplainTreeNode.JOIN_FILTER))
			{
				containsqualexpr = true;
				v.add(ExplainTreeNode.JOIN_FILTER);
			}

			if (s.contains(ExplainTreeNode.ONE_TIME_FILTER))
			{
				containsqualexpr = true;
				v.add(ExplainTreeNode.ONE_TIME_FILTER);
			}
                        
			if (s.contains(ExplainTreeNode.INDEX_COND))
			{
				containsqualexpr = true;
				v.add(ExplainTreeNode.INDEX_COND);
			}
                        
			if (s.contains(ExplainTreeNode.MERGE_COND))
			{
				containsqualexpr = true;
				v.add(ExplainTreeNode.MERGE_COND);
			}
                        
			if (s.contains(ExplainTreeNode.HASH_COND))
			{
				containsqualexpr = true;
				v.add(ExplainTreeNode.HASH_COND);
			}
                        
			if (s.contains(ExplainTreeNode.SORT_KEY))
			{
				containsqualexpr = true;
				v.add(ExplainTreeNode.SORT_KEY);
			}
                        
			if (containsqualexpr)
				v.add(BLANK);

			if (s.contains(ExplainTreeNode.SUBQUERY_ON))
			{
				containssubquery = true;
				v.add(ExplainTreeNode.SUBQUERY_ON);
			}
			if (s.contains(ExplainTreeNode.TABLE))
			{
				containssubquery = true;
				v.add(ExplainTreeNode.TABLE);
			}
			if (s.contains(ExplainTreeNode.TABLE_ALIAS))
			{
				containssubquery = true;
				v.add(ExplainTreeNode.TABLE_ALIAS);
			}
			if (s.contains(ExplainTreeNode.DIRECTION))
			{
				containssubquery = true;
				v.add(ExplainTreeNode.DIRECTION);
			}
			if (s.contains(ExplainTreeNode.INDEXES))
			{
				containssubquery = true;
				v.add(ExplainTreeNode.INDEXES);
			}
			if (s.contains(ExplainTreeNode.SETOP_ALL))
			{
				containssubquery = true;
				v.add(ExplainTreeNode.SETOP_ALL);
			}

			if (containssubquery)
				v.add(BLANK);

			if (s.contains(ExplainTreeNode.STARTUP_COST))
			{
				containscost = true;
				v.add(ExplainTreeNode.STARTUP_COST);
			}
			if (s.contains(ExplainTreeNode.NODE_COST))
			{
				containscost = true;
				v.add(ExplainTreeNode.NODE_COST);
			}
			if (s.contains(ExplainTreeNode.TOTAL_COST))
			{
				containscost = true;
				v.add(ExplainTreeNode.TOTAL_COST);
			}
			if (s.contains(ExplainTreeNode.ROWS))
			{
				containscost = true;
				v.add(ExplainTreeNode.ROWS);
			}
			if (s.contains(ExplainTreeNode.WIDTH))
			{
				containscost = true;
				v.add(ExplainTreeNode.WIDTH);
			}

			if (containscost)
				v.add(BLANK);

			if (s.contains(ExplainTreeNode.STARTUP_TIME))
			{
				containsactual = true;
				v.add(ExplainTreeNode.STARTUP_TIME);
			}
			if (s.contains(ExplainTreeNode.NODE_TIME))
			{
				containsactual = true;
				v.add(ExplainTreeNode.NODE_TIME);
			}
			if (s.contains(ExplainTreeNode.TOTAL_TIME))
			{
				containsactual = true;
				v.add(ExplainTreeNode.TOTAL_TIME);
			}
			if (s.contains(ExplainTreeNode.ACTUAL_ROWS))
			{
				containsactual = true;
				v.add(ExplainTreeNode.ACTUAL_ROWS);
			}
			if (s.contains(ExplainTreeNode.ACTUAL_LOOPS))
			{
				containsactual = true;
				v.add(ExplainTreeNode.ACTUAL_LOOPS);
			}
			if (s.contains(ExplainTreeNode.ACTUAL_NOTE))
			{
				containsactual = true;
				v.add(ExplainTreeNode.ACTUAL_NOTE);
			}

			if (containsactual)
				v.add(BLANK);

			if (s.contains(ExplainTreeNode.TOTAL_RUNTIME))
				v.add(ExplainTreeNode.TOTAL_RUNTIME);

			it = v.iterator();
			values = new String[v.size()][2];
			int maxlength = 0;
			int i = 0;
			while (it.hasNext())
			{
				str = (String) it.next();
				values[i][0] = str;

				if (str.equals(BLANK))
					values[i][1] = "";
				else
				{
					values[i][1] = n.getDetail(values[i][0]);
					if (values[i][1].length() > maxlength)
					{
						maxlength = values[i][1].length();
						vcolname[1] = " " + values[i][1];
					}
				}
				i++;
			}

			size = values.length;
			if (size > 0)
			{
				i = values.length - 1;
				while (values[i--][0].equals(BLANK))
					size--;
			}
		}

		public int getColumnCount() {
			return colnames.length;
		}

		public int getRowCount() {
			return size;
		}

		public String getColumnName(int col) {
			return vcolname[col];
		}

		public Object getValueAt(int row, int col) {
			if (col == 0 && values[row][0].equals(BLANK))
				return "";
			return values[row][col];
		}

		public boolean isCellEditable(int row, int col) {
			return false;
		}
	}

	/*
	 * Popup listener for the EXPLAIN and EXPLAIN VERBOSE output
	 * windows.
	 */
	private class PopupListener extends MouseAdapter {
		public void mousePressed(MouseEvent e)
		{
			maybeShowPopup(e);
		}

		public void mouseReleased(MouseEvent e)
		{
			maybeShowPopup(e);
		}

		private void maybeShowPopup(MouseEvent e)
		{
			if (e.isPopupTrigger())
			{
				if (e.getComponent() == plan_ta)
					popup_exp.show(e.getComponent(), e.getX(), e.getY());
				else if (e.getComponent() == dump_ta)
					popup_expvb.show(e.getComponent(), e.getX(), e.getY());
				else if (e.getComponent() == options_ta)
					popup_options.show(e.getComponent(), e.getX(), e.getY());
			}
		}
	}

	private class ExplainTreeDisplayOverviewWindow extends TreeDisplayOverviewWindow {
		public ExplainTreeDisplayOverviewWindow(Component owner)
		{
			super(owner, ExplainResources.getString(ExplainResources.OVERVIEW_TITLE));
		}

		public void propertyChange(java.beans.PropertyChangeEvent evt)
		{
			if (evt.getSource() != getSisterTreeDisplay()) return;

			String prop = evt.getPropertyName();
			if (prop.equals(ExplainTreeDisplay.LARGE_ICON_PROPERTY))
			{
				ExplainTreeDisplay td = (ExplainTreeDisplay) getTreeDisplay();
				td.setUseLargeIcons(((Boolean)evt.getNewValue()).booleanValue());
			}

			super.propertyChange(evt);
		}
	}

	private class DetailsTableCellRenderer extends DefaultTableCellRenderer {
		public Component getTableCellRendererComponent(JTable table, Object value,
									                   boolean isSelected, boolean hasFocus,
											    	   int row, int column)
		{
			String[] colnames = ExplainResources.getStringArray(ExplainResources.DETAIL_COLUMN_NAMES);
	     	JTableHeader header = null;
            // Inherit the colors and font from the header component
   	        if (table != null) {
       	        header = table.getTableHeader();
           	    if (header != null) {
               	    setForeground(header.getForeground());
                   	setBackground(header.getBackground());
       	            setFont(header.getFont());
           	    }
			}
					     
			if (value == null)
				setText("");
			else
			{
				if (header != null)
					setText((column == 1) ? ("      " + colnames[column]) : colnames[column]);
				else
					setText(value.toString());
			}
		    
   	        setBorder(UIManager.getBorder("TableHeader.cellBorder"));
       	    setHorizontalAlignment(JLabel.LEFT);
           	return this;
       	}
	}

}// ExplainView
