/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.io.*;
import java.util.ArrayList;
import java.util.Observable;
import java.util.Set;
import java.util.Iterator;
import java.util.StringTokenizer;

/**
 * Configuration model for Explain. Stores all configuration
 * stuff (last files opened, display properties, configured databases,
 * etc.). Provides load and save methods for loading and saving
 * preferences from/to a file.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 *
 * @see com.redhat.rhdb.vise.ConfigurationException
 */

public class ConfigurationModel extends Observable implements Cloneable {

	/** Maximum number of files in last files list. */
	public static final int MAX_LAST = 4;

	private ArrayList databases;	// List of configured databases
	
	private boolean antialias;		// Use anti-alias in the tree diagram
	private boolean uselargeicons;	// Use large icons in the tree diagram
	private boolean tooltips;		// Show tooltips
	private boolean analyze;		// Toolbar button runs Explain Analize
	private boolean reload;			// Reload last query and reconnect
	private boolean last_query;		// Indicates we had a query opened when we exited
	
	private String last_database;	// Name of last database we connected to
	
	private String[] last;			// Last files opened
	private int num_last = 0;		// Number of files currently in the list

	/* Keywords used in configuration files */
	private String antialias_text = "antialias";
	private String uselargeicons_text = "uselargeicons";
	private String tooltips_text ="tooltips";
	private String analyze_text = "analyze";
	private String reload_text = "reload";
	private String database_text = "database";
	private String query_text = "query";
	private String last_text = "last";
	private String true_text = "true";
	private String false_text = "false";
	
	private boolean dirty;			// Some change has been made and not yet saved
	
	/**
	 * Create a new <code>ConfigurationModel</code> instance.
	 */
	public ConfigurationModel()
	{
		databases = new ArrayList(10);
		antialias = false;
		uselargeicons = false;
		tooltips = true;
		analyze = false;
		reload = false;
		last_query = false;
		last_database = "";
		last = new String[MAX_LAST];
		dirty = false;
	}

	//
	// file i/o stuff - load, save
	//
	
	/**
	 * Load configuration from a given File.
	 *
	 * @param file a <code>File</code> value
	 * @throws IOException if an IOException occurs
	 * @throws ConfigurationException if there is an error in the
	 * configuration file.
	 */
	public void load(File file) throws IOException, ConfigurationException
	{
		final int STATE_UNKNOWN = 0;
		final int STATE_OPTIONS = 1;
		final int STATE_CONNECTION = 2;

		boolean hadoptions = false;
		boolean have_name, have_db;
		boolean seen_name, seen_db, seen_host, seen_port, seen_user, seen_pass;
		int state = STATE_UNKNOWN;
		String str, substr, first, second;
		StringTokenizer strtok;
		int cur_last = 1;

		if (file == null || !file.canRead())
			return;

		FileReader filein = new FileReader(file);
		BufferedReader reader = new BufferedReader(filein);
		DatabaseModel dm = null;

		seen_name = seen_db = seen_host = seen_port = seen_user = seen_pass = false;
		have_name = have_db = false;

		try {
			while ((str = reader.readLine()) != null)
			{
				str = str.trim();
				if (str.length() == 0 || str.startsWith("#"))
				{
					continue;
				}
				else if (str.startsWith("[") && str.endsWith("]"))
				{
					substr = str.substring(1, str.length() - 1).trim();
					strtok = new StringTokenizer(substr);

					int toks = strtok.countTokens();
					if (toks == 0 || toks > 2)
					{
						// bad section header
						throw new ConfigurationException("a");
					}
					
					first = strtok.nextToken();

					if (first.equals("options"))
					{
						if (toks == 2)
						{
							if (strtok.nextToken().equals("end") && state == STATE_OPTIONS)
							{
								state = STATE_UNKNOWN;
							}
							else
							{
								// bad line
								throw new ConfigurationException("b");
							}
						}
						else if (state == STATE_UNKNOWN && hadoptions)
						{
							// already done options - bad file
							throw new ConfigurationException("c");
						}
						else if (state == STATE_UNKNOWN)
						{
							state = STATE_OPTIONS;
							hadoptions = true;
						}
						else
						{
							// bad line
							throw new ConfigurationException("d");
						}
					}
					else if (first.equals("connection"))
					{
						if (toks == 2)
						{
							if (strtok.nextToken().equals("end") && state == STATE_CONNECTION)
							{
								state = STATE_UNKNOWN;
								if (have_name && have_db)
								{
									try {
										addDatabaseModel(dm);
									} catch (ConfigurationException e) {
										// ignore
									}
								}
								else
								{
									// missing info
									System.err.println("ConfigurationModel: missing database info");
								}
							}
							else
							{
								// bad line
								throw new ConfigurationException("e");
							}
						}
						else if (state == STATE_UNKNOWN)
						{
							state = STATE_CONNECTION;
							dm = new PostgreSQLDatabaseModel();
							seen_name = seen_db = seen_host = seen_port = seen_user = seen_pass = false;
							have_name = have_db = false;
						}
						else
						{
							// bad line
							throw new ConfigurationException("f: " + substr);
						}
					}
					else
					{
						// bad section header
						throw new ConfigurationException("g: " + substr);
					}
				}
				else if (str.startsWith("["))
				{
					// bad line
					throw new ConfigurationException("h");
				}
				else
				{
					switch (state)
					{
						case STATE_UNKNOWN:
							// bad line
							throw new ConfigurationException("i");
						case STATE_OPTIONS:
							int indexeq = str.indexOf('=');
							if (indexeq == -1)
							{
								// bad line
							}
							first = str.substring(0, indexeq);
							second = str.substring(indexeq+1, str.length());

							if (first.equals(antialias_text))
							{
								if (second.equals(true_text))
								{
									setAntiAlias(true);
								}
								else if (second.equals(false_text))
								{
									setAntiAlias(false);
								}
								else
								{
									// bad option value - ignore for now
								}
							}
							else if (first.equals(uselargeicons_text))
							{
								if (second.equals(true_text))
								{
									setUseLargeIcons(true);
								}
								else if (second.equals(false_text))
								{
									setUseLargeIcons(false);
								}
								else
								{
									// bad option value
								}
							}
							else if (first.equals(tooltips_text))
							{
								if (second.equals(true_text))
								{
									setShowToolTips(true);
								}
								else if (second.equals(false_text))
								{
									setShowToolTips(false);
								}
								else
								{
									// bad option value - ignore for now
								}
							}
							else if (first.equals(analyze_text))
							{
								if (second.equals(true_text))
								{
									setAnalyze(true);
								}
								else if (second.equals(false_text))
								{
									setAnalyze(false);
								}
								else
								{
									// bad option value - ignore for now
								}
							}
							else if (first.equals(reload_text))
							{
								if (second.equals(true_text))
								{
									setReload(true);
								}
								else if (second.equals(false_text))
								{
									setReload(false);
								}
								else
								{
									// bad option value - ignore for now
								}
							}
							else if (first.startsWith(query_text))
							{
								// Can't use setLastQuery() or it will cause
								// unecessary update of config file later
								
								if (second.equals(true_text))
								{
									last_query = true;
								}
								else if (second.equals(false_text))
								{
									last_query = false;
								}
								else
								{
									// bad option value - ignore for now
								}
							}
							else if (first.startsWith(database_text))
							{
								// Can't use setLastDatabase() or it will cause
								// unecessary update of config file later
								last_database = second;
							}
							else if (first.startsWith(last_text))
							{
								String number = first.substring(last_text.length());
								int num = -1;
								try {
									num = Integer.parseInt(number);
								} catch (NumberFormatException ex) { }
								if (num == cur_last && cur_last <= MAX_LAST)
								{
									last[cur_last - 1] = second;
									cur_last++;
									num_last++;
								}
								else
								{
									// ignore
								}
							}
							else
							{
								// bad option - ignore
							}
							
							break;
						case STATE_CONNECTION:
							indexeq = str.indexOf('=');
							if (indexeq == -1)
							{
								// bad line
							}
							first = str.substring(0, indexeq);
							second = str.substring(indexeq+1, str.length());

							if (first.equals("name") && !seen_name)
							{
								seen_name = true;
								if (getDatabaseModel(second) != null)
								{
									// duplicate entry
								}
								else
								{
									dm.setName(second);
									have_name = true;
								}
							}
							else if (first.equals("database") && !seen_db)
							{
								seen_db = true;
								dm.setDatabase(second);
								have_db = true;
							}
							else if (first.equals("host") && !seen_host)
							{
								seen_host = true;
								dm.setHost(second);
							}
							else if (first.equals("port") && !seen_port)
							{
								seen_port = true;
								dm.setPort(second);
							}
							else if (first.equals("user") && !seen_user)
							{
								seen_user = true;
								dm.setUser(second);
							}
							else if (first.equals("password") && !seen_pass)
							{
								seen_pass = true;
								dm.setPassword(second);
								dm.setSavePassword(true);
							}
							else
							{
								// unknown or duplicate option - ignore
							}
							
							break;
						default:
							break;
					}
				}
			}
		} finally {
			filein.close();
		}
	}

	/**
	 * Save the current configuration to the given File.
	 *
	 * @param file a <code>File</code> value
	 * @exception IOException if an error occurs
	 */
	public void save(File file) throws IOException
	{
		// If we don't need or can't, don't save
		if (!dirty || file == null || !file.canWrite())
			return;
		
		FileWriter fileout = new FileWriter(file);
		PrintWriter printer = new PrintWriter(new BufferedWriter(fileout));

		// options
		printer.println("[options]");
		printer.println(antialias_text + "=" + isAntiAlias());
		printer.println(uselargeicons_text + "=" + isUseLargeIcons());
		printer.println(tooltips_text + "=" + isShowToolTips());
		printer.println(analyze_text + "=" + isAnalyze());
		printer.println(reload_text + "=" + isReload());
		printer.println(database_text + "=" + getLastDatabase());
		printer.println(query_text + "=" + isLastQuery());
		for (int i = 0; i < num_last; i++)
		{
			printer.println(last_text + (i + 1) + "=" + last[i]);
		}
		printer.println("[options end]");

		// connections
		Iterator it = databases.iterator();
		while (it.hasNext())
		{
			DatabaseModel d = (DatabaseModel) it.next();

			printer.println();
			printer.println("[connection]");
			printer.println("name=" + d.getName());
			printer.println("database=" + d.getDatabase());
			printer.println("host=" + d.getHost());
			printer.println("port=" + d.getPort());
			printer.println("user=" + d.getUser());
			
			if (d.isSavePassword() && d.getPassword() != null)
				printer.println("password=" + d.getPassword());
			printer.println("[connection end]");
		}

		printer.flush();
		fileout.close();
		
		dirty = false;		// All changes have been saved
	}

	//
	// non-database model stuff
	//

	/**
	 * Is EXPLAIN ANALYZE to be attempted?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isAnalyze()
	{
		return analyze;
	}

	/**
	 * Turn on/off reloading/reconnecting.
	 *
	 * @param v a <code>boolean</code> value
	 */
	public void setAnalyze(boolean v)
	{
		this.analyze = v;

		updated();
	}

	/**
	 * Is the last query to be reloaded?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isReload()
	{
		return reload;
	}

	/**
	 * Turn on/off explain analyze when possible.
	 *
	 * @param v a <code>boolean</code> value
	 */
	public void setReload(boolean v)
	{
		// Mark a change in this option to be saved at the end
		if (this.reload != v)
			dirty = true;
			
		this.reload = v;
	
		/* Only affects next run, so don't notify observers. */
	}

	/**
	 * Name of last database we connected to
	 *
	 * @return a <code>String</code> value
	 */
	public String getLastDatabase()
	{
		return last_database;
	}

	/**
	 * Save the name of the last database we connected to.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setLastDatabase(String v)
	{
		// Mark a change in connection to be saved at the end
		if (!this.last_database.equals(v))
			dirty = true;
			
		this.last_database = v;

		/* Only affects next run, so don't notify observers. */
	}

	/**
	 * Indicates if we had a query in the text area
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isLastQuery()
	{
		return last_query;
	}

	/**
	 * Save indication if we had a query in the text area.
	 *
	 * @param v a <code>String</code> value
	 */
	public void setLastQuery(boolean v)
	{
		// Mark a change in text area status to be saved at the end
		if (this.last_query != v) {
			dirty = true;
            this.last_query = v;
        }

		/* Only affects next run, so don't notify observers. */
	}

	/**
	 * Are tooltips supposed to be shown?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isShowToolTips()
	{
		return tooltips;
	}

	/**
	 * Turn on/off display of tool tips.
	 *
	 * @param v a <code>boolean</code> value
	 */
	public void setShowToolTips(boolean v)
	{
		this.tooltips = v;

		updated();
	}

	/**
	 * Is the plan tree display supposed to do anti-aliasing when
	 * drawing?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isAntiAlias()
	{
		return antialias;
	}
	
	/**
	 * Turn on/off anti-aliasing of tree display component.
	 *
	 * @param v a <code>boolean</code> value
	 */
	public void setAntiAlias(boolean v)
	{
		this.antialias = v;

		updated();
	}

	/**
	 * Is the plan tree display supposed to use large icons?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isUseLargeIcons()
	{
		return uselargeicons;
	}

	/**
	 * Turn on/off use of large icons in tree display component.
	 *
	 * @param v a <code>boolean</code> value
	 */
	public void setUseLargeIcons(boolean v)
	{
		this.uselargeicons = v;

		updated();
	}

	/**
	 * Add given file to list of last files opened.
	 *
	 * @param f a <code>File</code> value
	 */
	public void addLast(File f)
	{
		int idx = getIndexOfLast(f);
		
		if (idx == 0)
			return;		// Already the first in the list

		if (idx == -1) {
			if (num_last < MAX_LAST)
				num_last++;

			idx = num_last - 1;
		}

		for (int i = idx; i > 0; i--)
			last[i] = last[i-1];

		last[0] = f.getAbsolutePath();
		
		updated();
	}

	/**
	 * Remove given file from list of last files opened.
	 *
	 * @param f a <code>File</code> value
	 */
	public void removeLast(File f)
	{
		int idx = getIndexOfLast(f);
		
		if (idx == -1)
			return;		// File not in the list (?!)

		// If it is not the last one, we promote the ones after it
		for (int i = idx; i < (num_last - 1); i++)
			last[i] = last[i+1];

		// Otherwise we just shorten the list
		num_last--;

		updated();
	}

	/**
	 * Get the number of files in the last files list.
	 *
	 * @return an <code>int</code> value
	 */
	public int getNumLast()
	{
		return num_last;
	}

	/**
	 * Get the File from the last files list that is in position i.
	 *
	 * @param i an <code>int</code> value
	 * @return a <code>File</code> value
	 * @throws IllegalArgumentException if i less than 0 or greater or
	 * equal to MAX_LAST.
	 */
	public File getLast(int i)
	{
		if (i < 0 || i >= MAX_LAST)
			throw new IllegalArgumentException();
		return new File(last[i]);
	}

	/**
	 * Get the index of File f in the last files list. Returns -1 if
	 * f is not in the list.
	 *
	 * @param f a <code>File</code> value
	 * @return an <code>int</code> value
	 */
	public int getIndexOfLast(File f)
	{
		for (int i = 0; i < num_last; i++)
		{
			if (f.getAbsolutePath().equals(last[i]))
				return i;
		}

		return -1;
	}

	/**
	 * Does the last files list contain the given File?
	 *
	 * @param f a <code>File</code> value
	 * @return a <code>boolean</code> value
	 */
	public boolean lastContains(File f)
	{
		for (int i = 0; i < num_last; i++)
		{
			if (f.getAbsolutePath().equals(last[i]))
				return true;
		}

		return false;
	}
	
	//
	// database model stuff
	//
	
	/**
	 * Set the DatabaseModel number <i>index</i> to the given
	 * model. Useful for replacing a certain DatabaseModel. If
	 * <i>index</i> is out of range, the configuration already
	 * contains <i>d</i> in another spot, or the configuration
	 * already contains a DatabaseModel with the same name as
	 * <i>d</i> then an exception is thrown.
	 *
	 * @param index an <code>int</code> value
	 * @param d a <code>DatabaseModel</code> value
	 * @exception ConfigurationException if an error occurs
	 */
	public void setDatabaseModel(int index, DatabaseModel d) throws ConfigurationException
	{
		if (index < 0 || index > databases.size())
			throw new ConfigurationException("Index out of range");

		if (databases.contains(d))
			throw new ConfigurationException("Adding a duplicate DatabaseModel: " + d.getName());

		if (getDatabaseModel(d.getName()) != null && getIndexOfDatabaseModel(getDatabaseModel(d.getName())) != index)
			throw new ConfigurationException("There is already a database configuration with the name '" + d.getName() + "'");

		if (index == databases.size())
			addDatabaseModel(d);
		else
			databases.set(index, d);

		updated();
	}
	
	/**
	 * Add the given DatabaseModel to the list of database in the
	 * configuration only if the model to add is not already in the
	 * configuration, the configuration doesn't already contain a
	 * model with the same name, or if the name of the model to add is
	 * null.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 * @exception ConfigurationException if an error occurs
	 */
	public void addDatabaseModel(DatabaseModel d) throws ConfigurationException
	{
		if (d == null)
			throw new ConfigurationException("Trying to add a null DatabaseModel");

		if (databases.contains(d))
			throw new ConfigurationException("Adding a duplicate DatabaseModel: " + d.getName());

		if (getDatabaseModel(d.getName()) != null)
			throw new ConfigurationException("There is already a database configuration with the name '" + d.getName() + "'");

		databases.add(d);

		updated();
	}

	/**
	 * Remove the given DatabaseModel from the configuration.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 */
	public void removeDatabaseModel(DatabaseModel d)
	{
		removeDatabaseModel(d.getName());
	}

	/**
	 * Remove the DatabaseModel with the given name from the
	 * configuration.
	 *
	 * @param name a <code>String</code> value
	 */
	public void removeDatabaseModel(String name)
	{
		DatabaseModel d = getDatabaseModel(name);

		if (d == null)
			return;
		
		databases.remove(databases.indexOf(d));

		updated();
	}

	/**
	 * Remove the DatabaseModel number <i>index</i> from the
	 * configuration.
	 *
	 * @param index an <code>int</code> value
	 */
	public void removeDatabaseModel(int index)
	{
		databases.remove(index);

		updated();
	}

	/**
	 * Get DatabaseModel <i>index</i>.
	 *
	 * @param index an <code>int</code> value
	 * @return a <code>DatabaseModel</code> value
	 */
	public DatabaseModel getDatabaseModel(int index)
	{
		return (DatabaseModel) databases.get(index);
	}

	/**
	 * Get the DatabaseModel with the given name.
	 *
	 * @param name a <code>String</code> value
	 * @return a <code>DatabaseModel</code> value
	 */
	public DatabaseModel getDatabaseModel(String name)
	{
		DatabaseModel rv = null;
		Iterator it = databases.iterator();
		
		while (it.hasNext())
		{
			DatabaseModel d = (DatabaseModel) it.next();
			if (name.equals(d.getName()))
			{
				rv = d;
				break;
			}
		}

		return rv;
	}

	/**
	 * Does the configuration contain the given DatabaseModel?
	 *
	 * @param d a <code>DatabaseModel</code> value
	 * @return a <code>boolean</code> value
	 */
	public boolean hasDatabaseModel(DatabaseModel d)
	{
		boolean rv = false;
		Iterator it = databases.iterator();

		if (d == null)
			return false;
		
		while (it.hasNext())
		{
			DatabaseModel e = (DatabaseModel) it.next();
			if (d == e)
			{
				rv = true;
				break;
			}
		}

		return rv;
	}

	/**
	 * Get the index of a given DatabaseModel.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 * @return an <code>int</code> value
	 */
	public int getIndexOfDatabaseModel(DatabaseModel d)
	{
		if (hasDatabaseModel(d))
			return databases.indexOf(d);
		return -1;
	}
	
	/**
	 * Get all DatabaseModels in an array.
	 *
	 * @return a <code>DatabaseModel[]</code> value
	 */
	public DatabaseModel[] getDatabases()
	{
		return (DatabaseModel[]) databases.toArray();
	}

	/**
	 * Get the number of DatabaseModels that this configuration stores.
	 *
	 * @return an <code>int</code> value
	 */
	public int getDatabaseCount()
	{
		return databases.size();
	}

	//
	// cloneable
	//

	/**
	 * Clone the configuration.
	 *
	 * @return an <code>Object</code> value
	 */
	public Object clone()
	{
		ConfigurationModel c = new ConfigurationModel();
		c.setAntiAlias(isAntiAlias());
		c.setUseLargeIcons(isUseLargeIcons());
		c.setShowToolTips(isShowToolTips());
		c.setAnalyze(isAnalyze());
		c.setReload(isReload());
		c.setLastQuery(isLastQuery());
		c.setLastDatabase(getLastDatabase());

		for (int i = getNumLast() - 1; i >= 0; i--)
		{
			c.addLast(getLast(i));
		}
		
		Iterator it = databases.iterator();
		while (it.hasNext())
		{
			DatabaseModel d = (DatabaseModel) it.next();
			DatabaseModel e = (DatabaseModel) d.clone();

			try {
				c.addDatabaseModel(e);
			} catch (ConfigurationException ex) {
				// we should never get this
			}
		}
		
		c.dirty = dirty;

		return c;
	}

	//
	// private methods
	//

	/*
	 * Note that configuration needs to be saved and notify observers.
	 */
	private void updated()
	{
		dirty = true;	// Need to save config file
		
		// Notify observers
		setChanged();
		notifyObservers();
	}
}// ConfigurationModel
