/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.io.File;
import com.redhat.rhdb.misc.RHDBUtils;

/**
 * QueryModel represents a query (to EXPLAIN). Functionality is
 * provided for saving the query.
 *
 * @author Liam Stewart
 * @author Maintained by <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 */
public class QueryModel {
	private String query;
	private File file;
	private boolean isnew;
	private boolean ismodified;

	/**
	 * Creates a new <code>QueryModel</code> instance.
	 */
	public QueryModel()
	{
		isnew = true;
		ismodified = false;
		file = null;
		query = "";
	}

	/**
	 * Creates a new <code>QueryModel</code> instance. The query is
	 * stored in the given File and will be loaded.
	 *
	 * @param f a <code>File</code> value
	 * @exception QueryModelException if an error occurs
	 */
	public QueryModel(File f) throws QueryModelException
	{
		isnew = false;
		ismodified = false;
		file = f;

		try {
			query = RHDBUtils.load(file);
		} catch (Exception ex) {
			file = null;	//	We didn't get it after all
			throw new QueryModelException(ex.getMessage());
		}
	}

	/**
	 * Creates a new <code>QueryModel</code> instance. The query is
	 * passed as a string, not loaded from the file.
	 * The file parameter is only used to provide the file name.
	 *
	 * @param f a <code>File</code> value
	 * @param query a <code>String</code> value
	 * @exception QueryModelException if an error occurs
	 */
	public QueryModel(File f, String query)
	{
		isnew = false;
		ismodified = false;
		file = f;

		this.query = query;
	}

	/**
	 * Is the QueryModel new (created, not loaded, but not saved)?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isNew()
	{
		return isnew;
	}

	/**
	 * Has the query represented by the QueryModel been modified in
	 * any way?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isModified()
	{
		return ismodified;
	}

	/**
	 * Save the query.
	 *
	 * @exception QueryModelException if an error occurs
	 */
	public void save() throws QueryModelException
	{
		if (isnew)
			throw new QueryModelException("Can't save a new file without a name.");

		// ASSERT: Callers already assert that we are called for real queries,
		// so this test is not strictly necessary
		String extension = RHDBUtils.getExtension(file);
		if ((extension != null) &&
			(extension.equalsIgnoreCase(ExplainResources.getString(ExplainResources.PLAN_EXTENSION))))
			throw new QueryModelException("Can't save - query did not come from a query file.");

		try {
			RHDBUtils.save(file, query);
		} catch (Exception ex) {
			throw new QueryModelException(ex.getMessage());
		}
	}
	
	/**
	 * Save the query to a new file.
	 *
	 * @param filename a <code>String</code> value
	 * @exception QueryModelException if an error occurs
	 */
	public void saveas(String filename) throws QueryModelException
	{
		saveas(new File(filename));
	}

	/**
	 * Save the query to a new file.
	 *
	 * @param f a <code>File</code> value
	 * @exception QueryModelException if an error occurs
	 */
	public void saveas(File f) throws QueryModelException
	{
		this.file = f;
		if (isnew) isnew = false;

		save();
	}
	
	/**
	 * Get the file that the query is loaded from / saved to.
	 *
	 * @return a <code>File</code> value
	 */
	public File getFile()
	{
		return file;
	}
	
	/**
	 * Get the query that the model represents.
	 *
	 * @return a <code>String</code> value
	 */
	public String getQuery()
	{
		return query;
	}

	/**
	 * Set the query.
	 *
	 * @param q a <code>String</code> value
	 */
	public void setQuery(String q)
	{
		this.query = q;

		ismodified = true;
	}
}// QueryModel
