/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.io.*;
import java.sql.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import java.text.*;
import java.util.*;
import javax.swing.plaf.metal.MetalLookAndFeel;
import javax.swing.filechooser.FileFilter;

import com.redhat.rhdb.explain.view.*;
import com.redhat.rhdb.explain.*;
import com.redhat.rhdb.treedisplay.*;
import com.redhat.rhdb.misc.RHDBUtils;

/**
 * Explain proper. This class contains a main method that runs
 * the Explain program.
 *
 * @author Liam Stewart
 * @author <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 * @version 1.2.0
 */
public class Vise extends JFrame implements ActionListener, ExplainListener,
								ItemListener, ExplainViewListener, Observer {
	private ExplainView expview;
	private Explain exp;
	private StatusBar statusbar;
	private ConnectionModel cm;
	private DatabaseModel dbm;
	private ConfigurationModel config;
	private DatabaseComboBoxModel dbcbm;
	private JComboBox dbcombobox;
	private JToggleButton overviewButton;
	private JButton explainButton;
	private JCheckBoxMenuItem overviewMenuitem, optionsMenuitem,
							  explainMenuitem, explainVerboseMenuitem;
	private JFileChooser filechooser, spfilechooser, mfilechooser;
	private JRadioButtonMenuItem actualsizeMenuitem, fitwidthMenuitem,
								 fitwindowMenuitem, noselectMenuitem;
	private JRadioButtonMenuItem topMenuitem, bottomMenuitem, leftMenuitem,
								 rightMenuitem;
	private ButtonGroup zoommodeGroup, orientationGroup;
	private QueryModel query;
	private JMenu filemenu;
	private JComboBox zoomcombobox;
	private NumberFormat formatter;
	private JTextArea query_ta;
	private JLabel query_ta_label;
	private boolean canSaveConfiguration;
	private boolean starting_up;	// We are in the tool startup code
	private boolean new_config;		// We have a new configuration model
									// and must reconnect
	private boolean running;		// A query is running in a separate thread
	private boolean modstatus = true;	// Indicates a modified text is in the
										// text area

	private ResourceBundle explainResources;
	
	private PreferencesDialog prefs;
	private DatabasesDialog dbases;
	private PlannerOptionsDialog poptions;
	private PasswordDialog passdialog;
	private AboutDialog aboutdialog;

	private HelpWindow helpwindow;
	
	private ActionFactory afactory;
	private GenericAction newAction, openAction, saveAction;
	private GenericAction cutAction, copyAction, pasteAction;
	private GenericAction aboutAction, exitAction, explainAction, analyzeAction;
	private GenericAction prefsAction, dbasesAction, poptsAction;
	private GenericAction zoominAction, zoomoutAction;
	private GenericAction saveasAction, refreshAction, stopAction;
	private GenericAction helpAction;

	private File conffile;
	private File lockfile;

	private final double[] zooms = {
					0.083, 0.125, 0.25, 0.33, 0.5, 0.66, 0.75,
					1.0,
					1.25, 1.5, 2.0, 3.0, 4.0, 6.0, 8.0, 12.0, 16.0 };
							   
	private final String sqlext = ExplainResources.getString(
									  ExplainResources.SQL_EXTENSION);
	private final String planext = ExplainResources.getString(
									   ExplainResources.PLAN_EXTENSION);
	
	/**
	 * Called by observable objects which have had this object added
	 * as an observer.
	 *
	 * @param o an <code>Observable</code> value
	 * @param arg an <code>Object</code> value
	 */
	public void update(Observable o, Object arg)
	{
		if (o != config)
			return;

		// configuration has changed, most likely because of user changes.
		update();
	}

	//
	// Update everything. This is usually called to populate things
	// for the first time or to refresh things after a configuration
	// change.
	//
	private void update()
	{
		expview.setAntiAlias(config.isAntiAlias());
		expview.setUseLargeIcons(config.isUseLargeIcons());

		ToolTipManager.sharedInstance().setEnabled(config.isShowToolTips());

		// ick ick ick .. needs to be fixed.
		DatabaseModel d = (DatabaseModel) dbcombobox.getSelectedItem();
		dbcbm = new DatabaseComboBoxModel(config);
		// Prevent the connection to be meddled with due to an spurious
		// event generated by 1.3.1 Java (both IBM and Sun)
		// by unresgistering us as listeners while setting the model
		// This does not happen with Sun 1.4.1
		dbcombobox.removeActionListener(this);
		dbcombobox.setModel(dbcbm);
		dbcombobox.addActionListener(this);
		if (d != null && config.getDatabaseModel(d.getName()) != null) {
			// Prevent an attempt to reopen the connection unecessarily
			if (!starting_up && !new_config)
				dbcombobox.removeActionListener(this);
 			dbcombobox.setSelectedItem(config.getDatabaseModel(d.getName()));
			// Other combobox events require that a connection be stablished
			if (!starting_up && !new_config)
				dbcombobox.addActionListener(this);
		} else {
			dbcombobox.setSelectedIndex(0);
		}
			
		// Check for button reconfig
		if (config.isAnalyze()) {
			analyzeAction.configureToolbarButton(explainButton);
		} else {
			explainAction.configureToolbarButton(explainButton);
		}

		modifyFileMenu(true);
	}

	//
	// Set the configuration model. Observer handling is done and
	// update is called to refresh.
	//
	private void setConfigurationModel(ConfigurationModel m)
	{
		config = m;
		config.addObserver(this);
		// We normally do not reopen a connection when things in the
		// configuration model change but in this case we _may_ have
		// a different set of connection parameters for the database
		// we are currently connected so we may need to reconnect
		new_config = true;
		update();
		new_config = false;
	}

	//
	// What to do when exiting the program.
	//
	private void quit()
	{
	        // Check for unsaved work before leaving
            // Don't use modstatus as we want to ignore empty text and
            // modifications that were undone
			if ((query != null) &&
				(query.getQuery() != null) &&
				(!query_ta.getText().trim().equals("")) &&
				(!query_ta.getText().equals(query.getQuery())))
			{
				// Ask for confirmation
        	    int result = JOptionPane.showConfirmDialog(this, 
		    					(query.isNew()
								? ExplainResources.getString(
									ExplainResources.MODIFIED_TEXT)
								: ExplainResources.getString(
									ExplainResources.MODIFIED_FILE,
									query.getFile().getName())
								),
                                ExplainResources.getString(
									ExplainResources.MODIFIED_FILE_TITLE),
								JOptionPane.YES_NO_OPTION);
            	switch (result) {
                	case JOptionPane.CLOSED_OPTION:
            	    case JOptionPane.NO_OPTION:
                	    break;
            	    case JOptionPane.YES_OPTION:
                	    save();
                    	break;
            	}
        	}

        quit_program();
	}

	//
	// What to do when exiting the program.
	//
	private void quit_program()
	{
		// close open connection if there is one
		if (cm.isConnectionOpen())
			cm.closeConnection();
			
		// Save configuration if it has changes
		saveconfig();

		// remove lock if we are owner
		if (lockfile != null)
			lockfile.delete();

		System.exit(0);
	}

	//
	// about the program
	//
	private void about()
	{
		aboutdialog.show();
	}

	//
	// help about the program
	//
	private void help()
	{
		helpwindow.show();
	}

	//
	// refresh views
	//
	private void refresh()
	{
		expview.repaint();
	}

	//
	// configure
	//
	private void configure()
	{
		ConfigurationModel cl = (ConfigurationModel) config.clone();
		prefs.setConfigurationModel(cl);
		prefs.show();

		switch (prefs.getResult())
		{
			case PreferencesDialog.OK_OPTION:
				// need to save
				setConfigurationModel(cl);
				saveconfig();
				break;
			case PreferencesDialog.CANCEL_OPTION:
			case PreferencesDialog.CLOSED_OPTION:
				// nothing to be done
				break;
			default:
				// shouldn't get this
		}
	}

	//
	// configure database connections
	//
	private void configure_databases()
	{
		ConfigurationModel cl = (ConfigurationModel) config.clone();
		dbases.setConfigurationModel(cl);
		dbases.show();

		switch (dbases.getResult())
		{
			case DatabasesDialog.OK_OPTION:
				// need to save
				setConfigurationModel(cl);
				saveconfig();
				break;
			case DatabasesDialog.CANCEL_OPTION:
			case DatabasesDialog.CLOSED_OPTION:
				// nothing to be done
				break;
			default:
				// shouldn't get this
		}
	}

	//
	// configure planner options
	//
	private void configure_planner_options()
	{
		PlannerOptions popts = cm.getPlannerOptions();
		poptions.setPlannerOptions(cm.getDefaultPlannerOptions(), popts);
		poptions.show();

		switch (poptions.getResult())
		{
			case PlannerOptionsDialog.OK_OPTION:
				cm.setPlannerOptions(poptions.getPlannerOptions());
				break;
			case PlannerOptionsDialog.CANCEL_OPTION:
			case PlannerOptionsDialog.CLOSED_OPTION:
				// nothing to be done
				break;
			default:
				// shouldn't get this
		}
	}

	//
	// save configuration
	//
	private void saveconfig()
	{
		if (canSaveConfiguration)
		{
			try {
				config.save(conffile);
			} catch (IOException ex) {
				ex.printStackTrace();
			}
		}
	}

	//
	// create lock file
	//
	private void createlock()
	{
		canSaveConfiguration = false;
		String errorstr = "", errortitle = "";
		lockfile = new File(ExplainResources.getString(
								ExplainResources.CONFIGURATION_DIR),
							ExplainResources.getString(
								ExplainResources.FILE_LOCK)
							);

		// try to create lock file. if that fails, then popup a
		// warning dialog and make sure that program is unable to
		// save changes. otherwise, set the lockfile to be deleted
		// on exit and make it so we can save configuration changes.
		try {
			if (lockfile.createNewFile())
			{
				lockfile.deleteOnExit();
				canSaveConfiguration = true;
			}
			else
			{
				canSaveConfiguration = false;
				lockfile = null;
				errorstr = ExplainResources.getString(
							   ExplainResources.ERROR_LOCK_EXISTS);
				errortitle = ExplainResources.getString(
								ExplainResources.ERROR_LOCK_EXISTS_TITLE);
			}
		} catch (Exception ex) {
			canSaveConfiguration = false;
			lockfile = null;
			errorstr = ExplainResources.getString(
						   ExplainResources.ERROR_CREATE_LOCK);
			errortitle = ExplainResources.getString(
							 ExplainResources.ERROR_CREATE_LOCK_TITLE);
		}	
				
		if (!canSaveConfiguration)
		{
			JOptionPane.showMessageDialog(null,
										  errorstr,
										  errortitle,
										  JOptionPane.WARNING_MESSAGE
										 );
		}
	}

	//
	// explain a query
	//
	private void explain(boolean analyze)
	{
		try {
			if (cm.isConnectionOpen())
			{
				String q = query_ta.getText();

				// is there selected text?
				String q2 = query_ta.getSelectedText();
				if (q2 != null)
					q = q2;
					
				exp.setQuery(q);
				exp.setPlannerOptions(cm.getPlannerOptions());
				exp.explain(cm.getConnection(), analyze);
			}
		} catch (ExplainException ex) {
			String s = RHDBUtils.stringRemove(ex.getMessage(), '\t');
			JOptionPane.showMessageDialog(
				this,
				s,
				ExplainResources.getString(
					ExplainResources.ERROR_GENERAL_TITLE),
				JOptionPane.ERROR_MESSAGE);
		}
	}
	
	//
	// Open a new file, either a query (.sql) or a plan (.pln)
	//
	private void open(File f)
	{
		if ((query != null) &&
			(query.getQuery() != null) &&
			(!query_ta.getText().trim().equals("")) &&
			(!query_ta.getText().equals(query.getQuery())))
		{
			// Ask for confirmation
            int result = JOptionPane.showConfirmDialog(
							this, 
		    				(query.isNew()
							? ExplainResources.getString(
								ExplainResources.MODIFIED_TEXT)
							: ExplainResources.getString(
								ExplainResources.MODIFIED_FILE,
							query.getFile().getName())
							),
                            ExplainResources.getString(
								ExplainResources.MODIFIED_FILE_TITLE),
							JOptionPane.YES_NO_CANCEL_OPTION);
            switch (result) {
                case JOptionPane.CLOSED_OPTION:
                case JOptionPane.CANCEL_OPTION:
                    return;
                case JOptionPane.NO_OPTION:
                    break;
                case JOptionPane.YES_OPTION:
                    save();
                    break;
            }
        }

		String extension = RHDBUtils.getExtension(f);
        
		if ((extension != null) && (extension.equalsIgnoreCase(planext)))
		{
			// Reload the Explain object
			try {
				ObjectInputStream in = new ObjectInputStream(
										   new FileInputStream(f));
		
				exp = (Explain) in.readObject();
				in.close();
			} catch (ClassNotFoundException ice) {
				System.out.println(ice.getMessage());
				JOptionPane.showMessageDialog(
					this,
					ExplainResources.getString(
						ExplainResources.ERROR_OPEN_FILE,
						f.getName(),
						ExplainResources.getString(
							ExplainResources.ERROR_OPEN_FILE_BADVERSION)),
					ExplainResources.getString(
						ExplainResources.ERROR_OPEN_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
				return;
			} catch (StreamCorruptedException sce) {
				JOptionPane.showMessageDialog(
					this,
					ExplainResources.getString(
						ExplainResources.ERROR_OPEN_FILE,
						f.getName(),
						ExplainResources.getString(
							ExplainResources.ERROR_OPEN_FILE_CORRUPTED)),
					ExplainResources.getString(
						ExplainResources.ERROR_OPEN_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
				return;
			} catch (IOException ioe) {
				JOptionPane.showMessageDialog(
					this,
				 	ExplainResources.getString(
						ExplainResources.ERROR_OPEN_FILE,
						f.getName(),
						ioe.getMessage()),
					ExplainResources.getString(
						ExplainResources.ERROR_OPEN_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
	            config.removeLast(f);
	            newquery();
		    	return;
			}

			// Load the query, but from the Explain object, not the file			
			query = new QueryModel(f, exp.getQuery());
			// If there is a connection opened
			// we must update its planner options to match
			// what the loaded plan has
			if (cm.isConnectionOpen())
				cm.setPlannerOptions(new PlannerOptions(
											cm.getDefaultPlannerOptions(),
											exp.getPlannerOptions())
									);
		}
		else // Everything else is assumed to be a query file
		{
			try {
				query = new QueryModel(f);
			} catch (QueryModelException ex) {
				JOptionPane.showMessageDialog(
					this,
					ExplainResources.getString(
						ExplainResources.ERROR_OPEN_FILE,
						f.getName(),
						ex.getMessage()),
				 	ExplainResources.getString(
						ExplainResources.ERROR_OPEN_FILE_TITLE),
				 	JOptionPane.ERROR_MESSAGE);
	            config.removeLast(f);
	            newquery();
		    	return;
			}

			exp = new Explain(query.getQuery());
		}

		exp.addExplainListener(this);
		exp.addObserver(expview);
		expview.setExplain(exp);
		query_ta.setText(query.getQuery());
		query_ta.setCaretPosition(0);
		setTitle(ExplainResources.getString(ExplainResources.PROGRAM_TITLE,
											query.getFile().getName())
				);
		setStatementBoxLabel(false);	// Update text area label
        config.setLastQuery(true);      // We currently have a query in the
										// text area
		if (config.getNumLast() == 0 ||
			!query.getFile().getAbsolutePath().equals(config.getLast(0).getAbsolutePath()))
		{
			config.addLast(query.getFile());
		}
	}

	//
	// create a new query
	//
	private void newquery()
	{
		if ((query != null) &&
			(query.getQuery() != null) &&
			(!query_ta.getText().trim().equals("")) &&
			(!query_ta.getText().equals(query.getQuery())))
		{
			// Ask for confirmation
            int result = JOptionPane.showConfirmDialog(
							this, 
		    				(query.isNew()
							? ExplainResources.getString(
								ExplainResources.MODIFIED_TEXT)
							: ExplainResources.getString(
								ExplainResources.MODIFIED_FILE,
								query.getFile().getName())
							),
                            ExplainResources.getString(
								ExplainResources.MODIFIED_FILE_TITLE),
							JOptionPane.YES_NO_CANCEL_OPTION);
            switch (result) {
                case JOptionPane.CLOSED_OPTION:
                case JOptionPane.CANCEL_OPTION:
                    return;
                case JOptionPane.NO_OPTION:
                    break;
                case JOptionPane.YES_OPTION:
                    save();
                    break;
            }
		}
		query = new QueryModel();
		query_ta.setText("");
		setStatementBoxLabel(false);    // Update text area label
        config.setLastQuery(false);     // We currently do not have a query
										// in the text area
        // delegate this to an explain controller?
		exp = new Explain("");
		exp.addExplainListener(this);
		exp.addObserver(expview);
		expview.setExplain(exp);
		setTitle(ExplainResources.getString(
					ExplainResources.PROGRAM_TITLE,
					ExplainResources.getString(
						ExplainResources.DEFAULT_FILE))
				);
		// If a connection is opened,
		// reset its planner options to the default
		if (cm.isConnectionOpen())
			cm.setPlannerOptions(new PlannerOptions(
										cm.getDefaultPlannerOptions())
								);
	}

	//
	// save current query (QueryModel object) or a plan (Explain object)
	// to a file
	//
	private void save()
	{
        // If it is a new query we need a file name first,
		// so it is more like a saveas()
		if (query.isNew())
		{
			saveas();
			return;
		}
		
		String extension = RHDBUtils.getExtension(query.getFile());
        
		if ((extension != null) && (extension.equalsIgnoreCase(planext)))
		{
			// It isa plan file
			if ((exp == null) || (exp.getExplainTree() == null)) {
				JOptionPane.showMessageDialog(this,
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE,
						query.getFile().getName(),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_NOPLAN)),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
				return;
			}

			// Check if the query in the text area matches the one in the plan
			// If not disallow the operation to prevent a mismatch between the
			// query and the plan tree in a plan file.
			if ((exp.getQuery() == null) ||
				(query_ta.getText().trim().equals("")) ||
				(!query_ta.getText().equals(exp.getQuery())))
			{
				JOptionPane.showMessageDialog(this,
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE,
						query.getFile().getName(),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_NOMATCH)),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
				return;
			}
			
			// Get the full path name of the file we are writting to
			String cannonicalFileName;
			try {
				cannonicalFileName = query.getFile().getCanonicalPath();
			}
			catch (IOException ioe) {
				JOptionPane.showMessageDialog(this,
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE,
						query.getFile().getName(),
						ioe.getMessage()),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
				return;
			}
			
			// Now try and write the file
			try {
				ObjectOutputStream out = new ObjectOutputStream(
					new FileOutputStream(cannonicalFileName));
		
				out.writeObject(exp);
				out.close();
			}
			catch (IOException ioe) {
				JOptionPane.showMessageDialog(this,
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE,
						query.getFile().getName(),
						ioe.getMessage()),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
				return;
			}
			
			// Update query model
            query.setQuery(query_ta.getText());
	    	// Update text area label
			setStatementBoxLabel(false);
		}
		else // Everything else is assumed to be a query file
		{
			// It is a query file
			
			// Update query model
            query.setQuery(query_ta.getText());

            try {
                query.save();
            } catch (QueryModelException ex) {
                JOptionPane.showMessageDialog(this,
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE,
						query.getFile().getName(),
						ex.getMessage()),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
            }
	    
	    	// Update text area label
			setStatementBoxLabel(false);
        }
	}

	//
	// save current query (QueryModel object) or plan (Explain object)
	//
	private void saveas()
	{
		File f;
		
		// First we need a file name
		// If there is a plan, use a multi filter,
		// Otherwise only offer to choose query files.
		if ((exp != null) && (exp.getExplainTree() != null)) {
			if (spfilechooser.showSaveDialog(this) ==
				JFileChooser.APPROVE_OPTION)
			{
				f = spfilechooser.getSelectedFile();
				if (spfilechooser.getFileFilter() instanceof SQLFileFilter)
					f = RHDBUtils.addExtension(f,
						ExplainResources.getString(
							ExplainResources.SQL_EXTENSION));
				if (spfilechooser.getFileFilter() instanceof PlanFileFilter)
					f = RHDBUtils.addExtension(f,
						ExplainResources.getString(
							ExplainResources.PLAN_EXTENSION));
		
				// Now we can save it
				saveas(f);
			}
		}
		else
		{
			if (filechooser.showSaveDialog(this) == JFileChooser.APPROVE_OPTION)
			{
				f = filechooser.getSelectedFile();
				if (filechooser.getFileFilter() instanceof SQLFileFilter)
					f = RHDBUtils.addExtension(f,
						ExplainResources.getString(
							ExplainResources.SQL_EXTENSION));
		
				// Now we can save it
				saveas(f);
			}
		}
	}

	//
	// save current query or plan to a (possibly) new file
	//
	private void saveas(File f)
	{
		if (f.exists())
		{
			int rv = JOptionPane.showConfirmDialog(this,
				ExplainResources.getString(
					ExplainResources.FILE_OVERWRITE,
					f.getName()),
				ExplainResources.getString(
					ExplainResources.FILE_OVERWRITE_TITLE),
				JOptionPane.YES_NO_OPTION);
			if (rv == JOptionPane.NO_OPTION)
				return;
		}

		String extension = RHDBUtils.getExtension(f);
        
		if ((extension != null) && (extension.equalsIgnoreCase(planext)))
		{
			// It is a plan file
			if ((exp == null) || (exp.getExplainTree() == null)) {
				JOptionPane.showMessageDialog(this,
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE,
							 f.getName(),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_NOPLAN)),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
				return;
			}

			// Check if the query in the text area matches the one in the plan
			// If not disallow the operation to prevent a mismatch between the
			// query and the plan tree in a plan file.
			if ((exp.getQuery() == null) ||
				(query_ta.getText().trim().equals("")) ||
				(!query_ta.getText().trim().equals(exp.getQuery())))
			{
				JOptionPane.showMessageDialog(this,
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE,
							 f.getName(),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_NOMATCH)),
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
				return;
			}
			
			try {
				ObjectOutputStream out = new ObjectOutputStream(
					new FileOutputStream(f));
		
				out.writeObject(exp);
				out.close();
			}
			catch (Exception e) {
				e.printStackTrace();
				return;
			}
		}
		else // Everything else is assumed to be a query file
		{
			// It is a query file

			// Update query model so that we save the modified version
			query.setQuery(query_ta.getText());

			try {
				query.saveas(f);
			} catch (QueryModelException ex) {
				JOptionPane.showMessageDialog(this,
					ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE,
						f.getName(),
						ex.getMessage()),
				 	ExplainResources.getString(
						ExplainResources.ERROR_SAVE_FILE_TITLE),
					JOptionPane.ERROR_MESSAGE);
			}
		}
			
		// Update query model
       	query = new QueryModel(f, query_ta.getText());
		// Update window title
		setTitle(ExplainResources.getString(
			ExplainResources.PROGRAM_TITLE,	f.getName()));
    	// Update text area label
		setStatementBoxLabel(false);
		// We now have a query in the text area
        config.setLastQuery(true);
		// Update list of most recently opened files
		if (config.getNumLast() == 0 ||
			!query.getFile().getAbsolutePath().equals(config.getLast(0).getAbsolutePath()))
		{
			config.addLast(query.getFile());
			try {
				config.save(conffile);
			} catch (Exception ex) { }
		}
	}
	//
	// open a connection
	//
	private void opencon()
	{
		DatabaseModel d =
		  (DatabaseModel)((DatabaseModel) dbcombobox.getSelectedItem()).clone();
		// This will close the connection, if one is open
		cm.setDatabaseModel(d);

		if (d instanceof DisconnectedDatabaseModel)
		{
			disconnected();
			return;
		}

		statusbar.setText(ExplainResources.getString(
							  ExplainResources.CONNECTION_OPENING,
							  d.getName())
						 );

		if (d.getPassword() == null)
		{
			passdialog.setDatabaseModel(d);
			passdialog.show();
			if (passdialog.getResult() != PasswordDialog.OK_OPTION)
			{
				disconnected();
				return;
			}
			d.setPassword(passdialog.getPassword());
		}
		
		// Open the new connection
		cm.openConnection();
	}

	private void conopened()
	{
		DatabaseModel d = (DatabaseModel) dbcombobox.getSelectedItem();
		DatabaseModel dclone = cm.getDatabaseModel();

		if (cm.isConnectionOpen())
		{
			// If a plan is loaded, must set the
			// new connection planner options to the plan's values
			// Note that this must be done after the connection process
			// has been completed
			if (!exp.getPlannerOptions().trim().equals(""))
				cm.setPlannerOptions(new PlannerOptions(
											cm.getDefaultPlannerOptions(),
											exp.getPlannerOptions())
									);

			Object[] args = {
							d.getName(), d.getDatabase(), d.getHost(), d.getPort(), d.getUser(),
							cm.getBackendVersion()
							};
			statusbar.setText(ExplainResources.getString(
								  ExplainResources.CONNECTION_OPEN,
								  args)
							 );
			// Now that we know we can connect with this password,
			// save it for the session
			if ((d.getPassword() == null) ||
				(!d.getPassword().equals(dclone.getPassword())))
				d.setPassword(dclone.getPassword());
			explainAction.setEnabled(true);
			analyzeAction.setEnabled(true);
			poptsAction.setEnabled(true);
			// Save index of last database we connected to for next session
			config.setLastDatabase(d.getName());
		}
		else
		{
			String error = RHDBUtils.removeIncluding(cm.getConnectionError(),
													 "Stack Trace:");
			error = RHDBUtils.stringFitWidth(error, 50);

			JOptionPane.showMessageDialog(
				this,
				ExplainResources.getString(
					ExplainResources.CONNECTION_ERROR,
					d.getName(),
					error),
				ExplainResources.getString(
					ExplainResources.CONNECTION_ERROR_TITLE),
				JOptionPane.ERROR_MESSAGE);
			disconnected();
		}
	}

	private void disconnected()
	{
		statusbar.setText(ExplainResources.getString(
							  ExplainResources.CONNECTION_CLOSED)
						 );
		dbcombobox.setSelectedIndex(0);
		cm.setDatabaseModel((DatabaseModel) dbcombobox.getSelectedItem());
		explainAction.setEnabled(false);
		analyzeAction.setEnabled(false);
		poptsAction.setEnabled(false);
		// Record that we were not connected for next session
		config.setLastDatabase("");
		update();
	}

	//
	// fit width
	//
	private void fitwidth()
	{
		expview.setZoomMode(TreeDisplay.FIT_WIDTH);
	}

	//
	// fit in window
	//
	private void fitwindow()
	{
		expview.setZoomMode(TreeDisplay.FIT_WINDOW);
	}

	//
	// actual size
	//
	private void actualsize()
	{
		expview.setZoomMode(TreeDisplay.ACTUAL_SIZE);
	}

	//
	// zoom to a given level
	//
	private void zoom(double level)
	{
		try {
			if (level == 1.0) throw new Exception();
			expview.setZoomLevel(level);
			noselectMenuitem.setSelected(true);
		} catch (Exception ex) {
			actualsize();
			actualsizeMenuitem.setSelected(true);
		}
	}

	//
	// find what zoom index is next if we want to zoom in
	//
	private int findNextZoomInIndex()
	{
		double zoom = expview.getZoomLevel();
		int rv = -1;

		if (zoom < zooms[0])
			rv = 0;
		if (zoom == zooms[0])
			rv = 1;
		else if (zoom >= zooms[zooms.length - 1])
			rv = zooms.length - 1;
		else
		{
			for (int i = 0; i < zooms.length - 1; i++)
			{
				if (zoom >= zooms[i] && zoom < zooms[i+1])
				{
					rv = i + 1;
					break;
				}
			}
		}

		return rv;
	}

	//
	// find what zoom index is next if we want to zoom out
	//
	private int findNextZoomOutIndex()
	{
		double zoom = expview.getZoomLevel();
		int rv = -1;

		if (zoom <= zooms[0])
			rv = 0;
		else if (zoom > zooms[zooms.length-1])
			rv = zooms.length - 1;
		else if (zoom == zooms[zooms.length - 1])
			rv = zooms.length - 1 - 1;
		else
		{
			for (int i = 1; i < zooms.length; i++)
			{
				if (zoom > zooms[i-1] && zoom <= zooms[i])
				{
					rv = i - 1;
					break;
				}
			}
		}

		return rv;
	}

	//
	// ExplainViewListener
	//

	/**
	 * EXPLAIN dialog has been closed.
	 *
	 * @param e an <code>ExplainViewEvent</code> value
	 */
	public void explainViewExplainWindowClosed(ExplainViewEvent e)
	{
		Object source = e.getSource();
		if (source != expview)
			return;

		explainMenuitem.setSelected(false);
	}

	/**
	 * EXPLAIN VERBOSE dialog has been closed.
	 *
	 * @param e an <code>ExplainViewEvent</code> value
	 */
	public void explainViewExplainVerboseWindowClosed(ExplainViewEvent e)
	{
		Object source = e.getSource();
		if (source != expview)
			return;

		explainVerboseMenuitem.setSelected(false);
	}

	/**
	 * Planner Options dialog has been closed.
	 *
	 * @param e an <code>ExplainViewEvent</code> value
	 */
	public void explainViewOptionsWindowClosed(ExplainViewEvent e)
	{
		Object source = e.getSource();
		if (source != expview)
			return;

		optionsMenuitem.setSelected(false);
	}
	
	/**
	 * Overview dialog has been closed.
	 *
	 * @param e an <code>ExplainViewEvent</code> value
	 */
	public void explainViewOverviewWindowClosed(ExplainViewEvent e)
	{
		Object source = e.getSource();
		if (source != expview)
			return;

		overviewButton.setSelected(false);
	}

	/**
	 * Explain view has been zoomed.
	 *
	 * @param e an <code>ExplainViewEvent</code> value
	 */
	public void explainViewZoomed(ExplainViewEvent e)
	{
		Object source = e.getSource();

		if (source != expview)
			return;

		// set the text in the editor to avoid an action event
		// being fired by the combo box, which can mess up
		// the zoom radio group
		zoomcombobox.getEditor().setItem(
									formatter.format(expview.getZoomLevel())
									);

		double level = expview.getZoomLevel();
				
		// enable/disable zoom in/out buttons based on zoom level
		if (level <= zooms[0])
		{
			zoominAction.setEnabled(true);
			zoomoutAction.setEnabled(false);
		}
		else if (level >= zooms[zooms.length-1])
		{
			zoominAction.setEnabled(false);
			zoomoutAction.setEnabled(true);
		}
		else
		{
			zoominAction.setEnabled(true);
			zoomoutAction.setEnabled(true);
		}
	}

	//
	// ItemListener
	//
	
	/**
	 * Item listener
	 *
	 * @param e an <code>ItemEvent</code> value
	 */
	public void itemStateChanged(ItemEvent e)
	{
		Object source = e.getItemSelectable();

		if (source == overviewButton || source == overviewMenuitem)
		{
			switch (e.getStateChange())
			{
				case ItemEvent.SELECTED:
					expview.showOverview();
					break;
				case ItemEvent.DESELECTED:
					expview.hideOverview();
					break;
				default:
					// do nothing
			}
		}
		else if (source == optionsMenuitem)
		{
			switch (e.getStateChange())
			{
				case ItemEvent.SELECTED:
					expview.showOptionsWindow();
					break;
				case ItemEvent.DESELECTED:
					expview.hideOptionsWindow();
					break;
				default:
					// do nothing
			}
		}
		else if (source == explainMenuitem)
		{
			switch (e.getStateChange())
			{
				case ItemEvent.SELECTED:
					expview.showExplainWindow();
					break;
				case ItemEvent.DESELECTED:
					expview.hideExplainWindow();
					break;
				default:
					// do nothing
			}
		}
		else if (source == explainVerboseMenuitem)
		{
			switch (e.getStateChange())
			{
				case ItemEvent.SELECTED:
					expview.showExplainVerboseWindow();
					break;
				case ItemEvent.DESELECTED:
					expview.hideExplainVerboseWindow();
					break;
				default:
					// do nothing
			}
		}
	}

	//
	// ExplainListener
	//
	
	/**
	 * Explain listener (handles ExplainEvent's).
	 *
	 * @param e an <code>ExplainEvent</code> value
	 */
	public void statusChanged(ExplainEvent e)
	{
		if (e.getStatus().equals("running"))
		{
			//
			// query execution thread started
			//

			running = true;
			
			explainAction.setEnabled(false);
			analyzeAction.setEnabled(false);
			stopAction.setEnabled(true);
			
			newAction.setEnabled(false);
			openAction.setEnabled(false);
			saveAction.setEnabled(false);
			saveasAction.setEnabled(false);
			aboutAction.setEnabled(false);
			exitAction.setEnabled(false);
			prefsAction.setEnabled(false);
			dbasesAction.setEnabled(false);
			poptsAction.setEnabled(false);
			zoominAction.setEnabled(false);
			zoomoutAction.setEnabled(false);
			helpAction.setEnabled(false);
			refreshAction.setEnabled(false);
			dbcombobox.setEnabled(false);
			fitwindowMenuitem.setEnabled(false);
			fitwidthMenuitem.setEnabled(false);

			modifyFileMenu(false);
		}
		else if (e.getStatus().equals("idle"))
		{
			//
			// query execution thread finished
			//

			running = false;

			// Only re-enable the Explain button if the
			// connection is still open (it may have dropped
			// due to a communication link error)
			if (cm.isConnectionOpen())
			{
				explainAction.setEnabled(true);
				analyzeAction.setEnabled(true);
			}
			stopAction.setEnabled(false);
			
			newAction.setEnabled(true);
			openAction.setEnabled(true);
			saveAction.setEnabled(true);
			saveasAction.setEnabled(true);
			aboutAction.setEnabled(true);
			exitAction.setEnabled(true);
			prefsAction.setEnabled(true);
			dbasesAction.setEnabled(true);
			poptsAction.setEnabled(true);
			zoominAction.setEnabled(true);
			zoomoutAction.setEnabled(true);
			helpAction.setEnabled(true);
			refreshAction.setEnabled(true);
			dbcombobox.setEnabled(true);
			fitwindowMenuitem.setEnabled(true);
			fitwidthMenuitem.setEnabled(true);

			modifyFileMenu(true);
		}
		else if (e.getStatus().equals("error"))
		{
			// Some exception
			Exception ex = e.getException();

			// Check for a communication link error,
			// in which case we close the connection
			if (ex.getMessage().startsWith("Communication link error")) {

				// close open connection if there is still one
				if (cm.isConnectionOpen())
					cm.closeConnection();

				// Update GUI to reflect that we are disconnected
				disconnected();
			}

			String s = RHDBUtils.stringRemove(ex.getMessage(), '\t');
			JOptionPane.showMessageDialog(
				this,
				s,
				ExplainResources.getString(
					ExplainResources.ERROR_GENERAL_TITLE),
				JOptionPane.ERROR_MESSAGE
				);
		}
		else
		{
			//
			// we shouldn't get this, but just in case ...
			//

			System.err.println("vise: unknown explain event: " + e.getStatus());
		}
	}

	//
	// ActionListener
	//
	
	/**
	 * Action listener (handles ActionEvent's).
	 *
	 * @param e an <code>ActionEvent</code> value
	 */
	public void actionPerformed(ActionEvent e)
	{
		String command = e.getActionCommand();

		// first check if we are being asked to open a dynamic recent
		// file
		boolean found = false;
		for (int i = 0; i < config.getNumLast(); i++)
		{
			if (command.equals("_VELastFile" + (i + 1)))
			{
				open(config.getLast(i));
				found = true;
				break;
			}
		}
		if (found)
			return;
			
		if (command.equals(dbcombobox.getActionCommand()))
		{
			//
			// new selection in combo box (database list)
			//

			// or the database is the same we were connected to?
			DatabaseModel dbnew = (DatabaseModel) dbcombobox.getSelectedItem();
			DatabaseModel dbold = cm.getDatabaseModel();
			if ((dbold != null) &&
				!(dbold instanceof DisconnectedDatabaseModel) &&
				(dbnew.getName().equals(dbold.getName())) &&
				(dbnew.getDatabase().equals(dbold.getDatabase())) &&
				(dbnew.getHost().equals(dbold.getHost())) &&
				(dbnew.getPort().equals(dbold.getPort())) &&
				(dbnew.getUser().equals(dbold.getUser()))
				)
				return; 

			// We must do this or the popup stays there while we are opening
			dbcombobox.setPopupVisible(false);

			// Invalidate the plan unless it is [Disconnected]
			// or we are just starting up
			if ((!starting_up) && (dbcombobox.getSelectedIndex() != 0))
				exp.setQuery(exp.getQuery());

			// Open the new connection
			opencon();
		}
		else if (command.equals(cm.getActionCommand()))
		{
			//
			// notification from connection model
			//

			conopened();
		}
		else if (command.equals(newAction.getActionCommand()))
		{
			//
			// new query
			//

			newquery();
		}
		else if (command.equals(openAction.getActionCommand()))
		{
			//
			// open a query or plan
			//

			if (mfilechooser.showOpenDialog(this) == JFileChooser.APPROVE_OPTION)
				open(mfilechooser.getSelectedFile());
		}
		else if (command.equals(saveAction.getActionCommand()))
		{
			//
			// save current query or plan
			//

			save();
		}
		else if (command.equals(saveasAction.getActionCommand()))
		{
			//
			// save current query or plan as ...
			//

			saveas();
		}
		else if (command.equals(exitAction.getActionCommand()))
		{
			//
			// quit
			//
			
	        // Check for unsaved work before leaving
			if ((query != null) &&
				(query.getQuery() != null) &&
				(!query_ta.getText().trim().equals("")) &&
				(!query_ta.getText().equals(query.getQuery())))
			{
				// Ask for confirmation
        	    int result = JOptionPane.showConfirmDialog(
								this,
		    					(query.isNew()
								? ExplainResources.getString(
									ExplainResources.MODIFIED_TEXT)
								: ExplainResources.getString(
									ExplainResources.MODIFIED_FILE,
									query.getFile().getName())
								),
                                ExplainResources.getString(
									ExplainResources.MODIFIED_FILE_TITLE),
								JOptionPane.YES_NO_CANCEL_OPTION
								);
            	switch (result) {
                	case JOptionPane.CLOSED_OPTION:
            	    case JOptionPane.CANCEL_OPTION:
                	    return;
            	    case JOptionPane.NO_OPTION:
                	    break;
            	    case JOptionPane.YES_OPTION:
                	    save();
                    	break;
            	}
        	}

			quit_program();
		}
		else if (command.equals(aboutAction.getActionCommand()))
		{
			//
			// about us
			//

			about();
		}
		else if (command.equals(helpAction.getActionCommand()))
		{
			//
			// help
			//

			help();
		}
		else if (command.equals(refreshAction.getActionCommand()))
		{
			//
			// refresh views
			//

			refresh();
		}
		else if (command.equals(explainAction.getActionCommand()))
		{
			//
			// explain current query
			//

			explain(false);
		}
		else if (command.equals(analyzeAction.getActionCommand()))
		{
			//
			// explain analyze current query
			//

			explain(true);
		}
		else if (command.equals(stopAction.getActionCommand()))
		{
			//
			// interrupt query execution
			//

			// Interrupt query execution
			System.out.println("Stop requested");
			exp.cancel();
		}
		else if (command.equals(prefsAction.getActionCommand()))
		{
			//
			// configure the program preferences
			//

			configure();
		}
		else if (command.equals(dbasesAction.getActionCommand()))
		{
			//
			// configure the program database connections
			//

			configure_databases();
		}
		else if (command.equals(poptsAction.getActionCommand()))
		{
			//
			// configure the planner options
			//

			configure_planner_options();
		}
		else if (command.equals(zoominAction.getActionCommand()))
		{
			//
			// zoom in
			//

			zoom(zooms[findNextZoomInIndex()]);
		}
		else if (command.equals(zoomoutAction.getActionCommand()))
		{
			//
			// zoom out
			//

			zoom(zooms[findNextZoomOutIndex()]);
		}
		else if (command.equals(actualsizeMenuitem.getActionCommand()))
		{
			//
			// zoom to 100%
			//

			actualsize();
		}
		else if (command.equals(fitwindowMenuitem.getActionCommand()))
		{
			//
			// zoom so that the whole tree just fits in the display area
			//

			fitwindow();
		}
		else if (command.equals(fitwidthMenuitem.getActionCommand()))
		{
			//
			// zoom so that the width of the tree just fits in the display area
			//

			fitwidth();
		}
		else if (command.equals(zoomcombobox.getActionCommand()))
		{
			//
			// user-selected zoom
			//

			Object sel = zoomcombobox.getSelectedItem();
			double d = ((Double) sel).doubleValue();

			zoom(d);
		}
		else if (command.equals(topMenuitem.getActionCommand()))
		{
			//
			// root node at top
			//

			expview.setOrientation(ExplainView.TOP);
		}
		else if (command.equals(bottomMenuitem.getActionCommand()))
		{
			//
			// root node at bottom
			//

			expview.setOrientation(ExplainView.BOTTOM);
		}
		else if (command.equals(leftMenuitem.getActionCommand()))
		{
			//
			// root node at left
			//

			expview.setOrientation(ExplainView.LEFT);
		}
		else if (command.equals(rightMenuitem.getActionCommand()))
		{
			//
			// root node at right
			//

			expview.setOrientation(ExplainView.RIGHT);
		}
		else
		{
			//
			// we shouldn't get this, but just in case ...
			//

			System.err.println("vise: unknown action command: " + command);
		}
	}

	//
	// create the application toolbar
	//
	private JToolBar createToolBar()
	{
		JToolBar toolbar = new JToolBar();

		toolbar.add(newAction.getToolbarButton());
		toolbar.add(openAction.getToolbarButton());
		toolbar.add(saveAction.getToolbarButton());
		
		toolbar.addSeparator();

		if (config.isAnalyze()) {
			explainButton = analyzeAction.getToolbarButton();
		} else {
			explainButton = explainAction.getToolbarButton();
		}
		toolbar.add(explainButton);
		toolbar.add(stopAction.getToolbarButton());

		toolbar.addSeparator();

		toolbar.add(overviewButton);
		
		toolbar.addSeparator();

		JToggleButton button;
		ButtonGroup bg = new ButtonGroup();

		button = new JToggleButton();
		button.setMargin(new Insets(0, 0, 0, 0));
		button.setIcon(
			new ImageIcon(
				getClass().getResource(
					ExplainResources.getString(ExplainResources.IMAGE_PATH) +
					ExplainResources.getString(
						ExplainResources.ORIENTATION_TOP_ICON)
					)
				)
			);
		button.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_TOP_TOOLTIP)
			);
		button.setModel(topMenuitem.getModel());
		bg.add(button);
		toolbar.add(button);
		
		button = new JToggleButton();
		button.setMargin(new Insets(0, 0, 0, 0));
		button.setIcon(
			new ImageIcon(
				getClass().getResource(
					ExplainResources.getString(ExplainResources.IMAGE_PATH) +
					ExplainResources.getString(
						ExplainResources.ORIENTATION_BOTTOM_ICON)
					)
				)
			);
		button.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_BOTTOM_TOOLTIP)
			);
		button.setModel(bottomMenuitem.getModel());
		bg.add(button);
		toolbar.add(button);
		
		button = new JToggleButton();
		button.setMargin(new Insets(0, 0, 0, 0));
		button.setIcon(
			new ImageIcon(
				getClass().getResource(
					ExplainResources.getString(ExplainResources.IMAGE_PATH) +
					ExplainResources.getString(
						ExplainResources.ORIENTATION_LEFT_ICON)
					)
				)
			);
		button.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_LEFT_TOOLTIP)
			);
		button.setModel(leftMenuitem.getModel());
		bg.add(button);
		toolbar.add(button);
		
		button = new JToggleButton();
		button.setMargin(new Insets(0, 0, 0, 0));
		button.setIcon(
			new ImageIcon(
				getClass().getResource(
					ExplainResources.getString(ExplainResources.IMAGE_PATH) +
					ExplainResources.getString(
						ExplainResources.ORIENTATION_RIGHT_ICON)
					)
				)
			);
		button.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_RIGHT_TOOLTIP)
			);
		button.setModel(rightMenuitem.getModel());
		bg.add(button);
		toolbar.add(button);

		toolbar.addSeparator();

		toolbar.add(zoominAction.getToolbarButton());
		toolbar.add(zoomoutAction.getToolbarButton());

		zoomcombobox = new FixedSizeComboBox(80, 24);
		zoomcombobox.setEditable(true);
		zoomcombobox.setActionCommand("ZoomComboBox");
		zoomcombobox.addActionListener(this);
		zoomcombobox.setMaximumSize(new Dimension(0, 24));
		zoomcombobox.setRenderer(new ZoomComboBoxRenderer());
		toolbar.add(zoomcombobox);
		
		toolbar.addSeparator();

		dbcombobox = new FixedSizeComboBox(100, 24);
		dbcbm = new DatabaseComboBoxModel(config);
		dbcombobox.setModel(dbcbm);
		dbcombobox.setActionCommand("ConnectionEvent");
		dbcombobox.addActionListener(this);
		toolbar.add(dbcombobox);
		
		return toolbar;
	}

	//
	// build the file menu
	//
	private void modifyFileMenu(boolean active)
	{
		filemenu.removeAll();

		filemenu.add(newAction.getMenuItem());
		filemenu.add(openAction.getMenuItem());

		filemenu.addSeparator();

		filemenu.add(saveAction.getMenuItem());
		filemenu.add(saveasAction.getMenuItem());
		
		filemenu.addSeparator();
		
		// add menu items
		if (config != null && config.getNumLast() > 0)
		{
			JMenuItem menuitem;

			for (int i = 0; i < config.getNumLast(); i++)
			{
				menuitem = new JMenuItem((i + 1) + " " +
										 config.getLast(i).getName());
				menuitem.setMnemonic(49 + i);
				menuitem.setActionCommand("_VELastFile" + (i + 1));
				menuitem.addActionListener(this);
				menuitem.setToolTipText(config.getLast(i).getAbsolutePath());
				filemenu.add(menuitem);
				if (!active)
					menuitem.setEnabled(false);
			}

			filemenu.addSeparator();
		}

		filemenu.add(exitAction.getMenuItem());
	}
	
	//
	// create the application menubar
	//
	private JMenuBar createMenuBar()
	{
		// Menubar
		JMenuBar menubar;
		JMenu menu, submenu;
		JMenuItem menuitem;
		JRadioButtonMenuItem rbmenuitem;

		menubar = new JMenuBar();

		// File menu
		filemenu = new JMenu(ExplainResources.getString(
								 ExplainResources.MENU_FILE));
		filemenu.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_FILE_MNEMONIC)).intValue()
			);
		menubar.add(filemenu);

		modifyFileMenu(true);

		// Edit menu
		menu = new JMenu(ExplainResources.getString(
							ExplainResources.MENU_EDIT));
		menu.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_EDIT_MNEMONIC)).intValue()
			);
		menubar.add(menu);

		menu.add(cutAction.getMenuItem());
		menu.add(copyAction.getMenuItem());
		menu.add(pasteAction.getMenuItem());

		menu.addSeparator();
		
		menu.add(poptsAction.getMenuItem());

		menu.addSeparator();
		
		menu.add(dbasesAction.getMenuItem());

		menu.addSeparator();
		
		menu.add(prefsAction.getMenuItem());

		// Statement menu
		menu = new JMenu(ExplainResources.getString(
							ExplainResources.MENU_STATEMENT));
		menu.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_STATEMENT_MNEMONIC)).intValue()
			);
		menubar.add(menu);

		menu.add(explainAction.getMenuItem());
		menu.add(analyzeAction.getMenuItem());

		// View menu
		menu = new JMenu(ExplainResources.getString(
							ExplainResources.MENU_VIEW));
		menu.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_VIEW_MNEMONIC)).intValue()
			);
		menubar.add(menu);

		menu.add(zoominAction.getMenuItem());
		menu.add(zoomoutAction.getMenuItem());

		menu.addSeparator();

		menu.add(actualsizeMenuitem);
		menu.add(fitwindowMenuitem);
		menu.add(fitwidthMenuitem);

		menu.addSeparator();

		// orientation submenu
		
		submenu = new JMenu(ExplainResources.getString(
								ExplainResources.MENU_ITEM_ORIENTATION));
		submenu.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_ORIENTATION_MNEMONIC)).intValue()
			);
		menu.add(submenu);

		submenu.add(topMenuitem);
		submenu.add(bottomMenuitem);
		submenu.add(leftMenuitem);
		submenu.add(rightMenuitem);

		menu.addSeparator();

		menu.add(refreshAction.getMenuItem());

		// Window menu
		menu = new JMenu(ExplainResources.getString(
							ExplainResources.MENU_WINDOW));
		menu.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_WINDOW_MNEMONIC)).intValue()
			);
		menubar.add(menu);

		menu.add(overviewMenuitem);
		menu.add(optionsMenuitem);
		menu.add(explainMenuitem);
		menu.add(explainVerboseMenuitem);

		// Some glue for spacing
		menubar.add(Box.createHorizontalGlue());
		
		// Help menu
		menu = new JMenu(ExplainResources.getString(
			ExplainResources.MENU_HELP));
		menu.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_HELP_MNEMONIC)).intValue()
			);
		menubar.add(menu);

		menu.add(helpAction.getMenuItem());
		menu.addSeparator();
		menu.add(aboutAction.getMenuItem());
		
		return menubar;
	}

	//
	// create the main area of the program - currently just one explainview
	//
	private JComponent createMainArea()
	{
		exp = new Explain();
		exp.addExplainListener(this);
		expview = new ExplainView(exp);
		expview.addExplainViewListener(this);
		exp.addObserver(expview);

		JPanel statementBox = createStatementBox();

		JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
		splitPane.setTopComponent(expview);
		splitPane.setBottomComponent(statementBox);
		splitPane.setDividerLocation(350);
		splitPane.setResizeWeight(1);

		return splitPane;
	}

	private JPanel createStatementBox()
	{
		JPanel pane = new JPanel();
		pane.setLayout(new BorderLayout());
		pane.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
		
		query_ta_label = new JLabel();
		query_ta_label.setText(
			ExplainResources.getString(
				ExplainResources.LABEL_SQL_ENTRY_AREA,
				ExplainResources.getString(ExplainResources.DEFAULT_FILE)
				)
			);
		query_ta_label.setBorder(BorderFactory.createEmptyBorder(0, 0, 5, 0));
		query_ta_label.setForeground(Color.black);
		pane.add(query_ta_label, BorderLayout.NORTH);
		
		query_ta = new JTextArea();
		query_ta.setLineWrap(true);
		query_ta.setWrapStyleWord(true);
        query_ta.getDocument().addDocumentListener(new DocumentListener() {
			public void insertUpdate(DocumentEvent de) {
	    		// Show modified marks in text area label
				setStatementBoxLabel(true);
			}
			public void removeUpdate(DocumentEvent de) {
	    		// Show modified marks in text area label
				setStatementBoxLabel(true);
			}
			public void changedUpdate(DocumentEvent de) {
			}
        });

		/* Control-Enter explain's query */
		Object key = "CTRLEnter";
		InputMap imap = query_ta.getInputMap();
		ActionMap amap = query_ta.getActionMap();

		imap.put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, Event.CTRL_MASK),
				 key);
		amap.put(key, explainAction);
		amap.put(key, analyzeAction);

		JScrollPane scroller = new JScrollPane(query_ta);

		pane.add(scroller, BorderLayout.CENTER);

		return pane;
	}

	private void setStatementBoxLabel(boolean modified) {
		String labeltext;
		
		if (modstatus && modified)
			return;  // If we already marked as modified just return
        
        modstatus = modified;
		
		if (modified) {
			labeltext = ExplainResources.LABEL_SQL_ENTRY_AREA_MOD;
		}
		else
		{
			labeltext = ExplainResources.LABEL_SQL_ENTRY_AREA;
		}
		query_ta_label.setText(
			ExplainResources.getString(
				labeltext,
				(query.getFile() == null)
				? ExplainResources.getString(ExplainResources.DEFAULT_FILE)
				: query.getFile().getName()
				)
			);
	}
	
	//
	// create ui
	//
	private void initComponents()
	{
		// shared stuff

		// plan overview buttons
		overviewMenuitem = new JCheckBoxMenuItem();
		overviewMenuitem.setText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWOVERVIEW)
			);
		overviewMenuitem.setAccelerator(
			(KeyStroke)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_SHOWOVERVIEW_ACCEL)
			);
		overviewMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_SHOWOVERVIEW_MNEMONIC)).intValue()
			);
		overviewMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWOVERVIEW_TOOLTIP)
			);
		overviewMenuitem.addItemListener(this);

		overviewButton = new JToggleButton();
		overviewButton.setIcon(
			new ImageIcon(
				getClass().getResource(
					ExplainResources.getString(ExplainResources.IMAGE_PATH) +
					ExplainResources.getString(
						ExplainResources.TOOLBAR_OVERVIEW_ICON)
					)
				)
			);
		overviewButton.setText(null);
		overviewButton.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWOVERVIEW_TOOLTIP)
			);
		overviewButton.setMargin(new java.awt.Insets(0, 0, 0, 0));
		overviewButton.setModel(overviewMenuitem.getModel());
		overviewButton.addItemListener(this);

		// explain output button
		optionsMenuitem = new JCheckBoxMenuItem();
		optionsMenuitem.setText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWOPTIONS)
			);
		optionsMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_SHOWOPTIONS_MNEMONIC)).intValue()
			);
		if (RHDBUtils.isJava14())
			optionsMenuitem.setDisplayedMnemonicIndex(
				((Integer)ExplainResources.getObject(
					ExplainResources.MENU_ITEM_SHOWOPTIONS_MNEMONIC_INDEX)
				).intValue()
				);
		optionsMenuitem.setAccelerator(
			(KeyStroke)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_SHOWOPTIONS_ACCEL)
			);
		optionsMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWOPTIONS_TOOLTIP)
			);
		optionsMenuitem.addItemListener(this);

		// explain output button
		explainMenuitem = new JCheckBoxMenuItem();
		explainMenuitem.setText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWEXPLAIN)
			);
		explainMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_SHOWEXPLAIN_MNEMONIC)).intValue()
			);
		if (RHDBUtils.isJava14())
			explainMenuitem.setDisplayedMnemonicIndex(
				((Integer)ExplainResources.getObject(
					ExplainResources.MENU_ITEM_SHOWEXPLAIN_MNEMONIC_INDEX)
				).intValue()
				);
		explainMenuitem.setAccelerator(
			(KeyStroke)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_SHOWEXPLAIN_ACCEL)
			);
		explainMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWEXPLAIN_TOOLTIP)
			);
		explainMenuitem.addItemListener(this);
		
		// explain verbose button
		explainVerboseMenuitem = new JCheckBoxMenuItem();
		explainVerboseMenuitem.setText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWEXPLAINVB)
			);
		explainVerboseMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_SHOWEXPLAINVB_MNEMONIC)).intValue()
			);
		explainVerboseMenuitem.setAccelerator(
			(KeyStroke)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_SHOWEXPLAINVB_ACCEL)
			);
		explainVerboseMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_SHOWEXPLAINVB_TOOLTIP)
			);
		explainVerboseMenuitem.addItemListener(this);

		// zoom modes
		actualsizeMenuitem = new JRadioButtonMenuItem();
		actualsizeMenuitem.setText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ACTUALSIZE)
			);
		actualsizeMenuitem.setAccelerator(
			(KeyStroke)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_ACTUALSIZE_ACCEL)
			);
		actualsizeMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_ACTUALSIZE_MNEMONIC)).intValue()
			);
		actualsizeMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ACTUALSIZE_TOOLTIP)
			);
		actualsizeMenuitem.setActionCommand("_VEActualSize");
		actualsizeMenuitem.addActionListener(this);

		fitwindowMenuitem = new JRadioButtonMenuItem();
		fitwindowMenuitem.setText(
			ExplainResources.getString(ExplainResources.MENU_ITEM_FITINWINDOW)
			);
		fitwindowMenuitem.setAccelerator(
			(KeyStroke)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_FITINWINDOW_ACCEL)
			);
		fitwindowMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_FITINWINDOW_MNEMONIC)).intValue()
			);
		if (RHDBUtils.isJava14())
			fitwindowMenuitem.setDisplayedMnemonicIndex(
				((Integer)ExplainResources.getObject(
					ExplainResources.MENU_ITEM_FITINWINDOW_MNEMONIC_INDEX)
				).intValue()
				);
		fitwindowMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_FITINWINDOW_TOOLTIP)
			);
		fitwindowMenuitem.setActionCommand("_VEFitwindow");
		fitwindowMenuitem.addActionListener(this);

		fitwidthMenuitem = new JRadioButtonMenuItem();
		fitwidthMenuitem.setText(
			ExplainResources.getString(ExplainResources.MENU_ITEM_FITWIDTH)
			);
		fitwidthMenuitem.setAccelerator(
			(KeyStroke)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_FITWIDTH_ACCEL)
			);
		fitwidthMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_FITWIDTH_MNEMONIC)).intValue()
			);
		fitwidthMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_FITWIDTH_TOOLTIP)
			);
		fitwidthMenuitem.setActionCommand("_VEFitwidth");
		fitwidthMenuitem.addActionListener(this);
		
		// invisible radio button
		noselectMenuitem = new JRadioButtonMenuItem();

		zoommodeGroup = new ButtonGroup();
		zoommodeGroup.add(actualsizeMenuitem);
		zoommodeGroup.add(fitwindowMenuitem);
		zoommodeGroup.add(fitwidthMenuitem);
		zoommodeGroup.add(noselectMenuitem);

		// orientations
		topMenuitem = new JRadioButtonMenuItem(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_TOP)
			);
		topMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_ORIENTATION_TOP_MNEMONIC)).intValue()
			);
		topMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_TOP_TOOLTIP)
			);
		topMenuitem.setActionCommand("_VEOrientationTop");
		topMenuitem.addActionListener(this);
		topMenuitem.setSelected(true);

		bottomMenuitem = new JRadioButtonMenuItem(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_BOTTOM)
			);
		bottomMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_ORIENTATION_BOTTOM_MNEMONIC)
			).intValue()
			);
		bottomMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_BOTTOM_TOOLTIP)
			);
		bottomMenuitem.setActionCommand("_VEOrientationBottom");
		bottomMenuitem.addActionListener(this);

		leftMenuitem = new JRadioButtonMenuItem(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_LEFT)
			);
		leftMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_ORIENTATION_LEFT_MNEMONIC)
			).intValue()
			);
		leftMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_LEFT_TOOLTIP)
			);
		leftMenuitem.setActionCommand("_VEOrientationLeft");
		leftMenuitem.addActionListener(this);

		rightMenuitem = new JRadioButtonMenuItem(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_RIGHT)
			);
		rightMenuitem.setMnemonic(
			((Integer)ExplainResources.getObject(
				ExplainResources.MENU_ITEM_ORIENTATION_RIGHT_MNEMONIC)
			).intValue()
			);
		rightMenuitem.setToolTipText(
			ExplainResources.getString(
				ExplainResources.MENU_ITEM_ORIENTATION_RIGHT_TOOLTIP)
			);
		rightMenuitem.setActionCommand("_VEOrientationRight");
		rightMenuitem.addActionListener(this);

		orientationGroup = new ButtonGroup();
		orientationGroup.add(topMenuitem);
		orientationGroup.add(bottomMenuitem);
		orientationGroup.add(leftMenuitem);
		orientationGroup.add(rightMenuitem);

		// Content
		JPanel content = new JPanel();
		content.setLayout(new BorderLayout());

		JToolBar toolbar = createToolBar();
		toolbar.setFloatable(false);

		statusbar = new StatusBar();
		statusbar.setBorder(BorderFactory.createEtchedBorder());
		
		JMenuBar menubar = createMenuBar();
		JComponent mainarea = createMainArea();
		
		content.add(toolbar, BorderLayout.NORTH);
		content.add(mainarea, BorderLayout.CENTER);
		content.add(statusbar, BorderLayout.SOUTH);
		content.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));

		filechooser = new JFileChooser();
		filechooser.addChoosableFileFilter(new SQLFileFilter());

		spfilechooser = new JFileChooser();
		spfilechooser.addChoosableFileFilter(new SQLFileFilter());
		spfilechooser.addChoosableFileFilter(new PlanFileFilter());
		spfilechooser.setAcceptAllFileFilterUsed(false);

		mfilechooser = new JFileChooser();
		mfilechooser.addChoosableFileFilter(new SQLFileFilter());
		mfilechooser.addChoosableFileFilter(new PlanFileFilter());
		mfilechooser.addChoosableFileFilter(new MultiFileFilter());

		this.setContentPane(content);
		this.setJMenuBar(menubar);
		this.pack();
	}
	
	/**
	 * Creates a new <code>Vise</code> instance.
	 */
	public Vise()
	{
		starting_up = true;
		new_config = false;

		// icon
		setIconImage(
			new ImageIcon(
				this.getClass().getResource(
					ExplainResources.getString(ExplainResources.ICON_IMAGE))
				).getImage()
			);

		// formatter for zoom levels
		formatter = NumberFormat.getPercentInstance();

		// Listeners
		this.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				quit();
			}
		});

		// Actions
		afactory = new ActionFactory(
			ExplainResources.getString(ExplainResources.IMAGE_PATH));

		newAction     = afactory.getAction(ActionFactory.ACTION_NEW);
		openAction    = afactory.getAction(ActionFactory.ACTION_OPEN);
		saveAction    = afactory.getAction(ActionFactory.ACTION_SAVE);
		saveasAction  = afactory.getAction(ActionFactory.ACTION_SAVEAS);
		cutAction     = afactory.getAction(ActionFactory.ACTION_CUT);
		copyAction    = afactory.getAction(ActionFactory.ACTION_COPY);
		pasteAction   = afactory.getAction(ActionFactory.ACTION_PASTE);
		aboutAction   = afactory.getAction(ActionFactory.ACTION_ABOUT);
		exitAction    = afactory.getAction(ActionFactory.ACTION_EXIT);
		explainAction = afactory.getAction(ActionFactory.ACTION_EXPLAIN);
		analyzeAction = afactory.getAction(ActionFactory.ACTION_ANALYZE);
		prefsAction   = afactory.getAction(ActionFactory.ACTION_PREFERENCES);
		dbasesAction   = afactory.getAction(ActionFactory.ACTION_DATABASES);
		poptsAction   = afactory.getAction(ActionFactory.ACTION_PLANNEROPTIONS);
		zoominAction  = afactory.getAction(ActionFactory.ACTION_ZOOMIN);
		zoomoutAction = afactory.getAction(ActionFactory.ACTION_ZOOMOUT);
		stopAction = afactory.getAction(ActionFactory.ACTION_STOP);
		helpAction    = afactory.getAction(ActionFactory.ACTION_MANUAL);
		refreshAction = afactory.getAction(ActionFactory.ACTION_REFRESH);
		
		newAction.addActionListener(this);
		openAction.addActionListener(this);
		saveAction.addActionListener(this);
		saveasAction.addActionListener(this);
		aboutAction.addActionListener(this);
		exitAction.addActionListener(this);
		explainAction.addActionListener(this);
		analyzeAction.addActionListener(this);
		prefsAction.addActionListener(this);
		dbasesAction.addActionListener(this);
		poptsAction.addActionListener(this);
		zoominAction.addActionListener(this);
		zoomoutAction.addActionListener(this);
		stopAction.addActionListener(this);
		helpAction.addActionListener(this);
		refreshAction.addActionListener(this);

		// We don't open a connection right away so explain is disabled
		explainAction.setEnabled(false);
		analyzeAction.setEnabled(false);
		stopAction.setEnabled(false);
		
		// configuration stuff - needs reworking
		config = new ConfigurationModel();
		(new File(
			ExplainResources.getString(
				ExplainResources.CONFIGURATION_DIR)
			)
		).mkdirs();
		conffile = new File(ExplainResources.getString(
								ExplainResources.CONFIGURATION_DIR),
							ExplainResources.getString(
								ExplainResources.FILE_CONFIGURATION)
							);
		// For backwards compatibility, we will look for the previous
		// config file name on this release
		File oldconffile = new File(ExplainResources.getString(
								        ExplainResources.CONFIGURATION_DIR),
							        "explain.conf"
						        	);

		createlock();
		try {
			if (!conffile.exists())
			{
			    if (oldconffile.exists())
			    {
			        config.load(oldconffile);
			    }
				conffile.createNewFile();
				// XXX: We can't use saveconfig as if there is a stale lock
				// it will not save the configuration and we need a file
				// in place.  So we bypass the lock mechanism and save to
				// the new config file anyway.  If there are two instances
				// of Vise running under the same userid started exactly at
				// the same time there is a chance in a million that one of
				// them can't write to the file and gets a stack trace here.  
				//saveconfig();
			    try {
				    config.save(conffile);
			    } catch (IOException ex) {
				    ex.printStackTrace();
			    }
			}
			else
			{
			    config.load(conffile);
			}
		} catch (IOException ex) {
			ex.printStackTrace();
		} catch (ConfigurationException ex) {
			ex.printStackTrace();
		} finally {
			config.addObserver(this);
		}

		cm = new ConnectionModel(this);
		cm.addActionListener(this);

		prefs = new PreferencesDialog(config, this);
		dbases = new DatabasesDialog(config, this);
		poptions = new PlannerOptionsDialog(new PlannerOptions(), this);
		
		// We are not running at the beginning
		running = false;

		// initialize components
		initComponents();

		// initial state of overview toggle buttons
		overviewButton.setSelected(expview.isOverviewVisible());

		// zoom stuff
		zoomcombobox.setModel(new ZoomComboBoxModel(zooms));
		actualsizeMenuitem.setSelected(true);
		actualsize();

		// Password dialog
		passdialog = new PasswordDialog(this);
		aboutdialog = new AboutDialog(this);
		aboutdialog.setResizable(false);

		// help window
		helpwindow = new HelpWindow();
		
		// query model
		// Check for auto-reloading of last query
		// and auto-connection to last database
		if (config.isReload()) {
			if (config.isLastQuery())
				open(config.getLast(0));
            else
                newquery();
			DatabaseModel savedmodel = config.getDatabaseModel(
										   config.getLastDatabase());
			// If the saved database name was found, we try and connect
			if (savedmodel != null)
				dbcombobox.setSelectedItem(savedmodel);
		}
        else
            newquery();

		// update based on loaded preferences
		update();

		starting_up = false;
	}
	
	/**
	 * Creates a new <code>Vise</code> (Visual Explain) instance.
	 *
	 * @param wname the window name, a <code>String</code> value
	 */
	public Vise(String wname)
	{
		this();

		setTitle(wname);
	}
	
	/**
	 * Main entrypoint into the Visual Explain program.
	 *
	 * @param args a <code>String[]</code> value
	 */
	public static void main(String[] args)
	{
		// FIXME Create a --version command line argument to display those
		// and our own version
		//System.out.println("Default Locale=" + Locale.getDefault());
		//System.out.println("Java Version=" +
		//					 System.getProperty("java.version"));
		//System.out.println("Java VM Version=" +
		//					 System.getProperty("java.vm.version"));
		//System.out.println("Java VM Specification Version=" +
		//					 System.getProperty("java.vm.specification.version"));
		//System.out.println("Java Specification Version=" +
		//					 System.getProperty("java.specification.version"));
        MetalLookAndFeel.setCurrentTheme(new ViseLookAndFeel());

 		SplashScreen splash = new SplashScreen(
			ExplainResources.getString(
				ExplainResources.SPLASH_IMAGE)
			);
		splash.setVisible(true);
		
		Vise win = new Vise();
		win.setSize(800, 600);
		Dimension screen = Toolkit.getDefaultToolkit().getScreenSize();
		win.setLocation((screen.width - win.getWidth()) / 2,
						(screen.height - win.getHeight()) / 2);
		win.setVisible(true);
		
		// Argh!  If we loaded a plan file before resizing and making the window
		// visible the table column calculations were made with the wrong data
		// as Swing will only set the values properly when painting the window.
		// So we force an update in the ExplainView. 
		win.expview.setExplain(win.exp);

		splash.dispose();
	}

	//
	// Private implementation of ComboBoxModel. Adapter for list of
	// databases that the configuration model stores.
	//
	private class DatabaseComboBoxModel extends AbstractListModel
										implements ComboBoxModel {
		private ConfigurationModel cm;
		private DisconnectedDatabaseModel ddm;
		private DatabaseModel sel;

		public DatabaseComboBoxModel(ConfigurationModel cm)
		{
			this.cm = cm;
			sel = null;
			ddm = new DisconnectedDatabaseModel();
		}

		public Object getElementAt(int index)
		{
			DatabaseModel d;
			
			if (index == 0)
			{
				d = ddm;
			}
			else
			{
				d = cm.getDatabaseModel(index - 1);
			}

			return d;
		}

		public int getSize()
		{
			if (cm == null)
				return 1;
			else
				return 1 + cm.getDatabaseCount();
		}
		
		public Object getSelectedItem()
		{
			return sel;
		}

		public void setSelectedItem(Object anItem)
		{
			sel = (DatabaseModel) anItem;
		}

		public ConfigurationModel getConfigurationModel()
		{
			return cm;
		}

		public void setConfigurationModel(ConfigurationModel cm)
		{
			this.cm = cm;
		}
	}

	//
	// model class for the zoom combo box
	//
	private class ZoomComboBoxModel extends AbstractListModel
									implements ComboBoxModel {
		private Double sel;
		private Double[] list;

		public ZoomComboBoxModel(Double[] list)
		{
			this.list = list;
		}

		public ZoomComboBoxModel(double[] list)
		{
			this.list = new Double[list.length];
			for (int i = 0; i < list.length; i++)
				this.list[i] = new Double(list[i]);
		}

		public Object getElementAt(int index)
		{
			return list[index];
		}

		public int getSize()
		{
			return list.length;
		}

		public Object getSelectedItem()
		{
			return sel;
		}

		public void setSelectedItem(Object anItem)
		{
			if (anItem instanceof String)
			{
				// first try to use the percent formatter to get the number.
				// if that doesn't work, then user might have entered a zoom
				// without a percent sign so parse the item and divide by 100.
				// if that doesn't work, give up and set the zoom to 1.0.
				try {
					Number n = formatter.parse((String) anItem);
					anItem = new Double(n.doubleValue());
				} catch (ParseException ex) {
					try {
						double d = Double.parseDouble((String) anItem) / 100;
						anItem = new Double(d);
					} catch (Exception exc) {
						anItem = new Double(1.0);
					}
				}
			}

			sel = (Double) anItem;
		}
	}

	//
	// renderer for the zoom combo box
	//
	private class ZoomComboBoxRenderer extends JLabel
									   implements ListCellRenderer {
		public ZoomComboBoxRenderer()
		{
			setOpaque(true);
		}

		public Component getListCellRendererComponent(
			JList list,
			Object value,
			int index,
			boolean isSelected,
			boolean cellHasFocus)
		{
			if (isSelected)
			{
				this.setBackground(list.getSelectionBackground());
				this.setForeground(list.getSelectionForeground());
			}
			else
			{
				this.setBackground(list.getBackground());
				this.setForeground(list.getForeground());
			}

			this.setText(formatter.format(((Double) value).doubleValue()));

			return this;
		}
	}

	//
	// private file filter class for query files
	//
	private class SQLFileFilter extends FileFilter {

		public boolean accept(File f)
		{
			String extension;

			if (f.isDirectory())
				return true;

			// get the file's extension
			if ((extension = RHDBUtils.getExtension(f)) == null)
				return false;

			if (extension.equalsIgnoreCase(sqlext))
				return true;

			return false;
		}

		public String getDescription()
		{
			return ExplainResources.getString(ExplainResources.SQL_FILTER,
											  "*." + sqlext);
		}
	}

	//
	// private file filter class for plan files
	//
	private class PlanFileFilter extends FileFilter {

		public boolean accept(File f)
		{
			String extension;

			if (f.isDirectory())
				return true;

			// get the file's extension
			if ((extension = RHDBUtils.getExtension(f)) == null)
				return false;

			if (extension.equalsIgnoreCase(planext))
				return true;

			return false;
		}

		public String getDescription()
		{
			return ExplainResources.getString(
					   ExplainResources.PLAN_FILTER,
					   "*." + planext
				   );
		}
	}

	//
	// private file filter class for query or plan files
	//
	private class MultiFileFilter extends FileFilter {

		public boolean accept(File f)
		{
			String extension;

			if (f.isDirectory())
				return true;

			// get the file's extension
			if ((extension = RHDBUtils.getExtension(f)) == null)
				return false;

			if (extension.equalsIgnoreCase(sqlext) ||
				extension.equalsIgnoreCase(planext))
				return true;

			return false;
		}

		public String getDescription()
		{
			return ExplainResources.getString(
					   ExplainResources.MULTI_FILTER,
					   "*." + sqlext + ", *." + planext
					   );
		}
	}
} // Vise
