/*
 * Seti@Home client integrated into panel
 * (C) 2000 Richard Kinder
 *
 * based on:
 * GNOME webcontrol applet and reading of the sources of many other applets.
 *
 * Author: Richard Kinder <r_kinder@yahoo.com>
 *
 * Date: 8/1/00
 *
 * modified: 26/3/00 by Craig Orsinger (CJO) <cjorsinger@earthlink.net>
 *            - Changed single callback to three; one each for reading
 *              state file, user file, and display (message[] array info)
 *            - Added prototypes for static functions (and removed them
 *              from "seti_applet.h").
 *            - Fixed bug in long term (years) printouts of CPU time.
 */

#define VERSION "0.2.2"
#define PACKAGE "seti_applet"
#include <gnome.h>
#include <applet-widget.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "seti_applet.h"
#include "seti_control.h"

#include "seti_applet-logo-red.xpm"
#include "seti_applet-logo-green.xpm"

GtkWidget *applet;
GtkWidget *frame;
GtkWidget *hbox_all;
GtkWidget *vbox;
GtkWidget *hbox_top;
GtkWidget *hbox_bottom;
GtkWidget *align;
GtkWidget *top_left_label;
GtkWidget *bottom_left_label;
GtkWidget *image_pixmap;

/* boolean ... is seti running? */
static gint seti_is_running = 0;

static gint update_state_file_timeout_id;
static gint update_user_file_timeout_id;
static gint display_timeout_id;
/*If seti@home has finished and requires user intervention, flash the icon*/
static gint attention_timeout_id;
static gint flash_index = 0;
static gint currentcount = 0;

static gint previous_status = SETI_STOPPED;

static gchar* percentage = NULL;
static gchar* unitsinfo = NULL;
static gchar* spike = NULL;
static gchar* gaussian = NULL;
static gchar* cpuinfo = NULL;

static PanelOrientType orientation;

static gchar* messages2[5];

setiapplet sa =
{
	NULL,
	10,
	120,
	0,
	TRUE,
	TRUE,
	TRUE,
	TRUE,
	TRUE,
	19,
	NULL,
	FALSE,
	NULL,
	TRUE
};

/*
 * (CJO) Prototypes of static functions
 */
static void applet_change_orient ( GtkWidget *, PanelOrientType, gpointer );
static void check_change_pixmap ( void );
static void start_callback ( void );
static void attention_callback ( void );
static void display_callback ( void );
static void state_file_callback ( void );
static void user_file_callback ( void );
static void parse_state_file ( char* );
static void parse_user_file ( char* );
static void update_display ( void );
static void re_read_state_file ( void );
static void re_read_user_file ( void );
static void parse_state_file ( char* );

static void
help_cb (AppletWidget *widget, gpointer data)
{
	static GnomeHelpMenuEntry help_entry = {NULL, "index.html"};
	help_entry.name = gnome_app_id;
	gnome_help_display ( NULL, &help_entry );
}

void
on_SetiAppletProperties_help (GnomePropertyBox *gnomepropertybox,
		gint             arg1,
		gpointer         user_data)
{
	help_cb(NULL,NULL);
}

void
on_SetiAppletProperties_apply (GnomePropertyBox *gnomepropertybox,
		gint             arg1,
		gpointer         user_data)
{
	gchar *new_text;
	/*final apply on close, prevent multiple calls of this method*/
	if ( arg1 == -1 )
	{
		sa.showpercentage = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(PercentageCheck) );
		sa.showgaussian = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(GaussianCheck) );
		sa.showspike = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(SpikeCheck) );
		sa.showunits = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(UnitsProcessedCheck) );
		sa.showcpu = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(TotalCPUCheck) );
		new_text = gtk_entry_get_text ( GTK_ENTRY (SetiLocationText) );
		if ( sa.setidir )
		{
			g_free ( sa.setidir );
		}
		sa.setidir = g_strdup(new_text);
		sa.updateinterval = gtk_spin_button_get_value_as_int ( GTK_SPIN_BUTTON ( UIUpdateSpin ) );
		sa.fileupdateinterval = gtk_spin_button_get_value_as_int ( GTK_SPIN_BUTTON ( FileUpdateSpin ) );
		sa.launchonstart = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(LaunchOnStart) );
		sa.nicevalue = (GTK_ADJUSTMENT(NiceAdjust))->value;
		if ( sa.extraparams )
		{
			g_free ( sa.extraparams );
		}
		new_text = gtk_entry_get_text ( GTK_ENTRY (ExtraParmText) );
		sa.extraparams = g_strdup(new_text);
		sa.separateexedir = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON(UseDiffDirExe) );
		if ( sa.separateexedir )
		{
			if ( sa.setiexedir )
			{
				g_free ( sa.setiexedir );
			}
			new_text = gtk_entry_get_text ( GTK_ENTRY (SetiAlternateDir) );
			sa.setiexedir = g_strdup(new_text);
		}
		re_read_state_file();
		re_read_user_file();
		start_callback();
        }
}

static void
applet_change_orient ( GtkWidget *widget, PanelOrientType or, gpointer data )
{
	orientation = or;
	if ( or == ORIENT_LEFT || or == ORIENT_RIGHT )
	{
		gtk_label_set_text ( GTK_LABEL(top_left_label), "  seti  ");
	}
	else
	{
		gtk_label_set_text ( GTK_LABEL ( top_left_label), "    seti@home    ");
	}
	re_read_state_file();
	re_read_user_file();
	update_display();
	start_callback();
}


static void
about_cb (AppletWidget *widget, gpointer data)
{
	GtkWidget *about;
	static const gchar *authors[2];
	authors[0] = "Richard Kinder <r_kinder@yahoo.com>";
	authors[1] = "Matt Herbert <mherbert@bit-net.com>";
	authors[2] = NULL;

	about = gnome_about_new (_("Seti@Home Applet"), VERSION,
			"(C) 2000",
			authors,
			_("An applet which shows the status of the seti@home client within the panel."),
			NULL);
	gtk_widget_show (about);
	return;
}

gchar *
re_read_file ( gchar *full_path )
{
	int fd;
	int stat_rv;
	int num_bytes_read;
	gchar* read_buf = g_malloc0 ( 700 );
	struct stat *stat_buf = g_malloc0 ( sizeof ( struct stat ) );

	/* Get file stats, check it exists, has size > 0 */
	stat_rv = stat ( full_path, stat_buf );
	if ( stat_rv == -1 )
	{
		g_free ( read_buf );
		read_buf = 0;
	}
	else
	{
		if ( ( stat_buf->st_size ) == 0 )
		{
			g_free ( read_buf );
			read_buf = 0;
		}
		else
		{

			fd = open ( full_path, O_RDONLY );
			if ( fd != -1 )
			{
				num_bytes_read = read ( fd, read_buf, 700 );
				close ( fd );
			}
			else
			{
				g_free ( read_buf );
				read_buf = 0;
			}
		}
	}
	g_free ( stat_buf );
	return read_buf;
}

static void
re_read_state_file()
{
	gchar *file_contents;
	gchar *full_path;
	/*Try the new seti state.sah file name*/
	full_path = g_strdup_printf ( "%s/%s", sa.setidir, STATE_SAH );
	file_contents = re_read_file ( full_path );
	/*Try the old seti state.txt file name*/
	if ( !file_contents )
	{
		g_free ( full_path );
		full_path = g_strdup_printf ( "%s/%s", sa.setidir, STATE_TXT );
		file_contents = re_read_file ( full_path );
	}
	parse_state_file ( file_contents );
	g_free ( full_path );
	/*file_contents may be a null pointer - re_read_file returns null if
	 * the open file returned an error. Only free the memory if we have
	 * a clean open file*/
	if ( file_contents )
	{
		g_free ( file_contents );
	}
}

static void
re_read_user_file()
{
	gchar *file_contents;
	gchar *full_path;
	full_path = g_strdup_printf ( "%s/%s", sa.setidir, USER_INFO_SAH );
	file_contents = re_read_file ( full_path );
	if ( !file_contents )
	{
		g_free ( full_path );
		full_path = g_strdup_printf ( "%s/%s", sa.setidir, USER_INFO_TXT );
		file_contents = re_read_file ( full_path );
	}
	parse_user_file ( file_contents );
	g_free ( full_path );
	/*As for re_read_state_file - dont try to free a null pointer*/
	if ( file_contents )
	{
		g_free ( file_contents );
	}
}

void
change_pixmap()
{
	if ( ( sa.current_image % 2 ) == 1 )
	{
		gnome_pixmap_load_xpm_d(GNOME_PIXMAP(image_pixmap), seti_applet_logo_green_xpm);
	}
	else
	{
		gnome_pixmap_load_xpm_d(GNOME_PIXMAP(image_pixmap), seti_applet_logo_red_xpm);
	}
	sa.current_image ++;
}

static void
attention_timeout(void)
{
	change_pixmap();
	attention_callback();
}

static void attention_timeout_end(void)
{
	if ( attention_timeout_id )
	{
		gtk_timeout_remove ( attention_timeout_id );
	}
}

static void
attention_callback(void)
{
	if ( attention_timeout_id )
	{
		gtk_timeout_remove ( attention_timeout_id );
	}
	if ( seti_status ( &sa ) == SETI_RUNNING )
	{
		attention_timeout_id = gtk_timeout_add ( ATTENTION_INTERVAL, (GtkFunction)attention_timeout, NULL );
	}
	else
	{
		sa.current_image = 0;
		check_change_pixmap();
	}
}

static void
parse_state_file ( char* read_buf )
{
	gdouble d_temp;
	gint attention_tmout;
	gchar *loc;

	/*Check one of the strings containing the values, if it is not null, we
	 * have been through here before, else it is the first time through and
	 * we must read in some values
	 */
	if ( ( percentage != NULL ) && ( seti_status ( &sa ) == SETI_STOPPED ) )
	{
		return;
	}
	if ( percentage )
	{
		g_free ( percentage );
	}
	if ( gaussian )
	{
		g_free ( gaussian );
	}
	if ( spike )
	{
		g_free ( spike );
	}
	if ( !read_buf )
	{
		/*Start the flash callback*/
		attention_timeout();
		if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
		{
			percentage = g_strdup ( _("!Path") );
			gaussian = g_strdup ( _("!Path") );
			spike = g_strdup ( _("!Path") );
		}
		else
		{
			if ( sa.showpercentage )
			{
				percentage = g_strdup ( _("Invalid path") );
			}
			else
			{
				percentage = g_strdup ( " " );
			}
			if ( sa.showgaussian )
			{
				gaussian = g_strdup ( _("Invalid path") );
			}
			else
			{
				gaussian = g_strdup ( " " );
			}
			if ( sa.showspike )
			{
				spike = g_strdup ( _("Invalid path") );
			}
			else
			{
				spike = g_strdup ( " " );
			}
		}
	}
	else
	{
		attention_timeout_end();
		loc = strstr ( read_buf, "prog=" );
		loc += 5;
		d_temp = g_strtod ( loc, NULL );
		if ( sa.showpercentage )
		{
			if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
			{
				percentage = g_strdup_printf ( "%2.1f%c", d_temp * 100, '%' );
			}
			else
			{
				percentage = g_strdup_printf ( "%6.2f%c", d_temp * 100, '%' );
			}
		}
		else
		{
			percentage = g_strdup ( " " );
		}
		loc = strstr ( loc, "bs_power=" );
		loc += 9;
		d_temp = g_strtod ( loc, NULL );
		if ( sa.showspike )
		{
			if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
			{
				spike = g_strdup_printf ( "S:%3.0f", d_temp );
			}
			else
			{
				spike = g_strdup_printf ( "Spike:%6.2f", d_temp );
			}
		}
		else
		{
			spike = g_strdup ( " " );
		}
		loc = strstr ( loc, "bg_score=" );
		loc += 9;
		d_temp = g_strtod ( loc, NULL );
		if ( sa.showgaussian )
		{
			if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
			{
				gaussian = g_strdup_printf ( "G:%1.2f", d_temp );
			}
			else
			{
				gaussian = g_strdup_printf ( "Gauss:%6.2f", d_temp );
			}
		}
		else
		{
			gaussian = g_strdup ( " " );
		}
	}
	messages2[0] = percentage;
	messages2[1] = spike;
	messages2[2] = gaussian;
}

static void
parse_user_file ( char* read_buf )
{
	gdouble d_temp;
	gchar *loc;

	if ( cpuinfo )
	{
		g_free ( cpuinfo );
	}
	if ( unitsinfo )
	{
		g_free ( unitsinfo );
	}
	if ( !read_buf )
	{
		if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
		{
			cpuinfo = g_strdup ( _("!Path") );
			unitsinfo = g_strdup ( _("!Path") );
		}
		else
		{
			cpuinfo = g_strdup ( _("Invalid path") );
			unitsinfo = g_strdup ( _("Invalid path") );
		}
	}
	else
	{
		loc = strstr ( read_buf, "nresults=" );
		loc += 9;
		d_temp = g_strtod ( loc, NULL );
		if ( sa.showunits )
		{
			if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
			{
				unitsinfo = g_strdup_printf ( "U:%3.0f", d_temp );
			}
			else
			{
				unitsinfo = g_strdup_printf ( "Units:%6.0f", d_temp );
			}
		}
		else
		{
			unitsinfo = g_strdup ( " " );
		}

		loc = strstr ( loc, "total_cpu=" );
		loc += 10;
		d_temp = g_strtod ( loc, NULL );
		if ( sa.showcpu )
		{
			if ( d_temp > 315440000000 )
			{
				if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
				{
					cpuinfo = g_strdup_printf ( "C:>10ky" );
				}
				else
				{
					cpuinfo = g_strdup_printf ( "CPU:>10000 yrs" );
				}
			}
			else if ( d_temp > 31557600 )
			{
				if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
				{
					cpuinfo = g_strdup_printf ( "C:%4.0f yr", d_temp / (60*60*24*365.25) );
				}
				else
				{
					cpuinfo = g_strdup_printf ( "CPU:%6.2f yrs", d_temp / (60*60*24*365.25) );
				}
			}
			else
			{
				if ( orientation == ORIENT_LEFT || orientation == ORIENT_RIGHT )
				{
					cpuinfo = g_strdup_printf ( "C:%3.0f", d_temp / (60*60*24) );
				}
				else
				{
					cpuinfo = g_strdup_printf ( "CPU:%6.0f days", d_temp / (60*60*24) );
				}
			}
		}
		else
		{
			cpuinfo = g_strdup ( " " );
		}
	}
	messages2[3] = unitsinfo;
	messages2[4] = cpuinfo;
}

static void
properties_cb (AppletWidget *widget, gpointer data)
{
	GtkWidget* propertiesDialog;

	propertiesDialog = create_SetiAppletProperties();
	gtk_widget_show ( propertiesDialog );
}

static void
update_display(void)
{
	gint internalcount = 0;
	check_change_pixmap();
	currentcount++;
	
	if ( currentcount >= 5 )
	{
		currentcount = currentcount%5;
	}
	while ( internalcount < 6 && g_strcasecmp ( messages2[currentcount%5], " " ) == 0 )
	{
		currentcount++;
		internalcount++;
	}
	if ( internalcount < 6 )
	{
		gtk_label_set_text ( GTK_LABEL(bottom_left_label), messages2[currentcount%5] );
	}
	else
	{
		gtk_label_set_text ( GTK_LABEL(bottom_left_label), "No data!");
	}
}


static void
display_timeout(void)
{
	update_display() ;
	display_callback() ;
}

static void
update_state_file_timeout(void)
{
	re_read_state_file();
	state_file_callback();
}

static void
update_user_file_timeout(void)
{
	re_read_user_file();
	user_file_callback();
}

static void
state_file_callback(void)
{
	gint delay ;
	
	if ( sa.fileupdateinterval < 1 )
	{
		delay = 120 ;	/* (CJO) changed because 2.4 updates every 60 sec.*/
	}
	else
	{
		delay = sa.fileupdateinterval ;
	}
	delay *= 1000 ;
	if ( update_state_file_timeout_id )
	{
		gtk_timeout_remove ( update_state_file_timeout_id );
	}
	update_state_file_timeout_id = gtk_timeout_add ( delay, (GtkFunction)update_state_file_timeout, NULL );
}

static void
user_file_callback(void)
{
	gint delay ;
	
	/*
	 * (CJO) for now, use the file update interval, just like the
	 *       state file callback does.
	 */
	if ( sa.fileupdateinterval < 1 )
	{
		delay = 120 ;	/* (CJO) changed because 2.4 updates every 60 sec.*/
	}
	else
	{
		delay = sa.fileupdateinterval ;
	}
	if ( update_user_file_timeout_id )
	{
		gtk_timeout_remove ( update_user_file_timeout_id );
	}
	/*
	 * (CJO) - changed timeout to every two minutes , because one hour 
	 *         just seems way longer than necessary.
	 */
	update_user_file_timeout_id = gtk_timeout_add ( 1000 * 60 * 2, (GtkFunction)update_user_file_timeout, NULL );
}

static void
display_callback(void)
{
	gint delay ;
	
	if ( sa.updateinterval < 1 )
	{
		/* (CJO) for now, try changing message every 5 sec. */
		delay = 5 ;
	}
	else
	{
		delay = sa.updateinterval ;
	}
	if ( display_timeout_id )
	{
		gtk_timeout_remove ( display_timeout_id );
	}
	delay *= 1000 ;
	display_timeout_id = gtk_timeout_add ( delay, (GtkFunction)display_timeout, NULL );
}

static void
start_callback(void)
{
	state_file_callback();
	user_file_callback();
	display_callback();
}

static void applet_get_session ( gchar* path )
{
	if ( sa.setidir )
	{
		g_free ( sa.setidir );
	}
	gnome_config_push_prefix ( path );
	sa.setidir = gnome_config_get_string ( "setiapplet/setidir=/usr/local/seti" );
	sa.updateinterval = gnome_config_get_int ( "setiapplet/updateinterval=10" );
	sa.fileupdateinterval = gnome_config_get_int ( "setiapplet/fileupdateinterval=120" );
	sa.showcpu = gnome_config_get_int ( "setiapplet/showcpu=1" );
	sa.showspike = gnome_config_get_int ( "setiapplet/showspike=1" );
	sa.showgaussian = gnome_config_get_int ( "setiapplet/showgaussian=1" );
	sa.current_image = 0;
	sa.showpercentage = gnome_config_get_int ( "setiapplet/showpercentage=1" );
	sa.showunits = gnome_config_get_int ( "setiapplet/showunits=1" );
	sa.nicevalue = gnome_config_get_int ( "setiapplet/nicevalue=19" );
	if ( sa.extraparams )
	{
		g_free ( sa.extraparams );
	}
	sa.extraparams = gnome_config_get_string ( "setiapplet/extraparams=");
	sa.separateexedir = gnome_config_get_int ( "setiapplet/separateexedir=0" );
	if ( sa.separateexedir == TRUE )
	{
		sa.setiexedir = gnome_config_get_string ( "setiapplet/setiexedir" );
	}
	sa.launchonstart = gnome_config_get_int ( "setiapplet/launchonstart=0" );
	gnome_config_pop_prefix();
}

/* sesion save signal handler*/
static gint
applet_save_session(GtkWidget *w,
		    const char *privcfgpath,
		    const char *globcfgpath)
{	
	gnome_config_push_prefix(privcfgpath);
	gnome_config_set_string("setiapplet/setidir", sa.setidir);
	gnome_config_set_int("setiapplet/updateinterval", sa.updateinterval);
	gnome_config_set_int("setiapplet/fileupdateinterval", sa.fileupdateinterval);
	gnome_config_set_int("setiapplet/showspike", sa.showspike);
	gnome_config_set_int("setiapplet/showgaussian", sa.showgaussian);
	gnome_config_set_int("setiapplet/showunits", sa.showunits);
	gnome_config_set_int("setiapplet/showcpu", sa.showcpu);
	gnome_config_set_int("setiapplet/showpercentage", sa.showpercentage);
	gnome_config_set_int("setiapplet/nicevalue", sa.nicevalue);
	gnome_config_set_string("setiapplet/extraparams", sa.extraparams);
	gnome_config_set_int("setiapplet/separateexedir", sa.separateexedir);
	gnome_config_set_int("setiapplet/launchonstart", sa.launchonstart);
	if ( sa.separateexedir == TRUE )
	{
		gnome_config_set_string("setiapplet/setiexedir", sa.setiexedir);
	}
	gnome_config_pop_prefix();

	gnome_config_sync();
	/* you need to use the drop_all here since we're all writing to
	   one file, without it, things might not work too well */
	gnome_config_drop_all();
		
	/* make sure you return FALSE, otherwise your applet might not
	   work compeltely, there are very few circumstances where you
	   want to return TRUE. This behaves similiar to GTK events, in
	   that if you return FALSE it means that you haven't done
	   everything yourself, meaning you want the panel to save your
	   other state such as the panel you are on, position,
	   parameter, etc ... */
	return FALSE;
}

/*
 * This code is borrowed from VSA.  It will make "toggling" a menu pick
 * between two values easier.
 */
static void register_applet_callbacks()
{

	AppletWidget *a = APPLET_WIDGET(applet);

	if ( !a )
		return;

	/* About Menu pick */
	applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					      "about",
					      GNOME_STOCK_MENU_ABOUT,
					      _("About..."),
					      about_cb,
					      NULL);
	
	/* Properties Menu Pick */
	applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					      "properties",
					      GNOME_STOCK_MENU_PROP,
					      _("Properties..."),
					      properties_cb,
					      NULL);

	/* Start/Stop Menu Toggle */
	applet_widget_register_callback(APPLET_WIDGET(applet),
					      "start_stop_seti",
					      seti_status(&sa) == SETI_RUNNING ?
					      _("Stop seti...") : _("Start seti..."),
					      start_stop_seti_cb,
					      &sa);

	/* Help Menu Pick */
	applet_widget_register_stock_callback(APPLET_WIDGET(applet),
					      "help",
					      GNOME_STOCK_PIXMAP_HELP,
					      _("Help..."),
					      help_cb,
					      NULL);
}

static void unregister_applet_callbacks()
{
	AppletWidget *a = APPLET_WIDGET(applet);
	
        if(!a)
		return;

	applet_widget_unregister_callback(a, "properties");
	applet_widget_unregister_callback(a, "about");
	applet_widget_unregister_callback(a, "help");
	applet_widget_unregister_callback(a, "start_stop_seti");
}

/**
 * Hack to change popup menu entries
 * This is borrowed from VAS (Visual Sound Analyzer)
 * Modified 28/2/00 RK, change the image from red->green or green->red
 * depending upon the status of the client app.
 */
void update_applet_menu()
{
	if(!applet)
		return;

	unregister_applet_callbacks();
	register_applet_callbacks();
	update_state_file_timeout();
	check_change_pixmap();
}

/**
 * check_change_pixmap, check if the process is running, change the image
 * appropriately.
 */
static void check_change_pixmap()
{
	int current_status = seti_status ( &sa );
	if ( current_status == SETI_RUNNING && previous_status == SETI_STOPPED )
	{
		gnome_pixmap_load_xpm_d(GNOME_PIXMAP(image_pixmap), seti_applet_logo_green_xpm);
	}
	else if ( current_status == SETI_STOPPED && previous_status == SETI_RUNNING )
	{
		gnome_pixmap_load_xpm_d(GNOME_PIXMAP(image_pixmap), seti_applet_logo_red_xpm);
	}
	previous_status = current_status;
}

int
main(int argc, char **argv)
{
	messages2[0] = percentage;
	messages2[1] = spike;
	messages2[2] = gaussian;
	messages2[3] = cpuinfo;
	messages2[4] = unitsinfo;

	/* Initialize the i18n stuff */
	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	/* intialize, this will basically set up the applet, corba and
	   call gnome_init */
	applet_widget_init("seti_applet", VERSION, argc, argv,
				    NULL, 0, NULL);

	/* create a new applet_widget */
	applet = applet_widget_new("seti_applet");
	/* in the rare case that the communication with the panel
	   failed, error out */
	if (!applet)
		g_error("Can't create applet!\n");
	orientation = ORIENT_UP;
	
	applet_get_session(APPLET_WIDGET(applet)->privcfgpath);
	
	if ( sa.launchonstart == TRUE )
	{
		if ( seti_status ( &sa ) == SETI_STOPPED )
		{
			start_stop_seti_cb ( APPLET_WIDGET(applet), &sa );
		}
	}
	/* create the widget we are going to put on the applet */
	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_widget_show(frame);

	align = gtk_alignment_new(0.5, 0.5, 0.0, 0.0);
	gtk_container_set_border_width(GTK_CONTAINER(align), 4);
	gtk_container_add(GTK_CONTAINER(frame), align);
	gtk_widget_show(align);

	hbox_all = gtk_hbox_new ( FALSE, FALSE );
	gtk_container_add ( GTK_CONTAINER (align), hbox_all );
	gtk_widget_show ( hbox_all );

	vbox = gtk_vbox_new(FALSE, FALSE);
	gtk_container_add(GTK_CONTAINER(hbox_all), vbox);
	gtk_widget_show(vbox);

	hbox_top = gtk_hbox_new(FALSE, FALSE);
	gtk_widget_show(hbox_top);
	gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox_top);

	top_left_label = gtk_label_new("    seti@home    ");

	gtk_box_pack_start_defaults(GTK_BOX(hbox_top), top_left_label);
	gtk_widget_show(top_left_label);

	if ( seti_status ( &sa ) == SETI_STOPPED )
	{
		image_pixmap = gnome_pixmap_new_from_xpm_d ( seti_applet_logo_red_xpm );
	}
	else
	{
		image_pixmap = gnome_pixmap_new_from_xpm_d ( seti_applet_logo_green_xpm );
	}
	gtk_box_pack_start_defaults(GTK_BOX(hbox_all), image_pixmap);
	gtk_widget_show ( image_pixmap );

	hbox_bottom = gtk_hbox_new(FALSE, FALSE);
	gtk_widget_show(hbox_bottom);
	gtk_box_pack_start_defaults(GTK_BOX(vbox), hbox_bottom);

	bottom_left_label = gtk_label_new("");

	gtk_box_pack_start_defaults(GTK_BOX(hbox_bottom), bottom_left_label);

	vbox = gtk_vbox_new(FALSE, FALSE);


	gtk_widget_show(bottom_left_label);
	
	/* add the widget to the applet-widget, and thereby actually
	   putting it "onto" the panel */
	applet_widget_add (APPLET_WIDGET (applet), frame);
 
	gtk_signal_connect(GTK_OBJECT(applet),"change_orient",
			GTK_SIGNAL_FUNC(applet_change_orient),
			NULL);
	
	/* bind the session save signal */
	gtk_signal_connect(GTK_OBJECT(applet),"save_session",
			   GTK_SIGNAL_FUNC(applet_save_session),
			   NULL);

	/* register the callbacks for the menu */
	register_applet_callbacks();
	
	/* add the widget to the applet-widget, and thereby actually
	   putting it "onto" the panel */
	gtk_widget_show (applet);

	/*Get some initial values or error message from the state and user
	  files*/
	re_read_state_file();
	re_read_user_file();
	update_display() ;

	/*Set the callback function to update the info displayed*/
	start_callback();

	/* special corba main loop */
	applet_widget_gtk_main ();

	/* NOTREACHED */
	return 0;
}
