/* ------------------------------------------------------------------------
 * $Id: Logger.hh,v 1.4 2001/08/16 11:32:20 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-07 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Logger_hh_
#define _Logger_hh_

// -- System Includes
#include <string>
#include <set>
#include <iostream>

// -- Local Includes
#include "Mutex.hh"

// -- Class Declarations

/**
 * Configurable logger class.
 **/
class Logger {

private:

    /// Stream lock struct forward declaration
    struct streamlock;
    friend class streamlock;
    
public:

    /**
     * Log a message 
     **/
    static streamlock log(const std::string &name);
        
private:
    
    /**
     * Emit a message
     **/
    template <class T> 
    static void emit(const std::string &name, const T &t) {
	if (isActive(name)) _output << t;
    }
    
    struct streamlock {
	streamlock(const std::string t) : owner(true), name(t) {
	    _mutex.lock();
	}
	streamlock(const streamlock &sl) : owner(true), name(sl.name) {
	    sl.owner = false;
	}
	~streamlock() {
	    if (owner) _mutex.unlock();
	};
	mutable bool owner;
	std::string name;
    };
        
    friend const streamlock &operator << (const streamlock &sl,
					  std::ostream &(func)(std::ostream &)) {
	Logger::emit(sl.name, func); return sl;
    }
    
    template <class T>
    friend const streamlock &operator << (const streamlock &sl, const T &t) {
	Logger::emit(sl.name, t); return sl;
    }
    
public:

    /**
     * Subscribe to a message group with the logging service. All messages
     * sent to this group will be forwarded to the output stream.
     *
     * @param name group name.
     **/ 
    static void subscribe(const std::string &name);

    /** 
     * Unsubscribe to a message group with the logging service.
     **/
    static void unsubscribe(const std::string &name);
    
private:

    static bool isActive(const std::string &name);
    
    static Mutex _mutex;
    static std::ostream &_output;
    static std::set<std::string> _groups;
};

#endif /* Logger.hh */
