/* ------------------------------------------------------------------------
 * $Id: Path.hh,v 1.3 2001/08/20 14:18:27 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-08-15 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _Path_hh_
#define _Path_hh_

// -- System Includes
#include <string>
#include <vector>

// -- Class Declarations

/**
 * UNIX path class. Common pathname manipulations.
 **/
class Path {
public:

    /**
     * Constructor.
     **/
    Path() { }

    /**
     * Constructor.
     *
     * @param path path string.
     * @param separator character used for separating several
     *                  directories (normally ':').
     **/
    Path(const std::string &path, char separator = ':');

    /**
     * Destructor.
     **/
    ~Path() { }

    /**
     * Append a directory to the internal list of directories
     * associated with the path.
     *
     * @param directory directory to append to the path.
     **/
    void append(const std::string &directory) {
	_directories.push_back(directory);
    }

    /**
     * Retrieve the size of the path (i.e. the number of directories
     * contained in it).
     * 
     * @return number of directories.
     **/ 
    int size() const { return _directories.size(); }

    /** 
     * Indexing operator.
     *
     * @param i index of directory to retrieve.
     * @return directory at the specified index.
     **/ 
    const std::string &operator [] (int i) {
	if (i >= 0 && i < size()) return _directories[i];
	else return empty;
    }

    /**
     * Expand a path with an initial component of "~" or "~user" (this
     * will replace these with the user's home directory).
     * 
     * @param path path string to expand.
     **/
    static std::string expandUser(const std::string &path);

    /**
     * Return the base name of a path. This will extract the last part
     * of the passed path, the part that looks to be a file. For
     * "/usr/include/stdlib.h", this call would return "stdlib.h", but
     * for "/usr/include/", it would return "", the empty string.
     *
     * @param path path to extract base name from.
     * @return basename of the given path.
     **/
    static std::string basename(const std::string &path);

    /**
     * Return the directory name of a path. This will extract the
     * first part of the path, so calling it on a string like
     * "/usr/bin/gzip" would return "/usr/bin/".
     *
     * @param path path to extract directory name from.
     * @return directory name of the given path.
     **/
    static std::string dirname(const std::string &path);

    /**
     * Split a path into a pair consisting of the directory name (up
     * to and including the last slash) and the basename. For
     * instance, calling this function on the string
     * "/home/foo/README" would return the pair ["/home/foo/",
     * "README"].
     *
     * @param path path to split.
     * @return pair consisting of the directory name and the base name.
     **/
    static std::pair<std::string, std::string> split(const std::string &path);
    
    /**
     * Normalize a path, thus eliminating redundant separators and
     * resolving relative references. This means that "/foo/../bar/"
     * becomes "/bar/".
     *
     * @param path path to normalize.
     *@return normalized path string.
     **/
    static std::string normpath(const std::string &path);
    
private:
    static std::string empty;
    std::vector<std::string> _directories;
};

#endif /* Path.hh */
