/* ------------------------------------------------------------------------
 * $Id: BSPTree.hh,v 1.4 2001/08/09 14:19:33 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-07-17 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _BSPTree_hh_
#define _BSPTree_hh_

// -- System Includes
#include <vector>

// -- 3Dwm Includes
#include "Celsius/Vector2D.hh"
#include "Celsius/Vector3D.hh"
#include "Celsius/Matrix3D.hh"
#include "Solid/Face.hh"
#include "Solid/BSPTreeBuilder.hh"

// -- Forward Declarations
class BSP;

// -- Class Declarations

/**
 * BSP tree container class. Each complete BSP tree is associated with
 * a BSPTree container object that holds all the tree-wide state such
 * as vertices, normals and texture coordinates.
 **/
class BSPTree {
public:

    /**
     * Constructor.
     **/
    BSPTree() : _tree(0) { }

    /**
     * Copy constructor. Will clone the contents of the source object,
     * including dynamically allocated stuff.
     *
     * @param t source tree to copy.
     **/
    BSPTree(const BSPTree &t) : _vertices(t._vertices), _normals(t._normals),
				_texCoords(t._texCoords) {
	// Clone the tree as well (deep copy)
	_tree = t._tree->clone();
    }
    
    /**
     * Destructor.
     **/
    ~BSPTree() { 
	if (_tree != 0) delete _tree;
    }

    /**
     * Retrieve the internal BSP tree. If no tree has been
     * constructed, this call will return a zero tree pointer.
     *
     * @return constructed BSP tree (zero if none).
     **/
    BSP *getTree() const {
	return _tree;
    }
    
    /**
     * Update the internal BSP tree. If a tree already exists, it will
     * be deallocated prior to being replaced by the new tree.
     * 
     * @param tree new BSP tree pointer.
     **/ 
    void setTree(BSP *tree) {
	_tree = tree;
    }

    /**
     * Retrieve a const vertex in the shared vertex list.
     *
     * @param index vertex index.
     * @return vertex at the specified index.
     **/
    const Vector3D &getVertex(int index) const {
	return _vertices[index];
    }

    /**
     * Retrieve a const normal in the shared normal list.
     *
     * @param index normal index.
     * @return normal at the specified index.
     **/
    const Vector3D &getNormal(int index) const {
	return _normals[index];
    }

    /**
     * Retrieve a const texture coordinate in the shared vertex list.
     *
     * @param index texture coordinate index index.
     * @return texture coordinate at the specified index.
     **/
    const Vector2D &getTexCoord(int index) const {
	return _texCoords[index];
    }

    /**
     * Retrieve a const reference to the shared vertex list contained
     * in the BSP tree.
     *
     * @return const reference to vertex list.
     **/
    const std::vector<Vector3D> &getVertices() const {
	return _vertices;
    }

    /**
     * Retrieve a const reference to the shared normal list contained
     * in the BSP tree.
     *
     * @return const reference to normal list.
     **/
    const std::vector<Vector3D> &getNormals() const {
	return _normals;
    }

    /**
     * Retrieve a const reference to the shared texture coordinate
     * list contained in the BSP tree.
     *
     * @return const reference to texture coordinate list.
     **/
    const std::vector<Vector2D> &getTexCoords() const {
	return _texCoords;
    }

    /**
     * Add a number of vertices to the shared vertex list of the BSP
     * tree.
     *
     * @param vs list of vertices to add.
     **/
    void addVertices(const std::vector<Vector3D> &vs) {
	_vertices.insert(_vertices.end(), vs.begin(), vs.end());
    }

    /**
     * Add a number of normals to the shared normal list of the BSP
     * tree.
     *
     * @param ns list of normals to add.
     **/
    void addNormals(const std::vector<Vector3D> &ns) {
	_normals.insert(_normals.end(), ns.begin(), ns.end());
    }

    /**
     * Add a number of texture coordinates to the shared texture
     * coordinate list of the BSP tree.
     *
     * @param tcs list of texture coordinates to add.
     **/
    void addTexCoords(const std::vector<Vector2D> &tcs) {
	_texCoords.insert(_texCoords.end(), tcs.begin(), tcs.end());
    }

    /**
     * Add a vertex to the shared vertex list. 
     *
     * @param vertex vertex to add to the shared vertex list.
     * @return index of the newly added vertex.
     **/
    int addVertex(const Vector3D &vertex);

    /**
     * Add a texture coordinate to the shared texture coordinate list.
     *
     * @param texCoord texture coordinate to add to the shared list.
     * @return index of the newly added texture coordinate.
     **/
    int addTexCoord(const Vector2D &texCoord);

    /**
     * Add a normal to the shared normal list. 
     *
     * @param normal normal to add to the shared normal list.
     * @return index of the newly added normal.
     **/
    int addNormal(const Vector3D &normal);

    /**
     * Retrieve the number of vertices in the shared vertex list.
     *
     * @return number of vertices in the shared vertex list.
     **/
    int getVertexCount() const {
	return _vertices.size();
    }

    /**
     * Retrieve the number of normals in the shared normal list.
     *
     * @return number of normals in the shared normal list.
     **/
    int getNormalCount() const {
	return _normals.size();
    }

    /**
     * Retrieve the number of texture coordinates in the shared vertex
     * list.
     *
     * @return number of vertices in the shared texture coordinate
     *         list.
     **/
    int getTexCoordCount() const {
	return _texCoords.size();
    }

    /**
     * Retrieve the number of faces contained in the entire BSP
     * tree. This will recursively query all subnodes of the tree, so
     * this call might be expensive.
     *
     * @return number of faces contained in the BSP tree.
     **/
    int getFaceCount() const {
	return _tree->getFaceCount(); 
    }
    
    /**
     * Perform an incremental boolean set operation on the current BSP
     * tree by combining a number of faces with it. The new faces may
     * not contain indices outside the geometry data of the BSP tree.
     *
     * @param face list of triangle faces to apply to the tree.
     * @param op set operation to use when combining.
     **/
    void combine(const std::vector<TriangleFace> &faces,
		 BSPTreeBuilder::SetOperation op) {

	// Delegate the call to the BSP tree itself
	_tree->combine(*this, faces, op);
    }
    
    /**
     * Complement the BSP tree. This will flip all the faces and
     * planes of the tree as well as the normals.
     **/
    void complement();

    /**
     * Transform the geometry data of the BSP tree with a given
     * transformation matrix. In addition, to avoid incorrect
     * lighting, normals will be scaled back to unit length.
     *
     * @param trafo transformation matrix.
     **/
    void transform(const Matrix3D &trafo);
    
private:
    BSP *_tree;
    std::vector<Vector3D> _vertices;
    std::vector<Vector3D> _normals;
    std::vector<Vector2D> _texCoords;
};

#endif /* BSPTree.hh */
