/* ------------------------------------------------------------------------
 * AST.hh
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-12 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

#ifndef _AST_hh_
#define _AST_hh_

// -- System Includes
#include <string>
#include <stdio.h>
#include <iostream>

// -- Zorn Includes
#include "ASTVisitor.hh"

namespace Zorn {

    // -- Forward Declarations
    class ASTVisitor;

    // -- Class Declarations

    /**
     * Abstract syntax tree node class.
     **/
    class AST {
    public:

	/**
	 * Constructor. 
	 **/
	AST(const std::string &name) : _name(name), _child(0), _sibling(0) { }

	/**
	 * Destructor.
	 **/
	virtual ~AST() {
	    if (_child) delete _child;
	    if (_sibling) delete _sibling;
	}
	
	/**
	 * Retrieve the first child of this AST node.
	 *
	 * @return first child of the node (0 if none).
	 **/
	AST *getFirstChild() const {
	    return _child;
	}
    
	/**
	 * Retrieve the next sibling of this AST node (an AST node on
	 * the same level/height as the current node).
	 *
	 * @return next sibling (0 if none).
	 **/
	AST *getNextSibling() const {
	    return _sibling;
	}
    
	/**
	 * Retrieve the name of the AST node. The name is usually the
	 * language construct that the node represents, like "action",
	 * "rule", "call", etc.
	 *
	 * @return name of the AST node.
	 **/
	const std::string &getName() const {
	    return _name;
	}

	/**
	 * Retrieve the text associated with the AST node. This text
	 * usually contains the state of the node, like the identifier
	 * of a variable or a call or the actual string literal of a
	 * string.
	 *
	 * @return node text.
	 **/
	const std::string &getText() const {
	    return _text;
	}

	/**
	 * Set the first child pointer. If the node already has a
	 * child, this child may optionally be deleted by setting the
	 * "garbage-collection" flag.
	 *
	 * @param child new child pointer.
	 * @param gc garbage collect any existing child?
	 **/
	void setFirstChild(AST *child, bool gc = true) {
	    if (_child && gc) delete _child;
	    _child = child;
	}
    
	/**
	 * Set the next sibling pointer. If the node already has a
	 * sibling, this sibling may optionally be deleted by setting
	 * the "garbage-collection" flag.
	 *
	 * @param sibling new child pointer.
	 * @param gc garbage collect any existing sibling?
	 **/
	void setNextSibling(AST *sibling, bool gc = true) {
	    if (_sibling && gc) delete _sibling;
	    _sibling = sibling;
	}
	
	/**
	 * Accept an AST node visitor for traversal. This is the entry
	 * point for the AST visitor when it is traversing the
	 * abstract syntax tree. Each AST node type should call the
	 * corresponding visit*() method in the visitor.
	 *
	 * @param visitor AST visitor instance to use for traversal.
	 **/
	virtual void accept(ASTVisitor *visitor) = 0;
	
    protected:
	
	/**
	 * Set the AST node name.
	 *
	 * @param name new AST node name.
	 **/
	void setName(const std::string &name) {
	    _name = name;
	}
    
	/**
	 * Set the AST node text.
	 *
	 * @param text new AST node text.
	 **/
	void setText(const std::string &text) {
	    _text = text;
	}
    
    private:
	std::string _name;
	std::string _text;
	AST *_child;
	AST *_sibling;
    };

    class ASTRule : public AST {
    public:
	ASTRule(AST *condition, AST *actions) : AST("rule") {
	    setFirstChild(condition);
	    condition->setNextSibling(actions);
	}
	void accept(ASTVisitor *visitor) { visitor->visitRule(this); }
    };

    class ASTAssign : public AST {
    public:
	ASTAssign(const std::string &id, AST *node) : AST("assign") {
	    setText(id);
	    setFirstChild(node);
	}    
	void accept(ASTVisitor *visitor) { visitor->visitAssign(this); }
    };

    class ASTActionCall : public AST {
    public:
	ASTActionCall(AST *call) : AST("action call") {
	    setFirstChild(call);
	}    
	void accept(ASTVisitor *visitor) { visitor->visitActionCall(this); }
    };

    class ASTCall : public AST {
    public:
	ASTCall(const std::string &id, AST *params) : AST("call") {
	    setText(id);
	    setFirstChild(params);
	}    
	void accept(ASTVisitor *visitor) { visitor->visitCall(this); }
    };

    class ASTParam : public AST {
    public:
	ASTParam(AST *param) : AST("param") {
	    setFirstChild(param);
	}    
	void accept(ASTVisitor *visitor) { visitor->visitParam(this); }
    };

    class ASTBinOp : public AST {
    public:
	ASTBinOp(AST *left, const std::string &op, AST *right) : AST(op) {
	    setFirstChild(left);
	    left->setNextSibling(right);
	}    
	void accept(ASTVisitor *visitor) { visitor->visitBinOp(this); }
    };

    class ASTUnOp : public AST {
    public:
	ASTUnOp(const std::string &op, AST *node) : AST(op) {
	    setFirstChild(node);
	}    
	void accept(ASTVisitor *visitor) { visitor->visitUnOp(this); }    
    };

    class ASTAnd : public AST {
    public:
	ASTAnd(AST *left, const std::string &op, AST *right) : AST(op) {
	    setFirstChild(left);
	    left->setNextSibling(right);
	}    
	void accept(ASTVisitor *visitor) { visitor->visitAnd(this); }
    };

    class ASTIf : public AST {
    public:
	ASTIf(AST *condition, AST *s1, AST *s2) : AST("if") {
	    setFirstChild(condition);
	    condition->setNextSibling(s1);
	    s1->setNextSibling(s2);
	}
	void accept(ASTVisitor *visitor) { visitor->visitIf(this); }
    };

    class ASTBool : public AST {
    public:
	ASTBool(bool b) : AST("bool"), _value(b) {
	    setText(b ? "true" : "false");
	}
	void accept(ASTVisitor *visitor) { visitor->visitBool(this); }
	bool getValue() const { return _value; }
    private:
	bool _value;
    };

    class ASTDouble : public AST {
    public:
	ASTDouble(double d) : AST("double"), _value(d) {
	    char buf[256];
	    sprintf(buf, "%0.2f", d);
	    setText(buf);
	}
	void accept(ASTVisitor *visitor) { visitor->visitDouble(this); }
	double getValue() const { return _value; }
    private:
	double _value;
    };

    class ASTInt : public AST {
    public:
	ASTInt(int i) : AST("int"), _value(i) {
	    char buf[256];
	    sprintf(buf, "%d", i);
	    setText(buf);
	}
	void accept(ASTVisitor *visitor) { visitor->visitInt(this); }
	int getValue() const { return _value; }
    private:
	int _value;
    };

    class ASTString : public AST {
    public:
	ASTString(const std::string &s) : AST("string") {
	    setText(s);
	}
	void accept(ASTVisitor *visitor) { visitor->visitString(this); }
    };

    class ASTId : public AST {
    public:
	ASTId(const std::string &id) : AST("identifier") {
	    setText(id);
	}
	void accept(ASTVisitor *visitor) { visitor->visitId(this); }
    };

    class ASTExists : public AST {
    public:
	ASTExists(const std::string &id) : AST("exists") {
	    setText(id);
	}
	void accept(ASTVisitor *visitor) { visitor->visitExists(this); }
    };

};

#endif /* AST.hh */
