/* ------------------------------------------------------------------------
 *  $Id: Vector3D.cc,v 1.7 2001/07/30 15:19:00 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 1999-06-02 by Niklas Elmqvist.
 *
 * Copyright (c) 1999-2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2001 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <string.h>
#include <math.h>
#include <iostream>

// -- Local Includes
#include "Math.hh"
#include "Vector3D.hh"

// -- Code Segment

Vector3D::Vector3D(float x, float y, float z, float w)
{
    _coord[0] = x; _coord[1] = y; _coord[2] = z; _coord[3] = w;
}

Vector3D::Vector3D(const float c[3])
{
    clear();
    memcpy(_coord, c, sizeof(float) * 3);
}

Vector3D::Vector3D(const Vector3D &v)
{
    *this = v;
}

Vector3D &Vector3D::operator += (const Vector3D &v)
{
    for (int i = 0; i < 3; i++)
	_coord[i] += v._coord[i];
    return *this;
}

Vector3D &Vector3D::operator -= (const Vector3D &v)
{    
    for (int i = 0; i < 3; i++)
        _coord[i] -= v._coord[i];
    return *this;
}

Vector3D Vector3D::operator + (const Vector3D &v) const
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
        r._coord[i] = _coord[i] + v._coord[i];
    return r;
}

Vector3D Vector3D::operator - (const Vector3D &v) const
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
	r._coord[i] = _coord[i] - v._coord[i];
    return r;
}

Vector3D Vector3D::operator / (const Vector3D &v) const
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
	r._coord[i] = _coord[i] / v._coord[i];
    return r;
}

Vector3D Vector3D::operator * (const Vector3D &v) const
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
	r._coord[i] = _coord[i] * v._coord[i];
    return r;
}

Vector3D Vector3D::operator * (const Matrix3D &m) const
{
    Vector3D r(0.0f, 0.0f, 0.0f, 0.0f);
    for (int i = 0; i < 4; i++) {
	for (int j = 0; j < 4; j++) {
	    r._coord[i] += m(i, j) * _coord[j];
	}
    }
    return r;
}

Vector3D &Vector3D::operator += (float f)
{
    for (int i = 0; i < 3; i++)
        _coord[i] += f;
    return *this;
}

Vector3D &Vector3D::operator -= (float f)
{
    for (int i = 0; i < 3; i++)
	_coord[i] -= f;
    return *this;
}

Vector3D &Vector3D::operator /= (float f)
{
    for (int i = 0; i < 3; i++)
	_coord[i] /= f;
    return *this;
}

Vector3D &Vector3D::operator *= (float f)
{
    for (int i = 0; i < 3; i++)
	_coord[i] *= f;
    return *this;
}

Vector3D Vector3D::operator + (float f) const
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
	r._coord[i] = _coord[i] + f;
    return r;
}

Vector3D Vector3D::operator - (float f) const
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
	r._coord[i] = _coord[i] - f;
    return r;
}

Vector3D Vector3D::operator / (float f) const
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
	r._coord[i] = _coord[i] / f;
    return r;
}

Vector3D Vector3D::operator * (float f) const
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
	r._coord[i] = _coord[i] * f;
    return r;
}

Vector3D Vector3D::operator - () const
{
    return Vector3D(-_coord[0], -_coord[1], -_coord[2]);
}

Vector3D &Vector3D::operator = (const Vector3D &v)
{
    memcpy(_coord, v._coord, sizeof(_coord));
    for (int i = 0; i < 3; i++)
	_coord[i] = v._coord[i];
    return *this;
}

float Vector3D::magnitude()
{
    return (float) sqrt(_coord[0] * _coord[0] + 
			_coord[1] * _coord[1] + 
			_coord[2] * _coord[2]);
}

void Vector3D::normalize()
{
    // Compute magnitude of vector
    float mag = magnitude();
    
    // Sanity check
    if (mag == 0.0f) return;
    
    // Normalize all components with magnitude
    for (int i = 0; i < 3; i++)
	_coord[i] /= mag;
}

void Vector3D::clear()
{
    for (int i = 0; i < 3; i++) _coord[i] = 0.0f;
    _coord[3] = 1.0f;
}

void Vector3D::zero() 
{
    clear();
}


float Vector3D::dot(const Vector3D &a, const Vector3D &b)
{
    float r = 0.0f;
    for (int i = 0; i < 3; i++)
	r += a(i) * b(i);
    return r;
}

Vector3D Vector3D::cross(const Vector3D &a, const Vector3D &b) 
{
    return Vector3D(a.y() * b.z() - a.z() * b.y(),
		    a.z() * b.x() - a.x() * b.z(),
		    a.x() * b.y() - a.y() * b.x());
}

Vector3D operator * (float f, const Vector3D &v)
{
    Vector3D r;
    for (int i = 0; i < 3; i++)
	r(i) = f * v(i);
    return r;
}

std::ostream &operator << (std::ostream &f, const Vector3D &v)
{
    f << "[" << v.x() << ", " << v.y() << ", "
      << v.z() << ", " << v.w() << "]'";
    return f;
}

bool operator == (const Vector3D &a, const Vector3D &b)
{
    for (int i = 0; i < 4; i++) 
	if (Math::equal(a(i), b(i)) == false)
	    return false;
    return true;
}
