/* ------------------------------------------------------------------------
 * PointGeometryImpl.hh
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-10-24 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Nobel/Renderer.hh"
#include "Polhem/VolumeImpl.hh"
#include "Polhem/PointGeometryImpl.hh"

using namespace Nobel;

// -- Code Segment 

void PointGeometryImpl::setVertexNumber(CORBA::Long n)
{
    Guard<Mutex> guard(_mutex);

    // Make sure the value is legal 
    if (n > 0) {

	// Update the limit and the sequence length
	_vertexNumber = n;
	_mesh.vertexList.length(n);
    }
}

void PointGeometryImpl::setVertex(CORBA::Long index, const Vertex3D &v)
{
    Guard<Mutex> guard(_mutex);
    
    // Make sure the index is within legal range
    if (index >= 0 && index < _vertexNumber) {

	// Update the value and mark the geometry as dirty
	_mesh.vertexList[index] = v;
	_dirty = true;
    }
}

Vertex3D PointGeometryImpl::getVertex(CORBA::Long index)
{
    // Make sure the index is with legal range
    if (index < 0 || index >= _vertexNumber)
	throw Nobel::Geometry::OutOfRange();

    // It is, so return the value
    return _mesh.vertexList[index];
}

void PointGeometryImpl::render(Renderer_ptr r)
{
    Guard<Mutex> guard(_mutex);

    // Send the entire mesh to the renderer
    r->renderPoints(_mesh);
}

PointMesh *PointGeometryImpl::getMesh()
{
    // Create a new point mesh and copy contents from the existing mesh
    PointMesh *mesh_ptr = new PointMesh(_mesh);
    return mesh_ptr;
}

void PointGeometryImpl::setMesh(const PointMesh &mesh)
{
    Guard<Mutex> guard(_mutex);
    
    // Copy contents of the new mesh into the internal mesh
    _mesh = mesh; 
    
    // Update geometry limits
    _vertexNumber = mesh.vertexList.length();
}

void PointGeometryImpl::recomputeBoundingVolume()
{
    Vertex3D minimum, maximum;
    
    // Find the extremes of the vertex data
    findMinMax(_mesh.vertexList, _vertexNumber, minimum, maximum);
    
    // Update the bounding volume
    _bounds->setBounds(minimum, maximum);

    // Okay, the geometry isn't dirty anymore
    _dirty = false;
}
