/* ------------------------------------------------------------------------
 * $Id: SDLPlatform.cc,v 1.10 2001/07/18 07:48:37 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-04-28 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Antony Suter <antony@mira.net>.
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2001 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- OpenGL Includes
#include <GL/gl.h>
#include <GL/glu.h>

// -- 3Dwm Includes
#include "Celsius/Exception.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Thread.hh"
#include "Celsius/Logger.hh"
#include "Polhem/LogGroup.hh"
#include "Polhem/InputDevice.hh"
#include "Polhem/RendererImpl.hh"
#include "Polhem/SDLPlatform.hh"

#include "Polhem/InterTraxDevice.hh"

// -- Constants

// Maximum number of input events in the event queue
const int maxInputEvents = 64;

// Include the SDL event listener class! (it's the same for all SDL
// platforms)
#include "SDLInputListener.inl"

// -- Code Segment

SDLPlatform::SDLPlatform(int &argc, char **argv, int w, int h, int d)
    : _width(w), _height(h), _eventQueue(maxInputEvents)
{
    // Initialize the SDL library
    if (SDL_Init(SDL_INIT_VIDEO) < 0) {
	throw Exception(SDL_GetError());
    }
    
    // Retrieve color depth
    switch (d) {
    case 15: _depth = 15; break;
    case 16: _depth = 16; break;
    case 24: _depth = 32; break;
    case 32: _depth = 32; break;
    default: _depth = 32; break;
    }
    
    _screen = SDL_SetVideoMode(_width, _height, _depth, SDL_OPENGL);
    if (_screen == 0) {
	throw Exception(SDL_GetError());
    }
    
    // Configure SDL
    SDL_WM_SetCaption("3Dwm", "3Dwm");
    SDL_EventState(SDL_QUIT, SDL_ENABLE);
    SDL_EventState(SDL_ACTIVEEVENT, SDL_IGNORE);
    SDL_EnableKeyRepeat(500, 10);

    // Create and add the desired input devices to the event queue
    addInputDevice(new SDLInputListener(&_eventQueue));
    //addInputDevice(new InterTraxDevice(&_eventQueue));
}

SDLPlatform::~SDLPlatform()
{
    // Kill and deallocate the device threads
    for (std::vector<Thread *>::iterator i = _deviceThreads.begin();
	 i != _deviceThreads.end(); i++) {
	(*i)->kill();
	delete (*i);
    }

    // Deallocate the input devices as well
    for (std::vector<InputDevice *>::iterator i = _devices.begin();
	 i != _devices.end(); i++)
	delete (*i);
    
    // Free the SDL surfaces
    SDL_FreeSurface(_screen);
    
    // Cleanup the SDL lib
    SDL_Quit();
}


RendererImpl *SDLPlatform::createRenderer()
{
    // No need to create the OpenGL context as SDL does it for
    // us in this case.
    return new RendererImpl(_width, _height);
}

InputEvent SDLPlatform::getNextInputEvent()
{
    return _eventQueue.pop();
}

// TODO: chase these remaining gl calls out of here

void SDLPlatform::clearDisplay()
{
    glClear(GL_DEPTH_BUFFER_BIT | GL_COLOR_BUFFER_BIT);
}

void SDLPlatform::swapDisplay()
{
    // Flush GL rendering pipeline and swap the buffers
    glFlush();
    SDL_GL_SwapBuffers();    
}

void SDLPlatform::addInputDevice(InputDevice *device)
{
    Logger::log(LogGroup::Platform) 
	<< "Adding input device \"" << device->getDeviceName() << "\"" 
	<< std::endl;

    // First, add the device to the list of devices. Note that the
    // devices in this list are owned by us, so we will be the ones
    // that deallocate them.
    _devices.push_back(device);

    // Create a thread for the device and start it!
    Thread *device_thread = new Thread(device);
    device_thread->start();
    _deviceThreads.push_back(device_thread);
}
