/* ------------------------------------------------------------------------
 * $Id: SystemCommand.cc,v 1.7 2001/08/28 13:18:52 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-15 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Vector2D.hh"
#include "Celsius/Vector3D.hh"
#include "Nobel/Event.hh"
#include "Polhem/View.hh"
#include "Polhem/SceneManager.hh"
#include "Polhem/SystemCommand.hh"

// Internal Zorn function struct
struct function_t {
    std::string id;
    SystemCommand::FunctionMethod method;
    int paramCount;
};

// -- Code Segment

SystemCommand::SystemCommand(SceneManager *smanager)
    : _smanager(smanager)
{
    function_t functionList[] = {
	{ "server.quit",	&SystemCommand::quit, 			0 },
	{ "view.orbit",		&SystemCommand::viewOrbit, 		4 },
	{ "view.track",		&SystemCommand::viewTrack, 		2 },
	{ "view.dolly",		&SystemCommand::viewDolly, 		1 },
	{ "view.zoom",		&SystemCommand::viewZoom, 		1 },
	{ "event.key",		&SystemCommand::keyEvent,		2 },
	{ "event.mousebutton",	&SystemCommand::mouseButtonEvent,	4 },
	{ "event.mousemove",	&SystemCommand::mouseMoveEvent,		2 }
    };
    
    // Add the Zorn functions to the internal map (for speed)
    for (int i = 0; i < int(sizeof(functionList) / sizeof(function_t)); i++) {
	_functions[functionList[i].id] = functionList[i].method;
	_paramCount[functionList[i].id] = functionList[i].paramCount;
    }
}

SystemCommand::~SystemCommand()
{
    // empty
}
    
Zorn::Value SystemCommand::execute(const std::string &id,
				    const Zorn::ValueList &vs,
				    Zorn::Store *store)
{
    return (this->*_functions[id])(vs, store);
}

int SystemCommand::getParamCount(const std::string &id)
{
    return _paramCount[id];
}
    
std::vector<std::string> SystemCommand::getFunctionNames() const
{
    std::vector<std::string> names;

    // Traverse through all items and add the function names
    for (std::map<std::string, FunctionMethod>::const_iterator
	     i = _functions.begin(); i != _functions.end(); i++) 
	names.push_back(i->first);
    
    return names;    
}
    
Zorn::Value SystemCommand::quit(const Zorn::ValueList &vs,
				 Zorn::Store *store)
{
    // Terminate the scene manager
    _smanager->terminate();
    return true;
}

Zorn::Value SystemCommand::viewOrbit(const Zorn::ValueList &vs,
				      Zorn::Store *store)
{
    // Orbit the current view using the angle and the axis
    _smanager->getCurrentView()->
	orbit(vs[0].getDouble(), Vector3D(vs[1].getDouble(),
					  vs[2].getDouble(),
					  vs[3].getDouble()));
    return true;
}

Zorn::Value SystemCommand::viewDolly(const Zorn::ValueList &vs,
				      Zorn::Store *store)
{
    // Dolly the current view using the delta
    _smanager->getCurrentView()->dolly(vs[0].getDouble());
    return true;
}

Zorn::Value SystemCommand::viewTrack(const Zorn::ValueList &vs,
				      Zorn::Store *store)
{
    // Track the current view using the delta vector
    _smanager->getCurrentView()->
	track(Vector2D(vs[0].getDouble(), vs[1].getDouble()));
    return true;
}

Zorn::Value SystemCommand::viewZoom(const Zorn::ValueList &vs,
				     Zorn::Store *store)
{
    // Zoom the current view using the delta
    _smanager->getCurrentView()->zoom(vs[0].getDouble());
    return true;
}

Zorn::Value SystemCommand::mouseButtonEvent(const Zorn::ValueList &vs,
					    Zorn::Store *store)
{
    Nobel::Event e;
    
    // Construct the event
    bool press = vs[3].getBool();
    e.type = press ? Nobel::ButtonPress : Nobel::ButtonRelease;
    e.dev = 1;
    e.key = vs[2].getInt();
    
    // Send the positional to the scene manager
    //double mx = vs[0].getDouble();
    //double my = vs[1].getDouble();
    _smanager->handlePositional(Vector2D(vs[0].getInt(), vs[1].getInt()), e);
    return true;
}

Zorn::Value SystemCommand::mouseMoveEvent(const Zorn::ValueList &vs,
					  Zorn::Store *store)
{
    Nobel::Event e;
    
    // Construct the event
    e.type = Nobel::Movement;
    e.dev = 1;

    // Send the positional event to the scene manager
    _smanager->handlePositional(Vector2D(vs[0].getInt(), vs[1].getInt()), e);
    return true;
}

Zorn::Value SystemCommand::keyEvent(const Zorn::ValueList &vs,
				    Zorn::Store *store)
{
    Nobel::Event e;

    // Initialize the event
    bool press = vs[1].getBool();
    e.type = press ? Nobel::ButtonPress : Nobel::ButtonRelease;
    e.key = vs[0].getInt();
    e.dev = 0;
    
    // Send the nonpositional event to the scene manager
    _smanager->handleNonPositional(e);
    return true;
}
