/* ------------------------------------------------------------------------
 * $Id: StandardLibrary.cc,v 1.2 2001/06/20 12:32:15 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2001-06-18 by Niklas Elmqvist.
 *
 * Copyright (c) 2001 Niklas Elmqvist <elm@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <stdlib.h>
#include <math.h>

// -- Zorn Includes
#include "Store.hh"
#include "StandardLibrary.hh"

// Internal Zorn function struct
struct function_t {
    std::string id;
    Zorn::StandardLibrary::FunctionMethod method;
    int paramCount;
};

// -- Code Segment

Zorn::StandardLibrary::StandardLibrary()
{
    function_t functionList[] = {
	{ "exists",		&StandardLibrary::exists, 		1 },
	{ "sqrt", 		&StandardLibrary::sqrt, 		1 },
	{ "abs", 		&StandardLibrary::abs, 			1 },
	{ "exp", 		&StandardLibrary::exp, 			1 },
	{ "pi", 		&StandardLibrary::pi, 			0 },
	{ "print", 		&StandardLibrary::print, 		1 },
	{ "println", 		&StandardLibrary::println, 		1 },
	{ "undefine", 		&StandardLibrary::undefine, 		1 },
	{ "define", 		&StandardLibrary::define, 		1 },
	{ "dump", 		&StandardLibrary::dump, 		0 }
    };

    // Add the Zorn functions to the internal map (for speed)
    for (int i = 0; i < int(sizeof(functionList) / sizeof(function_t)); i++) {
	_functions[functionList[i].id] = functionList[i].method;
	_paramCount[functionList[i].id] = functionList[i].paramCount;
    }
}

Zorn::StandardLibrary::~StandardLibrary()
{
    // empty
}

Zorn::Value Zorn::StandardLibrary::execute(const std::string &id,
			       const ValueList &vs,
			       Store *store)
{
    return (this->*_functions[id])(vs, store);
}

int Zorn::StandardLibrary::getParamCount(const std::string &id)
{
    return _paramCount[id];
}

std::vector<std::string> Zorn::StandardLibrary::getFunctionNames() const
{
    std::vector<std::string> names;

    // Traverse through all items and add the function names
    for (std::map<std::string, FunctionMethod>::const_iterator
	     i = _functions.begin(); i != _functions.end(); i++) 
	names.push_back(i->first);
    
    return names;
}

Zorn::Value Zorn::StandardLibrary::exists(const ValueList &vs, Store *store)
{
    return store->exists(vs[0].getString());
}

Zorn::Value Zorn::StandardLibrary::sqrt(const ValueList &vs, Store *store)
{
    return ::sqrt(vs[0].getDouble());
}

Zorn::Value Zorn::StandardLibrary::abs(const ValueList &vs, Store *store)
{
    if (vs[0].getType() == Value::type_int)
	return ::abs(vs[0].getInt());
    else 
	return ::fabs(vs[0].getDouble());
}

Zorn::Value Zorn::StandardLibrary::exp(const ValueList &vs, Store *store)
{
    return ::exp(vs[0].getDouble());
}

Zorn::Value Zorn::StandardLibrary::pi(const ValueList &vs, Store *store)
{
    return M_PI;
}

Zorn::Value Zorn::StandardLibrary::print(const ValueList &vs, Store *store)
{
    std::cerr << vs[0];
    return true;
}

Zorn::Value Zorn::StandardLibrary::println(const ValueList &vs, Store *store)
{
    std::cerr << vs[0] << std::endl;
    return true;
}

Zorn::Value Zorn::StandardLibrary::dump(const ValueList &vs, Store *store)
{
    store->dump(std::cerr);
    return true;
}

Zorn::Value Zorn::StandardLibrary::undefine(const ValueList &vs, Store *store)
{
    // Make sure the value is defined and then remove it 
    if (store->exists(vs[0].getString()) == false)
	return false;
    store->remove(vs[0].getString());
    return true;
}

Zorn::Value Zorn::StandardLibrary::define(const ValueList &vs, Store *store)
{
    // Make sure the value is *not* defined and then add it
    if (store->exists(vs[0].getString()) == true)
	return false;
    store->set(vs[0].getString(), Value(true));
    return true;
}
