# Message detection classes for C language.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/report"
require "adlint/message"
require "adlint/traits"
require "adlint/monitor"
require "adlint/c/syntax"
require "adlint/c/format"
require "adlint/c/option"
require "adlint/c/util"
require "adlint/cpp/syntax"

module AdLint #:nodoc:
module C #:nodoc:

  class W0001 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_grouped_expression += method(:enter_grouped_expression)
      visitor.leave_grouped_expression += method(:leave_grouped_expression)
      @group_depth = 0
    end

    private
    def enter_grouped_expression(node)
      @group_depth += 1
      W(:W0001, node.location) if @group_depth == 33
    end

    def leave_grouped_expression(node)
      @group_depth -= 1
    end
  end

  class W0002 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_kandr_function_definition += method(:check)
    end

    private
    def check(node)
      W(:W0002, node.location, node.identifier.value)
    end
  end

  class W0003 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_switch_statement += method(:check)
    end

    private
    def check(node)
      if node.statement.kind_of?(CompoundStatement)
        unless have_default_statement?(node.statement)
          W(:W0003, node.location)
        end
      end
    end

    def have_default_statement?(compound_statement)
      compound_statement.block_items.any? do |block_item|
        while block_item.kind_of?(GenericLabeledStatement)
          block_item = block_item.statement
        end
        block_item.kind_of?(DefaultLabeledStatement)
      end
    end
  end

  class W0007 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_switch_statement += method(:enter_switch_statement)
    end

    private
    def enter_switch_statement(node)
      previous = nil
      node.statement.block_items.each do |item|
        if previous
          case item
          when CaseLabeledStatement, DefaultLabeledStatement
            case previous
            when CompoundStatement
              unless end_with_break_or_return_statement?(item)
                W(:W0007, item.location)
                warn_inner_case_labeled_statement(item)
              end
            when BreakStatement, ReturnStatement
              ;
            else
              W(:W0007, item.location)
              warn_inner_case_labeled_statement(item)
            end
          end
        end

        item = item.statement while item.kind_of?(LabeledStatement)
        previous = item
      end
    end

    def warn_inner_case_labeled_statement(node)
      child = node
      loop do
        child = child.statement
        break unless child.kind_of?(CaseLabeledStatement)
        W(:W0007, child.location)
      end
    end

    def end_with_break_or_return_statement?(statement)
      case statement
      when CompoundStatement
        end_with_break_or_return_statement?(statement.block_Items.last)
      when BreakStatement
        true
      when ReturnStatement
        true
      end
    end
  end

  class W0010 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_conditional_expression += method(:check)
    end

    private
    def check(node)
      if node.then_expression.have_side_effect? ||
          node.else_expression.have_side_effect?
        W(:W0010, node.location)
      end
    end
  end

  class W0013 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_continue_statement += method(:check)
    end

    private
    def check(node)
      W(:W0013, node.location)
    end
  end

  class W0016 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        css = format.conversion_specifiers
        if css.any? { |cs| cs.field_width_value > 509 }
          W(:W0016, format.location)
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return PrintfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0017 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        css = format.conversion_specifiers
        if css.any? { |cs| cs.field_width_value > 509 }
          W(:W0017, format.location)
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return ScanfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0018 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        css = format.conversion_specifiers
        if css.any? { |cs| cs.precision_value > 509 }
          W(:W0018, format.location)
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return PrintfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0019 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check_explicit_conversion)
    end

    private
    def check_explicit_conversion(expression,
                                  original_variable, result_variable)
      orig_type = original_variable.type
      conv_type = result_variable.type

      if orig_type.pointer? && orig_type.unqualify.base_type.const? &&
          conv_type.pointer? && !conv_type.unqualify.base_type.const?
        W(:W0019, expression.location)
      end
    end
  end

  class W0021 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check_explicit_conversion)
    end

    private
    def check_explicit_conversion(expression,
                                  original_variable, result_variable)
      orig_type = original_variable.type
      conv_type = result_variable.type

      if orig_type.pointer? && orig_type.unqualify.base_type.volatile? &&
          conv_type.pointer? && !conv_type.unqualify.base_type.volatile?
        W(:W0021, expression.location)
      end
    end
  end

  class W0023 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_additive_expr_evaled += method(:check_binary)
      interp.on_multiplicative_expr_evaled += method(:check_binary)
      interp.on_shift_expr_evaled += method(:check_binary)
      interp.on_and_expr_evaled += method(:check_binary)
      interp.on_exclusive_or_expr_evaled += method(:check_binary)
      interp.on_inclusive_or_expr_evaled += method(:check_binary)
    end

    private
    def check_binary(binary_expression, lhs_variable, rhs_variable, result)
      case
      when lhs_variable.type.pointer? && rhs_variable.type.scalar?
        if rhs_variable.value.scalar? &&
            rhs_variable.value.must_be_greater_than?(ScalarValue.of(1))
          W(:W0023, binary_expression.location)
        end
      when rhs_variable.type.pointer? && lhs_variable.type.scalar?
        if lhs_variable.value.scalar? &&
            lhs_variable.value.must_be_greater_than?(ScalarValue.of(1))
          W(:W0023, binary_expression.location)
        end
      end
    end
  end

  class W0024 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_additive_expr_evaled += method(:check_binary)
      interp.on_prefix_increment_expr_evaled += method(:check_unary_prefix)
      interp.on_postfix_increment_expr_evaled += method(:check_unary_postfix)
      interp.on_prefix_decrement_expr_evaled += method(:check_unary_prefix)
      interp.on_postfix_decrement_expr_evaled += method(:check_unary_postfix)
    end

    private
    def check_binary(binary_expression, lhs_variable, rhs_variable, result)
      case
      when lhs_variable.type.pointer? && rhs_variable.type.scalar?
        if rhs_variable.value.scalar? &&
            rhs_variable.value.must_be_equal_to?(ScalarValue.of(1))
          W(:W0024, binary_expression.location)
        end
      when rhs_variable.type.pointer? && lhs_variable.type.scalar?
        if lhs_variable.value.scalar? &&
            lhs_variable.value.must_be_equal_to?(ScalarValue.of(1))
          W(:W0024, binary_expression.location)
        end
      end
    end

    def check_unary_prefix(unary_expression, operand_variable, original_value)
      if operand_variable.type.pointer?
        W(:W0024, unary_expression.location)
      end
    end

    def check_unary_postfix(postfix_expression,
                            operand_variable, result_variable)
      if operand_variable.type.pointer?
        W(:W0024, postfix_expression.location)
      end
    end
  end

  class W0027 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_relational_expr_evaled += method(:check_comparison)
      interp.on_equality_expr_evaled += method(:check_comparison)
    end

    private
    def check_comparison(binary_expression,
                         lhs_variable, rhs_variable, result)
      return if binary_expression.lhs_operand.kind_of?(NullConstantSpecifier)
      return if binary_expression.rhs_operand.kind_of?(NullConstantSpecifier)

      if lhs_variable.type.pointer? || rhs_variable.type.pointer?
        W(:W0027, binary_expression.location)
      end
    end
  end

  class W0028 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_indirection_expr_evaled += method(:check_indirection)
      interp.on_member_access_expr_evaled += method(:check_member_access)
      interp.on_array_subscript_expr_evaled += method(:check_array_subscript)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check_indirection(indirection_expression, variable, dereferenced)
      return unless indirection_expression.operand.constant?(@enum_tbl)

      if variable.value.scalar? &&
          variable.value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0028, indirection_expression.location)
      end
    end

    def check_member_access(member_access_expression,
                            outer_variable, member_variable)
      return unless outer_variable.type.pointer?
      return unless member_access_expression.expression.constant?(@enum_tbl)
      if outer_variable.value.scalar? &&
          outer_variable.value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0028, member_access_expression.location)
      end
    end

    def check_array_subscript(array_subscript_expression,
                              pointer_variable, subscript_variable,
                              array_variable, result_variable)
      return unless pointer_variable.type.pointer?
      return unless array_subscript_expression.expression.constant?(@enum_tbl)
      if pointer_variable.value.scalar? &&
          pointer_variable.value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0028, array_subscript_expression.location)
      end
    end
  end

  class W0030 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check_binary)
      interp.on_additive_expr_evaled += method(:check_binary)
      interp.on_shift_expr_evaled += method(:check_binary)
      interp.on_and_expr_evaled += method(:check_binary)
      interp.on_exclusive_or_expr_evaled += method(:check_binary)
      interp.on_inclusive_or_expr_evaled += method(:check_binary)
      interp.on_prefix_increment_expr_evaled += method(:check_unary_prefix)
      interp.on_postfix_increment_expr_evaled += method(:check_unary_postfix)
      interp.on_prefix_decrement_expr_evaled += method(:check_unary_prefix)
      interp.on_postfix_decrement_expr_evaled += method(:check_unary_postfix)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check_binary(binary_expression, lhs_variable, rhs_variable, result)
      lhs_type, lhs_value = lhs_variable.type, lhs_variable.value
      rhs_type, rhs_value = rhs_variable.type, rhs_variable.value

      if binary_expression.lhs_operand.constant?(@enum_tbl) &&
          lhs_type.pointer? && lhs_value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0030, binary_expression.lhs_operand.location)
      end

      if binary_expression.rhs_operand.constant?(@enum_tbl) &&
          rhs_type.pointer? && rhs_value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0030, binary_expression.rhs_operand.location)
      end
    end

    def check_unary_prefix(unary_expression, operand_variable, original_value)
      type, value = operand_variable.type, original_value

      if unary_expression.operand.constant?(@enum_tbl) &&
          type.pointer? && value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0030, unary_expression.operand.location)
      end
    end

    def check_unary_postfix(postfix_expression,
                            operand_variable, result_variable)
      type, value = operand_variable.type, operand_variable.value

      if postfix_expression.operand.constant?(@enum_tbl) &&
          type.pointer? && value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0030, postfix_expression.operand.location)
      end
    end
  end

  class W0031 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:start_function)
      interp.on_parameter_defined += method(:add_parameter)
      interp.on_variable_referred += method(:use_parameter)
      interp.on_variable_value_referred += method(:use_parameter)
      interp.on_function_ended += method(:check_unused_parameter)
      @parameters = nil
    end

    private
    def start_function(function_definition, function)
      @parameters = {}
    end

    def add_parameter(parameter_definition, variable)
      return unless @parameters

      if variable.named?
        @parameters[variable.name] = [parameter_definition, false]
      end
    end

    def use_parameter(expression, variable)
      return unless @parameters

      if variable.named?
        if parameter_definition = @parameters[variable.name]
          @parameters[variable.name] = [parameter_definition, true]
        end
      end
    end

    def check_unused_parameter(function_definition, function)
      return unless @parameters

      @parameters.each do |name, param|
        W(:W0031, param[0].location, name) unless param[1]
      end

      @parameters = nil
    end
  end

  class W0033 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:collect_labels)
      interp.on_goto_stmt_evaled += method(:use_label)
      interp.on_function_ended += method(:check_unused_label)
      @labels = nil
    end

    private
    def collect_labels(function_definition, function)
      collector = LabelCollector.new
      function_definition.function_body.accept(collector)
      @labels = collector.labels
    end

    def end_function(function_definition)
      @labels = nil
    end

    def use_label(goto_statement, label_name)
      return unless @labels

      if label = @labels[label_name]
        @labels[label_name] = [label, true]
      end
    end

    def check_unused_label(function_definition, function)
      return unless @labels

      @labels.each do |name, label|
        W(:W0033, label[0].location, name) unless label[1]
      end

      @labels = nil
    end

    class LabelCollector < SyntaxTreeVisitor
      def initialize
        @labels = {}
      end

      attr_reader :labels

      def visit_generic_labeled_statement(node)
        @labels[node.label.value] = [node.label, false]
      end
    end
    private_constant :LabelCollector
  end

  class W0035 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_logical_or_expr_evaled += method(:check_logical)
      interp.on_logical_and_expr_evaled += method(:check_logical)
      interp.on_conditional_expr_evaled += method(:check_conditional)
    end

    private
    def check_logical(binary_expression, lhs_variable, rhs_variable, result)
      unless rhs_variable.type.scalar?
        W(:W0035, binary_expression.location)
      end
    end

    def check_conditional(conditional_expression, controlling_variable, result)
      unless controlling_variable.type.scalar?
        W(:W0035, conditional_expression.location)
      end
    end
  end

  class W0036 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_init_declarator += method(:check_init_decl)
      visitor.enter_struct_declarator += method(:check_struct_decl)
      visitor.enter_parameter_declaration += method(:check_parameter_decl)
      visitor.enter_kandr_function_definition += method(:check_kandr_fundef)
      visitor.enter_ansi_function_definition += method(:check_ansi_fundef)
      visitor.enter_type_name += method(:check_type_name)
    end

    private
    def check_init_decl(init_declarator)
      declarator_num = DeclaratorCounter.new.tap { |counter|
        init_declarator.declarator.accept(counter)
      }.result

      W(:W0036, init_declarator.location) if declarator_num > 12
    end

    def check_struct_decl(struct_declarator)
      if declarator = struct_declarator.declarator
        declarator_num = DeclaratorCounter.new.tap { |counter|
          declarator.accept(counter)
        }.result

        W(:W0036, struct_declarator.location) if declarator_num > 12
      end
    end

    def check_parameter_decl(parameter_declaration)
      if declarator = parameter_declaration.declarator
        declarator_num = DeclaratorCounter.new.tap { |counter|
          declarator.accept(counter)
        }.result

        W(:W0036, parameter_declaration.location) if declarator_num > 12
      end
    end

    def check_kandr_fundef(kandr_function_definition)
      declarator_num = DeclaratorCounter.new.tap { |counter|
        kandr_function_definition.declarator.accept(counter)
      }.result

      W(:W0036, kandr_function_definition.location) if declarator_num > 12
    end

    def check_ansi_fundef(ansi_function_definition)
      declarator_num = DeclaratorCounter.new.tap { |counter|
        ansi_function_definition.declarator.accept(counter)
      }.result

      W(:W0036, ansi_function_definition.location) if declarator_num > 12
    end

    def check_type_name(type_name)
      if declarator = type_name.abstract_declarator
        declarator_num = DeclaratorCounter.new.tap { |counter|
          declarator.accept(counter)
        }.result

        W(:W0036, type_name.location) if declarator_num > 12
      end
    end

    class DeclaratorCounter < SyntaxTreeVisitor
      def initialize
        @result = 0
      end

      attr_reader :result

      def visit_identifier_declarator(node)
        super
        if pointer = node.pointer
          @result += pointer.count { |token| token.type == "*" }
        end
      end

      def visit_array_declarator(node)
        super
        @result += 1
        if pointer = node.pointer
          @result += pointer.count { |token| token.type == "*" }
        end
      end

      def visit_ansi_function_declarator(node)
        node.base.accept(self)
        @result += 1
        if pointer = node.pointer
          @result += pointer.count { |token| token.type == "*" }
        end
      end

      def visit_kandr_function_declarator(node)
        super
        @result += 1
        if pointer = node.pointer
          @result += pointer.count { |token| token.type == "*" }
        end
      end

      def visit_abbreviated_function_declarator(node)
        super
        @result += 1
        if pointer = node.pointer
          @result += pointer.count { |token| token.type == "*" }
        end
      end

      def visit_pointer_abstract_declarator(node)
        super
        @result += node.pointer.count { |token| token.type == "*" }
      end

      def visit_array_abstract_declarator(node)
        super
        @result += 1
      end

      def visit_function_abstract_declarator(node)
        node.base.accept(self) if node.base
        @result += 1
      end
    end
    private_constant :DeclaratorCounter
  end

  class W0037 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_struct_specifier += method(:enter_struct_specifier)
      visitor.leave_struct_specifier += method(:leave_struct_specifier)
      visitor.enter_union_specifier += method(:enter_union_specifier)
      visitor.leave_union_specifier += method(:leave_union_specifier)
      @nest_level = 0
    end

    private
    def enter_struct_specifier(node)
      @nest_level += 1
      W(:W0037, node.location) if @nest_level == 16
    end

    def leave_struct_specifier(node)
      @nest_level -= 1
    end

    def enter_union_specifier(node)
      @nest_level += 1
      W(:W0037, node.location) if @nest_level == 16
    end

    def leave_union_specifier(node)
      @nest_level -= 1
    end
  end

  class W0038 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_definition += method(:check)
    end

    private
    def check(node)
      if node.type.aligned_byte_size > 32767
        W(:W0038, node.location, node.identifier.value)
      end
    end
  end

  class W0039 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_translation_unit += method(:enter_block)
      visitor.leave_translation_unit += method(:leave_block)
      visitor.enter_compound_statement += method(:enter_block)
      visitor.leave_compound_statement += method(:leave_block)
      visitor.enter_variable_definition += method(:add_identifier)
      visitor.enter_variable_declaration += method(:add_identifier)
      visitor.enter_typedef_declaration += method(:add_identifier)
      @block_stack = []
    end

    private
    def enter_block(node)
      @block_stack.push(0)
    end

    def leave_block(node)
      @block_stack.pop
    end

    def add_identifier(node)
      @block_stack[-1] += 1
      W(:W0039, node.location) if @block_stack.last == 128
    end
  end

  class W0040 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_struct_type_declaration += method(:check)
      visitor.enter_union_type_declaration += method(:check)
    end

    private
    def check(node)
      node.struct_declarations.each do |struct_declaration|
        struct_declaration.items.each do |member_declaration|
          memb_type = member_declaration.type
          next unless memb_type.scalar? && memb_type.integer?
          if memb_type.bitfield? && !memb_type.explicitly_signed?
            W(:W0040, node.location)
            return
          end
        end
      end
    end
  end

  class W0041 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_struct_type_declaration += method(:check_struct)
      visitor.enter_union_type_declaration += method(:check_union)
    end

    private
    def check_struct(node)
      member_num = node.struct_declarations.map { |decl|
        decl.items.size
      }.reduce(0) { |sum, num| sum + num }

      if member_num > 127
        W(:W0041, node.location)
      end
    end

    def check_union(node)
      member_num = node.struct_declarations.map { |decl|
        decl.items.size
      }.reduce(0) { |sum, num| sum + num }

      if member_num > 127
        W(:W0041, node.location)
      end
    end
  end

  class W0042 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_enum_type_declaration += method(:check)
    end

    private
    def check(node)
      if node.enum_specifier.enumerators.size > 127
        W(:W0042, node.location)
      end
    end
  end

  class W0043 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_definition += method(:enter_variable_definition)
    end

    private
    def enter_variable_definition(node)
      return unless node.initializer
      return unless node.type.array?

      if initializers = node.initializer.initializers
        if initializer_depth(node.initializer) == type_depth(node.type)
          check_well_balanced(node.type, initializers, node.initializer)
        else
          check_flattened(node.type, initializers, node.initializer)
        end
      end
    end

    def check_well_balanced(array_type, initializers, parent_initializer)
      return unless array_type.length

      if !initializers.empty? && initializers.size < array_type.length
        if initializers.size == 1 and expr = initializers.first.expression
          unless expr.kind_of?(ConstantSpecifier) && expr.constant.value == "0"
            warn(expr, parent_initializer)
          end
        else
          warn(initializers.first, parent_initializer)
        end
      end

      if array_type.base_type.array?
        initializers.each do |ini|
          if ini.initializers
            check_well_balanced(array_type.base_type, ini.initializers, ini)
          end
        end
      end
    end

    def check_flattened(array_type, initializers, parent_initializer)
      unless total_length = total_length(array_type)
        # NOTE: Cannot check the incomplete array.
        return
      end

      flattener = lambda { |initializer|
        initializer.expression || initializer.initializers.map(&flattener)
      }

      exprs = initializers.map { |initializer|
        flattener[initializer]
      }.flatten.compact

      if !exprs.empty? && exprs.size < total_length
        if exprs.size == 1 and first = exprs.first
          if first.kind_of?(ObjectSpecifier) && first.constant.value != "0"
            warn(first, parent_initializer)
          end
        else
          warn(exprs.first, parent_initializer)
        end
      end
    end

    def warn(node, parent_initializer)
      if parent_initializer
        W(:W0043, parent_initializer.location)
      else
        W(:W0043, node.location)
      end
    end

    def initializer_depth(initializer)
      if initializers = initializer.initializers
        1 + initializers.map { |ini| initializer_depth(ini) }.max
      else
        0
      end
    end

    def type_depth(type)
      case
      when type.array?
        1 + type_depth(type.base_type)
      when type.composite?
        type.members.empty? ?
          1 : 1 + type.members.map { |memb| type_depth(memb.type) }.max
      else
        0
      end
    end

    def total_length(array_type)
      result = 1
      type = array_type
      while type.array?
        if type.length
          result *= type.length
          type = type.base_type
        else
          return nil
        end
      end
      result
    end
  end

  class W0049 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_if_statement += method(:enter_if_statement)
      visitor.leave_if_statement += method(:leave_if_statement)
      visitor.enter_switch_statement += method(:enter_switch_statement)
      visitor.leave_switch_statement += method(:leave_switch_statement)
      visitor.enter_while_statement += method(:enter_while_statement)
      visitor.leave_while_statement += method(:leave_while_statement)
      visitor.enter_do_statement += method(:enter_do_statement)
      visitor.leave_do_statement += method(:leave_do_statement)
      visitor.enter_for_statement += method(:enter_for_statement)
      visitor.leave_for_statement += method(:leave_for_statement)
      visitor.enter_c99_for_statement += method(:enter_c99_for_statement)
      visitor.leave_c99_for_statement += method(:leave_c99_for_statement)
      @control_statement_level = 0
    end

    private
    def enter_if_statement(node)
      @control_statement_level += 1
      W(:W0049, node.location) if @control_statement_level == 16
    end

    def leave_if_statement(node)
      @control_statement_level -= 1
    end

    def enter_switch_statement(node)
      @control_statement_level += 1
      W(:W0049, node.location) if @control_statement_level == 16
    end

    def leave_switch_statement(node)
      @control_statement_level -= 1
    end

    def enter_while_statement(node)
      @control_statement_level += 1
      W(:W0049, node.location) if @control_statement_level == 16
    end

    def leave_while_statement(node)
      @control_statement_level -= 1
    end

    def enter_do_statement(node)
      @control_statement_level += 1
      W(:W0049, node.location) if @control_statement_level == 16
    end

    def leave_do_statement(node)
      @control_statement_level -= 1
    end

    def enter_for_statement(node)
      @control_statement_level += 1
      W(:W0049, node.location) if @control_statement_level == 16
    end

    def leave_for_statement(node)
      @control_statement_level -= 1
    end

    def enter_c99_for_statement(node)
      @control_statement_level += 1
      W(:W0049, node.location) if @control_statement_level == 16
    end

    def leave_c99_for_statement(node)
      @control_statement_level -= 1
    end
  end

  class W0050 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_switch_statement += method(:enter_switch_statement)
      visitor.leave_switch_statement += method(:leave_switch_statement)
      visitor.enter_case_labeled_statement += method(:check)
      @label_num_stack = []
    end

    private
    def enter_switch_statement(node)
      @label_num_stack.push(0)
    end

    def leave_switch_statement(node)
      @label_num_stack.pop
    end

    def check(node)
      unless @label_num_stack.empty?
        @label_num_stack[-1] += 1
        W(:W0050, node.location) if @label_num_stack[-1] == 258
      end
    end
  end

  class W0051 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:add_variable)
      interp.on_variable_defined += method(:add_variable)
      interp.on_function_declared += method(:add_function)
      interp.on_function_defined += method(:add_function)
      interp.on_translation_unit_ended += method(:check)
      @identifiers = Hash.new { |hash, key| hash[key] = [] }
    end

    private
    def add_variable(decl_or_def, variable)
      if variable.named?
        @identifiers[mangle(variable.name)].push(
          [decl_or_def.identifier, variable.declared_as_extern?])
      end
    end

    def add_function(decl_or_def, function)
      if function.named?
        @identifiers[mangle(function.name)].push(
          [decl_or_def.identifier, function.declared_as_extern?])
      end
    end

    def check(*)
      @identifiers.each do |key, id_array|
        id_array.uniq! { |id| id[0].value }
        if id_array.size > 1
          id_array.each_with_index do |id, index|
            next unless id[1]
            pair = index == 0 ? id_array[1] : id_array[0]
            C(:C0001, pair[0].location, pair[0].value)
            W(:W0051, id[0].location, pair[0].value)
          end
        end
      end
    end

    def mangle(name)
      truncated = name.slice(0...Traits.instance.of_linker.identifier_max)

      if Traits.instance.of_linker.identifier_ignore_case
        truncated.upcase
      else
        truncated
      end
    end
  end

  class W0052 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_declaration += method(:add_variable_decl)
      visitor.enter_variable_definition += method(:add_variable_def)
      visitor.enter_function_declaration += method(:add_function_decl)
      visitor.enter_ansi_function_definition += method(:add_function_def)
      visitor.enter_kandr_function_definition += method(:add_function_def)
      visitor.leave_translation_unit += method(:check)
      @identifiers = Hash.new { |hash, key| hash[key] = [] }
    end

    private
    def add_variable_decl(node)
      @identifiers[mangle(node.identifier.value)].push(node.identifier)
    end

    def add_variable_def(node)
      @identifiers[mangle(node.identifier.value)].push(node.identifier)
    end

    def add_function_decl(node)
      @identifiers[mangle(node.identifier.value)].push(node.identifier)
    end

    def add_function_def(node)
      @identifiers[mangle(node.identifier.value)].push(node.identifier)
    end

    def check(node)
      @identifiers.each do |key, id_array|
        id_array.uniq! { |id| id.value }
        if id_array.size > 1
          id_array.each_with_index do |id, index|
            pair = index == 0 ? id_array[1] : id_array[0]
            C(:C0001, pair.location, pair.value)
            W(:W0052, id.location, pair.value)
          end
        end
      end
    end

    def mangle(name)
      name.slice(0...Traits.instance.of_compiler.identifier_max)
    end
  end

  class W0058 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_string_literal_specifier += method(:check)
    end

    private
    def check(node)
      if node.literal.value.length - 2 > 509
        W(:W0058, node.location)
      end
    end
  end

  class W0062 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_typeof_type_specifier += method(:check)
    end

    private
    def check(node)
      W(:W0062, node.location)
    end
  end

  class W0063 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_alignof_expression += method(:check)
      visitor.enter_alignof_type_expression += method(:check)
    end

    private
    def check(node)
      W(:W0063, node.location)
    end
  end

  class W0064 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_constant_referred += method(:check)
    end

    private
    def check(constant_specifier, variable)
      if constant_specifier.constant.value =~ /\A0b/
        W(:W0064, constant_specifier.location)
      end
    end
  end

  class W0065 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_simple_assignment_expression += method(:check)
    end

    private
    def check(node)
      if node.lhs_operand.kind_of?(CastExpression)
        W(:W0065, node.location)
      end
    end
  end

  class W0066 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:check_ansi_function)
      visitor.enter_kandr_function_definition += method(:check_kandr_function)
      @interp = context[:c_interpreter]
    end

    private
    def check_ansi_function(node)
      if node.identifier.value == "main"
        check(node)
      end
    end

    def check_kandr_function(node)
      if node.identifier.value == "main"
        check(node)
      end
    end

    def check(node)
      unless node.type.return_type == @interp.int_type
        W(:W0066, node.location)
        return
      end

      if node.type.parameter_types.size == 1 &&
          node.type.parameter_types[0] == @interp.void_type
        return
      end

      if node.type.parameter_types.size == 2 &&
          node.type.parameter_types[0] == argc_type &&
          node.type.parameter_types[1] == argv_type
        return
      end

      W(:W0066, node.location)
    end

    def argc_type
      @interp.int_type
    end

    def argv_type
      @interp.array_type(@interp.pointer_type(@interp.char_type))
    end
  end

  class W0067 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_member_access_expr_evaled += method(:check_member_access)
    end

    private
    def check_member_access(member_access_expression,
                            outer_variable, member_variable)
      if outer_variable.type.pointer? &&
          outer_variable.type.unqualify.base_type.incomplete?
        return
      end

      if outer_variable.type.composite? && outer_variable.type.incomplete?
        return
      end

      W(:W0067, member_access_expression.location) unless member_variable
    end
  end

  class W0068 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_bit_access_by_value_expression += method(:check)
      visitor.enter_bit_access_by_pointer_expression += method(:check)
    end

    private
    def check(node)
      W(:W0068, node.location)
    end
  end

  class W0070 < MessageDetection
    private
    def do_prepare(context) end

    def do_execute(context)
      system_headers = context[:sources].select { |src| src.system_header? }
      system_headers.each do |src|
        symbols = context[:symbol_table].symbols_appeared_in(src)
        if symbols.all? { |symbol| symbol.useless? }
          W(:W0070, src.included_at, src.fpath)
        end
      end
    end
  end

  class W0071 < MessageDetection
    private
    def do_prepare(context) end

    def do_execute(context)
      user_headers = context[:sources].select { |src| src.user_header? }
      user_headers.each do |src|
        symbols = context[:symbol_table].symbols_appeared_in(src)
        if symbols.all? { |symbol| symbol.useless? }
          W(:W0071, src.included_at, src.fpath)
        end
      end
    end
  end

  class W0076 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_constant_referred += method(:check)
    end

    private
    def check(constant_specifier, variable)
      if constant_specifier.constant.value =~ /\A0x[0-9A-F]+\z/i
        W(:W0076, constant_specifier.location)
      end
    end
  end

  class W0077 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_constant_referred += method(:check)
    end

    private
    def check(constant_specifier, variable)
      case constant_specifier.constant.value
      when /\A(?:[0-9]*\.[0-9]*[Ee][+-]?[0-9]+|[0-9]+\.?[Ee][+-]?[0-9]+).*l\z/,
           /\A(?:[0-9]*\.[0-9]+|[0-9]+\.).*l\z/,
           /\A(?:0x[0-9A-Fa-f]+|0b[01]+|[0-9]+)l\z/
        W(:W0077, constant_specifier.location)
      end
    end
  end

  class W0078 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_function_declaration += method(:check)
    end

    private
    def check(node)
      if node.type.parameter_types.empty?
        W(:W0078, node.location)
      end
    end
  end

  class W0079 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_definition += method(:check)
      @interp = context[:c_interpreter]
    end

    private
    def check(node)
      return unless node.initializer && node.initializer.expression

      if node.type.array? && node.type.base_type.same_as?(@interp.char_type)
        if length = node.type.length
          visitor = StringLiteralSpecifierFinder.new
          node.initializer.expression.accept(visitor)
          if string_literal_specifier = visitor.result
            string = unquote_string_literal(string_literal_specifier)
            W(:W0079, node.location) if length <= string.length
          end
        end
      end
    end

    def unquote_string_literal(string_literal_specifier)
      quoted = string_literal_specifier.literal.value
      quoted.sub(/\AL?"(.*)"\z/, "\\1")
    end

    class StringLiteralSpecifierFinder < SyntaxTreeVisitor
      def initialize
        @result = nil
      end

      attr_reader :result

      def visit_string_literal_specifier(node)
        @result = node
      end
    end
    private_constant :StringLiteralSpecifierFinder
  end

  class W0080 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check)
      interp.on_block_started += method(:enter_block)
      interp.on_block_ended += method(:leave_block)
      @block_level = 0
    end

    private
    def check(variable_definition, variable)
      if @block_level == 0
        if variable.declared_as_extern? || variable.declared_as_static?
          if variable.type.const? && variable_definition.initializer.nil?
            W(:W0080, variable_definition.location)
          end
        end
      end
    end

    def enter_block(*)
      @block_level += 1
    end

    def leave_block(*)
      @block_level -= 1
    end
  end

  class W0081 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_unary_arithmetic_expr_evaled += method(:check)
    end

    private
    def check(unary_arithmetic_expression, operand_variable, result_variable)
      if unary_arithmetic_expression.operator.type == "-"
        if operand_variable.type.same_as?(@interp.unsigned_int_type) ||
            operand_variable.type.same_as?(@interp.unsigned_long_type) ||
            operand_variable.type.same_as?(@interp.unsigned_long_long_type)
          W(:W0081, unary_arithmetic_expression.location)
        end
      end
    end
  end

  class W0082 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_unary_arithmetic_expr_evaled += method(:check)
    end

    private
    def check(unary_arithmetic_expression, operand_variable, result_variable)
      if unary_arithmetic_expression.operator.type == "-"
        if operand_variable.type.same_as?(@interp.unsigned_char_type) ||
            operand_variable.type.same_as?(@interp.unsigned_short_type) ||
            operand_variable.type.bitfield?
          W(:W0082, unary_arithmetic_expression.location)
        end
      end
    end
  end

  class W0084 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_comma_separated_expression += method(:check)
    end

    private
    def check(node)
      if node.expressions.size > 1
        node.expressions[0..-2].each do |expr|
          W(:W0084, expr.location) unless expr.have_side_effect?
        end
      end
    end
  end

  class W0085 < MessageDetection
    private
    def do_prepare(context) end

    def do_execute(context)
      context[:c_syntax_tree].accept(Visitor.new(context))
    end

    class Visitor < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context)
        @context = context
      end

      def visit_expression_statement(node)
        unless node.expression && node.expression.have_side_effect?
          W(:W0085, node.location)
        end
      end

      def visit_for_statement(node)
        node.body_statement.accept(self)
      end

      def visit_c99_for_statement(node)
        node.body_statement.accept(self)
      end

      private
      def report
        @context.report
      end
    end
    private_constant :Visitor
  end

  class W0086 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_conditional_expression += method(:check)
    end

    private
    def check(node)
      if node.then_expression.have_side_effect? !=
          node.else_expression.have_side_effect?
        W(:W0086, node.location)
      end
    end
  end

  class W0087 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_simple_assignment_expression += method(:enter_assignment)
      visitor.leave_simple_assignment_expression += method(:leave_assignment)
      visitor.enter_compound_assignment_expression += method(:enter_assignment)
      visitor.leave_compound_assignment_expression += method(:leave_assignment)
      visitor.enter_function_call_expression += method(:enter_assignment)
      visitor.leave_function_call_expression += method(:leave_assignment)
      visitor.enter_comma_separated_expression += method(:check)
      @assignment_depth = 0
    end

    private
    def enter_assignment(node)
      @assignment_depth += 1
    end

    def leave_assignment(node)
      @assignment_depth -= 1
    end

    def check(node)
      if !node.expressions.last.have_side_effect? && @assignment_depth == 0
        W(:W0087, node.location)
      end
    end
  end

  class W0088 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_expression_statement += method(:enter_expression_statement)
      visitor.leave_expression_statement += method(:leave_expression_statement)
      visitor.enter_logical_and_expression += method(:check)
      visitor.enter_logical_or_expression += method(:check)
      @in_expression_statement = false
    end

    private
    def enter_expression_statement(node)
      @in_expression_statement = true
    end

    def leave_expression_statement(node)
      @in_expression_statement = false
    end

    def check(node)
      if @in_expression_statement
        unless node.rhs_operand.have_side_effect?
          W(:W0088, node.operator.location)
        end
        @in_expression_statement = false
      end
    end
  end

  class W0093 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(multiplicative_expression, lhs_variable, rhs_variable, result)
      return if multiplicative_expression.operator.type == "*"
      return if multiplicative_expression.rhs_operand.constant?(@enum_tbl)

      return unless rhs_variable.type.scalar? && rhs_variable.value.scalar?
      return if rhs_variable.value.must_be_equal_to?(ScalarValue.of(0))

      if rhs_variable.value.may_be_equal_to?(ScalarValue.of(0))
        W(:W0093, multiplicative_expression.location)
      end
    end
  end

  class W0096 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(multiplicative_expression, lhs_variable, rhs_variable, result)
      return if multiplicative_expression.operator.type == "*"
      return unless multiplicative_expression.rhs_operand.constant?(@enum_tbl)

      return unless rhs_variable.type.scalar? && rhs_variable.value.scalar?

      if rhs_variable.value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0096, multiplicative_expression.location)
      end
    end
  end

  class W0097 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(multiplicative_expression, lhs_variable, rhs_variable, result)
      return if multiplicative_expression.operator.type == "*"
      return if multiplicative_expression.rhs_operand.constant?(@enum_tbl)

      return unless rhs_variable.type.scalar? && rhs_variable.value.scalar?

      if rhs_variable.value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0097, multiplicative_expression.location)
      end
    end
  end

  class W0100 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:define_variable)
      interp.on_variable_initialized += method(:init_variable)
      interp.on_variable_value_updated += method(:write_variable)
      interp.on_block_started += method(:enter_block)
      interp.on_block_ended += method(:leave_block)
      @variable_stack = [{}]
    end

    private
    def enter_block(compound_statement)
      @variable_stack.push({})
    end

    def leave_block(compound_statement)
      check_constant_variables(@variable_stack.last)
      @variable_stack.pop
    end

    def define_variable(variable_definition, variable)
      if variable.named?
        @variable_stack.last[variable] = [variable_definition.location, 0]
      end
    end

    def init_variable(variable_definition, variable, init_variable)
      if record = @variable_stack.last[variable]
        record[1] += 1
      end
    end

    def write_variable(expression, variable)
      variable = variable.owner while variable.kind_of?(InnerVariable)
      return unless variable.named?

      @variable_stack.reverse_each do |variables|
        if record = variables[variable]
          record[1] += 1
        end
      end
    end

    def check_constant_variables(variables)
      variables.each do |variable, (location, assign_count)|
        case
        when variable.type.pointer? &&
             variable.type.unqualify.base_type.function?
          next
        when variable.type.array?
          if assign_count <= 1
            base_type = variable.type.base_type
            while base_type.array? || base_type.pointer?
              base_type = base_type.base_type
            end
            W(:W0100, location, variable.name) unless base_type.const?
          end
        when !variable.type.const?
          W(:W0100, location, variable.name) if assign_count <= 1
        end
      end
    end
  end

  class W0101 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_assignment_expr_evaled += method(:check)
    end

    private
    def check(assignment_expression, lhs_variable, rhs_variable)
      return unless lhs_variable.type.pointer? && rhs_variable.type.pointer?

      if rhs_pointee = @interp.pointee_of(rhs_variable)
        if rhs_pointee.variable? && rhs_pointee.named?
          return unless rhs_pointee.binding.memory.dynamic?
          # NOTE: An array typed parameter can be considerd as an alias of the
          #       corresponding argument.  So, it is safe to return an address
          #       of the argument.
          return if rhs_pointee.type.parameter? && rhs_pointee.type.array?

          case
          when lhs_variable.binding.memory.static?
            W(:W0101, assignment_expression.location)
          when lhs_variable.scope.depth < rhs_pointee.scope.depth
            W(:W0101, assignment_expression.location)
          end
        end
      end
    end
  end

  class W0102 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_started += method(:start_function)
      @interp.on_function_ended += method(:end_function)
      @interp.on_parameter_defined += method(:add_parameter)
      @interp.on_indirection_expr_evaled += method(:relate_pointer)
      @interp.on_assignment_expr_evaled += method(:check_assignment)
      @parameters = nil
      @pointer_relationship = nil
    end

    private
    def start_function(function_definition, function)
      @parameters = Set.new
      @pointer_relationship = {}
    end

    def end_function(function_definition, function)
      @parameters = nil
      @pointer_relationship = nil
    end

    def add_parameter(parameter_definition, variable)
      return unless @parameters

      if variable.named?
        @parameters.add(variable.name)
      end
    end

    def relate_pointer(indirection_expression, variable, dereferenced)
      return unless @pointer_relationship

      @pointer_relationship[dereferenced] = variable
    end

    def check_assignment(assignment_expression, lhs_variable, rhs_variable)
      return unless @parameters && @pointer_relationship
      return unless lhs_variable.type.pointer? && rhs_variable.type.pointer?

      if pointer = @pointer_relationship[lhs_variable]
        return unless pointer.named? && @parameters.include?(pointer.name)
      else
        return
      end

      if rhs_pointee = @interp.pointee_of(rhs_variable)
        if rhs_pointee.variable? && rhs_pointee.named?
          return unless rhs_pointee.binding.memory.dynamic?
          # NOTE: An array typed parameter can be considerd as an alias of the
          #       corresponding argument.  So, it is safe to return an address
          #       of the argument.
          return if rhs_pointee.type.parameter? && rhs_pointee.type.array?
          W(:W0102, assignment_expression.location)
        end
      end
    end
  end

  class W0103 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_started += method(:start_function)
      @interp.on_function_ended += method(:end_function)
      @interp.on_variable_defined += method(:add_local_variable)
      @interp.on_parameter_defined += method(:add_local_variable)
      @interp.on_return_stmt_evaled += method(:check_return_statement)
      @local_variables = nil
    end

    private
    def start_function(function_definition, function)
      @local_variables = Set.new
    end

    def end_function(function_definition, function)
      @local_variables = nil
    end

    def add_local_variable(variable_definition, variable)
      return unless @local_variables
      return unless variable.binding.memory.dynamic?

      if variable.named?
        @local_variables.add(variable.name)
      end
    end

    def check_return_statement(return_statement, result_variable)
      return unless @local_variables
      return unless result_variable && result_variable.type.pointer?

      if pointee = @interp.pointee_of(result_variable)
        if pointee.variable? && pointee.named?
          # NOTE: An array typed parameter can be considerd as an alias of the
          #       corresponding argument.  So, it is safe to return an address
          #       of the argument.
          return if pointee.type.parameter? && pointee.type.array?
          if @local_variables.include?(pointee.name)
            W(:W0103, return_statement.location)
          end
        end
      end
    end
  end

  class W0104 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:start_function)
      interp.on_function_ended += method(:check_constant_parameters)
      interp.on_parameter_defined += method(:add_parameter)
      interp.on_variable_value_updated += method(:write_parameter)
      @parameters = nil
    end

    private
    def start_function(function_definition, function)
      @parameters = {}
    end

    def add_parameter(parameter_definition, variable)
      return unless @parameters

      if variable.named?
        @parameters[variable.name] =
          [false, variable, parameter_definition.location]
      end
    end

    def write_parameter(expression, variable)
      return unless @parameters

      if variable.named? && @parameters.include?(variable.name)
        @parameters[variable.name][0] = true
      end
    end

    def check_constant_parameters(function_definition, function)
      return unless @parameters

      @parameters.each do |name, (written, variable, location)|
        next if variable.type.const?
        next if variable.type.array? && variable.type.base_type.const?

        W(:W0104, location, name) unless written
      end

      @parameters = nil
    end
  end

  class W0105 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:start_function)
      interp.on_function_ended += method(:check_constant_parameters)
      interp.on_parameter_defined += method(:add_parameter)
      interp.on_variable_value_updated += method(:write_parameter)
      interp.on_indirection_expr_evaled += method(:handle_indirection)
      interp.on_array_subscript_expr_evaled += method(:handle_array_subscript)
      @variable_relationship = nil
      @parameters = nil
    end

    private
    def start_function(function_definition, function)
      @variable_relationship = {}
      @parameters = {}
    end

    def add_parameter(parameter_definition, variable)
      return unless @parameters

      if variable.type.pointer? && variable.named?
        @parameters[variable] = [parameter_definition.location, false]
      end
    end

    def write_parameter(expression, variable)
      return unless @variable_relationship && @parameters

      if pointer = @variable_relationship[variable]
        if @parameters.include?(pointer)
          @parameters[pointer][1] = true
        end
      end
    end

    def handle_indirection(indirection_expression, variable, dereferenced)
      return unless @variable_relationship

      @variable_relationship[dereferenced] = variable
    end

    def handle_array_subscript(array_subscript_expression,
                               array_or_pointer_variable, subscript_variable,
                               array_variable, result_variable)
      return unless @variable_relationship

      if array_or_pointer_variable.type.pointer?
        @variable_relationship[result_variable] = array_or_pointer_variable
      end
    end

    def check_constant_parameters(function_definition, function)
      return unless @parameters

      @parameters.each do |pointer, (location, written)|
        base_type = pointer.type.unqualify.base_type
        unless base_type.function?
          unless written || base_type.const?
            W(:W0105, location, pointer.name)
          end
        end
      end

      @variable_relationship = nil
      @parameters = nil
    end
  end

  class W0107 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_assignment_expr_evaled += method(:check_assignment)
    end

    private
    def check_assignment(assignment_expression, lhs_variable, rhs_variable)
      return unless lhs_variable.type.pointer? && rhs_variable.type.pointer?

      if rhs_pointee = @interp.pointee_of(rhs_variable)
        if rhs_pointee.variable? && rhs_pointee.named?
          return unless rhs_pointee.binding.memory.dynamic?
          # NOTE: An array typed parameter can be considerd as an alias of the
          #       corresponding argument.  So, it is safe to return an address
          #       of the argument.
          return if rhs_pointee.type.parameter? && rhs_pointee.type.array?

          if lhs_variable.scope.local? && lhs_variable.binding.memory.static?
            W(:W0107, assignment_expression.location)
          end
        end
      end
    end
  end

  class W0108 < MessageDetection
    private
    def do_prepare(context) end

    def do_execute(context)
      context[:c_syntax_tree].accept(Visitor.new(context))
    end

    class Visitor < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context)
        @context = context
      end

      def visit_if_statement(node)
        super
        warn(node.expression)
      end

      def visit_if_else_statement(node)
        super
        warn(node.expression)
      end

      def visit_while_statement(node)
        super
        warn(node.expression)
      end

      def visit_do_statement(node)
        super
        warn(node.expression)
      end

      def visit_for_statement(node)
        super
        warn(node.condition_statement.expression)
      end

      def visit_c99_for_statement(node)
        super
        warn(node.condition_statement.expression)
      end

      def visit_relational_expression(node)
        super
        warn(node.lhs_operand)
        warn(node.rhs_operand)
      end

      def visit_equality_expression(node)
        super
        warn(node.lhs_operand)
        warn(node.rhs_operand)
      end

      def visit_unary_arithmetic_expression(node)
        super
        warn(node.operand) if node.operator.type == "!"
      end

      def visit_logical_and_expression(node)
        super
        warn(node.lhs_operand)
        warn(node.rhs_operand)
      end

      def visit_logical_or_expression(node)
        super
        warn(node.lhs_operand)
        warn(node.rhs_operand)
      end

      def visit_conditional_expression(node)
        super
        warn(node.condition)
      end

      private
      def warn(node)
        node = node.expression while node.kind_of?(GroupedExpression)

        case node
        when SimpleAssignmentExpression, CompoundAssignmentExpression
          W(:W0108, node.location)
        end
      end

      def report
        @context.report
      end
    end
    private_constant :Visitor
  end

  class W0109 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.implicit? && function.named?
        W(:W0109, function_call_expression.location, function.name)
      end
    end
  end

  class W0110 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_for_stmt_started += method(:handle_for_statement)
      @interp.on_c99_for_stmt_started += method(:handle_for_statement)
    end

    private
    def handle_for_statement(for_statement)
      for_statement.accept(ForStatementAnalyzer.new(@context, @interp))
    end

    class ForStatementAnalyzer < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, interp)
        @context = context
        @interp = interp
        @reported = false
      end

      def visit_for_statement(node)
        node.condition_statement.accept(self)
        node.expression.accept(self) if node.expression
      end

      def visit_c99_for_statement(node)
        node.condition_statement.accept(self)
        node.expression.accept(self) if node.expression
      end

      def visit_object_specifier(node)
        return if @reported

        if variable = @interp.variable_named(node.identifier.value)
          if variable.type.scalar? && variable.type.floating?
            W(:W0110, node.location)
            @reported = true
          end
        end
      end

      private
      def report
        @context.report
      end
    end
    private_constant :ForStatementAnalyzer
  end

  class W0112 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_equality_expr_evaled += method(:check_equality)
    end

    private
    def check_equality(equality_expression, lhs_variable, rhs_variable, result)
      if lhs_variable.type.scalar? && lhs_variable.type.floating? &&
          rhs_variable.type.scalar? && rhs_variable.type.floating?
        W(:W0112, equality_expression.location)
      end
    end
  end

  class W0114 < MessageDetection
    private
    def do_prepare(context) end

    def do_execute(context)
      context[:c_syntax_tree].accept(Visitor.new(context))
    end

    class Visitor < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context)
        @context = context
        @logical_operation_num = 0
      end

      def visit_if_statement(node)
        @logical_operation_num = 0
        node.expression.accept(self)
        check_logical_operation(node.location)
        node.statement.accept(self)
      end

      def visit_if_else_statement(node)
        @logical_operation_num = 0
        node.expression.accept(self)
        check_logical_operation(node.location)
        node.then_statement.accept(self)
        node.else_statement.accept(self)
      end

      def visit_while_statement(node)
        @logical_operation_num = 0
        node.expression.accept(self)
        check_logical_operation(node.location)
        node.statement.accept(self)
      end

      def visit_do_statement(node)
        node.statement.accept(self)
        @logical_operation_num = 0
        node.expression.accept(self)
        check_logical_operation(node.location)
      end

      def visit_for_statement(node)
        @logical_operation_num = 0
        node.condition_statement.accept(self)
        check_logical_operation(node.location)
        node.body_statement.accept(self)
      end

      def visit_c99_for_statement(node)
        @logical_operation_num = 0
        node.condition_statement.accept(self)
        check_logical_operation(node.location)
        node.body_statement.accept(self)
      end

      def visit_unary_arithmetic_expression(node)
        if node.operator.type == "!"
          @logical_operation_num += 1
        end
        super
      end

      def visit_relational_expression(node)
        @logical_operation_num += 1
        super
      end

      def visit_equality_expression(node)
        @logical_operation_num += 1
        super
      end

      def visit_logical_and_expression(node)
        @logical_operation_num += 1
        super
      end

      def visit_logical_or_expression(node)
        @logical_operation_num += 1
        super
      end

      def visit_conditional_expression(node)
        @logical_operation_num = 0
        node.condition.accept(self)
        check_logical_operation(node.condition.location)
        node.then_expression.accept(self)
        node.else_expression.accept(self)
      end

      private
      def check_logical_operation(location)
        W(:W0114, location) if @logical_operation_num == 0
      end

      def report
        @context.report
      end
    end
    private_constant :Visitor
  end

  class W0115 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_shift_expr_evaled += method(:check)
    end

    private
    def check(shift_expression, lhs_variable, rhs_variable, result)
      operator = shift_expression.operator.type
      return unless operator == "<<" || operator == "<<="

      return if lhs_variable.type.signed?

      if must_overflow?(lhs_variable, rhs_variable)
        W(:W0115, shift_expression.location)
      end
    end

    def must_overflow?(lhs_variable, rhs_variable)
      result = lhs_variable.value << rhs_variable.value
      result.must_be_greater_than?(ScalarValue.of(lhs_variable.type.max_value))
    end
  end

  class W0116 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_shift_expr_evaled += method(:check)
    end

    private
    def check(shift_expression, lhs_variable, rhs_variable, result)
      operator = shift_expression.operator.type
      return unless operator == "<<" || operator == "<<="

      return if lhs_variable.type.signed?

      if !must_overflow?(lhs_variable, rhs_variable) &&
          may_overflow?(lhs_variable, rhs_variable)
        W(:W0116, shift_expression.location)
      end
    end

    def must_overflow?(lhs_variable, rhs_variable)
      result = lhs_variable.value << rhs_variable.value
      result.must_be_greater_than?(ScalarValue.of(lhs_variable.type.max_value))
    end

    def may_overflow?(lhs_variable, rhs_variable)
      result = lhs_variable.value << rhs_variable.value
      result.may_be_greater_than?(ScalarValue.of(lhs_variable.type.max_value))
    end
  end

  class W0117 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_declared += method(:declare_function)
      interp.on_variable_declared += method(:declare_variable)
      interp.on_function_defined += method(:check_function)
      interp.on_variable_defined += method(:check_variable)
      @target_fpath = context[:sources].first.fpath
      @external_symbols = Set.new
    end

    private
    def declare_function(function_declaration, function)
      if function.named? && function.declared_as_extern?
        @external_symbols.add(function.name)
      end
    end

    def declare_variable(variable_declaration, variable)
      if variable.named? && variable.declared_as_extern?
        @external_symbols.add(variable.name)
      end
    end

    def check_function(function_definition, function)
      if function.named? && function.declared_as_extern?
        return if function.name == "main"

        unless @external_symbols.include?(function.name)
          if function_definition.location.fpath == @target_fpath
            W(:W0117, function_definition.location, function.name)
          end
        end
      end
    end

    def check_variable(variable_definition, variable)
      if variable.named? && variable.declared_as_extern?
        unless @external_symbols.include?(variable.name)
          if variable_definition.location.fpath == @target_fpath
            W(:W0117, variable_definition.location, variable.name)
          end
        end
      end
    end
  end

  class W0118 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_declared += method(:check_function)
      interp.on_variable_declared += method(:check_variable)
      @target_fpath = context[:sources].first.fpath
      @external_symbols = Set.new
    end

    private
    def check_function(function_declaration, function)
      if function.named? && function.declared_as_extern?
        unless @external_symbols.include?(function.name)
          if function_declaration.location.fpath == @target_fpath
            W(:W0118, function_declaration.location, function.name)
          else
            @external_symbols.add(function.name)
          end
        end
      end
    end

    def check_variable(variable_declaration, variable)
      if variable.named? && variable.declared_as_extern?
        unless @external_symbols.include?(variable.name)
          if variable_declaration.location.fpath == @target_fpath
            W(:W0118, variable_declaration.location, variable.name)
          else
            @external_symbols.add(variable.name)
          end
        end
      end
    end
  end

  class W0119 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_implicit_conv_performed += method(:check)
    end

    private
    def check(initializer_or_expression, original_variable, result_variable)
      return if from_type == to_type

      if original_variable.type.same_as?(from_type) &&
          result_variable.type.same_as?(to_type)
        W(message_id, initializer_or_expression.location)
      end
    end

    def from_type
      @interp.char_type
    end

    def to_type
      @interp.signed_char_type
    end

    def message_id
      self.class.name.sub(/\A.*::/, "").intern
    end
  end

  class W0120 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0121 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0122 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0123 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0124 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0125 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0126 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0127 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0128 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0129 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0130 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0131 < W0119
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0132 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0133 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0134 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0135 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0136 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0137 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0138 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0139 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0140 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0141 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0142 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0143 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0144 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0145 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0146 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0147 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0148 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0149 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0150 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0151 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0152 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0153 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0154 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0155 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0156 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0157 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0158 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0159 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0160 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0161 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0162 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0163 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0164 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0165 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0166 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0167 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0168 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0169 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0170 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0171 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0172 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0173 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0174 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0175 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0176 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0177 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0178 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0179 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0180 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0181 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0182 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0183 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0184 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0185 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0186 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0187 < W0119
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0188 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0189 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0190 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0191 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0192 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0193 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0194 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0195 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0196 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0197 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0198 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0199 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0200 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0201 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0202 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0203 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0204 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0205 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0206 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0207 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0208 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0209 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0210 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0211 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0212 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0213 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0214 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0215 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0216 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0217 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0218 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0219 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0220 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0221 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0222 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0223 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0224 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0225 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0226 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0227 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0228 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0229 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0230 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0231 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0232 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0233 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0234 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0235 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0236 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0237 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0238 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0239 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0240 < W0119
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0241 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0242 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0243 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0244 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0245 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0246 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0247 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0248 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0249 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0250 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0251 < W0119
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0252 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0253 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0254 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0255 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_started += method(:start_function)
      @interp.on_function_ended += method(:end_function)
      @interp.on_return_stmt_evaled += method(:check_return_statement)
      @current_function = nil
    end

    private
    def start_function(function_definition, function)
      @current_function = function
    end

    def end_function(function_definition, function)
      @current_function = nil
    end

    def check_return_statement(return_statement, result_variable)
      return unless @current_function
      return if from_type == to_type
      return unless result_variable

      if result_variable.type.same_as?(from_type) &&
          @current_function.type.return_type.same_as?(to_type)
        W(message_id, return_statement.location, @current_function.name)
      end
    end

    def from_type
      @interp.char_type
    end

    def to_type
      @interp.signed_char_type
    end

    def message_id
      self.class.name.sub(/\A.*::/, "").intern
    end
  end

  class W0256 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0257 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0258 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0259 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0260 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0261 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0262 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0263 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0264 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0265 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0266 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0267 < W0255
    private
    def from_type
      @interp.char_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0268 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0269 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0270 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0271 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0272 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0273 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0274 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0275 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0276 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0277 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0278 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0279 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0280 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0281 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0282 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0283 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0284 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0285 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0286 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0287 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0288 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0289 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0290 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0291 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0292 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0293 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0294 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0295 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0296 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0297 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0298 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0299 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0300 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0301 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0302 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0303 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0304 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0305 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0306 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0307 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0308 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0309 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0310 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0311 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0312 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0313 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0314 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0315 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0316 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0317 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0318 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0319 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0320 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0321 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0322 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0323 < W0255
    private
    def from_type
      @interp.signed_char_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0324 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0325 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0326 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0327 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0328 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0329 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0330 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0331 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0332 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0333 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0334 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0335 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0336 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0337 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0338 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0339 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0340 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0341 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0342 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0343 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0344 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0345 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0346 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0347 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0348 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0349 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0350 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0351 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0352 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0353 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0354 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0355 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0356 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0357 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0358 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0359 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0360 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0361 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0362 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0363 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0364 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0365 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0366 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.char_type
    end
  end

  class W0367 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0368 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0369 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0370 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0371 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0372 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0373 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0374 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0375 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0376 < W0255
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0377 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0378 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0379 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0380 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.unsigned_long_long_type
    end
  end

  class W0381 < W0255
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0382 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0383 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0384 < W0255
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0385 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0386 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0387 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0388 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0389 < W0255
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0390 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0391 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0392 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0393 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0394 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0395 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0396 < W0255
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0397 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0398 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0399 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0400 < W0255
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0401 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0402 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0403 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0404 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0405 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0406 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0407 < W0255
    private
    def from_type
      @interp.unsigned_char_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0408 < W0255
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0409 < W0255
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0410 < W0255
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.long_long_type
    end
  end

  class W0411 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_enum_type_declaration += method(:check)
    end

    private
    def check(node)
      if enumerators = node.enum_specifier.enumerators
        init_exprs = enumerators.map { |enum| enum.expression }
        return if init_exprs.all? { |expr| expr.nil? }
        return if init_exprs.first &&
                  init_exprs[1..-1].all? { |expr| expr.nil? }
        return if init_exprs.all? { |expr| !expr.nil? }
        W(:W0411, node.location)
      end
    end
  end

  class W0413 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_if_statement += method(:check_if_statement)
      visitor.enter_if_else_statement += method(:check_if_else_statement)
      visitor.enter_while_statement += method(:check_while_statement)
      visitor.enter_do_statement += method(:check_do_statement)
      visitor.enter_for_statement += method(:check_for_statement)
      visitor.enter_c99_for_statement += method(:check_for_statement)
    end

    private
    def check_if_statement(node)
      unless node.header_terminator.location.line_no ==
          node.statement.head_location.line_no
        unless node.statement.kind_of?(CompoundStatement)
          W(:W0413, node.statement.location)
        end
      end
    end

    def check_if_else_statement(node)
      unless node.then_header_terminator.location.line_no ==
          node.then_statement.head_location.line_no
        unless node.then_statement.kind_of?(CompoundStatement)
          W(:W0413, node.then_statement.location)
        end
      end

      unless node.else_header_terminator.location.line_no ==
          node.else_statement.head_location.line_no
        case node.else_statement
        when CompoundStatement, IfStatement, IfElseStatement
        else
          W(:W0413, node.else_statement.location)
        end
      end
    end

    def check_while_statement(node)
      unless node.header_terminator.location.line_no ==
          node.statement.head_location.line_no
        unless node.statement.kind_of?(CompoundStatement)
          W(:W0413, node.statement.location)
        end
      end
    end

    def check_do_statement(node)
      unless node.header_terminator.location.line_no ==
          node.statement.head_location.line_no
        unless node.statement.kind_of?(CompoundStatement)
          W(:W0413, node.statement.location)
        end
      end
    end

    def check_for_statement(node)
      unless node.header_terminator.location.line_no ==
          node.body_statement.head_location.line_no
        unless node.body_statement.kind_of?(CompoundStatement)
          W(:W0413, node.body_statement.location)
        end
      end
    end
  end

  class W0414 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_if_statement += method(:check_if_statement)
      visitor.enter_if_else_statement += method(:check_if_else_statement)
      visitor.enter_while_statement += method(:check_while_statement)
      visitor.enter_do_statement += method(:check_do_statement)
      visitor.enter_for_statement += method(:check_for_statement)
      visitor.enter_c99_for_statement += method(:check_for_statement)
    end

    private
    def check_if_statement(node)
      if node.header_terminator.location.line_no ==
          node.statement.head_location.line_no
        unless node.statement.kind_of?(CompoundStatement)
          W(:W0414, node.statement.location)
        end
      end
    end

    def check_if_else_statement(node)
      if node.then_header_terminator.location.line_no ==
          node.then_statement.head_location.line_no
        unless node.then_statement.kind_of?(CompoundStatement)
          W(:W0414, node.then_statement.location)
        end
      end

      if node.else_header_terminator.location.line_no ==
          node.else_statement.head_location.line_no
        case node.else_statement
        when CompoundStatement, IfStatement, IfElseStatement
        else
          W(:W0414, node.else_statement.location)
        end
      end
    end

    def check_while_statement(node)
      if node.header_terminator.location.line_no ==
          node.statement.head_location.line_no
        unless node.statement.kind_of?(CompoundStatement)
          W(:W0414, node.statement.location)
        end
      end
    end

    def check_do_statement(node)
      if node.header_terminator.location.line_no ==
          node.statement.head_location.line_no
        unless node.statement.kind_of?(CompoundStatement)
          W(:W0414, node.statement.location)
        end
      end
    end

    def check_for_statement(node)
      if node.header_terminator.location.line_no ==
          node.body_statement.head_location.line_no
        unless node.body_statement.kind_of?(CompoundStatement)
          W(:W0414, node.body_statement.location)
        end
      end
    end
  end

  class W0421 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_indirection_expr_evaled += method(:check_indirection)
      interp.on_member_access_expr_evaled += method(:check_member_access)
      interp.on_array_subscript_expr_evaled += method(:check_array_subscript)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check_indirection(indirection_expression, variable, dereferenced)
      return if indirection_expression.operand.constant?(@enum_tbl)
      if variable.value.scalar? &&
          variable.value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0421, indirection_expression.location)
      end
    end

    def check_member_access(member_access_expression,
                            outer_variable, member_variable)
      return unless outer_variable.type.pointer?
      return if member_access_expression.expression.constant?(@enum_tbl)
      if outer_variable.value.scalar? &&
          outer_variable.value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0421, member_access_expression.location)
      end
    end

    def check_array_subscript(array_subscript_expression,
                              pointer_variable, subscript_variable,
                              array_variable, result_variable)
      return unless pointer_variable.type.pointer?
      return if array_subscript_expression.expression.constant?(@enum_tbl)
      if pointer_variable.value.scalar? &&
          pointer_variable.value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0421, array_subscript_expression.location)
      end
    end
  end

  class W0422 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_indirection_expr_evaled += method(:check_indirection)
      interp.on_member_access_expr_evaled += method(:check_member_access)
      interp.on_array_subscript_expr_evaled += method(:check_array_subscript)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check_indirection(indirection_expression, variable, dereferenced)
      return if indirection_expression.operand.constant?(@enum_tbl)
      return unless variable.value.scalar?

      if !variable.value.must_be_equal_to?(ScalarValue.of(0)) &&
          variable.value.may_be_equal_to?(ScalarValue.of(0))
        W(:W0422, indirection_expression.location)
      end
    end

    def check_member_access(member_access_expression,
                            outer_variable, member_variable)
      return unless outer_variable.type.pointer?
      return if member_access_expression.expression.constant?(@enum_tbl)
      return unless outer_variable.value.scalar?

      if !outer_variable.value.must_be_equal_to?(ScalarValue.of(0)) &&
          outer_variable.value.may_be_equal_to?(ScalarValue.of(0))
        W(:W0422, member_access_expression.location)
      end
    end

    def check_array_subscript(array_subscript_expression,
                              pointer_variable, subscript_variable,
                              array_variable, result_variable)
      return unless pointer_variable.type.pointer?
      return if array_subscript_expression.expression.constant?(@enum_tbl)
      return unless pointer_variable.value.scalar?

      if !pointer_variable.value.must_be_equal_to?(ScalarValue.of(0)) &&
          pointer_variable.value.may_be_equal_to?(ScalarValue.of(0))
        W(:W0422, array_subscript_expression.location)
      end
    end
  end

  class W0423 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check_binary)
      interp.on_additive_expr_evaled += method(:check_binary)
      interp.on_shift_expr_evaled += method(:check_binary)
      interp.on_and_expr_evaled += method(:check_binary)
      interp.on_exclusive_or_expr_evaled += method(:check_binary)
      interp.on_inclusive_or_expr_evaled += method(:check_binary)
      interp.on_prefix_increment_expr_evaled += method(:check_unary_prefix)
      interp.on_postfix_increment_expr_evaled += method(:check_unary_postfix)
      interp.on_prefix_decrement_expr_evaled += method(:check_unary_prefix)
      interp.on_postfix_decrement_expr_evaled += method(:check_unary_postfix)
    end

    private
    def check_binary(binary_expression, lhs_variable, rhs_variable, result)
      lhs_type, lhs_value = lhs_variable.type, lhs_variable.value
      rhs_type, rhs_value = rhs_variable.type, rhs_variable.value

      if lhs_type.pointer? && lhs_value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0423, binary_expression.lhs_operand.location)
      end

      if rhs_type.pointer? && rhs_value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0423, binary_expression.rhs_operand.location)
      end
    end

    def check_unary_prefix(unary_expression, operand_variable, original_value)
      type, value = operand_variable.type, original_value

      if type.pointer? && value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0423, unary_expression.operand.location)
      end
    end

    def check_unary_postfix(postfix_expression,
                            operand_variable, result_variable)
      type, value = operand_variable.type, operand_variable.value

      if type.pointer? && value.must_be_equal_to?(ScalarValue.of(0))
        W(:W0423, postfix_expression.operand.location)
      end
    end
  end

  class W0424 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check_binary)
      interp.on_additive_expr_evaled += method(:check_binary)
      interp.on_shift_expr_evaled += method(:check_binary)
      interp.on_and_expr_evaled += method(:check_binary)
      interp.on_exclusive_or_expr_evaled += method(:check_binary)
      interp.on_inclusive_or_expr_evaled += method(:check_binary)
      interp.on_prefix_increment_expr_evaled += method(:check_unary_prefix)
      interp.on_postfix_increment_expr_evaled += method(:check_unary_postfix)
      interp.on_prefix_decrement_expr_evaled += method(:check_unary_prefix)
      interp.on_postfix_decrement_expr_evaled += method(:check_unary_postfix)
    end

    private
    def check_binary(binary_expression, lhs_variable, rhs_variable, result)
      lhs_type, lhs_value = lhs_variable.type, lhs_variable.value
      rhs_type, rhs_value = rhs_variable.type, rhs_variable.value

      if lhs_type.pointer? &&
          !lhs_value.must_be_equal_to?(ScalarValue.of(0)) &&
           lhs_value.may_be_equal_to?(ScalarValue.of(0))
        W(:W0424, binary_expression.lhs_operand.location)
      end

      if rhs_type.pointer? &&
          !rhs_value.must_be_equal_to?(ScalarValue.of(0)) &&
           rhs_value.may_be_equal_to?(ScalarValue.of(0))
        W(:W0424, binary_expression.rhs_operand.location)
      end
    end

    def check_unary_prefix(unary_expression, operand_variable, original_value)
      type, value = operand_variable.type, original_value

      if type.pointer? &&
          !value.must_be_equal_to?(ScalarValue.of(0)) &&
           value.may_be_equal_to?(ScalarValue.of(0))
        W(:W0424, unary_expression.operand.location)
      end
    end

    def check_unary_postfix(postfix_expression,
                            operand_variable, result_variable)
      type, value = operand_variable.type, operand_variable.value

      if type.pointer? &&
          !value.must_be_equal_to?(ScalarValue.of(0)) &&
           value.may_be_equal_to?(ScalarValue.of(0))
        W(:W0424, postfix_expression.operand.location)
      end
    end
  end

  class W0425 < MessageDetection
    private
    def do_prepare(context) end

    def do_execute(context)
      context[:c_syntax_tree].accept(Visitor.new(context))
    end

    class Visitor < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context)
        @context = context
        @decl_location = Location.new
        @stmt_location = Location.new
      end

      def visit_function_declaration(node)
        check_declaration(node)
      end

      def visit_variable_declaration(node)
        check_declaration(node)
      end

      def visit_variable_definition(node)
        check_declaration(node)
      end

      def visit_typedef_declaration(node)
        check_declaration(node)
      end

      def visit_struct_type_declaration(node)
        check_declaration(node)
      end

      def visit_union_type_declaration(node)
        check_declaration(node)
      end

      def visit_enum_type_declaration(node)
        check_declaration(node)
      end

      def visit_generic_labeled_statement(node)
        check_statement(node)
      end

      def visit_case_labeled_statement(node)
        check_statement(node)
      end

      def visit_default_labeled_statement(node)
        check_statement(node)
      end

      def visit_expression_statement(node)
        check_statement(node)
      end

      def visit_if_statement(node)
        check_statement(node)
        node.statement.accept(self)
      end

      def visit_if_else_statement(node)
        check_statement(node)
        node.then_statement.accept(self)
        node.else_statement.accept(self)
      end

      def visit_switch_statement(node)
        check_statement(node)
        node.statement.accept(self)
      end

      def visit_while_statement(node)
        check_statement(node)
        node.statement.accept(self)
      end

      def visit_do_statement(node)
        check_statement(node)
        node.statement.accept(self)
      end

      def visit_for_statement(node)
        check_statement(node)
        node.body_statement.accept(self)
      end

      def visit_c99_for_statement(node)
        check_statement(node)
        node.body_statement.accept(self)
      end

      def visit_goto_statement(node)
        check_statement(node)
      end

      def visit_continue_statement(node)
        check_statement(node)
      end

      def visit_break_statement(node)
        check_statement(node)
      end

      def visit_return_statement(node)
        check_statement(node)
      end

      private
      def check_declaration(node)
        if @decl_location.fpath == node.location.fpath &&
            @decl_location.line_no == node.location.line_no
          W(:W0425, node.location)
        else
          @decl_location = node.location
        end
      end

      def check_statement(node)
        if @stmt_location.fpath == node.location.fpath &&
            @stmt_location.line_no == node.location.line_no
          W(:W0425, node.location)
        else
          @stmt_location = node.location
        end
      end

      def report
        @context.report
      end
    end
    private_constant :Visitor
  end

  class W0431 < MessageDetection
    include MonitorUtil
    include CodingStyleAccessor

    private
    def do_prepare(context)
      fpath = context[:sources].first.fpath
      token_array = context[:c_token_array]
      @tokens = token_array.select { |token| token.location.fpath == fpath }
      @index = 0
      @indent_level = 0
      @indent_widths = Hash.new(0)
      @paren_depth = 0
      @last_line_no = 0
    end

    def do_execute(context)
      while token = next_token
        case token.type
        when "{"
          on_left_brace(token)
        when "}"
          on_right_brace(token)
        when "("
          on_left_paren(token)
        when ")"
          on_right_paren(token)
        end

        case token.type
        when :IF, :FOR, :WHILE
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          skip_controlling_part
          unless token = peek_token and token.type == "{"
            skip_simple_substatement
          end
        when :ELSE
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          unless token = peek_token and token.type == :IF || token.type == "{"
            skip_simple_substatement
          end
        when :DO
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          unless token = peek_token and token.type == "{"
            skip_simple_substatement
          end
        else
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
        end
      end
    end

    def skip_controlling_part
      condition_paren_depth = 0
      while token = next_token
        case token.type
        when "("
          condition_paren_depth += 1
        when ")"
          condition_paren_depth -= 1
          break if condition_paren_depth == 0
        end
      end
    end

    def skip_simple_substatement
      paren_depth = 0
      while token = next_token
        case token.type
        when "("
          paren_depth += 1
        when ")"
          paren_depth -= 1
        end

        case token.type
        when :IF, :FOR, :WHILE
          skip_controlling_part
          unless token = peek_token and token.type == "{"
            skip_simple_substatement
            break
          end
        when :ELSE
          unless token = peek_token and token.type == :IF || token.type == "{"
            skip_simple_substatement
            break
          end
        when :DO
          unless token = peek_token and token.type == "{"
            skip_simple_substatement
            skip_simple_substatement
            break
          end
        when ";"
          break if paren_depth == 0
        end
      end
    end

    def next_token
      return nil unless token = peek_token
      @index += 1

      case token.type
      when :CASE
        while token = peek_token
          @index += 1
          break if token.type == ":"
        end
      when :IDENTIFIER, :DEFAULT
        if next_token = @tokens[@index] and next_token.type == ":"
          token = peek_token
          @index += 1
        end
      end

      token
    end

    def peek_token
      if token = @tokens[@index]
        @last_line_no = @tokens[[0, @index - 1].max].location.line_no
        checkpoint(token.location)
      end
      token
    end

    def on_left_brace(token)
      if indent_style == INDENT_STYLE_GNU && @indent_level > 0
        @indent_level += 2
      else
        @indent_level += 1
      end
    end

    def on_right_brace(token)
      if indent_style == INDENT_STYLE_GNU
        @indent_level -= 2
        @indent_level = 0 if @indent_level < 0
      else
        @indent_level -= 1
      end
    end

    def on_left_paren(token)
      @paren_depth += 1
    end

    def on_right_paren(token)
      @paren_depth -= 1
    end

    def on_beginning_of_line(token)
      return if @paren_depth > 0

      case token.type
      when "{"
        if @indent_level == 0
          widths_index = @indent_level
        else
          widths_index = @indent_level - 1
        end
      when "}"
        if indent_style == INDENT_STYLE_GNU && @indent_level > 0
          widths_index = @indent_level + 1
        else
          widths_index = @indent_level
        end
      else
        widths_index = @indent_level
      end

      expected_column_no = @indent_widths[widths_index]
      if token.location.appearance_column_no < expected_column_no
        W(:W0431, token.location)
      end

      @indent_widths[widths_index] = token.location.appearance_column_no
    end
  end

  class W0432 < MessageDetection
    include MonitorUtil
    include CodingStyleAccessor

    private
    def do_prepare(context)
      fpath = context[:sources].first.fpath
      token_array = context[:c_token_array]
      @tokens = token_array.select { |token| token.location.fpath == fpath }
      @index = 0
      @indent_level = 0
      @indent_width = indent_width
      @paren_depth = 0
      @last_line_no = 0
    end

    def do_execute(context)
      while token = next_token
        case token.type
        when "{"
          on_left_brace(token)
        when "}"
          on_right_brace(token)
        when "("
          on_left_paren(token)
        when ")"
          on_right_paren(token)
        end

        case token.type
        when :IF, :FOR, :WHILE
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          skip_controlling_part
          unless token = peek_token and token.type == "{"
            process_simple_substatement
          end
        when :ELSE
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          unless token = peek_token and token.type == :IF || token.type == "{"
            process_simple_substatement
          end
        when :DO
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          unless token = peek_token and token.type == "{"
            process_simple_substatement
          end
        else
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
        end
      end
    end

    def skip_controlling_part
      condition_paren_depth = 0
      while token = next_token
        case token.type
        when "("
          condition_paren_depth += 1
        when ")"
          condition_paren_depth -= 1
          break if condition_paren_depth == 0
        end
      end
    end

    def process_simple_substatement
      @indent_level += 1
      while token = next_token
        case token.type
        when "{"
          on_left_brace(token)
        when "}"
          on_right_brace(token)
        when "("
          on_left_paren(token)
        when ")"
          on_right_paren(token)
        end

        case token.type
        when :IF, :FOR, :WHILE
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          skip_controlling_part
          unless token = peek_token and token.type == "{"
            process_simple_substatement
            break
          end
        when :ELSE
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          unless token = peek_token and token.type == :IF || token.type == "{"
            process_simple_substatement
            break
          end
        when :DO
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          unless token = peek_token and token.type == "{"
            process_simple_substatement
            process_simple_substatement
            break
          end
        else
          if @last_line_no < token.location.line_no
            on_beginning_of_line(token)
          end
          break if token.type == ";"
        end
      end
      @indent_level -= 1
    end

    def next_token
      return nil unless token = peek_token
      @index += 1

      case token.type
      when :CASE
        while token = peek_token
          @index += 1
          break if token.type == ":"
        end
      when :IDENTIFIER, :DEFAULT
        if next_token = @tokens[@index] and next_token.type == ":"
          token = peek_token
          @index += 1
        end
      end

      token
    end

    def peek_token
      if token = @tokens[@index]
        @last_line_no = @tokens[[0, @index - 1].max].location.line_no
        checkpoint(token.location)
      end
      token
    end

    def on_left_brace(token)
      if indent_style == INDENT_STYLE_GNU && @indent_level > 0
        @indent_level += 2
      else
        @indent_level += 1
      end
    end

    def on_right_brace(token)
      if indent_style == INDENT_STYLE_GNU
        @indent_level -= 2
        @indent_level = 0 if @indent_level < 0
      else
        @indent_level -= 1
      end
    end

    def on_left_paren(token)
      @paren_depth += 1
    end

    def on_right_paren(token)
      @paren_depth -= 1
    end

    def on_beginning_of_line(token)
      return if @paren_depth > 0

      case token.type
      when "{"
        if @indent_level == 0
          expected_column_no = expected_indent_width(token)
        else
          expected_column_no = expected_indent_width(token, -1)
        end
      when "}"
        if indent_style == INDENT_STYLE_GNU && @indent_level > 0
          expected_column_no = expected_indent_width(token, +1)
        else
          expected_column_no = expected_indent_width(token)
        end
      else
        expected_column_no = expected_indent_width(token)
      end

      unless token.location.appearance_column_no == expected_column_no
        W(:W0432, token.location)
      end
    end

    def expected_indent_width(token, delta_level = 0)
      if @indent_width == 0 && @indent_level > 0
        @indent_width =
          (token.location.appearance_column_no - 1) / @indent_level
      end

      if @indent_width > 0
        @indent_width * @indent_level + @indent_width * delta_level + 1
      else
        token.location.appearance_column_no
      end
    end
  end

  class W0440 < PassiveMessageDetection
    include CodingStyleAccessor

    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_compound_statement += method(:check)
    end

    private
    def check(node)
      return if indent_style == INDENT_STYLE_K_AND_R

      unless node.head_location.appearance_column_no ==
          node.tail_location.appearance_column_no
        W(:W0440, node.tail_location)
      end
    end
  end

  class W0441 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_constant_referred += method(:check)
    end

    private
    def check(constant_specifier, variable)
      return unless variable.type.scalar? && variable.type.integer?
      return if constant_specifier.character?

      if constant_specifier.suffix.nil? && variable.type != @interp.int_type
        W(:W0441, constant_specifier.location)
      end
    end
  end

  class W0446 < MessageDetection
    private
    def do_prepare(context) end

    def do_execute(context)
      context[:c_syntax_tree].accept(Visitor.new(context))
    end

    class Visitor < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context)
        @context = context
      end

      def visit_simple_assignment_expression(node)
        super
        warn(node.lhs_operand)
        warn(node.rhs_operand)
      end

      def visit_compound_assignment_expression(node)
        super
        warn(node.lhs_operand)
        warn(node.rhs_operand)
      end

      def visit_function_call_expression(node)
        super
        node.argument_expressions.each { |expr| warn(expr) }
      end

      def visit_unary_arithmetic_expression(node)
        super
        if node.operator.type == "+" || node.operator.type == "-"
          warn(node.operand)
        end
      end

      def visit_multiplicative_expression(node)
        super
        warn(node.lhs_operand)
        warn(node.rhs_operand)
      end

      def visit_additive_expression(node)
        super
        warn(node.lhs_operand)
        warn(node.rhs_operand)
      end

      def visit_return_statement(node)
        super
        warn(node.expression) if node.expression
      end

      private
      def warn(node)
        node = node.expression while node.kind_of?(GroupedExpression)

        case node
        when SimpleAssignmentExpression, CompoundAssignmentExpression
          W(:W0446, node.location)
        end
      end

      def report
        @context.report
      end
    end
    private_constant :Visitor
  end

  class W0447 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_comma_separated_expression += method(:check)
      visitor.enter_for_statement += method(:enter_for_statement)
      visitor.leave_for_statement += method(:leave_for_statement)
      visitor.enter_c99_for_statement += method(:enter_for_statement)
      visitor.leave_c99_for_statement += method(:leave_for_statement)
      @in_for_statement = false
    end

    private
    def check(node)
      W(:W0447, node.location) unless @in_for_statement
    end

    def enter_for_statement(node)
      @in_for_statement = true
    end

    def leave_for_statement(node)
      @in_for_statement = false
    end
  end

  class W0456 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check_variable)
      interp.on_function_defined += method(:check_function)
      @target_fpath = context[:sources].first.fpath
    end

    private
    def check_variable(variable_definition, variable)
      if variable.declared_as_extern?
        unless variable_definition.location.fpath == @target_fpath
          W(:W0456, variable_definition.location, variable.name)
        end
      end
    end

    def check_function(function_definition, function)
      if function.declared_as_extern?
        unless function_definition.location.fpath == @target_fpath
          W(:W0456, function_definition.location, function.name)
        end
      end
    end
  end

  class W0457 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_function_declaration += method(:check_function)
      visitor.enter_ansi_function_definition += method(:check_function)
      visitor.enter_kandr_function_definition += method(:check_function)
    end

    private
    def check_function(node)
      W(:W0457, node.location) if node.implicitly_typed?
    end
  end

  class W0458 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_declaration += method(:check_variable)
      visitor.enter_variable_definition += method(:check_variable)
      visitor.enter_parameter_definition += method(:check_variable)
    end

    private
    def check_variable(node)
      W(:W0458, node.location) if node.implicitly_typed?
    end
  end

  class W0459 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_value_referred += method(:check)
    end

    private
    def check(expression, variable)
      return if variable.scope.global? || variable.binding.memory.static?

      if variable.named? && variable.value.must_be_undefined?
        W(:W0459, expression.location, variable.name)
      end
    end
  end

  class W0460 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_value_referred += method(:check)
    end

    private
    def check(expression, variable)
      return if variable.scope.global? || variable.binding.memory.static?
      return if variable.value.must_be_undefined?

      if variable.named? && variable.value.may_be_undefined?
        W(:W0460, expression.location, variable.name)
      end
    end
  end

  class W0461 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      args = arg_variables.zip(function.type.parameter_types)
      args.each_with_index do |(arg, type), index|
        next unless arg.variable? && arg.value.scalar?
        next unless type && type.pointer?

        base_type = type.unqualify.base_type
        next unless !base_type.function? && base_type.const?

        if pointee = @interp.pointee_of(arg) and pointee.variable?
          if !pointee.temporary? && pointee.value.must_be_undefined?
            W(:W0461,
              function_call_expression.argument_expressions[index].location)
          end
        end
      end
    end
  end

  class W0462 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      args = arg_variables.zip(function.type.parameter_types)
      args.each_with_index do |(arg, type), index|
        next unless arg.variable? && arg.value.scalar?
        next unless type && type.pointer?
        next unless type.unqualify.base_type.const?

        if pointee = @interp.pointee_of(arg) and pointee.variable?
          next if pointee.value.must_be_undefined?
          if !pointee.temporary? && pointee.value.may_be_undefined?
            W(:W0462,
              function_call_expression.argument_expressions[index].location)
          end
        end
      end
    end
  end

  class W0488 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @already_warned = false
        @group_depth = 0
        @ungrouped_array_subscript_expr = 0
        @ungrouped_function_call_expr = 0
        @ungrouped_member_access_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_array_subscript_expression(node)
        node.expression.accept(self)
        @ungrouped_array_subscript_expr += 1 if @group_depth == 0
        AmbiguousExpressionDetector.new(@context, node.array_subscript).execute
      end

      def visit_function_call_expression(node)
        node.expression.accept(self)
        @ungrouped_function_call_expr += 1 if @group_depth == 0
        node.argument_expressions.each do |expr|
          AmbiguousExpressionDetector.new(@context, expr).execute
        end
      end

      def visit_member_access_by_value_expression(node)
        super
        @ungrouped_member_access_expr += 1 if @group_depth == 0
      end

      def visit_member_access_by_pointer_expression(node)
        super
        @ungrouped_member_access_expr += 1 if @group_depth == 0
      end

      def visit_bit_access_by_value_expression(node)
        super
        @ungrouped_member_access_expr += 1 if @group_depth == 0
      end

      def visit_bit_access_by_pointer_expression(node)
        super
        @ungrouped_member_access_expr += 1 if @group_depth == 0
      end

      def visit_logical_and_expression(node)
        super
        unless @already_warned
          if include_ambiguous_expr?
            W(:W0488, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_logical_or_expression(node)
        super
        unless @already_warned
          if include_ambiguous_expr?
            W(:W0488, @expression.head_location)
            @already_warned = true
          end
        end
      end

      private
      def include_ambiguous_expr?
        @ungrouped_array_subscript_expr > 0 ||
          @ungrouped_function_call_expr > 0 ||
          @ungrouped_member_access_expr > 0
      end

      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0489 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @already_warned = false
        @group_depth = 0
        @ungrouped_unary_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_postfix_increment_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_postfix_decrement_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_prefix_increment_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_prefix_decrement_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_address_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_indirection_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_unary_arithmetic_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_sizeof_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_alignof_expression(node)
        super
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_cast_expression(node)
        node.operand.accept(self)
        @ungrouped_unary_expr += 1 if @group_depth == 0
      end

      def visit_logical_and_expression(node)
        super
        unless @already_warned
          if include_ambiguous_expr?
            W(:W0489, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_logical_or_expression(node)
        super
        unless @already_warned
          if include_ambiguous_expr?
            W(:W0489, @expression.head_location)
            @already_warned = true
          end
        end
      end

      private
      def include_ambiguous_expr?
        @ungrouped_unary_expr > 0
      end

      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0490 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @already_warned = false
        @group_depth = 0
        @ungrouped_highprec_binary_expr = 0
        @ungrouped_logical_and_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_array_subscript_expression(node)
        node.expression.accept(self)
        AmbiguousExpressionDetector.new(@context, node.array_subscript).execute
      end

      def visit_function_call_expression(node)
        node.expression.accept(self)
        node.argument_expressions.each do |expr|
          AmbiguousExpressionDetector.new(@context, expr).execute
        end
      end

      def visit_multiplicative_expression(node)
        super
        @ungrouped_highprec_binary_expr += 1 if @group_depth == 0
      end

      def visit_additive_expression(node)
        super
        @ungrouped_highprec_binary_expr += 1 if @group_depth == 0
      end

      def visit_shift_expression(node)
        super
        @ungrouped_highprec_binary_expr += 1 if @group_depth == 0
      end

      def visit_relational_expression(node)
        super
        @ungrouped_highprec_binary_expr += 1 if @group_depth == 0
      end

      def visit_equality_expression(node)
        super
        @ungrouped_highprec_binary_expr += 1 if @group_depth == 0
      end

      def visit_and_expression(node)
        super
        @ungrouped_highprec_binary_expr += 1 if @group_depth == 0
      end

      def visit_exclusive_or_expression(node)
        super
        @ungrouped_highprec_binary_expr += 1 if @group_depth == 0
      end

      def visit_inclusive_or_expression(node)
        super
        @ungrouped_highprec_binary_expr += 1 if @group_depth == 0
      end

      def visit_logical_and_expression(node)
        super
        @ungrouped_logical_and_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_highprec_binary_expr > 0
            W(:W0490, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_logical_or_expression(node)
        super
        unless @already_warned
          if @ungrouped_highprec_binary_expr + @ungrouped_logical_and_expr > 0
            W(:W0490, @expression.head_location)
            @already_warned = true
          end
        end
      end

      private
      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0491 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:define_variable)
      interp.on_variable_declared += method(:declare_variable)
      interp.on_struct_declared += method(:declare_struct)
      interp.on_union_declared += method(:declare_union)
      interp.on_enum_declared += method(:declare_enum)
      interp.on_typedef_declared += method(:declare_typedef)
      interp.on_parameter_defined += method(:define_parameter)
      interp.on_label_defined += method(:define_label)
      interp.on_block_started += method(:enter_scope)
      interp.on_block_ended += method(:leave_scope)
      @variable_names = [[]]
      @tag_names = [[]]
      @label_names = [[]]
    end

    private
    def define_variable(variable_definition, variable)
      variable_id = variable_definition.identifier

      (@tag_names + @label_names).each do |identifier_array|
        if identifier_array.any? { |id| id.value == variable_id.value }
          W(:W0491, variable_definition.location, variable_id.value)
          return
        end
      end

      @variable_names.last.push(variable_id)
    end

    def declare_variable(variable_declaration, variable)
      variable_id = variable_declaration.identifier

      (@tag_names + @label_names).each do |identifier_array|
        if identifier_array.any? { |id| id.value == variable_id.value }
          W(:W0491, variable_declaration.location, variable_id.value)
          return
        end
      end

      @variable_names.last.push(variable_id)
    end

    def declare_struct(struct_type_declaration)
      tag_id = struct_type_declaration.identifier
      return unless tag_id

      (@variable_names + @label_names).each do |identifier_array|
        if identifier_array.any? { |id| id.value == tag_id.value }
          W(:W0491, struct_type_declaration.location, tag_id.value)
          return
        end
      end

      @tag_names.last.push(tag_id)
    end

    def declare_union(union_type_declaration)
      tag_id = union_type_declaration.identifier
      return unless tag_id

      (@variable_names + @label_names).each do |identifier_array|
        if identifier_array.any? { |id| id.value == tag_id.value }
          W(:W0491, union_type_declaration.location, tag_id.value)
          return
        end
      end

      @tag_names.last.push(tag_id)
    end

    def declare_enum(enum_type_declaration)
      tag_id = enum_type_declaration.identifier
      return unless tag_id

      (@variable_names + @label_names).each do |identifier_array|
        if identifier_array.any? { |id| id.value == tag_id.value }
          W(:W0491, enum_type_declaration.location, tag_id.value)
          return
        end
      end

      @tag_names.last.push(tag_id)
    end

    def declare_typedef(typedef_declaration)
      typedef_id = typedef_declaration.identifier

      (@tag_names + @label_names).each do |identifier_array|
        if identifier_array.any? { |id| id.value == typedef_id.value }
          W(:W0491, typedef_declaration.location, typedef_id.value)
          return
        end
      end

      @variable_names.last.push(typedef_id)
    end

    def define_parameter(parameter_definition, variable)
      param_id = parameter_definition.identifier
      return unless param_id

      (@tag_names + @label_names).each do |identifier_array|
        if identifier_array.any? { |id| id.value == param_id.value }
          W(:W0491, parameter_definition.location, param_id.value)
          return
        end
      end

      @variable_names.last.push(param_id)
    end

    def define_label(generic_labeled_statement)
      label_id = generic_labeled_statement.label

      (@variable_names + @tag_names).each do |identifier_array|
        if identifier_array.any? { |id| id.value == label_id.value }
          W(:W0491, generic_labeled_statement.location, label_id.value)
          return
        end
      end

      @label_names.last.push(label_id)
    end

    def enter_scope(compound_statement)
      @variable_names.push([])
      @tag_names.push([])
      @label_names.push([])
    end

    def leave_scope(compound_statement)
      @variable_names.pop
      @tag_names.pop
      @label_names.pop
    end
  end

  class W0492 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:define_variable)
      interp.on_variable_declared += method(:declare_variable)
      interp.on_struct_declared += method(:declare_struct)
      interp.on_union_declared += method(:declare_union)
      interp.on_enum_declared += method(:declare_enum)
      interp.on_typedef_declared += method(:declare_typedef)
      interp.on_parameter_defined += method(:define_parameter)
      interp.on_label_defined += method(:define_label)
      interp.on_block_started += method(:enter_scope)
      interp.on_block_ended += method(:leave_scope)
      @variable_names = [[]]
      @tag_names = [[]]
      @label_names = [[]]
      @member_names = [[]]
    end

    private
    def define_variable(variable_definition, variable)
      variable_id = variable_definition.identifier

      @member_names.each do |identifier_array|
        if identifier_array.any? { |id| id.value == variable_id.value }
          W(:W0492, variable_definition.location, variable_id.value)
          return
        end
      end

      @variable_names.last.push(variable_id)
    end

    def declare_variable(variable_declaration, variable)
      variable_id = variable_declaration.identifier

      @member_names.each do |identifier_array|
        if identifier_array.any? { |id| id.value == variable_id.value }
          W(:W0492, variable_declaration.location, variable_id.value)
          return
        end
      end

      @variable_names.last.push(variable_id)
    end

    def declare_struct(struct_type_declaration)
      tag_id = struct_type_declaration.identifier
      return unless tag_id

      @member_names.each do |identifier_array|
        if identifier_array.any? { |id| id.value == tag_id.value }
          W(:W0492, struct_type_declaration.location, tag_id.value)
          break
        end
      end

      member_declarations = MemberExtractor.new.tap { |visitor|
        struct_type_declaration.accept(visitor)
      }.result

      (@variable_names + @tag_names + @label_names).each do |identifier_array|
        identifier_array.each do |identifier|
          member = member_declarations.find { |memb|
            memb.identifier.value == identifier.value
          }
          W(:W0492, member.location, member.identifier.value) if member
        end
      end

      @member_names.last.push(
        *member_declarations.map { |memb| memb.identifier })
    end

    def declare_union(union_type_declaration)
      tag_id = union_type_declaration.identifier
      return unless tag_id

      @member_names.each do |identifier_array|
        if identifier_array.any? { |id| id.value == tag_id.value }
          W(:W0492, union_type_declaration.location, tag_id.value)
          break
        end
      end

      member_declarations = MemberExtractor.new.tap { |visitor|
        union_type_declaration.accept(visitor)
      }.result

      (@variable_names + @tag_names + @label_names).each do |identifier_array|
        identifier_array.each do |identifier|
          member = member_declarations.find { |memb|
            memb.identifier.value == identifier.value
          }
          W(:W0492, member.location, member.identifier.value) if member
        end
      end

      @member_names.last.push(
        *member_declarations.map { |memb| memb.identifier })
    end

    def declare_enum(enum_type_declaration)
      tag_id = enum_type_declaration.identifier
      return unless tag_id

      @member_names.each do |identifier_array|
        if identifier_array.any? { |id| id.value == tag_id.value }
          W(:W0492, enum_type_declaration.location, tag_id.value)
          return
        end
      end

      @tag_names.last.push(tag_id)
    end

    def declare_typedef(typedef_declaration)
      typedef_id = typedef_declaration.identifier

      @member_names.each do |identifier_array|
        if identifier_array.any? { |id| id.value == typedef_id.value }
          W(:W0492, typedef_declaration.location, typedef_id.value)
          return
        end
      end

      @variable_names.last.push(typedef_id)
    end

    def define_parameter(parameter_definition, variable)
      param_id = parameter_definition.identifier
      return unless param_id

      @member_names.each do |identifier_array|
        if identifier_array.any? { |id| id.value == param_id.value }
          W(:W0492, parameter_definition.location, param_id.value)
          return
        end
      end

      @variable_names.last.push(param_id)
    end

    def define_label(generic_labeled_statement)
      label_id = generic_labeled_statement.label

      @member_names.each do |identifier_array|
        if identifier_array.any? { |id| id.value == label_id.value }
          W(:W0492, generic_labeled_statement.location, label_id.value)
          return
        end
      end

      @label_names.last.push(label_id)
    end

    def enter_scope(compound_statement)
      @variable_names.push([])
      @tag_names.push([])
      @label_names.push([])
      @member_names.push([])
    end

    def leave_scope(compound_statement)
      @variable_names.pop
      @tag_names.pop
      @label_names.pop
      @member_names.pop
    end

    class MemberExtractor < SyntaxTreeVisitor
      def initialize
        @result = []
      end

      attr_reader :result

      def visit_struct_type_declaration(node)
        if node.struct_declarations
          node.struct_declarations.each do |struct_declaration|
            struct_declaration.accept(self)
          end
        end
      end

      def visit_union_type_declaration(node)
        if node.struct_declarations
          node.struct_declarations.each do |struct_declaration|
            struct_declaration.accept(self)
          end
        end
      end

      def visit_struct_declaration(node)
        node.items.each { |item| item.accept(self) }
      end

      def visit_member_declaration(node)
        @result.push(node)
      end
    end
    private_constant :MemberExtractor
  end

  class W0493 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if arg_variables.any? { |arg| arg.type.composite? }
        W(:W0493, function_call_expression.location)
      end
    end
  end

  class W0495 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @group_depth = 0
        @ungrouped_add_expr = 0
        @ungrouped_sub_expr = 0
        @ungrouped_mul_expr = 0
        @ungrouped_div_expr = 0
        @ungrouped_mod_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_array_subscript_expression(node)
        AmbiguousExpressionDetector.new(@context, node.array_subscript).execute
      end

      def visit_function_call_expression(node)
        node.argument_expressions.each do |expr|
          AmbiguousExpressionDetector.new(@context, expr).execute
        end
      end

      def visit_additive_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "+"
            @ungrouped_add_expr += 1
          when "-"
            @ungrouped_sub_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0495, @expression.head_location)
        else
          super
        end
      end

      def visit_multiplicative_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "*"
            @ungrouped_mul_expr += 1
          when "/"
            @ungrouped_div_expr += 1
          when "%"
            @ungrouped_mod_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0495, @expression.head_location)
        else
          super
        end
      end

      private
      def include_ambiguous_expr?
        return false if @ungrouped_mod_expr == 0

        ungrouped_multiplicative_expr =
          @ungrouped_mul_expr + @ungrouped_div_expr + @ungrouped_mod_expr

        (@ungrouped_add_expr + @ungrouped_sub_expr) > 0 &&
          ungrouped_multiplicative_expr > 0 or
        ungrouped_multiplicative_expr > 1
      end

      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0496 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.expression.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.expression.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.expression.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.expression.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @already_warned = false
        @group_depth = 0
        @ungrouped_conditional_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_conditional_expression(node)
        super
        unless @already_warned
          if @ungrouped_conditional_expr > 0
            W(:W0496, @expression.head_location)
            @already_warned = true
          end
        end
        @ungrouped_conditional_expr += 1 if @group_depth == 0
      end

      private
      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0497 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @group_depth = 0
        @ungrouped_shr_expr = 0
        @ungrouped_shl_expr = 0
        @ungrouped_lt_expr = 0
        @ungrouped_gt_expr = 0
        @ungrouped_le_expr = 0
        @ungrouped_ge_expr = 0
        @ungrouped_eq_expr = 0
        @ungrouped_ne_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_array_subscript_expression(node)
        AmbiguousExpressionDetector.new(@context, node.array_subscript).execute
      end

      def visit_function_call_expression(node)
        node.argument_expressions.each do |expr|
          AmbiguousExpressionDetector.new(@context, expr).execute
        end
      end

      def visit_shift_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "<<"
            @ungrouped_shl_expr += 1
          when ">>"
            @ungrouped_shr_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0497, @expression.head_location)
        else
          super
        end
      end

      def visit_relational_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "<"
            @ungrouped_lt_expr += 1
          when ">"
            @ungrouped_gt_expr += 1
          when "<="
            @ungrouped_le_expr += 1
          when ">="
            @ungrouped_ge_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0497, @expression.head_location)
        else
          super
        end
      end

      def visit_equality_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "=="
            @ungrouped_eq_expr += 1
          when "!="
            @ungrouped_ne_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0497, @expression.head_location)
        else
          super
        end
      end

      private
      def include_ambiguous_expr?
        @ungrouped_shl_expr > 1 || @ungrouped_shr_expr > 1 ||
          @ungrouped_lt_expr > 1 || @ungrouped_gt_expr > 1 ||
          @ungrouped_le_expr > 1 || @ungrouped_ge_expr > 1 ||
          @ungrouped_eq_expr > 1 || @ungrouped_ne_expr > 1
      end

      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0498 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @group_depth = 0
        @ungrouped_add_expr = 0
        @ungrouped_sub_expr = 0
        @ungrouped_mul_expr = 0
        @ungrouped_div_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_array_subscript_expression(node)
        AmbiguousExpressionDetector.new(@context, node.array_subscript).execute
      end

      def visit_function_call_expression(node)
        node.argument_expressions.each do |expr|
          AmbiguousExpressionDetector.new(@context, expr).execute
        end
      end

      def visit_additive_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "+"
            @ungrouped_add_expr += 1
          when "-"
            @ungrouped_sub_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0498, @expression.head_location)
        else
          super
        end
      end

      def visit_multiplicative_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "*"
            @ungrouped_mul_expr += 1
          when "/"
            @ungrouped_div_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0498, @expression.head_location)
        else
          super
        end
      end

      private
      def include_ambiguous_expr?
        @ungrouped_add_expr > 0 && @ungrouped_sub_expr > 0 or
        @ungrouped_mul_expr > 0 && @ungrouped_div_expr > 0
      end

      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0499 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @group_depth = 0
        @ungrouped_shr_expr = 0
        @ungrouped_shl_expr = 0
        @ungrouped_lt_expr = 0
        @ungrouped_gt_expr = 0
        @ungrouped_le_expr = 0
        @ungrouped_ge_expr = 0
        @ungrouped_eq_expr = 0
        @ungrouped_ne_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_array_subscript_expression(node)
        AmbiguousExpressionDetector.new(@context, node.array_subscript).execute
      end

      def visit_function_call_expression(node)
        node.argument_expressions.each do |expr|
          AmbiguousExpressionDetector.new(@context, expr).execute
        end
      end

      def visit_shift_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "<<"
            @ungrouped_shl_expr += 1
          when ">>"
            @ungrouped_shr_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0499, @expression.head_location)
        else
          super
        end
      end

      def visit_relational_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "<"
            @ungrouped_lt_expr += 1
          when ">"
            @ungrouped_gt_expr += 1
          when "<="
            @ungrouped_le_expr += 1
          when ">="
            @ungrouped_ge_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0499, @expression.head_location)
        else
          super
        end
      end

      def visit_equality_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "=="
            @ungrouped_eq_expr += 1
          when "!="
            @ungrouped_ne_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0499, @expression.head_location)
        else
          super
        end
      end

      private
      def include_ambiguous_expr?
        @ungrouped_shl_expr > 0 && @ungrouped_shr_expr > 0 or
        @ungrouped_lt_expr > 0 &&
          (@ungrouped_gt_expr + @ungrouped_le_expr + @ungrouped_ge_expr) > 0 or
        @ungrouped_gt_expr > 0 &&
          (@ungrouped_lt_expr + @ungrouped_le_expr + @ungrouped_ge_expr) > 0 or
        @ungrouped_le_expr > 0 &&
          (@ungrouped_lt_expr + @ungrouped_gt_expr + @ungrouped_ge_expr) > 0 or
        @ungrouped_ge_expr > 0 &&
          (@ungrouped_lt_expr + @ungrouped_gt_expr + @ungrouped_le_expr) > 0 or
        @ungrouped_eq_expr > 0 && @ungrouped_ne_expr > 0
      end

      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0500 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @group_depth = 0
        @ungrouped_add_expr = 0
        @ungrouped_sub_expr = 0
        @ungrouped_mul_expr = 0
        @ungrouped_div_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_conditional_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_array_subscript_expression(node)
        AmbiguousExpressionDetector.new(@context, node.array_subscript).execute
      end

      def visit_function_call_expression(node)
        node.argument_expressions.each do |expr|
          AmbiguousExpressionDetector.new(@context, expr).execute
        end
      end

      def visit_multiplicative_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "*"
            @ungrouped_mul_expr += 1
          when "/"
            @ungrouped_div_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0500, @expression.head_location)
        else
          super
        end
      end

      def visit_additive_expression(node)
        if @group_depth == 0
          case node.operator.type
          when "+"
            @ungrouped_add_expr += 1
          when "-"
            @ungrouped_sub_expr += 1
          end
        end

        if include_ambiguous_expr?
          W(:W0500, @expression.head_location)
        else
          super
        end
      end

      private
      def include_ambiguous_expr?
        (@ungrouped_add_expr + @ungrouped_sub_expr) > 0 &&
          (@ungrouped_mul_expr + @ungrouped_div_expr) > 0
      end

      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0501 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.expression.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.expression.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.expression.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ConditionalExpressionExtractor.new.tap { |o|
        node.expression.accept(o)
      }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @already_warned = false
        @group_depth = 0
        @ungrouped_binary_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_multiplicative_expression(node)
        check_binary_expression
        super
      end

      def visit_additive_expression(node)
        check_binary_expression
        super
      end

      def visit_shift_expression(node)
        check_binary_expression
        super
      end

      def visit_relational_expression(node)
        check_binary_expression
        super
      end

      def visit_equality_expression(node)
        check_binary_expression
        super
      end

      def visit_and_expression(node)
        check_binary_expression
        super
      end

      def visit_exclusive_or_expression(node)
        check_binary_expression
        super
      end

      def visit_inclusive_or_expression(node)
        check_binary_expression
        super
      end

      def visit_logical_and_expression(node)
        check_binary_expression
        super
      end

      def visit_logical_or_expression(node)
        check_binary_expression
        super
      end

      def visit_simple_assignment_expression(node)
        check_binary_expression
        super
      end

      def visit_compound_assignment_expression(node)
        check_binary_expression
        super
      end

      private
      def check_binary_expression
        @ungrouped_binary_expr += 1 if @group_depth == 0
        unless @already_warned
          if include_ambiguous_expr?
            W(:W0501, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def include_ambiguous_expr?
        @ungrouped_binary_expr > 0
      end

      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0502 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_initializer += method(:check_initializer)
      visitor.enter_expression_statement += method(:check_expr_statement)
      visitor.enter_if_statement += method(:check_selection_statement)
      visitor.enter_if_else_statement += method(:check_selection_statement)
      visitor.enter_switch_statement += method(:check_selection_statement)
      visitor.enter_while_statement += method(:check_iteration_statement)
      visitor.enter_do_statement += method(:check_iteration_statement)
      visitor.enter_for_statement += method(:check_iteration_statement)
      visitor.enter_c99_for_statement += method(:check_iteration_statement)
      visitor.enter_return_statement += method(:check_return_statement)
    end

    private
    def check_initializer(node)
      extractor = ExpressionExtractor.new.tap { |o| node.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_expr_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_selection_statement(node)
      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_iteration_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    def check_return_statement(node)
      return unless node.expression

      extractor = ExpressionExtractor.new.tap { |o| node.expression.accept(o) }
      extractor.expressions.each do |expr|
        AmbiguousExpressionDetector.new(@context, expr).execute
      end
    end

    class AmbiguousExpressionDetector < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context, expression)
        @context = context
        @expression = expression
        @already_warned = false
        @group_depth = 0
        @ungrouped_multiplicative_expr = 0
        @ungrouped_additive_expr = 0
        @ungrouped_shift_expr = 0
        @ungrouped_relational_expr = 0
        @ungrouped_equality_expr = 0
        @ungrouped_and_expr = 0
        @ungrouped_xor_expr = 0
        @ungrouped_or_expr = 0
        @ungrouped_logical_and_expr = 0
        @ungrouped_logical_or_expr = 0
      end

      def execute
        @expression.accept(self)
      end

      def visit_grouped_expression(node)
        @group_depth += 1
        super
        @group_depth -= 1
      end

      def visit_multiplicative_expression(node)
        super
        @ungrouped_multiplicative_expr += 1 if @group_depth == 0
      end

      def visit_additive_expression(node)
        super
        @ungrouped_additive_expr += 1 if @group_depth == 0
      end

      def visit_shift_expression(node)
        super
        @ungrouped_shift_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_multiplicative_expr + @ungrouped_additive_expr > 0
            W(:W0502, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_relational_expression(node)
        super
        @ungrouped_relational_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_multiplicative_expr + @ungrouped_additive_expr +
              @ungrouped_shift_expr > 0
            W(:W0502, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_equality_expression(node)
        super
        @ungrouped_equality_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_multiplicative_expr + @ungrouped_additive_expr +
              @ungrouped_shift_expr + @ungrouped_relational_expr > 0
            W(:W0502, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_and_expression(node)
        super
        @ungrouped_and_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_multiplicative_expr + @ungrouped_additive_expr +
              @ungrouped_shift_expr + @ungrouped_relational_expr +
              @ungrouped_equality_expr > 0
            W(:W0502, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_exclusive_or_expression(node)
        super
        @ungrouped_xor_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_multiplicative_expr + @ungrouped_additive_expr +
              @ungrouped_shift_expr + @ungrouped_relational_expr +
              @ungrouped_equality_expr + @ungrouped_and_expr > 0
            W(:W0502, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_inclusive_or_expression(node)
        super
        @ungrouped_or_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_multiplicative_expr + @ungrouped_additive_expr +
              @ungrouped_shift_expr + @ungrouped_relational_expr +
              @ungrouped_equality_expr + @ungrouped_and_expr +
              @ungrouped_xor_expr > 0
            W(:W0502, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_logical_and_expression(node)
        super
        @ungrouped_logical_and_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_multiplicative_expr + @ungrouped_additive_expr +
              @ungrouped_shift_expr + @ungrouped_relational_expr +
              @ungrouped_equality_expr + @ungrouped_and_expr +
              @ungrouped_xor_expr + @ungrouped_or_expr > 0
            W(:W0502, @expression.head_location)
            @already_warned = true
          end
        end
      end

      def visit_logical_or_expression(node)
        super
        @ungrouped_logical_or_expr += 1 if @group_depth == 0

        unless @already_warned
          if @ungrouped_multiplicative_expr + @ungrouped_additive_expr +
              @ungrouped_shift_expr + @ungrouped_relational_expr +
              @ungrouped_equality_expr + @ungrouped_and_expr +
              @ungrouped_xor_expr + @ungrouped_or_expr +
              @ungrouped_logical_and_expr > 0
            W(:W0502, @expression.head_location)
            @already_warned = true
          end
        end
      end

      private
      def report
        @context.report
      end
    end
    private_constant :AmbiguousExpressionDetector
  end

  class W0508 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_logical_and_expression += method(:check)
      visitor.enter_logical_or_expression += method(:check)
    end

    private
    def check(node)
      W(:W0508, node.location) if node.rhs_operand.have_side_effect?
    end
  end

  class W0512 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_expression_statement += method(:enter_expression_statement)
      visitor.leave_expression_statement += method(:leave_expression_statement)
      visitor.enter_postfix_increment_expression += method(:check)
      visitor.enter_postfix_decrement_expression += method(:check)
      visitor.enter_prefix_increment_expression += method(:check)
      visitor.enter_prefix_decrement_expression += method(:check)
      @current_statement = nil
    end

    private
    def enter_expression_statement(node)
      @current_statement = node
    end

    def leave_expression_statement(node)
      @current_statement = nil
    end

    def check(node)
      return unless @current_statement

      unless @current_statement.expression == node
        W(:W0512, node.location)
      end
    end
  end

  class W0525 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_member_declaration += method(:check)
    end

    private
    def check(node)
      return unless node.type.scalar? && node.type.integer?
      if node.type.bitfield? && node.type.signed? && node.type.bit_size == 1
        W(:W0525, node.location)
      end
    end
  end

  class W0529 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_constant_specifier += method(:check)
    end

    private
    def check(node)
      if node.constant.value =~ /\A0[0-9]+[UL]*\z/i
        W(:W0529, node.location)
      end
    end
  end

  class W0530 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_constant_specifier += method(:check_constant)
      visitor.enter_string_literal_specifier += method(:check_string_literal)
    end

    private
    def check_constant(node)
      if node.constant.value =~ /\AL?'.*\\0[0-9]+.*'\z/i
        W(:W0530, node.location)
      end
    end

    def check_string_literal(node)
      if node.literal.value =~ /\AL?".*\\0[0-9]+.*"\z/i
        W(:W0530, node.location)
      end
    end
  end

  class W0532 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_switch_statement += method(:check_switch_statement)
    end

    private
    def check_switch_statement(node)
      return unless node.statement.kind_of?(CompoundStatement)

      node.statement.block_items.each_with_index do |item, index|
        effective_breaks = EffectiveBreakCollector.new.execute(item)
        unless effective_breaks.empty?
          if next_item = node.statement.block_items[index + 1]
            while next_item.kind_of?(GenericLabeledStatement)
              next_item = next_item.statement
            end

            case next_item
            when CaseLabeledStatement, DefaultLabeledStatement
              ;
            else
              effective_breaks.each do |effective_break|
                W(:W0532, effective_break.location)
              end
            end
          end
        end
      end
    end

    class EffectiveBreakCollector < SyntaxTreeVisitor
      def initialize
        @result = []
      end

      def execute(node)
        node.accept(self)
        @result
      end

      def visit_switch_statement(node) end

      def visit_while_statement(node) end

      def visit_do_statement(node) end

      def visit_for_statement(node) end

      def visit_c99_for_statement(node) end

      def visit_break_statement(node)
        @result.push(node)
      end
    end
    private_constant :EffectiveBreakCollector
  end

  class W0534 < PassiveMessageDetection
    include SyntaxNodeCollector

    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_for_statement += method(:check_for_statement)
      visitor.enter_c99_for_statement += method(:check_c99_for_statement)
    end

    private
    def check_for_statement(node)
      initialized = collect_object_specifiers(node.initial_statement)

      if ctrl_var_name = deduct_ctrl_variable_name(node, initialized)
        unless initialized.any? { |os| os.identifier.value == ctrl_var_name }
          W(:W0534, node.initial_statement.location, ctrl_var_name)
        end
      end
    end

    def check_c99_for_statement(node)
      initialized = collect_identifier_declarators(node.declaration)

      if ctrl_var_name = deduct_ctrl_variable_name(node, initialized)
        unless initialized.any? { |os| os.identifier.value == ctrl_var_name }
          W(:W0534, node.declaration.location, ctrl_var_name)
        end
      end
    end

    def deduct_ctrl_variable_name(node, init_vars)
      histogram = Hash.new(0)

      cond_vars = collect_object_specifiers(node.condition_statement)
      cond_vars.map { |os| os.identifier.value }.each do |name|
        histogram[name] += 100
      end

      init_vars.map { |os_or_id| os_or_id.identifier.value }.each do |name|
        histogram[name] += 10
      end

      expr_vars = collect_object_specifiers(node.expression)
      expr_vars.map { |os| os.identifier.value }.each do |name|
        histogram[name] += 1
      end

      if histogram.empty?
        nil
      else
        histogram.to_a.sort { |lhs, rhs| lhs.last <=> rhs.last }.last.first
      end
    end
  end

  class W0535 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_comma_separated_expression += method(:check)
      visitor.enter_for_statement += method(:enter_for_statement)
      visitor.leave_for_statement += method(:leave_for_statement)
      visitor.enter_c99_for_statement += method(:enter_for_statement)
      visitor.leave_c99_for_statement += method(:leave_for_statement)
      @in_for_statement = false
    end

    private
    def check(node)
      W(:W0535, node.location) if @in_for_statement
    end

    def enter_for_statement(node)
      @in_for_statement = true
    end

    def leave_for_statement(node)
      @in_for_statement = false
    end
  end

  class W0538 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_switch_statement += method(:check)
    end

    private
    def check(node)
      return unless node.statement.kind_of?(CompoundStatement)

      default_labeled_statement, index =
        find_default_labeled_statement(node.statement)

      if default_labeled_statement
        unless final_clause?(index, node.statement)
          W(:W0538, default_labeled_statement.location)
        end
      end
    end

    def find_default_labeled_statement(compound_statement)
      compound_statement.block_items.each_with_index do |block_item, index|
        while block_item.kind_of?(GenericLabeledStatement)
          block_item = block_item.statement
        end

        case block_item
        when DefaultLabeledStatement
          return block_item, index
        end
      end
      return nil, nil
    end

    def final_clause?(index, compound_statement)
      index += 1
      while block_item = compound_statement.block_items[index]
        while block_item.kind_of?(GenericLabeledStatement)
          block_item = block_item.statement
        end

        case block_item
        when CaseLabeledStatement
          return false
        else
          index += 1
        end
      end
      true
    end
  end

  class W0540 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:check)
    end

    private
    def check(node)
      if node.declarator.kind_of?(AbbreviatedFunctionDeclarator)
        W(:W0540, node.declarator.location)
      end
    end
  end

  class W0542 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_function_declaration += method(:check)
    end

    private
    def check(node)
      node.init_declarator.accept(Visitor.new(@context, node))
    end

    class Visitor < SyntaxTreeVisitor
      include ReportUtil
      include SyntaxNodeCollector

      def initialize(context, function_decl)
        @context = context
        @function_decl = function_decl
      end

      def visit_parameter_type_list(node)
        return unless node.parameters

        param_has_name = node.parameters.map { |param_decl|
          if param_decl.declarator
            collect_identifier_declarators(param_decl.declarator).count > 0
          else
            false
          end
        }

        unless param_has_name.all? || param_has_name.none?
          W(:W0542, @function_decl.location)
        end
      end

      private
      def report
        @context.report
      end
    end
    private_constant :Visitor
  end

  class W0543 < PassiveMessageDetection
    include SyntaxNodeCollector

    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:check)
      @interp = context[:c_interpreter]
    end

    private
    def check(node)
      function = @interp.function_named(node.identifier.value)
      return unless function

      param_lists = function.declarations_and_definitions.map { |decl_or_def|
        case decl_or_def
        when FunctionDeclaration
          extract_param_names(decl_or_def.init_declarator)
        when FunctionDefinition
          extract_param_names(decl_or_def.declarator)
        end
      }

      if param_lists.size > 1
        param_lists.first.zip(*param_lists[1..-1]) do |names|
          unless names.tap { |ary| ary.delete("") }.uniq.size == 1
            W(:W0543, node.location)
            break
          end
        end
      end
    end

    def extract_param_names(node)
      collect_identifier_declarators(node).map { |decl| decl.identifier.value }
    end
  end

  class W0544 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_variable_initialized += method(:check_initialization)
      @interp.on_assignment_expr_evaled += method(:check_assignment)
    end

    private
    def check_initialization(variable_definition, variable, init_variable)
      lhs_type = variable.type.unqualify
      rhs_type = init_variable.type.unqualify

      if lhs_type.pointer? && lhs_type.base_type.function? &&
          rhs_type.pointer? && rhs_type.base_type.function?
        check(variable_definition, lhs_type.base_type, rhs_type.base_type)
      end
    end

    def check_assignment(assignment_expression, lhs_variable, rhs_variable)
      lhs_type = lhs_variable.type.unqualify
      rhs_type = rhs_variable.type.unqualify

      if lhs_type.pointer? && lhs_type.base_type.function? &&
          rhs_type.pointer? && rhs_type.base_type.function?
        check(assignment_expression, lhs_type.base_type, rhs_type.base_type)
      end
    end

    def check(node, lhs_func_type, rhs_func_type)
      param_types =
        lhs_func_type.parameter_types.zip(rhs_func_type.parameter_types)
      if param_types.any? { |l, r| l && r && l.param_name != r.param_name }
        W(:W0544, node.location)
      end
    end
  end

  class W0546 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_switch_statement += method(:check)
    end

    private
    def check(node)
      if node.statement.kind_of?(CompoundStatement)
        body = node.statement
      else
        return
      end

      body.block_items.each do |block_item|
        block_item.accept(Visitor.new(@context))
      end
    end

    class Visitor < SyntaxTreeVisitor
      include ReportUtil

      def initialize(context)
        @context = context
        @block_level = 0
      end

      def visit_case_labeled_statement(node)
        super
        W(:W0546, node.location) if @block_level > 0
      end

      def visit_default_labeled_statement(node)
        super
        W(:W0546, node.location) if @block_level > 0
      end

      def visit_compound_statement(node)
        @block_level += 1
        super
        @block_level -= 1
      end

      def visit_if_statement(node)
        @block_level += 1
        node.statement.accept(self)
        @block_level -= 1
      end

      def visit_if_else_statement(node)
        @block_level += 1
        node.then_statement.accept(self)
        node.else_statement.accept(self)
        @block_level -= 1
      end

      def visit_switch_statement(node)
      end

      def visit_while_statement(node)
        @block_level += 1
        node.statement.accept(self)
        @block_level -= 1
      end

      def visit_do_statement(node)
        @block_level += 1
        node.statement.accept(self)
        @block_level -= 1
      end

      def visit_for_statement(node)
        @block_level += 1
        node.body_statement.accept(self)
        @block_level -= 1
      end

      def visit_c99_for_statement(node)
        @block_level += 1
        node.body_statement.accept(self)
        @block_level -= 1
      end

      private
      def report
        @context.report
      end
    end
    private_constant :Visitor
  end

  class W0551 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_union_type_declaration += method(:check)
    end

    private
    def check(node)
      W(:W0551, node.location)
    end
  end

  class W0552 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check)
    end

    private
    def check(variable_definition, variable)
      W(:W0552, variable_definition.location) if variable.type.union?
    end
  end

  class W0553 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      lhs_type = original_variable.type.unqualify
      rhs_type = result_variable.type.unqualify

      return unless lhs_type.pointer? && lhs_type.base_type.function?
      return unless rhs_type.pointer? && rhs_type.base_type.function?

      unless lhs_type.base_type.same_as?(rhs_type.base_type)
        W(:W0553, expression.location)
      end
    end
  end

  class W0556 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_started += method(:enter_function)
      @interp.on_function_ended += method(:leave_function)
      @interp.on_function_call_expr_evaled += method(:check)
      @functions = []
    end

    private
    def enter_function(function_definition, function)
      @functions.push(function)
    end

    def leave_function(function_definition, function)
      @functions.pop
    end

    def check(function_call_expression, function, arg_variables,
              result_variable)
      if current_function = @functions.last
        if function == current_function
          W(:W0556, function_call_expression.location)
        end
      end
    end
  end

  class W0559 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_unary_arithmetic_expression += method(:check)
    end

    private
    def check(node)
      if node.operator.type == "!"
        if Visitor.new.tap { |v| node.operand.accept(v) }.bitwise_expr > 0
          W(:W0559, node.location)
        end
      end
    end

    class Visitor < SyntaxTreeVisitor
      def initialize
        @bitwise_expr = 0
      end

      attr_reader :bitwise_expr

      def visit_unary_arithmetic_expression(node)
        super unless node.operator.type == "!"
      end

      def visit_and_expression(node)
        super
        @bitwise_expr += 1
      end

      def visit_inclusive_or_expression(node)
        super
        @bitwise_expr += 1
      end
    end
    private_constant :Visitor
  end

  class W0560 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_unary_arithmetic_expression += method(:check)
    end

    private
    def check(node)
      if node.operator.type == "~"
        if Visitor.new.tap { |v| node.operand.accept(v) }.logical_expr > 0
          W(:W0560, node.location)
        end
      end
    end

    class Visitor < SyntaxTreeVisitor
      def initialize
        @logical_expr = 0
      end

      attr_reader :logical_expr

      def visit_unary_arithmetic_expression(node)
        super unless node.operator.type == "~"
      end

      def visit_relational_expression(node)
        super
        @logical_expr += 1
      end

      def visit_equality_expression(node)
        super
        @logical_expr += 1
      end

      def visit_logical_and_expression(node)
        super
        @logical_expr += 1
      end

      def visit_logical_or_expression(node)
        super
        @logical_expr += 1
      end
    end
    private_constant :Visitor
  end

  class W0561 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_expression_statement += method(:check)
    end

    private
    def check(node)
      case expression = node.expression
      when IndirectionExpression
        case expression.operand
        when PostfixIncrementExpression
          W(:W0561, node.location)
        end
      end
    end
  end

  class W0562 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_definition += method(:check)
    end

    private
    def check(node)
      if initializer = node.initializer
        if initializer_depth(initializer) > type_depth(node.type)
          W(:W0562, initializer.location)
        end
      end
    end

    def initializer_depth(initializer)
      if initializers = initializer.initializers
        1 + initializers.map { |ini| initializer_depth(ini) }.max
      else
        0
      end
    end

    def type_depth(type)
      case
      when type.array?
        1 + type_depth(type.base_type)
      when type.composite?
        type.members.empty? ?
          1 : 1 + type.members.map { |memb| type_depth(memb.type) }.max
      else
        0
      end
    end
  end

  class W0563 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_compound_statement += method(:enter_block)
      visitor.leave_compound_statement += method(:leave_block)
      visitor.enter_generic_labeled_statement += method(:check)
      @blocks = []
    end

    private
    def check(node)
      return if node.referrers.empty?

      if @blocks.size > 1
        current_block = @blocks.last
        anterior_goto = node.referrers.find { |goto|
          goto.location.line_no < current_block.head_location.line_no
        }
        return unless anterior_goto

        # FIXME: Must consider that the declaration may appear at anywhere in
        #        ISO C99.
        if current_block.block_items.any? { |item| item.kind_of?(Declaration) }
          W(:W0563, node.location, node.label.value)
        end
      end
    end

    def enter_block(node)
      @blocks.push(node)
    end

    def leave_block(node)
      @blocks.pop
    end
  end

  class W0564 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_compound_statement += method(:enter_block)
      visitor.leave_compound_statement += method(:leave_block)
      visitor.enter_generic_labeled_statement += method(:check)
      @blocks = []
    end

    private
    def check(node)
      return if node.referrers.empty?

      if @blocks.size > 1
        current_block = @blocks.last
        posterior_goto = node.referrers.find { |goto|
          goto.location.line_no > current_block.tail_location.line_no
        }
        return unless posterior_goto

        # FIXME: Must consider that the declaration may appear at anywhere in
        #        ISO C99.
        if current_block.block_items.any? { |item| item.kind_of?(Declaration) }
          W(:W0564, posterior_goto.location, node.label.value)
        end
      end
    end

    def enter_block(node)
      @blocks.push(node)
    end

    def leave_block(node)
      @blocks.pop
    end
  end

  class W0565 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      lhs_type = original_variable.type.unqualify
      rhs_type = result_variable.type.unqualify

      if lhs_type.integer? && !lhs_type.pointer? &&
          rhs_type.pointer? && rhs_type.base_type.volatile?
        W(:W0565, expression.location)
        return
      end

      if rhs_type.integer? && !rhs_type.pointer? &&
          lhs_type.pointer? && lhs_type.base_type.volatile?
        W(:W0565, expression.location)
        return
      end
    end
  end

  class W0566 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      lhs_type = original_variable.type.unqualify
      rhs_type = result_variable.type.unqualify

      case
      when lhs_type.integer? && !lhs_type.pointer? &&
           rhs_type.pointer? && rhs_type.base_type.function?
        W(:W0566, expression.location)
      when rhs_type.integer? && !rhs_type.pointer? &&
           lhs_type.pointer? && lhs_type.base_type.function?
        W(:W0566, expression.location)
      end
    end
  end

  class W0567 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      lhs_type = original_variable.type.unqualify
      rhs_type = result_variable.type.unqualify

      if lhs_type.integer? && !lhs_type.pointer? &&
          rhs_type.pointer? && !rhs_type.base_type.volatile?
        W(:W0567, expression.location)
        return
      end

      if rhs_type.integer? && !rhs_type.pointer? &&
          lhs_type.pointer? && !lhs_type.base_type.volatile?
        W(:W0567, expression.location)
        return
      end
    end
  end

  class W0568 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_shift_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(shift_expression, lhs_variable, rhs_variable, result)
      operator = shift_expression.operator.type
      return unless operator == "<<" || operator == "<<="

      return unless shift_expression.lhs_operand.constant?(@enum_tbl)
      return unless lhs_variable.type.signed?

      if lhs_variable.value.must_be_less_than?(ScalarValue.of(0)) or
          lhs_variable.value.must_be_greater_than?(ScalarValue.of(0)) &&
          must_overflow?(lhs_variable, rhs_variable)
        W(:W0568, shift_expression.location)
      end
    end

    def must_overflow?(lhs_variable, rhs_variable)
      result = lhs_variable.value << rhs_variable.value
      result.must_be_greater_than?(ScalarValue.of(lhs_variable.type.max_value))
    end
  end

  class W0569 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_shift_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(shift_expression, lhs_variable, rhs_variable, result)
      operator = shift_expression.operator.type
      return unless operator == "<<" || operator == "<<="

      return if shift_expression.lhs_operand.constant?(@enum_tbl)
      return unless lhs_variable.type.signed?

      if lhs_variable.value.must_be_less_than?(ScalarValue.of(0)) or
          lhs_variable.value.must_be_greater_than?(ScalarValue.of(0)) &&
          must_overflow?(lhs_variable, rhs_variable)
        W(:W0569, shift_expression.location)
      end
    end

    def must_overflow?(lhs_variable, rhs_variable)
      result = lhs_variable.value << rhs_variable.value
      result.must_be_greater_than?(ScalarValue.of(lhs_variable.type.max_value))
    end
  end

  class W0570 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_shift_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(shift_expression, lhs_variable, rhs_variable, result)
      operator = shift_expression.operator.type
      return unless operator == "<<" || operator == "<<="

      return if shift_expression.lhs_operand.constant?(@enum_tbl)
      return unless lhs_variable.type.signed?

      if !lhs_variable.value.must_be_less_than?(ScalarValue.of(0)) &&
          lhs_variable.value.may_be_less_than?(ScalarValue.of(0)) or
         !must_overflow?(lhs_variable, rhs_variable) &&
          may_overflow?(lhs_variable, rhs_variable)
        W(:W0570, shift_expression.location)
      end
    end

    def must_overflow?(lhs_variable, rhs_variable)
      result = lhs_variable.value << rhs_variable.value
      result.must_be_greater_than?(ScalarValue.of(lhs_variable.type.max_value))
    end

    def may_overflow?(lhs_variable, rhs_variable)
      result = lhs_variable.value << rhs_variable.value
      result.may_be_greater_than?(ScalarValue.of(lhs_variable.type.max_value))
    end
  end

  class W0571 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_shift_expr_evaled += method(:check)
    end

    private
    def check(shift_expression, lhs_variable, rhs_variable, result)
      operator = shift_expression.operator.type
      return unless operator == ">>" || operator == ">>="

      W(:W0571, shift_expression.location) if lhs_variable.type.signed?
    end
  end

  class W0572 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_shift_expr_evaled += method(:check_shift)
      interp.on_and_expr_evaled += method(:check_binary)
      interp.on_inclusive_or_expr_evaled += method(:check_binary)
      interp.on_exclusive_or_expr_evaled += method(:check_binary)
      interp.on_unary_arithmetic_expr_evaled += method(:check_unary)
    end

    private
    def check_shift(shift_expression, lhs_variable, rhs_variable, result)
      if lhs_variable.type.scalar? && lhs_variable.type.signed?
        W(:W0572, shift_expression.location)
      end
    end

    def check_binary(binary_expression, lhs_variable, rhs_variable, result)
      if lhs_variable.type.scalar? && lhs_variable.type.signed?
        W(:W0572, binary_expression.location)
        return
      end

      if rhs_variable.type.scalar? && rhs_variable.type.signed?
        W(:W0572, binary_expression.location)
        return
      end
    end

    def check_unary(unary_arithmetic_expression, variable, result)
      return unless unary_arithmetic_expression.operator.type == "~"
      if variable.type.scalar? && variable.type.signed?
        W(:W0572, unary_arithmetic_expression.location)
      end
    end
  end

  class W0578  < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_implicit_conv_performed += method(:check)
      interp.on_function_started += method(:clear_rvalues)
      interp.on_unary_arithmetic_expr_evaled += method(:handle_unary)
      interp.on_shift_expr_evaled += method(:handle_shift)
      interp.on_additive_expr_evaled += method(:handle_additive)
      interp.on_multiplicative_expr_evaled += method(:handle_multiplicative)
      @rvalues = nil
    end

    private
    def check(initializer_or_expression, original_variable, result_variable)
      return unless @rvalues
      return unless original_variable.type.integer?

      src_type = original_variable.type
      dst_type = result_variable.type

      if src_type.integer_conversion_rank < dst_type.integer_conversion_rank
        case @rvalues[original_variable]
        when UnaryArithmeticExpression, ShiftExpression,
             AdditiveExpression, MultiplicativeExpression
          W(:W0578, initializer_or_expression.location)
        end
      end
    end

    def clear_rvalues(function_definition, function)
      @rvalues = {}
    end

    def handle_unary(unary_arithmetic_expression,
                     operand_variable, result_variable)
      return unless unary_arithmetic_expression.operator == "~"
      memorize_rvalue_derivation(result_variable, unary_arithmetic_expression)
    end

    def handle_shift(shift_expression,
                     lhs_variable, rhs_variable, result_variable)
      return unless shift_expression.operator.type == "<<"
      memorize_rvalue_derivation(result_variable, shift_expression)
    end

    def handle_additive(additive_expression,
                        lhs_variable, rhs_variable, result_variable)
      memorize_rvalue_derivation(result_variable, additive_expression)
    end

    def handle_multiplicative(multiplicative_expression,
                              lhs_variable, rhs_variable, result_variable)
      return if multiplicative_expression.operator.type == "%"
      memorize_rvalue_derivation(result_variable, multiplicative_expression)
    end

    def memorize_rvalue_derivation(rvalue_holder, expression)
      @rvalues[rvalue_holder] = expression if @rvalues
    end
  end

  class W0579 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
      interp.on_function_started += method(:clear_rvalues)
      interp.on_unary_arithmetic_expr_evaled += method(:handle_unary)
      interp.on_shift_expr_evaled += method(:handle_shift)
      interp.on_additive_expr_evaled += method(:handle_additive)
      interp.on_multiplicative_expr_evaled += method(:handle_multiplicative)
      @rvalues = nil
    end

    private
    def check(cast_expression, original_variable, result_variable)
      return unless @rvalues
      return unless original_variable.type.integer?

      src_type = original_variable.type
      dst_type = result_variable.type

      if src_type.integer_conversion_rank < dst_type.integer_conversion_rank
        case @rvalues[original_variable]
        when UnaryArithmeticExpression, ShiftExpression,
             AdditiveExpression, MultiplicativeExpression
          W(:W0579, cast_expression.location)
        end
      end
    end

    def clear_rvalues(function_definition, function)
      @rvalues = {}
    end

    def handle_unary(unary_arithmetic_expression,
                     operand_variable, result_variable)
      return unless unary_arithmetic_expression.operator == "~"
      memorize_rvalue_derivation(result_variable, unary_arithmetic_expression)
    end

    def handle_shift(shift_expression,
                     lhs_variable, rhs_variable, result_variable)
      return unless shift_expression.operator.type == "<<"
      memorize_rvalue_derivation(result_variable, shift_expression)
    end

    def handle_additive(additive_expression,
                        lhs_variable, rhs_variable, result_variable)
      memorize_rvalue_derivation(result_variable, additive_expression)
    end

    def handle_multiplicative(multiplicative_expression,
                              lhs_variable, rhs_variable, result_variable)
      return if multiplicative_expression.operator.type == "%"
      memorize_rvalue_derivation(result_variable, multiplicative_expression)
    end

    def memorize_rvalue_derivation(rvalue_holder, expression)
      @rvalues[rvalue_holder] = expression if @rvalues
    end
  end

  class W0580 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_function_started += method(:start_function)
      @interp.on_function_ended += method(:end_function)
      @interp.on_parameter_defined += method(:add_parameter)
      @interp.on_indirection_expr_evaled += method(:relate_pointer)
      @interp.on_assignment_expr_evaled += method(:check)
      @parameters = nil
      @pointer_relationship = nil
    end

    private
    def start_function(function_definition, function)
      @parameters = Set.new
      @pointer_relationship = {}
    end

    def end_function(function_definition, function)
      @parameters = nil
      @pointer_relationship = nil
    end

    def add_parameter(parameter_definition, variable)
      return unless @parameters

      if variable.named?
        @parameters.add(variable.name)
      end
    end

    def relate_pointer(indirection_expression, variable, dereferenced)
      return unless @pointer_relationship

      @pointer_relationship[dereferenced] = variable
    end

    def check(assignment_expression, lhs_variable, rhs_variable)
      return unless @parameters && @pointer_relationship
      return unless lhs_variable.type.pointer? && rhs_variable.type.pointer?

      if rhs_pointee = @interp.pointee_of(rhs_variable) and
          rhs_pointee.variable? && rhs_pointee.named? &&
          rhs_pointee.scope.local? && rhs_pointee.binding.memory.static?
        if lhs_variable.scope.global?
          W(:W0580, assignment_expression.location)
        else
          if pointer = @pointer_relationship[lhs_variable]
            if pointer.named? && @parameters.include?(pointer.name)
              W(:W0580, assignment_expression.location)
            end
          end
        end
      end
    end
  end

  class W0581 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @function_calls = Hash.new { |hash, key| hash[key] = [] }
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      return unless function.named?
      return if prototype_declaration_of(function)

      arg_types = arg_variables.map { |var| var.type.unqualify }

      @function_calls[function.name].each do |prev_arg_types|
        if prev_arg_types.size == arg_types.size
          conformed = prev_arg_types.zip(arg_types).all? { |prev, last|
            case
            when prev.array? && last.array?,
                 prev.array? && last.pointer?,
                 prev.pointer? && last.array?
              prev.base_type == last.base_type
            else
              prev == last
            end
          }
        else
          conformed = false
        end

        unless conformed
          W(:W0581, function_call_expression.location)
          break
        end
      end

      @function_calls[function.name].push(arg_types)
    end

    def prototype_declaration_of(function)
      function.declarations_and_definitions.find do |decl_or_def|
        decl_or_def.kind_of?(FunctionDeclaration)
      end
    end
  end

  class W0582 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:call_function)
      interp.on_function_declared += method(:check)
      @function_calls = Hash.new { |hash, key| hash[key] = [] }
    end

    private
    def check(function_declaration, function)
      return unless function.named?
      return if function.type.have_va_list?

      param_types = function.type.parameter_types.reject { |type| type.void? }

      @function_calls[function.name].each do |expr, arg_types|
        if arg_types.size == param_types.size
          conformed = arg_types.zip(param_types).all? { |atype, ptype|
            atype.convertible?(ptype)
          }
        else
          conformed = false
        end

        W(:W0582, expr.location) unless conformed
      end
    end

    def call_function(expression, function, arg_variables, result_variable)
      if function.named?
        arg_types = arg_variables.map { |var| var.type }
        @function_calls[function.name].push([expression, arg_types])
      end
    end
  end

  class W0583 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:call_function)
      interp.on_function_defined += method(:check)
      @function_calls = Hash.new { |hash, key| hash[key] = [] }
    end

    private
    def check(function_definition, function)
      return unless function.named?
      return if function.type.have_va_list?

      param_types = function.type.parameter_types.reject { |type| type.void? }

      @function_calls[function.name].each do |expr, arg_types|
        if arg_types.size == param_types.size
          conformed = arg_types.zip(param_types).all? { |atype, ptype|
            atype.convertible?(ptype)
          }
        else
          conformed = false
        end

        W(:W0583, expr.location) unless conformed
      end
    end

    def call_function(expression, function, arg_variables, result_variable)
      if function.named?
        arg_types = arg_variables.map { |var| var.type }
        @function_calls[function.name].push([expression, arg_types])
      end
    end
  end

  class W0584 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(expression, function, arg_variables, result_variable)
      return unless function.named?
      return unless kandr_style_definition_of(function)
      return if function.type.have_va_list?

      arg_types = arg_variables.map { |var| var.type }
      param_types = function.type.parameter_types.reject { |type| type.void? }

      if arg_types.size == param_types.size
        arg_types.zip(param_types).each_with_index do |(atype, ptype), index|
          unless atype.convertible?(ptype)
            W(:W0584,
              expression.argument_expressions[index].location, index + 1)
          end
        end
      end
    end

    def kandr_style_definition_of(function)
      function.declarations_and_definitions.find do |decl_or_def|
        decl_or_def.kind_of?(KandRFunctionDefinition)
      end
    end
  end

  class W0585 < PassiveMessageDetection
    include SyntaxNodeCollector

    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_for_statement += method(:check_for_statement)
      visitor.enter_c99_for_statement += method(:check_c99_for_statement)
    end

    private
    def check_for_statement(node)
      return unless node.condition_statement.expression
      return unless node.expression

      initialized = collect_object_specifiers(node.initial_statement)
      if ctrl_var_name = deduct_ctrl_variable_name(node, initialized)
        varying_var_names = collect_varying_variable_names(node.expression)

        unless varying_var_names.any? { |name| ctrl_var_name == name }
          W(:W0585,  node.expression.location, ctrl_var_name)
        end
      end
    end

    def check_c99_for_statement(node)
      return unless node.condition_statement.expression
      return unless node.expression

      initialized = collect_identifier_declarators(node.declaration)
      if ctrl_var_name = deduct_ctrl_variable_name(node, initialized)
        varying_var_names = collect_varying_variable_names(node.expression)
        unless varying_var_names.any? { |name| ctrl_var_name == name }
          W(:W0585,  node.expression.location, ctrl_var_name)
        end
      end
    end

    def deduct_ctrl_variable_name(node, init_vars)
      histogram = Hash.new(0)

      cond_vars = collect_object_specifiers(node.condition_statement)
      cond_vars.map { |os| os.identifier.value }.each do |name|
        histogram[name] += 100
      end

      init_vars.map { |os_or_id| os_or_id.identifier.value }.each do |name|
        histogram[name] += 10
      end

      if histogram.empty?
        nil
      else
        histogram.to_a.sort { |lhs, rhs| lhs.last <=> rhs.last }.last.first
      end
    end

    def collect_varying_variable_names(node)
      varying_var_names = Set.new

      collect_simple_assignment_expressions(node).each do |expr|
        if expr.lhs_operand.kind_of?(ObjectSpecifier)
          varying_var_names.add(expr.lhs_operand.identifier.value)
        end
      end

      collect_compound_assignment_expressions(node).each do |expr|
        if expr.lhs_operand.kind_of?(ObjectSpecifier)
          varying_var_names.add(expr.lhs_operand.identifier.value)
        end
      end

      collect_prefix_increment_expressions(node).each do |expr|
        if expr.operand.kind_of?(ObjectSpecifier)
          varying_var_names.add(expr.operand.identifier.value)
        end
      end

      collect_prefix_decrement_expressions(node).each do |expr|
        if expr.operand.kind_of?(ObjectSpecifier)
          varying_var_names.add(expr.operand.identifier.value)
        end
      end

      collect_postfix_increment_expressions(node).each do |expr|
        if expr.operand.kind_of?(ObjectSpecifier)
          varying_var_names.add(expr.operand.identifier.value)
        end
      end

      collect_postfix_decrement_expressions(node).each do |expr|
        if expr.operand.kind_of?(ObjectSpecifier)
          varying_var_names.add(expr.operand.identifier.value)
        end
      end

      varying_var_names
    end
  end

  class W0597 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_sequence_point_reached += method(:commit_changes)
      interp.on_variable_value_updated += method(:update_variable)
      @update_count = Hash.new(0)
    end

    private
    def commit_changes(sequence_point)
      @update_count.each do |var, count|
        W(:W0597, sequence_point.last_node.location, var.name) if count > 1
      end
      @update_count = Hash.new(0)
    end

    def update_variable(expression, variable)
      @update_count[variable] += 1 if variable.named?
    end
  end

  class W0598 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_sequence_point_reached += method(:commit_changes)
      interp.on_variable_value_updated += method(:update_variable)
      @update_count = [Hash.new(0)]
    end

    private
    def commit_changes(sequence_point)
      if sequence_point.obvious?
        updated_vars = @update_count.map { |hash| hash.keys }.flatten.uniq
        updated_vars.each do |var|
          if @update_count.count { |hash| hash.include?(var) } > 1
            if @update_count.map { |hash| hash[var] }.max == 1
              W(:W0598, sequence_point.last_node.location, var.name)
            end
          end
        end
        @update_count = [Hash.new(0)]
      else
        @update_count.push(Hash.new(0))
      end
    end

    def update_variable(expression, variable)
      @update_count.last[variable] += 1 if variable.named?
    end
  end

  class W0599 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_sequence_point_reached += method(:commit_changes)
      interp.on_variable_value_referred += method(:refer_variable)
      interp.on_variable_value_updated += method(:update_variable)
      @refer_count = Hash.new(0)
      @update_count = Hash.new(0)
    end

    private
    def commit_changes(sequence_point)
      (@refer_count.keys & @update_count.keys).each do |var|
        if @refer_count[var] > 0 && @update_count[var] > 0
          W(:W0599, sequence_point.last_node.location, var.name)
        end
      end
      @refer_count = Hash.new(0)
      @update_count = Hash.new(0)
    end

    def refer_variable(expression, variable)
      @refer_count[variable] += 1 if variable.named?
    end

    def update_variable(expression, variable)
      if variable.named?
        case expression
        when SimpleAssignmentExpression, CompoundAssignmentExpression
          # NOTE: The expression-statement `i = i + j;' should not be warned.
          #       But the expression-statement `i = i++ + j;' should be warned.
          #       So, side-effects of the assignment-expression are given
          #       special treatment.
        else
          @update_count[variable] += 1
        end
      end
    end
  end

  class W0600 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_sequence_point_reached += method(:commit_changes)
      interp.on_variable_value_referred += method(:refer_variable)
      interp.on_variable_value_updated += method(:update_variable)
      @refer_count = [Hash.new(0)]
      @update_count = [Hash.new(0)]
    end

    private
    def commit_changes(sequence_point)
      if sequence_point.obvious?
        updated_vars = @update_count.map { |hash| hash.keys }.flatten.uniq
        access_count = @refer_count.zip(@update_count)

        updated_vars.each do |var|
          count = access_count.count { |rhash, uhash|
            rhash.include?(var) && !uhash.include?(var)
          }
          W(:W0600, sequence_point.last_node.location, var.name) if count > 0
        end
        @refer_count = [Hash.new(0)]
        @update_count = [Hash.new(0)]
      else
        @refer_count.push(Hash.new(0))
        @update_count.push(Hash.new(0))
      end
    end

    def refer_variable(expression, variable)
      @refer_count.last[variable] += 1 if variable.named?
    end

    def update_variable(expression, variable)
      if variable.named?
        case expression
        when SimpleAssignmentExpression, CompoundAssignmentExpression
          # NOTE: The expression-statement `i = i + j;' should not be warned.
          #       But the expression-statement `i = i++ + j;' should be warned.
          #       So, side-effects of the assignment-expression are given
          #       special treatment.
        else
          @update_count.last[variable] += 1
        end
      end
    end
  end

  class W0605 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_break_statement += method(:check)
      visitor.enter_switch_statement += method(:enter_breakable_statement)
      visitor.leave_switch_statement += method(:leave_breakable_statement)
      visitor.enter_while_statement += method(:enter_breakable_statement)
      visitor.leave_while_statement += method(:leave_breakable_statement)
      visitor.enter_do_statement += method(:enter_breakable_statement)
      visitor.leave_do_statement += method(:leave_breakable_statement)
      visitor.enter_for_statement += method(:enter_breakable_statement)
      visitor.leave_for_statement += method(:leave_breakable_statement)
      visitor.enter_c99_for_statement += method(:enter_breakable_statement)
      visitor.leave_c99_for_statement += method(:leave_breakable_statement)
      @breakable_statements = []
    end

    private
    def check(break_statement)
      @breakable_statements.last[1] += 1

      if @breakable_statements.last[1] > 1 &&
          @breakable_statements.last[0].kind_of?(IterationStatement)
        W(:W0605, break_statement.location)
      end
    end

    def enter_breakable_statement(statement)
      @breakable_statements.push([statement, 0])
    end

    def leave_breakable_statement(statement)
      @breakable_statements.pop
    end
  end

  class W0607 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      orig_type = original_variable.type
      conv_type = result_variable.type

      unless orig_type.scalar? && orig_type.integer? &&
          conv_type.scalar? && conv_type.integer? && conv_type.unsigned?
        return
      end

      orig_value = original_variable.value
      return unless orig_value.scalar?

      lower_test = orig_value < ScalarValue.of(0)

      if lower_test.must_be_true?
        W(:W0607, expression.location)
      end
    end
  end

  class W0608 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      orig_type = original_variable.type
      conv_type = result_variable.type

      unless orig_type.scalar? && orig_type.integer? &&
          conv_type.scalar? && conv_type.integer? && conv_type.unsigned?
        return
      end

      orig_value = original_variable.value
      return unless orig_value.scalar?

      lower_test = orig_value < ScalarValue.of(0)

      if !lower_test.must_be_true? && lower_test.may_be_true?
        W(:W0608, expression.location)
      end
    end
  end

  class W0609 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_relational_expr_evaled += method(:check)
      interp.on_equality_expr_evaled += method(:check)
      interp.on_logical_and_expr_evaled += method(:check)
      interp.on_logical_or_expr_evaled += method(:check)
    end

    private
    def check(binary_expression, lhs_variable, rhs_variable, result_variable)
      if result_variable.value.must_be_true?
        W(:W0609, binary_expression.location)
      end
    end
  end

  class W0610 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_relational_expr_evaled += method(:check)
      interp.on_equality_expr_evaled += method(:check)
      interp.on_logical_and_expr_evaled += method(:check)
      interp.on_logical_or_expr_evaled += method(:check)
    end

    private
    def check(binary_expression, lhs_variable, rhs_variable, result_variable)
      if result_variable.value.must_be_false?
        W(:W0610, binary_expression.location)
      end
    end
  end

  class W0611 < PassiveMessageDetection
    include SyntaxNodeCollector

    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_while_stmt_started += method(:enter_while_stmt)
      interp.on_do_stmt_started += method(:enter_do_stmt)
      interp.on_for_stmt_started += method(:enter_for_stmt)
      interp.on_c99_for_stmt_started += method(:enter_c99_for_stmt)
      interp.on_while_ctrlexpr_evaled += method(:memorize_ctrlexpr_value)
      interp.on_do_ctrlexpr_evaled += method(:memorize_ctrlexpr_value)
      interp.on_for_ctrlexpr_evaled += method(:memorize_ctrlexpr_value)
      interp.on_c99_for_ctrlexpr_evaled += method(:memorize_ctrlexpr_value)
      interp.on_variable_value_updated += method(:update_ctrl_var)
      interp.on_while_stmt_ended += method(:check)
      interp.on_do_stmt_ended += method(:check)
      interp.on_for_stmt_ended += method(:check)
      interp.on_c99_for_stmt_ended += method(:check)
      @enum_tbl = interp.environment.enumerator_table
      @iteration_stmts = []
    end

    private
    def enter_while_stmt(while_statement)
      enter_iteration_stmt(while_statement.expression)
    end

    def enter_do_stmt(do_statement)
      enter_iteration_stmt(do_statement.expression)
    end

    def enter_for_stmt(for_statement)
      enter_iteration_stmt(for_statement.condition_statement.expression)
    end

    def enter_c99_for_stmt(c99_for_statement)
      enter_iteration_stmt(c99_for_statement.condition_statement.expression)
    end

    IterationStmt = Struct.new(:ctrlexpr, :ctrlexpr_value, :ctrl_vars)

    def enter_iteration_stmt(ctrlexpr)
      if ctrlexpr
        @iteration_stmts.push(IterationStmt.new(ctrlexpr, nil,
                                                deduct_ctrl_vars(ctrlexpr)))
      else
        @iteration_stmts.push(IterationStmt.new(nil, nil, nil))
      end
    end

    def deduct_ctrl_vars(ctrlexpr)
      collect_object_specifiers(ctrlexpr).each_with_object({}) { |os, hash|
        hash[os.identifier.value] = false
      }
    end

    def memorize_ctrlexpr_value(iteration_statement, ctrlexpr_value)
      @iteration_stmts.last.ctrlexpr_value = ctrlexpr_value
    end

    def update_ctrl_var(expression, variable)
      if variable.named?
        @iteration_stmts.reverse_each do |iteration_stmt|
          next unless iteration_stmt.ctrlexpr
          if iteration_stmt.ctrl_vars.include?(variable.name)
            iteration_stmt.ctrl_vars[variable.name] = true
          end
        end
      end
    end

    def check(iteration_statement)
      if ctrlexpr = @iteration_stmts.last.ctrlexpr
        unless ctrlexpr.constant?(@enum_tbl)
          ctrlexpr_value = @iteration_stmts.last.ctrlexpr_value
          ctrl_vars = @iteration_stmts.last.ctrl_vars
          if ctrlexpr_value && ctrlexpr_value.must_be_true? and
              ctrl_vars && ctrl_vars.values.none?
            W(:W0611, ctrlexpr.location)
          end
        end
      end
      @iteration_stmts.pop
    end
  end

  class W0612 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_if_ctrlexpr_evaled += method(:check)
      interp.on_if_else_ctrlexpr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(selection_statement, ctrlexpr_value)
      if ctrlexpr = selection_statement.expression
        unless ctrlexpr.constant?(@enum_tbl)
          W(:W0612, ctrlexpr.location) if ctrlexpr_value.must_be_true?
        end
      end
    end
  end

  class W0613 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_if_ctrlexpr_evaled += method(:check_if_stmt)
      interp.on_if_else_ctrlexpr_evaled += method(:check_if_else_stmt)
      interp.on_while_ctrlexpr_evaled += method(:check_while_stmt)
      interp.on_for_ctrlexpr_evaled += method(:check_for_stmt)
      interp.on_c99_for_ctrlexpr_evaled += method(:check_c99_for_stmt)
    end

    private
    def check_if_stmt(if_statement, ctrlexpr_value)
      if ctrlexpr_value.must_be_false?
        W(:W0613, if_statement.expression.location)
      end
    end

    def check_if_else_stmt(if_else_statement, ctrlexpr_value)
      if ctrlexpr_value.must_be_false?
        W(:W0613, if_else_statement.expression.location)
      end
    end

    def check_while_stmt(while_statement, ctrlexpr_value)
      if ctrlexpr_value.must_be_false?
        W(:W0613, while_statement.expression.location)
      end
    end

    def check_for_stmt(for_statement, ctrlexpr_value)
      # NOTE: This method is called only if the for-statement has a controlling
      #       expression.
      if ctrlexpr_value.must_be_false?
        W(:W0613, for_statement.condition_statement.expression.location)
      end
    end

    def check_c99_for_stmt(c99_for_statement, ctrlexpr_value)
      # NOTE: This method is called only if the c99-for-statement has a
      #       controlling expression.
      if ctrlexpr_value.must_be_false?
        W(:W0613, c99_for_statement.condition_statement.expression.location)
      end
    end
  end

  class W0614 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_do_ctrlexpr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(do_statement, ctrlexpr_value)
      unless do_statement.expression.constant?(@enum_tbl)
        if ctrlexpr_value.must_be_false?
          W(:W0614, do_statement.expression.location)
        end
      end
    end
  end

  class W0622 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_declared += method(:check)
      interp.on_block_started += method(:enter_block)
      interp.on_block_ended += method(:leave_block)
      @block_level = 0
    end

    private
    def check(function_declaration, function)
      if @block_level > 0 && function.declared_as_extern?
        W(:W0622, function_declaration.location)
      end
    end

    def enter_block(*)
      @block_level += 1
    end

    def leave_block(*)
      @block_level -= 1
    end
  end

  class W0623 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:check)
      interp.on_block_started += method(:enter_block)
      interp.on_block_ended += method(:leave_block)
      @block_level = 0
    end

    private
    def check(variable_declaration, variable)
      if @block_level > 0 && variable.declared_as_extern?
        W(:W0623, variable_declaration.location)
      end
    end

    def enter_block(*)
      @block_level += 1
    end

    def leave_block(*)
      @block_level -= 1
    end
  end

  class W0624 < PassiveMessageDetection
    include Cpp::SyntaxNodeCollector

    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_ansi_function_definition += method(:check)
      visitor.enter_kandr_function_definition += method(:check)
      @directives =
        collect_define_lines(context[:cpp_syntax_tree]) +
        collect_undef_lines(context[:cpp_syntax_tree])
    end

    private
    def check(function_definition)
      @directives.select { |node|
        in_block?(function_definition.function_body, node)
      }.each { |node| W(:W0624, node.location) }
    end

    def in_block?(outer_node, inner_node)
      outer_node.location.fpath == inner_node.location.fpath &&
        outer_node.head_token.location.line_no < inner_node.location.line_no &&
        outer_node.tail_token.location.line_no > inner_node.location.line_no
    end
  end

  class W0625 < PassiveMessageDetection
    # NOTE: W0625 may be duplicative when the same typedef is used twice or
    #       more.
    ensure_uniqueness_of :W0625

    include SyntaxNodeCollector

    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      interp = context[:c_interpreter]
      interp.on_typedef_declared += method(:declare_typedef)
      interp.on_variable_defined += method(:check)
      interp.on_variable_declared += method(:check)
      interp.on_function_declared += method(:check)
      interp.on_function_defined += method(:check)
      @typedef_types = {}
    end

    private
    def declare_typedef(typedef_declaration)
      typedef_name = typedef_declaration.identifier.value

      if @fpath == typedef_declaration.location.fpath
        @typedef_types[typedef_name] = typedef_declaration
      else
        @typedef_types.delete(typedef_name)
      end
    end

    def check(decl_or_def, object, *)
      return unless object.declared_as_extern?

      if declaration_specifiers  = decl_or_def.declaration_specifiers
        find_bad_typedef_decls(declaration_specifiers).each do |decl|
          W(:W0625, decl.location, decl.identifier.value)
          break
        end
      end
    end

    def find_bad_typedef_decls(node)
      collect_typedef_type_specifiers(node).map { |type_spec|
        @typedef_types[type_spec.identifier.value]
      }.compact
    end
  end

  class W0626 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_constant_specifier += method(:check_constant)
      visitor.enter_string_literal_specifier += method(:check_string_literal)
    end

    private
    def check_constant(node)
      W(:W0626, node.location) if node.prefix =~ /\AL\z/i
    end

    def check_string_literal(node)
      W(:W0626, node.location) if node.prefix =~ /\AL\z/i
    end
  end

  class W0627 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_compound_statement += method(:enter_block)
      visitor.leave_compound_statement += method(:leave_block)
      visitor.enter_expression_statement += method(:enter_expression_statement)
      visitor.enter_if_statement += method(:update_last_statement)
      visitor.enter_if_else_statement += method(:update_last_statement)
      visitor.enter_switch_statement += method(:update_last_statement)
      visitor.enter_while_statement += method(:update_last_statement)
      visitor.enter_do_statement += method(:update_last_statement)
      visitor.enter_for_statement += method(:enter_for_statement)
      visitor.enter_c99_for_statement += method(:enter_for_statement)
      visitor.enter_goto_statement += method(:update_last_statement)
      visitor.enter_continue_statement += method(:update_last_statement)
      visitor.enter_break_statement += method(:update_last_statement)
      visitor.enter_return_statement += method(:update_last_statement)
      @last_statements = []
      @excepted_statements = Set.new
    end

    private
    def enter_block(node)
      @last_statements.push(nil)
    end

    def leave_block(node)
      @last_statements.pop
    end

    def enter_expression_statement(node)
      return if @excepted_statements.include?(node)

      unless node.expression
        if last_statement = @last_statements.last
          tail_location = last_statement.tail_location
          head_location = node.head_location

          if tail_location.fpath == head_location.fpath &&
              tail_location.line_no == head_location.line_no
            W(:W0627, node.location)
          end
        end
      end
      update_last_statement(node)
    end

    def enter_for_statement(node)
      @excepted_statements.add(node.condition_statement)
      update_last_statement(node)
    end

    def update_last_statement(node)
      @last_statements[-1] = node
    end
  end

  class W0629 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_defined += method(:define_function)
      interp.on_function_referred += method(:refer_function)
      interp.on_translation_unit_ended += method(:check)
      @static_functions = {}
    end

    private
    def check(translation_unit)
      @static_functions.map { |name, (count, location)|
        count == 0 ? [name, location] : nil
      }.compact.each do |name, location|
        W(:W0629, location, name)
      end
    end

    def define_function(function_definition, function)
      if function.declared_as_static?
        @static_functions[function.name] ||= [0, function_definition.location]
        @static_functions[function.name][1] ||= function_definition.location
      end
    end

    def refer_function(expression, function)
      return unless function.named?

      if record = @static_functions[function.name]
        record[0] += 1
      else
        @static_functions[function.name] = [1, nil]
      end
    end
  end

  class W0635 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          if cs.wellformed?
            if cs.consume_arguments? && cs.conversion_argument
              W(:W0635, format.location, index + 1) unless cs.acceptable?
            end
          end
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return PrintfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0636 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          if cs.wellformed?
            if cs.consume_arguments? && cs.conversion_argument.nil?
              W(:W0636, format.location, index + 1)
            end
          end
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return PrintfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0637 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        unless format.extra_arguments.empty?
          arg_expr = find_extra_argument_expr(function_call_expression,
                                              format.extra_arguments)
          if arg_expr
            W(:W0637, arg_expr.location)
          else
            W(:W0637, function_call_expression.location)
          end
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return PrintfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end

    def find_extra_argument_expr(function_call_expression, extra_arguments)
      index = -extra_arguments.size
      function_call_expression.argument_expressions[index]
    end
  end

  class W0638 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        if function_call_expression.argument_expressions.empty?
          W(:W0638, function_call_expression.location)
        end
      end
    end
  end

  class W0639 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          if cs.wellformed?
            if cs.consume_arguments? && cs.conversion_argument
              W(:W0639, format.location, index + 1) unless cs.acceptable?
            end
          end
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return ScanfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0640 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each do |cs|
          if arg = cs.conversion_argument and !arg.type.pointer?
            index = arg_variables.index(arg)
            if index and
                arg_expr = function_call_expression.argument_expressions[index]
              W(:W0640, arg_expr.location)
            else
              W(:W0640, function_call_expression.location)
              break
            end
          end
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return ScanfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0642 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_address_expr_evaled += method(:check_address_expression)
      interp.on_implicit_conv_performed += method(:check_conversion)
    end

    private
    def check_address_expression(expression, object, pointer_variable)
      if object.declared_as_register?
        W(:W0642, expression.location)
      end
    end

    def check_conversion(expression, original_variable, result_variable)
      if original_variable.declared_as_register?
        W(:W0642, expression.location)
      end
    end
  end

  class W0653 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_initialized += method(:check)
    end

    private
    def check(variable_definition, variable, init_variable)
      if initializer = variable_definition.initializer
        init_depth = initializer_depth(initializer)
        case
        when init_depth == 0 && init_variable.type.array?,
             init_depth == 0 && init_variable.type.composite?
          return
        when init_depth == 0 && type_depth(variable.type) > 0
          W(:W0653, initializer.location)
        end
      end
    end

    def initializer_depth(initializer)
      if initializers = initializer.initializers
        1 + initializers.map { |ini| initializer_depth(ini) }.max
      else
        0
      end
    end

    def type_depth(type)
      case
      when type.array?
        1 + type_depth(type.base_type)
      when type.composite?
        type.members.empty? ?
          1 : 1 + type.members.map { |memb| type_depth(memb.type) }.max
      else
        0
      end
    end
  end

  class W0654 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_initialized += method(:check)
    end

    private
    def check(variable_definition, variable, init_variable)
      if variable.type.struct? || variable.type.union? and
          !variable.type.same_as?(init_variable.type)
        W(:W0654, variable_definition.location)
      end
    end
  end

  class W0655 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_sizeof_expr_evaled += method(:check)
    end

    private
    def check(expression, operand_variable, result_variable)
      operand_type = operand_variable.type

      if operand_type.scalar? && operand_type.integer? &&
          operand_type.bitfield?
        W(:W0655, expression.location)
      end
    end
  end

  class W0656 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          if cs.complete?
            if cs.undefined? || cs.illformed?
              W(:W0656, format.location, index + 1)
            end
          end
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return PrintfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0657 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          unless cs.valid_length_modifier?
            warn(format, cs.conversion_specifier_character, index)
          end
        end
      end
    end

    def warn(format, cs_char, index)
      if target_conversion_specifiers.include?(cs_char)
        W(message_id, format.location, index + 1)
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return PrintfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end

    def target_conversion_specifiers
      ["i", "d"]
    end

    def message_id
      self.class.name.sub(/\A.*::/, "").intern
    end
  end

  class W0658 < W0657
    private
    def target_conversion_specifiers
      ["o"]
    end
  end

  class W0659 < W0657
    private
    def target_conversion_specifiers
      ["u"]
    end
  end

  class W0660 < W0657
    private
    def target_conversion_specifiers
      ["x"]
    end
  end

  class W0661 < W0657
    private
    def target_conversion_specifiers
      ["X"]
    end
  end

  class W0662 < W0657
    private
    def target_conversion_specifiers
      ["f"]
    end
  end

  class W0663 < W0657
    private
    def target_conversion_specifiers
      ["e"]
    end
  end

  class W0664 < W0657
    private
    def target_conversion_specifiers
      ["E"]
    end
  end

  class W0665 < W0657
    private
    def target_conversion_specifiers
      ["g"]
    end
  end

  class W0666 < W0657
    private
    def target_conversion_specifiers
      ["G"]
    end
  end

  class W0667 < W0657
    private
    def target_conversion_specifiers
      ["c"]
    end
  end

  class W0668 < W0657
    private
    def target_conversion_specifiers
      ["%"]
    end
  end

  class W0669 < W0657
    private
    def target_conversion_specifiers
      ["s"]
    end
  end

  class W0670 < W0657
    private
    def target_conversion_specifiers
      ["n"]
    end
  end

  class W0671 < W0657
    private
    def target_conversion_specifiers
      ["p"]
    end
  end

  class W0672 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*printf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          W(:W0672, format.location, index + 1) if cs.incomplete?
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return PrintfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0673 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          if cs.complete?
            if cs.undefined? || cs.illformed?
              W(:W0673, format.location, index + 1)
            end
          end
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return ScanfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0674 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          unless cs.valid_length_modifier?
            warn(format, cs.conversion_specifier_character, index)
          end
        end
      end
    end

    def warn(format, cs_char, index)
      if target_conversion_specifiers.include?(cs_char)
        W(message_id, format.location, index + 1)
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return ScanfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end

    def target_conversion_specifiers
      ["d", "i", "n"]
    end

    def message_id
      self.class.name.sub(/\A.*::/, "").intern
    end
  end

  class W0675 < W0674
    private
    def target_conversion_specifiers
      ["o"]
    end
  end

  class W0676 < W0674
    private
    def target_conversion_specifiers
      ["u"]
    end
  end

  class W0677 < W0674
    private
    def target_conversion_specifiers
      ["x", "X"]
    end
  end

  class W0678 < W0674
    private
    def target_conversion_specifiers
      ["e", "E", "f", "g", "G"]
    end
  end

  class W0679 < W0674
    private
    def target_conversion_specifiers
      ["s"]
    end
  end

  class W0680 < W0674
    private
    def target_conversion_specifiers
      ["p"]
    end
  end

  class W0681 < W0674
    private
    def target_conversion_specifiers
      ["%"]
    end
  end

  class W0682 < W0674
    private
    def target_conversion_specifiers
      ["["]
    end
  end

  class W0683 < W0674
    private
    def target_conversion_specifiers
      ["c"]
    end
  end

  class W0684 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @environ = interp.environment
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if function.named? && function.name =~ /\A.*scanf\z/
        format = create_format(function_call_expression,
                               format_str_index_of(function_call_expression),
                               arg_variables, @environ)
        return unless format

        format.conversion_specifiers.each_with_index do |cs, index|
          W(:W0684, format.location, index + 1) if cs.incomplete?
        end
      end
    end

    def format_str_index_of(function_call_expression)
      function_call_expression.argument_expressions.index do |arg_expr|
        arg_expr.kind_of?(StringLiteralSpecifier)
      end
    end

    def create_format(function_call_expression,
                      format_str_index, arg_variables, environment)
      if format_str_index
        format_str =
          function_call_expression.argument_expressions[format_str_index]
        if format_str && format_str.literal.value =~ /\AL?"(.*)"\z/i
          location = format_str.location
          trailing_args = arg_variables[(format_str_index + 1)..-1] || []
          return ScanfFormat.new($1, location, trailing_args, environment)
        end
      end
      nil
    end
  end

  class W0703 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_struct_declared += method(:declare_struct)
      interp.on_union_declared += method(:declare_union)
      interp.on_enum_declared += method(:declare_enum)
      interp.on_block_started += method(:enter_scope)
      interp.on_block_ended += method(:leave_scope)
      @tag_names = [Set.new]
    end

    private
    def declare_struct(struct_type_declaration)
      tag_name = struct_type_declaration.identifier.value
      if @tag_names.any? { |names| names.include?(tag_name) }
        W(:W0703, struct_type_declaration.location, tag_name)
      end
      @tag_names.last.add(tag_name)
    end

    def declare_union(union_type_declaration)
      tag_name = union_type_declaration.identifier.value
      if @tag_names.any? { |names| names.include?(tag_name) }
        W(:W0703, union_type_declaration.location, tag_name)
      end
      @tag_names.last.add(tag_name)
    end

    def declare_enum(enum_type_declaration)
      tag_name = enum_type_declaration.identifier.value
      if @tag_names.any? { |names| names.include?(tag_name) }
        W(:W0703, enum_type_declaration.location, tag_name)
      end
      @tag_names.last.add(tag_name)
    end

    def enter_scope(compound_statement)
      @tag_names.push(Set.new)
    end

    def leave_scope(compound_statement)
      @tag_names.pop
    end
  end

  class W0704 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:define_variable)
      interp.on_function_declared += method(:declare_function)
      interp.on_variable_declared += method(:declare_variable)
      interp.on_enum_declared += method(:declare_enum)
      interp.on_typedef_declared += method(:declare_typedef)
      interp.on_function_defined += method(:define_function)
      interp.on_parameter_defined += method(:define_parameter)
      interp.on_block_started += method(:enter_scope)
      interp.on_block_ended += method(:leave_scope)
      @vdcls = [Set.new]
      @vdefs = [Set.new]
      @fdcls = [Set.new]
      @fdefs = [Set.new]
      @tdefs = [Set.new]
      @enums = [Set.new]
    end

    private
    def define_variable(variable_definition, variable)
      identifier = variable_definition.identifier.value
      if wider_identifiers_of_variable_definition.include?(identifier)
        W(:W0704, variable_definition.location, identifier)
      end
      @vdefs.last.add(identifier)
    end

    def wider_identifiers_of_variable_definition
      (@vdcls[0..-2] + @vdefs[0..-2] + @fdcls[0..-2] + @fdefs[0..-2] +
       @tdefs[0..-2] + @enums[0..-2]).reduce(Set.new) { |r, s| r + s }
    end

    def declare_function(function_declaration, function)
      identifier = function_declaration.identifier.value
      if wider_identifiers_of_function_declaration.include?(identifier)
        W(:W0704, function_declaration.location, identifier)
      end
      @fdcls.last.add(identifier)
    end

    def wider_identifiers_of_function_declaration
      (@vdcls[0..-2] + @vdefs[0..-2] + @fdcls[0..-2] + @fdefs[0..-2] +
       @tdefs[0..-2] + @enums[0..-2]).reduce(Set.new) { |r, s| r + s }
    end

    def declare_variable(variable_declaration, variable)
      identifier = variable_declaration.identifier.value
      if wider_identifiers_of_variable_declaration.include?(identifier)
        W(:W0704, variable_declaration.location, identifier)
      end
      @vdcls.last.add(identifier)
    end

    def wider_identifiers_of_variable_declaration
      (@vdefs[0..-2] + @fdcls[0..-2] + @fdefs[0..-2] + @tdefs[0..-2] +
       @enums[0..-2]).reduce(Set.new) { |r, s| r + s }
    end

    def declare_enum(enum_type_declaration)
      enum_type_declaration.enumerators.each do |enum|
        identifier = enum.identifier.value
        if wider_identifiers_of_enum_declaration.include?(identifier)
          W(:W0704, enum.location, identifier)
        end
        @enums.last.add(identifier)
      end
    end

    def wider_identifiers_of_enum_declaration
      (@vdcls[0..-2] + @vdefs[0..-2] + @fdcls[0..-2] + @fdefs[0..-2] +
       @tdefs[0..-2] + @enums[0..-2]).reduce(Set.new) { |r, s| r + s }
    end

    def declare_typedef(typedef_declaration)
      identifier = typedef_declaration.identifier.value
      if wider_identifiers_of_typedef_declaration.include?(identifier)
        W(:W0704, typedef_declaration.location, identifier)
      end
      @enums.last.add(identifier)
    end

    def wider_identifiers_of_typedef_declaration
      (@vdcls[0..-2] + @vdefs[0..-2] + @fdcls[0..-2] + @fdefs[0..-2] +
       @tdefs[0..-2] + @enums[0..-2]).reduce(Set.new) { |r, s| r + s }
    end

    def define_function(function_definition, function)
      identifier = function_definition.identifier.value
      if wider_identifiers_of_function_definition.include?(identifier)
        W(:W0704, function_definition.location, identifier)
      end
      @fdefs.last.add(identifier)
    end

    def wider_identifiers_of_function_definition
      (@vdcls[0..-2] + @vdefs[0..-2] + @fdefs[0..-2] + @tdefs[0..-2] +
       @enums[0..-2]).reduce(Set.new) { |r, s| r + s }
    end

    def define_parameter(parameter_definition, variable)
      identifier = parameter_definition.identifier.value
      if wider_identifiers_of_parameter_definition.include?(identifier)
        W(:W0704, parameter_definition.location, identifier)
      end
      @vdefs.last.add(identifier)
    end

    def wider_identifiers_of_parameter_definition
      (@vdcls[0..-2] + @vdefs[0..-2] + @fdcls[0..-2] + @fdefs[0..-2] +
       @tdefs[0..-2] + @enums[0..-2]).reduce(Set.new) { |r, s| r + s }
    end

    def enter_scope(compound_statement)
      @vdcls.push(Set.new)
      @vdefs.push(Set.new)
      @fdcls.push(Set.new)
      @fdefs.push(Set.new)
      @tdefs.push(Set.new)
      @enums.push(Set.new)
    end

    def leave_scope(compound_statement)
      @vdcls.pop
      @vdefs.pop
      @fdcls.pop
      @fdefs.pop
      @tdefs.pop
      @enums.pop
    end
  end

  class W0705 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_array_subscript_expr_evaled += method(:check)
    end

    private
    def check(array_subscript_expression,
              array_or_pointer_variable, subscript_variable,
              array_variable, result_variable)
      return unless array_variable && array_variable.type.length
      return unless subscript_variable.value.scalar?

      lower_bound = ScalarValue.of(0)
      upper_bound = ScalarValue.of(array_variable.type.length - 1)

      lower_test = subscript_variable.value < lower_bound
      upper_test = subscript_variable.value > upper_bound

      if !lower_test.must_be_true? && lower_test.may_be_true? or
          !upper_test.must_be_true? && upper_test.may_be_true?
        W(:W0705, array_subscript_expression.array_subscript.location)
      end
    end
  end

  class W0708 < PassiveMessageDetection
    include SyntaxNodeCollector

    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_for_stmt_started += method(:enter_for_statement)
      interp.on_for_stmt_ended += method(:leave_for_statement)
      interp.on_c99_for_stmt_started += method(:enter_c99_for_statement)
      interp.on_c99_for_stmt_ended += method(:leave_c99_for_statement)
      interp.on_variable_value_updated += method(:update_variable)
      @ctrl_variables = []
    end

    private
    def enter_for_statement(node)
      initialized = collect_object_specifiers(node.initial_statement)
      ctrl_var_name = deduct_ctrl_variable_name(node, initialized)
      @ctrl_variables.push([ctrl_var_name, node])
    end

    def leave_for_statement(node)
      @ctrl_variables.pop
    end

    def enter_c99_for_statement(node)
      initialized = collect_identifier_declarators(node.declaration)
      ctrl_var_name = deduct_ctrl_variable_name(node, initialized)
      @ctrl_variables.push([ctrl_var_name, node])
    end

    def leave_c99_for_statement(node)
      @ctrl_variables.pop
    end

    def update_variable(expression, variable)
      return unless variable.named?

      unless @ctrl_variables.empty?
        if @ctrl_variables.last.first == variable.name &&
            !in_ctrl_part(@ctrl_variables.last.last, expression)
          W(:W0708, expression.location, variable.name)
        end
      end
    end

    def deduct_ctrl_variable_name(node, init_vars)
      histogram = Hash.new(0)

      cond_vars = collect_object_specifiers(node.condition_statement)
      cond_vars.map { |os| os.identifier.value }.each do |name|
        histogram[name] += 100
      end

      init_vars.map { |os_or_id| os_or_id.identifier.value }.each do |name|
        histogram[name] += 10
      end

      expr_vars = collect_object_specifiers(node.expression)
      expr_vars.map { |os| os.identifier.value }.each do |name|
        histogram[name] += 1
      end

      if histogram.empty?
        nil
      else
        histogram.to_a.sort { |lhs, rhs| lhs.last <=> rhs.last }.last.first
      end
    end

    def in_ctrl_part(for_or_c99_for_statement, expr)
      case for_or_c99_for_statement
      when ForStatement
        contain_expr?(for_or_c99_for_statement.initial_statement, expr) ||
          contain_expr?(for_or_c99_for_statement.condition_statement, expr) ||
          contain_expr?(for_or_c99_for_statement.expression, expr)
      when C99ForStatement
        contain_expr?(for_or_c99_for_statement.condition_statement, expr) ||
          contain_expr?(for_or_c99_for_statement.expression, expr)
      end
    end

    def contain_expr?(node, expr)
      node ? node.accept(Visitor.new(expr)) : false
    end

    class Visitor < SyntaxTreeVisitor
      def initialize(expression)
        @expression = expression
      end

      def visit_function_call_expression(node)
        return true if node == @expression
        super
      end

      def visit_postfix_increment_expression(node)
        return true if node == @expression
        super
      end

      def visit_postfix_decrement_expression(node)
        return true if node == @expression
        super
      end

      def visit_prefix_increment_expression(node)
        return true if node == @expression
        super
      end

      def visit_prefix_decrement_expression(node)
        return true if node == @expression
        super
      end

      def visit_simple_assignment_expression(node)
        return true if node == @expression
        super
      end

      def visit_compound_assignment_expression(node)
        return true if node == @expression
        super
      end
    end
    private_constant :Visitor
  end

  class W0720 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      orig_type = original_variable.type
      conv_type = result_variable.type

      unless orig_type.scalar? && orig_type.floating? &&
          conv_type.scalar? && conv_type.integer?
        return
      end

      orig_value = original_variable.value
      return unless orig_value.scalar?

      lower_test = orig_value < ScalarValue.of(conv_type.min_value - 1)
      upper_test = orig_value > ScalarValue.of(conv_type.max_value + 1)

      if lower_test.must_be_true? || upper_test.must_be_true?
        W(:W0720, expression.location)
      end
    end
  end

  class W0721 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      orig_type = original_variable.type
      conv_type = result_variable.type

      unless orig_type.pointer? && conv_type.scalar? && conv_type.integer?
        return
      end

      if orig_type.min_value < conv_type.min_value ||
          orig_type.max_value > conv_type.max_value
        W(:W0721, expression.location)
      end
    end
  end

  class W0722 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check)
      interp.on_additive_expr_evaled += method(:check)
    end

    private
    def check(binary_expression, lhs_variable, rhs_variable, result_variable)
      return unless lhs_variable.type.scalar? && lhs_variable.type.signed?
      return unless rhs_variable.type.scalar? && rhs_variable.type.signed?

      return unless lhs_variable.value.scalar? && rhs_variable.value.scalar?

      case binary_expression.operator.type
      when "+"
        unbound_value = lhs_variable.value + rhs_variable.value
      when "-"
        unbound_value = lhs_variable.value - rhs_variable.value
      when "*"
        unbound_value = lhs_variable.value * rhs_variable.value
      else
        return
      end

      result_type = result_variable.type
      lower_test = unbound_value < ScalarValue.of(result_type.min_value)
      upper_test = unbound_value > ScalarValue.of(result_type.max_value)

      if lower_test.must_be_true? || upper_test.must_be_true?
        W(:W0722, binary_expression.location)
      end
    end
  end

  class W0723 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check)
      interp.on_additive_expr_evaled += method(:check)
    end

    private
    def check(binary_expression, lhs_variable, rhs_variable, result_variable)
      return unless lhs_variable.type.scalar? && lhs_variable.type.signed?
      return unless rhs_variable.type.scalar? && rhs_variable.type.signed?

      return unless lhs_variable.value.scalar? && rhs_variable.value.scalar?

      case binary_expression.operator.type
      when "+"
        unbound_value = lhs_variable.value + rhs_variable.value
      when "-"
        unbound_value = lhs_variable.value + rhs_variable.value
      when "*"
        unbound_value = lhs_variable.value * rhs_variable.value
      else
        return
      end

      result_type = result_variable.type
      lower_test = unbound_value < ScalarValue.of(result_type.min_value)
      upper_test = unbound_value > ScalarValue.of(result_type.max_value)

      if !lower_test.must_be_true? && lower_test.may_be_true? or
          !upper_test.must_be_true? && upper_test.may_be_true?
        W(:W0723, binary_expression.location)
      end
    end
  end

  class W0727 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
      interp.on_implicit_conv_performed += method(:check)
    end

    private
    def check(initializer_or_expression, original_variable, result_variable)
      return unless result_variable.type.enum?

      value = original_variable.value.unique_sample

      enumerators = result_variable.type.enumerators
      unless enumerators.any? { |enum| value == enum.value }
        W(:W0727, initializer_or_expression.location)
      end
    end
  end

  class W0728 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(expression, function, arg_variables, result_variable)
      args = arg_variables.zip(function.type.parameter_types)
      args.each_with_index do |(arg_variable, param_type), index|
        next unless param_type && param_type.enum?

        arg_expr = expression.argument_expressions[index]
        next unless arg_expr.constant?(@enum_tbl)

        if arg_variable.type.enum?
          unless arg_variable.type.same_as?(param_type)
            W(:W0728, arg_expr.location)
          end
        end
      end
    end
  end

  class W0729 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_assignment_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(expression, lhs_variable, rhs_variable)
      return unless lhs_variable.type.enum?
      return unless expression.rhs_operand.constant?(@enum_tbl)

      if rhs_variable.type.enum?
        unless lhs_variable.type.same_as?(rhs_variable.type)
          W(:W0729, expression.location)
        end
      end
    end
  end

  class W0730 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:start_function)
      interp.on_function_ended += method(:end_function)
      interp.on_return_stmt_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
      @current_function = nil
    end

    private
    def start_function(function_definition, function)
      @current_function = function
    end

    def end_function(function_definition, function)
      @current_function = nil
    end

    def check(return_statement, result_variable)
      return unless @current_function && result_variable

      return unless return_type = @current_function.type.return_type
      return unless return_type.enum?
      return unless return_statement.expression.constant?(@enum_tbl)

      if result_variable.type.enum?
        unless return_type.same_as?(result_variable.type)
          W(:W0730, return_statement.expression.location)
        end
      end
    end
  end

  class W0731 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_switch_stmt_ended += method(:end_switch_statement)
      interp.on_switch_ctrlexpr_evaled += method(:memorize_switch_ctrlexpr)
      interp.on_case_ctrlexpr_evaled += method(:check)
      @switch_ctrlexpr_stack = []
    end

    private
    def end_switch_statement(node)
      @switch_ctrlexpr_stack.pop
    end

    def memorize_switch_ctrlexpr(node, ctrlexpr_result)
      @switch_ctrlexpr_stack.push(ctrlexpr_result)
    end

    def check(case_labeled_statement, ctrlexpr_result)
      unless switch_ctrlexpr_result = @switch_ctrlexpr_stack.last
        return
      end

      return unless switch_ctrlexpr_result.type.enum?
      expected_type = switch_ctrlexpr_result.type

      value = ctrlexpr_result.value.unique_sample
      unless expected_type.enumerators.any? { |enum| value == enum.value }
        W(:W0731, case_labeled_statement.expression.location,
          case_labeled_statement.expression.to_s)
      end
    end
  end

  class W0736 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:define_variable)
      interp.on_variable_referred += method(:refer_variable)
      interp.on_function_started += method(:enter_function)
      interp.on_function_ended += method(:leave_function)
      interp.on_translation_unit_ended += method(:check)
      @static_variables = {}
      @functions = []
    end

    private
    def define_variable(variable_definition, variable)
      return unless @functions.empty?

      if variable.declared_as_static?
        @static_variables[variable] = [variable_definition, Set.new]
      end
    end

    def refer_variable(object_specifier, variable)
      return if @functions.empty?

      if @static_variables.include?(variable)
        @static_variables[variable].last.add(@functions.last)
      end
    end

    def enter_function(function_definition, function)
      @functions.push(function)
    end

    def leave_function(function_definition, function)
      @functions.pop
    end

    def check(translation_unit)
      @static_variables.each do |variable, (definition, accessors)|
        if accessors.size == 1
          W(:W0736, definition.location, variable.name)
        end
      end
    end
  end

  class W0737 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:check)
      interp.on_variable_defined += method(:check)
    end

    private
    def check(decl_or_def, variable)
      return unless decl_or_def.type.enum?

      if decl_or_def.type.incomplete?
        W(:W0737, decl_or_def.location, decl_or_def.type.name)
      end
    end
  end

  class W0738 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_implicit_conv_performed += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(initializer_or_expression, original_variable, result_variable)
      case initializer_or_expression
      when Initializer
        unless expression = initializer_or_expression.expression
          return
        end
      when Expression
        expression = initializer_or_expression
      end

      return unless expression.constant?(@enum_tbl)

      orig_type = original_variable.type
      conv_type = result_variable.type

      unless orig_type.scalar? && orig_type.integer? &&
          conv_type.scalar? && conv_type.integer? && conv_type.unsigned?
        return
      end

      orig_value = original_variable.value
      return unless orig_value.scalar?

      upper_test = orig_value > ScalarValue.of(conv_type.max_value)

      W(:W0738, expression.location) if upper_test.must_be_true?
    end
  end

  class W0739 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_additive_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(additive_expression, lhs_variable, rhs_variable, result_variable)
      return unless additive_expression.operator.type == "-"

      return unless additive_expression.lhs_operand.constant?(@enum_tbl)
      return unless additive_expression.rhs_operand.constant?(@enum_tbl)

      return unless lhs_variable.type.scalar? && lhs_variable.type.unsigned?
      return unless rhs_variable.type.scalar? && rhs_variable.type.unsigned?

      return unless lhs_variable.value.scalar? && rhs_variable.value.scalar?

      unbound_value = lhs_variable.value - rhs_variable.value

      result_type = result_variable.type
      lower_test = unbound_value < ScalarValue.of(result_type.min_value)

      W(:W0739, additive_expression.location) if lower_test.must_be_true?
    end
  end

  class W0740 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_additive_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(additive_expression, lhs_variable, rhs_variable, result_variable)
      return unless additive_expression.operator.type == "+"

      return unless additive_expression.lhs_operand.constant?(@enum_tbl)
      return unless additive_expression.rhs_operand.constant?(@enum_tbl)

      return unless lhs_variable.type.scalar? && lhs_variable.type.unsigned?
      return unless rhs_variable.type.scalar? && rhs_variable.type.unsigned?

      return unless lhs_variable.value.scalar? && rhs_variable.value.scalar?

      unbound_value = lhs_variable.value + rhs_variable.value

      result_type = result_variable.type
      upper_test = unbound_value > ScalarValue.of(result_type.max_value)

      W(:W0740, additive_expression.location) if upper_test.must_be_true?
    end
  end

  class W0741 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(multiplicative_expression, lhs_variable, rhs_variable,
              result_variable)
      return unless multiplicative_expression.operator.type == "*"

      return unless multiplicative_expression.lhs_operand.constant?(@enum_tbl)
      return unless multiplicative_expression.rhs_operand.constant?(@enum_tbl)

      return unless lhs_variable.type.scalar? && lhs_variable.type.unsigned?
      return unless rhs_variable.type.scalar? && rhs_variable.type.unsigned?

      return unless lhs_variable.value.scalar? && rhs_variable.value.scalar?

      unbound_value = lhs_variable.value * rhs_variable.value

      result_type = result_variable.type
      upper_test = unbound_value > ScalarValue.of(result_type.max_value)

      W(:W0741, multiplicative_expression.location) if upper_test.must_be_true?
    end
  end

  class W0742 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_implicit_conv_performed += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(initializer_or_expression, original_variable, result_variable)
      unless original_variable.type.scalar? && result_variable.type.scalar? &&
          original_variable.type.signed? && result_variable.type.unsigned?
        return
      end

      if initializer_or_expression.kind_of?(Initializer)
        expression = initializer_or_expression.expression
      else
        expression = initializer_or_expression
      end

      if expression && expression.constant?(@enum_tbl) &&
          original_variable.value.must_be_less_than?(ScalarValue.of(0))
        W(:W0742, expression.location)
      end
    end
  end

  class W0743 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_implicit_conv_performed += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(initializer_or_expression, original_variable, result_variable)
      case initializer_or_expression
      when Initializer
        unless expression = initializer_or_expression.expression
          return
        end
      when Expression
        expression = initializer_or_expression
      end

      return unless expression.constant?(@enum_tbl)

      orig_type = original_variable.type
      conv_type = result_variable.type

      unless orig_type.scalar? && orig_type.integer? &&
          conv_type.scalar? && conv_type.integer? && conv_type.signed?
        return
      end

      orig_value = original_variable.value
      return unless orig_value.scalar?

      lower_test = orig_value < ScalarValue.of(conv_type.min_value)
      upper_test = orig_value > ScalarValue.of(conv_type.max_value)

      if lower_test.must_be_true? || upper_test.must_be_true?
        W(:W0743, expression.location)
      end
    end
  end

  class W0744 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_if_ctrlexpr_evaled += method(:check_if_statement)
      interp.on_if_else_ctrlexpr_evaled += method(:check_if_else_statement)
      interp.on_while_ctrlexpr_evaled += method(:check_while_statement)
      interp.on_for_ctrlexpr_evaled += method(:check_for_statement)
      interp.on_c99_for_ctrlexpr_evaled += method(:check_c99_for_statement)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check_if_statement(if_statement, ctrlexpr_value)
      ctrlexpr = if_statement.expression
      if ctrlexpr.constant?(@enum_tbl) && ctrlexpr_value.must_be_false?
        W(:W0744, ctrlexpr.location)
      end
    end

    def check_if_else_statement(if_else_statement, ctrlexpr_value)
      ctrlexpr = if_else_statement.expression
      if ctrlexpr.constant?(@enum_tbl) && ctrlexpr_value.must_be_false?
        W(:W0744, ctrlexpr.location)
      end
    end

    def check_while_statement(while_statement, ctrlexpr_value)
      ctrlexpr = while_statement.expression
      if ctrlexpr.constant?(@enum_tbl) && ctrlexpr_value.must_be_false?
        W(:W0744, ctrlexpr.location)
      end
    end

    def check_for_statement(for_statement, ctrlexpr_value)
      ctrlexpr = for_statement.condition_statement.expression
      if ctrlexpr.constant?(@enum_tbl) && ctrlexpr_value.must_be_false?
        W(:W0744, ctrlexpr.location)
      end
    end

    def check_c99_for_statement(c99_for_statement, ctrlexpr_value)
      ctrlexpr = c99_for_statement.condition_statement.expression
      if ctrlexpr.constant?(@enum_tbl) && ctrlexpr_value.must_be_false?
        W(:W0744, ctrlexpr.location)
      end
    end
  end

  class W0745 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_array_subscript_expr_evaled += method(:check)
    end

    private
    def check(array_subscript_expression,
              array_or_pointer_variable, subscript_variable,
              array_variable, result_variable)
      return unless array_variable && array_variable.type.length
      return unless subscript_variable.value.scalar?

      lower_bound = ScalarValue.of(0)
      upper_bound = ScalarValue.of(array_variable.type.length - 1)

      lower_test = subscript_variable.value < lower_bound
      upper_test = subscript_variable.value > upper_bound

      if lower_test.must_be_true? || upper_test.must_be_true?
        W(:W0745, array_subscript_expression.array_subscript.location)
      end
    end
  end

  class W0747 < W0119
    private
    def from_type
      @interp.short_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0748 < W0119
    private
    def from_type
      @interp.unsigned_short_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0749 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0750 < W0119
    private
    def from_type
      @interp.int_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0751 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0752 < W0119
    private
    def from_type
      @interp.unsigned_int_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0753 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0754 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0755 < W0119
    private
    def from_type
      @interp.long_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0756 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0757 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0758 < W0119
    private
    def from_type
      @interp.unsigned_long_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0759 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.signed_char_type
    end
  end

  class W0760 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.short_type
    end
  end

  class W0761 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.int_type
    end
  end

  class W0762 < W0119
    private
    def from_type
      @interp.long_long_type
    end

    def to_type
      @interp.long_type
    end
  end

  class W0763 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.unsigned_char_type
    end
  end

  class W0764 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.unsigned_short_type
    end
  end

  class W0765 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.unsigned_int_type
    end
  end

  class W0766 < W0119
    private
    def from_type
      @interp.unsigned_long_long_type
    end

    def to_type
      @interp.unsigned_long_type
    end
  end

  class W0767 < W0119
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0768 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0769 < W0119
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0771 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:declare_variable)
      interp.on_function_declared += method(:declare_function)
      interp.on_translation_unit_ended += method(:check)
      @variable_declarations = Hash.new { |hash, key| hash[key] = [] }
      @function_declarations = Hash.new { |hash, key| hash[key] = [] }
    end

    private
    def declare_variable(variable_declaration, variable)
      if variable.named? && variable.declared_as_extern?
        @variable_declarations[variable.name].push(variable_declaration)
      end
    end

    def declare_function(function_declaration, function)
      if function.named? && function.declared_as_extern?
        @function_declarations[function.name].push(function_declaration)
      end
    end

    def check(*)
      @variable_declarations.each_value do |declarations|
        decl_locations = declarations.map { |decl|
          decl.location
        }.uniq { |location| location.fpath }
        if decl_locations.size > 1
          decl_locations.each do |location|
            W(:W0771, location, declarations.first.identifier.value)
          end
        end
      end

      @function_declarations.each_value do |declarations|
        decl_locations = declarations.map { |decl|
          decl.location
        }.uniq { |location| location.fpath }
        if decl_locations.size > 1
          decl_locations.each do |location|
            W(:W0771, location, declarations.first.identifier.value)
          end
        end
      end
    end
  end

  class W0774 < W0255
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0775 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.float_type
    end
  end

  class W0776 < W0255
    private
    def from_type
      @interp.long_double_type
    end

    def to_type
      @interp.double_type
    end
  end

  class W0777 < PassiveMessageDetection
    def initialize(context)
      super
      @interp = context[:c_interpreter]
      @interp.on_implicit_conv_performed += method(:check)
      @interp.on_function_started += method(:clear_rvalues)
      @interp.on_additive_expr_evaled += method(:handle_additive)
      @interp.on_multiplicative_expr_evaled += method(:handle_multiplicative)
      @rvalues = nil
    end

    private
    def check(initializer_or_expression, original_variable, result_variable)
      return unless @rvalues
      return unless original_variable.type.floating?

      case expression = @rvalues[original_variable]
      when AdditiveExpression, MultiplicativeExpression
        if original_variable.type.same_as?(from_type) &&
            result_variable.type.same_as?(to_type)
          W(message_id, expression.location)
        end
      end
    end

    def clear_rvalues(function_definition, function)
      @rvalues = {}
    end

    def handle_additive(additive_expression,
                        lhs_variable, rhs_variable, result_variable)
      memorize_rvalue_derivation(result_variable, additive_expression)
    end

    def handle_multiplicative(multiplicative_expression,
                              lhs_variable, rhs_variable, result_variable)
      return if multiplicative_expression.operator.type == "%"
      memorize_rvalue_derivation(result_variable, multiplicative_expression)
    end

    def memorize_rvalue_derivation(rvalue_holder, expression)
      @rvalues[rvalue_holder] = expression if @rvalues
    end

    def from_type
      @interp.float_type
    end

    def to_type
      @interp.double_type
    end

    def message_id
      self.class.name.sub(/\A.*::/, "").intern
    end
  end

  class W0778 < W0777
    private
    def from_type
      @interp.float_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0779 < W0777
    private
    def from_type
      @interp.double_type
    end

    def to_type
      @interp.long_double_type
    end
  end

  class W0785 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_struct_declared += method(:declare_struct)
      interp.on_union_declared += method(:declare_union)
      interp.on_enum_declared += method(:declare_enum)
      @declared_tag_names = Set.new
    end

    private
    def declare_struct(struct_type_declaration)
      # NOTE: Unique autogenerated tag name is assigned to the unnamed struct
      #       declarations in parsing phase.
      tag_name = struct_type_declaration.identifier.value

      if @declared_tag_names.include?(tag_name)
        W(:W0785, struct_type_declaration.location, tag_name)
      end

      @declared_tag_names.add(tag_name)
    end

    def declare_union(union_type_declaration)
      # NOTE: Unique autogenerated tag name is assigned to the unnamed union
      #       declarations in parsing phase.
      tag_name = union_type_declaration.identifier.value

      if @declared_tag_names.include?(tag_name)
        W(:W0785, union_type_declaration.location, tag_name)
      end

      @declared_tag_names.add(tag_name)
    end

    def declare_enum(enum_type_declaration)
      # NOTE: Unique autogenerated tag name is assigned to the unnamed enum
      #       declarations in parsing phase.
      tag_name = enum_type_declaration.identifier.value

      if @declared_tag_names.include?(tag_name)
        W(:W0785, enum_type_declaration.location, tag_name)
      end

      @declared_tag_names.add(tag_name)
    end
  end

  class W0786 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_struct_type_declaration += method(:check)
      visitor.enter_union_type_declaration += method(:check)
      @interp = context[:c_interpreter]
    end

    private
    def check(node)
      node.struct_declarations.each do |struct_declaration|
        struct_declaration.items.each do |member_declaration|
          type = member_declaration.type
          next unless type.scalar? && type.integer? && type.bitfield?

          case type.base_type
          when @interp.int_type, @interp.unsigned_int_type,
               @interp.signed_int_type
            next
          else
            W(:W0786, node.location)
            return
          end
        end
      end
    end
  end

  class W0787 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:check_object_declaration)
      interp.on_variable_defined += method(:check_object_declaration)
      interp.on_function_declared += method(:check_object_declaration)
      interp.on_function_defined += method(:check_object_declaration)
      interp.on_typedef_declared += method(:check_typedef_declaration)
      interp.on_enum_declared += method(:check_enum_declaration)
      interp.on_block_started += method(:start_block)
      interp.on_block_ended += method(:end_block)

      @object_decls_stack = [Hash.new { |hash, key| hash[key] = [] }]
      @typedef_decls_stack = [Hash.new { |hash, key| hash[key] = [] }]
      @enumerators_stack = [Hash.new(0)]

      @object_decls_in_other_scope = Hash.new { |hash, key| hash[key] = [] }
      @typedef_decls_in_other_scope = Hash.new { |hash, key| hash[key] = [] }
      @enumerators_in_other_scope = Hash.new(0)
    end

    private
    def check_object_declaration(declaration_or_definition, *)
      name = declaration_or_definition.identifier.value
      type = declaration_or_definition.type

      same_name_object_decls = @object_decls_in_other_scope[name]
      unless same_name_object_decls.all? { |decl| decl.type == type }
        W(:W0787, declaration_or_definition.location, name)
        return
      end

      same_name_typedef_decls = @typedef_decls_in_other_scope[name]
      unless same_name_typedef_decls.empty?
        W(:W0787, declaration_or_definition.location, name)
        return
      end

      same_name_enumerators = @enumerators_in_other_scope[name]
      unless same_name_enumerators == 0
        W(:W0787, declaration_or_definition.location, name)
        return
      end
    ensure
      unless @object_decls_stack.empty?
        @object_decls_stack.last[name].push(declaration_or_definition)
      end
    end

    def check_typedef_declaration(typedef_declaration)
      name = typedef_declaration.identifier.value
      type = typedef_declaration.type

      same_name_object_decls = @object_decls_in_other_scope[name]
      unless same_name_object_decls.empty?
        W(:W0787, typedef_declaration.location, name)
        return
      end

      same_name_typedef_decls = @typedef_decls_in_other_scope[name]
      unless same_name_typedef_decls.all? { |decl| decl.type == type }
        W(:W0787, typedef_declaration.location, name)
        return
      end

      same_name_enumerators = @enumerators_in_other_scope[name]
      unless same_name_enumerators == 0
        W(:W0787, typedef_declaration.location, name)
        return
      end
    ensure
      unless @typedef_decls_stack.empty?
        @typedef_decls_stack.last[name].push(typedef_declaration)
      end
    end

    def check_enum_declaration(enum_type_declaration)
      enum_type_declaration.enumerators.each { |enum| check_enumerator(enum) }
    end

    def check_enumerator(enumerator)
      name = enumerator.identifier.value

      same_name_object_decls = @object_decls_in_other_scope[name]
      unless same_name_object_decls.empty?
        W(:W0787, enumerator.location, name)
        return
      end

      same_name_typedef_decls = @typedef_decls_in_other_scope[name]
      unless same_name_typedef_decls.empty?
        W(:W0787, enumerator.location, name)
        return
      end

      same_name_enumerators = @enumerators_in_other_scope[name]
      unless same_name_enumerators == 0
        W(:W0787, enumerator.location, name)
        return
      end
    ensure
      unless @enumerators_stack.empty?
        @enumerators_stack.last[name] += 1
      end
    end

    def start_block(compound_statement)
      @object_decls_stack.push(Hash.new { |hash, key| hash[key] = [] })
      @typedef_decls_stack.push(Hash.new { |hash, key| hash[key] = [] })
      @enumerators_stack.push(Hash.new(0))
    end

    def end_block(compound_statement)
      unless @object_decls_stack.empty?
        @object_decls_stack.last.each do |name, decls|
          @object_decls_in_other_scope[name].concat(decls)
        end
        @object_decls_stack.pop
      end

      unless @typedef_decls_stack.empty?
        @typedef_decls_stack.last.each do |name, decls|
          @typedef_decls_in_other_scope[name].concat(decls)
        end
        @typedef_decls_stack.pop
      end

      unless @enumerators_stack.empty?
        @enumerators_stack.last.each do |name, decl_num|
          @enumerators_in_other_scope[name] += decl_num
        end
      end
    end
  end

  class W0788 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:check_object_declaration)
      interp.on_variable_defined += method(:check_object_declaration)
      interp.on_function_declared += method(:check_object_declaration)
      interp.on_function_defined += method(:check_object_declaration)
      interp.on_typedef_declared += method(:check_typedef_declaration)
      interp.on_enum_declared += method(:check_enum_declaration)
      interp.on_block_started += method(:start_block)
      interp.on_block_ended += method(:end_block)

      @object_decls_stack = [Hash.new { |hash, key| hash[key] = [] }]
      @typedef_decls_stack = [Hash.new { |hash, key| hash[key] = [] }]
      @enumerators_stack = [Hash.new(0)]
    end

    private
    def check_object_declaration(declaration_or_definition, *)
      name = declaration_or_definition.identifier.value
      type = declaration_or_definition.type

      unless @object_decls_stack.empty?
        same_name_object_decls = @object_decls_stack.last[name]
        unless same_name_object_decls.all? { |decl| decl.type == type }
          W(:W0788, declaration_or_definition.location, name)
          return
        end
      end

      unless @typedef_decls_stack.empty?
        same_name_typedef_decls = @typedef_decls_stack.last[name]
        unless same_name_typedef_decls.empty?
          W(:W0788, declaration_or_definition.location, name)
          return
        end
      end

      unless @enumerators_stack.empty?
        same_name_enumerators = @enumerators_stack.last[name]
        unless same_name_enumerators == 0
          W(:W0788, declaration_or_definition.location, name)
          return
        end
      end
    ensure
      unless @object_decls_stack.empty?
        @object_decls_stack.last[name].push(declaration_or_definition)
      end
    end

    def check_typedef_declaration(typedef_declaration)
      name = typedef_declaration.identifier.value
      type = typedef_declaration.type

      unless @object_decls_stack.empty?
        same_name_object_decls = @object_decls_stack.last[name]
        unless same_name_object_decls.empty?
          W(:W0788, typedef_declaration.location, name)
          return
        end
      end

      unless @typedef_decls_stack.empty?
        same_name_typedef_decls = @typedef_decls_stack.last[name]
        unless same_name_typedef_decls.all? { |decl| decl.type == type }
          W(:W0788, typedef_declaration.location, name)
          return
        end
      end

      unless @enumerators_stack.empty?
        same_name_enumerators = @enumerators_stack.last[name]
        unless same_name_enumerators == 0
          W(:W0788, typedef_declaration.location, name)
          return
        end
      end
    ensure
      unless @typedef_decls_stack.empty?
        @typedef_decls_stack.last[name].push(typedef_declaration)
      end
    end

    def check_enum_declaration(enum_type_declaration)
      enum_type_declaration.enumerators.each { |enum| check_enumerator(enum) }
    end

    def check_enumerator(enumerator)
      name = enumerator.identifier.value

      unless @object_decls_stack.empty?
        same_name_object_decls = @object_decls_stack.last[name]
        unless same_name_object_decls.empty?
          W(:W0788, enumerator.location, name)
          return
        end
      end

      unless @typedef_decls_stack.empty?
        same_name_typedef_decls = @typedef_decls_stack.last[name]
        unless same_name_typedef_decls.empty?
          W(:W0788, enumerator.location, name)
          return
        end
      end

      unless @enumerators_stack.empty?
        same_name_enumerators = @enumerators_stack.last[name]
        unless same_name_enumerators == 0
          W(:W0788, enumerator.location, name)
          return
        end
      end
    ensure
      unless @enumerators_stack.empty?
        @enumerators_stack.last[name] += 1
      end
    end

    def start_block(compound_statement)
      @object_decls_stack.push(Hash.new { |hash, key| hash[key] = [] })
      @typedef_decls_stack.push(Hash.new { |hash, key| hash[key] = [] })
      @enumerators_stack.push(Hash.new(0))
    end

    def end_block(compound_statement)
      @object_decls_stack.pop
      @typedef_decls_stack.pop
      @enumerators_stack.pop
    end
  end

  class W0789 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:check_object_declaration)
      interp.on_variable_defined += method(:check_object_declaration)
      interp.on_function_declared += method(:check_object_declaration)
      interp.on_function_defined += method(:check_object_declaration)
      interp.on_typedef_declared += method(:check_typedef_declaration)
      interp.on_enum_declared += method(:check_enum_declaration)
      interp.on_block_started += method(:start_block)
      interp.on_block_ended += method(:end_block)

      @object_decls_stack = [Hash.new { |hash, key| hash[key] = [] }]
      @typedef_decls_stack = [Hash.new { |hash, key| hash[key] = [] }]
      @enumerators_stack = [Hash.new(0)]
    end

    private
    def check_object_declaration(declaration_or_definition, *)
      name = declaration_or_definition.identifier.value
      type = declaration_or_definition.type

      same_name_object_decls =
        merge_same_name_object_decls(name, @object_decls_stack[0..-2])
      unless same_name_object_decls.all? { |decl| decl.type == type }
        W(:W0789, declaration_or_definition.location, name)
        return
      end

      same_name_typedef_decls =
        merge_same_name_typedef_decls(name, @typedef_decls_stack[0..-2])
      unless same_name_typedef_decls.empty?
        W(:W0789, declaration_or_definition.location, name)
        return
      end

      same_name_enumerators =
        merge_same_name_enumerators(name, @enumerators_stack[0..-2])
      unless same_name_enumerators == 0
        W(:W0789, declaration_or_definition.location, name)
        return
      end
    ensure
      unless @object_decls_stack.empty?
        @object_decls_stack.last[name].push(declaration_or_definition)
      end
    end

    def check_typedef_declaration(typedef_declaration)
      name = typedef_declaration.identifier.value
      type = typedef_declaration.type

      same_name_object_decls =
        merge_same_name_object_decls(name, @object_decls_stack[0..-2])
      unless same_name_object_decls.empty?
        W(:W0789, typedef_declaration.location, name)
        return
      end

      same_name_typedef_decls =
        merge_same_name_typedef_decls(name, @typedef_decls_stack[0..-2])
      unless same_name_typedef_decls.all? { |decl| decl.type == type }
        W(:W0789, typedef_declaration.location, name)
        return
      end

      same_name_enumerators =
        merge_same_name_enumerators(name, @enumerators_stack[0..-2])
      unless same_name_enumerators == 0
        W(:W0789, typedef_declaration.location, name)
        return
      end
    ensure
      unless @typedef_decls_stack.empty?
        @typedef_decls_stack.last[name].push(typedef_declaration)
      end
    end

    def check_enum_declaration(enum_type_declaration)
      enum_type_declaration.enumerators.each { |enum| check_enumerator(enum) }
    end

    def check_enumerator(enumerator)
      name = enumerator.identifier.value

      same_name_object_decls =
        merge_same_name_object_decls(name, @object_decls_stack[0..-2])
      unless same_name_object_decls.empty?
        W(:W0789, enumerator.location, name)
        return
      end

      same_name_typedef_decls =
        merge_same_name_typedef_decls(name, @typedef_decls_stack[0..-2])
      unless same_name_typedef_decls.empty?
        W(:W0789, enumerator.location, name)
        return
      end

      same_name_enumerators =
        merge_same_name_enumerators(name, @enumerators_stack[0..-2])
      unless same_name_enumerators == 0
        W(:W0789, enumerator.location, name)
        return
      end
    ensure
      unless @enumerators_stack.empty?
        @enumerators_stack.last[name] += 1
      end
    end

    def start_block(compound_statement)
      @object_decls_stack.push(Hash.new { |hash, key| hash[key] = [] })
      @typedef_decls_stack.push(Hash.new { |hash, key| hash[key] = [] })
      @enumerators_stack.push(Hash.new(0))
    end

    def end_block(compound_statement)
      @object_decls_stack.pop
      @typedef_decls_stack.pop
      @enumerators_stack.pop
    end

    def merge_same_name_object_decls(name, scopes)
      scopes.each_with_object([]) { |hash, result| result.concat(hash[name]) }
    end

    def merge_same_name_typedef_decls(name, scopes)
      scopes.each_with_object([]) { |hash, result| result.concat(hash[name]) }
    end

    def merge_same_name_enumerators(name, scopes)
      scopes.reduce(0) { |result, hash| result + hash[name] }
    end
  end

  class W0790 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check_variable)
      interp.on_function_defined += method(:check_function)
      interp.on_block_started += method(:enter_block)
      interp.on_block_ended += method(:leave_block)
      @global_var_names = Set.new
      @global_fun_names = Set.new
      @block_level = 0
    end

    private
    def check_variable(variable_definition, variable)
      return unless @block_level == 0 && variable.declared_as_extern?

      name = variable_definition.identifier.value
      if @global_var_names.include?(name) || @global_fun_names.include?(name)
        W(:W0790, variable_definition.location, name)
      else
        @global_var_names.add(name)
      end
    end

    def check_function(function_definition, function)
      return unless @block_level == 0 && function.declared_as_extern?

      name = function_definition.identifier.value
      if @global_var_names.include?(name) || @global_fun_names.include?(name)
        W(:W0790, function_definition.location, name)
      else
        @global_fun_names.add(name)
      end
    end

    def enter_block(*)
      @block_level += 1
    end

    def leave_block(*)
      @block_level -= 1
    end
  end

  class W0795 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if prototype = prototype_declaration_of(function)
        param_types = prototype.type.parameter_types
        if arg_variables.size < param_types.select { |type| !type.void? }.size
          W(:W0795, function_call_expression.location)
        end
      end
    end

    def prototype_declaration_of(function)
      function.declarations_and_definitions.find do |decl_or_def|
        decl_or_def.kind_of?(FunctionDeclaration)
      end
    end
  end

  class W0796 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      if prototype = prototype_declaration_of(function)
        return if prototype.type.have_va_list?

        param_types = prototype.type.parameter_types
        if !param_types.empty? &&
            arg_variables.size > param_types.select { |type| !type.void? }.size
          W(:W0796, function_call_expression.location)
        end
      end
    end

    def prototype_declaration_of(function)
      function.declarations_and_definitions.find do |decl_or_def|
        decl_or_def.kind_of?(FunctionDeclaration)
      end
    end
  end

  class W0797 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
    end

    private
    def check(function_call_expression, function, arg_variables,
              result_variable)
      unless prototype_declaration_of(function)
        if definition = kandr_style_definition_of(function)
          case function_declarator = definition.function_declarator
          when KandRFunctionDeclarator
            param_num = function_declarator.identifier_list.size
          else
            return
          end

          unless arg_variables.size == param_num
            W(:W0797, function_call_expression.location)
          end
        end
      end
    end

    def prototype_declaration_of(function)
      function.declarations_and_definitions.find do |decl_or_def|
        decl_or_def.kind_of?(FunctionDeclaration)
      end
    end

    def kandr_style_definition_of(function)
      function.declarations_and_definitions.find do |decl_or_def|
        decl_or_def.kind_of?(KandRFunctionDefinition)
      end
    end
  end

  class W0798 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_indirection_expr_evaled += method(:check_indirection)
      interp.on_member_access_expr_evaled += method(:check_member_access)
      interp.on_array_subscript_expr_evaled += method(:check_array_subscript)
    end

    private
    def check_indirection(indirection_expression, variable, dereferenced)
      return unless variable.type.pointer?

      base_type = variable.type.unqualify.base_type
      if base_type.union? && base_type.incomplete?
        W(:W0798, indirection_expression.location)
      end
    end

    def check_member_access(member_access_expression,
                            outer_variable, member_variable)
      return unless outer_variable.type.pointer?

      base_type = outer_variable.type.unqualify.base_type
      if base_type.union? && base_type.incomplete?
        W(:W0798, member_access_expression.location)
      end
    end

    def check_array_subscript(array_subscript_expression,
                              pointer_variable, subscript_variable,
                              array_variable, result_variable)
      return unless pointer_variable.type.pointer?

      base_type = pointer_variable.type.unqualify.base_type
      if base_type.union? && base_type.incomplete?
        W(:W0798, array_subscript_expression.location)
      end
    end
  end

  class W0799 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_indirection_expr_evaled += method(:check_indirection)
      interp.on_member_access_expr_evaled += method(:check_member_access)
      interp.on_array_subscript_expr_evaled += method(:check_array_subscript)
    end

    private
    def check_indirection(indirection_expression, variable, dereferenced)
      return unless variable.type.pointer?

      base_type = variable.type.unqualify.base_type
      if base_type.struct? && base_type.incomplete?
        W(:W0799, indirection_expression.location)
      end
    end

    def check_member_access(member_access_expression,
                            outer_variable, member_variable)
      return unless outer_variable.type.pointer?

      base_type = outer_variable.type.unqualify.base_type
      if base_type.struct? && base_type.incomplete?
        W(:W0799, member_access_expression.location)
      end
    end

    def check_array_subscript(array_subscript_expression,
                              pointer_variable, subscript_variable,
                              array_variable, result_variable)
      return unless pointer_variable.type.pointer?

      base_type = pointer_variable.type.unqualify.base_type
      if base_type.struct? && base_type.incomplete?
        W(:W0799, array_subscript_expression.location)
      end
    end
  end

  class W0800 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check)
    end

    private
    def check(variable_definition, variable)
      unless variable.declared_as_extern?
        if variable.type.incomplete?
          W(:W0800, variable_definition.location, variable.name)
        end
      end
    end
  end

  class W0810 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_if_statement += method(:enter_outer_if_stmt)
      visitor.leave_if_statement += method(:leave_outer_if_stmt)
      visitor.enter_if_else_statement += method(:check_inner_if_else_stmt)
      @if_stmt_stack = []
    end

    private
    def enter_outer_if_stmt(if_statement)
      @if_stmt_stack.push(if_statement)
    end

    def leave_outer_if_stmt(if_statement)
      @if_stmt_stack.pop
    end

    def check_inner_if_else_stmt(if_else_statement)
      if outer_if_statement = @if_stmt_stack.last and
          !outer_if_statement.statement.kind_of?(CompoundStatement)
        W(:W0810, outer_if_statement.location)
      end
    end
  end

  class W0827 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_definition += method(:enter_variable_definition)
    end

    private
    def enter_variable_definition(node)
      if outmost_initializer = node.initializer
        if initializers = outmost_initializer.initializers
          case
          when node.type.array?
            check(node.type.base_type, initializers.first)
          when node.type.struct?
            node.type.members.zip(initializers).each do |memb, init|
              check(memb.type, init) if init
            end
          end
        end
      end
    end

    def check(type, initializer)
      return unless initializer

      if initializers = initializer.initializers
        case
        when type.array?
          check(type.base_type, initializers.first)
        when type.struct?
          type.members.zip(initializers).each do |memb, init|
            check(memb.type, init) if init
          end
        end
      else
        W(:W0827, initializer.location) if type.struct?
      end
    end
  end

  class W0828 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_definition += method(:enter_variable_definition)
    end

    private
    def enter_variable_definition(node)
      if outmost_initializer = node.initializer
        if initializers = outmost_initializer.initializers
          case
          when node.type.array?
            check(node.type.base_type, initializers.first)
          when node.type.struct?
            node.type.members.zip(initializers).each do |memb, init|
              check(memb.type, init) if init
            end
          end
        end
      end
    end

    def check(type, initializer)
      return unless initializer

      case
      when initializers = initializer.initializers
        case
        when type.array?
          check(type.base_type, initializers.first)
        when type.struct?
          type.members.zip(initializers).each do |memb, init|
            check(memb.type, init) if init
          end
        end
      when initializer.expression.kind_of?(StringLiteralSpecifier)
      else
        W(:W0828, initializer.location) if type.array?
      end
    end
  end

  class W0947 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_string_literal_specifier += method(:check)
    end

    private
    def check(string_literal_specifier)
      unless string_literal_specifier.literal.replaced?
        W(:W0947, string_literal_specifier.location)
      end
    end
  end

  class W0948 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_constant_specifier += method(:check)
    end

    private
    def check(constant_specifier)
      if constant_specifier.character? &&
          !constant_specifier.constant.replaced?
        W(:W0948, constant_specifier.location,
          constant_specifier.constant.value)
      end
    end
  end

  class W0949 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_struct_declarator += method(:check)
      @c_interp = Interpreter.new(context[:c_type_table])
    end

    private
    def check(node)
      if node.kind_of?(StructDeclarator)
        if expr = node.expression and
            expr.kind_of?(ConstantSpecifier) && !expr.constant.replaced?
          bitfield_width = compute_bitfield_width(expr)
          if bitfield_width > 1
            W(:W0949, expr.location, expr.constant.value)
          end
        end
      end
    end

    def compute_bitfield_width(expression)
      object = @c_interp.execute(expression, InterpreterOptions::QUIET)
      if object.variable? && object.value.scalar?
        return object.value.unique_sample || 0
      end
      0
    end
  end

  class W0950 < PassiveMessageDetection
    include SyntaxNodeCollector

    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_variable_definition += method(:check)
    end

    private
    def check(variable_definition)
      return unless variable_definition.type.array?

      declarator = variable_definition.init_declarator.declarator
      array_declarators = collect_array_declarators(declarator)

      array_declarators.each do |array_declarator|
        if expression = array_declarator.size_expression
          constants = collect_constant_specifiers(expression)
          if immediate = constants.find { |c| !c.constant.replaced? }
            W(:W0950, immediate.location, immediate.to_s)
          end
        end
      end
    end
  end

  class W1027 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check)
      interp.on_variable_declared += method(:check)
    end

    private
    def check(definition_or_declaration, variable)
      if variable.type.array? && variable.type.base_type.function?
        W(:W1027, definition_or_declaration.location)
      end
    end
  end

  class W1028 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check)
      interp.on_variable_declared += method(:check)
    end

    private
    def check(definition_or_declaration, variable)
      return unless variable.type.array?

      type = variable.type.base_type
      while type.array?
        if type.length
          type = type.base_type
        else
          W(:W1028, definition_or_declaration.location)
          break
        end
      end
    end
  end

  class W1029 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check)
      interp.on_variable_declared += method(:check)
    end

    private
    def check(definition_or_declaration, variable)
      return unless variable.type.array?

      if variable.type.base_type.composite? || variable.type.base_type.void?
        if variable.type.base_type.incomplete?
          W(:W1029, definition_or_declaration.location)
        end
      end
    end
  end

  class W1031 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_declared += method(:declare_variable)
      interp.on_variable_defined += method(:define_variable)
      interp.on_function_declared += method(:declare_function)
      interp.on_function_defined += method(:define_function)
    end

    private
    def declare_variable(variable_declaration, variable)
      if variable.named?
        case
        when variable.declared_as_extern?
          sc_specifier = variable_declaration.storage_class_specifier
          if sc_specifier && sc_specifier.type == :STATIC
            W(:W1031, variable_declaration.location, variable.name)
          end
        when variable.declared_as_static?
          sc_specifier = variable_declaration.storage_class_specifier
          if sc_specifier && sc_specifier.type == :EXTERN
            W(:W1031, variable_declaration.location, variable.name)
          end
        end
      end
    end

    def define_variable(variable_definition, variable)
      if variable.named?
        case
        when variable.declared_as_extern?
          sc_specifier = variable_definition.storage_class_specifier
          if sc_specifier && sc_specifier.type == :STATIC
            W(:W1031, variable_definition.location, variable.name)
          end
        when variable.declared_as_static?
          sc_specifier = variable_definition.storage_class_specifier
          if sc_specifier && sc_specifier.type == :EXTERN
            W(:W1031, variable_definition.location, variable.name)
          end
        end
      end
    end

    def declare_function(function_declaration, function)
      if function.named?
        case
        when function.declared_as_extern?
          sc_specifier = function_declaration.storage_class_specifier
          if sc_specifier && sc_specifier.type == :STATIC
            W(:W1031, function_declaration.location, function.name)
          end
        when function.declared_as_static?
          sc_specifier = function_declaration.storage_class_specifier
          if sc_specifier && sc_specifier.type == :EXTERN
            W(:W1031, function_declaration.location, function.name)
          end
        end
      end
    end

    def define_function(function_definition, function)
      if function.named?
        case
        when function.declared_as_extern?
          sc_specifier = function_definition.storage_class_specifier
          if sc_specifier && sc_specifier.type == :STATIC
            W(:W1031, function_definition.location, function.name)
          end
        when function.declared_as_static?
          sc_specifier = function_definition.storage_class_specifier
          if sc_specifier && sc_specifier.type == :EXTERN
            W(:W1031, function_definition.location, function.name)
          end
        end
      end
    end
  end

  class W1032 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_variable_defined += method(:check)
    end

    private
    def check(variable_definition, variable)
      if variable.declared_as_static? && variable.type.incomplete?
        W(:W1032, variable_definition.location, variable.name)
      end
    end
  end

  class W1034 < PassiveMessageDetection
    def initialize(context)
      super
      visitor = context[:c_visitor]
      visitor.enter_function_declaration += method(:check)
      visitor.enter_compound_statement += method(:enter_block)
      visitor.leave_compound_statement += method(:leave_block)
      @block_level = 0
    end

    private
    def check(declaration)
      if @block_level > 0
        if scs = declaration.storage_class_specifier and scs.type == :STATIC
          W(:W1034, declaration.location, declaration.identifier.value)
        end
      end
    end

    def enter_block(*)
      @block_level += 1
    end

    def leave_block(*)
      @block_level -= 1
    end
  end

  class W1049 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      orig_type = original_variable.type
      conv_type = result_variable.type

      unless orig_type.scalar? && orig_type.integer? &&
          conv_type.scalar? && conv_type.integer? && conv_type.signed?
        return
      end

      orig_value = original_variable.value
      return unless orig_value.scalar?

      lower_test = orig_value < ScalarValue.of(conv_type.min_value)
      upper_test = orig_value > ScalarValue.of(conv_type.max_value)

      if !lower_test.must_be_true? && lower_test.may_be_true? or
          !upper_test.must_be_true? && upper_test.may_be_true?
        W(:W1049, expression.location)
      end
    end
  end

  class W1050 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_explicit_conv_performed += method(:check)
    end

    private
    def check(expression, original_variable, result_variable)
      orig_type = original_variable.type
      conv_type = result_variable.type

      unless orig_type.scalar? && orig_type.integer? &&
          conv_type.scalar? && conv_type.integer? && conv_type.signed?
        return
      end

      orig_value = original_variable.value
      return unless orig_value.scalar?

      lower_test = orig_value < ScalarValue.of(conv_type.min_value)
      upper_test = orig_value > ScalarValue.of(conv_type.max_value)

      if lower_test.must_be_true? || upper_test.must_be_true?
        W(:W1050, expression.location)
      end
    end
  end

  class W1051 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check)
      interp.on_additive_expr_evaled += method(:check)
    end

    private
    def check(binary_expression, lhs_variable, rhs_variable, result_variable)
      return unless lhs_variable.type.scalar? && lhs_variable.type.unsigned?
      return unless rhs_variable.type.scalar? && rhs_variable.type.unsigned?

      return unless lhs_variable.value.scalar? && rhs_variable.value.scalar?

      case binary_expression.operator.type
      when "+"
        unbound_value = lhs_variable.value + rhs_variable.value
      when "-"
        unbound_value = lhs_variable.value - rhs_variable.value
      when "*"
        unbound_value = lhs_variable.value * rhs_variable.value
      else
        return
      end

      result_type = result_variable.type
      lower_test = unbound_value < ScalarValue.of(result_type.min_value)
      upper_test = unbound_value > ScalarValue.of(result_type.max_value)

      if lower_test.must_be_true? || upper_test.must_be_true?
        W(:W1051, binary_expression.location)
      end
    end
  end

  class W1052 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_multiplicative_expr_evaled += method(:check)
      interp.on_additive_expr_evaled += method(:check)
    end

    private
    def check(binary_expression, lhs_variable, rhs_variable, result_variable)
      return unless lhs_variable.type.scalar? && lhs_variable.type.unsigned?
      return unless rhs_variable.type.scalar? && rhs_variable.type.unsigned?

      return unless lhs_variable.value.scalar? && rhs_variable.value.scalar?

      case binary_expression.operator.type
      when "+"
        unbound_value = lhs_variable.value + rhs_variable.value
      when "-"
        unbound_value = lhs_variable.value - rhs_variable.value
      when "*"
        unbound_value = lhs_variable.value * rhs_variable.value
      else
        return
      end

      result_type = result_variable.type
      lower_test = unbound_value < ScalarValue.of(result_type.min_value)
      upper_test = unbound_value > ScalarValue.of(result_type.max_value)

      if !lower_test.must_be_true? && lower_test.may_be_true? or
          !upper_test.must_be_true? && upper_test.may_be_true?
        W(:W1052, binary_expression.location)
      end
    end
  end

  class W1053 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(expression, function, arg_variables, result_variable)
      args = arg_variables.zip(function.type.parameter_types)
      args.each_with_index do |(arg_variable, param_type), index|
        next unless param_type && param_type.enum?

        arg_expr = expression.argument_expressions[index]
        next unless arg_expr.constant?(@enum_tbl)

        unless arg_variable.type.enum?
          W(:W1053, arg_expr.location)
        end
      end
    end
  end

  class W1054 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_assignment_expr_evaled += method(:check)
    end

    private
    def check(expression, lhs_variable, rhs_variable)
      return unless lhs_variable.type.enum?
      W(:W1054, expression.location) unless rhs_variable.type.enum?
    end
  end

  class W1055 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:start_function)
      interp.on_function_ended += method(:end_function)
      interp.on_return_stmt_evaled += method(:check)
      @current_function = nil
    end

    private
    def start_function(function_definition, function)
      @current_function = function
    end

    def end_function(function_definition, function)
      @current_function = nil
    end

    def check(return_statement, result_variable)
      return unless @current_function && result_variable

      return unless return_type = @current_function.type.return_type
      return unless return_type.enum?

      unless result_variable.type.enum?
        W(:W1055, return_statement.expression.location)
      end
    end
  end

  class W1056 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(expression, function, arg_variables, result_variable)
      args = arg_variables.zip(function.type.parameter_types)
      args.each_with_index do |(arg_variable, param_type), index|
        next unless param_type && param_type.enum?

        arg_expr = expression.argument_expressions[index]
        next if arg_expr.constant?(@enum_tbl)

        if arg_variable.type.enum?
          unless arg_variable.type.same_as?(param_type)
            W(:W1056, arg_expr.location)
          end
        end
      end
    end
  end

  class W1057 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_assignment_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(expression, lhs_variable, rhs_variable)
      return unless lhs_variable.type.enum?
      return if expression.rhs_operand.constant?(@enum_tbl)

      if rhs_variable.type.enum?
        unless lhs_variable.type.same_as?(rhs_variable.type)
          W(:W1057, expression.location)
        end
      end
    end
  end

  class W1058 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:start_function)
      interp.on_function_ended += method(:end_function)
      interp.on_return_stmt_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
      @current_function = nil
    end

    private
    def start_function(function_definition, function)
      @current_function = function
    end

    def end_function(function_definition, function)
      @current_function = nil
    end

    def check(return_statement, result_variable)
      return unless @current_function && result_variable

      return unless return_type = @current_function.type.return_type
      return unless return_type.enum?
      return if return_statement.expression.constant?(@enum_tbl)

      if result_variable.type.enum?
        unless return_type.same_as?(result_variable.type)
          W(:W1058, return_statement.expression.location)
        end
      end
    end
  end

  class W1059 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(expression, function, arg_variables, result_variable)
      args = arg_variables.zip(function.type.parameter_types)
      args.each_with_index do |(arg_variable, param_type), index|
        next if param_type.nil? || param_type.enum?

        if arg_variable.type.enum?
          W(:W1059, expression.argument_expressions[index].location)
        end
      end
    end
  end

  class W1060 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:start_function)
      interp.on_function_ended += method(:end_function)
      interp.on_return_stmt_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
      @current_function = nil
    end

    private
    def start_function(function_definition, function)
      @current_function = function
    end

    def end_function(function_definition, function)
      @current_function = nil
    end

    def check(return_statement, result_variable)
      return unless @current_function && result_variable

      return unless return_type = @current_function.type.return_type
      return if return_type.enum?

      if result_variable.type.enum?
        W(:W1060, return_statement.expression.location)
      end
    end
  end

  class W1061 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_call_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(expression, function, arg_variables, result_variable)
      args = arg_variables.zip(function.type.parameter_types)
      args.each_with_index do |(arg_variable, param_type), index|
        next unless param_type && param_type.enum?

        arg_expr = expression.argument_expressions[index]
        next if arg_expr.constant?(@enum_tbl)

        W(:W1061, arg_expr.location) unless arg_variable.type.enum?
      end
    end
  end

  class W1062 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_assignment_expr_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
    end

    private
    def check(expression, lhs_variable, rhs_variable)
      return unless lhs_variable.type.enum?
      return if expression.rhs_operand.constant?(@enum_tbl)

      W(:W1062, expression.location) unless rhs_variable.type.enum?
    end
  end

  class W1063 < PassiveMessageDetection
    def initialize(context)
      super
      interp = context[:c_interpreter]
      interp.on_function_started += method(:start_function)
      interp.on_function_ended += method(:end_function)
      interp.on_return_stmt_evaled += method(:check)
      @enum_tbl = interp.environment.enumerator_table
      @current_function = nil
    end

    private
    def start_function(function_definition, function)
      @current_function = function
    end

    def end_function(function_definition, function)
      @current_function = nil
    end

    def check(return_statement, result_variable)
      return unless @current_function && result_variable

      return unless return_type = @current_function.type.return_type
      return unless return_type.enum?
      return if return_statement.expression.constant?(@enum_tbl)

      unless result_variable.type.enum?
        W(:W1063, return_statement.expression.location)
      end
    end
  end

  class W1064 < W0731
    private
    def check(case_labeled_statement, ctrlexpr_result)
      unless switch_ctrlexpr_result = @switch_ctrlexpr_stack.last
        return
      end

      return unless switch_ctrlexpr_result.type.enum?

      unless ctrlexpr_result.type.enum?
        W(:W1064, case_labeled_statement.expression.location)
      end
    end
  end

  class W1065 < W0731
    private
    def check(case_labeled_statement, ctrlexpr_result)
      unless switch_ctrlexpr_result = @switch_ctrlexpr_stack.last
        return
      end

      return unless switch_ctrlexpr_result.type.enum?
      expected_type = switch_ctrlexpr_result.type

      if ctrlexpr_result.type.enum?
        unless ctrlexpr_result.type.same_as?(expected_type)
          W(:W1065, case_labeled_statement.expression.location)
        end
      end
    end
  end

  class W9001 < PassiveMessageDetection
    def initialize(context)
      super
      @fpath = context[:sources].first.fpath
      visitor = context[:c_visitor]
      visitor.enter_error_statement += method(:check)
      visitor.enter_generic_labeled_statement += method(:check)
      visitor.enter_case_labeled_statement += method(:check)
      visitor.enter_default_labeled_statement += method(:check)
      visitor.enter_expression_statement += method(:check)
      visitor.enter_if_statement += method(:check)
      visitor.enter_if_else_statement += method(:check)
      visitor.enter_switch_statement += method(:check)
      visitor.enter_while_statement += method(:check)
      visitor.enter_do_statement += method(:check)
      visitor.enter_for_statement += method(:check)
      visitor.enter_c99_for_statement += method(:check)
      visitor.enter_goto_statement += method(:check)
      visitor.enter_continue_statement += method(:check)
      visitor.enter_break_statement += method(:check)
      visitor.enter_return_statement += method(:check)
    end

    private
    def check(node)
      return unless @fpath == node.location.fpath
      W(:W9001, node.location) unless node.executed?
    end
  end

end
end
