# Progress monitoring mechanism.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/token"
require "adlint/error"
require "adlint/report"

module AdLint #:nodoc:

  module MonitorUtil
    def monitored_region(title, total = 1)
      ProgressMonitor.instance.start(title, total)
      result = yield(ProgressMonitor.instance)
      ProgressMonitor.instance.finish
      result
    rescue Error
      ProgressMonitor.instance.abort
      raise
    rescue => ex
      ProgressMonitor.instance.abort
      raise InternalError.new(ex, ProgressMonitor.instance.location)
    end
    module_function :monitored_region

    def checkpoint(location_or_numeric)
      case location_or_numeric
      when Location
        ProgressMonitor.instance.location = location_or_numeric
      when Numeric
        ProgressMonitor.instance.progress = location_or_numeric
      end
    end
    module_function :checkpoint
  end

  class ProgressMonitor
    include Singleton

    attr_reader :location
    attr_reader :progress

    def reset(fpath, phase_num)
      @fpath = fpath
      @phase_num = phase_num
      @current_phase = 0
      @start_time = Time.now
    end

    def start(title, total = 1)
      @total = total
      @title = title
      @location = nil
      @progress = 0
      @current_phase += 1
      draw
    end

    def finish
      @progress = @total
      if @current_phase == @phase_num
        draw_finished
      else
        draw
      end
    end

    def abort
      draw_aborted
    end

    def location=(location)
      @location = location
      if false && File.identical?(location.fpath, @fpath)
        self.progress = location.line_no
      end
    end

    def progress=(value)
      @progress = value
      draw
    end

    private
    def draw
      return unless verbose?
      draw_bar(@fpath, @title)
      print " %3d%%" % (total_progress * 100).to_i
    end

    def draw_finished
      return unless verbose?
      draw_bar(@fpath, "fin")
      puts " %.3fs" % (Time.now - @start_time)
    end

    def draw_aborted
      return unless verbose?
      draw_bar(@fpath, @title)
      puts " %.3fs!" % (Time.now - @start_time)
    end

    def total_progress
      phase_start = (@current_phase - 1).to_f / @phase_num
      phase_progress = @progress.to_f / @total / @phase_num
      phase_start + phase_progress
    end

    def draw_bar(fpath, title)
      print "\r%30.30s [%3.3s] |" % [fpath.to_s.scan(/.{,30}\z/).first, title]
      print "=" * (28 * total_progress).to_i
      print " " * (28 - (28 * total_progress).to_i)
      print "|"
    end
  end

end
