/*
   Project: Adun

   Copyright (C) 2007 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
#ifndef ADUN_MMFORCEFIELD
#define ADUN_MMFORCEFIELD

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdbool.h>
#include "Base/AdVector.h"
#include "Base/AdMatrix.h"
#include "Base/AdLinkedList.h"
#include "Base/AdForceFieldFunctions.h"
#include "AdunKernel/AdunForceField.h"
#include "AdunKernel/AdunNonbondedTerm.h"
#include "AdunKernel/AdunPureNonbondedTerm.h"

/*!
\ingroup Inter
AdMolecularMechanicsForceField is an abstract class that defines the interface for objects that calculate the
energy and forces associated with the elements of an AdSystem or AdInteractionSystem object using a molecular mechanics type
potential function. 

A molecular mechanics force field is usually defined by a function which is the sum of a number of independant mathematical terms.

\f[ U(\vec r) = \sum T_{1} + \sum T_{2} + ... + \sum T_{n} \f]

Each term represents an interaction that can occur between the elements of a system e.g. bonded, electrostatic etc.
Applying the force field function to a set of elements gives their energy.

The application of a force field involves identifying all occurances of each interaction type in the system.
Then the corresponding mathematical term is used to evaluate each occurances energy and finally all the results are summed.
It is usual that the individual terms are parameterisable functions with the parameters depending on the element types
involved in the interaction.

Each AdMolecularMechanicsForceField subclass represents a different force field function i.e. a different combination of terms.
It is important to note that they only represent the mathematical function and contain no parameter information.
The lists of occurances of each interaction along with the parameters are provided by AdSystem or AdInteractionSystem objects.
AdMolecularMechanicsForceField subclassses associate a name with each of their terms. The coreTerms() method returns
an array containing the names.  

When given a system an AdMolecularMechanicsForceField object checks which interactions are present in it 
using AdSystem::availableInteractions. 
It then extracts information for those interactions which correspond to terms in its force field using the methods
defined by the AdSystemDataSource protocol. The information required for each
term depends on the force field. For example for calculating the bonded term a force field may require a list
of all bonded elements aswell as parameters for each. The individual subclasses detail what exactly they require.

\note
This class implementes a lot of common functionality for its current subclasses since they are very similar. 
However this means that it is tightly coupled to a force field which has the following core terms :
HarmonicBond, HarmonicAngle, FourierTorsion, HarmonicImproperTorsion, CoulombElectrostatic and an VdW interaction.
If other subclasses are implemented it may be necessary to refactor this class to be more general (e.g move some functionality
back to the current subclasses).
*/

@interface AdMolecularMechanicsForceField:  AdForceField
{
	int no_of_atoms;
	int updateInterval;
	BOOL harmonicBond;
	BOOL harmonicAngle;
	BOOL fourierTorsion;
	BOOL improperTorsion;
	BOOL nonbonded;
	double bnd_pot;
	double ang_pot;
	double tor_pot;
	double itor_pot;
	double vdw_pot;
	double est_pot;
	double total_energy;
	double cutoff;
	double *reciprocalMasses;
	AdMatrix *bonds;
	AdMatrix *angles;
	AdMatrix *torsions;
	AdMatrix *improperTorsions;
	AdMatrix* forceMatrix;
	AdMatrix* accelerationMatrix;
	AdListHandler* nonbondedHandler;
	id system;
	id nonbondedTerm;  
	id systemKeywords;
	id coreTerms;		//the core terms of the force field
	id availableTerms;	//the core terms that the current system provides information for
	id state;		//the current state of the simulator
	id customTerms;
	id bondedInteractions;
	id nonbondedInteractionTypes;
	NSString* vdwInteractionType; 	//Identifies which vdw interaction is used.
}	
/**
As initWithSystem:() passing nil for \e system
*/
- (id) init;
/**
As initWithSystem:nobondedTerm() passing nil for \e anObject
*/
- (id) initWithSystem: (id) system;
/**
As initWithSystem:nonbondedTerm:customTerms:() passing nil for \e aDict.
*/
- (id) initWithSystem: (id) system nonbondedTerm: (AdNonbondedTerm*) anObject;
/**
Initialises an new AdMolecularMechanicsForceField instance for calculating the force and energies
of \e system using the enzymix force field. 
\param system An AdSystem or AdInteractionSystem object.
\param anObject An AdNonbondedTerm object that will handle the nonbonded terms of the force field.
If the system associated with \e anObject is not the same
as \e system it is replaced by it. In addition the lennard
jones type used by \e anObject is automatically set to "A".
If \e anObject is nil and \system is not nil a default nonbonded term object is created.
This is an AdPureNonbondedTerm
instance with a cutoff of 12 and update interval of 20.
\param aDict An optional dictionary whose keys are term names and whose values
are objects that conform to the AdForceFieldTerm protocol. They will
be added to the force field using addCustomTerms:().
*/
- (id) initWithSystem: (id) system 
	nonbondedTerm: (AdNonbondedTerm*) anObject
	customTerms: (NSDictionary*) aDict;
/**
Sets the receiver to use \e anObject for calculating the nonbonded terms of
the force field. If the system associated with \e anObject is not the same
as the revceivers \e system then it is replaced with it. The lennard jones 
type used by \e anObject is set to "A" if it is not already.
*/
- (void) setNonbondedTerm: (AdNonbondedTerm*) anObject;
/**
Returns the object used to calculate the nonbonded terms of the force field.
*/
- (AdNonbondedTerm*) nonbondedTerm;
/**
\todo Partial Implementation - Does not provided energies for custom terms.
*/
- (NSArray*) arrayOfEnergiesForTerms: (NSArray*) terms notFoundMarker: (id) anObject;
/**
\bug Incorrect handling of situtation where \e name already exists.
*/
- (void) addCustomTerm: (id) object withName: (NSString*) name;
/** \bug - Does not handle the case where \e system does not contain the necessary parameters. Such
an occurance will produce a segmentation fault.
*/
- (void) setSystem: (id) anObject;
/**
Replaces the current custom terms with those in \e aDict.
All the objects being added as terms must conform to the AdForceFieldTerm
protocol. If any object does not conform to this protocol an exception is
raised. In this case the object is left in its initial state.
*/
- (void) setCustomTerms: (NSDictionary*) aDict;
/**
Returns the names of the core terms of the force field.
*/
- (NSArray*) coreTerms;
/**
Returns an array containing the last calculated energies associated
with each core term. The order of the returned array corresponds to the order
of the array returned by coreTerms().
*/
- (NSArray*) arrayOfCoreTermEnergies;
/**
Returns a dictionary whose keys are core term names and whose values are the
energies for each.
*/
- (NSDictionary*) dictionaryOfCoreTermEnergies;
/**
Returns the vdw function type used by receiver
*/
- (NSString*) vdwInteractionType;
@end

#endif
