/**********************************************************************
 * options.c                                                 March 2001
 * Horms                                             horms@vergenet.net
 *
 * Read in command line options
 * Code based on man getopt(3), later translated to popt
 *
 * aggregate
 * CIDR network aggregation and filtering
 * Copyright (C) 2001  Horms
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
 * 02111-1307  USA
 *
 **********************************************************************/

#include "options.h"
#include "cidr_net.h"
#include "log.h"
#include "config.h"
#include <popt.h>
#include <stdlib.h>

options_t opt;


/**********************************************************************
 * options
 * Read in command line options
 * pre: argc: number or elements in argv
 *      argv: array of strings with command line-options
 *      flag: see options.h for flag values
 *            ignores errors otherwise
 * post: global opt is seeded with values according to argc and argv
 **********************************************************************/

int options(int argc, char **argv){
  int c=0;
  char *optarg;
  poptContext context;

  static struct poptOption options[] =
  {
    {"help",           'h', POPT_ARG_NONE,   NULL, 'h'},
    {"maximum_prefix", 'M', POPT_ARG_STRING, NULL, 'M'},
    {"minimim_prefix", 'm', POPT_ARG_STRING, NULL, 'm'},
    {"output_format",  'o', POPT_ARG_STRING, NULL, 'o'},
    {"input_format",   'i', POPT_ARG_STRING, NULL, 'i'},
    {NULL,             0,   0,               NULL, 0  }
  };

  if(argc==0 || argv==NULL) return(0);

  opt.maximum_prefix=DEFAULT_MAXIMUM_PREFIX;
  opt.minimum_prefix=DEFAULT_MINIMUM_PREFIX;
  opt.output_format=DEFAULT_OUTPUT_FORMAT;
  opt.input_format=DEFAULT_INPUT_FORMAT;
  
  context=poptGetContext("aggregate", argc, (char **)argv, options, 0);

  while ((c=poptGetNextOpt(context)) >= 0){
    optarg=poptGetOptArg(context);
    switch (c){
      case 'h':
	usage(0); 
	break;
      case 'M':
        opt.maximum_prefix=(prefix_t)atoi(optarg);
        break;
      case 'm':
        opt.minimum_prefix=(prefix_t)atoi(optarg);
        break;
      case 'o':
        if(strcasecmp("prefix", optarg)==0){
	  opt.output_format=CIDR_NET_PREFIX;
	}
	else if(strcasecmp("netmask", optarg)==0){
	  opt.output_format=CIDR_NET_NETMASK;
	}
	else if(strcasecmp("range", optarg)==0){
	  opt.output_format=CIDR_ADDRESS_RANGE;
	}
	else {
	  AGGREGATE_LOG(LOG_DEBUG, "Invalid output format: %s", optarg);
          usage(-1);
	}
        break;
      case 'i':
        if(strcasecmp("prefix", optarg)==0){
	  opt.input_format=CIDR_NET_PREFIX;
	}
	else if(strcasecmp("netmask", optarg)==0){
	  opt.input_format=CIDR_NET_NETMASK;
	}
	else if(strcasecmp("range", optarg)==0){
	  opt.input_format=CIDR_ADDRESS_RANGE;
	}
	else {
	  AGGREGATE_LOG(LOG_DEBUG, "Invalid input format: %s", optarg);
          usage(-1);
	}
        break;
    }
  }

  if (c < -1) {
    AGGREGATE_LOG(
      LOG_DEBUG,
      "options: %s: %s\n",
      poptBadOption(context, POPT_BADOPTION_NOALIAS),
      poptStrerror(c)
    );
    usage(-1);
  }
  
  poptFreeContext(context);

  return(0);
}


/**********************************************************************
 * usage
 * Display usage information
 * Printed to stdout if exit_status=0, stderr otherwise
 **********************************************************************/

void usage(int exit_status){
  FILE *stream;

  stream=(exit_status)?stderr:stdout;
  
  fprintf(
    stream, 
    "aggregate version %s Copyright Horms\n"
    "\n"
    "aggregate is tool for aggregating CIDR networks\n"
    "\n"
    "Usage: aggregate [options]\n"
    "\n"
    "Options:\n"
    " -h|--help:\n"
    "    Display this message\n"
    " -i|--input-format:\n"
    "    Format for input. One of: prefix, netmask, range.\n"
    "    (default \"prefix\")\n" 
    " -M|--maximum_prefix:\n"
    "    Supress netorks whose prefix is greater than this value.\n"
    "    Valid values 0-32. -1 for no maximum prefix.\n"
    "    (default %d)\n"
    " -m|--minimum_prefix:\n"
    "    Supress netorks whose prefix is less than this value.\n"
    "    Valid values 0-32. -1 for no minimum prefix.\n"
    "    (default %d)\n"
    " -o|--output-format:\n"
    "    Format for output. One of: prefix, netmask, range.\n"
    "    (default \"prefix\")\n",
    VERSION,
    DEFAULT_MAXIMUM_PREFIX,
    DEFAULT_MINIMUM_PREFIX
  );

  fflush(stream);
  exit(exit_status);
}
