/**********************************************************************
 ** Book class: Handles book objects, which are objects that point to a
 **             file so that when the book is read, the file is displayed
 ** 
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef BOOK_C
#define BOOK_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mudobject.h"
#include "objtype.h"
#include "book.h"
#include "pager.h"
#include "player.h"
#include "inp_handler.h"
#include "inp_funct.h"
#include "flags.h"
#include "newfuncts.h"
#include "memchk.h"
#include "global.h"

/***********************************************************************
 ** Book (constructor)
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Book::Book(void)
{
}


/***********************************************************************
 ** Book (constructor) - creates the item
 **
 ** Parameters: the_name - the name of the marker
 **             the_area - the area the marker belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Book::Book(char *the_name, char *the_area)
{
   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_BOOK;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   size = weight = state = 0;  
}

/***********************************************************************
 ** ~Book (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Book::~Book()
{
}


/***********************************************************************
 ** write_object - writes the book to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Book::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nbook %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);
   write_moveable_attrib(the_file);

   fprintf(the_file, "%s\n", get_book_subject());
}

/***********************************************************************
 ** describe - describes the book to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Book::describe(Builder *the_builder)
{
   the_builder->send_bldr(_("\n&+GBook: \t\t&+M%s&*\n"), get_name());
   the_builder->send_bldr(_("&+GTitle: \t\t&+w%s&*\n"), get_title());
   the_builder->send_bldr(_("&+GAltnames: \t&+g%s&*\n"), get_altname());
   the_builder->send_bldr(_("&+GClones: \t&+g%s&*\n"), get_clones());
   the_builder->send_bldr(_("&+GSpecials: \t&+g%s&*\n"), get_special_str());
   the_builder->send_bldr(_("&+GGuards: \t&+g%s&*\n"), get_guards());
   the_builder->send_bldr(_("&+GLocation: \t&+M%s&*\n"), get_location());
   the_builder->send_bldr(_("&+GSize: \t\t&+w%d&*\n"), get_size());
   the_builder->send_bldr(_("&+GWeight:\t\t&+w%d&*\n"), get_weight());
   the_builder->send_bldr(_("&+GBrief0:\n&*%s\n"), get_brief(0));
   the_builder->send_bldr(_("&+GBrief1:\n&*%s\n"), get_brief(1));
   the_builder->send_bldr(_("&+GSubject:&*\t&+M%s &+c(file: %s/%s%s%s)\n"),
                get_book_subject(),the_config.basedir.str_show(),BOOKDIR,
                                          get_book_subject(),BOOKEXTENTION);
   the_builder->send_bldr(_("&+GDesc:&*\n%s\n"), get_desc());

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the book to a player
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Book::describe(Player *the_player)
{
   MudObject *tmp_container;
   Strings   container_name;

   the_player->send_plr(_("\n&+GBook: \t\t&+M%s&*\n"), get_name());
   the_player->send_plr(_("&+GTitle: \t\t&+w%s&*\n"), get_title());
   the_player->send_plr(_("&+GAltnames: \t&+g%s&*\n"), get_altname());
   the_player->send_plr(_("&+GClones: \t&+g%s&*\n"), get_clones());
   the_player->send_plr(_("&+GSpecials: \t&+g%s&*\n"), get_special_str());
   the_player->send_plr(_("&+GGuards: \t&+g%s&*\n"), get_guards());
   the_player->send_plr(_("&+GStartLoc: \t&+M%s&*\n"), get_location());

   tmp_container = get_contained_by();
   if (tmp_container == NULL)
      container_name = _("nowhere");
   else
      container_name.sprintf("%s@%s", tmp_container->get_name(), 
                                         tmp_container->get_area());

   the_player->send_plr(_("&+GCurrentLoc: \t&+M%s&*\n"), 
                                                 container_name.str_show());

   the_player->send_plr(_("&+GSize: \t\t&+w%d&*\n"), get_size());
   the_player->send_plr(_("&+GWeight:\t\t&+w%d&*\n"), get_weight());
   the_player->send_plr(_("&+GBrief0:\n&*%s\n"), get_brief(0));
   the_player->send_plr(_("&+GBrief1:\n&*%s\n"), get_brief(1));
   the_player->send_plr(_("&+GSubject:&*\t&+M%s &+c(file: %s/%s%s%s)\n"),
                get_book_subject(),the_config.basedir.str_show(),BOOKDIR,
                                            get_book_subject(),BOOKEXTENTION);
   the_player->send_plr(_("&+GDesc:&*\n%s\n"), get_desc());

   list_specials(the_player);
   the_player->send_plr(_("&+YSize: \t\t\t&+W%d&*\n"), get_mem_size());

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Book::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->send_bldr(_("You can set the following attributes "
             "on a book.\n"
             "   title, weight, size, altnames, clones, desc, itemflags,\n"
             "   content, brief0, brief1, location, guards, specials,\n"
             "   and subject\n"));
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("title"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("weight"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_weight(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("size"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_size(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("guards"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("altnames"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("clones"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("desc"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("specials"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("itemflags"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("content"),
                               strlen(the_parsed->get_target1())))
   {
      FILE *fp;
      Strings tmp_str;
      Strings tmp_fn;

      if (the_booksubject.str_show() == NULL)
      {  the_builder->send_bldr(_("Set the subject of the book first.\n"));
         return -1;
      }

      if (the_builder->get_long_input(&tmp_str) < 0)
      {
         the_builder->send_bldr(_("Error reading in input, failed!\n"));
         return -1;
      }
     
      tmp_fn.sprintf("%s/%s%s%s", the_config.basedir.str_show(), BOOKDIR, 
                                 the_booksubject.str_show(), BOOKEXTENTION); 
      if ((fp = xfopen(tmp_fn.str_show(),"w", "write_book_file")) == NULL)
      {   the_builder->send_bldr(_("Could not open content file, "
                                                    "changes not saved\n"));
          return -1;
      }

      fprintf(fp,"%s",tmp_str.str_show());
      xfclose(fp, "write_book_file");
      the_builder->send_bldr(_("Content on book saved in %s.\n"),
                                                         tmp_fn.str_show());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("brief0"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 0);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("brief1"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 1);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("location"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_loc(the_parsed, the_builder);
   }

   if (!STRNCASECMP(the_parsed->get_target1(),_("subject"),
                    strlen(the_parsed->get_target1())))
   {
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr(_("You need to specify a subject to set to.\n"));
         return -1;
      }
      if (strlen(the_parsed->get_speech()) > 10 || strchr(the_parsed->get_speech(),' ') != NULL)
      {  the_builder->send_bldr(_("Please limit the subject to 1 word of 10 chars maximum.\n"));
         return -1;
      }

      set_book_subject(the_parsed->get_speech());
      the_builder->send_bldr(_("Subject on %s set to: %s\n"), get_name(),
                                                           get_book_subject());
      set_modified(1);
      return 1;
   }

   the_builder->send_bldr(_("The attribute '%s' is not a moveable attribute.\n"),
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** read_book - shows the contents of the book (paged) to the user.
 **
 ** Parameters: the_user - the user who wants to read the book, either
 **                        a player or a mobile 
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
int Book::read_book(MudObject *the_user)
{   Strings fullname;
    Pager   *the_pager;
    Player  *the_player = NULL;
    Builder *the_builder = NULL;

    switch (the_user->get_type()) {
    case OBJ_TYPE_PLAYER:  the_player = (Player *)the_user; break;
    case OBJ_TYPE_BUILDER: the_builder = (Builder *)the_user; break;
    default:
        return -1;
    }

    fullname.sprintf("%s/%s%s%s",the_config.basedir.str_show(), BOOKDIR, 
                                     the_booksubject.str_show(),BOOKEXTENTION);

    if (the_player != NULL)
       the_pager = new_Pager(fullname.str_show(), 
                                              the_player->get_pager_lines());
    else
       the_pager = new_Pager(fullname.str_show(), 
                                              the_builder->get_pager_lines());

    if (the_pager && the_pager->file_available())
    {
        Inp_Handler *tmp_handler;
        int results;

        if (the_player)
        {
           tmp_handler = the_player->get_input_handler();
           results = the_pager->paged_read(the_player->get_connection());
        }
        else
	{
           tmp_handler = the_builder->get_input_handler();
           results = the_pager->paged_read(the_builder->get_connection());
        }

        if (results)
        {
           delete_Pager(the_pager);
        }
        else
        {
           tmp_handler->push_input_handler(pager_funct, 
             _("&+c<&+G******************&+Bpress return to continue, "
             "q to quit&+G**********************&+c>&*"), HANDLER_DATA_PAGER);
           tmp_handler->set_data((void *) the_pager);
        }
        return 0;
    }
    return -1;
}

/***********************************************************************
 ** get_book_subject - shows the subject of the book (paged) to the user.
 **
 ** Parameters: none
 **
 ** Returns: NULL if no subject available, otherwise a string with the 
 **          subject 
 **
 ***********************************************************************/
char *Book::get_book_subject()
{   return the_booksubject.str_show();
}

/***********************************************************************
 ** set_book_subject - Sets the subject of the book.
 **
 ** Parameters: subject - the subject of the book 
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Book::set_book_subject(char *subject)
{   the_booksubject = subject;
    return 1;
}


/***********************************************************************
 ** copy_object - copies the book to another object of a different name
 **
 ** Parameters: copy_obj - the object we are copying from 
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Book::copy_object(Entity *copy_obj)
{
   Book *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_BOOK)
      return 0;

   copy_from = (Book *) copy_obj;

   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /***** set the item attributes ********/
   copy_item_attrib((Item *) copy_from);

   /****** set the moveable attributes ******/
   copy_moveable_attrib((Moveable *) copy_from);

   /****** set the book attributes *******/
   set_book_subject(copy_from->get_book_subject());

   return 1;
}

/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Book *Book::operator = (Book *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** read_book_attrib - reads in books attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Book::read_book_attrib(FILE *read_file, ErrLog *error_log)
{
   token_record *the_token;

   error_log = NULL;

   /* get the book subject */
   the_token = get_token(read_file, '\0');
   set_book_subject(the_token->the_string);

   return 1;
}

/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Book::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Book::get_mem_size_dynamic()
{
   int  size = 0;

   size += the_booksubject.get_mem_size_dynamic();   
   size += get_mem_size_moveable();
   size += get_mem_size_item();
   size += get_mem_size_mudobj();
   size += get_mem_size_entity();

   return size;
}

#endif



