/**********************************************************************
 ** Mailer class: Maintains the mail for this user, including the mail
 **               file and passing mail to other user's mail
 **
 **
 **
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef MAILER_C
#define MAILER_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "mailer.h"
#include "pager.h"
#include "global.h"
#include "utils.h"
#include "lexer.h"
#include "gameflags.h"
#include "newfuncts.h"
#include "memchk.h"
#include "inp_handler.h"

/***********************************************************************
 ** Mailer (constructor) - Creates the mailer and fills it with info
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Mailer::Mailer(char *the_username, load_type the_mode)
{
   char user_dir;

   newmail = 0;

   username = the_username;
   username.lowercase();
   user_dir = tolower(*(username.str_show()));

   mailfilename.sprintf("%s/%s/%c/%s.mail", the_config.basedir.str_show(), 
                              USERDATADIR, user_dir, username.str_show()); 

   loaded_mail = NULL;
   load_mail(the_mode);

   current_ltr = NULL;

}


/***********************************************************************
 ** ~Mailer (destructor) - closes the mailer
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Mailer::~Mailer()
{ 
   delete_mail();
}


/***********************************************************************
 ** load_mail - loads in the mail of the user to memory
 **
 ** Parameters: the_mode - the mode to load (all mail or just the header) 
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

int Mailer::load_mail(load_type the_mode)
{
   Strings      holder;
   token_record *the_token;
   Letter       *new_ltr;
   Letter       *prev_ltr = NULL;

   if (load_mode == the_mode)
      return 0;

   if (the_mode == None)
   {
      delete_mail();
      load_mode = None;
      return 1;
   }

   /* if the user's mail account does not exist, we create info */
   if ((the_mailfile = xfopen(mailfilename.str_show(), "r", 
                                                   "mail_file")) == NULL)
   {
      /* create header information */
 
      return 0;
   }

   /* read in the header information */

   /* Get newmail attribute */
   the_token = get_token(the_mailfile, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.
         sprintf("Invalid format in mailer file for player '%s'", 
                                                     username.str_show());

      mainstruct->log_error(holder.str_show(), "load_mail");
      return -1;
   }
   newmail = atoi(the_token->the_string);

   load_mode = the_mode;

   the_token = get_token(the_mailfile, '\0');
   while ((the_token->token_type > 0) && 
          (the_token->token_type != T_POUND))
   {
      new_ltr = new_Letter();
      if (new_ltr->read_letter(the_mailfile) == -1)
      {
         delete_Letter(new_ltr);
         the_token = get_token(the_mailfile, '\0');
         continue;
      }

      if (prev_ltr == NULL)
      {
         loaded_mail = new_ltr;
      }
      else
      {
         prev_ltr->set_next_ltr(new_ltr);
      }
      prev_ltr = new_ltr;
      the_token = get_token(the_mailfile, '\0');
   }

   if (load_mode != Full)
   {
      xfclose(the_mailfile, "mail_file");
      return 0;
   }
   
   /* read in the mail messages */

   xfclose(the_mailfile, "mail_file");
   return 1;
}

/***********************************************************************
 ** delete_mail - deletes all the mail in the linked list
 **
 ** Parameters: None
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

int Mailer::delete_mail()
{
   Letter *tmp_ltr;

   tmp_ltr = loaded_mail;
   while (loaded_mail != NULL)
   {
      tmp_ltr = loaded_mail;
      loaded_mail = loaded_mail->get_next_ltr();
      delete_Letter(tmp_ltr);
   }
   return 1;
}



/***********************************************************************
 ** write_mail - writes the mail of the user to disk
 **
 ** Parameters: None
 **
 ** Returns: 1 if success, -1 if error
 **
 ***********************************************************************/

int Mailer::write_mail()
{
   Strings      holder;
   Letter       *tmp_ltr;

   if (load_mode != Full)
   {
      load_mail(Full);
   }

   /* open the mail file for writing */
   if ((the_mailfile = xfopen(mailfilename.str_show(), "w", 
                                                     "mail_file")) == NULL)
   {
      /* create header information */
      holder.sprintf("Could not open user '%s' mailfile.", 
                                                      username.str_show());
      mainstruct->log_error(holder.str_show(), "write_mail");
      return 0;
   }

   /* write the header information */
   fprintf(the_mailfile, "%d\n", newmail);

   fprintf(the_mailfile, "\n");

   /* write the mail messages */
   tmp_ltr = loaded_mail;
   while (tmp_ltr != NULL)
   {
      tmp_ltr->write_letter(the_mailfile);
      tmp_ltr = tmp_ltr->get_next_ltr();
   }
   fprintf(the_mailfile, "#\n");
   
   xfclose(the_mailfile, "mail_file");
   return 1;
}


/***********************************************************************
 ** display_help - displays help for the mailer to the player
 **
 ** Parameters: the_player - who to send to
 **
 ***********************************************************************/

void Mailer::display_help(Player *the_player)
{
   the_player->send_plr("\n&+WCommands for Mailer:&*\n");
   the_player->send_plr("&+B---------------------------&*\n");
   the_player->send_plr("&+Gquit &+W- &*quit from the mailer\n");
   the_player->send_plr("&+Ghelp &+W- &*this help display\n");
   the_player->send_plr("&+Glist &+W- &*list all mail messages\n");
   the_player->send_plr("&+Gread &+W- &*read a message\n");
   the_player->send_plr("&+Gmail &+W- &*mail a message one or more players\n");
   the_player->send_plr("&+Greply &+W- &*reply to sender or all in message\n");
   the_player->send_plr("&+Gforward &+W- &*forward message to someone\n");
   the_player->send_plr("&+Gdelete &+W- &*delete a message\n");

   the_player->send_plr("\n");
}


/***********************************************************************
 ** get_current_ltr - passes a pointer to the current letter
 **
 ** Parameters: the_player - who to send to
 **
 ***********************************************************************/

Letter *Mailer::get_current_ltr()
{
   return current_ltr;
}


/***********************************************************************
 ** set_current_ltr - passes a pointer to the current letter
 **
 ** Parameters: new_ltr - the new letter to assign this to
 **
 ***********************************************************************/

void Mailer::set_current_ltr(Letter *new_ltr)
{
   current_ltr = new_ltr;
}


/***********************************************************************
 ** remove_current_ltr - removes the current letter and deletes if necessary
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Mailer::remove_current_ltr()
{
   if (current_ltr == NULL)
      return 0;

   if (current_ltr->get_status() == Composing)
   {
      delete_Letter(current_ltr);
   }
   current_ltr = NULL;
   return 1;
}


/***********************************************************************
 ** add_mail - adds mail to the player's mail
 **
 ** Parameters: the_ltr - the letter to add to the mail
 **
 ***********************************************************************/

int Mailer::add_mail(Letter *the_ltr)
{
   Letter *tmp_ltr;
   int    counter = 2;

   if (the_ltr == NULL)
      return 0;

   tmp_ltr = loaded_mail;

   if (tmp_ltr == NULL)
   {
      loaded_mail = the_ltr;
      return 1;
   }

   while (tmp_ltr->get_next_ltr() != NULL)
   {
      counter++;
      tmp_ltr = tmp_ltr->get_next_ltr();
   }
   
   tmp_ltr->set_next_ltr(the_ltr);

   return counter;
}


/***********************************************************************
 ** delete_mail - deletes mail from the player's mail
 **
 ** Parameters: the_num - the number of the mail to delete
 **
 ** Returns: 1 for deleted, 0 for not found, -1 for failure
 **
 ***********************************************************************/

int Mailer::delete_mail(int the_num)
{
   Letter *tmp_ltr;
   Letter *prev_ltr = NULL;
   int    counter = 1;

   if (the_num <= 0)
      return -1;

   tmp_ltr = loaded_mail;
   while ((tmp_ltr != NULL) && (counter < the_num))
   {
      prev_ltr = tmp_ltr;
      tmp_ltr = tmp_ltr->get_next_ltr();
      counter++;
   }

   if (tmp_ltr == NULL)
      return 0;

   if (prev_ltr == NULL)
   {
      loaded_mail = loaded_mail->get_next_ltr();
   }
   else
   {
      prev_ltr->set_next_ltr(tmp_ltr->get_next_ltr());
   }

   delete_Letter(tmp_ltr);

   return 1;
}


/***********************************************************************
 ** get_load_mode - returns the mode the mailer is set to
 **
 ** Parameters: the_ltr - the letter to add to the mail
 **
 ***********************************************************************/

load_type Mailer::get_load_mode()
{
   return load_mode;
}


/***********************************************************************
 ** list_mail - lists the current mail in the user's mailbox
 **
 ** Parameters: the_player - who to send to
 **
 ***********************************************************************/

int Mailer::list_mail(Player *the_player)
{
   Letter *the_ltr;
   int    counter = 1;
   Strings holder;
   char *the_year;
   Strings the_date;
   char   *color;

   the_player->send_plr("&+BCurrent Mail:&*\n\n");
   the_player->send_plr(
        "&+W No.   From           Date Sent                 Subject&*\n");
   the_player->send_plr(
        "&+B-----------------------------------------------------------&*\n");

   the_ltr = loaded_mail;
   while (the_ltr != NULL)
   {
      if (the_ltr->get_status() == NotRead)
         color = "&+C";
      else
         color = "&+c";

      the_year = the_ltr->get_date();
      the_year += 20;
      the_date = the_ltr->get_date();
      the_date.truncate(12);
      the_date.str_cat(the_year);
      the_player->send_plr("&+g[&*%3.3d&+g]  %s%-14s %s  %s&*\n", counter, 
                             color, the_ltr->get_from(), 
                             the_date.str_show(), the_ltr->get_subject());
      counter++;
      the_ltr = the_ltr->get_next_ltr();
   }
   the_player->send_plr("\n");
   return counter;
}


/***********************************************************************
 ** find_mail - finds a particular number of mail and sets current_ltr
 **             to it
 **
 ** Parameters: the_num - the number of mail we find down the list
 **
 ***********************************************************************/

int Mailer::find_mail(int the_num)
{
   int    counter = 1;
   Letter *tmp_ltr;

   if (the_num <= 0)
   {
      current_ltr = NULL;
      return -1;
   }

   tmp_ltr = loaded_mail;
   while ((tmp_ltr != NULL) && (counter != the_num))
   {
      tmp_ltr = tmp_ltr->get_next_ltr();
      counter++;
   }

   current_ltr = tmp_ltr;
   return 1;
}


/***********************************************************************
 ** has_newmail - does the player have new mail
 **
 ** Returns: 1 for new mail, 0 for none
 **
 ***********************************************************************/

int Mailer::has_newmail()
{
   return newmail;
}


/***********************************************************************
 ** clr_newmail - clears the new mail value
 **
 ** Parameters: None
 **
 ***********************************************************************/

void Mailer::clr_newmail()
{
   newmail = 0;
}


/***********************************************************************
 ** set_newmail - sets the new mail value
 **
 ** Parameters: None
 **
 ***********************************************************************/

void Mailer::set_newmail()
{
   newmail = 1;
}


/***********************************************************************
 **               Letter methods
 ***********************************************************************/


/***********************************************************************
 ** Letter (constructor) - Creates the letter
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Letter::Letter()
{
   status = Composing;
   next_letter = NULL;
}


/***********************************************************************
 ** ~Letter (destructor) - deletes the letter
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Letter::~Letter()
{ 
}


/***********************************************************************
 ** set_to - sets names in the to field of this letter
 **
 ** Parameters: the_str - the string to set the to field to
 **
 ***********************************************************************/

void Letter::set_to(char *the_str)
{
   to = the_str;
}


/***********************************************************************
 ** set_from - sets a name in the from field
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

void Letter::set_from(char *the_str)
{
   from = the_str;
}


/***********************************************************************
 ** get_date - gets the date string
 **
 ***********************************************************************/

char *Letter::get_date()
{
   return date.str_show();
}


/***********************************************************************
 ** get_to - gets the to string
 **
 ***********************************************************************/

char *Letter::get_to()
{
   return to.str_show();
}


/***********************************************************************
 ** get_from - gets the from string
 **
 ***********************************************************************/

char *Letter::get_from()
{
   return from.str_show();
}


/***********************************************************************
 ** get_subject - gets the subject string
 **
 ***********************************************************************/

char *Letter::get_subject()
{
   return subject.str_show();
}


/***********************************************************************
 ** get_body - gets the body string
 **
 ***********************************************************************/

char *Letter::get_body()
{
   return body.str_show();
}

/***********************************************************************
 ** set_subject - sets a subject for this letter
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

void Letter::set_subject(char *the_str)
{
   subject = the_str;
}


/***********************************************************************
 ** set_body - sets the body for this letter
 **
 ** Parameters: the_str - the string to set the field to
 **
 ***********************************************************************/

void Letter::set_body(char *the_str)
{
   body = the_str;
}


/***********************************************************************
 ** edit_body - edits the body of the mail message
 **
 ** Parameters: the_player - the player to start editing with
 **
 ***********************************************************************/

int Letter::edit_body(Player *the_player, int is_reply)
{
   Editor      *input_editor;
   Flags       *tmp_gameflags;
   Inp_Handler *tmp_handler;
   int         results;

   tmp_handler = the_player->get_input_handler();

   input_editor = new Editor();

   tmp_gameflags = the_player->get_gameflags();

   if (!tmp_gameflags->get_flag(GAMEFLAG_FULLEDIT))
      input_editor->set_simple_mode();

   tmp_handler->pop_input_handler();

   the_player->send_plr("&+gMessage body:&*\n");
   results = input_editor->start_editor(the_player, &body);

   tmp_handler = the_player->get_input_handler();

   if (is_reply)
      tmp_handler->add_pop_function(reply_on_pop, the_player->get_name(), 
                                               body.str_show(), NULL, NULL);
   else
      tmp_handler->add_pop_function(mail_on_pop, the_player->get_name(), 
                                                          NULL, NULL, NULL);

   return results;
}


/***********************************************************************
 ** get_status - gets the status of the letter
 **
 ** Returns: the current status of the letter
 **
 ***********************************************************************/

letter_status Letter::get_status()
{
   return status;
}


/***********************************************************************
 ** deliver_letter - sends the letter to the applicable players
 **
 ** Parameters: the_player - the player who is sending the letter
 **
 ** Returns: 1 for success
 **
 ***********************************************************************/

int Letter::deliver_letter(Player *the_player)
{
   Strings    to_who;
   Player     *tmp_player;
   Mailer     *target_mailer;
   User_Dbase *tmp_dbase;
   int        which = 1;
   int        is_on = 0;
   Strings    filename;
   Strings    username;
   char       user_dir;
   FILE       *the_file = NULL;
   int        an_error = 0;
   Letter     *new_ltr;

   status = Delivering;
   to_who.assign_word(to.str_show(), which);
   date = get_time_str();
   while (to_who.str_show() != NULL)
   {
      /* get the mailer...if they are not on, load it */
      if ((tmp_player = mainstruct->get_player(to_who.str_show())) != NULL)
      {
         is_on = 1;
         target_mailer = tmp_player->get_mailer();
      }
      else
      {
         is_on = 0;

	 /* see if the user exists before we open a mailer for them */
         tmp_dbase = mainstruct->get_user_database();

         user_dir = (*(to_who.str_show()));
         user_dir = (char) tolower(user_dir);   

         to_who.lowercase();

         filename.sprintf("%s/%s/%c/%s%s", the_config.basedir.str_show(),
                     USERDATADIR, user_dir, to_who.str_show(), USEREXTENTION);

         if ((the_file = xfopen(filename.str_show(), "r", 
                                            "hunt_mail_opening")) != NULL)
         {
            xfclose(the_file, "hunt_mail_opening");
         }
         else
	 {
            the_player->send_plr(
             "Could not send to player '%s', does not exist.\n", 
                                                          to_who.str_show());
            an_error = 1;
            which++;
            to_who.assign_word(to.str_show(), which);
            continue;
         }

         target_mailer = new_Mailer(to_who.str_show(), Full);
      }

      if (target_mailer->get_load_mode() != Full)
      {
         target_mailer->load_mail(Full);
      }

      new_ltr = new_Letter();
      new_ltr->copy_ltr(this);
      new_ltr->set_status(NotRead);
      target_mailer->add_mail(new_ltr);

      if (!is_on)
         target_mailer->set_newmail();

      target_mailer->write_mail();

      if (!is_on)
      {
         delete_Mailer(target_mailer);
      }
      else
      {
         tmp_player->send_plr("You have received new mail from %s.\n", 
                                                     the_player->get_title());
      }
      which++;
      to_who.assign_word(to.str_show(), which);
   }   

   if (!an_error)
   {
      the_player->
           send_plr("Mail delivered successfully to all recipients.\n");
   }
   else
   {
      the_player->send_plr(
            "Error encountered delivering to one or more recepients.\n");
   }
   return 1;
}


/***********************************************************************
 ** get_next_ltr - gets the next letter attached to this one
 **
 **
 ***********************************************************************/

Letter *Letter::get_next_ltr()
{
   return next_letter;
}


/***********************************************************************
 ** set_next_ltr - sets the next letter attached to this one
 **
 ** Parameters: new_ltr - the letter to add as next
 **
 ***********************************************************************/

void Letter::set_next_ltr(Letter *new_ltr)
{
   next_letter = new_ltr;
}


/***********************************************************************
 ** set_status - sets the status of the letter to something else
 **
 ** Parameters: new_status - the new status of the letter
 **
 ***********************************************************************/

void Letter::set_status(letter_status new_status)
{
   status = new_status;
}


/***********************************************************************
 ** copy_ltr - copies the letter to another letter
 **
 ** Parameters: new_ltr - the letter to add as next
 **
 ***********************************************************************/

void Letter::copy_ltr(Letter *copy_from)
{
   date = copy_from->get_date();
   to = copy_from->get_to();
   from = copy_from->get_from();
   subject = copy_from->get_subject();
   body = copy_from->get_body();
   status = copy_from->get_status();
}


/***********************************************************************
 ** write_letter - writes the letter to a file
 **
 ** Parameters: the_file - the file to write the letter to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Letter::write_letter(FILE *the_file)
{
   fprintf(the_file, "+\n");
   fprintf(the_file, "^%s^\n", date.str_show());
   fprintf(the_file, "^%s^\n", to.str_show());
   fprintf(the_file, "^%s^\n", from.str_show());
   fprintf(the_file, "^%s^\n", subject.str_show());
   fprintf(the_file, "^%s^\n", body.str_show()); 
   fprintf(the_file, "%d\n\n", (int) status);
   return 1;
}


/***********************************************************************
 ** read_letter - reads the letter from a file
 **
 ** Parameters: the_file - the file to read the letter from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Letter::read_letter(FILE *the_file)
{
   token_record *the_token;
   Strings      holder;
   char         *temp_body;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      mainstruct->log_error("Invalid format in mail file", "read_letter");
      return -1;
   }

   /* get the date of this letter */
   the_token = get_token(the_file, '^');
   date = the_token->the_string;   


   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      mainstruct->log_error("Invalid format in mail file", "read_letter");
      return -1;
   }

   /* get the to of this letter */
   the_token = get_token(the_file, '^');
   to = the_token->the_string;   


   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      mainstruct->log_error("Invalid format in mail file", "read_letter");
      return -1;
   }

   /* get the from of this letter */
   the_token = get_token(the_file, '^');
   from = the_token->the_string;   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      mainstruct->log_error("Invalid format in mail file", "read_letter");
      return -1;
   }

   /* get the date of this letter */
   the_token = get_token(the_file, '^');
   subject = the_token->the_string;   


   temp_body = read_desc_type(the_file, mainstruct->get_log(), NULL);
   
   if (temp_body == NULL)
      return -1;
   body = temp_body;
   delete temp_body;


   /* Get status attribute */
   the_token = get_token(the_file, '\0');
     
   if (the_token->token_type != T_NUMERICAL)
   {
      mainstruct->log_error("Invalid format in mail file", "read_letter");
 
      return -1;
   }
   status = (letter_status) atoi(the_token->the_string);

   return 1;
}


/***********************************************************************
 ** display_letter - displays the letter to the screen
 **
 ** Parameters: the_player - who to send to
 **
 ***********************************************************************/

void Letter::display_letter(Player *the_player)
{
   the_player->send_plr("\n&+gTo:&* %s\n", to.str_show());
   the_player->send_plr("&+gFrom:&* %s\n", from.str_show());
   the_player->send_plr("&+gDate:&* %s\n", date.str_show());
   the_player->send_plr("&+gSubject:&* %s\n", subject.str_show());
   the_player->send_plr("&+gBody:&* \n%s\n", body.str_show());
   the_player->send_plr("\n");
}


/***********************************************************************
 ** start_single_reply - starts a reply going to only the sender
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Letter::start_single_reply(Player *the_player)
{
   Strings holder;
   Letter  *new_ltr;
   Mailer  *the_mailer;

   the_mailer = the_player->get_mailer();
   new_ltr = new_Letter();
   new_ltr->set_to(from.str_show());
   new_ltr->set_from(the_player->get_name());
   holder = "Re: ";
   holder.str_cat(subject.str_show());
   new_ltr->set_subject(holder.str_show());

   the_player->send_plr("\n&+gTo: &*%s\n", new_ltr->get_to());
   the_player->send_plr("&+gFrom: &*%s\n", new_ltr->get_from());
   the_player->send_plr("&+gSubject: &*%s\n", new_ltr->get_subject());

   the_mailer->set_current_ltr(new_ltr);
   return new_ltr->edit_body(the_player, 1);
}


/***********************************************************************
 ** start_multiple_reply - starts a reply going to all names in the from
 **                        and to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Letter::start_multiple_reply(Player *the_player)
{
   Letter  *new_ltr;
   Strings holder;
   Strings to_str;
   Strings tmp_word;
   int     counter = 1;
   Mailer  *the_mailer;

   the_mailer = the_player->get_mailer();

   new_ltr = new_Letter();
   to_str = from.str_show();
   while (tmp_word.assign_word(to.str_show(), counter) > 0)
   {
      if (STRCASECMP(tmp_word.str_show(), the_player->get_name()))
      {
         to_str.str_cat(" ");
         to_str.str_cat(tmp_word.str_show());
      }
      counter++;
   }

   new_ltr->set_to(to_str.str_show());
   new_ltr->set_from(the_player->get_name());
   holder = "Re: ";
   holder.str_cat(subject.str_show());
   new_ltr->set_subject(holder.str_show());

   the_player->send_plr("\n&+gTo: &*%s\n", new_ltr->get_to());
   the_player->send_plr("&+gFrom: &*%s\n", new_ltr->get_from());
   the_player->send_plr("&+gSubject: &*%s\n", new_ltr->get_subject());

   the_mailer->set_current_ltr(new_ltr);
   return new_ltr->edit_body(the_player, 1);
}


/***********************************************************************
 ** start_forward - starts a forward going to the sender
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Letter::start_forward(Player *the_player, char *to_who)
{
   Strings holder;
   Letter  *new_ltr;
   Mailer  *the_mailer;

   the_mailer = the_player->get_mailer();

   new_ltr = new_Letter();
   new_ltr->set_to(to_who);
   new_ltr->set_from(the_player->get_name());
   holder = "Fwd: ";
   holder.str_cat(subject.str_show());
   new_ltr->set_subject(holder.str_show());

   the_player->send_plr("\n&+gTo: &*%s\n", new_ltr->get_to());
   the_player->send_plr("&+gFrom: &*%s\n", new_ltr->get_from());
   the_player->send_plr("&+gSubject: &*%s\n", new_ltr->get_subject());

   the_mailer->set_current_ltr(new_ltr);
   return new_ltr->edit_body(the_player, 1);
}



/**************************************************************************
 **    Mailer input handlers
 **************************************************************************/

/**************************************************************************
 ** mailer_command_handler - Main input handler for the mailer
 **
 ** This input handler is the one the mailer should start in.. It handles
 ** the input of the mailer when this is in command mode. 
 **
 ** Parameters - the_obj : the player using the mailer
 **              the_input : the command
 **
 ** returns 1 on success, -1 on error
 ** 
 **************************************************************************/

int mailer_command_handler(MudObject *the_obj, char *the_input)
{
   Player      *the_player;
   Inp_Handler *tmp_handler;
   Strings     the_inpstr;
   Strings     the_cmd;
   Mailer      *the_mailer;
 
   the_player = (Player *) the_obj;

   tmp_handler = the_player->get_input_handler();
           
   the_mailer = (Mailer *)tmp_handler->get_data();

   /* get the input that was passed in and remove the newline */
   the_inpstr = the_input;
   the_inpstr.remove_newline();


   /* get the first word of the input, it is all we care about */
   the_cmd.assign_word(the_inpstr.str_show(),1);

   if (!STRNCASECMP(the_cmd.str_show(), "quit", the_cmd.str_len()))
   {
      the_player->send_plr("&+B[&+WQuitting mailer&+b]&*\n");
      the_mailer->write_mail();
      tmp_handler->pop_input_handler();
   }
   else if ((!STRNCASECMP(the_cmd.str_show(), "help", the_cmd.str_len())) ||
            ((*(the_cmd.str_show())) == '?'))
   {
      the_mailer->display_help(the_player);
   }
   else if ((!STRNCASECMP(the_cmd.str_show(), "list", the_cmd.str_len())) ||
            ((*(the_cmd.str_show())) == '?'))
   {
      the_mailer->list_mail(the_player);
   }
   else if ((!STRNCASECMP(the_cmd.str_show(), "mail", the_cmd.str_len())) ||
            ((*(the_cmd.str_show())) == '?'))
   {
      Letter *new_ltr;
      Strings tmp_to;
      Strings *prompt;

      tmp_to.assign_phrase(the_inpstr.str_show(), 1);

      if (tmp_to.str_show() == NULL)
      {
         the_player->send_plr("You need to specify players to mail to.\n");
         return 1;
      }
      new_ltr = new_Letter();
      new_ltr->set_to(tmp_to.str_show());
      new_ltr->set_from(the_player->get_title());
      the_player->send_plr("\n&+gTo: &*%s\n", tmp_to.str_show());
      the_player->send_plr("&+gFrom: &*%s\n", the_player->get_title());
      the_mailer->set_current_ltr(new_ltr);

      prompt = new_Strings("&+gSubject:&* ");

      /* get the subject from the user */
      tmp_handler->push_input_handler(mailer_get_subject, 
                                  prompt, 1, HANDLER_DATA_MAILER, 0);
      tmp_handler->set_data(the_player->get_mailer());

   }

   else if (!STRNCASECMP(the_cmd.str_show(), "read", the_cmd.str_len()))
   {
      Letter *cur_ltr;
      Strings the_num;

      the_num.assign_word(the_inpstr.str_show(), 2);

      if (the_num.str_show() == NULL)
      {
         if ((cur_ltr = the_mailer->get_current_ltr()) == NULL)
            the_player->send_plr(
                         "You need to specify a message number to read.\n");
         else
	 {
            cur_ltr->set_status(Read);
            cur_ltr->display_letter(the_player);
	 }
         return 1;
      }

      if (!isdigit(*(the_num.str_show())))
      {
         the_player->send_plr("Format: read <messagenumber>.\n");
         return 0;
      }

      the_mailer->find_mail(atoi(the_num.str_show()));
      cur_ltr = the_mailer->get_current_ltr();

      if (cur_ltr == NULL)
      {
         the_player->send_plr("That letter number does not exist.\n");
         return 0;
      }

      cur_ltr->set_status(Read);
      cur_ltr->display_letter(the_player);
   }

   else if (!STRNCASECMP(the_cmd.str_show(), "reply", the_cmd.str_len()))
   {
      Letter *cur_ltr = NULL;
      Strings the_num;
      int     use_all = 0;

      the_num.assign_word(the_inpstr.str_show(), 2);

      if (the_num.str_show() == NULL)
      {
         if ((cur_ltr = the_mailer->get_current_ltr()) == NULL)
            the_player->send_plr(
                      "You need to specify a message number to reply to.\n");
         else
	 {
            cur_ltr->start_single_reply(the_player);
	 }
         return 1;
      }

      if (!STRCASECMP(the_num.str_show(), "all"))
      {
         use_all = 1;
         the_num.assign_word(the_inpstr.str_show(), 3);
      }

      if (!isdigit(*(the_num.str_show())))
      {
         if ((cur_ltr = the_mailer->get_current_ltr()) == NULL)
	 {
            the_player->send_plr("Format: reply [all] <messagenumber>.\n");
            return 0;
	 }
      }

      if (cur_ltr == NULL)
      {
         the_mailer->find_mail(atoi(the_num.str_show()));
         cur_ltr = the_mailer->get_current_ltr();
      }

      if (cur_ltr == NULL)
      {
         the_player->send_plr("That letter number does not exist.\n");
         return 0;
      }

      if (use_all)
         cur_ltr->start_multiple_reply(the_player);
      else
         cur_ltr->start_single_reply(the_player);
   }

   else if (!STRNCASECMP(the_cmd.str_show(), "forward", the_cmd.str_len()))
   {
      Letter *cur_ltr = NULL;
      Strings the_num;
      Strings to_who;
      int     offset = 1;

      the_num.assign_word(the_inpstr.str_show(), offset);

      if (the_num.str_show() == NULL)
      {
         the_player->send_plr(
                    "Format: forward <messagenumber> <players>.\n");
         return 0;
      }

      if (!isdigit(*(the_num.str_show())))
      {
         if ((cur_ltr = the_mailer->get_current_ltr()) == NULL)
	 {
            the_player->send_plr(
                    "Format: forward <messagenumber> <players>.\n");
            return 0;
	 }
      }
      else
         offset++;

      if (cur_ltr == NULL)
      {
         the_mailer->find_mail(atoi(the_num.str_show()));
         cur_ltr = the_mailer->get_current_ltr();
      }

      if (cur_ltr == NULL)
      {
         the_player->send_plr("That letter number does not exist.\n");
         return 0;
      }

      to_who.assign_phrase(the_inpstr.str_show(), offset);

      if (to_who.str_show() == NULL)
      {
         the_player->send_plr("You need to indicate who to forward to.\n");
         return 0;
      }

      cur_ltr->start_forward(the_player, to_who.str_show());
   }

   else if (!STRNCASECMP(the_cmd.str_show(), "delete", the_cmd.str_len()))
   {
      Strings the_num;

      the_num.assign_word(the_inpstr.str_show(), 2);

      if (the_num.str_show() == NULL)
      {
         the_player->send_plr(
                    "Format: delete <messagenumber>.\n");
         return 0;
      }

      if (!isdigit(*(the_num.str_show())))
      {
         the_player->send_plr(
                    "Format: delete <messagenumber>.\n");
         return 0;

      }

      if (the_mailer->delete_mail(atoi(the_num.str_show())) == 0)
      {
         the_player->send_plr("That letter number does not exist.\n");
         return 0;
      }

      the_player->send_plr("Message number &+W%s&* deleted.\n",
                                                        the_num.str_show());
   }

   else
   {
      the_player->send_plr("Unknown command.\n");
      return -1;
   }
   return 1;
}


/**************************************************************************
 ** mailer_get_subject - the input handler to get the subject from the user
 **
 **
 ** Parameters - the_obj : the player using the mailer
 **              the_input : the command
 **
 ** returns 1 on success, -1 on error
 ** 
 **************************************************************************/

int mailer_get_subject(MudObject *the_obj, char *the_input)
{
   Player      *the_player;
   Inp_Handler *tmp_handler;
   Strings     the_inpstr;
   Strings     the_cmd;
   Mailer      *the_mailer;
   Letter      *the_ltr;
 
   the_player = (Player *) the_obj;

   tmp_handler = the_player->get_input_handler();
           
   the_mailer = (Mailer *)tmp_handler->get_data();

   /* get the input that was passed in and remove the newline */
   the_inpstr = the_input;
   the_inpstr.remove_newline();

   if ((the_ltr = the_mailer->get_current_ltr()) == NULL)
   {
      mainstruct->log_error("Potentially fatal error, in subject input "
            "handler with no letter set current", "mailer_command_handler");
      return -1;
   }
  
   the_ltr->set_subject(the_inpstr.str_show());

   return the_ltr->edit_body(the_player, 0);
}


/***********************************************************************
 ** mail_on_pop - when the letter has been finished, deliver it!
 **
 ** Parameters: player_name - the name of the player with the letter
 **             notused1 - should be set to null
 **             notused2 - should be set to null
 **             notused3 - should be set to null
 **             
 **
 ** Returns:  1 if success
 **          -1 if error
 **
 ***********************************************************************/

int mail_on_pop(Inp_Handler *the_handler, char *player_name, char *notused1, 
                                    char *notused2, Strings *notused3)
{
   Player   *the_player;
   Mailer   *the_mailer;
   Letter   *the_letter;

   if (player_name == NULL)
      return -1;

   if ((the_player = mainstruct->get_player(player_name)) == NULL)
      return -1;

   the_mailer = the_player->get_mailer();
   
   if ((the_letter = the_mailer->get_current_ltr()) == NULL)
      return -1;

   the_letter->deliver_letter(the_player);
   the_mailer->remove_current_ltr();


   /* get rid of compile errors */
   notused1 = notused2 = NULL;
   notused3 = NULL;
   the_handler = NULL;

   return 1;
}


/***********************************************************************
 ** reply_on_pop - when the letter has been finished, deliver it!
 **                different from mail_on_pop in that it includes the
 **                original message at the bottom
 **
 ** Parameters: player_name - the name of the player with the letter
 **             old_body - the old body of this message
 **             notused2 - should be set to null
 **             notused3 - should be set to null
 **             
 **
 ** Returns:  1 if success
 **          -1 if error
 **
 ***********************************************************************/

int reply_on_pop(Inp_Handler *the_handler, char *player_name, char *old_body, 
                 char *notused2, Strings *notused3)
{
   Player   *the_player;
   Mailer   *the_mailer;
   Letter   *the_letter;
   Strings  tmp_body;

   if (player_name == NULL)
      return -1;

   if ((the_player = mainstruct->get_player(player_name)) == NULL)
      return -1;

   the_mailer = the_player->get_mailer();
   
   if ((the_letter = the_mailer->get_current_ltr()) == NULL)
      return -1;

   tmp_body = the_letter->get_body();
   tmp_body.str_cat("\n\n-----------Original message-------------\n\n");
   tmp_body.str_cat(old_body);

   the_letter->deliver_letter(the_player);
   the_mailer->remove_current_ltr();

   /* get rid of compile errors */
   notused2 = NULL;
   notused3 = NULL;
   the_handler = NULL;

   return 1;
}


#endif








