/**********************************************************************
 ** shop class: this class handles a shop that is attached to a mobile. 
 **             It maintains data and provides methods that allows for
 **             buying and selling by players
 **
 **   
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef SHOP_C
#define SHOP_C

#include "config.h"
#include "sysdep.h"
#include "mudtypes.h"
#include "strings.h"
#include "shop.h"
#include "money.h"
#include "global.h"
#include "location.h"
#include "mobile.h"
#include "merger.h"
#include "location.h"
#include "code.h"
#include "objtype.h"
#include "newfuncts.h"

/***********************************************************************
 ** Shop (constructor) - creates the shop object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Shop::Shop()
{
   buy_list = NULL;
   request_list = NULL;
}


/***********************************************************************
 ** Shop (destructor) - destroys this shop object
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
Shop::~Shop()
{
   shop_item *tmp_list;
   shop_item *del_ptr;

   tmp_list = buy_list;
   while (tmp_list != NULL)
   {
      del_ptr = tmp_list;
      tmp_list = tmp_list->next_item;
      delete_shop_item(del_ptr);
   }
}


/***********************************************************************
 ** get_name - gets the name of the shop
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the name
 **
 ***********************************************************************/

char *Shop::get_name()
{
   return name.str_show();
}


/***********************************************************************
 ** set_name - sets the name of this shop
 **
 ** Parameters: the_name - the name of this shop
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Shop::set_name(char *the_name)
{
   if (the_name == NULL)
      return -1;

   name = the_name;
   return 1;
}


/***********************************************************************
 ** get_currency - gets the name of the currency this shop used
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the currency name
 **
 ***********************************************************************/

char *Shop::get_currency()
{
   return currency.str_show();
}


/***********************************************************************
 ** set_currency - sets the currency string for the name of the currency 
 **                they use
 **
 ** Parameters: the_name - the name they assign to this
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Shop::set_currency(char *the_name)
{
   if (the_name == NULL)
      return -1;

   currency = the_name;
   return 1;
}


/***********************************************************************
 ** add_shop_item - adds a shop item to the list of items the shop will
 **                 sell
 **
 ** Parameters: the_name - the mudobject name of this object
 **             alias - the alias name of this store item, it is the name
 **                     that the store will use to identify this item
 **             value - the value of this item, in relation to the global
 **                     price index
 **
 ** Returns: 1 for success, -1 for failure, -2 for alias already used
 **
 ***********************************************************************/

int Shop::add_shop_item(char *the_name, char *alias, int value)
{
   shop_item *tmp_list;
   shop_item *new_item;

   new_item = new_shop_item();
   new_item->itemname = the_name;
   new_item->alias = alias;
   new_item->value = value;
   new_item->the_type = Dual;
   new_item->num_of = 0;
   new_item->description = "No Desc.\n";
   new_item->next_item = NULL;

   if ((tmp_list = buy_list) == NULL)
   {
      buy_list = new_item;
      return 1;
   }

   if (new_item->alias.str_cmp(tmp_list->alias.str_show()))
   {
      delete_shop_item(new_item);
      return -2;
   }
   while (tmp_list->next_item != NULL)
   {
      tmp_list = tmp_list->next_item;
      if (new_item->alias.str_cmp(tmp_list->alias.str_show()))
      {
         delete_shop_item(new_item);
         return -2;
      }
   }

   tmp_list->next_item = new_item;
   return 1;
}


/***********************************************************************
 ** write_shop - writes this shop to a specified file
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Shop::write_shop(FILE *the_file)
{
   shop_item *tmp_item;

   fprintf(the_file, "^%s^\n", (get_name() == NULL) ? "NoName" : get_name());
   fprintf(the_file, "%s\n", (get_currency() == NULL) ? "None" : 
                                                           get_currency());
   tmp_item = buy_list;

   while (tmp_item != NULL)
   {
      fprintf(the_file, "+\n%s\n", (tmp_item->itemname.str_show() == NULL) ? 
                                "None" : tmp_item->itemname.str_show());
      fprintf(the_file, "%s\n", (tmp_item->alias.str_show() == NULL) ? 
                                "None" : tmp_item->alias.str_show());
      fprintf(the_file, "%d\n", tmp_item->value);
      fprintf(the_file, "%d\n", (int) tmp_item->the_type);
      fprintf(the_file, "%d\n", tmp_item->num_of);
      fprintf(the_file, "^\n%s\n^\n", 
                          (tmp_item->description.str_show() == NULL) ? 
                          "No Desc" : tmp_item->description.str_show());
      tmp_item = tmp_item->next_item;
   }
   return 1;
}


/***********************************************************************
 ** display_shop - displays shop attributes to the builder 
 **
 ** Parameters: the_builder - the builder to send all this to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Shop::display_shop(Builder *the_builder)
{
   Strings holder;
   shop_item *the_list;
   char      *exchname[] = {"Buy", "Sell", "Dual"};

   the_builder->send_bldr("&+W\nShop Name&*: \t%s\n", get_name());
   the_builder->send_bldr("&+GCurrency: \t&*%s\n", get_currency());

   the_list = buy_list;
   while (the_list != NULL)
   {
      the_builder->send_bldr("\n&+GAlias: \t\t&+M%s\n", 
                                       the_list->alias.str_show());
      the_builder->send_bldr("&+GItemName: \t&*%s\n", 
                                       the_list->itemname.str_show());
      the_builder->send_bldr("&+GType: \t\t&+G%s&*\n", 
                                   exchname[(int) the_list->the_type]);
      the_builder->send_bldr("&+GValue: \t\t&*%d\n", the_list->value);
      the_builder->send_bldr("&+GNumberOf: \t&*%d%s\n", the_list->num_of, 
                       (the_list->num_of == 9999) ? " &+m(Unlimited)&*" : "");
      the_builder->send_bldr("&+GItemDesc:&* \n%s\n", 
                                       the_list->description.str_show());
      the_list = the_list->next_item;
   }
   the_builder->send_bldr("\n");

   return;
}


/***********************************************************************
 ** get_shop_item - gets a shop item that has a certain alias
 **
 ** Parameters: the_alias - the alias for this shop item we seek
 **
 ** Returns: pointer to the item if found, NULL if not found
 **
 ***********************************************************************/

shop_item *Shop::get_shop_item(char *the_alias)
{
   shop_item *tmp_item;

   tmp_item = buy_list;
   while (tmp_item != NULL)
   {
      if (!STRCASECMP(the_alias, tmp_item->alias.str_show()))
         return tmp_item;
      tmp_item = tmp_item->next_item;
   }
   return NULL;
}


/***********************************************************************
 ** set_itemname - sets the item name for a particular alias where the
 **                item name is the object name that will be cloned, or
 **                the special name that will be run
 **
 ** Parameters: the_alias - the alias for this itemname
 **             the_itemname - the itemname to set this to
 **
 ** Returns: 1 for success, -1 for failure, -2 for alias doesn't exist
 **
 ***********************************************************************/

int Shop::set_itemname(char *the_alias, char *the_itemname)
{
   shop_item *item_holder;

   if ((the_alias == NULL) || (the_itemname == NULL))
      return -1;

   if ((item_holder = get_shop_item(the_alias)) == NULL)
      return -2;

   item_holder->itemname = the_itemname;
   return 1;
}


/***********************************************************************
 ** set_value - sets the value index this item will have, or the cost of it
 **
 ** Parameters: the_alias - the alias for this itemname
 **             the_value - the value to set this to
 **
 ** Returns: 1 for success, -1 for failure, -2 for alias doesn't exist
 **
 ***********************************************************************/

int Shop::set_value(char *the_alias, int the_value)
{
   shop_item *item_holder;

   if (the_alias == NULL)
      return -1;

   if ((item_holder = get_shop_item(the_alias)) == NULL)
      return -2;

   item_holder->value = the_value;
   return 1;
}


/***********************************************************************
 ** set_type - sets the type of this exchange, buy, sell, or dual
 **
 ** Parameters: the_alias - the alias for this itemname
 **             the_value - the value to set this to
 **
 ** Returns: 1 for success, -1 for failure, -2 for alias doesn't exist
 **
 ***********************************************************************/

int Shop::set_type(char *the_alias, exchange_type the_value)
{
   shop_item *item_holder;

   if (the_alias == NULL)
      return -1;

   if ((item_holder = get_shop_item(the_alias)) == NULL)
      return -2;

   item_holder->the_type = the_value;
   return 1;
}


/***********************************************************************
 ** set_num_of - sets the number of this item that is for sale
 **
 ** Parameters: the_alias - the alias for this itemname
 **             the_value - the value to set this to
 **
 ** Returns: 1 for success, -1 for failure, -2 for alias doesn't exist
 **
 ***********************************************************************/

int Shop::set_num_of(char *the_alias, int the_value)
{
   shop_item *item_holder;

   if (the_alias == NULL)
      return -1;

   if ((item_holder = get_shop_item(the_alias)) == NULL)
      return -2;

   item_holder->num_of = the_value;
   return 1;
}

/***********************************************************************
 ** display_wares - shows the player what wares are for sale
 **
 ** Parameters: the_player - the player we send all messages to
 **
 ** Returns: 1 for success, -1 for failure, -2 for alias doesn't exist
 **
 ***********************************************************************/

int Shop::display_wares(Player *the_plr)
{
   shop_item *tmp_item;
   Money     *currency_obj;
   int value;

   currency_obj = get_currency_ptr();
   /* if the currency object pointer is not set, we have big probs */
   if (currency_obj == NULL)
   {
      the_plr->send_plr("ERROR - unexpected NULL pointer for shop "
                        "currency!\n");
      mainstruct->log_error("Unexpected NULL pointer for shop currency!\n", 
                                                           "display_wares");
      return -1;
   }

   the_plr->send_plr("%s prices:\n\n", name.str_show());
   tmp_item = buy_list;
   
   while (tmp_item != NULL)
   {
      value = tmp_item->value / currency_obj->get_money_index();

      if ((value * currency_obj->get_money_index()) < tmp_item->value)
         value++;
 
      the_plr->send_plr("\t%s costing %d %s\n", tmp_item->alias.str_show(), 
                         value, currency_obj->get_title());
      tmp_item = tmp_item->next_item;
   }
   the_plr->send_plr("\n");
   return 1;
}


/***********************************************************************
 ** display_will_buy - shows which items that the player has that this
 **                    shopkeeper will buy
 **
 ** Parameters: the_player - the player we send all messages to
 **
 ** Returns: num items given for success, 
 **          -1 for failure, 
 **          -2 for alias doesn't exist
 **
 ***********************************************************************/

int Shop::display_will_buy(Player *the_plr)
{
   shop_item  *tmp_item;
   Money      *currency_obj;
   LinkedList *plr_inventory;
   MudObject  *the_obj;
   int        value;
   Strings    full_name;
   int        count = 0;

   currency_obj = get_currency_ptr();
   /* if the currency object pointer is not set, we have big probs */
   if (currency_obj == NULL)
   {
      the_plr->send_plr("ERROR - unexpected NULL pointer for shop "
                        "currency!\n");
      mainstruct->log_error("Unexpected NULL pointer for shop currency!\n", 
                                                     "display_will_buy");
      return -1;
   }

   the_plr->send_plr("Items %s will buy from you:\n\n", name.str_show());

   plr_inventory = the_plr->get_inventory();

   plr_inventory->reset_current();
   the_obj = plr_inventory->get_next();
   while (the_obj != NULL)
   {
      if (the_obj->get_parent() == NULL)
         full_name.sprintf("%s@%s", the_obj->get_name(), the_obj->get_area());
      else
         full_name.sprintf("%s@%s", the_obj->get_parent(), the_obj->get_area());

      tmp_item = buy_list;
      while (tmp_item != NULL)
      {
         if (!STRCASECMP(full_name.str_show(), tmp_item->itemname.str_show()))
	 {
            value = (int) (((float) tmp_item->value) * .10);
            value = value / currency_obj->get_money_index();  
            if ((value * currency_obj->get_money_index()) < tmp_item->value)
               value++;
            the_plr->send_plr("\t%s for %d %s\n", tmp_item->alias.str_show(), 
                           value, currency_obj->get_title());
            count++;
         }
         tmp_item = tmp_item->next_item;
      }
      the_obj = plr_inventory->get_next();
   }
   if (!count)
      the_plr->send_plr("\tNothing.\n");

   the_plr->send_plr("\n");
   return count;
}


/***********************************************************************
 ** get_currency_ptr - gets the currency pointer for this shop object
 **
 ** Parameters: None
 **
 ** Returns: A pointer to the Money object for this object
 **
 ***********************************************************************/

Money *Shop::get_currency_ptr(void)
{
   MudObject *tmp_obj;
   Strings   holder;

   if ((tmp_obj = mainstruct->get_object(currency.str_show())) == NULL)
   {
      holder.sprintf("Currency '%s' does not exist for Shop '%s'.", 
         currency.str_show(), name.str_show());
      mainstruct->log_error(holder.str_show(), "get_currency_ptr");
      return NULL;
   }

   if (tmp_obj->get_type() != OBJ_TYPE_MONEY)
   {
      holder.sprintf("Currency '%s' not money type for Shop '%s'.", 
         currency.str_show(), name.str_show());
      mainstruct->log_error(holder.str_show(), "get_currency_ptr");
      return NULL;
   }

   return (Money *) tmp_obj;
}


/***********************************************************************
 ** add_request - adds a request to the request list
 **
 ** Parameters: the_player - the player whos name will be on this request
 **             the_obj - the object that they are requesting
 **             num_of - number of the object they are requesting
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Shop::add_request(Player *the_player, char *alias, int num_of)
{
   buy_request *tmp_request;
   buy_request *new_request;
   buy_request *prev_request = NULL;
   shop_item *tmp_item;

   if ((the_player == NULL) || (alias == NULL) || (num_of <= 0))
      return -1;

   if ((tmp_item = get_shop_item(alias)) == NULL)
      return -1;

   new_request = new_buy_request();
   tmp_request = request_list;

   new_request->playername = the_player->get_name();
   new_request->item = tmp_item;
   new_request->num_of = num_of;
   new_request->next_request = NULL;

   if (tmp_request == NULL)
   {
      request_list = new_request;
      return 1;
   }

   while (tmp_request != NULL)
   {
      if (!STRCASECMP(tmp_request->playername.str_show(), 
                      new_request->playername.str_show()))
      {
         if (prev_request == NULL)
            request_list = new_request;
         else
            prev_request->next_request = new_request;

         new_request = tmp_request->next_request;
         delete_buy_request(tmp_request);
         return 1;
      }
      prev_request = tmp_request;
      tmp_request = tmp_request->next_request;
   }
   
   prev_request->next_request = new_request;

   return 1;
}


/***********************************************************************
 ** remove_request - removes a request from the list
 **
 ** Parameters: the_name - the name for the request that will be removed
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Shop::remove_request(char *the_name)
{
   buy_request *tmp_request;
   buy_request *prev_request = NULL;

   if (the_name == NULL)
      return -1;

   tmp_request = request_list;

   while (tmp_request != NULL)
   {
      if (!STRCASECMP(tmp_request->playername.str_show(), the_name))
      {
         if (prev_request == NULL)
	 {
            request_list = tmp_request->next_request;
            delete_buy_request(tmp_request);
         }
         else
	 {
            prev_request->next_request = tmp_request->next_request;
            delete_buy_request(tmp_request);
         }

         return 1;
      }
      prev_request = tmp_request;
      tmp_request = tmp_request->next_request;
   }
   
   return -1;
}


/***********************************************************************
 ** udpate_requests - looks for and removes outdated requests
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Shop::update_requests(Location *the_loc)
{
   buy_request *tmp_request;
   buy_request *prev_request = NULL;
   int int_holder = 1;

   tmp_request = request_list;

   while (tmp_request != NULL)
   {  
      if (the_loc->find_contained(tmp_request->playername.str_show(), 
                                                      &int_holder) == NULL)
      {
         if (prev_request == NULL)
	 {
            tmp_request = tmp_request->next_request;
            delete_buy_request(request_list);
            request_list = tmp_request;
         }
         else
	 {
            prev_request->next_request = tmp_request->next_request;
            delete_buy_request(tmp_request);
         }
      }
      if (tmp_request != NULL)
      {
         prev_request = tmp_request;
         tmp_request = tmp_request->next_request;
      }
   }
   return 1;
}


/***********************************************************************
 ** get_request_cost - gets the total cost index for this request
 **
 ** Parameters: None
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Shop::get_request_cost(Player *the_player)
{
   buy_request *tmp_request;

   if (get_currency_ptr() == NULL)
      return -1;

   tmp_request = request_list;
   while (tmp_request != NULL)
   {
      if (!STRCASECMP(tmp_request->playername.str_show(), 
                                                  the_player->get_name()))
         break;

      tmp_request = tmp_request->next_request;
   }   
 
   if (tmp_request != NULL)
      return (((tmp_request->item)->value) * (tmp_request->num_of));

   return -1;
}


/***********************************************************************
 ** complete_transaction - handles the transaction for the player, given
 **                        that the player passes in enough money
 **
 ** Parameters: the_player - the player who is doing the transaction
 **             the_mobile - the shopkeeper
 **             the_money - the money that was passed to the shopkeeper
 **
 ** Returns:  1 for success, 
 **          -1 for failure
 **          -2 if we should return the money
 **
 ***********************************************************************/

int Shop::complete_transaction(Player *the_player, Mobile *the_mobile, 
                                                      Money *the_money)
{
   buy_request    *the_request;
   int            cost; 
   int            change;
   MudObject      *buy_obj;
   MudObject      *bought_obj = NULL;
   Strings        holder;
   Merger         *the_merger;
   Specials       *the_special;
   special_holder *the_holder = NULL;
   in_params      fill_param;
   int            special_used = 0;
   Strings        full_name;
   int            number_of;

   the_request = request_list;
   while (the_request != NULL)
   {
      if (!STRCASECMP(the_request->playername.str_show(), 
                                                  the_player->get_name()))
         break;

      the_request = the_request->next_request;
   }   

   the_merger = (Merger *) the_money;   

   if (the_request == NULL)
   {
      the_mobile->tell_ind(the_player, 
                    "You need to tell me what you want to buy first.");
      the_mobile->move_merger(the_player, &the_merger, 
                                             the_money->get_number_of());
      the_player->send_plr("%s returns the %s to you.\n",
                           the_mobile->get_title(), the_money->get_title()); 
      return -2;
   }

   if (the_request->item->num_of == 0)
   {

      the_mobile->tell_ind(the_player, 
                         "Oops, I seem to have run out.  Try back later.");  
      the_mobile->move_merger(the_player, &the_merger, 
                                             the_money->get_number_of());
      the_player->send_plr("%s returns the %s to you.\n",
                           the_mobile->get_title(), the_money->get_title()); 
      return -2;
   }

   cost = get_request_cost(the_player);
   if (the_money->get_total_sum_index() < cost)
   {

      the_mobile->tell_ind(the_player, "That is not enough.");  
   
      the_mobile->move_merger(the_player, &the_merger, 
                                             the_money->get_number_of());
      the_player->send_plr("%s returns the %s to you.\n",
                           the_mobile->get_title(), the_money->get_title()); 
      return -2;
   }

   holder = the_request->item->itemname.str_show();
   if (holder.num_char('@') == 0)
      full_name.sprintf("%s@%s", holder.str_show(), the_mobile->get_area());
   else
      full_name = holder.str_show();

   if ((buy_obj = mainstruct->get_object(full_name.str_show())) == NULL)
   {

      if ((the_special = mainstruct->get_special(
                                          full_name.str_show())) != NULL)
      {
         fill_param.primary_obj = NULL;
         fill_param.secondary_obj = NULL;
         fill_param.this_obj = the_mobile;

         the_holder = new_special_holder();
         the_holder->next_special = NULL;
         the_holder->the_special = the_special;
         the_holder->environment.exec_vars = NULL;
         the_holder->environment.stack_pointer = NULL;
         the_holder->environment.counter = 0;

         if (the_special->run_special(the_player, 
                  &fill_param, &the_holder->environment) != 3)
	 {
            special_used = 1;
            change = (the_money->get_total_sum_index() - cost) / 
                                            the_money->get_money_index();
            if (change > 0)
            {
               the_player->send_plr("%s gives you your change.\n", 
                                               the_mobile->get_title());
               the_mobile->move_merger(the_player, &the_merger, change);
 
            }

            if (the_request->item->num_of != 9999)
               the_request->item->num_of--;

            remove_request(the_player->get_name());

            return 1;
         }
         else
            special_used = 1;
      }
      if (!special_used)
      {
         the_mobile->tell_ind(the_player, 
                               "Sorry, I can't complete your transaction.");
         holder.sprintf("Shop item %s, itemname %s couldn't be created.", 
               the_request->item->itemname.str_show(), full_name.str_show());
         mainstruct->log_error(holder.str_show(), "complete_transaction");
      }

      the_merger = (Merger *) the_money;      
      the_mobile->move_merger(the_player, &the_merger, 
                                             the_money->get_number_of());

      the_player->send_plr("%s returns the %s to you.\n",
                           the_mobile->get_title(), the_money->get_title()); 

      return -2;


   } 


   if (the_request->num_of == 0)
      the_request->num_of = 1;

   number_of = the_request->num_of;

   if (the_player->get_capacity() < (the_player->get_size_held() + 
                           (((Moveable *) buy_obj)->get_size() * number_of)))
   {
      the_mobile->tell_ind(the_player, 
              "You're carrying too much,  try again when you can carry it.");  
      the_mobile->move_merger(the_player, &the_merger, 
                                             the_money->get_number_of());
      the_player->send_plr("%s returns the %s to you.\n",
                           the_mobile->get_title(), the_money->get_title()); 
      return -2;
   }

   if (the_player->get_weight_capacity() < (the_player->get_weight_held() + 
               (((Moveable *) buy_obj)->get_weight_held())) + 
                ((Moveable *) buy_obj)->get_weight())
   {
      the_mobile->tell_ind(the_player, 
              "It is too heavy, try again when you can carry it.");  
      the_mobile->move_merger(the_player, &the_merger, 
                                             the_money->get_number_of());
      the_player->send_plr("%s returns the %s to you.\n",
                           the_mobile->get_title(), the_money->get_title()); 
      return -2;
   }

   while (number_of != 0)
   {
      bought_obj = (mainstruct->get_dbase())->
                                     clone_object((MudObject *) buy_obj);

      if (bought_obj->is_merger())
      {
         ((Merger *) bought_obj)->set_number_of(the_request->num_of);
         number_of = 1;
      }
      if (bought_obj == NULL)
         return -1;

      if (bought_obj->is_a_moveable())
         ((Moveable *) bought_obj)->set_moved();

      the_player->add_contained((MudObject *) bought_obj);
      number_of--;
      
   }

   change = (the_money->get_total_sum_index() - cost) / 
                                            the_money->get_money_index();
 
   the_player->send_plr("%s gives you the %s.\n", the_mobile->get_title(), 
                                             bought_obj->get_title());
   holder.sprintf("%s gives the %s to %s.\n", the_mobile->get_title(), 
               bought_obj->get_title(), the_player->get_title());

   if (change > 0)
   {
      the_player->send_plr("%s gives you your change.\n", 
                                               the_mobile->get_title());
      the_mobile->move_merger(the_player, &the_merger, change);
   }

   // Decrease the number left if it is set up that way 
   if ((the_request->item)->num_of != 9999)
      the_request->item->num_of--;

   (the_player->get_loc())->send_location(holder.str_show(), the_player);
   remove_request(the_player->get_title());
   
   return 1;
}


/***********************************************************************
 ** complete_sell - handles the sale of an item to the shopkeeper
 **
 ** Parameters: the_player - the player who is doing the transaction
 **             the_mobile - the shopkeeper
 **             the_obj - the object that was passed to the shopkeeper
 **
 ** Returns:  1 for success, 
 **          -1 for failure
 **          -2 if the object has been returned
 **
 ***********************************************************************/

int Shop::complete_sell(Player *the_player, Mobile *the_mobile, 
                                                      MudObject *the_obj)
{
   shop_item      *item_data;
   int            value; 
   Money          *currency_obj;
   Money          *new_money;
   Object_List    *tmp_dbase;
   Strings        full_name; 
   Strings        holder;
   /*
   Merger         *the_merger;
   Specials       *the_special;
   special_holder *the_holder = NULL;
   in_params      fill_param;
   int            special_used = 0;
   int            number_of; */


   if (the_obj->get_parent() == NULL)
      full_name.sprintf("%s@%s", the_obj->get_name(), the_obj->get_area());
   else
      full_name.sprintf("%s@%s", the_obj->get_parent(), the_obj->get_area());

   item_data = buy_list;
   while ((item_data != NULL) && (STRCASECMP(item_data->itemname.str_show(), 
                                                    full_name.str_show())))
   {
      item_data = item_data->next_item;
   }
   if ((item_data == NULL) || (item_data->the_type == Buy))
   {
      holder.sprintf("Sorry, I have no use for the %s.", the_obj->get_title());
      the_mobile->tell_ind(the_player, holder.str_show());

      if (the_obj->is_merger())
         the_mobile->move_merger(the_player, ((Merger **) &the_obj), 
                                ((Merger *)the_obj)->get_number_of());
      else
      {
         the_mobile->remove_contained(the_obj);
         the_player->add_contained(the_obj);
      }
      the_player->send_plr("%s returns the %s to you.\n",
                           the_mobile->get_title(), the_obj->get_title());
      return -2;    
   }

   currency_obj = get_currency_ptr();

   /* calculate how much to give the person for the object */
   value = (int) (((float) item_data->value) * .25);
   value = value / currency_obj->get_money_index();  
   if ((value * currency_obj->get_money_index()) < item_data->value)
      value++;

   tmp_dbase = mainstruct->get_dbase();
   new_money = (Money *) tmp_dbase->clone_object(currency_obj);
   new_money->set_number_of(value);

   the_player->send_plr("%s gives you %d %s.\n", the_mobile->get_title(),
                       new_money->get_number_of(), currency_obj->get_title());
   the_mobile->move_merger(the_player, ((Merger **) &new_money), value);

   if (item_data->num_of != 9999)
      item_data->num_of++;

   return 1;
}


/***********************************************************************
 ** get_desc - gets the description of the shop item
 **
 ** Parameters: the_alias - the alias for this itemname
 **
 ** Returns: pointer for success, NULL for alias doesn't exist
 **
 ***********************************************************************/

char *Shop::get_desc(char *the_alias)
{
   shop_item *item_holder;

   if (the_alias == NULL)
      return NULL;

   if ((item_holder = get_shop_item(the_alias)) == NULL)
      return NULL;

   return item_holder->description.str_show();
}

/***********************************************************************
 ** set_desc - sets the description that will appear for this object
 **
 ** Parameters: the_alias - the alias for this itemname
 **             the_desc - the desc to set this to
 **
 ** Returns: 1 for success, -1 for failure, -2 for alias doesn't exist
 **
 ***********************************************************************/

int Shop::set_desc(char *the_alias, char *the_desc)
{
   shop_item *item_holder;

   if (the_alias == NULL)
      return -1;

   if ((item_holder = get_shop_item(the_alias)) == NULL)
      return -2;

   item_holder->description = the_desc;
   return 1;   
}


/***********************************************************************
 ** bldr_set_desc - starts the builder setting the item description
 **
 ** Parameters: the_alias - the alias of the item we are setting
 **             the_builder - the builder we are setting it for
 **
 ** Returns: 1 for success, -1 for failure, -2 for alias doesn't exist
 **
 ***********************************************************************/

int Shop::bldr_set_desc(char *the_alias, Builder *the_builder)
{
   shop_item *item_holder;

   if (the_alias == NULL)
      return -1;

   if ((item_holder = get_shop_item(the_alias)) == NULL)
      return -2;

   if (the_builder->get_long_input(&(item_holder->description), 1) < 0)
   {
      the_builder->send_bldr("Error reading in input, failed!\n");
      return -1;
   }
   return 1;
}

/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Shop::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Shop::get_mem_size_dynamic()
{
   int  size = 0;
   shop_item *tmp_item;
   buy_request *tmp_req;

   tmp_item = buy_list;
   while (tmp_item != NULL)
   {
      size += tmp_item->itemname.get_mem_size_dynamic();
      size += tmp_item->alias.get_mem_size_dynamic();
      size += tmp_item->description.get_mem_size_dynamic();
      size += sizeof(*tmp_item);
      tmp_item = tmp_item->next_item;
   }

   tmp_req = request_list;
   while (tmp_req != NULL)
   {
      size += tmp_req->playername.get_mem_size_dynamic();
      tmp_req = tmp_req->next_request;
   }
   size += name.get_mem_size_dynamic();
   size += currency.get_mem_size_dynamic();

   return size;
}

/***********************************************************************
 ** del_shop_item - deletes a shop item from the item list
 **
 ** Parameters: the_alias - the alias for this shop item we want to delete
 **
 ** Returns: 1 if successful, -1 if failed
 **
 ***********************************************************************/

int Shop::del_shop_item(char *alias)
{
   shop_item *tmp_item;
   shop_item *last_item = NULL;

   tmp_item = buy_list;
   while (tmp_item != NULL)
   {
      if (!STRCASECMP(alias, tmp_item->alias.str_show()))
         break;
      last_item = tmp_item;
      tmp_item = tmp_item->next_item;
   }

   if (tmp_item == NULL)
      return -1;

   if (last_item == NULL)
      buy_list = buy_list->next_item;
   else
      last_item->next_item = tmp_item->next_item;

   delete_shop_item(tmp_item);
   return 1;
}


/***********************************************************************
 ** rename_shop_item - renames a shop item alias name
 **
 ** Parameters: the_alias - the alias for the item to rename
 **
 ** Returns: 1 if successful, -1 if failed
 **
 ***********************************************************************/

int Shop::rename_shop_item(char *alias, char *new_alias)
{
   shop_item *tmp_item;

   if (get_shop_item(new_alias) != NULL)
      return -2;

   tmp_item = buy_list;
   
   while (tmp_item != NULL)
   {
      if (!STRCASECMP(alias, tmp_item->alias.str_show()))
         break;
      tmp_item = tmp_item->next_item;
   }

   if (tmp_item == NULL)
      return -1;

    tmp_item->alias.str_copy(new_alias);

    return 1;
}




#endif














