/**********************************************************************
 ** Special_Func: this is a listing of all the functions that you can
 **               call in specials code. They are called using a
 **               corresponding stringname located in a table
 **    
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate), Kelly Gerke
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 **
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef SPECIAL_FUNC_C
#define SPECIAL_FUNC_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "code.h"
#include "specials.h"
#include "player.h"
#include "global.h"
#include "objtype.h"
#include "flags.h"
#include "itemflags.h"
#include "indflags.h"
#include "utils.h"
#include "mobile.h"
#include "adminflags.h"
#include "door.h"
#include "newfuncts.h"
#include "gameflags.h"
#include "special_func.h"
#include "race.h"
#include "inp_funct.h"

/***********************************************************************
 ** send_actor - sends it to the player who is doing the action
 **    format: send_actor(<phrase>);
 **
 ***********************************************************************/
params *send_actor(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *param_list;
   Strings holder;

   param_list = the_params;
 
   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }
   
   *results = 0;
   the_player->send_plr(holder.str_show());

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** send_all_room - sends it to all players in the room
 **   format: send_all_room(<phrase>);
 **
 ***********************************************************************/
params *send_all_room(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params   *param_list;
   MudObject *tmp_obj;
   Location *the_loc;
   Strings  holder;

   param_list = the_params;

   if (the_player != NULL)
      the_loc = the_player->get_loc();
   else if (player_params->this_obj->get_type() == OBJ_TYPE_LOCATION)
      the_loc = (Location *) player_params->this_obj;
   else
   {
      if ((tmp_obj = mainstruct->
               get_object(player_params->this_obj->get_location())) == NULL)
      {
         holder.sprintf("Could not find object '%s' location '%s'", 
                                 player_params->this_obj->get_name(), 
                                 player_params->this_obj->get_location());
         mainstruct->log_error(holder.str_show(), "send_all_room");
         *results = -1;
         return NULL;
      }

      if (tmp_obj->get_type() != OBJ_TYPE_LOCATION)
      {
         holder.sprintf("Object '%s' location '%s' is not a location type", 
                                 player_params->this_obj->get_name(), 
                                 player_params->this_obj->get_location());
         mainstruct->log_error(holder.str_show(), "send_all_room");
         *results = -1;
         return NULL;
      }

      the_loc = (Location *) tmp_obj;
   }

   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }
   
   *results = 0;

   the_loc->send_location(holder.str_show(), NULL);

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** send_all_room_except - sends it to all players in the room, except the
 **                        player who is running this
 **    send_all_room_except(<phrase>);
 **
 ***********************************************************************/
params *send_all_room_except(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params   *param_list;
   Location *the_loc;
   Strings holder;

   param_list = the_params;

   the_loc = the_player->get_loc();
   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }
   the_loc->send_location(holder.str_show(), the_player);
   
   *results = 0;

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** send_room_except_dual - sends it to all players in the room, except
 **                         the two indicated, if they are players
 **    send_room_except_dual(<this|primary|secondary|actor>, 
 **                          <this|primary|secondary|actor>, <phrase>);
 **
 ***********************************************************************/
params *send_room_except_dual(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params    *param_list;
   Location  *the_loc;
   Strings   holder;
   MudObject *first;
   MudObject *second;
   Player    *tmp_player[2];
   int       i;

   param_list = the_params;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "send_room_except_dual");
      *results = -1;
      return NULL;
   }

   if ((first = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "send_room_except_dual", the_player, 
                                                   player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }


   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "send_room_except_dual");
      *results = -1;
      return NULL;
   }

   if ((second = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "send_room_except_dual", the_player, 
                                                   player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if ((!first->is_an_individual()) || (!second->is_an_individual()))
   {
      mainstruct->
         log_error("First and second parameters must both be individuals.", 
                                        "send_room_except_dual (special)");
      *results = -1;
      return NULL;
   }

   tmp_player[0] = NULL;
   tmp_player[1] = NULL;
   i = 0;
   if (first->get_type() == OBJ_TYPE_PLAYER)
   {
      tmp_player[i] = (Player *) first;
      i++;
   }

   if (second->get_type() == OBJ_TYPE_PLAYER)
   {
      tmp_player[i] = (Player *) second;
   }

   the_loc = the_player->get_loc();
   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }
   the_loc->send_location(holder.str_show(), tmp_player[0], tmp_player[1]);
   
   *results = 0;

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** send_obj_room - sends it to all players in the room the object is in
 **    usage: send_all_room(<primary|secondary|this|actor>, <phrase>);
 **
 ***********************************************************************/
params *send_obj_room(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params   *param_list;
   MudObject *the_obj;
   MudObject *tmp_obj;
   Location *the_loc;
   Strings  holder;

   param_list = the_params;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "send_obj_room");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "send_obj_room", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;

   if (the_obj->get_type() == OBJ_TYPE_LOCATION)
   {
      tmp_obj = the_obj;
   }
   else 
   {   
      if ((tmp_obj = mainstruct->get_object(the_obj->get_location())) == NULL)
      {
         holder.sprintf("Could not find object '%s' location '%s'", 
                the_obj->get_name(), the_obj->get_location());
         mainstruct->log_error(holder.str_show(), "send_obj_room (special)");
         *results = -1;
         return NULL;
      } 

      if (tmp_obj->get_type() != OBJ_TYPE_LOCATION)
      {
         holder.sprintf("Object '%s' location '%s' is not a location type", 
                the_obj->get_name(), the_obj->get_location());
         mainstruct->log_error(holder.str_show(), "send_param (special)");
         *results = -1;
         return NULL;
      }
   }

   the_loc = (Location *) tmp_obj;

   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }
   
   *results = 0;

   the_loc->send_location(holder.str_show(), NULL);

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   the_player = NULL;
   return NULL;
}


/***********************************************************************
 ** send_all_except - sends it to all players in the mud, except the
 **                   one indicated
 **    send_all_except(<this|primary|secondary|actor>, <phrase>);
 **
 ***********************************************************************/
params *send_all_except(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params    *param_list;
   Strings   holder;
   MudObject *the_obj;
   Player    *tmp_player;

   param_list = the_params;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "send_all_except");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "send_all_except", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;

   if (!the_obj->is_an_individual())
   {
      mainstruct->
         log_error("First parameter must be an individual.", 
                                        "send_all_except (special)");
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
      tmp_player = NULL;
   else
      tmp_player = (Player *) the_obj;

   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }

   mainstruct->send_all_players(holder.str_show(), tmp_player);
   
   *results = 0;

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** send_all - sends it to all players in the mud
 **    send_all(<phrase>);
 **
 ***********************************************************************/
params *send_all(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params    *param_list;
   Strings   holder;

   param_list = the_params;

   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }

   mainstruct->send_all_players(holder.str_show(), NULL);
   
   *results = 0;

   /* to eliminate the compile warnings */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** send_to - sends it to an indicated player
 **    send_to(<this|primary|secondary|actor>, <phrase>);
 **
 ***********************************************************************/
params *send_to(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params    *param_list;
   Strings   holder;
   MudObject *target;
   Player    *tmp_player;

   param_list = the_params;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "send_to");
      *results = -1;
      return NULL;
   }

   if ((target = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "send_to", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }


   if (!target->is_an_individual())
   {
      mainstruct->
         log_error("First parameters must be an individual.", 
                                        "send_to (special)");
      *results = -1;
      return NULL;
   }

   if (target->get_type() == OBJ_TYPE_PLAYER)
   {
      tmp_player = (Player *) target;
   }
   else
   {
      *results = 0;
      return NULL;
   }

   param_list = param_list->next_param;

   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }
   tmp_player->send_plr(holder.str_show());
   
   *results = 1;

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   the_player = NULL;
   return NULL;
}


/***********************************************************************
 ** send_holder - sends to either the player if the player is holding
 **               the object, or to the room if it is lying on the
 **               ground
 **    format: send_holder(<primary|secondary|this|actor>, <message>);
 ***********************************************************************/
params *send_holder(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params    *param_list;
   Strings   holder;
   MudObject *the_obj;
   MudObject *container;

   param_list = the_params;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "send_holder");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "send_holder", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }


   if ((container = the_obj->get_contained_by()) == NULL)
   {
      mainstruct->log_error("Error, could not find container of object.", 
                                                "send_holder");
      *results = -1;
      return NULL;
   }

   if ((container->get_type() != OBJ_TYPE_PLAYER) &&
       (container->get_type() != OBJ_TYPE_LOCATION))
   {
      *results = 1;
      return NULL;
   }

   param_list = param_list->next_param;
   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }
   
   *results = 0;

   if (container->get_type() == OBJ_TYPE_LOCATION)
      ((Location*) container)->send_location(holder.str_show(), NULL);
   else
      ((Player *) container)->send_plr(holder.str_show());
      

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_player = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** tell_ind - sends a tell from the first parameter string to the
 **            individual indicated in the second parameter string
 **    format: tell_ind(<primary|secondary|this|actor>,
 **                     <primary|secondary|this|actor>, <phrase>);
 **
 ***********************************************************************/
params *tell_ind(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *param_list;
   Strings holder;
   Strings formatted;
   Strings tell_name;
   MudObject *tmp_obj;
   Player  *targ_player;

   param_list = the_params;
 
   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      mainstruct->log_error("First parameter needed, must be string", 
                                                             "tell_ind");
      *results = -1;
      return NULL;
   }

   if ((tmp_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "tell_ind", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }
   
   if (!tmp_obj->is_an_individual())
   {
      holder.sprintf("First param '%s' object '%s' is not an individual", 
         param_list->the_param.a_string->str_show(), tmp_obj->get_title());
      mainstruct->log_error(holder.str_show(), "tell_ind (special)");
      *results = -1;
      return NULL;

   }
   
   tell_name = tmp_obj->get_title();

   param_list = param_list->next_param;

   if ((tmp_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "tell_ind", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (!tmp_obj->is_an_individual())
   {
      holder.sprintf("Second param '%s' object '%s' is not an individual", 
         param_list->the_param.a_string->str_show(), tmp_obj->get_title());
      mainstruct->log_error(holder.str_show(), "tell_ind (special)");
      *results = -1;
      return NULL;
   }

   if (tmp_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      *results = 0;
      return NULL;
   }

   targ_player = (Player *) tmp_obj;

   param_list = param_list->next_param;

   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         holder.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }

   formatted.format_for_comm(holder.str_show(), 
                  (strlen(tell_name.str_show()) + 12), the_config.max_line_len);
   targ_player->send_plr("&+B%s&+w tells &+Byou &+W'&+G%s&+W'&*\n", 
                               tell_name.str_show(), formatted.str_show());
   
   *results = 0;

   /* to eliminate the compile warnings */
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}

/******************* flow control functions *************************/

/***********************************************************************
 ** goto_spec - goto in all cases
 **    format: goto(<codemarkername>);
 **
 ***********************************************************************/
params *goto_spec(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *goto_param;
   vars   *the_var;

   goto_param = the_params;
   if (goto_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "goto_spec");
      *results = -1;
      return NULL;
   }

   if ((the_var = 
          the_special->get_var(goto_param->the_param.a_string->str_show()))
                                                                  == NULL)
   {
      mainstruct->log_error("First parameter not a valid code marker", 
                                                 "goto_spec");
      *results = -1;
      return NULL;

   }
 
   if (the_var->vartype != VAR_TYPE_GOTO)
   {
      mainstruct->log_error("First parameter is not a code marker", 
                                                        "goto_spec");
      *results = -1;
      return NULL;
   }

   environment->stack_pointer = (*(the_var->the_ptr.goto_code));
   *results = 2;
   return NULL;

   /* to remove the silly warnings on compile */
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** goto_if_eq - goto the indicated code marker if the two numbers are equal
 **    format: goto_if_eq(<codemarkername>, <number1>, <number2>);
 **
 ***********************************************************************/
params *goto_if_eq(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *goto_param;
   params *first_num;
   params *second_num;
   vars   *the_var;
   int    is_string = 0;

   goto_param = the_params;
   if (goto_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "goto_if_eq");
      *results = -1;
      return NULL;
   }

   if ((the_var = 
          the_special->get_var(goto_param->the_param.a_string->str_show()))
                                                                  == NULL)
   {
      mainstruct->log_error("First parameter not a valid code marker", 
                                                 "goto_if_eq");
      *results = -1;
      return NULL;

   }
 
   if (the_var->vartype != VAR_TYPE_GOTO)
   {
      mainstruct->log_error("First parameter is not a code marker", 
                                                        "goto_if_eq");
      *results = -1;
      return NULL;
   }

   first_num = the_params->next_param;
   
   if ((first_num == NULL) || (first_num->param_type == VAR_TYPE_INT))
   {
      is_string = 0;
   }
   else if ((first_num == NULL) || 
                              (first_num->param_type == VAR_TYPE_STRING))
   {
      is_string = 1;
   }
   else
   {
      mainstruct->
          log_error("Second parameter should be an integer or string", 
                                                        "goto_if_eq");
      *results = -1;
      return NULL;
   }

   second_num = (the_params->next_param)->next_param;
   
   if ((second_num == NULL) || 
                      (second_num->param_type != first_num->param_type))
   {
      mainstruct->log_error("Third parameter should be same type as second", 
                                                        "goto_if_eq");
      *results = -1;
      return NULL;
   }

   if (((is_string) && (!STRCASECMP(
                           first_num->the_param.a_string->str_show(), 
                             second_num->the_param.a_string->str_show()))) ||
       ((!is_string) && (*(first_num->the_param.an_int) == 
                                           *(second_num->the_param.an_int))))
   {
      environment->stack_pointer = (*(the_var->the_ptr.goto_code));
      *results = 2;
   }
   else
      *results = 0;
   return NULL;

   /* to remove the silly warnings on compile */
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** goto_if_neq - goto the indicated code marker if the two numbers are not
 **               equal
 **    format: goto_if_eq(<codemarkername>, <number1>, <number2>);
 **
 ***********************************************************************/
params *goto_if_neq(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *goto_param;
   params *first_num;
   params *second_num;
   vars   *the_var;
   int    is_string = 0;

   goto_param = the_params;
   if (goto_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "goto_if_neq");
      *results = -1;
      return NULL;
   }

   if ((the_var = 
          the_special->get_var(goto_param->the_param.a_string->str_show()))
                                                                  == NULL)
   {
      mainstruct->log_error("First parameter not a valid code marker", 
                                                 "goto_if_neq");
      *results = -1;
      return NULL;

   }
 
   if (the_var->vartype != VAR_TYPE_GOTO)
   {
      mainstruct->log_error("First parameter is not a code marker", 
                                                        "goto_if_neq");
      *results = -1;
      return NULL;
   }

   first_num = the_params->next_param;

   if ((first_num == NULL) || (first_num->param_type == VAR_TYPE_INT))
   {
      is_string = 0;
   }
   else if ((first_num == NULL) || 
                              (first_num->param_type == VAR_TYPE_STRING))
   {
      is_string = 1;
   }
   else
   {
      mainstruct->
          log_error("Second parameter should be an integer or string", 
                                                        "goto_if_neq");
      *results = -1;
      return NULL;
   }

   second_num = (the_params->next_param)->next_param;
   
   if ((second_num == NULL) || 
                      (second_num->param_type != first_num->param_type))
   {
      mainstruct->log_error("Third parameter should be same type as second", 
                                                        "goto_if_neq");
      *results = -1;
      return NULL;
   }

   if (((is_string) && (STRCASECMP(
                           first_num->the_param.a_string->str_show(), 
                             second_num->the_param.a_string->str_show()))) ||
       ((!is_string) && (*(first_num->the_param.an_int) != 
                                           *(second_num->the_param.an_int))))
   {
      environment->stack_pointer = (*(the_var->the_ptr.goto_code));
      *results = 2;
   }
   else
      *results = 0;

   return NULL;

   /* to remove the silly warnings on compile */
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** goto_if_less - goto the indicated code marker if number1 is less than
 **                number 2
 **    format: goto_if_less(<codemarkername>, <number1>, <number2>);
 **
 ***********************************************************************/
params *goto_if_less(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *goto_param;
   params *first_num;
   params *second_num;
   vars   *the_var;

   goto_param = the_params;
   if (goto_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "goto_if_less");
      *results = -1;
      return NULL;
   }

   if ((the_var = 
          the_special->get_var(goto_param->the_param.a_string->str_show()))
                                                                  == NULL)
   {
      mainstruct->log_error("First parameter not a valid code marker", 
                                                 "goto_if_less");
      *results = -1;
      return NULL;

   }
 
   if (the_var->vartype != VAR_TYPE_GOTO)
   {
      mainstruct->log_error("First parameter is not a code marker", 
                                                     "goto_if_less");
      *results = -1;
      return NULL;
   }

   first_num = the_params->next_param;
   
   if ((first_num == NULL) || (first_num->param_type != VAR_TYPE_INT))
   {
      mainstruct->log_error("Second parameter should be an integer", 
                                                        "goto_if_less");
      *results = -1;
      return NULL;
   }

   second_num = (the_params->next_param)->next_param;
   
   if ((second_num == NULL) || (second_num->param_type != VAR_TYPE_INT))
   {
      mainstruct->log_error("Third parameter should be an integer", 
                                                        "goto_if_less");
      *results = -1;
      return NULL;
   }

   if (*(first_num->the_param.an_int) < *(second_num->the_param.an_int))
   {
      environment->stack_pointer = (*(the_var->the_ptr.goto_code));
      *results = 2;
   }
   else
      *results = 0;
   return NULL;

   /* to remove the silly warnings on compile */
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** ret_inv_crit - returns from the code having not met certain
 **                           criteria for successful execution
 **
 **    format: ret_inv_crit();
 **
 ***********************************************************************/
params *ret_inv_crit(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   *results = 4;
   return NULL;

   /* get rid of all those compile warnings */
   the_special = NULL;
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** force_termination - forces termination of the ability, usually because
 **                     a special is going to take over
 **
 **    format: force_termination();
 **
 ***********************************************************************/
params *force_termination(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   *results = 8;


   return NULL;

   /* get rid of all those compile warnings */
   the_special = NULL;
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** force_success - exits the special and forces success of an ability
 **
 **    format: force_success();
 **
 ***********************************************************************/
params *force_success(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   *results = 6;


   return NULL;

   /* get rid of all those compile warnings */
   the_special = NULL;
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** force_failure - exits the special and forces failure of an ability
 **
 **    format: force_failure();
 **
 ***********************************************************************/
params *force_failure(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   *results = 5;


   return NULL;

   /* get rid of all those compile warnings */
   the_special = NULL;
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/**************** Functions for All MudObjects *************************/

/***********************************************************************
 ** get - gets an attribute on an object
 **
 **    format: get("<this|primary|secondary|actor>", "<attribute>");
 **
 ** Attibute can be: name, title, strength
 **
 ***********************************************************************/
params *get(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   params    *param_list;
   Strings   attribute;
   Strings   holder;
   params    *new_param;

   param_list = the_params;
   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "get (special)");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "get", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   param_list = the_params->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                              "get");
      *results = -1;
      return NULL;
   }

   attribute = param_list->the_param.a_string->str_show();

   new_param = new_params();
   new_param->param_type = VAR_TYPE_STRING;

   if (!STRCASECMP(attribute.str_show(), "name"))
   {
      holder.sprintf("%s@%s", the_obj->get_name(), the_obj->get_area());
      new_param->param_type = VAR_TYPE_STRING;
      new_param->the_param.a_string = new_Strings(holder.str_show());
      *results = 1;   
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "title"))
   {
      new_param->param_type = VAR_TYPE_STRING;
      new_param->the_param.a_string = new_Strings(the_obj->get_title());
      *results = 1;   
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "location"))
   {
      new_param->param_type = VAR_TYPE_STRING;
      new_param->the_param.a_string = new_Strings(the_obj->get_location());
      *results = 1;   
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "desc"))
   {
      new_param->param_type = VAR_TYPE_STRING;
      new_param->the_param.a_string = new_Strings(the_obj->get_desc());
      *results = 1;   
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "strength"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only get strength on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "get");
         *results = -1;
         return NULL;
      }
      new_param->param_type = VAR_TYPE_INT;
      new_param->the_param.an_int = new_long();
      *(new_param->the_param.an_int) = ((Individual *)the_obj)->get_strength();
      *results = 1;
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "dexterity"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only get dexterity on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "get");
         *results = -1;
         return NULL;
      }
      new_param->param_type = VAR_TYPE_INT;
      new_param->the_param.an_int = new_long();
      *(new_param->the_param.an_int) = ((Individual *)the_obj)->get_dexterity();
      *results = 1;
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "health"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only get health on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "get");
         *results = -1;
         return NULL;
      }
      new_param->param_type = VAR_TYPE_INT;
      new_param->the_param.an_int = new_long();
      *(new_param->the_param.an_int) = ((Individual *)the_obj)->get_health();
      *results = 1;
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "maxhealth"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only get maxhealth on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "get");
         *results = -1;
         return NULL;
      }
      new_param->param_type = VAR_TYPE_INT;
      new_param->the_param.an_int = new_long();
      *(new_param->the_param.an_int) = ((Individual *)the_obj)->get_maxhealth();
      *results = 1;
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "intelligence"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only get intelligence on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "get");
         *results = -1;
         return NULL;
      }
      new_param->param_type = VAR_TYPE_INT;
      new_param->the_param.an_int = new_long();
      *(new_param->the_param.an_int) = ((Individual *)the_obj)->get_intel();
      *results = 1;
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "constitution"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only get constitution on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "get");
         *results = -1;
         return NULL;
      }
      new_param->param_type = VAR_TYPE_INT;
      new_param->the_param.an_int = new_long();
      *(new_param->the_param.an_int) = ((Individual *)the_obj)->get_constitution();
      *results = 1;
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "experience"))
   {
      if (the_obj->get_type() != OBJ_TYPE_PLAYER)
      {
         holder.sprintf("You can only get experience on players, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "get");
         *results = -1;
         return NULL;
      }
      new_param->param_type = VAR_TYPE_INT;
      new_param->the_param.an_int = new_long();
      *(new_param->the_param.an_int) = ((Player *)the_obj)->get_exp();
      *results = 1;
      return new_param;
   }
   else if (!STRCASECMP(attribute.str_show(), "state"))
   {
      if (!the_obj->is_an_item())
      {
         holder.sprintf("You can only get state on items, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "get");
         *results = -1;
         return NULL;
      }
      new_param->param_type = VAR_TYPE_INT;
      new_param->the_param.an_int = new_long();
      *(new_param->the_param.an_int) = 
                               (long) ((Item *)the_obj)->get_state();
      *results = 1;
      return new_param;
   }
   else
   {
      holder.sprintf("Second parameter attribute type '%s' not recognized "
                     "in special %s@%s", attribute.str_show(), 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "set");
      *results = -1;
      return NULL;
   }

   environment = NULL;
}


/***********************************************************************
 ** set - sets an attribute on an object to a specific value
 **
 **    format: set("<this|primary|secondary|actor>", "<attribute>", <value>);
 **
 ** Attibute can be: name, 
 **
 ***********************************************************************/
params *set(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   params    *param_list;
   Strings   attribute;
   Strings   holder;

   param_list = the_params;
   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "set (special)");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "set", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "set");
      *results = -1;
      return NULL;
   }

   attribute = param_list->the_param.a_string->str_show();

   param_list = param_list->next_param;

   if (!STRCASECMP(attribute.str_show(), "title"))
   {
      if (param_list->param_type != VAR_TYPE_STRING)
      {
         holder.sprintf("Third parameter of special %s@%s must be a string "
             "to set title", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      the_obj->set_title(param_list->the_param.a_string->str_show());
   }
   else if (!STRCASECMP(attribute.str_show(), "desc"))
   {
      if (param_list->param_type != VAR_TYPE_STRING)
      {
         holder.sprintf("Third parameter of special %s@%s must be a string "
             "to set desc", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      the_obj->set_desc(param_list->the_param.a_string->str_show());
   }

   else if (!STRCASECMP(attribute.str_show(), "strength"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only set strength on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      if (param_list->param_type != VAR_TYPE_INT)
      {
         holder.sprintf("Third parameter of special %s@%s must be a number "
            "to set strength", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }
    
      ((Individual *)the_obj)->set_strength(*param_list->the_param.an_int);
      *results = 1;
      return NULL;
   }
   else if (!STRCASECMP(attribute.str_show(), "dexterity"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only set dexterity on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      if (param_list->param_type != VAR_TYPE_INT)
      {
         holder.sprintf("Third parameter of special %s@%s must be a number "
           "to set dexterity", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }
    
      ((Individual *)the_obj)->set_dexterity(*param_list->the_param.an_int);
      *results = 1;
      return NULL;
   }
   else if (!STRCASECMP(attribute.str_show(), "constitution"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only set constitution on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      if (param_list->param_type != VAR_TYPE_INT)
      {
         holder.sprintf("Third parameter of special %s@%s must be a number "
                           "to set contitution", the_special->get_name(), 
                                                     the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }
    
      ((Individual *)the_obj)->set_constitution(*param_list->the_param.an_int);
      *results = 1;
      return NULL;
   }
   else if (!STRCASECMP(attribute.str_show(), "health"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only set health on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      if (param_list->param_type != VAR_TYPE_INT)
      {
         holder.sprintf("Third parameter of special %s@%s must be a number "
           "to set health", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }
    
      ((Individual *)the_obj)->set_health(*param_list->the_param.an_int);
      *results = 1;
      return NULL;
   }
   else if (!STRCASECMP(attribute.str_show(), "intelligence"))
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("You can only set intelligence on individuals, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      if (param_list->param_type != VAR_TYPE_INT)
      {
         holder.sprintf("Third parameter of special %s@%s must be a number "
           "to set intelligence", the_special->get_name(), 
                                                    the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }
    
      ((Individual *)the_obj)->set_intel(*param_list->the_param.an_int);
      *results = 1;
      return NULL;
   }
   else if (!STRCASECMP(attribute.str_show(), "experience"))
   {
      if (the_obj->get_type() != OBJ_TYPE_PLAYER)
      {
         holder.sprintf("You can only set experience on players, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      if (param_list->param_type != VAR_TYPE_INT)
      {
         holder.sprintf("Third parameter of special %s@%s must be a number "
           "to set experience", the_special->get_name(), 
                                                    the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }
    
      ((Player *)the_obj)->set_exp(*param_list->the_param.an_int);
      *results = 1;
      return NULL;
   }

   else if (!STRCASECMP(attribute.str_show(), "state"))
   {
      if (!the_obj->is_an_item())
      {
         holder.sprintf("You can only set state on item objects, "
            "special %s@%s", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }

      if (param_list->param_type != VAR_TYPE_INT)
      {
         holder.sprintf("Third parameter of special %s@%s must be a number "
            "to set state", the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "set");
         *results = -1;
         return NULL;
      }
    
      ((Item *)the_obj)->set_state(*param_list->the_param.an_int);
      *results = 1;
      return NULL;
   }
   else
   {
      holder.sprintf("Second parameter attribute type '%s' not recognized "
                     "in special %s@%s", attribute.str_show(), 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "set");
      *results = -1;
      return NULL;
   }
 
   *results = 1;
   return NULL;

   environment = NULL;
}



/***********************************************************************
 ** destroy_this_obj - destroys the object that this special is attached
 **                    to, deleting it from it's container and the
 **                    database, and returns the function
 **
 **    format: destroy_this_obj();
 **
 ***********************************************************************/

params *destroy_this_obj(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   *results = 3;


   return NULL;

   /* get rid of all those compile warnings */
   the_special = NULL;
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** get_parent_name - returns the parent name of this object with area
 **
 **    format: get_parent_name(<which var>);
 **
 ***********************************************************************/
params *get_parent_name(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params    *param_list;
   MudObject *the_obj;
   char *parent;
   Strings   holder;
   params    *new_param;

   param_list = the_params;


   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "get_parent_name (special)");

      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "get_parent_name", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if ((parent = the_obj->get_parent()) == NULL)
      parent = the_obj->get_name();

   holder.sprintf("%s@%s", parent, the_obj->get_area());

   new_param = new_params();

   new_param->param_type = VAR_TYPE_STRING;
   new_param->the_param.a_string = new_Strings(holder.str_show());;

   *results = 1;

   return new_param;

   /* to eliminate the compile warnings */
   the_player = NULL;
   the_special = NULL;
   player_params = NULL;
   environment = NULL;
   return NULL;   
}


/***********************************************************************
 ** is_individual - returns true (1) if object is an individual, false (0) 
 **                   if not
 **    format: is_individual(<this|primary|secondary>);
 **
 ***********************************************************************/

params *is_individual(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *new_param;

   if (the_params->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "is_individual");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "is_individual", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = (long) the_obj->is_an_individual();

   *results = 0;   
   return new_param;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** clone_object - clones the given object and places it in the inventory
 **                of the second attribute specified
 **    format: clone_object(<objname>, 
 **                          <this|primary|secondary|inloc|actor|none>);
 **
 ***********************************************************************/

params *clone_object(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   MudObject *temp_obj;
   MudObject *parent_obj;
   MudObject *new_obj;
   Strings   holder;
   Strings   objname;
   params    *new_param;
   params    *param_list;
   Object_List *obj_dbase;

   param_list = the_params;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be an object name", 
                                                 "clone_object");
      *results = -1;
      return NULL;
   }
   objname = param_list->the_param.a_string->str_show();

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "clone_object");
      *results = -1;
      return NULL;
   }

   if (!STRCASECMP(param_list->the_param.a_string->str_show(), "primary"))
   {
      the_obj = player_params->primary_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "secondary"))
   {
      the_obj = player_params->secondary_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "this"))
   {
      the_obj = player_params->this_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "actor"))
   {
      the_obj = the_player;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "inloc"))
   {
      if ((temp_obj = player_params->this_obj) == NULL)
      {
         if ((temp_obj = player_params->primary_obj) == NULL)
	 {
	   holder.sprintf("'This' object is pointed to null!."); 
	   mainstruct->
	     log_error(holder.str_show(), "clone_object (special)");
	   *results = -1;
	   return NULL;
	 }
      }

      while ((temp_obj->get_type() != OBJ_TYPE_LOCATION) && 
             (temp_obj->is_a_mudobject()) &&
             (temp_obj->get_contained_by() != NULL))
         temp_obj = temp_obj->get_contained_by();
      if (temp_obj->get_type() != OBJ_TYPE_LOCATION)
      {
         holder.sprintf("'This' object does not exist in a location!.\n"); 
         mainstruct->
            log_error(holder.str_show(), "clone_object (special)");
         *results = -1;
         return NULL;
      }
      the_obj = temp_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "none"))
   {
      the_obj = NULL;
   }
   else
   {
      holder.sprintf("%s is not a valid second parameter.  "
        "Must be either 'primary', 'secondary', 'this', 'inloc' or 'none'\n", 
                            param_list->the_param.a_string->str_show());
      mainstruct->
            log_error(holder.str_show(), "clone_object (special)");
      *results = -1;
      return NULL;
   }

   if ((parent_obj = mainstruct->get_object(objname.str_show())) == NULL)
   {
      holder.sprintf("MudObject '%s' not found.", objname.str_show());
      mainstruct->
            log_error(holder.str_show(), "clone_object (special)");
      *results = -1;
      return NULL;
   }

   if (parent_obj->get_type() == OBJ_TYPE_LOCATION)
   {
      holder.sprintf("Object '%s' is a location, can't be cloned.", 
                                                        objname.str_show());
      mainstruct->
            log_error(holder.str_show(), "clone_object (special)");
      *results = -1;
      return NULL;
   }

   obj_dbase = mainstruct->get_dbase();   
   if ((new_obj = obj_dbase->clone_object(parent_obj)) == NULL)
   {
      holder.sprintf("Error cloning object '%s'.", objname.str_show());
      mainstruct->
            log_error(holder.str_show(), "clone_object (special)");
      *results = -1;
      return NULL;
   }
   
   if (the_obj != NULL)
   {
      the_obj->add_contained(new_obj);
   }

   if (the_obj->is_a_moveable())
      ((Moveable *) the_obj)->set_moved();
   
   holder.sprintf("%s@%s", new_obj->get_name(), new_obj->get_area());
   new_param = new_params();

   new_param->param_type = VAR_TYPE_OBJNAME;
   new_param->the_param.a_string = new_Strings(holder.str_show());

   *results = 0;   
   return new_param;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** spec_add_status - adds a status to the mudobject indicated
 **    format: add_status(<this|primary|secondary>, <status_string>);
 **
 ***********************************************************************/

params *spec_add_status(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *param_list;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "add_status");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "is_individual", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "add_status");
      *results = -1;
      return NULL;
   }

   the_obj->add_status(param_list->the_param.a_string->str_show());

   *results = 0;

   /* to eliminate the compile warnings */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** remove_status - removes a status from the mudobject indicated
 **    format: remove_status(<this|primary|secondary>, <status_string>);
 **
 ***********************************************************************/

params *remove_status(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *param_list;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "remove_status");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "is_individual", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "remove_status");
      *results = -1;
      return NULL;
   }

   *results = 0;
   the_obj->remove_status(param_list->the_param.a_string->str_show());

   /* to eliminate the compile warnings */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
   return NULL;
}


/***********************************************************************
 ** has_status - returns if the object has status (1) or not (0)
 **    format: add_status(<this|primary|secondary>, <status_string>);
 **
 ***********************************************************************/

params *has_status(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *new_param;
   params    *param_list;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "has_status");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "is_individual", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "has_status");
      *results = -1;
      return NULL;
   }

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = 
      (long) the_obj->has_status(param_list->the_param.a_string->str_show());

   *results = 0;   
   return new_param;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** move_object - moves the given object and places it in a new location
 **    format: move_object(<objname>, 
 **                          <this|primary|secondary|inloc|actor|none>);
 **
 ***********************************************************************/

params *move_object(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *loc_obj;
   MudObject *prev_loc;
   MudObject *tmp_obj;
   MudObject *move_obj;
   Strings   holder;
   Strings   objname;
   params    *param_list;

   param_list = the_params;

   if (param_list->param_type != VAR_TYPE_OBJNAME)
   {
      mainstruct->log_error("First parameter should be an object name", 
                                                 "move_object");
      *results = -1;
      return NULL;
   }
   objname = param_list->the_param.a_string->str_show();

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "move_object");
      *results = -1;
      return NULL;
   }

   if (!STRCASECMP(param_list->the_param.a_string->str_show(), "primary"))
   {
      loc_obj = player_params->primary_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "secondary"))
   {
      loc_obj = player_params->secondary_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "this"))
   {
      loc_obj = player_params->this_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "actor"))
   {
      loc_obj = the_player;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "inloc"))
   {
      tmp_obj = player_params->this_obj;
      while ((tmp_obj->get_type() != OBJ_TYPE_LOCATION) && 
             (tmp_obj->is_a_mudobject()) &&
             (tmp_obj->get_contained_by() != NULL))
         tmp_obj = tmp_obj->get_contained_by();
      if (tmp_obj->get_type() != OBJ_TYPE_LOCATION)
      {
         holder.sprintf("'This' object does not exist in a location!.\n"); 
         mainstruct->
            log_error(holder.str_show(), "move_object (special)");
         *results = -1;
         return NULL;
      }
      loc_obj = tmp_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "none"))
   {
      loc_obj = NULL;
   }
   else
   {
      holder.sprintf("%s is not a valid second parameter.  "
        "Must be either 'primary', 'secondary', 'this', 'inloc' or 'none'\n", 
                            param_list->the_param.a_string->str_show());
      mainstruct->
            log_error(holder.str_show(), "move_object (special)");
      *results = -1;
      return NULL;
   }

   if ((move_obj = mainstruct->get_object(objname.str_show())) == NULL)
   {
      holder.sprintf("MudObject '%s' not found.", objname.str_show());
      mainstruct->
            log_error(holder.str_show(), "move_object (special)");
      *results = -1;
      return NULL;
   }

   if (move_obj->get_type() == OBJ_TYPE_LOCATION)
   {
      holder.sprintf("Object '%s' is a location, can't be moved.", 
                                                        objname.str_show());
      mainstruct->
            log_error(holder.str_show(), "move_object (special)");
      *results = -1;
      return NULL;
   }
   
   prev_loc = move_obj->get_contained_by();
   
   move_obj->set_location(loc_obj, prev_loc);   

   if (move_obj->is_a_moveable())
      ((Moveable *) move_obj)->set_moved();
   
   *results = 0;   
   return NULL;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** object_exists - returns if the object has status (1) or not (0)
 **    format: object_exists(<this|primary|secondary>);
 **
 ***********************************************************************/

params *object_exists(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *new_param;
   params    *param_list;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "has_status");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "object_exists", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();

   if (the_obj == NULL)
      *(new_param->the_param.an_int) = 0;
   else
      *(new_param->the_param.an_int) = 1;
      
   *results = 1;   
   return new_param;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** attach_special - attaches a special to an object
 **    format: attach_special(<this|primary|secondary|actor>);
 **
 ***********************************************************************/

params *attach_special(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *param_list;
   Specials  *tmp_special;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "attach_special");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "attach_special", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                              "attach_special (special)");
      *results = -1;
      return NULL;
   }

   if ((tmp_special = mainstruct->get_special(
                     param_list->the_param.a_string->str_show())) == NULL)
   {
      holder.sprintf("Could not find special '%s' to attach",
                         param_list->the_param.a_string->str_show());
      mainstruct->log_error(holder.str_show(), "attach_special (special)");
      *results = -1;
      return NULL;
   }

   the_obj->add_special(tmp_special);

   *results = 1;
   return NULL;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** detach_special - removes the special that is executing from the obj
 **
 **    format: detach_special();
 **
 ***********************************************************************/
params *detach_special(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   *results = 7;


   return NULL;

   /* get rid of all those compile warnings */
   the_special = NULL;
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}



/***********************************************************************
 ** is_itemflag_set - returns true (1) if an itemflag is set, false (0) 
 **                   if not
 **    format: is_itemflag_set(<this|primary|secondary>, <flagname>);
 **
 ***********************************************************************/

params *is_itemflag_set(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *flag_param;
   params *new_param;
   MudObject *the_obj;
   Item   *the_item;
   Flags  *tmp_itemflags;
   int    flagnum;
   Strings holder;

   if (the_params->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "is_itemflag_set");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "is_itemflag_set", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   flag_param = the_params->next_param;
   if (flag_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "is_itemflag_set");
      *results = -1;
      return NULL;
   }

   if ((the_obj == NULL) || (!the_obj->is_an_item()))
   {
      mainstruct->log_error("Function can't be used on non-item objects", 
                                                      "is_itemflag_set");
      *results = -1;
      return NULL;
   }

   the_item = (Item *) the_obj;
   tmp_itemflags = the_item->get_itemflags();

   if ((flagnum = tmp_itemflags->
          get_by_name(flag_param->the_param.a_string->str_show(), 
                                              itemflagnames)) < 0)
   {
      mainstruct->log_error("First parameter not a valid itemflag", 
                                                 "is_itemflag_set");
      *results = -1;
      return NULL;
   }

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = (long) tmp_itemflags->get_flag(flagnum);
   
   return new_param;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** set_itemflag - sets an itemflag to true
 **
 **    format: set_itemflag(<this|primary|secondary>, <flagname>);
 **
 ***********************************************************************/

params *set_itemflag(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params  *flag_param;
   MudObject *the_obj;
   Item    *the_item;
   Flags   *tmp_itemflags;
   int     flagnum;
   Strings holder;

   if (the_params->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "set_itemflag");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "set_itemflag", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   flag_param = the_params->next_param;

   if (flag_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "set_itemflag");
      *results = -1;
      return NULL;
   }


   if ((the_obj == NULL) || (!the_obj->is_an_item()))
   {
      mainstruct->log_error("Function can't be used on non-item objects", 
                                                      "set_itemflag");
      *results = -1;
      return NULL;
   }

   the_item = (Item *) the_obj;
   tmp_itemflags = the_item->get_itemflags();

   if ((flagnum = tmp_itemflags->
          get_by_name(flag_param->the_param.a_string->str_show(), 
                                              itemflagnames)) < 0)
   {
      mainstruct->log_error("First parameter not a valid itemflag", 
                                                 "set_itemflag");
      *results = -1;
      return NULL;
   }

   *results = tmp_itemflags->set_flag(flagnum);
   
   return NULL;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** clr_itemflag - sets an itemflag to false
 **
 **    format: clr_itemflag(<primary|secondary|this|actor>, <flagname>);
 **
 ***********************************************************************/

params *clr_itemflag(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params  *flag_param;
   MudObject *the_obj;
   Item    *the_item;
   Flags   *tmp_itemflags;
   int     flagnum;
   Strings holder;

   if (the_params->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "set_itemflag");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "set_itemflag", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   flag_param = the_params->next_param;

   if (flag_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "set_itemflag");
      *results = -1;
      return NULL;
   }


   if ((the_obj == NULL) || (!the_obj->is_an_item()))
   {
      mainstruct->log_error("Function can't be used on non-item objects", 
                                                      "set_itemflag");
      *results = -1;
      return NULL;
   }

   the_item = (Item *) the_obj;
   tmp_itemflags = the_item->get_itemflags();

   if ((flagnum = tmp_itemflags->
          get_by_name(flag_param->the_param.a_string->str_show(), 
                                              itemflagnames)) < 0)
   {
      mainstruct->log_error("First parameter not a valid itemflag", 
                                                 "set_itemflag");
      *results = -1;
      return NULL;
   }

   *results = tmp_itemflags->clr_flag(flagnum);
   
   return NULL;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** ambush_individual - clones a mobile and places it in the player's room.  
 **                     sets the mobile attacking the player
 **     
 **
 **    format: ambush_individual(<mobile name>, <target name>)
 **
 ***********************************************************************/

params *ambush_individual(params *the_params, Specials *the_special, 
      Player *the_player, in_params *player_params, special_env *environment, 
                                                                int *results)
{
   params      *ambush_param;
   MudObject   *the_mudobj;
   Mobile      *the_mobile;
   Individual  *the_ind = NULL;
   Mobile      *cloned_mob;
   Object_List *the_dbase;
   MudObject   *tmp_ind;
   Strings     holder;

   ambush_param = the_params;
   if ((ambush_param == NULL) ||
       (ambush_param->param_type != VAR_TYPE_STRING))
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "ambush_individual");
      *results = -1;
      return NULL;
   }


   if (((the_mudobj = mainstruct->get_object(
                  ambush_param->the_param.a_string->str_show())) == NULL) ||
       (the_mudobj->get_type() != OBJ_TYPE_MOBILE))
   {
      mainstruct->log_error("First parameter not a valid mobile", 
                                                 "ambush_individual");
      *results = -1;
      return NULL;

   }
   the_mobile = (Mobile *) the_mudobj;

   ambush_param = ambush_param->next_param;
   if ((ambush_param == NULL) ||
       (ambush_param->param_type != VAR_TYPE_STRING))
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "ambush_individual");
      *results = -1;
      return NULL;
   }

   if ((the_ind = mainstruct->
           get_player(ambush_param->the_param.a_string->str_show())) == NULL)
   {
      if (((tmp_ind = mainstruct->get_object(
             ambush_param->the_param.a_string->str_show())) == NULL) ||
          (tmp_ind->get_type() != OBJ_TYPE_MOBILE))
      {
         mainstruct->
                log_error("Second parameter not a valid player or mobile", 
                                                 "ambush_individual");
         *results = -1;
         return NULL;
      }
      the_ind = (Individual *) tmp_ind;
   }

   the_dbase = mainstruct->get_dbase();
   cloned_mob = (Mobile *) the_dbase->clone_object((MudObject *) the_mobile);

   cloned_mob->add_clones(mainstruct->get_dbase(), mainstruct->get_log());
   cloned_mob->start_wield(mainstruct->get_dbase(), mainstruct->get_log());
   cloned_mob->start_wear(mainstruct->get_dbase(), mainstruct->get_log());

   cloned_mob->set_location(the_ind->get_loc(), cloned_mob->get_loc());
   

   holder.sprintf("%s erupts from the shadows, lunging at %s!\n", 
                cloned_mob->get_title(), the_ind->get_title());
   if (the_ind->get_type() == OBJ_TYPE_PLAYER)
   {
      ((Player *) the_ind)->send_plr("%s lunges at you from the shadows!\n",
                             cloned_mob->get_title());
      (the_ind->get_loc())->
                     send_location(holder.str_show(), (Player *) the_ind);
   }
   else
   {
      (the_ind->get_loc())->send_location(holder.str_show(), NULL);
   }

   *results = 1;
   if (cloned_mob->request_fight((Individual *) the_ind) < 0)
   {
      printf("fight refused!\n");
      return NULL;
   }
   cloned_mob->start_fight(the_ind);

   return NULL;

   the_player = NULL;
   the_special = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** damage_individual - causes damage to an individual, killing if
 **                     necessary
 **     
 **
 **    format: damage_individual(<this|primary|secondary>, <damage_number>)
 **
 ***********************************************************************/

params *damage_individual(params *the_params, Specials *the_special, 
      Player *the_player, in_params *player_params, special_env *environment, 
                                                                int *results)
{
   MudObject   *the_obj;
   Individual  *the_ind = NULL;
   Strings     holder;
   params      *second_param;
   int         damage;
   int         prev_health;
   int         damage_done = -1;
   Flags       *tmp_indflags;

   if ((the_params == NULL) || (the_params->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "damage_individual (special)");

      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "damage_individual", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (!the_obj->is_an_individual())
   {
      mainstruct->log_error("Damage can only be done to individuals.", 
                                        "damage_individual (special)");
      *results = -1;
      return NULL;
   }

   the_ind = (Individual *) the_obj;

   tmp_indflags = the_ind->get_indflags();
   
   if ((tmp_indflags->get_flag(INDFLAG_CORPSE)) ||
       (tmp_indflags->get_flag(INDFLAG_GHOST)))
   {
      mainstruct->log_error("Special attempted to directly damage non-living individual (ghost, corpse, etc)", "damage_individual");
      *results = -1;
      return NULL;
   }

   second_param = the_params->next_param;
   if ((second_param == NULL) ||
       (second_param->param_type != VAR_TYPE_INT))
   {
      mainstruct->log_error("Second parameter should be a number", 
                                                 "damage_individual");
      *results = -1;
      return NULL;
   }

   if ((damage = *second_param->the_param.an_int) < 1)
   {
      mainstruct->log_error("Damage must be greater than 0", 
                                                      "damage_individual");
      *results = -1;
      return NULL;
   }   

   prev_health = the_ind->get_health();
   
   if (the_ind->do_damage(damage, WEARFLAG_CHEST) <= 0)
   {
      *results = 0;
      return NULL;
   }

   if ((the_ind != NULL) && (damage_done < 0) && (the_player != NULL))
   {
      if (the_ind->get_health() <= 0)
         damage_done = prev_health;
      else
         damage_done = prev_health - the_ind->get_health();
   }

   *results = 0;
   return NULL;

   the_player = NULL;
   the_special = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** fight_player - individual indicated starts fight with player
 **     
 **
 **    format: fight_player(<this|primary|secondary|>)
 **
 ***********************************************************************/

params *fight_player(params *the_params, Specials *the_special, 
      Player *the_player, in_params *player_params, special_env *environment, 
                                                                int *results)
{
   MudObject   *the_obj;
   Individual  *the_ind = NULL;
   Strings     holder;

   if ((the_params == NULL) || (the_params->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "fight_player (special)");

      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "fight_player", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (((Individual *)the_obj) == ((Individual *) the_player))
   {
      mainstruct->log_error("Fights can't be started with the actor.", 
                                        "fight_player (special)");
      *results = -1;
      return NULL;
   } 

   if (!the_obj->is_an_individual())
   {
      mainstruct->log_error("Fights can only be started with individuals.", 
                                        "fight_player (special)");
      *results = -1;
      return NULL;
   }

   the_ind = (Individual *) the_obj;

   if (the_ind->request_fight((Individual *) the_player) < 0)
   {
      *results = -1;
      return NULL;
   }
   the_ind->start_fight(the_player);

   *results = 0;
   return NULL;

   the_special = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** is_indflag_set - returns true (1) if an indflag is set, false (0) 
 **                   if not
 **    format: is_indflag_set(<this|primary|secondary>, <flagname>);
 **
 ***********************************************************************/

params *is_indflag_set(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params      *flag_param;
   params      *new_param;
   MudObject   *the_obj;
   Individual  *the_ind;
   Flags       *tmp_indflags;
   int         flagnum;
   Strings     holder;

   if (the_params->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                               "is_indflag_set");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "is_indflag_set", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   flag_param = the_params->next_param;
   if (flag_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "is_indflag_set");
      *results = -1;
      return NULL;
   }

   if ((the_obj == NULL) || (!the_obj->is_an_individual()))
   {
      mainstruct->
         log_error("Function can't be used on non-individual objects", 
                                                      "is_indflag_set");
      *results = -1;
      return NULL;
   }

   the_ind = (Individual *) the_obj;
   tmp_indflags = the_ind->get_indflags();

   if ((flagnum = tmp_indflags->
          get_by_name(flag_param->the_param.a_string->str_show(), 
                                              indflagnames)) < 0)
   {
      mainstruct->log_error("First parameter not a valid indflag", 
                                                 "is_indflag_set");
      *results = -1;
      return NULL;
   }

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = (long) tmp_indflags->get_flag(flagnum);
   
   return new_param;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** set_indflag - sets an indflag to true
 **
 **    format: set_indflag(<this|primary|secondary>, <flagname>);
 **
 ***********************************************************************/

params *set_indflag(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params      *flag_param;
   MudObject   *the_obj;
   Individual  *the_ind;
   Flags       *tmp_indflags;
   int         flagnum;
   Strings     holder;

   if (the_params->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "is_individual");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "set_indflag", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }


   flag_param = the_params->next_param;

   if (flag_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "set_indflag");
      *results = -1;
      return NULL;
   }


   if ((the_obj == NULL) || (!the_obj->is_an_individual()))
   {
      mainstruct->
         log_error("Function can't be used on non-individual objects", 
                                                      "set_indflag");
      *results = -1;
      return NULL;
   }

   the_ind = (Individual *) the_obj;
   tmp_indflags = the_ind->get_indflags();

   if ((flagnum = tmp_indflags->
          get_by_name(flag_param->the_param.a_string->str_show(), 
                                              indflagnames)) < 0)
   {
      mainstruct->log_error("First parameter not a valid indflag", 
                                                 "set_indflag");
      *results = -1;
      return NULL;
   }

   *results = tmp_indflags->set_flag(flagnum);

   /* here is where we also set other flags linked to this indflag */
   if ((flagnum = INDFLAG_NIGHTVISION) &&
       (the_ind->get_type() == OBJ_TYPE_PLAYER))
      (((Player *) the_ind)->get_gameflags())->set_flag(GAMEFLAG_NIGHTVISION);
   

   return NULL;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}



/***********************************************************************
 ** clr_indflag - sets an indflag to false
 **
 **    format: clr_indflag(<this|primary|secondary>, <flagname>);
 **
 ***********************************************************************/

params *clr_indflag(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params      *flag_param;
   MudObject   *the_obj;
   Individual  *the_ind;
   Flags       *tmp_indflags;
   int         flagnum;
   Strings     holder;

   if (the_params->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "clr_indflag");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "clr_indflag", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }


   flag_param = the_params->next_param;

   if (flag_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "clr_indflag");
      *results = -1;
      return NULL;
   }


   if ((the_obj == NULL) || (!the_obj->is_an_individual()))
   {
      mainstruct->
         log_error("Function can't be used on non-individual objects", 
                                                      "clr_indflag");
      *results = -1;
      return NULL;
   }

   the_ind = (Individual *) the_obj;
   tmp_indflags = the_ind->get_indflags();

   if ((flagnum = tmp_indflags->
          get_by_name(flag_param->the_param.a_string->str_show(), 
                                              indflagnames)) < 0)
   {
      mainstruct->log_error("First parameter not a valid indflag", 
                                                 "clr_indflag");
      *results = -1;
      return NULL;
   }

   *results = tmp_indflags->clr_flag(flagnum);
   
   return NULL;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** trans_player - transports a player to another room, displaying the
 **                room information to them
 **    format: trans_player("<roomname>@<roomarea>");
 **
 ***********************************************************************/

params *trans_player(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Location  *the_loc;
   Strings   holder;
   params    *param_list;
   Flags     *tmp_gameflags;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be an object name", 
                                                 "trans_player");
      *results = -1;
      return NULL;
   }

   if ((the_obj = mainstruct->get_object(
                     param_list->the_param.a_string->str_show())) == NULL)
   {
      holder.sprintf("Location '%s' in special %s@%s doesn't exist, ", 
                 param_list->the_param.a_string->str_show(), 
                 the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "trans_player");
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_LOCATION)
   {
      holder.sprintf("Object '%s' in special %s@%s not a Location type", 
                 param_list->the_param.a_string->str_show(), 
                 the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "trans_player");
      *results = -1;
      return NULL;
   }

   the_loc = (Location *) the_obj;

   if (the_player->set_location(the_loc, the_player->get_loc()) == -1)
   {
      holder.sprintf("Could not set player loc to '%s' in special %s@%s", 
                 param_list->the_param.a_string->str_show(), 
                 the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "trans_player");
      *results = -1;
      return NULL;
   }

   tmp_gameflags = the_player->get_gameflags();
   the_loc->show_location(the_player, 
                                    tmp_gameflags->get_flag(GAMEFLAG_BRIEF));
      
   *results = 1;   
   return NULL;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
   player_params = NULL;
}


/***********************************************************************
 ** add_ability - adds an ability to the player indicated
 **   format: add_ability_player(<primary|secondary|this|actor>, 
 **                                                      <ability_name>);
 **
 ***********************************************************************/
params *add_ability(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   Ability   *the_ability;
   params    *param_list;
   MudObject *the_obj;
   Player    *target_plr;
   

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "add_ability");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "add_ability_indflag", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      mainstruct->log_error("First parameter must point to a player object", 
                                                 "add_ability");
      *results = -1;
      return NULL;
     
   }

   target_plr = (Player *) the_obj;

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "add_ability");
      *results = -1;
      return NULL;
   }

   if ((the_ability = mainstruct->get_ability(
                      param_list->the_param.a_string->str_show())) == NULL)
   {
      mainstruct->log_error("Second parameter not a valid ability", 
                                                 "add_ability");
      *results = -1;
      return NULL;

   }
 
   target_plr->add_rank(the_ability->get_name(), the_ability->get_type(),
                                                            0,0);
   *results = 0;

   return NULL;

   /* to remove the silly warnings on compile */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
}


/***********************************************************************
 ** remove_ability - removes an ability from the player indicated
 **   format: add_ability_player(<primary|secondary|this|actor>, 
 **                                                      <ability_name>);
 **
 ***********************************************************************/
params *remove_ability(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   Ability   *the_ability;
   params    *param_list;
   MudObject *the_obj;
   Player    *target_plr;
   

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "remove_ability");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "remove_ability", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      mainstruct->log_error("First parameter must point to a player object", 
                                                 "remove_ability");
      *results = -1;
      return NULL;
     
   }

   target_plr = (Player *) the_obj;

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "remove_ability");
      *results = -1;
      return NULL;
   }

   if ((the_ability = mainstruct->get_ability(
                      the_params->the_param.a_string->str_show())) == NULL)
   {
      mainstruct->log_error("Second parameter not a valid ability", 
                                                 "remove_ability");
      *results = -1;
      return NULL;

   }
 
   target_plr->remove_rank(the_ability->get_name());
   *results = 0;

   return NULL;

   /* to remove the silly warnings on compile */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
}


/***********************************************************************
 ** has_ability - returns 1 if the player has the ability, 0 if no
 **   format: has_ability(<primary|secondary|this|actor>, <ability_name>);
 **
 ***********************************************************************/
params *has_ability(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params   *new_param;
   params    *param_list;
   MudObject *the_obj;
   Player    *target_plr;
   

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "has_ability");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "has_ability", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      mainstruct->log_error("First parameter must point to a player object", 
                                                 "has_ability");
      *results = -1;
      return NULL;
     
   }

   target_plr = (Player *) the_obj;

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "has_ability");
      *results = -1;
      return NULL;
   }

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = (long ) target_plr->find_ability(
                              the_params->the_param.a_string->str_show());
 
   *results = 0;

   return new_param;

   /* to remove the silly warnings on compile */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
}


/***********************************************************************
 ** boot_off - kicks the player off the mud
 **
 **    format: boot_off();
 **
 ***********************************************************************/
params *boot_off(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{

   the_player->set_off();
   *results = 1;

   return NULL;

   /* get rid of all those compile warnings */
   the_special = NULL;
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** award_quests - award a quest to the actor
 **    format: award_quest(<questname>);
 **
 ***********************************************************************/

params *award_quest(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   Quest       *the_quest;
   Strings     holder;
   params      *param_list;
   Object_List *the_dbase;

   if (!the_config.conf_flags->get_flag(CF_USEQUESTS))
   {
      mainstruct->log_error("Attempted to award quest with quests turned off.",
                                                "award_quest");
      *results = -1;
      return NULL;
   }

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "award_quest");
      *results = -1;
      return NULL;
   }

   the_dbase = mainstruct->get_dbase();
   if ((the_quest = the_dbase->get_quest_obj(
            param_list->the_param.a_string->str_show())) == NULL)
   {
      holder.sprintf("Attempted to add quest '%s', does not exist.", 
                    param_list->the_param.a_string->str_show());
      mainstruct->log_error(holder.str_show(), "award_quest");
      *results = -1;
      return NULL;
   }

   if (the_player->find_quest(the_quest->get_name()))
   {
      the_player->send_plr(
         "Congratulations!  You have completed the %s quest...again.\n", 
                                                    the_quest->get_name());
      holder.sprintf(
        "&+Y[&+M%s&+W has completed the &+Y%s&+W quest&+y (again)&+Y]&*\n",
                           the_player->get_title(), the_quest->get_name());

      *results = 1;
   }
   else
   {
      the_player->send_plr(
         "Congratulations!  You have completed the %s quest!\n", 
                                                   the_quest->get_name());
      the_player->add_quest(the_quest->get_name());
      holder.sprintf("&+Y[&+M%s&+W has completed the &+Y%s&+W quest&+Y]&*\n",
                           the_player->get_title(), the_quest->get_name());

      *results = 1;

   }

   mainstruct->send_all_players(holder.str_show(), NULL, 
                                                     ADMINFLAG_SEEQUESTS);
   return NULL; 

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** advance_experience - for abilities, advances the experience of both
 **                      the global experience and the ability exp. for
 **                      the actor of the special
 **    format: advance_experience(<primary|secondary|this|actor>, 
 **                               <abilityname>, <success|failure>, 
 **                                                          <number>);
 **
 ***********************************************************************/

params *advance_experience(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   Strings   holder;
   params    *param_list;
   Ability   *the_ability;
   MudObject *the_obj;
   Player    *target_plr;
   int       difficulty;
   int       divide_by;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "advance_experience");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "advance_experience", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      mainstruct->log_error("Attempted to assign experience to non-player.",
                                                 "advance_experience");
      *results = -1;
      return NULL;
   }

   target_plr = (Player *) the_obj;

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "advance_experience");
      *results = -1;
      return NULL;
   }

   if ((the_ability = mainstruct->get_ability(param_list->
                                  the_param.a_string->str_show())) == NULL)
   {
      holder.sprintf("Ability '%s' not found, special %s.\n",
        param_list->the_param.a_string->str_show(), the_special->get_name());
      mainstruct->
            log_error(holder.str_show(), "advance_experience (special)");
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Third parameter should be a string", 
                                                 "advance_experience");
      *results = -1;
      return NULL;
   }

   if (!STRCASECMP(param_list->the_param.a_string->str_show(), "success"))
      divide_by = 1;
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                              "failure"))
      divide_by = 2;
   else
   {
      holder.sprintf("Invalid status '%s' for param two, special %s.\n",
        param_list->the_param.a_string->str_show(), the_special->get_name());
      mainstruct->
            log_error(holder.str_show(), "advance_experience (special)");
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("Expected integer as fourth parameter, not found",
                                      "advance_experience (special)");
      *results = -1;
      return NULL;
   }

   difficulty = (int) *(param_list->the_param.an_int);

   target_plr->increase_rank_exp(difficulty, the_ability, divide_by);

      
   *results = 1;   
   return NULL;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
   player_params = NULL;
}


/***********************************************************************
 ** push_prompt - pushes a particular prompt onto a player
 **    format: push_prompt(<this|primary|secondary|actor>, <promptname>, 
 **                                                      <promptstr>);
 **
 ***********************************************************************/

params *push_prompt(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Player    *the_plr;
   Strings   holder;
   Strings   promptname;
   params    *param_list;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "push_prompt");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "push_prompt", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      mainstruct->log_error("First parameter is not a player", 
                                             "push_prompt (special)");
      *results = -1;
      return NULL;
   
   }

   the_plr = (Player *) the_obj;

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                              "push_prompt (special)");
      *results = -1;
      return NULL;
   }

   promptname = param_list->the_param.a_string->str_show();

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Third parameter should be a string", 
                                              "push_prompt (special)");
      *results = -1;
      return NULL;
   }

   the_plr->push_prompt(promptname.str_show(), 
                        param_list->the_param.a_string->str_show());

   *results = 0;   
   return NULL;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** pop_prompt - removes a particular prompt from a player
 **    format: pop_prompt(<this|primary|secondary|actor>, <promptname>);
 **
 ***********************************************************************/

params *pop_prompt(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Player    *the_plr;
   Strings   holder;
   Strings   promptname;
   params    *param_list;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "pop_prompt");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "pop_prompt", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      mainstruct->log_error("First parameter is not a player", 
                                             "pop_prompt (special)");
      *results = -1;
      return NULL;
   
   }

   the_plr = (Player *) the_obj;

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                              "pop_prompt (special)");
      *results = -1;
      return NULL;
   }

   promptname = param_list->the_param.a_string->str_show();

   if (promptname.str_cmp("main"))
   {
      holder.sprintf("Prompt name can't be 'main', special %s", 
                                        the_special->get_name());
      mainstruct->log_error(holder.str_show(), "pop_prompt (special)");
      *results = -1;
      return NULL;
   }

   if (the_plr->remove_prompt(promptname.str_show()) <= 0)
   {
      holder.sprintf("Prompt name '%s' not found, special %s", 
                            promptname.str_show(), the_special->get_name());
      mainstruct->log_error(holder.str_show(), "pop_prompt (special)");
      *results = -1;
      return NULL;
   }

   *results = 0;   
   return NULL;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** start_busy - marks the player as busy, raising the busy flag
 **    format: start_busy(<this|primary|secondary|actor>);
 **
 ***********************************************************************/

params *start_busy(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *param_list;
   Flags     *tmp_gameflags;
   Player    *the_plr;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "start_busy");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "start_busy", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      holder.sprintf("%s is not a player.", the_obj->get_name());
      mainstruct->log_error(holder.str_show(), "start_busy (special)");
      *results = -1;
      return NULL;

   }

   the_plr = (Player *) the_obj;

   tmp_gameflags = the_plr->get_gameflags();
   if (tmp_gameflags->get_flag(GAMEFLAG_BUSY))
   {
      holder.sprintf("Busy gameflag already set for player %s", 
                                       the_plr->get_title());
      mainstruct->log_error(holder.str_show(), "start_busy (special)");
      *results = -1;
      return NULL;

   }

   tmp_gameflags->set_flag(GAMEFLAG_BUSY);
   *results = 1;
   return NULL;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** end_busy - marks the player as not busy, clearing the busy flag
 **    format: end_busy(<this|primary|secondary|actor>);
 **
 ***********************************************************************/

params *end_busy(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *param_list;
   Flags     *tmp_gameflags;
   Player    *the_plr;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "end_busy");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "end_busy", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      holder.sprintf("%s is not a player.", the_obj->get_name());
      mainstruct->log_error(holder.str_show(), "end_busy (special)");
      *results = -1;
      return NULL;

   }

   the_plr = (Player *) the_obj;

   tmp_gameflags = the_plr->get_gameflags();
   if (!tmp_gameflags->get_flag(GAMEFLAG_BUSY))
   {
      holder.sprintf("Busy gameflag not set for player %s", 
                                       the_plr->get_title());
      mainstruct->log_error(holder.str_show(), "end_busy (special)");
      *results = -1;
      return NULL;

   }

   tmp_gameflags->clr_flag(GAMEFLAG_BUSY);
   *results = 1;
   return NULL;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** decrement_num_of - decreases the number of this object by the number
 **                    indicated, returns the number left
 **    format: decrement_number_of(<this|primary|secondary>, <number>);
 **
 ***********************************************************************/

params *decrement_number_of(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Merger    *the_merger;
   Strings   holder;
   params    *new_param;
   params    *param_list;
   int       the_num;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "decrement_number_of");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
       get_indicated_param(param_list->the_param.a_string->str_show(), 
       the_special, "decrement_number_of", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (!the_obj->is_merger())
   {
      holder.sprintf("%s is not a valid merger type, this special only "
                     "works with mergers.\n", 
                            param_list->the_param.a_string->str_show());
      mainstruct->
            log_error(holder.str_show(), "decrement_number_of (special)");
      *results = -1;
      return NULL;
   }

   the_merger = (Merger *) the_obj;
   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("Second parameter should be a number", 
                                        "decrement_number_of (special)");
      *results = -1;
      return NULL;
   }

   the_num = *(param_list->the_param.an_int);

   if (the_merger->get_number_of() < the_num)
      the_merger->set_number_of(0);
   else
      the_merger->set_number_of(the_merger->get_number_of() - the_num);

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = (long) the_merger->get_number_of();

   *results = 0;   
   return new_param;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** increment_num_of - increases the number of this object by the number
 **                    indicated, returns the number total
 **    format: decrement_number_of(<this|primary|secondary>, <number>);
 **
 ***********************************************************************/

params *increment_number_of(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Merger    *the_merger;
   Strings   holder;
   params    *new_param;
   params    *param_list;
   int       the_num;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "increment_number_of");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
        get_indicated_param(param_list->the_param.a_string->str_show(), 
        the_special, "increment_number_of", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (!the_obj->is_merger())
   {
      holder.sprintf("%s is not a valid merger type, this special only "
                     "works with mergers.\n", 
                            param_list->the_param.a_string->str_show());
      mainstruct->
            log_error(holder.str_show(), "increment_number_of (special)");
      *results = -1;
      return NULL;
   }

   the_merger = (Merger *) the_obj;
   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("Second parameter should be a number", 
                                        "increment_number_of (special)");
      *results = -1;
      return NULL;
   }

   the_num = *(param_list->the_param.an_int);

   the_merger->set_number_of(the_merger->get_number_of() + the_num);

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = (long) the_merger->get_number_of();

   *results = 0;   
   return new_param;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** get_number_of - gets the number of this merger item (mergers only)
 **    format: get_number_of(<this|primary|secondary>);
 **
 ***********************************************************************/

params *get_number_of(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Merger    *the_merger;
   Strings   holder;
   params    *new_param;
   params    *param_list;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "is_individual");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "get_number_of", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (!the_obj->is_merger())
   {
      holder.sprintf("%s is not a valid merger type, this special only "
                     "works with mergers.\n", 
                            param_list->the_param.a_string->str_show());
      mainstruct->
            log_error(holder.str_show(), "get_number_of (special)");
      *results = -1;
      return NULL;
   }

   the_merger = (Merger *) the_obj;

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = (long) the_merger->get_number_of();

   *results = 0;   
   return new_param;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** set_door_state - sets the state of the door, open, closed, etc
 **    format: set_door_state(<this|primary|secondary>, 
 **                           <open|closed|locked|mlocked>);
 **
 ***********************************************************************/

params *set_door_state(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Door      *the_door;
   Strings   holder;
   params    *param_list;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "set_door_state");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "set_door_state", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }


   if (the_obj->get_type() != OBJ_TYPE_DOOR)
   {
      mainstruct->log_error("First parameter is not a door type", 
                                             "set_door_state (special)");
      *results = -1;
      return NULL;
   
   }

   the_door = (Door *) the_obj;

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                              "set_door_state (special)");
      *results = -1;
      return NULL;
   }

   if (!STRCASECMP(param_list->the_param.a_string->str_show(), "open"))
   {
      the_door->set_door_state(DOOR_OPEN);  
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                               "closed"))
   {
      the_door->set_door_state(DOOR_CLOSED);  
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                               "locked"))
   {
      the_door->set_door_state(DOOR_LOCKED);  
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                               "mlocked"))
   {
      the_door->set_door_state(DOOR_MLOCKED);  
   }
   else
   {
      holder.sprintf("Parameter '%s' not a recognized door state", 
                       param_list->the_param.a_string->str_show());
      mainstruct->
            log_error(holder.str_show(), "set_door_state (special)");
      *results = -1;
      return NULL;
   }

   *results = 0;   
   return NULL;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** get_door_state - gets the state of the door, open, closed, etc
 **    format: get_door_state(<this|primary|secondary>);
 **
 ***********************************************************************/

params *get_door_state(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Door      *the_door;
   Strings   holder;
   params    *param_list;
   params    *new_param;
   Strings   tmp_return;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "get_door_state");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "get_door_state", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_DOOR)
   {
      mainstruct->log_error("First parameter is not a door type", 
                                             "get_door_state (special)");
      *results = -1;
      return NULL;
   
   }

   the_door = (Door *) the_obj;

   switch(the_door->get_door_state())
   {
      case DOOR_OPEN:
         tmp_return = "Open";
         break;
      case DOOR_CLOSED:
         tmp_return = "Closed";
         break;
      case DOOR_LOCKED:
         tmp_return = "Locked";
         break;
      case DOOR_MLOCKED:
         tmp_return = "MLocked";
         break;
      default:
         fault("Error: unrecognized door state!\n");
         return NULL;
   }
   new_param = new_params();

   new_param->param_type = VAR_TYPE_STRING;
   new_param->the_param.a_string = new_Strings(tmp_return.str_show());

   the_special = NULL;
   environment = NULL;
   the_player = NULL;
   *results = 0;
   return new_param;
}


/***********************************************************************
 ** decrement_counter - makes the counter one less
 **    format: decrement_counter();
 **
 ***********************************************************************/
params *decrement_counter(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   environment->counter = environment->counter - 1;
   *results = 1;

   return NULL;

   /* get rid of all those compile warnings */
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
   the_special = NULL;
}


/***********************************************************************
 ** increment_counter - makes the counter one more
 **    format: decrement_counter();
 **
 ***********************************************************************/
params *increment_counter(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   environment->counter = environment->counter - 1;
   *results = 1;

   return NULL;

   /* get rid of all those compile warnings */
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
   the_special = NULL;
}


/***********************************************************************
 ** get_counter - returns the counter value of a specified obj/special
 **    format: get_counter(<primary|secondary|this>, "specialname");
 **
 ***********************************************************************/
params *get_counter(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params         *new_param;
   params         *param_list;
   params         *second_param;
   MudObject      *the_obj;
   special_holder *obj_special;
   special_env    *the_env;
   Strings        holder;

   param_list = the_params;

   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "get_counter (special)");

      *results = -1;
      return NULL;
   }

   if (((second_param = param_list->next_param) == NULL) ||
        (second_param->param_type != VAR_TYPE_STRING))
   {
      mainstruct->
            log_error("Expected string as second parameter, not found", 
                                                "get_counter (special)");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "get_counter", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (!STRCASECMP(second_param->the_param.a_string->str_show(), "this"))
   {
      the_env = environment;
   }
   else if ((obj_special = the_obj->
        find_special(second_param->the_param.a_string->str_show())) == NULL) 
   {
      holder.sprintf("Special with trigger '%s' not found in object '%s'\n", 
          second_param->the_param.a_string->str_show(), the_obj->get_name());
      mainstruct->
            log_error(holder.str_show(), "get_counter (special)");
      *results = -1;
      return NULL;
   }
   else
   {
      the_env = &(obj_special->environment);
   }

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = the_env->counter;

   *results = 1;

   return new_param;

   /* get rid of all those compile warnings */
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
   the_special = NULL;
}



/***********************************************************************
 ** set_counter - returns the counter value of a specified obj/special
 **    format: set_counter(<primary|secondary|this>, "specialname", <value>);
 **
 ***********************************************************************/
params *set_counter(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params         *param_list;
   params         *second_param;
   MudObject      *the_obj;
   special_holder *obj_special;
   special_env    *the_env;
   Strings        holder;
   int            the_num;

   param_list = the_params;

   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "set_counter (special)");

      *results = -1;
      return NULL;
   }

   if (((second_param = param_list->next_param) == NULL) ||
        (second_param->param_type != VAR_TYPE_STRING))
   {
      mainstruct->
            log_error("Expected string as second parameter, not found", 
                                                "set_counter (special)");
      *results = -1;
      return NULL;
   }

   if ((second_param->next_param == NULL) ||
       (second_param->next_param->param_type != VAR_TYPE_INT))
   {
      mainstruct->
            log_error("Expected number as third parameter, not found", 
                                                "set_counter (special)");
      *results = -1;
      return NULL;
   }
   the_num = (int) *(second_param->next_param->the_param.an_int);

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "set_counter", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (!STRCASECMP(second_param->the_param.a_string->str_show(), "this"))
   {
      the_env = environment;
   }
   else if ((obj_special = the_obj->
        find_special(second_param->the_param.a_string->str_show())) == NULL) 
   {
      holder.sprintf("Special with trigger '%s' not found in object '%s'\n", 
          second_param->the_param.a_string->str_show(), the_obj->get_name());
      mainstruct->
            log_error(holder.str_show(), "get_a_counter (special)");
      *results = -1;
      return NULL;
   }
   else
   {
      the_env = &(obj_special->environment);
   }

   the_env->counter = the_num;


   *results = 1;

   return NULL;

   /* get rid of all those compile warnings */
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
   the_special = NULL;
}


/***********************************************************************
 ** set_param - sets an object to one of the params, either 'primary', 
 **             'secondary', or 'this' parameter
 **
 **    format: set_param(<which param>, <object name>);
 **
 ***********************************************************************/
params *set_param(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params    *param_list;
   params    *second_param;
   MudObject *new_obj;
   Strings   holder;   

   param_list = the_params;


   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "set_param (special)");

      *results = -1;
      return NULL;
   }

   if (((second_param = param_list->next_param) == NULL) ||
        (second_param->param_type != VAR_TYPE_STRING))
   {
      mainstruct->
            log_error("Expected string as second parameter, not found", 
                                                "set_param (special)");
      *results = -1;
      return NULL;
   }

   if ((new_obj = mainstruct->
        get_object(second_param->the_param.a_string->str_show())) == NULL) 
   {
      holder.sprintf("Object '%s' not found\n", 
                            second_param->the_param.a_string->str_show());
      mainstruct->
            log_error(holder.str_show(), "set_param (special)");
      *results = -1;
      return NULL;
   }

   if (!STRCASECMP(param_list->the_param.a_string->str_show(), "primary"))
   {
      player_params->primary_obj = new_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "secondary"))
   {
      player_params->secondary_obj = new_obj;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                            "this"))
   {
      player_params->this_obj = new_obj;
   }
   else
   {
      holder.sprintf("%s is not a valid first parameter.  "
                     "Must be either 'primary', 'secondary', or 'this'\n", 
                            param_list->the_param.a_string->str_show());
      mainstruct->
            log_error(holder.str_show(), "set_param (special)");
      *results = -1;
      return NULL;
   }

   *results = 0;

   /* to eliminate the compile warnings */
   the_player = NULL;
   the_special = NULL;
   player_params = NULL;
   environment = NULL;
   return NULL;   
}


/***********************************************************************
 ** set_store_int - sets a special attribute not of this special
 **                      that is attached to a specified object
 **
 **    format: set_store_int(<new value>);
 **
 ***********************************************************************/
params *set_store_int(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *param_list;
   Strings holder;   

   param_list = the_params;


   if (param_list->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("Expected integer as first parameter, not found",
                                      "set_store_int (special)");
      *results = -1;
      return NULL;
   }

   environment->store_int = (int) *(param_list->the_param.an_int);

   /* to eliminate the compile warnings */
   the_player = NULL;
   the_special = NULL;
   player_params = NULL;
   return NULL;   
}


/***********************************************************************
 ** get_store_int - returns the storage int value
 **    format: get_store_int();
 **
 ***********************************************************************/
params *get_store_int(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *new_param;

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = environment->store_int;

   *results = 1;

   return new_param;

   /* get rid of all those compile warnings */
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
   the_special = NULL;
}


/***********************************************************************
 ** get_target_str - gets the string in the target_str slot, usually the
 **                  string that the player input
 **
 ***********************************************************************/
params *get_target_str(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params   *new_param;

   new_param = new_params();

   new_param->param_type = VAR_TYPE_STRING;
   new_param->the_param.a_string = 
                             new_Strings(environment->target_str.str_show());

   the_player = NULL;
   the_params = NULL;
   the_special = NULL;
   player_params = NULL;
   environment = NULL;
   *results = 0;

   return new_param;
}


/***********************************************************************
 ** increment_number - increments the number given one and returns the
 **                    new number
 **   increment_number(<number>, <offset>);
 **
 ***********************************************************************/
params *increment_number(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *param_list;
   Strings holder;   
   long    first_num;
   long    offset;
   params   *new_param;

   param_list = the_params;

   /* this is the number to be incremented */
   if (param_list->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("Expected integer as first parameter, not found", 
                                      "increment_number (special)");
      *results = -1;
      return NULL;
   }

   first_num = *(param_list->the_param.an_int);
   
   param_list = param_list->next_param;

   /* this is the offset to increment it */
   if (param_list->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("Expected integer as second parameter, not found", 
                                      "increment_number (special)");
      *results = -1;
      return NULL;
   }

   offset = *(param_list->the_param.an_int);
   
   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = first_num + offset;

   return new_param;

   /* to eliminate the compile warnings */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
}


/***********************************************************************
 ** decrement_number - decrements the number given one and returns the
 **                    new number
 **   decrement_number(<number>, <offset>);
 **
 ***********************************************************************/
params *decrement_number(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params *param_list;
   Strings holder;   
   long    first_num;
   long    offset;
   params   *new_param;

   param_list = the_params;

   /* this is the number to be decremented */
   if (param_list->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("Expected integer as first parameter, not found", 
                                      "decrement_number (special)");
      *results = -1;
      return NULL;
   }

   first_num = *(param_list->the_param.an_int);
   
   param_list = param_list->next_param;

   /* this is the offset to decrement it */
   if (param_list->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("Expected integer as second parameter, not found", 
                                      "decrement_number (special)");
      *results = -1;
      return NULL;
   }

   offset = *(param_list->the_param.an_int);
   
   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = first_num - offset;

   return new_param;

   /* to eliminate the compile warnings */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
}

/***********************************************************************
 ** get_random - gets a random number between 0 and the number specified
 **              as the parameter
 **
 **    format: get_random(<number>);
 **
 ***********************************************************************/

params *get_random(params *the_params, Specials *the_special, 
      Player *the_player, in_params *player_params, special_env *environment, 
                                                                int *results)
{
   params *int_param;
   params *new_param;
   long   the_num;
   long   rand_results;

   int_param = the_params;
   if (int_param->param_type != VAR_TYPE_INT)
   {
      mainstruct->log_error("First parameter should be a number", 
                                                 "get_random");
      *results = -1;
      return NULL;
   }

   the_num = *(int_param->the_param.an_int);
   
   rand_results = 1+ (long) (((float) the_num) *(rand()/(RAND_MAX+1.0)));
   
   new_param = new_params();
   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = rand_results;
   
   *results = 1;
   return new_param;

   the_special = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** select_rand_param - selects one of the parameters given and returns
 **                     it, random param
 **     
 **
 **    format: select_rand_param(<param>, <param>, <param>, .....);
 **
 ***********************************************************************/

params *select_rand_param(params *the_params, Specials *the_special, 
      Player *the_player, in_params *player_params, special_env *environment, 
                                                                int *results)
{
   params *list_param;
   params *new_param;
   int    num_params = 0;
   int    rand_results;
   int    i;

   list_param = the_params;
   while (list_param != NULL)
   {
      num_params++;
      list_param = list_param->next_param;
   }
   
   rand_results = (int) (((float) num_params) *(rand()/(RAND_MAX+1.0)));

   list_param = the_params;
   for (i=0; i<rand_results; i++)
   {
      list_param = list_param->next_param;
   }

   new_param = new_params();
   new_param->param_type = list_param->param_type;
   switch(new_param->param_type)
   {
      case VAR_TYPE_STRING:
         new_param->the_param.a_string = 
                 new_Strings(list_param->the_param.a_string->str_show());
         break;

      case VAR_TYPE_INT:
         new_param->the_param.an_int = new_long();
         *(new_param->the_param.an_int) = *(list_param->the_param.an_int);
         break;

      default: 
         mainstruct->log_error("Parameter must be a string or a number", 
                                                 "select_rand_param");
         *results = -1;
         return NULL; 
         break;
   }

   *results = 1;
   return new_param;

   the_special = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** ind_holds - returns if the individual indicated holds a specified
 **             object or type of object
 **    format: ind_holds(<this|primary|secondary|player>, <object>);
 **
 **    possible types: fire, lit, any object name
 **
 ***********************************************************************/

params *ind_holds(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Strings   holder;
   params    *new_param;
   params    *param_list;
   int       the_num;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "ind_holds");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "ind_holds", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }


   if (the_obj == NULL)
   {
      holder.sprintf("Parameter points to a NULL object");
      mainstruct->
            log_error(holder.str_show(), "ind_holds (special)");
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "ind_holds");
      *results = -1;
      return NULL;
   }


   if (!STRCASECMP(param_list->the_param.a_string->str_show(), "fire"))
   {
      if (the_obj->find_fire() == NULL)
         the_num = 0;
      else
         the_num = 1;
   }
   else if (!STRCASECMP(param_list->the_param.a_string->str_show(), "lit"))
   {
      the_num = the_obj->contains_lit();
   }
   else
   {
      LinkedList *obj_inv;
      MudObject  *tmp_obj;
      Strings    this_obj;
      char       *areaname;
      char       *objname;

      the_num = 0;

      obj_inv = the_obj->get_inventory();
      obj_inv->reset_current();
      tmp_obj = obj_inv->get_next();
      while (tmp_obj != NULL)
      {

	 /* get the name and area of this object */
         objname = param_list->the_param.a_string->str_show();
         areaname = objname;
         while ((*areaname) && (*areaname != '@'))
            areaname++;
         if (*areaname)
            areaname++;

         if (tmp_obj->get_parent() != NULL)
            this_obj = tmp_obj->get_parent();
  
         if ((!STRNCASECMP(objname, this_obj.str_show(), 
                                              this_obj.str_len())) &&
             (!STRCASECMP(areaname, tmp_obj->get_area())))
            the_num = 1;
         tmp_obj = obj_inv->get_next();
      }
   }
   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();
   *(new_param->the_param.an_int) = (long) the_num;

   *results = 0;   
   return new_param;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}







/***********************************************************************
 ** is_string_eq - is the first string equal to any of the strings that 
 **                follow it
 **   format: is_string_eq(<string1>, <string2>, <string3>, ...);
 **
 ***********************************************************************/
params *is_string_eq(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params  *param_list;
   params  *new_param;
   Strings compare_str;

   param_list = the_params;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "is_string_eq");
      *results = -1;
      return NULL;
   }

   compare_str = param_list->the_param.a_string->str_show();

   param_list = the_params->next_param;
   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "is_string_eq");
      *results = -1;
      return NULL;
   }

   new_param = new_params();
   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();

   while (param_list != NULL)
   {
      if (!STRCASECMP(param_list->the_param.a_string->str_show(), 
                                                 compare_str.str_show()))
      {
         *(new_param->the_param.an_int) = 1;
         return new_param;
      }

      param_list = param_list->next_param;
      if ((param_list != NULL) && (param_list->param_type != VAR_TYPE_STRING))
      {
         mainstruct->log_error("All parameters must be strings", 
                                                 "is_string_eq");
         *results = -1;
         return NULL;
      }
   }

   *results = 0;
   *(new_param->the_param.an_int) = 0;
   return new_param;

   /* to remove the silly warnings on compile */
   the_player = NULL;
   player_params = NULL;
   the_special = NULL;
   environment = NULL;
}




/***********************************************************************
 ** test_for_success - for abilities, tests if the ability succeeds this
 **                    round.  Returns the success index (0-100)
 **    format: test_for_success(<abilityname>, <difficulty>);
 **
 ***********************************************************************/

params *test_for_success(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   Strings   holder;
   params    *new_param;
   params    *param_list;
   Ability   *the_ability;
   int       difficulty;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                                 "test_for_success");
      *results = -1;
      return NULL;
   }

   if ((the_ability = mainstruct->get_ability(param_list->
                                  the_param.a_string->str_show())) == NULL)
   {
      holder.sprintf("Ability '%s' not found, special %s.\n",
        param_list->the_param.a_string->str_show(), the_special->get_name());
      mainstruct->
            log_error(holder.str_show(), "test_for_success (special)");
      *results = -1;
      return NULL;
   }

   param_list = param_list->next_param;

   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_INT))
   {
      mainstruct->log_error("Expected integer as second parameter, not found",
                                      "test_for_success (special)");
      *results = -1;
      return NULL;
   }

   difficulty = (int) *(param_list->the_param.an_int);

   new_param = new_params();

   new_param->param_type = VAR_TYPE_INT;
   new_param->the_param.an_int = new_long();

   *(new_param->the_param.an_int) = the_ability->test_success(the_player,
                                                               difficulty);
      
   *results = 1;   
   return new_param;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
   player_params = NULL;
}




/***********************************************************************
 ** assign - assigns a value to a variable.
 **    format: <variable> = <value>;
 **
 ***********************************************************************/

params *assign(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   vars    *tmp_var;
   char    *typname = NULL;
   Strings holder;

   tmp_var = find_var(environment, environment->stack_pointer->get_var_name());

   if (the_params == NULL)
   {
      holder.sprintf("No parameter passed into assignment of '%s'.\n", 
                                  environment->stack_pointer->get_var_name());
      mainstruct->log_error(holder.str_show(), "assign");
      *results = 0;
      return NULL;
   }

   /* if the variable has been assigned a type already and we are assigning
      it another type, raise an error */
   if ((tmp_var->vartype != 0) && 
       (tmp_var->vartype != the_params->param_type))
   {
      switch(tmp_var->vartype)
      {
         case VAR_TYPE_STRING:
            typname = "string";
            break;

         case VAR_TYPE_INT:
            typname = "int";
            break;

         case VAR_TYPE_OBJNAME:
            typname = "object name";
            break;

         case VAR_TYPE_CODE:
         case VAR_TYPE_GOTO:
            mainstruct->log_error("Assignment object type was code or goto.",
                                         "assign");
            *results = 0;
            return NULL;
            break;

         default:
            break;
      }
 
      holder.sprintf("Assignment to %s must be of type %s.", 
               environment->stack_pointer->get_var_name(), typname);
      mainstruct->log_error(holder.str_show(), "assign");
      *results = 0;
      return NULL;
   }

   /* if it has not been assigned a type yet */
   if (tmp_var->vartype == 0)
   {
      tmp_var->vartype = the_params->param_type;
   }

   switch(tmp_var->vartype)
   {
      case VAR_TYPE_STRING:
         tmp_var->the_ptr.a_string = 
             new_Strings(the_params->the_param.a_string->str_show());
         break;

      case VAR_TYPE_INT:
         tmp_var->the_ptr.an_int = new_long();
         *(tmp_var->the_ptr.an_int) = *(the_params->the_param.an_int);

         break;

      case VAR_TYPE_OBJNAME:
         tmp_var->the_ptr.a_string = 
              new_Strings(the_params->the_param.a_string->str_show());
         break;

      default:
         printf("Error in logic for assignment function.\n");
         return NULL;
         break;
   }
   
   *results = 1;
   return NULL;

   the_params = NULL;
   the_special = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}


/***********************************************************************
 ** get_loc_by_dir - returns the location string based on the direction
 **                  passed in
 **    format: get_loc_by_dir("<currentlocname>@<area>", "<the_direction>");
 **
 ***********************************************************************/

params *get_loc_by_dir(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject *the_obj;
   Location  *cur_loc;
   Strings   holder;
   Strings   locname;
   params    *param_list;
   Flags     *tmp_gameflags;
   int       dir_num;
   params    *new_param;

   param_list = the_params;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be an object name", 
                                                 "get_loc_by_dir");
      *results = -1;
      return NULL;
   }

   new_param = new_params();
   new_param->param_type = VAR_TYPE_STRING;

   if ((the_obj = mainstruct->get_object(
                     param_list->the_param.a_string->str_show())) == NULL)
   {
      holder.sprintf("Location '%s' param 1 in special %s@%s doesn't exist, ", 
                 param_list->the_param.a_string->str_show(), 
                 the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "get_loc_by_dir");
      *results = -1;
      new_param->the_param.a_string = new_Strings("none");
      return new_param;
   }

   if (the_obj->get_type() != OBJ_TYPE_LOCATION)
   {
     holder.sprintf("Object '%s' param 1 in special %s@%s not a Location type", 
                 param_list->the_param.a_string->str_show(), 
                 the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "get_loc_by_dir");
      *results = -1;
      new_param->the_param.a_string = new_Strings("none");
      return new_param;      
   }

   cur_loc = (Location *) the_obj;

   param_list = the_params->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                 "get_loc_by_dir");
      *results = -1;
      return NULL;
   }

   if (param_list->the_param.a_string->str_show() == NULL)
   {
      mainstruct->log_error("Second parameter is an empty string.  "
                            "Check for proper flag settings", 
                                                 "get_loc_by_dir");
      *results = -1;
      return NULL;
   }

   if ((dir_num = 
        cur_loc->dir_to_num(param_list->the_param.a_string->str_show())) == -1)
   {
      if ((dir_num = cur_loc->short_dir_to_num(
                          param_list->the_param.a_string->str_show())) == -1)
      {
         new_param->the_param.a_string = new_Strings("none");
         *results = 0;
         return new_param;
      }
   }
   tmp_gameflags = the_player->get_gameflags();


   holder = cur_loc->get_exit(dir_num);
   if (holder.str_show() == NULL)
   {
      new_param->the_param.a_string = new_Strings("none");
      *results = 0;
      return new_param;
   }
      
   if (!holder.num_char('@'))
      locname.sprintf("%s@%s", holder.str_show(), cur_loc->get_area());
   else
      locname = holder.str_show();

   if ((the_obj = mainstruct->get_object(locname.str_show())) == NULL)
   {
      *results = -1;
      new_param->the_param.a_string = new_Strings("none");
      return new_param;
   }

   /* if we have a door, get the other side */
   if (the_obj->get_type() == OBJ_TYPE_DOOR)
   {
      Door        *the_door;
      Object_List *the_dbase;
      int         tmp_results = 0;      


      the_door = (Door *) the_obj;
      the_dbase = mainstruct->get_dbase();
      holder = the_door->get_link(cur_loc, &tmp_results);

      if (!holder.num_char('@'))
         locname.sprintf("%s@%s", holder.str_show(), cur_loc->get_area());
      else
         locname = holder.str_show();

   }

   new_param->the_param.a_string = new_Strings(locname.str_show());
           
   *results = 1;   
   return new_param;

   /* eliminate compile warnings */
   the_special = NULL;
   the_player = NULL;
   environment = NULL;
   player_params = NULL;
}


/***********************************************************************
 ** display_loc - displays the location of a specific object to the first
 **               parameter object
 **    format: get_location("<this|primary|secondary|actor>", 
 **                                                "<locname>@<area>");
 **
 ***********************************************************************/
params *display_loc(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params    *param_list;
   MudObject *the_obj;
   Strings   the_locname;
   Strings   holder;
   Player    *tmp_player;
   Location  *the_loc;

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "get_location", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      *results = 0;
      return NULL;
   }

   tmp_player = (Player *) the_obj;

   param_list = the_params->next_param;
   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a location name", 
                                                 "display_loc");
      *results = -1;
      return NULL;
   }

   if ((the_obj = mainstruct->get_object(
                     param_list->the_param.a_string->str_show())) == NULL)
   {
      holder.sprintf("Location '%s' param 2 in special %s@%s doesn't exist, ", 
                 param_list->the_param.a_string->str_show(), 
                 the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "display_loc");
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_LOCATION)
   {
      holder.sprintf("Object '%s' in special %s@%s not a Location type", 
                 param_list->the_param.a_string->str_show(), 
                 the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "display_loc");
      *results = -1;
      return NULL;
   }

   the_loc = (Location *) the_obj;
   
   the_loc->show_location(tmp_player, 0);

   player_params = NULL;
   environment = NULL;
   *results = 0;

   return NULL;
}


/***********************************************************************
 ** display_objects - displays all objects of an indicated type in the 
 **                   location indicated by the second parameter and the
 **                   object location to the one indicated in the first 
 **                   parameter
 **
 **    format: display_objects("<this|primary|secondary|actor>", 
 **                            "<this|primary|secondary|actor>", "<attribute>");
 **
 ** Type can be: all, individuals
 **
 ***********************************************************************/
params *display_objects(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   MudObject  *the_obj;
   Location   *the_loc;
   Player     *to_plr;
   params     *param_list;
   Strings    objtype;
   Strings    holder;
   int        show_inds = 0;
   LinkedList *room_contents;

   param_list = the_params;
   if ((param_list == NULL) || (param_list->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "display_objects");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "display_objects", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_PLAYER)
   {
      holder.sprintf("First parameter must be a player in special '%s@%s'", 
		    the_special->get_name(), the_player->get_area());
      mainstruct->log_error(holder.str_show(), "display_objects");
      *results = -1;
      return NULL;
   }

   to_plr = (Player *) the_obj;

   param_list = the_params->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a string", 
                                                              "display_objects");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "display_objects", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   if (the_obj->get_type() != OBJ_TYPE_LOCATION)
   {
      holder.sprintf("Second parameter must be a location in special '%s@%s'", 
		    the_special->get_name(), the_player->get_area());
      mainstruct->log_error(holder.str_show(), "display_objects");
      *results = -1;
      return NULL;
   }

   the_loc = (Location *) the_obj;

   param_list = param_list->next_param;

   if (param_list->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Third parameter should be a string", 
                                                              "display_objects");
      *results = -1;
      return NULL;
   }

   objtype = param_list->the_param.a_string->str_show();

   if (!STRCASECMP(objtype.str_show(), "all"))
   {
      show_inds = 1;
   }
   else if (!STRCASECMP(objtype.str_show(), "individuals"))
   {
      show_inds = 1;
   }
   else
   {
      holder.sprintf("Third parameter object type '%s' not recognized "
                     "in special %s@%s", objtype.str_show(), 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "display_objects");
      *results = -1;
      return NULL;
   }

   room_contents = the_loc->get_inventory();
   room_contents->reset_current();
   the_obj = room_contents->get_next();
   while (the_obj != NULL)
   {   
      if ((the_obj->is_an_individual()) && (show_inds))
	to_plr->send_plr("  &+M%-15s &*&+B: &*%s\n", the_obj->get_title(), 
			 the_loc->get_title());
      the_obj = room_contents->get_next();
   }
   *results = 1;
   return NULL;

   environment = NULL;
}


/***********************************************************************
 ** exit_tutorial - exits the tutorial.  Only works if marked as in a
 **                 tutorial already
 **    format: exit_tutorial();
 **
 ***********************************************************************/
params *exit_tutorial(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   Inp_Handler *the_handler;
   Race        *the_race;
   Strings     cur_tutorial;
   Strings     next_tutorial;
   Strings     holder;
   Object_List *the_dbase;

   the_handler = the_player->get_input_handler();

   cur_tutorial = the_player->get_status_string("currenttutorial");
   if (cur_tutorial.str_show() == NULL)
   {
      mainstruct->log_error("exit_tutorial run while player not in tutorial", 
                            "exit_tutorial");
      *results = -1;
      return NULL;
   }

   if ((the_race = mainstruct->get_race(the_player->get_race())) == NULL)
   {
      holder.sprintf("Player %s's race %s doesn't seem to exist.", 
                  the_player->get_name(), the_player->get_race());
      mainstruct->log_error(holder.str_show(), "exit_tutorial");
      *results = -1;
      return NULL;
   }
   
   next_tutorial = the_race->get_next_tutorial(cur_tutorial.str_show());

   if (next_tutorial.str_show() == NULL)
   {
      the_dbase = mainstruct->get_dbase();
      if (the_dbase->has_inclinations())
      {
         mainstruct->display_inclinations(the_player);
         the_handler->push_input_handler(ask_inclinations, 
                         "Which inclination would you prefer? ", 0);
      }
      else if (the_dbase->has_talents())
      {
         the_player->load_talent_list();
         the_player->display_talents();
         the_handler->push_input_handler(ask_talents, 
                     "Which talents would you prefer? (q when done) ", 0);
      }
      else
      {	
         the_handler->push_input_handler(ask_see_intro, 
                         "Do you want to display the intro? (y/n) ", 0);
      }
      the_player->send_plr("\n");
      *results = -1;
      return NULL;
   }

   /* first we remove the old tutorial */
   the_player->remove_status("currenttutorial");

   /* we mark which tutorial they are in */
   the_player->add_status("currenttutorial", next_tutorial.str_show());

   if (the_race->get_tutorial_mandatory(next_tutorial.str_show()))
   {
      *results = 4;
      the_race->start_tutorial(next_tutorial.str_show(), the_player);
      return NULL;
   }

   the_handler->push_input_handler(handle_tutorial, 
                  the_race->get_tutorial_prompt(next_tutorial.str_show()), 0);
   the_player->send_plr("\n\n");

   *results = 4;
   return NULL;

   /* get rid of all those compile warnings */
   the_special = NULL;
   the_params = NULL;
   the_player = NULL;
   player_params = NULL;
   environment = NULL;

   return NULL;
}


/***********************************************************************
 ** apply_special_loc - applies a particular special to any number of
 **                     locations that exist a particular distance out
 **                     from the object marked in the first parameter.
 **
 **    format: apply_special_loc("<this|primary|secondary|actor>", 
 **                              "<specialname>@<area>", <distance>);
 **
 ***********************************************************************/

params *apply_special_loc(params *the_params, Specials *the_special, 
   Player *the_player, in_params *player_params, special_env *environment, 
                                                             int *results) 
{
   params      *tmp_param;
   MudObject   *the_obj;
   Location    *obj_loc;
   Specials    *tmp_special;
   int         distance;
   Strings     holder;

   if (the_params->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("First parameter should be a string", 
                                        "apply_special_loc (special)");
      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "apply_special_loc", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   while ((the_obj != NULL) && (the_obj->get_contained_by() != NULL))
      the_obj = the_obj->get_contained_by();

   if ((the_obj == NULL) || 
       (the_obj->get_type() != OBJ_TYPE_LOCATION))
   {
      holder.sprintf("Object indicated by parameter one not contained by a "
    "location, special %s@%s", the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "apply_special_loc (special)");
      *results = -1;
      return NULL;
   }
 
   obj_loc = (Location *) the_obj;

   tmp_param = the_params->next_param;

   if (tmp_param->param_type != VAR_TYPE_STRING)
   {
      mainstruct->log_error("Second parameter should be a special name string", 
                                              "apply_special_loc (special)");
      *results = -1;
      return NULL;
   }

   if ((tmp_special = mainstruct->get_special(
                           tmp_param->the_param.a_string->str_show())) == NULL)
   {
      holder.sprintf(
           "Special '%s' not found for second parameter in special '%s@%s'", 
           tmp_param->the_param.a_string->str_show(), the_special->get_name(), 
                                                      the_special->get_area());
      mainstruct->log_error(holder.str_show(), "apply_special_loc (special)");
      *results = -1;
      return NULL;
   }

   tmp_param = tmp_param->next_param;

   if ((tmp_param == NULL) || (tmp_param->param_type != VAR_TYPE_INT))
   {
      mainstruct->log_error("Expected integer as third parameter, not found",
                                      "apply_special_loc (special)");
      *results = -1;
      return NULL;
   }

   distance = (int) *(tmp_param->the_param.an_int);

   /* now apply the special to all in the room.  We use 10 for dir_from to
      specify it comes from the room itself so don't execute the special in
      this room */
   obj_loc->apply_special(the_player, tmp_special, distance, 10);
   obj_loc->cleanup_marks(distance, 10);
   
   return NULL;

   the_special = NULL;
   the_player = NULL;
   environment = NULL;
}


/***********************************************************************
 ** damage_all_room - causes damage to all individuals in a room, killing
 **                   if necessary
 **     
 **
 **    format: damage_all_room(<this|primary|secondary|actor|none>, 
 **        <this|primary|secondary|actor|none>, <damage_number>, 
 **        <damage_str>, <damage_str>, ....);
 **
 ***********************************************************************/

params *damage_all_room(params *the_params, Specials *the_special, 
      Player *the_player, in_params *player_params, special_env *environment, 
                                                                int *results)
{
   MudObject   *the_obj;
   Location    *obj_loc;
   Individual  *the_ind = NULL;
   Individual  *damage_ind;
   Strings     holder;
   Strings     damage_text;
   params      *param_list;
   int         damage;
   LinkedList  *the_list;
   int         prev_health;
   Flags       *tmp_indflags;

   if ((the_params == NULL) || (the_params->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "damage_all_room (special)");

      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "damage_all_room", the_player, player_params)) 
                                                                  == NULL)
   {
      *results = -1;
      return NULL;
   }

   while ((the_obj != NULL) && (the_obj->get_contained_by() != NULL))
      the_obj = the_obj->get_contained_by();

   if ((the_obj == NULL) || 
       (the_obj->get_type() != OBJ_TYPE_LOCATION))
   {
      holder.sprintf("Object indicated by parameter one not contained by a "
    "location, special %s@%s", the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "damage_all_room (special)");
      *results = -1;
      return NULL;
   }
 
   obj_loc = (Location *) the_obj;

   param_list = the_params->next_param;

   the_obj = get_indicated_param(param_list->the_param.a_string->str_show(), 
         the_special, "damage_all_room", the_player, player_params);

   if (the_obj != NULL)
   {
      if (!the_obj->is_an_individual())
      {
         holder.sprintf("Parameter two in special %s@%s must be an individual.",
                           the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "damage_all_room (special)");
         *results = -1;
         return NULL;
      }

      the_ind = (Individual *) the_obj;

      tmp_indflags = the_ind->get_indflags();
   
      if ((tmp_indflags->get_flag(INDFLAG_CORPSE)) ||
          (tmp_indflags->get_flag(INDFLAG_GHOST)))
      {
         holder.sprintf(
            "Can't use corpse or ghost as param two for special %s@%s",
                           the_special->get_name(), the_special->get_area());
         mainstruct->log_error(holder.str_show(), "damage_all_room (special)");
         *results = -1;
         return NULL;
      }

   }
   param_list = param_list->next_param;
   if ((param_list == NULL) ||
       (param_list->param_type != VAR_TYPE_INT))
   {
      holder.sprintf("Third parameter of special %s@%s should be a number",
                              the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "damage_all_room (special)");
      *results = -1;
      return NULL;
   }

   if ((damage = *param_list->the_param.an_int) < 1)
   {
      holder.sprintf("Damage must be greater than 0 in special %s@%s",
                              the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "damage_individual");
      *results = -1;
      return NULL;
   }   

   while (param_list != NULL)
   {
      if (param_list->param_type == VAR_TYPE_STRING)
      {
         damage_text.str_cat(param_list->the_param.a_string->str_show());
      }
      param_list = param_list->next_param;
   }

   the_list = obj_loc->get_inventory();
   the_list->reset_current();
   the_obj = the_list->get_next();

   /* search through the location inventory for individuals */
   while (the_obj != NULL)
   {
      /* we have found the invidividual and they are not the exempt
         individual */
      if ((the_obj->is_an_individual()) &&
          (((Individual *) the_obj) != the_ind))
      {
         damage_ind = (Individual *) the_obj;
         tmp_indflags = damage_ind->get_indflags();
   
         /* if they are not already dead, time to cause some damage */
         if ((!tmp_indflags->get_flag(INDFLAG_CORPSE)) &&
             (!tmp_indflags->get_flag(INDFLAG_GHOST)))
         {
 
	    /* first we tell them what is happening if we have damage text */
            if ((damage_ind->get_type() == OBJ_TYPE_PLAYER) &&
                (damage_text.str_show() != NULL))
               ((Player *) damage_ind)->send_plr("%s\n", damage_text.str_show());

            prev_health = damage_ind->get_health();
            damage_ind->do_damage(damage, WEARFLAG_CHEST);
         }
      }
      the_obj = the_list->get_next();
   }

   *results = 1;
   return NULL;

   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}



/***********************************************************************
 ** light_all_room - lights all lightable objects in the room on fire that
 **                  are not carried by anyone
 **     
 **
 **    format: light_all_room(<this|primary|secondary|actor|none>);
 **
 ***********************************************************************/

params *light_all_room(params *the_params, Specials *the_special, 
      Player *the_player, in_params *player_params, special_env *environment, 
                                                                int *results)
{
   MudObject   *the_obj;
   Location    *obj_loc;
   Strings     holder;
   LinkedList  *the_list;
   Flags       *tmp_itemflags;


   if ((the_params == NULL) || (the_params->param_type != VAR_TYPE_STRING))
   {
      holder.sprintf("Expected string as first param in %s@%s, not found", 
                        the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "light_all_room (special)");

      *results = -1;
      return NULL;
   }

   if ((the_obj = 
         get_indicated_param(the_params->the_param.a_string->str_show(), 
         the_special, "light_all_room", the_player, player_params)) == NULL)
   {
      *results = -1;
      return NULL;
   }

   while ((the_obj != NULL) && (the_obj->get_contained_by() != NULL))
      the_obj = the_obj->get_contained_by();

   if ((the_obj == NULL) || 
       (the_obj->get_type() != OBJ_TYPE_LOCATION))
   {
      holder.sprintf("Object indicated by parameter one not contained by a "
    "location, special %s@%s", the_special->get_name(), the_special->get_area());
      mainstruct->log_error(holder.str_show(), "light_all_room (special)");
      *results = -1;
      return NULL;
   }
 
   obj_loc = (Location *) the_obj;

   the_list = obj_loc->get_inventory();
   the_list->reset_current();
   the_obj = the_list->get_next();

   /* search through the location inventory for lightable objects */
   while (the_obj != NULL)
   {
      /* we have found the invidividual and they are not the exempt
         individual */
      if (the_obj->is_an_item())
      {
         tmp_itemflags = ((Item *) the_obj)->get_itemflags();

         if ((tmp_itemflags->get_flag(ITEMFLAG_LIGHTABLE)) &&
             (!tmp_itemflags->get_flag(ITEMFLAG_LIT)))
            tmp_itemflags->set_flag(ITEMFLAG_LIT);
      }
      the_obj = the_list->get_next();
   }

   *results = 1;
   return NULL;

   the_player = NULL;
   player_params = NULL;
   environment = NULL;
}



#endif

